/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_ascii
  Output of text files. One for each image. 

2 PURPOSE
  This program writes an ascii file that contains the value of
  each pixel. A header can preceed the file. The first column and/or
  row can contain the array, image, real or saxs coordinates of the file. 

  If +var is set a second output file with the same layout as the first file
  is written that contains the standard deviation of each data point
  (square root of variance array). If the standard deviation could not be 
  calculated VarDummy, a negative integer number, is written.

  Calculation:       I0 = I1
  Error propagation: V0 = V1

  Arguments:
  saxs_ascii [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> 
             <header print> <swap output> <head 1> <head 2>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <output dummy>      : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <header print>      : on   * to print the header in the ascii file
  <swap output>       : off  * to transpose the output in the file
  <head 1>            : on   * to see the i_1 world coordinate at the top
  <head 2>            : on   * to see the i_2 world coordinate on the left

  Specific options:
  -pref 'prefix'      : specify the prefix of the ouptput file
  -ext 'extension'    : extension of the output file (default: "txt")
  -hdmk 'marker'      : first character in a header line (default: "#")
  -spc 'spacer'       : spacer string between columns (default: "\t")

  +-head              : enables/disables the header printing
  +-swap              : enables/disables the swap
  +-hd1               : enables/disables the i_1 coordinate printing
  +-hd2               : enables/disables the i_2 coordinate printing
  +-abs               : absolut values of coordinates
  -scf <factor>       : multiplication of coordinates with float <factor>
  +-hist              : writes/do not write history into header
  +-noi               : suppress/do not suppress image number extensions in 
                        output file
  +-waxs              : calculate exact scattering vector length from 
                        saxs-coordinate. This is only necessary if the data 
                        were not treated with the program saxs_waxs and if the 
                        distance between sample and detector is small. The 
                        option +waxs will set automatically the reference 
                        system to -rsys saxs.

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  1995        Pierre Jourde
  01-Dec-1995 PB
  27-Nov-1996 PB error in i_1 header in non-swapped data corrected
  03-Dec-1996 PB option +abs (absolut values of coordinates) and option -scf
  01-Aug-1998 PB V3.21 Segmentation error in Linux version due to an error 
                       in writing info pcb->Lin[NPrefix]: .V extension
                       forgotten, now corrected.
  01-Aug-1998 PB V3.3  All new lines '\n' replaced by '\r\n'
  25-Feb-1999 PB V3.4  Header marker can be changed with -hdmk <marker>.
                       Default is "#".
  27-Mar-2000 PB V3.5  option for spacer between columns (-spc <spacer>)
  2000-08-04  PB V3.5  %f->%g
  2000-08-15  PB V3.6  first column header changed
  2000-09-15  PB V3.7  option -hedl for head line string
  2000-11-18  PB V3.8  WriteHistory, WriteHeader, Time keyword into header
                       new options: +/-hist (NHist), -len  (NHlen)
  2000-11-18  PB V3.9  default header line starts with #
  2001-03-02  PB V3.10 space before \r\n
  2001-03-30  PB V3.11a space not in label line
  2001-07-03  PB V3.12 filename_full
  2001-07-07  PB V3.13 IO_DontReadWrite
  2001-07-08  PB V3.14 FFirst
  2001-07-09  PB V3.15 ib[0].FileNumber
  2001-07-09  PB V4.00 new loop, no repetition
  2001-08-30  PB V4.01 
  2001-09-02  PB V4.02 option noi (switch on/off suppression of image numbers)
  2002-05-31  PB V4.50 ave 
  2002-07-26  PB V4.51 hedl also for multiple files 
  2002-10-09  PB V4.52 option waxs 
  2003-02-02  PB V4.53 num[BlockNum] 
  2003-02-09  PB V4.54 Dummy, DDummy, Dim IO_*exp.
                       Read WaveLength only when WaxsI is set.
  2003-03-30  PB V4.55 ImageLoop: Function call changed
  2003-08-11  PB V4.56 option_define, all option numbers removed
  2003-11-30  PB V4.57 BinSiz
  2004-05-30  PB V4.58 Writes also the last history line 
  2004-09-26  PB V4.59 To avoid a wrong calculation of scattering vectors
                       when the projection type of the output is already
                       Waxs, WaxsI is set to zero in this case. In this
                       case the scattering vector s is identical to the
                       saxs-coordinate. To overcome this behavious (why?)
                       use the option -opro saxs.
  2004-10-02  PB V4.60 Function names changed to avoid conflicts with 
                       public definitions:
                       WriteHistory => ascii_write_history, 
                       WriteHeader => ascii_write_header
  2004-10-31  PB V4.61 MAKE_FUNCTION
  2005-07-11  PB V4.62 exit -> return, ERROUT
  2005-07-15  PB V4.63 ReportSaxsStatus parameter list updated,
                       SetSaxsErrorExternal added
  2005-08-04  PB V4.64 PrintImageHeaderBlock paramters updated
  2005-09-17  PB V4.65 VarDat
  2006-07-26  PB V4.66 stddev written into headline of stddev file
  2007-04-23  PB V4.67 for (cnt;...) -> for (;...)
  2007-06-11  PB V4.68 ImageLoop parameters updated
  2008-08-05  PB V4.69 SaxsAsciiArgvFilenames: Dummy, Dim_1 and Dim_2 
                       are not globally set any more for the output file.
  2008-12-05  PB V4.70 open output text file as binary file to avoid
                       doubling of \r (\r\r\n) at the end of the line when
                       compiled on windows.
  2011-02-22  PB V4.71 include <string.h>
  2011-10-22  PB V4.72 include "strlib.h": strtok -> strlib_tok
  2012-01-31  PB V4.73 ImageLoop parameters updated

---*/
# define Version  "saxs_ascii V4.73 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <math.h>

# include "strlib.h"
# include "SaxsPrograms.h"

// line
# define SPrefix    "pref"
# define SExtension "ext"
# define SHdmk      "hdmk"
# define SSpacer    "spc"
# define SHeadL     "hedl"
// flag 
# define SihbPrint  "head" // write header
# define SHist      "hist" // write history
# define Si1Head    "hd1"  // write header 1
# define Si2Head    "hd2"  // write header 2
# define SAbs       "abs"  // write absolute values of coordinates
# define SSwap      "swap" // swap columns and lines in output file
# define SNoImNum   "noi"  // suppress image number in output, if not required
# define SWaxs      "waxs" // calculate 2*sin(Theta) from tan(2*Theta) 
// flexp
# define SScf       "scf"  // scale factor for header column
// lvexp
# define SHlen      "len"  // minimum number of header lines (if +head)

# define E0Extension "_stddev"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc>\n\
                <odum> <odim1> <odim2>\n\
                <header> <swap> <head_1> <head_2>\n\
Purpose: Transformation into ASCII format\n\
Options: +-head +-swap +-hd1 +-hd2 -hdmk <marker> -ext <extension> +-noi +-waxs"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

const char *delim = " ";

char * str_tok( char * string, const char * delimiter )
{ char * label=(char*) NULL;
  label=strlib_tok(string, delimiter);

  return( label );
} /* str_tok */

/*---------------------------------------------------------------------------
SaxsAsciiArgvFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

If blkno >= 1 the header block of the first image is read into ihb[1].
The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void SaxsAsciiArgvFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long image_1st, image_lst;
  char linebuffer[IO_len];
  int nameblk;
  long num[BlockNum];
  IO_line * pPrefix;

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib, blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile( pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {

    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadWrite | IO_DontOpen;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- Argument  : text file prefix */
    if (ib[0].Name.I) nameblk = 0; else nameblk = 1;
    pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );
    if (*pstatus!=Success) return;

    if (!pPrefix->I) {
      /* Determine prefix from input file name */
      // remove extension
      filename_body ( pPrefix->V, IO_len, ib[nameblk].Name.V );
      // remove path
      filename_name ( pPrefix->V, IO_len, pPrefix->V );
      pPrefix->I = TRUE;
      } 

    /*--- Prefix of output file name */
    argv_line( pcb,"Ascii file prefix",pPrefix,pPrefix->V, 
               pstatus);
    if (*pstatus!=Success) return;

    /* extract output filenumbers from ASCII file name */
    extract_filenumbers( &ib[0], pPrefix, TRUE, pstatus );
    if (*pstatus!=Success) return;

    /* Expand Prefix to the full file name */
    if (!filename_full (pPrefix->V,IO_len,pPrefix->V)) {
      fprintf(ERROUT,"Input file name or path not allowed\n");
      *pstatus = FileNameError; return;
    }

    /*--- Take prefix as default for output filename */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",pPrefix->V);
      } 

    } /* output file */

  /* ---  Image numbers --- */

  if (first_block==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    num[1] = ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

    /* Defaults for output file */
    if (!ib[0].Dummy.I) sprintf(ib[0].Dummy.V,"%g",ihb[1].Dummy.V);
    if (!ib[0].Dim[1].I) sprintf(ib[0].Dim[1].V,"%d",ihb[1].Dim[1]);
    if (!ib[0].Dim[2].I) sprintf(ib[0].Dim[2].V,"%d",ihb[1].Dim[2]);

  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;
    num[blkno] = ib[blkno].First.V;

  } /* all other input files */

  if (block_1st <= 0) {

    if (ib[0].First.I) num[0] = ib[0].First.V;
    else num[0] = image_1st;

    if ( (pcb->ImgBlkLen>1) && (ihb[1].I) ) {
      NewImageHeader(pcb,ib,0,num,ihb,&(ihb[1]),NULL,pstatus);
    } else {
      NewImageHeader(pcb,ib,0,num,ihb,NULL,NULL,pstatus);
    }
    if (*pstatus!=Success) { free(num); return; }

    /*--- Argument : output dummy */
    sprintf(linebuffer,"Output dummy, e.g. %g",ihb[0].Dummy.V);
    argv_flexp( pcb, linebuffer, &ib[0].Dummy, "", pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 1 */
    sprintf(linebuffer,"Output dimension 1, e.g. %d",ihb[0].Dim[1]);
    argv_lvexp(pcb,linebuffer,
      &ib[0].Dim[1],ib[0].Dim[1].I?ib[0].Dim[1].V:"",pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 2 */
    sprintf(linebuffer,"Output dimension 2, e.g. %d",ihb[0].Dim[2]);
    argv_lvexp(pcb,linebuffer,
      &ib[0].Dim[2],ib[0].Dim[2].I?ib[0].Dim[2].V:"", pstatus);
    if (*pstatus!=Success) return;

  }

  for (blkno = blkno = MAX2(1,first_block);blkno<=block_lst;blkno++) {
    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;
  } /* close all input files */

} /* SaxsAsciiArgvFilenames */

/*----------------------------------------------------------------------------
NAME
  ascii_write_history

DESCRIPTION
  Writes the full history into the output file 'out'

RETURN VALUE
  Number of written lines

HISTORY
  18-Nov-2000 PB
----------------------------------------------------------------------------*/
long ascii_write_history( FILE * out, CmdBlk * pcb, long num[], ImgHeadBlk ihb[], 
                   int * pstatus )
{
  long cnt=0;
  IO_line Hdmk;

  const char * header_key="header_tmp";
  const char * key, * value;
 
  int ErrorValue;

  *pstatus = Failed;

  /* write history to a separate header */

  if ( !edf_new_header( header_key ) ) return( cnt );
  edf_history_write_header ( header_key, SaxsHistoryKey,
                             &ErrorValue, pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }

  Hdmk = option_line ( SHdmk, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }

  edf_first_header_element ( header_key, &key, &value, &ErrorValue, pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }

  while (key) {
    fprintf(out,"%s%s = %s \r\n", Hdmk.V, key, value);
    cnt++;
    edf_next_header_element ( header_key, &key, &value, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { edf_free_header( header_key ); return( cnt ); }
  }

  if ( !edf_free_header( header_key ) ) return( cnt );

  *pstatus = Success;
  return(cnt);

} /* ascii_write_history */

/*----------------------------------------------------------------------------
NAME
  ascii_write_header 

DESCRIPTION
  Writes the header 

RETURN VALUE
  Number of written lines

HISTORY
  18-Nov-2000 PB
----------------------------------------------------------------------------*/
long ascii_write_header( FILE *out, CmdBlk *pcb, long num[], ImgHeadBlk ihb[], 
                  int *pstatus )
{ long cnt=0;
  IO_long cntmax, Hist;
  IO_line Hdmk;

  Hdmk = option_line ( SHdmk, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return( cnt );

  Hist = option_flag ( SHist, pcb, pstatus );
  if (*pstatus!=Success) return( cnt );

  cntmax = option_long ( SHlen, pcb, pcb->ib, num, 1, pstatus ); // header lines
  if (*pstatus!=Success) return( cnt );

  fprintf(out,"%s{\r\n",Hdmk.V); cnt++;
  fprintf(out,"%s%s = % d \r\n", Hdmk.V, KImage, ihb->ImNum); cnt++;
  fprintf(out,"%s%s = % d \r\n", Hdmk.V, "Size", ihb->ImgByteLen); cnt++;
  fprintf(out,"%s%s = % d \r\n", Hdmk.V, "Dim_1", ihb->Dim[1]); cnt++;
  fprintf(out,"%s%s = % d \r\n", Hdmk.V, "Dim_2", ihb->Dim[2]); cnt++;
  if (ihb->Dummy.I)        { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KDummy, ihb->Dummy.V); cnt++; }
  if (ihb->DDummy.I)       { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KDDummy, ihb->DDummy.V); cnt++; }
  if (ihb->Offset[1].I)    { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KOffset_1, ihb->Offset[1].V); cnt++; }
  if (ihb->Offset[2].I)    { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KOffset_2, ihb->Offset[2].V); cnt++; }
  if (ihb->Center[1].I)    { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KCenter_1, ihb->Center[1].V); cnt++; }
  if (ihb->Center[2].I)    { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KCenter_2, ihb->Center[2].V); cnt++; }
  if (ihb->PixSiz[1].I)    { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KPixSiz_1, ihb->PixSiz[1].V); cnt++; }
  if (ihb->PixSiz[2].I)    { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KPixSiz_2, ihb->PixSiz[2].V); cnt++; }
  if (ihb->SampleDistance.I) { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KSampleDistance, ihb->SampleDistance.V); cnt++; }
  if (ihb->WaveLength.I)   { fprintf(out,"%s%s = % e \r\n",
    Hdmk.V, KWaveLength, ihb->WaveLength.V); cnt++; }
  if (ihb->Title.I)        { fprintf(out,"%s%s = % s \r\n",
    Hdmk.V, KTitle, ihb->Title.V); cnt++; }
  if (ihb->Time.I)         { fprintf(out,"%s%s = % s \r\n",
    Hdmk.V, KTime, ihb->Time.V); cnt++; }
  if (Hist.V)
    cnt += ascii_write_history( out, pcb, num, ihb, pstatus );
  if (*pstatus != Success) return( cnt );

  for (;cnt<cntmax.V-1;cnt++) {
    fprintf(out,"%s\r\n", Hdmk.V);
    }
  fprintf(out,"%s}\r\n", Hdmk.V); cnt ++;

  return( cnt );

} /* ascii_write_header */

//float floattest( float in )
//{ return( in );
//} // floattest

/*---------------------------------------------------------------------------
1 saxs_ascii

2 PURPOSE
  Output of ascii data
---------------------------------------------------------------------------*/

void saxs_ascii (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int   I1Ave;
  float *I1Data, *E1Data; 
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  char OutputFileName[InputLineLength];
  FILE *OutputFileId;
  char OutputErrorFileName[InputLineLength];
  FILE *OutputErrorFileId;

  IO_float Scf, *ptmp;
  IO_line Hdmk, Spacer, HeadL, Extension, Prefix;
  IO_long ihbPrint, Hist, i1Head, i2Head, Abs, Swap, NoImNum, Waxs;

  char *label=(char *) NULL, *nextlabel=(char *) NULL, *headl=(char *) NULL;
  char fname_full[IO_len];

  /* boolean indicators */
  int SwapI, i1HeadI, i2HeadI, AbsI, WaxsI;

  /* output coordinates */
  float O_1, O_2;
  float V_1, V_2;
  float K; 

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\nCalculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

 /* options */
  Hdmk = option_line ( SHdmk, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  Spacer = option_line ( SSpacer, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  HeadL = option_line ( SHeadL, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  Extension = option_line ( SExtension, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  Prefix = option_line ( SPrefix, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return; 

  ihbPrint = option_flag ( SihbPrint, pcb, pstatus );
  if (*pstatus!=Success) return;

  Hist = option_flag ( SHist, pcb, pstatus );
  if (*pstatus!=Success) return;

  i1Head = option_flag ( Si1Head, pcb, pstatus );
  if (*pstatus!=Success) return;

  i2Head = option_flag ( Si2Head, pcb, pstatus );
  if (*pstatus!=Success) return;

  Abs = option_flag ( SAbs, pcb, pstatus );
  if (*pstatus!=Success) return;

  Swap = option_flag ( SSwap, pcb, pstatus );
  if (*pstatus!=Success) return;

  NoImNum = option_flag ( SNoImNum, pcb, pstatus );
  if (*pstatus!=Success) return;

  Waxs = option_flag ( SWaxs, pcb, pstatus );
  if (*pstatus!=Success) return;

  Scf = option_float ( SScf, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return; 

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* loop over the output array */
  W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
 
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {
 
     if ( E0Data ) {
        // V0 = V1
        if ( Isum2ldwE(I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
               f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1],
               &I1Sum, &I1Weight, &E1Sum, &E1Weight) ) {
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
          UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
 
          if (E1Sum >= 0.0) {
            E1Value = E1Sum; if (I1Ave) E1Value /= E1Weight*E1Weight;
            UPDATE( *pE0Data, E1Value, VarDummy, VarDDummy );
          }
        }
      } else {
        if (Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
              f_1[1],f_2[1],f_1[1]+Df_1[1],f_2[1]+Df_2[1],
              &I1Sum,&I1Weight)) {
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
          UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
        } /* if (Ipol2d ... */
      }

      pI0Data++;
      pE0Data++;
      W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */

    W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
  } /* for i_2 ... */

  /* construct the file name from Prefix.V and the image number */
  
  /* Generate filename of ASCII file */
  filename_parameter (fname_full,IO_len,Prefix.V,0);
  filename_pattern (fname_full,IO_len,fname_full,(pcb->ib)[0].FileNumber);
  filename_full (fname_full,IO_len,fname_full);

  if ((NoImNum.V) && ImageIsFirst(pcb, 0, ihb ) && ImageIsLast(pcb, 0, ihb )) {
    /* suppress image number if not required */
    sprintf(OutputFileName,"%s%s",fname_full, Extension.V);
    if (E0Data) sprintf(OutputErrorFileName,"%s%s%s",
      fname_full, E0Extension, Extension.V);
  } else {
    /* write image number always */
    sprintf(OutputFileName,"%s_%d%s",fname_full,ihb[0].ImNum, Extension.V);
    if (E0Data) sprintf(OutputErrorFileName,"%s_%d%s%s",
      fname_full,ihb[0].ImNum, E0Extension, Extension.V);
  }

  /* open ascii file "Prefix"_NNN.txt" */
  if ((OutputFileId=fopen(OutputFileName,"wb")) == NULL) return;
  if (E0Data) 
    if ((OutputErrorFileId=fopen(OutputErrorFileName,"wb")) == NULL) return;

  printf("Writing output ascii file : %s",OutputFileName);
  if (E0Data) printf(", %s",OutputErrorFileName);
  printf("\n");

  /* writing of the image header block into the ouput file */
  if (ihbPrint.V) {
    /* Writing header */
    ascii_write_header( OutputFileId, pcb, num, ihb, pstatus );
    if (*pstatus!=Success) return;
    if (E0Data) {
      ascii_write_header( OutputErrorFileId, pcb, num, ihb, pstatus );
      if (*pstatus!=Success) return;
    }
  }

  /* set the boolean indicators */
  SwapI   = Swap.V;
  i1HeadI = i1Head.V;
  i2HeadI = i2Head.V;
  AbsI    = Abs.V;
  WaxsI   = Waxs.V;
  /* Do not set WaxsI if the image is already in Waxs Projection */
  if (ihb[0].Projection.V==IO_ProWaxs) WaxsI=0;

  if (WaxsI) 
    if (ihb[0].WaveLength.I) K = WAVENUMBER(ihb[0].WaveLength.V);
      else {
      *pstatus=Failed;
      fprintf(ERROUT,
        "Missing parameter WaveLength for scattering vector calculation\n"); 
      return;
    }

  if (pcb->TestBit) 
    printf("\nFlag : Head=%d, Swap=%d, Hd1=%d, Hd2=%d, Abs=%d, Waxs=%d\n",
           ihbPrint.V,SwapI, i1HeadI, i2HeadI, AbsI, Waxs.V);

  /* writing of the x (or y if swap) heading line if wished */
  W_1 = Wmin_1; W_2 = Wmin_2;

  if (WaxsI) { V_1 = 2*K*sin(0.5*atan(W_1/K)); V_2 = 2*K*sin(0.5*atan(W_2/K)); 
  } else { V_1 = W_1; V_2 = W_2; }

  if (AbsI) { O_1 = fabs(Scf.V*V_1);  O_2 = fabs(Scf.V*V_2); 
   } else { O_1 = Scf.V*V_1; O_2 = Scf.V*V_2; }

  /* Writing data */

  if (HeadL.I) {
    headl = malloc(strlen(HeadL.V)+1);
    nextlabel = strcpy(headl,HeadL.V);
  }


  if (SwapI) {
    /* Swapped */
    if (i2HeadI) {
      if (nextlabel) label = str_tok(nextlabel, delim);

      if (i1HeadI) {

        if (label) {
          fprintf(OutputFileId,"%s%s",label,Spacer.V);
          if (E0Data) fprintf(OutputErrorFileId,"%s%s",label,Spacer.V);
        } else {
          fprintf(OutputFileId,"#i_1v/i_2>%s",Spacer.V);
          if (E0Data) fprintf(OutputErrorFileId,"#i_1v/i_2>%s",Spacer.V);
        }
        if (nextlabel) label = str_tok((char*) NULL, delim);

      } // if (i1HeadI)

      if (label) {
        fprintf(OutputFileId,"%s",label);
        if (E0Data) fprintf(OutputErrorFileId,"stddev(%s)",label);
      } else {
        fprintf(OutputFileId,"%g",O_2);
        if (E0Data) fprintf(OutputErrorFileId,"stddev(%g)",O_2);
      }

      for (i_2=1; i_2<I0Dim_2; i_2++) {
        W_2 += DW_2;
        if (WaxsI) { V_2 = 2*K*sin(0.5*atan(W_2/K)); } else { V_2 = W_2; }
        if (AbsI) O_2 = fabs(Scf.V*V_2); else O_2 = Scf.V*V_2; 
  
        if (nextlabel) label = str_tok((char *) NULL, delim);
 
        if (label) {
          fprintf(OutputFileId,"%s%s",Spacer.V,label);
          if (E0Data) fprintf(OutputErrorFileId,"%sstddev(%s)",Spacer.V,label);
        } else {
          fprintf(OutputFileId,"%s%g",Spacer.V,O_2);
          if (E0Data) fprintf(OutputErrorFileId,"%sstddev(%g)",Spacer.V,O_2);
        }
      }
      fprintf(OutputFileId,"\r\n");
      if (E0Data) fprintf(OutputErrorFileId,"\r\n");
    } // if (i2HeadI)
      
  } else {

    /* not swapped */
    if (i1HeadI) {
      if (nextlabel) label = str_tok(nextlabel, delim);

      if (i2HeadI) {

        if (label) {
          fprintf(OutputFileId,"%s%s",label,Spacer.V);
          if (E0Data) fprintf(OutputErrorFileId,"%s%s",label,Spacer.V);
        } else {
          fprintf(OutputFileId,"i_2v/i_1>%s",Spacer.V);
          if (E0Data) fprintf(OutputErrorFileId,"i_2v/i_1>%s",Spacer.V);
        }
        if (nextlabel) label = str_tok((char *) NULL, delim);
      }

      if (label) { 
        fprintf(OutputFileId,"%s",label);
        if (E0Data) fprintf(OutputErrorFileId,"stddev(%s)",label);
      } else {
        fprintf(OutputFileId,"%g",O_1);
        if (E0Data) fprintf(OutputErrorFileId,"stddev(%g)",O_1);
      }

      for (i_1=1; i_1<I0Dim_1; i_1++) {
        W_1 += DW_1;
        if (WaxsI) { V_1 = 2*K*sin(0.5*atan(W_1/K)); } else { V_1 = W_1; }
        if (AbsI) O_1 = fabs(Scf.V*V_1); else O_1 = Scf.V*V_1; 
        if (nextlabel) label = str_tok((char *) NULL, delim);

        if (label) {
          fprintf(OutputFileId,"%s%s",Spacer.V,label);
          if (E0Data) fprintf(OutputErrorFileId,"%sstddev(%s)",Spacer.V,label);
        } else {
          fprintf(OutputFileId,"%s%g",Spacer.V,O_1);
          if (E0Data) fprintf(OutputErrorFileId,"%sstddev(%g)",Spacer.V,O_1);
        }
      }
      fprintf(OutputFileId,"\r\n");
      if (E0Data) fprintf(OutputErrorFileId,"\r\n");
    } // if (i1Head1)

  } // if not swapped 

  /* Write the array I0Data[I0Dim_1,I0Dim_2] to the file
     array is written by lines : each row of the line is separated
     from the next by a tab caracter (ASCII 9), each line by a RC */

  if (SwapI) { /* if swapped output */
    W_1= Wmin_1;

    if (WaxsI) { V_1 = 2*K*sin(0.5*atan(W_1/K)); } else { V_1 = W_1; }
    if (AbsI) { O_1 = fabs(Scf.V*V_1);  O_2 = fabs(Scf.V*V_2);
      } else { O_1 = Scf.V*V_1; O_2 = Scf.V*V_2; }

    for (i_1=0; i_1<I0Dim_1; i_1++) {
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,0);
      pE0Data = E0Data-I0Data+pI0Data;

      if (i1HeadI) {
        fprintf(OutputFileId,"%g%s",O_1,Spacer.V);
        if (E0Data) fprintf(OutputErrorFileId,"%g%s",O_1,Spacer.V);
        W_1 += DW_1;
        if (WaxsI) { V_1 = 2*K*sin(0.5*atan(W_1/K)); } else { V_1 = W_1; }
        if (AbsI) O_1 = fabs(Scf.V*V_1); else O_1 = Scf.V*V_1;
      }

      fprintf(OutputFileId,"%g",*pI0Data);
      if (E0Data) fprintf(OutputErrorFileId,"%g",
                    (*pE0Data<0)?VarDummy:sqrt(*pE0Data));
      for (i_2=1; i_2<I0Dim_2; i_2++) {
        pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);
        pE0Data = E0Data-I0Data+pI0Data;
        fprintf(OutputFileId,"%s%g",Spacer.V,*pI0Data);
        if (E0Data) fprintf(OutputErrorFileId,"%s%g",Spacer.V,
                      (*pE0Data<0)?VarDummy:sqrt(*pE0Data));
      }

      fprintf(OutputFileId," \r\n");
      if (E0Data) fprintf(OutputErrorFileId," \r\n");
    }
  } else { /* if not swapped output */
    W_2= Wmin_2;
    if (WaxsI) { V_2 = 2*K*sin(0.5*atan(W_2/K)); } else { V_2 = W_2; }
    if (AbsI) O_2 = fabs(Scf.V*V_2); else O_2 = Scf.V*V_2; 
    for (i_2=0; i_2<I0Dim_2; i_2++) {
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,0,i_2);
      pE0Data = E0Data-I0Data+pI0Data;
 
      if (i2HeadI) {
        fprintf(OutputFileId,"%g%s",O_2,Spacer.V);
        if (E0Data) fprintf(OutputErrorFileId,"%g%s",O_2,Spacer.V);
        W_2 += DW_2;
        if (WaxsI) { V_2 = 2*K*sin(0.5*atan(W_2/K)); } else { V_2 = W_2; }
        if (AbsI) O_2 = fabs(Scf.V*V_2); else O_2 = Scf.V*V_2; 
        }

      fprintf(OutputFileId,"%g",*pI0Data);
      if (E0Data) fprintf(OutputErrorFileId,"%g",
                    (*pE0Data<0)?VarDummy:sqrt(*pE0Data));

      for (i_1=1; i_1<I0Dim_1; i_1++) {
        pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);
        pE0Data = E0Data-I0Data+pI0Data;
        fprintf(OutputFileId,"%s%g",Spacer.V,*pI0Data);
        if (E0Data) fprintf(OutputErrorFileId,"%s%g",Spacer.V,
                      (*pE0Data<0)?VarDummy:sqrt(*pE0Data));
      }

      fprintf(OutputFileId," \r\n");
      if (E0Data) fprintf(OutputErrorFileId," \r\n");
    }
  }

  /* close ascii file */

  free(headl);

  if (fclose(OutputFileId)!=0) return;
  if (E0Data) if (fclose(OutputErrorFileId)!=0) return;

} /* saxs_ascii*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_line *pPrefix;
  IO_long *pihbPrint, *pi1Head, *pi2Head, *pSwap;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,
           InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  /* Expand Prefix to the full file name */
  pPrefix = (IO_line*) option_parameter_search( SPrefix, pcb, pstatus );
  if (*pstatus!=Success) return;

  if (!filename_full (pPrefix->V,IO_len,pPrefix->V)) {
    fprintf(ERROUT,"% s. Input file name or path not allowed\n",progfn);
    *pstatus = FileNameError; return;
  }
 
  SaxsAsciiArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : image header block printing authorization */
  pihbPrint = (IO_long*) option_parameter_search( SihbPrint, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flag(pcb,"Write header block (yes|no)",
            pihbPrint,pihbPrint->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : swapping of the output file */
  pSwap = (IO_long*) option_parameter_search( SSwap, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flag(pcb,"Swap output (yes|no)",
            pSwap,pSwap->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : i_1 heading in the output file authorization */
  pi1Head = (IO_long*) option_parameter_search( Si1Head, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flag(pcb,"Write column coordinates (yes|no)",
            pi1Head,pi1Head->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : i_2 heading in the output file authorization */
  pi2Head = (IO_long*) option_parameter_search( Si2Head, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flag(pcb,"Write row coordinates (yes|no)",
            pi2Head,pi2Head->V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)   printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)   printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)  printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)   printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)    printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)  printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I) printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I) printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (pihbPrint->I)
                      printf("header print       : %d\n",pihbPrint->V);
  if (pSwap->I)
                      printf("swap               : %d\n",pSwap->V);
  if (pi1Head->I)
                      printf("i_1 head           : %d\n",pi1Head->V);
  if (pi2Head->I)
                      printf("i_2 head           : %d\n",pi2Head->V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_ascii
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  IO_long Waxs;

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpline, SPrefix, "datafile", &cb );
  option_define ( IO_tpline, SExtension, ".txt", &cb );
  option_define ( IO_tpline, SHdmk, "#", &cb ); /* default is # */
  option_define ( IO_tpline, SSpacer, "\t", &cb ); /* default is tab */
  option_define ( IO_tpline, SHeadL, "", &cb ); /* default is "" */
  option_define ( IO_tpflag, SihbPrint, "TRUE", &cb ); /* default on */
  option_define ( IO_tpflag, SHist, "TRUE", &cb );
  option_define ( IO_tpflag, Si1Head, "TRUE", &cb ); /* default on */
  option_define ( IO_tpflag, Si2Head, "TRUE", &cb ); /* default on */
  option_define ( IO_tpflag, SAbs, "FALSE", &cb );
  option_define ( IO_tpflag, SSwap, "FALSE", &cb ); /* default off */
  option_define ( IO_tpflag, SNoImNum, "FALSE", &cb );
  option_define ( IO_tpflag, SWaxs, "FALSE", &cb );
  option_define ( IO_tpflexp, SScf, "1.0", &cb );
  option_define ( IO_tplvexp, SHlen, "25", &cb ); /* at least 25 header lines */

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

 /* Force reference system to SAXS */
  if (status==Success)
    Waxs = option_flag( SWaxs, &cb, &status );
  if (status==Success) 
    if (Waxs.V) cb.RSys.V = IO_Saxs;

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, saxs_ascii, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

