/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/**************************************************************************/
/*                                                                        */
/* EDFTiff.c                                                              */
/*                                                                        */
/*DESCRIPTION                                                             */
/* Routine to write a simple tiff file with comment and pixel size        */
/*                                                                        */
/*HISTORY                                                                 */
/* 1999-10-18 Peter Boesecke                                              */
/* 1999-10-31 PB INTERNAL_BYTE_ORDER                                      */
/*               INTERNAL_BYTE_ORDER should be a function that uses       */
/*               BYTE_ORDER when defined                                  */
/* 1999-11-08 PB float2unsigned                                           */
/* 1999-11-09 PB TIFF_GREYSCALE changed from 0 to 1 (0:dark,maximum:white)*/
/*               DUMMYDEFINED                                             */
/* 1999-11-10 PB in float2unsigned unsigned char*, short*, long*          */
/* 1999-11-11 PB autorange sets dummyout to maximum value                 */
/* 1999-11-12 PB automin and automax, error in autoscale corrected        */
/* 1999-11-16 PB BYTE_ORDER replaced by INTERNAL_BYTE_ORDER               */ 
/* 1999-11-26 PB cc on dec alpha: underflow for eps=1e-40, initialized    */
/*               with FLT_MIN (<float.h>).                                */
/* 1999-12-21 PB                                                          */
/* 2000-01-07 PB fwrite ( &(pheader->byteorder ), ... ); changed to       */
/*               fwrite ( (pheader->byteorder ), ... );                   */
/* 2000-01-07 PB machine_sizes                                            */
/* 2005-02-24 PB INTERNAL_BYTE_ORDER -> edf_byteorder                     */ 
/* 2007-04-19 PB -Wall compiler warnings resolved                         */
/*               %zu formats complemented by %lu                          */
/* 2007-04-25 PB comparisons between signed and unsigned removed          */
/* 2010-12-18 PB write_tiff: first argument of strncpy is now (char*)     */
/*                                                                        */
/**************************************************************************/

/**************************************************************************/
/* edftiff.c                                                              */
/**************************************************************************/

# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <ctype.h>
# include <errno.h>
# include <float.h>

# include "SaxsPrograms.h"
# include "edfio.h"
# include "edftiff.h"


/* CEILMOD: Round N up to the next full multiple of DIV */
#ifndef CEILMOD
# define CEILMOD(N,DIV) (((N)%(DIV))?((DIV)+(N)-(N)%(DIV)):(N))
#endif

// adaptable TIFF parameters
# define TIFF_BLOCK_SIZE                      4
# define TIFF_RESOLUTION_QUOTIENT        100000
# define TIFF_MAXTAGS                        32
# define TIFF_BUFLEN                       1024

// fixed TIFF specific definitions
# define TIFF_DATATYPE_BYTE                   1
# define TIFF_DATATYPE_ASCII                  2
# define TIFF_DATATYPE_SHORT                  3
# define TIFF_DATATYPE_LONG                   4
# define TIFF_DATATYPE_RATIONAL               5
# define NUMBER_OF_TIFF_DATATYPES             5

# define TIFF_NOCOMPRESSION                   1
# define TIFF_GREYSCALE                       1
# define TIFF_PPINCH                          2
# define TIFF_DPCM                            3

# define TIFF_TAG_NewSubfileType            254 
# define TIFF_TAG_ImageWidth                256
# define TIFF_TAG_ImageLength               257
# define TIFF_TAG_BitsPerSample             258
# define TIFF_TAG_Compression               259
# define TIFF_TAG_PhotometricInterpretation 262
# define TIFF_TAG_ImageDescription          270
# define TIFF_TAG_StripOffsets              273
# define TIFF_TAG_SamplesPerPixel           277
# define TIFF_TAG_RowsPerStrip              278
# define TIFF_TAG_StripByteCounts           279
# define TIFF_TAG_XResolution               282
# define TIFF_TAG_YResolution               283
# define TIFF_TAG_ResolutionUnit            296

# define TIFF_HEADER_SIZE                     8 // tiff header
# define TIFF_TAG_SIZE                       12 // tiff tag
# define TIFF_DIRHEADTAIL_SIZE                6 // tiff directory head+tail

# define TIFF_BYTE_SIZE                       1
# define TIFF_ASCII_SIZE                      1
# define TIFF_SHORT_SIZE                      2
# define TIFF_LONG_SIZE                       4
# define TIFF_RATIONAL_SIZE                   (TIFF_LONG_SIZE+TIFF_LONG_SIZE)

typedef struct tiffhead {
  char               byteorder[2];
  unsigned short int version;
  unsigned long int  byteoffset;
  } TiffHead;

typedef struct tiffrational {
  unsigned long a;
  unsigned long b; // the value is a/b
  } TiffRational;

typedef union tiffelementdata {
  unsigned char      *bytedata;     // TIFF_DATATYPE_BYTE
  unsigned char      *asciidata;    // TIFF_DATATYPE_ASCII
  unsigned short int *shortdata;    // TIFF_DATATYPE_SHORT
  unsigned long int  *longdata;     // TIFF_DATATYPE_LONG
  TiffRational       *rationaldata; // TIFF_DATATYPE_RATIONAL
  void               *data;
  } TiffElementData;

typedef struct tiffelement {
  size_t             size; // external data size (in file)
  TiffElementData    u;
  } TiffElement;

typedef struct tifftag {
  unsigned short int tagtype;
  unsigned short int datatype;
  unsigned long  int numberofelements;
  TiffElement        element;
  unsigned long  int byteoffset; // offset to the data, null if stored in tag
  } TiffTag;

typedef struct tiffdir {
  unsigned short int numberoftags;
  unsigned long int  byteoffset;
} TiffDir;

/**************************************************************************/
/* internal variables                                                     */
/**************************************************************************/
enum IType { InValidIType,
             IUnsignedChar=1, IUnsignedShort, IUnsignedInteger, IUnsignedLong,
             NumberOfITypes };

const char * ITypeStrings[NumberOfITypes+1] =
           { "Invalid",
             "unsigned char",  "unsigned short", 
             "unsigned integer", "unsigned long",
             (const char *) NULL };

int TiffType[NUMBER_OF_TIFF_DATATYPES];
int TiffTypeInit = 0;    // not initialized

/*---------------------------------------------------------------------------
NAME
  itype2s ---  returns a itype as string

SYNOPSIS
  const char * itype2s ( int itype )

DESCRIPTION
  shows internal data types
---------------------------------------------------------------------------*/
const char * itype2s ( int itype )
{ return( ITypeStrings[itype] );
} /* itype2s */

/*---------------------------------------------------------------------------
NAME
   inittifftypes 

SYNOPSIS
   void inittifftypes ( int tifftype[NUMBER_OF_TIFF_DATATYPES] )

DESCRIPTION
  inits tiff type array
---------------------------------------------------------------------------*/
void inittifftypes ( void )
{ int i;
  int * tifftype = TiffType;

  for (i=0;i<NUMBER_OF_TIFF_DATATYPES;i++) tifftype[i] = InValidIType;
  
  // unsigned char 
  switch (sizeof( unsigned char ) ) {
     case 1 : tifftype[TIFF_DATATYPE_BYTE]  = IUnsignedChar; 
              tifftype[TIFF_DATATYPE_ASCII] = IUnsignedChar;
              break; 
     case 2 : tifftype[TIFF_DATATYPE_SHORT] = IUnsignedChar;
              break;
     }

  // unsigned short int 
  switch (sizeof( unsigned short int ) ) {
     case 1 : tifftype[TIFF_DATATYPE_BYTE]  = IUnsignedShort;
              tifftype[TIFF_DATATYPE_ASCII] = IUnsignedShort;
              break;
     case 2 : tifftype[TIFF_DATATYPE_SHORT] = IUnsignedShort; 
              break;
     case 4 : tifftype[TIFF_DATATYPE_LONG]  = IUnsignedShort;
              break;
     }

  // unsigned int  
  switch (sizeof( unsigned int ) ) {
     case 1 : tifftype[TIFF_DATATYPE_BYTE]  = IUnsignedInteger;
              tifftype[TIFF_DATATYPE_ASCII] = IUnsignedInteger;
              break;
     case 2 : tifftype[TIFF_DATATYPE_SHORT] = IUnsignedInteger; 
              break;
     case 4 : tifftype[TIFF_DATATYPE_LONG]  = IUnsignedInteger;
              break;
     }

  // unsigned long
  switch (sizeof( unsigned long ) ) {
     case 1 : tifftype[TIFF_DATATYPE_BYTE]  = IUnsignedLong;
              tifftype[TIFF_DATATYPE_ASCII] = IUnsignedLong;
              break;
     case 2 : tifftype[TIFF_DATATYPE_SHORT] = IUnsignedLong;
              break;
     case 4 : tifftype[TIFF_DATATYPE_LONG]  = IUnsignedLong;
              break;
     }

  TiffTypeInit = 1;

} /* inittifftype */

/*---------------------------------------------------------------------------
NAME
  ttype2i ---  returns the internal data type of a tiff type 

SYNOPSIS
  int ttype2i ( int tifftype )

---------------------------------------------------------------------------*/
int ttype2i ( int tifftype )
{ if (!TiffTypeInit) inittifftypes();
   return( TiffType[tifftype] );
} /* ttype2i */

/*---------------------------------------------------------------------------
NAME
   showtifftypes
 
SYNOPSIS
   void showtifftypes ( void )

DESCRIPTION
  shows tiff types and the corresponding internal data types
---------------------------------------------------------------------------*/
void showtifftypes ( void )
{ 
  printf(" TIFF_DATATYPE_BYTE  = %s\n",itype2s(ttype2i(TIFF_DATATYPE_BYTE)));
  printf(" TIFF_DATATYPE_ASCII = %s\n",itype2s(ttype2i(TIFF_DATATYPE_ASCII)));
  printf(" TIFF_DATATYPE_SHORT = %s\n",itype2s(ttype2i(TIFF_DATATYPE_SHORT)));
  printf(" TIFF_DATATYPE_LONG  = %s\n",itype2s(ttype2i(TIFF_DATATYPE_LONG)));

} /* showtifftypes */

/*--------------------------------------------------------------------------
NAME

   pad_file --- pads output file with spaces 

SYNOPSIS

  int pad_file ( FILE * output, size_t padlen )

DESCRIPTION
Writes exactly padlen spaces to output

ARGUMENTS
FILE * output : output channel
size_t padlen : number of bytes to write

RETURN VALUE
  int status : TIFF_STATUS_SUCCESS or TIFF_STATUS_ERROR 

---------------------------------------------------------------------------*/
int pad_file ( FILE * output, size_t padlen )
{ size_t i_pad = (size_t) 0;
  while (i_pad<padlen) {
    if ( fputc ( ' ' , output ) < 0 ) return(TIFF_STATUS_ERROR);
    i_pad++;
    } /* while */
  return(TIFF_STATUS_SUCCESS);
} /* pad_file */

size_t tiff_sizeof ( short tiff_datatype )
{ // external data sizes (in file)
  size_t size;

  switch (tiff_datatype) { 
    case TIFF_DATATYPE_BYTE    :
           size = (size_t) TIFF_BYTE_SIZE; break;
    case TIFF_DATATYPE_ASCII   :
           size = (size_t) TIFF_BYTE_SIZE; break;
    case TIFF_DATATYPE_SHORT   :
           size = (size_t) TIFF_SHORT_SIZE; break;
    case TIFF_DATATYPE_LONG    :
           size = (size_t) TIFF_LONG_SIZE; break;
    case TIFF_DATATYPE_RATIONAL:
           size = (size_t) (TIFF_LONG_SIZE+TIFF_LONG_SIZE); break;
    default:
           size = (size_t) NULL; break;
    }
 
  return( size );

} /* tiff_sizeof */

/*--------------------------------------------------------------------------
NAME

   new_tiffdata --- allocate tiff data buffer

SYNOPSIS

   void * new_tiffdata ( unsigned short int datatype,
                         unsigned long int  numberofelements,
                         TiffElement *      tiffelement )

DESCRIPTION
Memory allocation of numberofelements datatype elements. The pointer to 
the allocated buffer is returned as void * value and is written to 
tiffelement.u.bytedata, .asciidata, .shortdata, .longdata or .rationaldata, 
according to dataype. The actually allocated memory size in bytes is 
written to tiffelement.size. The minimum allocated size is 4 bytes.

ARGUMENTS
unsigned short int datatype        : TIFF_DATATYPE_BYTE, _ASCII, _SHORT etc. 
unsigned long int numberofelements : number of elements to allocate
TiffElement * tiffelement          : structure containing size and data buffer 

RETURN VALUE
 void *                            : pointer to the allocated data buffer,
                                     NULL if nothing was allocated

SEE ALSO
new_tiffdata, free_tiffdata
new_tiffelement, free_tiffelement
---------------------------------------------------------------------------*/
void * new_tiffdata ( unsigned short int datatype,
                      unsigned long int  numberofelements,
                      TiffElement *      tiffelement ) 
{ size_t size;

  switch (datatype) { // internal data size
    case TIFF_DATATYPE_BYTE    : 
           size = sizeof(unsigned char); break;
    case TIFF_DATATYPE_ASCII   :
           size = sizeof(unsigned char); break;
    case TIFF_DATATYPE_SHORT   :
           size = sizeof(unsigned short int); break;
    case TIFF_DATATYPE_LONG    :
           size = sizeof(unsigned long int); break;
    case TIFF_DATATYPE_RATIONAL:
           size = sizeof(TiffRational); break;
    default: 
           size = (size_t) NULL; break;
    }
  size = size * numberofelements;
  size = (size<TIFF_DATATYPE_LONG)?TIFF_DATATYPE_LONG:size;

  tiffelement->u.data = (void *) malloc( size );
  // external data size
  if (tiffelement->u.data) {
    tiffelement->size = tiff_sizeof(datatype)*numberofelements;
    memset(tiffelement->u.data,(unsigned char) 0,size); // clear memory
   } else tiffelement->size = (size_t) NULL;

  return( tiffelement->u.data );

} /* new_tiffdata */

/*--------------------------------------------------------------------------
NAME

   new_tiffelement --- allocate tiff data buffer

SYNOPSIS

   void * new_tiffelement( TiffTag * ptag );

DESCRIPTION
Allocates a tiff data element buffer in *ptag with new_tiffdata according 
to ptag->datatype and ptag->numberofelements with new_tiffdata. The 
minimum allocated size is 4 bytes.

ARGUMENTS
TiffTag * ptag : pointer to tag

RETURN VALUE
 void *                            : pointer to the allocated data buffer,
                                     NULL if nothing was allocated

SEE ALSO
new_tiffdata, free_tiffdata
new_tiffelement, free_tiffelement
---------------------------------------------------------------------------*/
void * new_tiffelement( TiffTag * ptag )
{ return( new_tiffdata ( ptag->datatype, ptag->numberofelements,
                         &(ptag->element) ) ); 
} /* new_tiffelement */

/*--------------------------------------------------------------------------
NAME

   free_tiffdata --- release previously allocated tiff data buffer 

SYNOPSIS

   void free_tiffdata ( TiffElement * tiffelement )

DESCRIPTION
Releases the data buffer in tiffelement. In case of success, the pointer
tiffelement.u.data is set to NULL and tiffelement.size is set to NULL.

ARGUMENTS
TiffElement * tiffelement          : structure containing size and data buffer

SEE ALSO
new_tiffdata, free_tiffdata
new_tiffelement, free_tiffelement
---------------------------------------------------------------------------*/
void free_tiffdata ( TiffElement * tiffelement ) {
  if (tiffelement) {
    if (tiffelement->u.data) free(tiffelement->u.data);
    tiffelement->u.data = (void *) NULL;
    tiffelement->size = (size_t) NULL;
    }
} /* free_tiffdata */

/*--------------------------------------------------------------------------
NAME

   free_tiffelement --- release previously allocated tiff element in *ptag

SYNOPSIS

   void free_tiffelement ( TiffTag * ptag )

DESCRIPTION
Releases the data buffer in ptag->element. In case of success, the pointer
(ptag->element).data is set to NULL and (ptag->element).size is set to NULL.

ARGUMENTS
TiffTag * ptag : pointer to tag

SEE ALSO
new_tiffdata, free_tiffdata
new_tiffelement, free_tiffelement
---------------------------------------------------------------------------*/
void free_tiffelement ( TiffTag * ptag ) {
  if (ptag) free_tiffdata ( &(ptag->element) );
} /* free_tiffdata */

void init_tags( TiffTag tag[], long taglen )
{ long int i;
  for (i=0;i<taglen;i++) {
    tag[i].tagtype = (unsigned short int) 0;
    tag[i].datatype = (unsigned short int) 0;
    tag[i].numberofelements = (unsigned long int) 0;
    tag[i].element.size = (size_t) 0;
    tag[i].element.u.data = (void *) 0;
    tag[i].byteoffset = (unsigned long int) 0;
    }
} /* init_tags */

void free_tags( TiffTag tag[], long taglen )
{ long int i;
  for (i=0;i<taglen;i++) { free_tiffelement ( &(tag[i]) ); }
} /* free_tags */

/*--------------------------------------------------------------------------
NAME

   tiff_sizecheck --- check the length of internal data types 

SYNOPSIS

   int tiff_sizecheck( void )

DESCRIPTION
Checks, whether the internal data type sizes are compatible to the used 
sizes. The messages are written to the error stream.

ARRGUMENTS
void

RETURN VALUE
int  0: OK
    -1: difference found
--------------------------------------------------------------------------*/
int tiff_sizecheck( void )
{ char buffer[TIFF_BUFLEN];
  int status = 0;

  if (sizeof(char) != TIFF_BYTE_SIZE) {
//    sprintf(buffer,"sizeof(char) = %zu != 1\n",sizeof(char));
    sprintf(buffer,"sizeof(char) = %zu|%lu != 1\n",
      sizeof(char),sizeof(char));
    perror(buffer); status = -1;
    }

  if (sizeof(short) != TIFF_SHORT_SIZE) {
//    sprintf(buffer,"sizeof(short) = %zu != 2\n",sizeof(short));
    sprintf(buffer,"sizeof(short) = %zu|%lu != 2\n",
      sizeof(short), sizeof(short));
    perror(buffer); status = -1;
    }

  if (sizeof(long) != TIFF_LONG_SIZE) {
//    sprintf(buffer,"sizeof(long) = %zu != 4\n",sizeof(long));
    sprintf(buffer,"sizeof(long) = %zu|%lu != 4\n",
      sizeof(long), sizeof(long));
    perror(buffer); status = -1; 
    }

  return( status );

} /* tiff_sizecheck */

int wtd_byte( FILE * output, unsigned long value )
{ unsigned char vuchar; 
  unsigned short vushort;
  unsigned int vuint; 
  unsigned long vulong;
  void * pointer;
  int written;

  switch ( ttype2i(TIFF_DATATYPE_BYTE) ) {
    case IUnsignedChar     :  
       vuchar = (unsigned char) value; pointer = &vuchar; break;
    case IUnsignedShort    :
       vushort = (unsigned short) value; pointer = &vushort; break;
    case IUnsignedInteger  :
       vuint = (unsigned int) value; pointer = &vuint; break;
    case IUnsignedLong     : 
       vulong = (unsigned long) value; pointer = &vulong; break;
    default          : return(-1);
    }
  if ( (written = fwrite ( pointer, TIFF_BYTE_SIZE,  1, output )) != 1 ) {
     return(-1); }

  return(0);
} /* wtd_byte */

int wtd_ascii( FILE * output, unsigned long value )
{ return( wtd_byte( output, value ) );
} /* wtd_ascii */

int wtd_short( FILE * output, unsigned long value )
{ unsigned char vuchar;
  unsigned short vushort;
  unsigned int vuint;
  unsigned long vulong;
  void * pointer;
  int written;

  switch ( ttype2i(TIFF_DATATYPE_SHORT) ) {
    case IUnsignedChar     :
       vuchar = (unsigned char) value; pointer = &vuchar; break;
    case IUnsignedShort    :
       vushort = (unsigned short) value; pointer = &vushort; break;
    case IUnsignedInteger  :
       vuint = (unsigned int) value; pointer = &vuint; break;
    case IUnsignedLong     :
       vulong = (unsigned long) value; pointer = &vulong; break;
    default          : return(-1);
    }
  if ( (written = fwrite ( pointer, TIFF_SHORT_SIZE,  1, output )) != 1 ) {
     return(-1); }
  return(0);
} /* wtd_short */

int wtd_long( FILE * output, unsigned long value )
{ unsigned char vuchar;
  unsigned short vushort;
  unsigned int vuint;
  unsigned long vulong;
  void * pointer;
  int written;

  switch ( ttype2i(TIFF_DATATYPE_LONG) ) {
    case IUnsignedChar     :
       vuchar = (unsigned char) value; pointer = &vuchar; break;
    case IUnsignedShort    :
       vushort = (unsigned short) value; pointer = &vushort; break;
    case IUnsignedInteger  :
       vuint = (unsigned int) value; pointer = &vuint; break;
    case IUnsignedLong     :
       vulong = (unsigned long) value; pointer = &vulong; break;
    default          : return(-1);
    }
  if ( (written = fwrite ( pointer, TIFF_LONG_SIZE,  1, output )) != 1 ) {
     return(-1); }
  return(0);
} /* wtd_long */

/*--------------------------------------------------------------------------
NAME

   write_tiff_header --- write tiff header to output file

SYNOPSIS

   int write_tiff_header ( FILE * output, TiffHead header, *pstatus )

DESCRIPTION

ARRGUMENTS
 FILE * output     : output stream
 TiffHead * pheader : pointer to header

RETURN VALUE
int status  0: success 
           -1: failed 
SEE ALSO
--------------------------------------------------------------------------*/
int write_tiff_header ( FILE * output, TiffHead * pheader )
{ int status = 0;

  // tiff header
  if ( (status = wtd_byte( output, (pheader->byteorder)[0] )) ) {
       return( status ); }
  if ( (status = wtd_byte( output, (pheader->byteorder)[1] )) ) {
       return( status ); }
  if ( (status = wtd_short( output, pheader->version )) ) {
       return( status ); }
  if ( (status = wtd_long( output, pheader->byteoffset )) ) {
       return( status ); }

  return( status );

} /* write_tiff_header */

int write_tiff_numberoftags ( FILE * output, TiffDir * pdirectory )
{ int status = 0; 

  // tiff directory.numberoftags
  if ( wtd_short( output, pdirectory->numberoftags ) ) {
    status = -1; return( status ); }

  return( status );

} /* write_tiff_numberoftags */

int write_tiff_byteoffset ( FILE * output, TiffDir * pdirectory )
{ int status = 0;

  // tiff directory.byteoffset
  if ( wtd_long( output, pdirectory->byteoffset ) ) {
    status = -1; return( status ); }

  return( status );

} /* write_tiff_byteoffset */

int write_tiff_tags ( FILE * output, TiffTag tag[], long taglen )
{ int status = 0;
  long i;

  // tiff tags 
  for (i=0; i<taglen; i++) {
    if ( wtd_short( output, tag[i].tagtype ) ) {
      status = -1; return( status ); }
    if ( wtd_short( output, tag[i].datatype ) ) {
      status = -1; return( status ); }
    if ( wtd_long( output, tag[i].numberofelements ) ) {
      status = -1; return( status ); }

    if ( tag[i].byteoffset ) {
      // write exactly 4 bytes
      if ( wtd_long( output, tag[i].byteoffset ) ) {
        status = -1; return( status ); }
      } else {
      // write exactly 4 bytes
      switch ( tag[i].datatype ) {
        case TIFF_DATATYPE_BYTE :
          if ( (status = wtd_byte( output, (tag[i].element.u.bytedata)[0] )) ) {
            return( status ); }
          if ( (status = wtd_byte( output, (tag[i].element.u.bytedata)[1] )) ) {
            return( status ); }
          if ( (status = wtd_byte( output, (tag[i].element.u.bytedata)[2] )) ) {
            return( status ); }
          if ( (status = wtd_byte( output, (tag[i].element.u.bytedata)[3] )) ) {
            return( status ); }
          break; 
        case TIFF_DATATYPE_ASCII :
          if ( (status = wtd_ascii( output, (tag[i].element.u.asciidata)[0] )) ) {
            return( status ); }
          if ( (status = wtd_ascii( output, (tag[i].element.u.asciidata)[1] )) ) {
            return( status ); }
          if ( (status = wtd_ascii( output, (tag[i].element.u.asciidata)[2] )) ) {
            return( status ); }
          if ( (status = wtd_ascii( output, (tag[i].element.u.asciidata)[3] )) ) {
            return( status ); }
          break;
        case TIFF_DATATYPE_SHORT :
          if ( (status = wtd_short( output, (tag[i].element.u.shortdata)[0] )) ) {
            return( status ); }
          if ( (status = wtd_short( output, (tag[i].element.u.shortdata)[1] )) ) {
            return( status ); }
          break; 
        case TIFF_DATATYPE_LONG :
          if ( (status = wtd_long( output, (tag[i].element.u.longdata)[0] )) ) {
            return( status ); }
          break; 
        default : status = -1; return( status ); 
          break;
        } // switch
      } // if ( tag[i].byteoffset )
    } // for

  return( status );

} /* write_tiff_tags */

/*--------------------------------------------------------------------------
NAME

   write_tiff_offsetdata --- write offset data to output file 

SYNOPSIS

   int write_tiff_offsetdata ( FILE * output, TiffTag tag[], long taglen )

DESCRIPTION
Writes the element data of all data tags with byteoffset > 0 to the file. 

ARRGUMENTS
 FILE * output     : output stream
 Tifftag tag[]     : tag array with taglen elements
 long taglen       : number of elements

RETURN VALUE
int status  0: success
           -1: failed
SEE ALSO
--------------------------------------------------------------------------*/
int write_tiff_offsetdata ( FILE * output, TiffTag tag[], long taglen )
{ 
  int status = -1;
  long i;
  unsigned long int j ;

  TiffRational * rationaldata;
  unsigned char * bval;
  unsigned char * aval;
  unsigned short * sval;
  unsigned long * lval;
  unsigned long int numberofelements ;

  for (i=0;i<taglen;i++)
    if ( tag[i].byteoffset ) { // write offset data
      numberofelements = tag[i].numberofelements;
      switch ( tag[i].datatype ) {
        case TIFF_DATATYPE_RATIONAL : 
          rationaldata = tag[i].element.u.rationaldata;
          for (j=0;j<numberofelements;j++) {
            if ( (status = wtd_long( output, rationaldata[j].a )) ) 
              return( status ); 
            if ( (status = wtd_long( output, rationaldata[j].b )) )
              return( status ); 
            }
          break;
        case TIFF_DATATYPE_BYTE :
          bval = tag[i].element.u.data;
          for (j=0;j<numberofelements;j++) {
            if ( (status = wtd_byte( output, bval[j] )) )
              return( status );
            }
          break;
        case TIFF_DATATYPE_ASCII :
          aval = tag[i].element.u.data;
          for (j=0;j<numberofelements;j++) {
            if ( (status = wtd_ascii( output, aval[j] )) )
              return( status );
            }
          break;
        case TIFF_DATATYPE_SHORT : 
          sval = tag[i].element.u.data;
          for (j=0;j<numberofelements;j++) {
            if ( (status = wtd_short( output, sval[j] )) )
              return( status );
            }
          break;
        case TIFF_DATATYPE_LONG : 
          lval = tag[i].element.u.data;
          for (j=0;j<numberofelements;j++) {
            if ( (status = wtd_long( output, lval[j] )) )
              return( status );
            }
          break;
 
        default            : 
          status = -1;
          return( status );
          break;
        }

    } /* if ( tag[i].byteoffset ) */

  status = 0;
  return(status);

} /* write_tiff_offsetdata */

/*--------------------------------------------------------------------------
NAME

   calculate_byteoffsets --- calculate byteoffsets of element data 

SYNOPSIS

   unsigned long int calculate_byteoffsets( TiffTag tag[], long taglen,
                                            unsigned long int byteoffset )


DESCRIPTION
Calculates the byteoffsets of tag element data. The data of all element 
with longer size than TIFF_LONG_SIZE bytes (4 bytes) are written at an 
offset. byteoffset is incremented and returned.

ARRGUMENTS
TiffTag tag[]                : array of tags
long taglen                  : number of tags
unsigned long int byteoffset : byteoffset for first tag data

RETURN VALUE
unsigned long int : incremented byteoffset
--------------------------------------------------------------------------*/
unsigned long int calculate_byteoffsets( TiffTag tag[], long taglen, 
                                         unsigned long int byteoffset )
{ long i;

  for (i=0;i<taglen;i++) { 
    if ( (tag[i].element.size) > TIFF_LONG_SIZE ) {
      // write offset data
      tag[i].byteoffset = byteoffset ;
      byteoffset += (unsigned long int) tag[i].element.size;
      } else tag[i].byteoffset = (unsigned long int) NULL; 
    } // for
  return( byteoffset );

} /* calculate_byteoffsets */

/*---------------------------------------------------------------------------
NAME

   float2unsigned --- transform a float image to an unsigned integer image

SYNOPSIS

   void  * float2unsigned (float * src, float dummy, float ddummy,
                           long xdim, long ydim, int autorange, 
                           int automin, float vmin, int automax, float vmax,
                           short bps, char comment[], int * pstatus);

DESCRIPTION
Transforms the float image src with dimension xdim, ydim to the
output image data with the same dimension. Each pixel in the output
images occupies bps bits. The value range [min..max] in the input image
is mapped to [0..2^bps-1] in the output image. Only the following values
of bps are supported:
  bps = 8  (unsigned byte),
  bps = 16 (unsigned short),
  bps = 32 (unsigned long) or (unsigned int) depending on machine
A values outside the mapping range is replaced by 0, if it is smaller than 0
or to 2^bps-1, if it is larger than 2^bps-1. The required memory for the
output image is allocated. The pointer to this array is returned. It must
be released.

ARGUMENTS
  float * src         input float image
  long dim_1, dim_2   dimension 1 and 2 of the input image
  float dummy, ddummy dummy and ddummy values in the input image
  int automin         0: vmin is used, 1: vmin is calculated from minimum
                         in the input image
  float vmin          minimum value of the mapping range in the input image
  int automax         0: vmax is used, 1: vmax is calculated from maximum 
                         in the input image
  float vmax          maximum value of the mapping range in the input image
  short bps           bits per sample in the output image
  char comment[]      output buffer with length InputLineLength.
                      It describes the mapping range
  int * pstatus       returned status (0: OK, -1: error, array is released)

RETURN VALUE
  void *              pointer to output image or NULL in case of an error

HISTORY
  1999-10-31          Peter Boesecke
---------------------------------------------------------------------------*/
void  * float2unsigned (float * src, float dummy, float ddummy,
                        long xdim, long ydim, int autorange,
                        int automin, float vmin, int automax, float vmax,
                        short bps, char comment[], int * pstatus)
{ char buffer[TIFF_BUFLEN];
  const float eps = FLT_MIN;

  char *   funcname = "float2unsigned";
  size_t   item, datalen;
  double   range0, range1;
  void *   data = (void *) NULL;
  float    minimum, maximum;
  double   rangefac = (double) 0.0;
  float *  psrc;
  float    Value;
  unsigned char  *  pbyte;
  unsigned short *  pshort;
  unsigned int   *  pint;
  unsigned long  *  plong;
  unsigned long dummyout;
  long     i;
  int      itype;

  if ((xdim<1) || (ydim<1)) {
    sprintf(buffer,"function %s : xdim=%ld, ydim=%ld out of bounds",
                    funcname, xdim, ydim);
              perror(buffer); *pstatus = -1; return(data);
    }

  // search for minimum and maximum
  if (automin || automax) {
    // find first non-dummy
    i=0; psrc = src;
    while ( (DUMMY(*psrc, dummy, ddummy)) && (i<xdim*ydim-1) ) {
      i++; psrc++; }
    i++; minimum=*psrc++; maximum=minimum; 
    // find minimum and maximum
    for (i=i;i<xdim*ydim;i++) {
      Value = *psrc++;
      if ( !(DUMMY(Value, dummy, ddummy)) ) {
        minimum = MIN2(Value, minimum); maximum = MAX2(Value, maximum);
        }
      }
    if (automin) vmin = minimum;
    if (automax) vmax = maximum;
    } /* automin || automax */

  // calculate transformation factor
  range0   = 0.0;
  range1   = pow((double) 2.0,(double) bps) - 1.0;

  // set dummyout to range1 if autorange is set
  if (autorange) dummyout = range1--;
    else dummyout = (unsigned long) MAX2(range0,MIN2(range1, dummy));

  if ((fabs(vmax-vmin)) > eps)
    rangefac = (range1-range0+1.0)/(vmax-vmin);
   else rangefac = 0.0;

  sprintf(buffer,"function %s : bps = %d", funcname, bps);
  switch (bps) {
    case  8 : item = TIFF_BYTE_SIZE; itype = ttype2i(TIFF_DATATYPE_BYTE); break;
    case 16 : item = TIFF_SHORT_SIZE; itype = ttype2i(TIFF_DATATYPE_SHORT); break;
    case 32 : item = TIFF_LONG_SIZE; itype = ttype2i(TIFF_DATATYPE_LONG); break;
    default : sprintf(buffer,"function %s : unsupported bps = %d",
                      funcname, bps);
              perror(buffer); *pstatus = -1; return(data);
    }

  datalen = item*xdim*ydim;
  data = (void *) malloc( datalen );
  if (!data) {
//    sprintf(buffer,"function %s : data = malloc( %zu ) failed",funcname,datalen);
    sprintf(buffer,"function %s : data = malloc( %zu | %lu ) failed",
      funcname,datalen,datalen);
    perror(buffer); *pstatus = -1; return(data);
    }

  psrc = src;
  switch (itype) {
    case  IUnsignedChar : 
              pbyte = (unsigned char *) data;
              for (i=0;i<xdim*ydim;i++) {
                Value = *psrc++;
                if (DUMMY(Value, dummy, ddummy)) 
                  *pbyte++ = (unsigned char) dummyout;
                 else {
                  Value = range0 + (Value-vmin)*rangefac + 0.5;
                  *pbyte++ = (unsigned char)
                             ( MAX2(range0,MIN2(range1, Value)) );
                  }
                } break;
    case IUnsignedShort :
              pshort = (unsigned short *) data;
              for (i=0;i<xdim*ydim;i++) {
                Value = *psrc++;
                if (DUMMY(Value, dummy, ddummy))
                  *pshort++ = (unsigned short) dummyout;
                 else {
                  Value = range0 + (Value-vmin)*rangefac + 0.5;
                  *pshort++ = (unsigned short)
                              ( MAX2(range0,MIN2(range1, Value)) );
                  }
                } break;
    case IUnsignedInteger :
              pint = (unsigned int *) data;
              for (i=0;i<xdim*ydim;i++) {
                Value = *psrc++;
                if (DUMMY(Value, dummy, ddummy))
                  *pint++ = (unsigned int) dummyout;
                 else {
                  Value = range0 + (Value-vmin)*rangefac + 0.5;
                  *pint++ = (unsigned int)
                              ( MAX2(range0,MIN2(range1, Value)) );
                  }
                } break;
    case IUnsignedLong : 
              plong = (unsigned long *) data;
              for (i=0;i<xdim*ydim;i++) {
                Value = *psrc++;
                if (DUMMY(Value, dummy, ddummy)) 
                  *plong++ = dummyout;
                 else {
                  Value = range0 + (Value-vmin)*rangefac + 0.5;
                  *plong++ = (unsigned long)
                             ( MAX2(range0,MIN2(range1, Value)) );
                  }
                } break;
    }

  if (DUMMYDEFINED(dummy,ddummy)) {
    sprintf(comment,
  "Intensity range [%g..%g] mapped to [%0.0f..%0.0f]\r\nDummy values set to %lu",
    vmin,vmax,range0,range1,dummyout);
    } else {
    sprintf(comment,
  "Intensity range [%g..%g] mapped to [%0.0f..%0.0f]",
    vmin,vmax,range0,range1);
    } /* DUMMYDEFINED */

  *pstatus = 0;
  return(data);

} /* float2unsigned */
 
/*--------------------------------------------------------------------------
NAME

   write_tiff --- write image data in TIFF format

SYNOPSIS

   int write_tiff ( const char * fname, long xdim, long ydim, short bps,
                    short compression, const void * data, int orientation,
                    const char * description, float xdpcm, float ydpcm,
                    int * pstatus )


DESCRIPTION 
Write data in TIFF format. The data is either an unsigned char array (bps=8),
an unsigned short int array (bps=16) or an unsigned long int array (bps=32).  

ARGUMENTS
  const char * fname       output file name
  long xdim, ydim          horizontal and vertical dimension of the image
  short bps                bitspersample = size of 1 element in bits,
                           the elements are packed as dense as possible:
                            8 :   unsigned char 
                           16 :   unsigned short
                           32 :   unsigned long integer
  short compression        compression type of the supplied data 
                           (only TIFF_NOCOMPRESSION)
  const void * data        pointer to the data array, horizontal elements 
                           must follow each other
  int orientation          1: (x,y),
                           2: (-x,y)  x-axis inversed,
                           3: (x,-y)  y-axis inversed,
                           4: (-x,-y) x- and y-axis inversed
                           5: (y,x)   axes swapped,
                           6: (y,-x)  axes swapped and x-axis inversed,
                           7: (-y,x)  axes swapped and y-axis inversed,
                           8: (-y,-x) axes swapped and both axes inversed
  const char * description character string describing the image, e.g title
                           if description==NULL, description is not written
  float xdpcm, ydpcm       horizontal and vertical number of pixels per cm
                           (will be converted to a closeby rational number)
  int * pstatus            return status

RETURN VALUE
  int status               same as *pstatus

---------------------------------------------------------------------------*/
int write_tiff ( const char * fname, long xdim, long ydim, short bps,
                 short compression, const void * data, int orientation,
                 const char * description, float xdpcm, float ydpcm,
                 int * pstatus )
{ const char * funcname = "write_tiff";
  FILE *output;
  char buffer1[TIFF_BUFLEN], buffer[TIFF_BUFLEN];
  long  ntag;
  unsigned long int istripe;
  long  stripbytecounts_tag, stripoffset_tag; 
  unsigned long int byteoffset;

  char     byteorderchar; 
  TiffTag  tag[TIFF_MAXTAGS];
//  TiffRational *pxdpcm, *pydpcm; // unused
  long     taglen;
  size_t   padding1, padding2;
  size_t   item, datalen;
  long     data_dim[4];
  void *   out = (void *) NULL;
  const void * dataout;  
  float    ftmp;
  long     ltmp;

  TiffHead header;
  TiffDir  directory;

  // check internal sizes
/* no longer necessary
  if ( tiff_sizecheck() ) {
    sprintf(buffer,"function %s : size mismatch", funcname);
    perror(buffer); *pstatus=TIFF_STATUS_SIZEMISMATCH; return( *pstatus );
    }
*/
  // stop, if fname==NULL
  if ( fname==(char *) NULL ) {
    sprintf(buffer,"function %s : output file name missing", funcname);
    perror(buffer); *pstatus=TIFF_STATUS_ERROR; return( *pstatus );
    }

  // stop, if data==NULL
  if ( data==(void *) NULL ) {
    sprintf(buffer,"function %s : NULL pointer to source data", funcname);
    perror(buffer); *pstatus=TIFF_STATUS_ERROR; return( *pstatus );
    }

  // get byte order 
  switch (edf_byteorder()) {
    case LowByteFirst : 
                byteorderchar = 'I';
                break;
    case HighByteFirst : 
                byteorderchar = 'M';
                break;
    default : sprintf(buffer,"function %s : unsupported byte order %d",
                      funcname, edf_byteorder());
              perror(buffer); *pstatus=TIFF_STATUS_ERROR; return( *pstatus );
    }

  // reorder image
  item    = (size_t) CEILMOD(bps,8)/8;
  datalen = (size_t) CEILMOD(xdim*ydim*bps,8)/8; // up to next full byte
  if (orientation==1) {
    dataout = data;
    } else {
    out = (void *) malloc( datalen );
    if (!out) {
//     sprintf(buffer,"function %s : out = malloc( %zu ) failed",funcname,datalen);
     sprintf(buffer,"function %s : out = malloc( %zu | %lu ) failed",
       funcname,datalen,datalen);
     perror(buffer); *pstatus = -1; return( *pstatus );
     }

    if (orientation>4) {
      // exchange all x- and y-parameters
      ltmp = ydim; ydim = xdim; xdim = ltmp;
      ftmp = ydpcm; ydpcm = xdpcm; xdpcm = ftmp;
      }

    data_dim[0] = 2; 
    data_dim[1] = xdim; data_dim[2] = ydim;
    data_dim[3] = datalen;

    if (edf_raster_normalization ( out, data, data_dim, orientation, item )) {
      sprintf(buffer,"function %s : edf_raster_normalization failed",funcname);
      perror(buffer); *pstatus = -1; return( *pstatus );
      }

    dataout = out;

  } /* if (orientation!=1) */

  // create tiff header
  header.byteorder[0] = byteorderchar;
  header.byteorder[1] = byteorderchar;
  header.version      = (unsigned short) 42;
  header.byteoffset   = TIFF_HEADER_SIZE;

  // create tiff tags
  init_tags( tag, TIFF_MAXTAGS );

  sprintf(buffer,"function %s : Increase TIFF_MAXTAGS %d",
          funcname, TIFF_MAXTAGS);
  sprintf(buffer1,"function %s : memory allocation error", funcname);

  ntag = 0; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_NewSubfileType;
  tag[ntag].datatype           = TIFF_DATATYPE_LONG;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) { 
    // full resolution image
    *(tag[ntag].element.u.longdata) = (unsigned long) 0;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_ImageWidth;
  tag[ntag].datatype           = TIFF_DATATYPE_LONG;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // horizontal dimension
    *(tag[ntag].element.u.longdata) = (unsigned long) xdim; 
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_ImageLength;
  tag[ntag].datatype           = TIFF_DATATYPE_LONG;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // vertical dimension
    *(tag[ntag].element.u.longdata) = (unsigned long) ydim;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_BitsPerSample;
  tag[ntag].datatype           = TIFF_DATATYPE_SHORT;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // bits per sample 
    *(tag[ntag].element.u.shortdata) = (unsigned short) bps;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_Compression;
  tag[ntag].datatype           = TIFF_DATATYPE_SHORT;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // compression 
    *(tag[ntag].element.u.shortdata) = (unsigned short) compression;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_PhotometricInterpretation;
  tag[ntag].datatype           = TIFF_DATATYPE_SHORT;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // grey_scale type 
    *(tag[ntag].element.u.shortdata) = (unsigned short) TIFF_GREYSCALE;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  if ( description!=(char *) NULL ) {
    if (strlen(description)>0) {
     if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
     tag[ntag].tagtype            = TIFF_TAG_ImageDescription;
     tag[ntag].datatype           = TIFF_DATATYPE_ASCII;
     tag[ntag].numberofelements=(unsigned long)CEILMOD(strlen(description)+1,2);
     if ( new_tiffelement( &(tag[ntag]) ) ) {
       // description 
       strncpy((char*) tag[ntag].element.u.asciidata,description,
               tag[ntag].element.size);
       } else {
       perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
       free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
       }
     ntag++; 
     }
    }

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_StripOffsets;
  tag[ntag].datatype           = TIFF_DATATYPE_LONG;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // strip offsets 
    (tag[ntag].element.u.longdata)[0] = (unsigned long) NULL; // strip 1
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  stripoffset_tag = ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_SamplesPerPixel;
  tag[ntag].datatype           = TIFF_DATATYPE_SHORT;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // 1 sample per pixel 
    *(tag[ntag].element.u.shortdata) = (unsigned short) 1;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++;

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_RowsPerStrip;
  tag[ntag].datatype           = TIFF_DATATYPE_LONG;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // all rows in 1 strip 
    *(tag[ntag].element.u.longdata) = (unsigned long) ydim;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype             = TIFF_TAG_StripByteCounts;
  tag[ntag].datatype            = TIFF_DATATYPE_LONG;
  tag[ntag].numberofelements    = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // bytes per strip
    *(tag[ntag].element.u.longdata) = (unsigned long)CEILMOD(xdim*ydim*bps,8)/8;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  stripbytecounts_tag = ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_XResolution;
  tag[ntag].datatype           = TIFF_DATATYPE_RATIONAL;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // x-resolution 
    (tag[ntag].element.u.rationaldata)->a = 
                (unsigned long) (xdpcm*TIFF_RESOLUTION_QUOTIENT);
    (tag[ntag].element.u.rationaldata)->b = 
                (unsigned long) TIFF_RESOLUTION_QUOTIENT;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_YResolution;
  tag[ntag].datatype           = TIFF_DATATYPE_RATIONAL;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // y-resolution
    (tag[ntag].element.u.rationaldata)->a =
                (unsigned long) (ydpcm*TIFF_RESOLUTION_QUOTIENT);
    (tag[ntag].element.u.rationaldata)->b =
                (unsigned long) TIFF_RESOLUTION_QUOTIENT;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  if (ntag>=TIFF_MAXTAGS) { perror(buffer); exit(-1); }
  tag[ntag].tagtype            = TIFF_TAG_ResolutionUnit;
  tag[ntag].datatype           = TIFF_DATATYPE_SHORT;
  tag[ntag].numberofelements   = (unsigned long) 1;
  if ( new_tiffelement( &(tag[ntag]) ) ) {
    // pixel size units 
    *(tag[ntag].element.u.shortdata) = (unsigned short) TIFF_DPCM;
    } else {
    perror(buffer1); *pstatus= TIFF_STATUS_ERROR;
    free_tags( tag, ntag+1 ); if (out) free(out); return(*pstatus);
    }
  ntag++; 

  taglen = ntag;

  directory.numberoftags       = (unsigned short) taglen;
  directory.byteoffset         = (unsigned long int) 0; // no other directory
 
  // calculate byteoffset after fixed length tags 
  byteoffset = (unsigned long int) ( TIFF_HEADER_SIZE + TIFF_DIRHEADTAIL_SIZE +
                                     TIFF_TAG_SIZE*taglen );

  byteoffset = calculate_byteoffsets( tag, taglen, byteoffset );

  // padding to multiple of TIFF_BLOCK_SIZE before start of strips
  padding1 = CEILMOD(byteoffset,TIFF_BLOCK_SIZE ) - byteoffset;
  byteoffset += padding1;

  // calculate strip offsets
  for (istripe=0;istripe<tag[stripoffset_tag].numberofelements;istripe++) {
    (tag[stripoffset_tag].element.u.longdata)[istripe] += byteoffset;
      byteoffset += *(tag[stripbytecounts_tag].element.u.longdata);
    }

  // padding to multiple of TIFF_BLOCK_SIZE after strips
  datalen = (size_t) CEILMOD(xdim*ydim*bps,8)/8; // up to next full byte
  padding2 = CEILMOD(datalen,TIFF_BLOCK_SIZE );

  // open binary output file
  output = fopen ( fname, "wb" );
  if (!output) { 
    sprintf(buffer,"function %s : open file %s", funcname, fname);
    perror(buffer); *pstatus=TIFF_STATUS_OPENERROR; 
    free_tags( tag, taglen ); if (out) free(out); return(*pstatus); 
    }

  // write tiff file
  // write tiff header
  if ( write_tiff_header ( output, &header ) ) {
    sprintf(buffer,"%s->write_tiff_header( ... )", funcname);
    perror(buffer); *pstatus= TIFF_STATUS_ERROR; 
    free_tags( tag, taglen ); if (out) free(out); return(*pstatus); 
    }
  // write tiff directory.numberoftags
  if ( write_tiff_numberoftags ( output, &directory ) ) { 
    sprintf(buffer,"%s->write_tiff_numberoftags( ... )", funcname);
    perror(buffer); *pstatus=TIFF_STATUS_ERROR; 
    free_tags( tag, taglen ); if (out) free(out); return(*pstatus); 
    }
  // write tiff tags
  if ( write_tiff_tags ( output, tag, taglen ) ) {
    sprintf(buffer,"%s->write_tiff_tags( ... )", funcname);
    perror(buffer); *pstatus=TIFF_STATUS_ERROR; 
    free_tags( tag, taglen ); if (out) free(out); return(*pstatus); 
    }
  // tiff directory.byteoffset
  if ( write_tiff_byteoffset ( output, &directory ) ) {
    sprintf(buffer,"%s->write_tiff_byteoffset( ... )", funcname);
    perror(buffer); *pstatus=TIFF_STATUS_ERROR; 
    free_tags( tag, taglen ); if (out) free(out); return(*pstatus); 
    }
  // write offset data 
  if ( write_tiff_offsetdata ( output, tag, taglen ) ) {
    sprintf(buffer,"%s->write_tiff_offsetdata( ... )", funcname);
    perror(buffer); *pstatus=TIFF_STATUS_ERROR; 
    free_tags( tag, taglen ); if (out) free(out); return(*pstatus); 
    }

  // pad to multiples of TIFF_BLOCK_SIZE 
  if ( pad_file ( output, padding1 ) ) {
//    sprintf(buffer,"%s->pad_file( output, %zu, ... )", funcname, padding1 );
    sprintf(buffer,"%s->pad_file( output, %zu|%lu, ... )", 
      funcname, padding1, padding1);
    perror(buffer); *pstatus=TIFF_STATUS_ERROR; 
    free_tags( tag, taglen ); if (out) free(out); return(*pstatus); 
    }

  // write image data 
  if ( datalen > 0)
    if ( fwrite( dataout, 1, datalen, output ) < datalen ) {
//    sprintf(buffer,"%s->fwrite( data, 1, %zu, ... )", funcname, datalen );
    sprintf(buffer,"%s->fwrite( data, 1, %zu|%lu, ... )", 
      funcname, datalen, datalen);
    perror(buffer); *pstatus=TIFF_STATUS_ERROR; 
    free_tags( tag, taglen ); if (out) free(out); return(*pstatus); 
    }

  // pad to multiples of TIFF_BLOCK_SIZE
  if ( pad_file ( output, padding2 ) ) {
//    sprintf(buffer,"%s->pad_file( output, %zu, ... )", funcname, padding2 );
    sprintf(buffer,"%s->pad_file( output, %zu|%lu, ... )",
      funcname, padding2, padding2);
    perror(buffer); *pstatus=TIFF_STATUS_ERROR; 
    free_tags( tag, taglen ); if (out) free(out); return(*pstatus); 
    }

  fclose( output );

  free_tags( tag, taglen );
  if (out) free(out);

  *pstatus=TIFF_STATUS_SUCCESS; 
  return(*pstatus);

} /* write_tiff */

