/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 SaxsInput.c
  General prompting routines for keyboard i/o.

2 DESCRIPTION
  General prompting routines for keyboard i/o.
  The routines prompt for integer, float and strings.
  A default value is proposed when 'flag' is 1.
  It can be accepted by typing <enter>.
  The output status of the routines is one of the following:
  Success, Abort, ScanError, NoDefault.
  They can be displayed with the routine 'ReportSaxsStatus'.
  The input from the keyboard can be aborted by typing
  $exit at the beginning of a line (no leading blanks) or by typing
  <ctrl-D> at any position in the line.
  An empty line accepts the default.
  An empty line can be read by 'GetString' and 'GetLine' only by typing
  $null at the beginning of a line.
  The output value of 'GetString' is a pointer to a char string.
  It must be declared as char ** value; .
  The output value of 'GetLine' is a char string with fixed length.
  It must be declared as char value[InputLineLength]; .

2 CALL
char *SaxsInputVersion ( void );
float floatexpr ( const char * s, int * pstatus);
double doubleexpr ( const char * s, int * pstatus);
long  longexpr  ( const char * s, int * pstatus);
int GetLong (File *Fptr, long int *Value, int Flag, char *Prompt, long DefVal);
int GetInteger (FILE *Fptr, int *Value, int Flag, char *Prompt, int DefVal);
int GetReal (FILE *Fptr, float *Value, int Flag, char *Prompt, float DefVal);
int GetDouble (FILE *Fptr, double *Value, int Flag, \
                                      char *Prompt, double DefVal):
int GetString (FILE *Fptr, char **Value, int Flag, char *Prompt, char *DefVal);
int GetLine (FILE *Fptr  , char  Value[InputLineLength], int Flag, \
						char *Prompt, char * DefVal);
void CopyLine ( const char * s1, char * sout, int len , int conversion );
char *RightFR ( char * s1, char * s2, char * sout, int len ),
char *LeftFR  ( char * s1, char * s2, char * sout, int len ),
char *RightFL ( char * s1, char * s2, char * sout, int len ),
char *LeftFL  ( char * s1, char * s2, char * sout, int len );

2	HISTORY
	26-Dez-1994 PB routines from P. Daly
	29-Dez-1994 PB
	01-Jan-1995 PB new CopyLine function (SCopyLine from Separation.c)
	10-Jan-1995 PB installed on RED, separation into input.c, input.h
	12-Jan-1995 PB 2.1     GetLong
        09-Sep-1995 PB 2.2     WildCard defined
        11-Sep-1995 PB 3.0     fpexpr() and lexpr()
        16-Sep-1995 PB         floatexpr(), longexpr()
        17-Sep-1995 PB 3.1     math. functions in floatexpr
        14-Nov-1995 PB 3.4     Terminator3, SkipInput
	23-Nov-1995 PB         EditLine and EditFlags specified in input.h
	24-Nov-1995 PB         ReportSaxsStatus
        22-Dec-1995 PB 3.41 a) error corrected in definitions of EditFlags
                               (all bits inverted)
                            b) SkipToken changed to three dots ..., to allow
                               the specifications of directories ..\abcd
        26-Dec-1995 PB 3.42 a) ReportSaxsError modified
                            b) test_image.h included for EsrfDataFormat.h
        26-Jan-1996 PB 3.51    FileReadError
        28-Jan-1996 PB 3.52    UnknownDataType
        29-Mar-1996 PB 3.53    FilePositioningError
        10-Oct-1996 PB 3.54    include separation.inc, expression.inc
                               remamed to SaxsInput.c and SaxsInput.h
        11-Oct-1996 PB 3.6     ReportSaxsStatus -> SaxsError V1.0
        14-Oct-1996 PB 3.7     SaxsInputVersion
        26-Jun-1999 PB 3.8     DevSeparator, DirSeparator, ExtSeparator in
                               SaxsInput.h defined
        17-Jul-1999 PB 4.0     InputLineLength increased from 128 to 256
        02-Aug-2000 PB 4.1     GetDouble added
        2000-08-04  PB 4.2     %-e -> %-g
        2000-12-04  PB 4.3     -> SaxsExpression.c, .h
        2001-09-14  PB 4.31    include edfio.h for EdfMaxLinLen
        2003-02-04  PB 4.32    SaxsExpression: constant inf
        2003-08-03  PB 4.33    SaxsExpression: XXXexpr with const char *
                               CopyLine: char *s1 ->  const char * s1
        2004-02-19  PB 4.34    SaxsExpression: new functions: min, max
        2004-03-02  PB 4.35    SaxsExpression: new underscore operator '_' 
        2004-03-16  PB 4.36    SaxsExpression: now only interface to numio
        2004-04-02  PB 4.37    SaxsExpression: changes for new numio version 
        2004-09-27  PB 4.38    GetString, GetLine:
                               Skip input if default value is SkipToken
                               SkipCheck: The full input string must now match
                                          SkipToken to skip input.
        2005-12-01  PB 4.39    (c == ' ') replaced by isspace(c) to be 
                               sensitive for all white spaces, comment character
                               changed to '#'
        2007-04-19  PB 4.40    -Wall compiler warnings resolved

---*/

/***************************************************************************/
/* SaxsInput.c                                                                 */
/***************************************************************************/
# define SI_Version "SaxsInput V4.40 Peter Boesecke 2007-04-19"

# include "SaxsError.h"
# include "SaxsExpression.h"
# include "SaxsInput.h"

#	define Terminator1 "$exi"
#	define Terminator2 "$eod"
#	define Terminator3 "\\\\"
#	define NullToken1  "$nul"
#	define NullToken2  "$nix"
#	define EOD         4
#	define BufLen      1024

#	ifdef CommentChar
#	  undef CommentChar
#	endif
//#	define CommentChar '\n'
#	define CommentChar '#'

char buffer[BufLen];

/*----------------------------------------------------------------------------
 Private include files
----------------------------------------------------------------------------*/

# include "SaxsSeparation.inc"

/****************************************************************************
   declaration of internal functions
****************************************************************************/

/****************************************************************************
1 SaxsInputVersion

2 DESCRIPTION
  Returns a pointer to the version string.
  char * SaxsInputVersion ( void )

2 HISTORY
  14-Oct-1996 PB
****************************************************************************/
char * SaxsInputVersion ( void )
{ static char * Version = SI_Version;
  return ( Version );
} /* SaxsInputVersion */

/****************************************************************************
*  CopyLine : copies not more than len-1 characters from string s1 to
*	      string sout. A string end sign is respected.
*	      String sout ends with '\0'.
*	      conversion = -1 : conversion to small letters
*	      conversion = 0  : no conversion
*	      conversion = 1  : converion to big letters
****************************************************************************/
void CopyLine( const char * s1, char * sout, int len , int conversion )
{
  register int i;

  if (s1)
  switch (conversion)
   {  	case -1 : for (i=0;(i<len-1) && (*s1);i++) *sout++ = tolower(*s1++);
			break;
  	case 1  : for (i=0;(i<len-1) && (*s1);i++) *sout++ = toupper(*s1++);
			break;
  	default : for (i=0;(i<len-1) && (*s1);i++) *sout++ = *s1++;
                        break;
   }
  *sout++ = '\0';  /* terminate string with NULL */
} /* CopyLine */

/****************************************************************************
*  AbortCheck : checks s1 on appearance of EOD and on the token Terminator  *
****************************************************************************/

int AbortCheck (char *s1)
{
  register int i;
  char buf[5];
  if (strchr (s1,EOD) != NULL) return (-1); CopyLine(s1,buf,5,0);
 /*check first four characters for special token*/
  for (i=0;buf[i];i++) buf[i]=tolower(buf[i]);
  if (strstr (buf, Terminator1) != NULL) return (-1);
  if (strstr (buf, Terminator2) != NULL) return (-1);
  if (strstr (buf, Terminator3) != NULL) return (-1);
  return(0);
} /* AbortCheck */

/****************************************************************************
*  SkipCheck : checks s1 on appearance of SkipToken                         *
****************************************************************************/
int SkipCheck (char *s1)
{ int slen;

  slen=strlen(s1); 
  /* do not include terminating control character */
  if (slen>1) 
    if ( s1[slen-1] < (char) 16 ) slen=slen-1;
  /* check the whole string for special token */
  if ( (strlen(SkipToken)==slen) && ( ! strncmp( SkipToken, s1, slen ) ) ) 
    return(-1);
  else return(0);

} /* SkipCheck */

/****************************************************************************
*  NullCheck : Checks s1 on appearance of NullToken                         *
****************************************************************************/
int NullCheck (char *s1)
{
register int i;
char buf[5];
CopyLine(s1,buf,5,0); /*check first four characters for special token*/
	for (i=0;buf[i];i++) buf[i]=tolower(buf[i]);
	if (strstr (buf, NullToken1) != NULL) return (-1);
	if (strstr (buf, NullToken2) != NULL) return (-1);
	return(0);
} /* NullCheck */

/****************************************************************************
*	 DefaultCheck : Checks s1 on default                                *
***********************************************************************o**/
int DefaultCheck (char *s1)
{
	if (*s1 == '\0') return(-1);
	return(0);
} /* DefaultCheck */

/****************************************************************************
*  EditLine                                                                 *
****************************************************************************/
void EditLine (int EditFlag, char *str)
{
register int i;
char * start = str, * line;

if (str != (char *) NULL) {

  for (i = strlen (str) - 1; i>= 0; i--)
      if (str[i] == '\t') str[i] = ' ';    /* convert all tabs to spaces */

  if (EditFlag & UnComment) { /* remove comments => makes line shorter */
      if ((line = strchr (str, CommentChar)) != (char *) NULL)
         *line = '\0';
	}
  line = str;

  if (EditFlag & Collapse) {
    EditFlag = EditFlag | Trim;   /* if Collapse also invoke Trim */
    while (*line != '\0') {
      while (isspace(*line)) line++;
      *str++ = *line++;   /* remove all white spaces */
      }
    *str = *line;
    } else {
    while (*line != '\0') {    /* remove multiple blanks */
      *str++ = *line++;
      if (isspace(*line)) {
         *str++ = ' '; line++;
	 while (isspace(*line)) line++;
	 }
    }
  *str = *line;
  }
line = start;

if (EditFlag & Trim) {
  while (isspace(*line)) line++;   /* remove leading white spaces */
  for (i = strlen (line) -1; i>=0 && isspace(line[i]); i--);
  line[++i] = '\0';        /* remove trailing blanks */
  }

if ((EditFlag & LowerCase) || (EditFlag & UpperCase))
  for (i = strlen (line) - 1; i>= 0; i--)
      line[i] = (EditFlag & LowerCase ? tolower (line[i])
	    : toupper (line[i]));
  str = start;
  while ((*str++ = *line++) != '\0'); /*copy the result*/
 }
} /* EditLine */

/****************************************************************************
*	 GetInteger
*	 FILE		*Fptr	input file
*	 int		*Value  output value
*	 int		Flag	0: no default, 1: use DefVal as default
*	 char		*Prompt prompt string
*	 int		DefVal  default value
****************************************************************************/
int GetInteger (FILE *Fptr, int *Value, int Flag, char *Prompt, int DefVal)
{
  int status;

  buffer[0] = '\0';
  (void) fprintf ( stdout, "%s ", Prompt);
  if (Flag) (void) fprintf (stdout, "[%-d] ", DefVal);
  (void) fprintf ( stdout, ": ");
  (void) fgets (buffer, BufLen, Fptr);
  if (AbortCheck(buffer)) return(Abort);
  if (SkipCheck(buffer)) return(SkipInput);
  EditLine ( Trim | UnComment, buffer);
  if (DefaultCheck(buffer)) {
    if (Flag) *Value = DefVal; else return(NoDefault);
    } else {
    *Value = (int) longexpr(buffer,&status);
    if (status!=Success) return (status);
    }
  return (Success);
} /* GetInteger */

/****************************************************************************
*        GetLong
*        FILE           *Fptr   input file
*        long           *Value  output value
*        int            Flag    0: no default, 1: use DefVal as default
*        char           *Prompt prompt string
*        int            DefVal  default value
****************************************************************************/
int GetLong (FILE *Fptr,long int *Value,int Flag,char *Prompt,long int DefVal)
{
  int status;

  buffer[0] = '\0';
  (void) fprintf ( stdout, "%s ", Prompt);
  if (Flag) (void) fprintf (stdout, "[%-ld] ", DefVal);
  (void) fprintf ( stdout, ": ");
  (void) fgets (buffer, BufLen, Fptr);
  if (AbortCheck(buffer)) return(Abort);
  if (SkipCheck(buffer)) return(SkipInput);
  EditLine ( Trim | UnComment, buffer);
  if (DefaultCheck(buffer)) {
    if (Flag) *Value = DefVal; else return(NoDefault);
    } else {
    *Value = longexpr(buffer,&status);
    if (status!=Success) return (status);
    }
return (Success);
} /* GetLong */

/****************************************************************************
*	 GetReal			                                        *
*	 FILE		*Fptr	input file
*	 float	*Value  output value
*	 int		Flag	0: no default, 1: use DefVal as default
*	 char		*Prompt prompt string
*	 float	DefVal  default value
****************************************************************************/
int GetReal (FILE *Fptr, float *Value, int Flag, char *Prompt, float DefVal)
{
  int status;

  buffer[0] = '\0';
  (void) fprintf ( stdout, "%s ", Prompt);
  if (Flag) (void) fprintf (stdout, "[%-g] ", DefVal);
  (void) fprintf ( stdout, ": ");
  (void) fgets (buffer, BufLen, Fptr);
  if (AbortCheck(buffer)) return(Abort);
  if (SkipCheck(buffer)) return(SkipInput);
  EditLine ( Trim | UnComment, buffer);
  if (DefaultCheck(buffer)) {
    if (Flag) *Value = DefVal; else return(NoDefault);
    } else {
    *Value = floatexpr(buffer,&status);
    if (status!=Success) return (status);
    }
return (Success);
} /* GetReal */

/****************************************************************************
*        GetDouble
*
*        FILE           *Fptr   input file
*        float  *Value  output value
*        int            Flag    0: no default, 1: use DefVal as default
*        char           *Prompt prompt string
*        float  DefVal  default value
****************************************************************************/
int GetDouble (FILE *Fptr, double *Value, int Flag, char *Prompt, double DefVal)
{
  int status;

  buffer[0] = '\0';
  (void) fprintf ( stdout, "%s ", Prompt);
  if (Flag) (void) fprintf (stdout, "[%-g] ", DefVal);
  (void) fprintf ( stdout, ": ");
  (void) fgets (buffer, BufLen, Fptr);
  if (AbortCheck(buffer)) return(Abort);
  if (SkipCheck(buffer)) return(SkipInput);
  EditLine ( Trim | UnComment, buffer);
  if (DefaultCheck(buffer)) {
    if (Flag) *Value = DefVal; else return(NoDefault);
    } else {
    *Value = doubleexpr(buffer,&status);
    if (status!=Success) return (status);
    }
return (Success);
} /* GetDouble */

/****************************************************************************
*	 GetString                                                          *
*	 FILE		*Fptr   input file
*	 char		**Value output string
*	 int		Flag    0: no default, 1: use DefVal as default
*	 char		*Prompt prompt string
*	 char		*DefVal default string
****************************************************************************/
int GetString (FILE *Fptr, char **Value, int Flag, char *Prompt, char *DefVal)
{
  buffer[0] = '\0';
  (void) fprintf ( stdout, "%s ", Prompt);
  if (Flag) (void) fprintf (stdout, "[%-s] ", DefVal);
  (void) fprintf ( stdout, ": ");
  (void) fgets (buffer, BufLen, Fptr);
  if (AbortCheck(buffer)) return(Abort);
  if (SkipCheck(buffer)) return(SkipInput);
  if (NullCheck(buffer)) {
    *Value = (char *) malloc (1);
    **Value = '\0';
    } else {
    EditLine (Trim | UnComment, buffer);
    if (DefaultCheck(buffer)) {
      if (Flag) {
        /* Skip input if default value is SkipToken */
        if (SkipCheck(DefVal)) return(SkipInput);
        *Value = (char *) malloc (1 + strlen (DefVal));
	(void) strcpy (*Value, DefVal);
        } else return(NoDefault);
      } else {
      *Value = (char *) malloc (1 + strlen (buffer));
      (void) strcpy (*Value, buffer);
      }
    }
  return (Success);
} /* GetString */

/****************************************************************************
*	 GetLine                                                            *
*	 FILE		*Fptr   input file
*	 char		*Value  output string
*	 int		Flag    0: no default, 1: use DefVal as default
*	 char		*Prompt prompt string
*	 char		*DefVal default string
****************************************************************************/
int GetLine (FILE *Fptr  , char  Value[InputLineLength], int Flag, \
             char *Prompt, char * DefVal)
{
  buffer[0] = '\0';
  (void) fprintf ( stdout, "%s ", Prompt);
  if (Flag) (void) fprintf (stdout, "[%-s] ", DefVal);
  (void) fprintf ( stdout, ": ");
  (void) fgets (buffer, BufLen, Fptr);
  if (AbortCheck(buffer)) return(Abort);
  if (SkipCheck(buffer)) return(SkipInput);
  if (NullCheck(buffer)) {
    *Value = '\0';
    } else {
    EditLine (Trim | UnComment, buffer);
    if (DefaultCheck(buffer)) {
      if (Flag) {
        /* Skip input if default value is SkipToken */
        if (SkipCheck(DefVal)) return(SkipInput);
        (void) CopyLine (DefVal, Value, InputLineLength, 0);
	} else return(NoDefault);
      } else {
      (void) CopyLine (buffer, Value, InputLineLength, 0);
      }
    }
  return (Success);
} /* GetLine */

/****************************************************************************/
