/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 SaxsImage.c

2 PURPOSE
Input and output routines for time-resolved small-angle x-ray scattering
data that are stored in the ESRF data format (esrf_data_format.h).
A special sub-structure of this data format is defined which is
suitable for two dimensional time resolved data. The data are
stored in a float array of the size Dim_1 * Dim_2 (columns, rows).

Image data headers are marked with the keyword "image" followed by a positive
number larger than 0.

2 KEYWORDS and UNITS
Only SI units are used (meter, kilogramm, second).
Application programs may use other units, but they have to store their
values in SI units. The axes 1, 2 and 3 of the images form a right-handed
orthogonal coordinate system.
Because the direction of the axes are hardware depending they are not fixed
in the real world. Generally axis 1 is the horizontal axis, axis 2 the
vertical axis and axis 3 points from the detector to the sample.

The following keywords are used by the routines (default in square brackets):

Image header

  ImNum         = <image number>
  Dim[0]        = dimensions (2)
  Dim[1]        = number of columns
  Dim[2]        = number of rows
  Dummy         = dummy value (0.0 = no dummy) [0.0]
  DDummy        = delta dummy (range around a dummy) [0.1]
  Offset[1]     = column offset of all pixels [0.0]
  Offset[2]     = row offset of all pixels [0.0]
  Center[1]     = column center of the image
  Center[2]     = row center of the image
  BinSiz[1]     = distance in pixels of original image between two columns
  BinSiz[2]     = distance in pixels of original image between two rows

 optional

  PixSiz[1]       = column width in meters
  PixSiz[2]       = row width in meters
  SampleDistance  = distance of the sample in meters from the center that is
                    described by (Center_1, Center_2)
  WaveLength      = photon wavelength in meters
  DetRot[1]       = first detector rotation around axis 1
  DetRot[2]       = second detector rotation around axis 2
  DetRot[3]       = third detector rotation around axis 3

  Title           = one line with length InputLineLength-1 characters or shorter
  Time            = one line with start time,
                    preferred iso time: YYYY-MM-DD hh:mm:ss[.ssssss]
  Orientation     = orientation of the array (numbers from 1 to 16),
                    default 1
  Projection      = projection type of the array 
                    (IO_tppro: IO_ProSaxs, IO_ProWaxs, ...)
                  = default IO_ProSaxs
  AxisType        = axis type (IO_tpaxis: IO_Distance, IO_Angle, IO_Numerator)
                  = default IO_AxisTypeDistance

The keywords and the used coordinate systems are defined in the file
SaxsDefinition.h.

If a keyword has been found in the image header the corresponding .I flag
in the image header block is set to TRUE (+1) and the parameter is read.
It is also possible that a .I flag is set to -TRUE (-1).  This is the case
if a header block value has been overwritten by an image block value. The
test whether a flag is set must be : if (xxx.I!=0) the isset.

HISTORY
16-Nov-1995	Definition of return value of ReadImage and ReadImageHeader
		changed.
24-Nov-1995 PB	
27-Nov-1995 PB  ImgHeadBlock.I
04-Dec-1995 PB  IO_DontReadData in NewImage
21-Dec-1995 PB  all data format specific routines in external file
                (GetData2d.inc)
23-Dec-1995 PB  splitting in test_image.c and test_image.h
24-Dec-1995 PB  ReportSaxsError + more debug information for +test
25-Dec-1995 PB  ImageNumber int --> long
26-Dec-1995 PB  long --> long int
26-Jan-1996 PB  IO_DontRead --> IO_DontReadData (replaced 1:1)
                IO_DontWrite --> IO_DontReadWrite
                IO_DontReadWrite is now also checked in the read routines
                ReadImage and ReadImageHeader
27-Jan-1996 PB  ReadImageDimension added
28-Feb-1996 PB  alloc_image_memory: special case for ImgByteLen==0
29-Mar-1996 PB  MinClip and MaxClip
19-May-1996 PB  Gnaw, the order of operation for read and write is:
                clip, gnaw, rebin and transform image and image header
10-Oct-1996 PB  renamed from test_image.c to SaxsImage.c,
                include "ImageLoop.inc"
12-Oct-1996 PB  Error block modified, ReportSaxsError changed, SetSaxsError
14-Oct-1996 PB  V1.0 SAXSIMAGEVERSION="1.0" removed, replaced by
                SIM_Version, SaxsImageVersion
13-Dec-1996 PB  rebin_float... has only checked ..Bin[1].I, now also Bin[2].I
28-Aug-1997 PB  debug information added (SIM_Version V1.2)
31-Dec-1997 PB	SaxsGetData2d.inc -> ndf.inc (SIM_Version V1.3)
01-Jan-1997 PB  ndf.inc renamed to SaxsNewData2d.inc
12-Jan-1998 PB  all free(xyz); replaced against {free(xyz);xyz=NULL;}
                to avoid a second attempt to free the data (SIM_Version V1.31)
12-Jan-1998 PB  call to open_data_file now with: new, old, any (instead of
                any only) V1.32
17-Mar-1998 PB  include edfio.h V1.33
17-Mar-1998 PB  GeneralBlock option V1.34 in OpenImageFile 
17-May-1998 PB  call to open_data_file now with: new, old, any and read V1.35
18-Apr-1999 PB  new names of edfio routines V1.36
30-Apr-1999 PB  status set to true before calling edfio routines V1.37
25-Jun-1999 PB  edf_history V1.40:
                edf_read_header_history -> read_saxs_image 
                edf_write_header_history --> write_image_header
                The history is read from blkno 1 and written to
                blkno 0 together with the argument history (SaxsOption)
26-Jun-1999 PB  Time added to image header block V1.50
17-Jul-1999 PB  Comment corrected V1.51
17-Jul-1999 PB  V1.52
                definition of SAXSDATAVERSION removed (now in SaxsDefinition.h)
09-Nov-1999 PB  V1.53 DUMMYDEFINED updated
26-Nov-1999 PB  V1.54 cc on dec alpha: statements of the type *pd++=*pd+xyz;
                      changed to { *pd=*pd+xyz; pd++; }
21-Dec-1999 PB  V1.55 DummyFileName in OpenImageFile, IO_Dummy,
                      and alloc_image_memory in read_saxs_image 
18-Mar-2000 PB  V1.60 memory replaces ImageNumber 
                      modified: write_image_header, WriteImage,
                      read_saxs_image, ReadImage, ReadImageHeader,
                      SearchMinMaxImage, ReadImageDimension,
                      ReadHeaderLine, ReadHeaderLong, ReadHeaderFloat,
                      saxs_image_loop (default of output memory number
                      is memory number of first input)
07-Apr-2000 PB  V1.61 set_max_line_width
30-Jul-2000 PB  V1.62 Orientation added to image header block
                      orientation_float_2d, orientation_header_2d
                      included in read_saxs_image and WriteImage
31-Aug-2000 PB  V1.63 debug info in ReadHeader...
10-Nov-2000 PB  V1.7  condition_2d, changes for edfio V1.30
12-Nov-2000 PB  V1.71 changes for edfio V1.32
                V1.72 long ImgByteLen -> size_t ImgByteLen
                V1.73 changes for edfio V1.34
                      long ihb.Dim[3] -> long ihb.Dim[4]
2000-12-30      V1.74 changes for edfio V1.36
                      memory allocated for input image
2000-12-31      V1.75 element Dim[3] not used any more
2001-01-11      V1.76 NewImageHeader
2001-01-12      V1.77 u_long -> unsigned long
2001-07-03      V1.78 SaxsGetName.inc replaced by SaxsFilename.h
2001-07-06      V1.79 OpenImageFile uses edf_search_stream
2001-07-06      V1.80 default for ib[].Stream is -1 (no stream opened)
2001-07-07      V1.81 DoNotScaleWith
2001-07-08      V1.82 OpenImageFile(...,FileNumber,...)
2001-07-09      V1.83 OpenImageFile sets ib[].FileNumber and ib[].FileName
2001-07-10      V1.84 New SaxsImageLoop
2001-07-12      V1.85 New SaxsImageLoop.inc, ActualLoopCount
2001-08-19      V1.86 WriteHeaderLine, WriteHeaderFloat, WriteHeaderLong
                      ImageNumber removed from argument list of WriteImage
2001-08-20      V1.87 NewImageHeader
2001-08-30      V1.88 pihb->Write, SaxsImageMessage,
                      ImageIsNew, ImageToWrite, ImageIsFirst, ImageIsLast
2001-09-02      V1.89 edf_set_datatype, edf_set_datavalueoffset
2001-09-13      V1.90 history_key
2001-09-16      V1.91 PassHeader, header key in image header block, Header in
                      read_saxs_image, New_Image_Header, free_image_memory
2001-09-17      V1.92 write_image_header
2001-11-25      V1.93 (Re)OpenImageFile: edf_keyorder_set_table
2002-02-01      V1.94 condition_2d into header file
2002-05-28      V1.95 condition_2d(...,Ave,...)
2002-06-01      V1.96 adaptation to SaxsOption V6.33 (pcb->Ave -> pib->Ave)
2002-06-09      V1.97 LoopNumber in SaxsImageLoop
2002-09-01      V1.98 saxs_image_loop modified, ImageToWrite private
2003-02-02      V1.99 balance_header_block, read_saxs_image, ReadImage, 
                      ReadImageHeader: ImageNumber -> num[blkno]
2003-02-03      V2.00 SaxsReplacekey.h, 
                      balance_header_block( CmdBlk * pcb, ... , int * pstatus)
                      Title and Time with replacekey
2003-02-04      V2.01 balance_header_block: MinClip, MaxClip, Const, Factor
                      (preparation for replacekey)
2003-02-06      V2.02 floatkeyvalue, longkeyvalue, linekeyvalue
2003-02-07      V2.03 balance_header_block: Bin, Gnaw
2003-02-08      V2.04 balance_header_block: Dummy, DDummy, Dim
2003-02-10      V2.05 balance_header_block: Shift
2003-04-06      V2.06 SaxsImageLoop.inc
2003-04-13      V2.07 SaxsImageLoop.inc
2003-04-13      V2.08 SaxsImageLoop.inc
2003-04-15      V2.09 SaxsImageLoop.inc 
2003-05-25      V2.10 balance_header_block: Offset, Center, PixSiz,
                      WaveLength, SampleDistance
2003-05-29      V2.11 balance_header_block: Orientation 
                      WriteImage: pib->Orientation.V -> pihb->Orientation.V
                      NewImage: pib->Orientation.V is copied to 
                      pihb->Orientation.V, rori 
2003-05-30      V2.12 ...keyvalue with new parameter list,
                      ImageLoop with new parameter list (long Num[] added)
2003-11-30      V2.13 BinSiz 
2004-03-04      V2.14 AREBIN1 -> AREBIN 
2004-04-10      V2.15 SaxsImageLoop.inc
2004-04-12      V2.16 clip_float_2d: comparison for maxclip without a dummy 
                                     corrected.
                                     Values exceeding the limits of minclip
                                     or maxclip are set to minclip or maxclip
                                     if no dummy is set. They used to be set 
                                     to 0.
2004-07-08      V2.17 rebin_float_2d: copy always if input and output buffers 
                                      are different
2004-07-17      V2.18 MinHeaderSize
2004-09-25      V2.19 Projection added to image header block
2004-10-01      V2.20 new function: WriteHeader
2004-10-04      V2.21 DetRot added to image header block 
                      new function: WriteHeaderUnit
2004-10-14            reset_header_block: loop for DetRot[3].V corrected
2005-02-11      V2.22 BalanceHeaderBlock
2005-03-03      V2.23 OpenImageFile corrected
2005-07-11      V2.24 IO_prog Val, ErrVal, 
                      condition_2d: argument Val added
                      reset_header_block -> init_header_block
                      set_header_defaults -> set_output_header_defaults
2005-07-14      V2.25 SetSaxsError (seb removed from cb), also in
                      SaxsImageLoop.inc
                      ReOpenImageFile -> reopen_image_file.
2005-08-04      V2.26 pihb->blkno, init_header_block: parameter blkno added
                      PrintImageHeaderBlock: parameter i=blkno removed
2005-08-07      V2.27 VarDat(variance data), init_header_block, 
                      alloc_image_memory, free_image_memory,
                      clip_data_2d, gnaw_data_2d, rebin_data_2d,
                      prog_data_2d, prog_vardat_2d,
                      condition_2d updated
                      saxs_image_loop: default of CreateVariance set
                      variance calculation is done if input image 1 has a
                      variance array
                      Output variance array is initialized with dummy (-1.0)
                      to allow use of macro UPDATE
2005-09-17      V2.28 NewImage: Create VarDat only if image memory is positive
                      read_saxs_image, WriteImage: read/write VarDat only 
                      if image memory is positive,
                      VarDummy used for variance dummies
                      saxs_image_loop: CreateVariance default set to 1
                      if ErrVal.I flag of any block is set.
                      Write VarDummy into header of error block. 
2005-10-13      V2.29 PrintImageHeaderBlock: format error corrected
                      SaxsImageLoop.inc (for cc)
2005-11-30      V2.30 parameter list of linekeyvalue updated,
                      keys of Val and ErrVal read only first parameter
2005-12-01      V2.31 balance_header_block: tmp_line collapsed before calling 
                                            num_str2prog
2005-12-04      V2.32 include ipol.h
2005-12-05      V2.33 include ipol.h already in image.h, Ipolmode added
2005-12-11      V2.34 PrintImageHeaderBlock: program sizes added
2006-03-15      V2.35 PrintImageHeaderBlock: printf argument mismatch corrected
                      AlreadyOpened moved before SaxsImageLoop.inc
2007-03-30      V2.36 sizeof is always first operand (for data type)
2007-04-19            -Wall compiler warnings resolved 
2007-04-20            %zu complemented with %lu
2007-06-10      V2.37 Parameter InitFunction added to ImageLoop, NewImage and
                      new_image_header
2007-06-17      V2.38 SaxsFilename -> filename
2007-12-01      V2.39 Axis type
2008-05-07      V2.40 PrintImageHeaderBlock: some .I flags included
2008-05-25      V2.41 Ipolweight added
2008-11-30      V2.42 Functions Single2Multi, Multi2Single added
2010-11-08      V2.43 saxs_open_all_images
                      replace header keys in file names,
                      to change image block and parameter numbers
                      filename.c needs to be adapted. In order to read the
                      filename parameters it must ignore everything between 
                      square brackets []
2011-01-05      V2.44 WriteImage: output compression
2011-06-06      V2.45 SaxsImageLoop: CommonOrientation added,
                      condition_2d: no orientation transformation
2011-06-08      V2.46 check orientation number
2011-06-23      V2.47 read_saxs_image: missing return value TRUE added
2011-07-16      V2.48 SaxsImageLoop
2012-01-29      V2.49 SaxsImageLoop, ihb[].Write suppressed
2013-01-25      V2.50 orientation_header_2d: omod8 added

---*/

/****************************************************************************
* SaxsImage.c                                                               *
****************************************************************************/
# define SIM_Version  "SaxsImage : V2.50 Peter Boesecke 2013-01-25"

# include <stdio.h>
# include <string.h>
# include <unistd.h>
# include <stdlib.h>

# include "SaxsImage.h"
# include "SaxsReplacekey.h"
# include "filename.h"
# include "edfio.h"                        /* data format specific routines */
# include "sx.h"                                             /* sx geometry */

/*----------------------------------------------------------------------------
 Version String
----------------------------------------------------------------------------*/
static char * VersionString = (char *) NULL;

/*----------------------------------------------------------------------------
 Constants
----------------------------------------------------------------------------*/
const char * ModuleName    = "SaxsImage";
const char * DummyFileName = DUMMYFILENAME;

/*----------------------------------------------------------------------------
 Static Messages 
----------------------------------------------------------------------------*/
static char SaxsImageMessage[IO_len];

/*----------------------------------------------------------------------------
1 SaxsImageVersion

2 DESCRIPTION
  Returns a pointer to a string which explains the version of the used
  modules
  char * SaxsImageVersion ( void )
  "'SIM_Version' (Files = 'edf_dataformat_version')"

2 EXTERNAL FILES
  SaxsGetData2d.inc (edf_dataformat_version)

2 HISTORY
  14-Oct-1996 PB
  12-Jan-1997 PB free(xyz) -> { free(xyz);xyz=NULL;}
----------------------------------------------------------------------------*/
char * SaxsImageVersion ( void )
{ size_t t1, t2, t3;
  if ( VersionString!=(char *) NULL ) { 
    free( VersionString ); VersionString=NULL; }
  t1 = strlen(SIM_Version);
  t2 = strlen((char *) edf_dataformat_version() );
  t3 = strlen(SAXSDATAVERSION);
  VersionString =  (char *) malloc ( 30 + t1 + t2 + t3 + 1 );
  if ( VersionString!=(char *) NULL )
    sprintf(VersionString,"%s (SDV = %s, GDV = %s)",
              SIM_Version, SAXSDATAVERSION, edf_dataformat_version() );
  return ( VersionString );
} /* SaxsImageVersion */

/*--------------------------------------------------------------------------
NAME

  orientation_float_2d --- change orientation of 2d float data array

SYNOPSIS

  void orientation_float_2d( float *dataIn, float * dataOut,
                             long int dim_1, long int dim_2,
                             long relative_orientation,
                             int * pstatus );

DESCRIPTION
Change orientation of 2d float data array.

ARGUMENTS
  float * dataIn  : input data array
  float * dataOut : output data array (must be different from dataIn)
  long int dim_1  : dimensions of input and output arrays
  long int dim_2
  long relative_orientation : orientation of input array relative to the
                              desired final position
  int * pstatus   : Success or Failed

RETURN VALUE
  void

HISTORY
  2000-07-30 Peter Boesecke
--------------------------------------------------------------------------*/
void orientation_float_2d( float *dataIn, float * dataOut,
                           long int dim_1, long int dim_2,
                           long relative_orientation,
                           int * pstatus )
{ long data_dim[4];

  *pstatus = Success;

  if (relative_orientation<0)
    relative_orientation = edf_raster_inversion(-relative_orientation);

  data_dim[0] = 2l;
  if (relative_orientation<=4l) {
    data_dim[1] = dim_1; data_dim[2] = dim_2; 
    } else {
    data_dim[1] = dim_2; data_dim[2] = dim_1;
    }

  if ( edf_raster_normalization ( (void *) dataOut, (const void *) dataIn,
                data_dim, relative_orientation, sizeof( float ) ) ) {
    *pstatus = Failed;
    }

} /* orientation_float_2d */

/*--------------------------------------------------------------------------
NAME

  orientation_header_2d --- update header after a change of orientation

SYNOPSIS

  void orientation_header_2d ( ImgHeadBlk * pihb, long relative_orientation );

DESCRIPTION
The header values are changed according to a data transformation from
relative_orientation to 1

ARGUMENTS
  ImgHeadBlk * pihb         : image header block of the current data
  long relative_orientation : orientation relative to the final position

RETURN VALUE
  long orientation  > 0 final orientation of the data array
                   == 0 error
                    < 0 inverse orientation

HISTORY
  2000-07-30 Peter Boesecke
  2003-05-29 relative_orientation < 0
--------------------------------------------------------------------------*/
long orientation_header_2d ( ImgHeadBlk * pihb, long relative_orientation )
{ long tmpl;
  IO_float tmpf;
  long inverse_orientation;
  long orientation; 
  long omod2, omod4, omod8;

  if (relative_orientation<0)
    relative_orientation = edf_raster_inversion( -relative_orientation );
  inverse_orientation = edf_raster_inversion( relative_orientation );
  orientation =
     edf_raster_multiplication( inverse_orientation, pihb->Orientation.V );
  omod2 = (inverse_orientation-1) % 2l;
  omod4 = (inverse_orientation-1) % 4l;
  omod8 = (inverse_orientation-1) % 8l;

  if ( omod2 >= 1l ) { // invert first coordinate
    pihb->Center[1].V = CSWAP(pihb->Center[1].V,pihb->Offset[1].V,pihb->Dim[1]);
  }

  if ( omod4 >= 2l ) { // invert second coordinate
    pihb->Center[2].V = CSWAP(pihb->Center[2].V,pihb->Offset[2].V,pihb->Dim[2]);
  }

//+++++++++++  if ( inverse_orientation > 4l ) { // swap coordinates }
  if ( omod8 >= 4l ) { // swap coordinates
    tmpl=pihb->Dim[1];    pihb->Dim[1]   =pihb->Dim[2];    pihb->Dim[2]   =tmpl;
    tmpf=pihb->Offset[1]; pihb->Offset[1]=pihb->Offset[2]; pihb->Offset[2]=tmpf;
    tmpf=pihb->Center[1]; pihb->Center[1]=pihb->Center[2]; pihb->Center[2]=tmpf;
    tmpf=pihb->BinSiz[1]; pihb->BinSiz[1]=pihb->BinSiz[2]; pihb->BinSiz[2]=tmpf;
    tmpf=pihb->PixSiz[1]; pihb->PixSiz[1]=pihb->PixSiz[2]; pihb->PixSiz[2]=tmpf;
  }

  pihb->Orientation.V = orientation; pihb->Orientation.I = TRUE;

  return ( orientation );

} /* orientation_header_2d */

/*----------------------------------------------------------------------------
rebin_data_2d
  Two dimensional rebinning of data array and variane array. The result is 
  written into the output arrays and the new dimensions are returned. 
  Input and output arrays can be the same.
Parameters
  float * dataIn  (i)     pointer to input array
  float * dataOut (o)     pointer to output array
  float * dataIn  (i)     pointer to input variance array (NULL, if not used)
  float * dataOut (o)     pointer to output variance array (NULL, if not used)
  long * dim_1    (i/o)   pointer to input/output dimension 1
  long * dim_2    (i/o)   pointer to input/output dimension 2
  float dummy     (i)     dummy value
  float ddummy    (i)     ddummy value
  long bin_1      (i)     rebin factor 1 (>=1)
  long bin_2      (i)     rebin factor 2 (>=1)
  long ave        (i)     average!=0/sum==0 pixel values
-----------------------------------------------------------------------------*/
void rebin_data_2d ( float * dataIn, float * dataOut,
                     float * vardatIn, float * vardatOut,
                     long int * dim_1, long int * dim_2,
                     float dummy, float ddummy,
                     long int bin_1, long int bin_2,
                     long int ave )
{ register long int j_1, j_2, i_1, i_2;
  float * pdatain, * pdataout;
  float * pvardatin, * pvardatout;
  float count;
  float datavalue, datasum;
  float vardatvalue=0.0, vardatsum=0.0;
 
  bin_1 = MAX2(1,bin_1);
  bin_2 = MAX2(1,bin_2);

  if ((bin_1>1) || (bin_2>1) || (dataOut!=dataIn)) {
    pdataout = dataOut;
    pvardatout = vardatOut;
    for (j_2=0;j_2<=*dim_2-bin_2;j_2+=bin_2)
      for (j_1=0;j_1<=*dim_1-bin_1;j_1+=bin_1) {
        count = 0.0;
        datasum = 0.0; 
        vardatsum = 0.0; 
        for (i_2=j_2;i_2<j_2+bin_2;i_2++) {
          pdatain = ABSPTR(dataIn,*dim_1,*dim_2,j_1,i_2);
          pvardatin = ABSPTR(vardatIn,*dim_1,*dim_2,j_1,i_2);
          for (i_1=j_1;i_1<j_1+bin_1;i_1++) {
            datavalue = *(pdatain++);
            if (vardatIn) {
              vardatvalue = *(pvardatin++);
              if (vardatvalue<0.0) vardatvalue=0.0; 
            }
            if NODUMMY(datavalue,dummy,ddummy) {
              count += 1.0;
              datasum += datavalue;
              vardatsum += vardatvalue;
            }
          }
        }
        if (count>0.0) {
          *(pdataout++)   = ave?datasum/count:datasum;
          if (vardatOut) *(pvardatout++)=ave?vardatsum/(count*count):vardatsum;
        } else {
          *(pdataout++) = dummy;
          if (vardatOut) *(pvardatout++) = VarDummy;
        }
      }
    *dim_1 /= bin_1; *dim_2 /= bin_2;
  }
 
} /* rebin_data_2d */

/*----------------------------------------------------------------------------
rebin_float_2d
  Two dimensional rebinning of an array. The result is written into the
  output array and the new dimensions are returned. Input and output 
  arrays can be the same.
Parameters
  float * dataIn  (i)     pointer to input array
  float * dataOut (o)     pointer to output array
  long * dim_1    (i/o)   pointer to input/output dimension 1
  long * dim_2    (i/o)   pointer to input/output dimension 2
  float dummy     (i)     dummy value
  float ddummy    (i)     ddummy value
  long bin_1      (i)     rebin factor 1 (>=1)
  long bin_2      (i)     rebin factor 2 (>=1)
  long ave        (i)     average!=0/sum==0 pixel values
-----------------------------------------------------------------------------*/
void rebin_float_2d ( float * dataIn, float * dataOut,
                      long int * dim_1, long int * dim_2,
                      float dummy, float ddummy,
                      long int bin_1, long int bin_2,
                      long int ave )
{ register long int j_1, j_2, i_1, i_2;
  float * pin, * pout;
  float value, sum, count;

  bin_1 = MAX2(1,bin_1);
  bin_2 = MAX2(1,bin_2);

  if ((bin_1>1) || (bin_2>1) || (dataOut!=dataIn)) {
    pout = dataOut;
    for (j_2=0;j_2<=*dim_2-bin_2;j_2+=bin_2)
      for (j_1=0;j_1<=*dim_1-bin_1;j_1+=bin_1) {
        sum = 0.0; count = 0.0;
        for (i_2=j_2;i_2<j_2+bin_2;i_2++) {
          pin = ABSPTR(dataIn,*dim_1,*dim_2,j_1,i_2);
          for (i_1=j_1;i_1<j_1+bin_1;i_1++) {
            value = *(pin++);
            if NODUMMY(value,dummy,ddummy)
              { sum += value; count += 1.0; }
          }
        }
        if (count>0.0) *(pout++) = ave?sum/count:sum; 
        else *(pout++) = dummy;
      }
    *dim_1 /= bin_1; *dim_2 /= bin_2;
  }

} /* rebin_float_2d */

/*---------------------------------------------------------------------------
  transform_float_2d
  Multiplies the data field with a factor and adds a constant.
  ---------------------------------------------------------------------------*/
void transform_float_2d( float *data, long int dim_1, long int dim_2,
                         float dummy, float ddummy,
                         float factor, float constant )
{ register long int i;
  long int datanum;
  const float eps1 = 1e-9, eps2 = 1e-30;
  float *pdatain;
  float *pdataout;

  datanum = dim_1 * dim_2;
  pdatain = data;
  pdataout = data;
  if ((ABS(factor-1.0)>eps1) || (ABS(constant)>eps2)) { 

    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdatain,dummy,ddummy) 
           *pdataout = *pdatain * factor + constant;
        pdataout +=1; pdatain +=1;
        }
      } else for (i=0;i<datanum;i++) {
               *pdataout = *pdatain * factor + constant; 
               pdataout++; pdatain++;
               }
    } /* if ((ABS ...  */ 

} /* transform_float_2d */

/*---------------------------------------------------------------------------
  prog_data_2d
  Runs prog(iNval) over all pixels
  ---------------------------------------------------------------------------*/
void prog_data_2d( float *data, float *vardat, long int dim_1, long int dim_2,
                   float dummy, float ddummy,
                   NumProg *prog, int *pstatus )
{ register long int i;
  long int datanum;
  float *pdata, *pvardat;
  double value, datain, vardatin;
  double value2, datain2, difference;
  int use_var1;

  int errval;

  *pstatus = Success;

  datanum = dim_1 * dim_2;
  pdata = data; pvardat = vardat;

  if ( (data)&&(prog) ) {
    // check, whether prog uses the first variable (datain)
    use_var1 = num_chkvar ( prog, 1, &errval );

    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
          datain = (double) *pdata;
          value = num_runprog ( prog, &errval, datain );
          if (!errval) *pdata = (float) value;

          if (vardat) {
            // Program operations error propagation
            if (use_var1) {
              // calculate propagated variance value
              vardatin = (double) *pvardat;
              if (vardatin>0.0) {
                datain2  = datain + sqrt( vardatin );
                value2 = num_runprog ( prog, &errval, datain2 );
                if (!errval) {
                  difference = value - value2;
                  *pvardat = (float) (difference*difference);
                }
              }
            } else *pvardat = 0.0; // There is no variance value to propagate
          }

        } else if (vardat) *pvardat = VarDummy;
        pdata++; pvardat++;
      }
    } else {
      for (i=0;i<datanum;i++) {
        datain = (double) *pdata;
        value = num_runprog ( prog, &errval, datain );
        if (!errval) *pdata = (float) value;

        if (vardat) {
          // Program operations error propagation
          if (use_var1) {
            // calculate propagated variance value
            vardatin = (double) *pvardat;
            if (vardatin>0.0) {
              datain2  = datain + sqrt( vardatin );
              value2 = num_runprog ( prog, &errval, datain2 );
              if (!errval) {
                difference = value - value2;
                *pvardat = (float) (difference*difference);
              }
            } else *pvardat = 0.0; // There are no dummies
          } else *pvardat = 0.0; // There is no variance value to propagate
        }

        pdata++; pvardat++;
      }
    }
  } /* if (prog)  */

} /* prog_data_2d */

/*---------------------------------------------------------------------------
  prog_vardat_2d
  Runs prog(iNerr) over all pixels
  ---------------------------------------------------------------------------*/
void prog_vardat_2d( float *data, float *vardat, long int dim_1, long int dim_2,
                     float dummy, float ddummy,
                     NumProg *prog, int *pstatus )
{ register long int i;
  long int datanum;
  float *pdata, *pvardat;
  double value, variance;
 
  int errval;
 
  *pstatus = Success;
 
  datanum = dim_1 * dim_2;
  pdata = data;
  pvardat = vardat;
 
  if ( (data)&&(vardat)&&(prog) ) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
          if (*pvardat>=0.0) variance=(double) *pvardat; 
          else variance=(double) 0.0;
          value = num_runprog ( prog, &errval, (double) *pdata, variance );
          if (!errval) *pvardat = (float) value;
        } else *pvardat = VarDummy;
        pdata++; pvardat++;
      }
    } else {
      for (i=0;i<datanum;i++) {
        if (*pvardat>=0.0) variance=(double) *pvardat; 
        else variance=(double) 0.0;
        value = num_runprog ( prog, &errval, (double) *pdata, variance );
        if (!errval) *pvardat = (float) value;
        pdata++; pvardat++;
      }
    }
  } /* if (prog)  */
 
} /* prog_vardat_2d */

/*---------------------------------------------------------------------------
  clip_data_2d
  Sets all values of the data field outside minclip and maxclip to dummy.
  If vardat exists, clipped values are set to -1 
  ---------------------------------------------------------------------------*/
void clip_data_2d( float *data, float *vardat, long int dim_1, long int dim_2,
                    float dummy, float ddummy,
                    float minclip, int minclipI,
                    float maxclip, int maxclipI )
{ register long int i;
  long int datanum;
  float *pdata, *pvardat;

  datanum = dim_1 * dim_2;
  pdata = data; pvardat = vardat;
  if ( (data)&&(minclipI) ) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
          if (minclip>*pdata) { 
            *pdata = dummy;
            if (vardat) *pvardat = VarDummy;
          }
        }
        pdata++; pvardat++;
      }
    } else for (i=0;i<datanum;i++) {
             if (minclip>*pdata) {
               *pdata = minclip;
               if (vardat) *pvardat = VarDummy;
             }
             pdata++; pvardat++;
           }
  } /* if (minclipI) */

  pdata = data; pvardat = vardat;
  if ( (data)&&(maxclipI) ) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
          if (maxclip<*pdata) {
            *pdata = dummy;
            if (vardat) *pvardat = VarDummy;
          }
        }
        pdata++; pvardat++;
      }
    } else for (i=0;i<datanum;i++) {
             if (maxclip<*pdata) {
               *pdata = maxclip;
               if (vardat) *pvardat = VarDummy;
             }
             pdata++; pvardat++;
           }
  } /* if (maxclipI) */

} /* clip_data_2d */

/*----------------------------------------------------------------------------
gnaw_data_2d
  Sets all pixels inside an ellipse around a dummy to dummy.
  The result is written into the input array. The routine
  allocates a short integer array with the same dimensions as the
  data array for intermediate results. Its memory is released after
  successful termination of gnaw_data_2d.
  If vardat exists its values are set to -1 where a dummy is set.
Parameters
  float * data    (i)     pointer to input/output array
  float * vardat  (i)     pointer to input/output variance array
  long    dim_1   (i)     input/output/tmp dimension 1
  long    dim_2   (i)     input/output/tmp dimension 2
  float   dummy   (i)     dummy value
  float   ddummy  (i)     ddummy value
  long    gnaw_1  (i)     axis 1 of the ellipse (>=0)
  long    gnaw_2  (i)     axis 2 of the ellipse (>=0)
  int   * pstatus (o)     exit status
-----------------------------------------------------------------------------*/
void gnaw_data_2d ( float * data, float * vardat,
                     long int dim_1, long int dim_2,
                     float dummy, float ddummy,
                     long int gnaw_1, long int gnaw_2,
                     int * pstatus )
{ register long int i, j_1, j_2, i_1, i_2;
  long int i_1min;
  float * pdata, * pvardat;
  short int * tmp, * ptmp;
  size_t t_tmp;

  float value;
  long int gnaw2_1, gnaw2_2, gnaw2;
  long int dist2_1, dist2_2;

  *pstatus = Success;

  gnaw_1 = MAX2(0,gnaw_1);
  gnaw_2 = MAX2(0,gnaw_2);

  if ( (data) && ((gnaw_1>0) || (gnaw_2>0)) ) {

    /* allocate memory for tmp array*/
    t_tmp = sizeof(short int)*dim_1*dim_2;
    tmp = (short int *) malloc (t_tmp);
    if (tmp == (short int *) NULL) {
      *pstatus=NotEnoughMemoryAvailable; return; }

    /* clear tmp array */
    ptmp = tmp;
    for (i=0;i<dim_1*dim_2;i++) *ptmp++=0;

    /* search dummies in data array */
    gnaw2_1 = gnaw_1 * gnaw_1 + 1; /* to avoid stars with gnaw_1 = 1 */
    gnaw2_2 = gnaw_2 * gnaw_2 + 1; /* to avoid stars with gnaw_2 = 1 */
    gnaw2   = gnaw2_1*gnaw2_2;
    pdata = data;
    for (j_2=0;j_2<dim_2;j_2++)
      for (j_1=0;j_1<dim_1;j_1++) {
        value = *(pdata++);
        if DUMMY(value,dummy,ddummy) {
          /* create dummy ellipse around j_1, j_2 in tmp */
          for (i_2=MAX2(0,j_2-gnaw_2);i_2<MIN2(dim_2,j_2+gnaw_2+1);i_2++) {
            dist2_2 = i_2 - j_2; dist2_2 *= dist2_2;
            i_1min = MAX2(0,j_1-gnaw_1);
            ptmp = ABSPTR(tmp,dim_1,dim_2,i_1min,i_2);
            for (i_1=i_1min;i_1<MIN2(dim_1,j_1+gnaw_1+1);i_1++) {
              dist2_1 = i_1 - j_1; dist2_1 *= dist2_1;
              if (dist2_1*gnaw2_2+dist2_2*gnaw2_1<=gnaw2) *ptmp = 1;
              ptmp++;
              }
            }
          }
        }

    /* copy dummies from tmp to data */
    ptmp = tmp;
    pdata = data; pvardat = vardat;
    for (i=0;i<dim_1*dim_2;i++) {
      if (*ptmp++) {
        *pdata=dummy;
        if (vardat) *pvardat=VarDummy;
      }
      pdata++; pvardat++;
    }

    /* free tmp array */
    free(tmp); tmp=NULL;

  }

} /* gnaw_data_2d */

/*----------------------------------------------------------------------------
  NAME
    condition_2d

  DESCRIPTION

    clip, gnaw, rebin, orient and transform image and image header 

  ARGUMENTS
    ImgHeadBlk * pihb     pointer to image header block
    ...
    long Orientation      1 to 8, inverse transformations -1 to -8
                          transform data orientation from 
                          Orientation to 1, e.g.
                          read data:  pihb->Orientation.V
                          write data: edf_raster_inversion(pib->Orientation.V) 
                          or alternatively write data: -pihb->Orientation.V
                          0: no orientation transformation
    int Simulate          TRUE:  condition header only, do not access arrays
                          FALSE: condition header and data
    int TestBit           debug flag
    int *pstatus          exit status
-----------------------------------------------------------------------------*/
void condition_2d( ImgHeadBlk * pihb,
                   IO_float MinClip, IO_float MaxClip,
                   IO_long  Gnaw_1,  IO_long  Gnaw_2,
                   IO_long  Bin_1,   IO_long  Bin_2,
                   IO_prog  Val,     IO_prog  ErrVal,
                   IO_float Factor,  IO_float Const,
                   IO_long  Ave,     long Orientation,
                   int Simulate, int TestBit, int *pstatus )
{
  float * ptmp;
  float * pdatain = (float *) NULL;
  float * pdataout = (float *) NULL;
  float * TmpDataBuffer = (float *) NULL;
  size_t  TmpDataBufferSize = (size_t) NULL;
  float * pvardatin = (float *) NULL;
  float * pvardatout = (float *) NULL;
  float * TmpVarDatBuffer = (float *) NULL;
  size_t  TmpVarDatBufferSize = (size_t) NULL;

    // Clipping
    if (MinClip.I || MaxClip.I) {
      if (!Simulate) {
        if (TestBit>1) printf(" --- clip_data_2d\n");
        clip_data_2d( pihb->Data, pihb->VarDat, pihb->Dim[1], pihb->Dim[2],
                      pihb->Dummy.V, pihb->DDummy.V,
                      MinClip.V, MinClip.I,
                      MaxClip.V, MaxClip.I );
        } // if (!Simulate)
      } // Clipping

    // Gnawing
    if (Gnaw_1.I || Gnaw_2.I) {
      if (!Simulate) {
        if (TestBit>1) printf(" --- gnaw_data_2d\n");
        gnaw_data_2d ( pihb->Data, pihb->VarDat, pihb->Dim[1], pihb->Dim[2],
                        pihb->Dummy.V, pihb->DDummy.V,
                        Gnaw_1.V, Gnaw_2.V,
                        pstatus );
        if (*pstatus!=Success) return;
        } // if (!Simulate)
      } // Gnawing

    // Binning
    if (!Bin_1.I) Bin_1.V = 1l; else Bin_1.V = MAX2(1,Bin_1.V);
    if (!Bin_2.I) Bin_2.V = 1l; else Bin_2.V = MAX2(1,Bin_2.V);
    if ((Bin_1.V>1) || (Bin_2.V>1)) {
      if (!Simulate) {
        if (pihb->VarDat) {
          if (TestBit>1) printf(" --- rebin_data_2d\n");
          rebin_data_2d ( pihb->Data, pihb->Data,
                          pihb->VarDat, pihb->VarDat,
                          &pihb->Dim[1], &pihb->Dim[2],
                          pihb->Dummy.V, pihb->DDummy.V,
                          Bin_1.V, Bin_2.V, Ave.V );
        } else {
          if (TestBit>1) printf(" --- rebin_float_2d\n");
          rebin_float_2d ( pihb->Data, pihb->Data,
                           &pihb->Dim[1], &pihb->Dim[2],
                           pihb->Dummy.V, pihb->DDummy.V,
                           Bin_1.V, Bin_2.V, Ave.V );
        }
      } else {
        // update dimension
        pihb->Dim[1] /= Bin_1.V;
        pihb->Dim[2] /= Bin_2.V;
      } // if (!Simulate)
      // update header
      AREBIN(pihb->Offset[1].V,pihb->BinSiz[1].V,\
             pihb->PixSiz[1].V,pihb->Center[1].V,Bin_1.V);
      pihb->BinSiz[1].I = TRUE;
      AREBIN(pihb->Offset[2].V,pihb->BinSiz[2].V,\
             pihb->PixSiz[2].V,pihb->Center[2].V,Bin_2.V);
      pihb->BinSiz[2].I = TRUE;
    } // Binning

    /* During orientation transformatino the input and output buffers
       must be different. Data is moved between the two buffers and
       finally copied to pihb->Data, either with transform or memcpy */

    pdatain  = pihb->Data; // pointer to actual input data
    pdataout = (float *) NULL; // currently no separate output buffer

    // Orientation
    if (Orientation<0l)
      Orientation = edf_raster_inversion( -Orientation );

    if (Orientation > 1l) {
      if (!Simulate) {

        if (!pdataout) { // allocate output buffer
          if (TestBit>1) printf(" --- allocate memory for orientation\n");
          TmpDataBufferSize = sizeof(float)*pihb->Dim[1]*pihb->Dim[2];
          TmpDataBuffer = (float *) malloc ( TmpDataBufferSize );
          if (!TmpDataBuffer) { *pstatus=NotEnoughMemoryAvailable;return; }
          pdataout = TmpDataBuffer; // pointer to new output buffer
        } // allocate output buffer

        // transform data from Orientation to orientation 1
        if (TestBit>1) printf(" --- orientation_float_2d\n");
        orientation_float_2d( pdatain, pdataout,
                              pihb->Dim[1], pihb->Dim[2],
                              Orientation,
                              pstatus );
        if (*pstatus!=Success) {
          if (TmpDataBuffer) { free(TmpDataBuffer); TmpDataBuffer = NULL; } 
          return;
        }
        ptmp = pdatain; pdatain = pdataout; pdataout = ptmp;
      } // if (!Simulate)

      // update header
      if (TestBit>1) printf(" --- orientation_header_2d\n");
      if (!orientation_header_2d ( pihb, Orientation )) {
        if (TmpDataBuffer) { free(TmpDataBuffer); TmpDataBuffer = NULL; }
        *pstatus=Failed; return; }
    } // Orientation of data array

    if (pihb->VarDat) {
      pvardatin  = pihb->VarDat;
      pvardatout = (float *) NULL; // currently no separate output buffer

      if (Orientation > 1l) {
        if (!Simulate) {
 
          if (!pvardatout) { // allocate output buffer
            if (TestBit>1) 
             printf(" --- allocate memory for orientation of variance array\n");
            TmpVarDatBufferSize = sizeof(float)*pihb->Dim[1]*pihb->Dim[2];
            TmpVarDatBuffer = (float *) malloc ( TmpVarDatBufferSize );
            if (!TmpVarDatBuffer) { 
              *pstatus=NotEnoughMemoryAvailable;
              if (TmpDataBuffer) { free(TmpDataBuffer); TmpDataBuffer = NULL; }
              return; 
            }
            pvardatout = TmpVarDatBuffer; // pointer to new output buffer
          } // allocate output buffer
 
          // transform variance data from Orientation to orientation 1
          if (TestBit>1) printf(" --- orientation_float_2d variance array\n");
          orientation_float_2d( pvardatin, pvardatout,
                                pihb->Dim[1], pihb->Dim[2],
                                Orientation,
                                pstatus );
          if (*pstatus!=Success) {
            if (TmpVarDatBuffer) { 
              free(TmpVarDatBuffer); TmpVarDatBuffer = NULL; }
            if (TmpDataBuffer)   { 
              free(TmpDataBuffer); TmpDataBuffer = NULL; }
            return;
          }
          ptmp = pvardatin; pvardatin = pvardatout; pvardatout = ptmp;
        } // if (!Simulate)
 
      } // Orientation of variance array
    }

    // Program operations
    if (Val.I) {
      if (!Simulate) {
        if (TestBit>1) printf(" --- prog_data_2d\n");
        prog_data_2d( pdatain, pvardatin, pihb->Dim[1], pihb->Dim[2],
                      pihb->Dummy.V, pihb->DDummy.V,
                      Val.V, pstatus );
        if (*pstatus!=Success) {
          if (TmpVarDatBuffer) { 
            free(TmpVarDatBuffer); TmpVarDatBuffer = NULL; }
          if (TmpDataBuffer)   { 
            free(TmpDataBuffer); TmpDataBuffer = NULL; } 
          return;
        }
      } // if (!Simulate)
    }

    if (pihb->VarDat) {
      // Program operations variance data
      if (ErrVal.I) {
        if (!Simulate) {
          if (TestBit>1) printf(" --- prog_vardat_2d\n");
          prog_vardat_2d( pdatain, pvardatin, pihb->Dim[1], pihb->Dim[2],
                          pihb->Dummy.V, pihb->DDummy.V,
                          ErrVal.V, pstatus );
          if (*pstatus!=Success) {
            if (TmpVarDatBuffer) { 
              free(TmpVarDatBuffer); TmpVarDatBuffer = NULL; }
            if (TmpDataBuffer) { 
              free(TmpDataBuffer); TmpDataBuffer = NULL; } 
            return;
          }
        } // if (!Simulate)
      }
    }

    // Scalar operations
    if (Factor.I || Const.I ) {
      if (!Simulate) {
        if (TestBit>1) printf(" --- transform_float_2d\n");
        transform_float_2d( pdatain, pihb->Dim[1], pihb->Dim[2],
                            pihb->Dummy.V, pihb->DDummy.V,
                            Factor.V, Const.V );
        if (pihb->VarDat) {
          if (TestBit>1) printf(" --- transform_float_2d variance array\n");
          transform_float_2d( pvardatin, pihb->Dim[1], pihb->Dim[2],
                              VarDummy, DDSET(VarDummy),
                              Factor.V*Factor.V, 0.0 );
        }
      } // if (!Simulate)
    } // Scalar operations

    if (!Simulate) {
      if (pdatain!=pihb->Data) { // data is still in temporary buffer
        // copy temporary buffer to image array
        if (TestBit>1) printf(" --- copy temporary buffer to image array\n");
        memcpy((void *) pihb->Data, (void *) TmpDataBuffer, TmpDataBufferSize); 
      }
      if (pihb->VarDat) {
        if (pvardatin!=pihb->VarDat) { // data is still in temporary buffer
          // copy temporary buffer to variance array
          if (TestBit>1) 
            printf(" --- copy temporary buffer to variance array\n");
          memcpy((void *) pihb->VarDat, 
            (void *) TmpVarDatBuffer, TmpVarDatBufferSize); 
        }
      }

      // release temporary buffers
      if (TmpVarDatBuffer) { 
        free(TmpVarDatBuffer); TmpVarDatBuffer = NULL; }
      if (TmpDataBuffer) { 
        free(TmpDataBuffer); TmpDataBuffer = (float *) NULL; }
    } // if (!Simulate)

} // condition_2d

/*--------------------------------------------------------------------------
  set_output_header_defaults
  --------------------------------------------------------------------------*/
void set_output_header_defaults( ImgHeadBlk * ihb )
{
        /* standard SAXS header values */
        if (!ihb->Dummy.I)     {ihb->Dummy.V = 0.0;ihb->Dummy.I = TRUE;}
        if (!ihb->DDummy.I)    {ihb->DDummy.V = DDSET(ihb->Dummy.V);
                               ihb->DDummy.I=TRUE;}
        if (!ihb->Offset[1].I) {ihb->Offset[1].V=0.0;ihb->Offset[1].I=TRUE;}
        if (!ihb->Offset[2].I) {ihb->Offset[2].V=0.0;ihb->Offset[2].I=TRUE;}
        /* The two following lines must be uncommented to write always
           the BinSize key */
        // ihb->BinSiz[1].I=TRUE;
        // ihb->BinSiz[2].I=TRUE;

        /* optional SAXS header values
        if (!ihb->BinSiz[1].I)     ;
        if (!ihb->BinSiz[2].I)     ;
        if (!ihb->Center[1].I)     ;
        if (!ihb->Center[2].I)     ;
        if (!ihb->PixSiz[1].I)     ;
        if (!ihb->PixSiz[2].I)     ;
        if (!ihb->SampleDistance.I);
        if (!ihb->WaveLength.I)    ;
        if (!ihb->DetRot[1].I)     ;
        if (!ihb->DetRot[2].I)     ;
        if (!ihb->DetRot[3].I)     ;
        if (!ihb->Title.I)         ;
        if (!ihb->Time.I)          ; 
        if (!ihb->Orientation.I)   ;
        if (!ihb->Projection.I)    ;
        if (!ihb->AxisType[1].I)   ;
        if (!ihb->AxisType[2].I)   ;
        */

} /*  set_output_header_defaults */

/*+++------------------------------------------------------------------------
1 geo_header_copy

2 DESCRIPTION
  Copies the geometrical keywords from pin to pout
----------------------------------------------------------------------------*/
void geo_header_copy ( ImgHeadBlk *pin, ImgHeadBlk *pout ) 
{ register int i;

  for (i=0;i<3;i++) {
    pout->Offset[i].I   = (pin->Offset[i].I)   ?  TRUE : FALSE;
    pout->Center[i].I   = (pin->Center[i].I)   ?  TRUE : FALSE;
    pout->BinSiz[i].I   = (pin->BinSiz[i].I)   ?  TRUE : FALSE;
    pout->PixSiz[i].I   = (pin->PixSiz[i].I)   ?  TRUE : FALSE;
    pout->AxisType[i].I = (pin->AxisType[i].I) ?  TRUE : FALSE; }
  pout->SampleDistance.I = (pin->SampleDistance.I) ?  TRUE : FALSE;
  pout->WaveLength.I =  (pin->WaveLength.I) ?  TRUE : FALSE;
  for (i=0;i<4;i++) {
    pout->DetRot[i].I = (pin->DetRot[i].I) ?  TRUE : FALSE; }
  pout->Orientation.I = (pin->Orientation.I) ? TRUE : FALSE;
  pout->Projection.I = (pin->Projection.I) ? TRUE : FALSE;

  for (i=0;i<3;i++) {
    pout->Offset[i].V = pin->Offset[i].V;
    pout->Center[i].V = pin->Center[i].V;
    pout->BinSiz[i].V = pin->BinSiz[i].V;
    pout->PixSiz[i].V = pin->PixSiz[i].V;
    pout->AxisType[i].V = pin->AxisType[i].V; }
  pout->SampleDistance.V = pin->SampleDistance.V;
  pout->WaveLength.V =  pin->WaveLength.V;
  for (i=0;i<4;i++) {
    pout->DetRot[i].V = pin->DetRot[i].V; }
  pout->Orientation.V = pin->Orientation.V;
  pout->Projection.V = pin->Projection.V;

} /* geo_header_copy */

/*+++------------------------------------------------------------------------
1 template_header_block

2 DESCRIPTION
  Use the image header block *pin as a template for *pout.
  The output header block must have been initialized with
  init_header_block.

  return( void )                :

  ImgHeadBlk *pin       (i)     : input image header block
  ImgHeadBlk *pout      (o)     : output image header block


2 HISTORY
  12-Feb-1995   Peter Boesecke creation
  22-Feb-1995   PB      Copy renamed in Template
  26-Feb-1995   PB      Factor and Const added
  05-Mar-1995   PB      LoopCount added
  08-Sep-1995   PB      Input .I flags checked with if (xxx.I).
                        All output .I flags are set to TRUE if input flag
                        is set.
  04-Dec-1995   Stream and StreamOpenMode
  07-Dec-1995   geo_header_copy
  12-Jul-2001   ActualLoopCount
  28-Aug-2001   Write
  09-Jul-2005   Val
----------------------------------------------------------------------------*/
void template_header_block ( long int ImageNumber, 
                             long int LoopCount, long int ActualLoopCount,
                             size_t ImgByteLen, long int Dim[], float * Data,
                             ImgHeadBlk *pin, ImgHeadBlk *pout )
{ register int i;
  /* control header */
  /*  LoopCount and ActualLoopCount not changed */
  pout->ImNum =         ImageNumber;
  pout->Stream =        -1;
  pout->StreamOpenMode =0l;
  pout->LoopCount =      LoopCount;
  pout->ActualLoopCount=ActualLoopCount;
  pout->ImgByteLen =    ImgByteLen;
  for (i=0;i<3;i++) { pout->Dim[i] = Dim[i]; }
  pout->Data =          Data;
  pout->DoNotScale =    pin->DoNotScale;
  pout->Header.I =      (pin->Header.I) ? TRUE : FALSE;
  CopyLine(pin->Header.V,pout->Header.V,IO_len,0);

  /* transformation parameters are reset to defaults */
  pout->rori =          1l;
  pout->MinClip.I =     FALSE; pout->MinClip.V = -FLT_MAX;
  pout->MaxClip.I =     FALSE; pout->MaxClip.V =  FLT_MAX;
  pout->Const.I =       FALSE; pout->Const.V = 0.0;
  pout->Factor.I =      FALSE; pout->Factor.V = 1.0;
  pout->Val.I =         FALSE; pout->Val.V=NULL;
  pout->ErrVal.I =      FALSE; pout->ErrVal.V=NULL;
  pout->Bin[0].I =      FALSE; pout->Bin[0].V = 2l;
  pout->Bin[1].I =      FALSE; pout->Bin[1].V = 1l;
  pout->Bin[2].I =      FALSE; pout->Bin[2].V = 1l;
  pout->Gnaw[0].I =     FALSE; pout->Gnaw[0].V = 2l;
  pout->Gnaw[1].I =     FALSE; pout->Gnaw[1].V = 0l;
  pout->Gnaw[2].I =     FALSE; pout->Gnaw[2].V = 0l;
  pout->Shift[0].I =    FALSE; pout->Shift[0].V = 2.0;
  pout->Shift[1].I =    FALSE; pout->Shift[1].V = 0.0;
  pout->Shift[2].I =    FALSE; pout->Shift[2].V = 0.0;

  /* SAXS keywords */
  pout->Dummy.I =       (pin->Dummy.I) ?  TRUE : FALSE;
  pout->DDummy.I =      (pin->DDummy.I) ?  TRUE : FALSE;
  pout->Title.I =       (pin->Title.I) ?  TRUE : FALSE;
  pout->Time.I =        (pin->Time.I) ? TRUE : FALSE;

  pout->Dummy.V =       pin->Dummy.V;
  pout->DDummy.V =      pin->DDummy.V;

  strncpy(pout->Title.V,pin->Title.V,IO_len);
  strncpy(pout->Time.V,pin->Time.V,IO_len);

  geo_header_copy(pin,pout);

} /* template_header_block */

/*+++------------------------------------------------------------------------
1 balance_header_block

2 DESCRIPTION
  Fill the image header block *pihb with all set values of the image block
  *pib.
  return( void )                :

  CmdBlk     *pcb               : control block
  long int      ImageNumber (i)	: image number;
  ImgBlk     *pib       (i)     : input image block
  ImgHeadBlk *pihb      (i/o)   : image header block
  int        *pstatus   (o)     : output status

2 HISTORY
  12-Feb-1995   Peter Boesecke creation
  21-Sep-1995   Offset set to defaults (0.0) if not set in ihb.
  08-Oct-1995   .I flags -TRUE (-1), if set by image block, others unchanged.
  04-Dec-1995   Actual Stream and StreamOpenMode copied from image block
  02-Feb-2003   ImageNum -> num[blkno], *pib->ib[blkno], *pihb -> ihb[blkno]
  03-Feb-2003   *pstatus, replacekey, pcb and pstatus added to parameter list
  04-Feb-2003   MinClip, MaxClip, Const, Factor
  06-Feb-2003   floatkeyvalue, linekeyvalue
  07-Feb-2003   Bin[3], Gnaw[3] 
  08-Feb-2003   Dummy, DDummy, Dim[3]
  25-May-2003   Offset[3], Center[3], PixSiz[3], WaveLength, SampleDistance
  29-May-2003   rori is only set when set in image block
  04-Oct-2004   DetRot[4]
  09-Jul-2005   Val, ErrVal
 
  ----------------------------------------------------------------------------*/
void balance_header_block ( CmdBlk *pcb, int blkno, long num[], 
                            ImgBlk ib[], ImgHeadBlk ihb[], int *pstatus )
{
  register int i;

  ImgBlk       *pib;
  ImgHeadBlk   *pihb;

  char tmp_line[IO_len];
  char valnam[IO_len];
  char errvalnam[IO_len];

  int  errval;
 
  *pstatus = Success;

  sprintf(valnam,"oval");
  if (blkno!=0) { sprintf(valnam,"i%uval",blkno); }
  sprintf(errvalnam,"oerr");
  if (blkno!=0) { sprintf(errvalnam,"i%uerr",blkno); }

  pib  = &(ib[blkno]);
  pihb = &(ihb[blkno]);

  pihb->ImNum = num[blkno];

  /* copy actual stream and actual stream open mode */
  pihb->Stream = pib->Stream;
  pihb->StreamOpenMode = pib->StreamOpenMode;

  for (i=0;i<3;i++)
    if (pib->Dim[i].I) {
      pihb->Dim[i] = longkeyvalue( pib->Dim[i].V,
                        pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return;
    }

  /* image transformation parameters */
  if (pib->MinClip.I) {
    pihb->MinClip.V = floatkeyvalue( pib->MinClip.V,
                      pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return;
    pihb->MinClip.I = (-TRUE); 
  }
  if (pib->MaxClip.I) {
    pihb->MaxClip.V = floatkeyvalue( pib->MaxClip.V,
                      pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return;
    pihb->MaxClip.I = (-TRUE); 
  }
  if (pib->Const.I) {
    pihb->Const.V = floatkeyvalue( pib->Const.V,
                      pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return;
    pihb->Const.I = (-TRUE);
  }
  if (pib->Factor.I) {
    pihb->Factor.V = floatkeyvalue( pib->Factor.V,
                      pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return;
    pihb->Factor.I = (-TRUE);
  }
  if (pib->Val.I) {
    linekeyvalue( tmp_line, IO_len, pib->Val.V, 1,
                  pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return;

    EditLine(Collapse | UnComment, tmp_line);
    pihb->Val.V = 
      num_str2prog ( valnam, tmp_line, NULL, &errval, 1, valnam );
    *pstatus=errval2status( errval );
    if (*pstatus!=Success) {
      SetSaxsErrorMessage( pib->Val.V );
      return;
    }
    pihb->Val.I = (-TRUE);
  }
  if (pib->ErrVal.I) {
    linekeyvalue( tmp_line, IO_len, pib->ErrVal.V, 1,
                  pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return;

    EditLine(Collapse | UnComment, tmp_line);

    pihb->ErrVal.V =
      num_str2prog ( errvalnam, tmp_line, NULL, &errval, 2, valnam, errvalnam );
    *pstatus=errval2status( errval );
    if (*pstatus!=Success) {
      SetSaxsErrorMessage( pib->ErrVal.V );
      return;
    }

    pihb->ErrVal.I = (-TRUE);
  }
  for (i=0;i<3;i++) 
    if (pib->Bin[i].I) {
      pihb->Bin[i].V = longkeyvalue( pib->Bin[i].V,
                        pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return;
      pihb->Bin[i].I = (-TRUE);
    }
  for (i=0;i<3;i++)
    if (pib->Gnaw[i].I) {
      pihb->Gnaw[i].V = longkeyvalue( pib->Gnaw[i].V,
                        pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return;
      pihb->Gnaw[i].I = (-TRUE);
    }
  if (blkno==0) // output block only
    for (i=0;i<3;i++)
      if (pcb->Shift[i].I) {
        pihb->Shift[i].V = floatkeyvalue( pcb->Shift[i].V,
                           pcb, ib, num, blkno, pstatus );
        if (*pstatus!=Success) return;
        pihb->Shift[i].I = (-TRUE);
      }

  /* Dummy and DDummy must always be set. If not set use default values for
     Dummy and DDummy and set .I flags to (+TRUE).
     Check whether DDummy has a sound number.
     Result overwritten and .I flags are set to (-TRUE) if
     values for Dummy and DDummy are found in the image block pib */
  /*    Set defaults */
  if (!pihb->Dummy.I) { /* if Dummy is not set, recalculate DDummy */
      pihb->Dummy.V = 0; pihb->DDummy.V = DDSET( 0 );
      pihb->Dummy.I = pihb->DDummy.I = TRUE; }
  if (!pihb->DDummy.I) { /* if DDummy is still not set, set it */
    pihb->DDummy.V = DDSET( pihb->Dummy.V ); pihb->DDummy.I=TRUE;
    } else { /* if DDummy is set, check it */
    pihb->DDummy.V = MAX2( DDSET( pihb->Dummy.V ), pihb->DDummy.V );
    }
  /*    Overwrite with image block values */
  if (pib->Dummy.I) { /* if new Dummy is used, recalculate DDummy */
    pihb->Dummy.V = floatkeyvalue( pib->Dummy.V,
                      pcb, ib, num, blkno, pstatus ); 
    if (*pstatus!=Success) return;
    pihb->DDummy.V = DDSET( pihb->Dummy.V );
    pihb->Dummy.I = pihb->DDummy.I = (-TRUE); }
  if (pib->DDummy.I) { /* if one really wants another one, OK */
    pihb->DDummy.V = floatkeyvalue( pib->DDummy.V,
                      pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return;
    pihb->DDummy.I = (-TRUE); }

  /* Offset[1] and Offset[2] must always be set, set default values,
     and flags to (+TRUE) if not set */
  /*    Set defaults */
  for (i=0;i<3;i++) {
    if (!pihb->Offset[i].I) {
      pihb->Offset[i].V = 0.0;
      pihb->Offset[i].I = TRUE; } /* This value is not taken from the image
                                     block, therefore not (-TRUE) */
    }
  /*    Overwrite with image block values */
  for (i=0;i<3;i++) {
    if (pib->Offset[i].I) {
      pihb->Offset[i].V = floatkeyvalue( pib->Offset[i].V,
                            pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return;
      pihb->Offset[i].I = (-TRUE); }
    if (pib->Center[i].I) {
      pihb->Center[i].V = floatkeyvalue( pib->Center[i].V,
                            pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return;
      pihb->Center[i].I = (-TRUE); }
    if (pib->BinSiz[i].I) {
      pihb->BinSiz[i].V = floatkeyvalue( pib->BinSiz[i].V,
                            pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return; 
      pihb->BinSiz[i].I = (-TRUE); }
    if (pib->PixSiz[i].I) {
      pihb->PixSiz[i].V = floatkeyvalue( pib->PixSiz[i].V,
                            pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return; 
      pihb->PixSiz[i].I = (-TRUE); }
    if (pib->AxisType[i].I) {
      pihb->AxisType[i].V = pib->AxisType[i].V;
      if (*pstatus!=Success) return;
      pihb->AxisType[i].I = (-TRUE); }
    }
  if (pib->SampleDistance.I) {
      pihb->SampleDistance.V = floatkeyvalue( pib->SampleDistance.V,
                            pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return;
    pihb->SampleDistance.I = (-TRUE); }
  if (pib->WaveLength.I) {
      pihb->WaveLength.V = floatkeyvalue( pib->WaveLength.V,
                            pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return;
    pihb->WaveLength.I = (-TRUE); }
  for (i=0;i<4;i++) {
    if (pib->DetRot[i].I) {
      pihb->DetRot[i].V = floatkeyvalue( pib->DetRot[i].V,
                            pcb, ib, num, blkno, pstatus );
      if (*pstatus!=Success) return;
      pihb->DetRot[i].I = (-TRUE); }
    }
  if (pib->Title.I) {
    linekeyvalue ( pihb->Title.V, IO_len, pib->Title.V, 0,
                   pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return; 
    pihb->Title.I = (-TRUE); }
  if (pib->Time.I) {
    linekeyvalue ( pihb->Time.V, IO_len, pib->Time.V, 0,
                   pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return; 
    pihb->Time.I = (-TRUE); }
  if (pib->Orientation.I) {
    // orientationkeyvalue ++++++++++
    pihb->Orientation.V = longkeyvalue( pib->Orientation.V,
                      pcb, ib, num, blkno, pstatus );
    if (*pstatus!=Success) return;
    pihb->rori = pihb->Orientation.V;
    pihb->Orientation.I = (-TRUE); }
  if (pib->Projection.I) {
    pihb->Projection.V = pib->Projection.V;
    if (*pstatus!=Success) return;
    pihb->Projection.I = (-TRUE); }

} /* balance_header_block */

/*+++------------------------------------------------------------------------
1 BalanceHeaderBlock 

2 DESCRIPTION
  Public interface of balance_header_block

2 HISTORY
  2005-02-11 PB  creation
----------------------------------------------------------------------------*/
void BalanceHeaderBlock ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                          long num[], ImgHeadBlk ihb[], int *pstatus )
{  balance_header_block ( pcb, blkno, num, ib, ihb, pstatus );
} /* BalanceHeaderBlock */

/*+++------------------------------------------------------------------------
1 init_header_block

2 DESCRIPTION
  Sets all values of a header to default values. Must be used to initialize
  a new input header or a new output header.
  Sets header control to 0, all .I flags to FALSE and all .V to 0.
  return( void )                :

  ImgHeadBlk *pihb      (i/o)   : image header block
  int        blkno              : block number

2 HISTORY

----------------------------------------------------------------------------*/
void   init_header_block( ImgHeadBlk ihb[], int blkno )
{ register int i;

  ImgHeadBlk *pihb;

  pihb=&(ihb[blkno]);

  /* header control */
  pihb->I          = 0; /* header not valid */
  pihb->blkno      = blkno;
  pihb->ImNum      = 0l;
  pihb->Stream     = -1;
  pihb->StreamOpenMode   = 0l;
  pihb->LoopCount  = 0l;
  pihb->ActualLoopCount = 0l;
  pihb->ImgByteLen = (size_t) NULL;
  pihb->Dim[0]     = pihb->Dim[1] = pihb->Dim[2] = 0;
  pihb->Data       = (float *) NULL;
  pihb->VarDat     = (float *) NULL;
  pihb->DoNotScale = FALSE;
  pihb->Header.I   = FALSE;
  *(pihb->Header.V)= '\0';

  /* transformation parameters */
  pihb->rori       = 1l;
  pihb->MinClip.I  = FALSE; pihb->MinClip.V = -FLT_MAX;
  pihb->MaxClip.I  = FALSE; pihb->MaxClip.V =  FLT_MAX;
  pihb->Const.I    = FALSE; pihb->Const.V = 0.0;
  pihb->Factor.I   = FALSE; pihb->Factor.V = 1.0; 
  pihb->Val.I      = FALSE; pihb->Val.V = NULL; 
  pihb->ErrVal.I   = FALSE; pihb->ErrVal.V = NULL; 
  pihb->Bin[0].I   = FALSE; pihb->Bin[0].V = 2l;
  pihb->Bin[1].I   = FALSE; pihb->Bin[1].V = 1l;
  pihb->Bin[2].I   = FALSE; pihb->Bin[2].V = 1l;
  pihb->Gnaw[0].I  = FALSE; pihb->Gnaw[0].V = 2l;
  pihb->Gnaw[1].I  = FALSE; pihb->Gnaw[1].V = 0l;
  pihb->Gnaw[2].I  = FALSE; pihb->Gnaw[2].V = 0l;
  pihb->Shift[0].I = FALSE; pihb->Shift[0].V = 2.0;
  pihb->Shift[1].I = FALSE; pihb->Shift[1].V = 0.0;
  pihb->Shift[2].I = FALSE; pihb->Shift[2].V = 0.0;

  /* SAXS keywords */
  pihb->Dummy.I    = FALSE;
  pihb->DDummy.I   = FALSE;
  for (i=0;i<3;i++) {
    pihb->Offset[i].I = FALSE;
    pihb->Center[i].I = FALSE;
    pihb->BinSiz[i].I = FALSE;
    pihb->PixSiz[i].I = FALSE;
    pihb->AxisType[i].I = FALSE;
    } /* for */
  pihb->SampleDistance.I = FALSE;
  pihb->WaveLength.I = FALSE;
  for (i=0;i<4;i++) {
    pihb->DetRot[i].I = FALSE;
    } /* for */
  pihb->Title.I    = FALSE;
  pihb->Time.I     = FALSE;
  pihb->Orientation.I = FALSE;
  pihb->Projection.I = FALSE;

  /* SAXS keywords */
  pihb->Dummy.V    = 0.0;
  pihb->DDummy.V   = 0.0;
  for (i=0;i<3;i++) {
    pihb->Offset[i].V = 0.0;
    pihb->Center[i].V = 0.0;
    pihb->BinSiz[i].V = 1.0;
    pihb->PixSiz[i].V = 1.0;
    pihb->AxisType[i].V = IO_AxisTypeDistance; // default Distance
    } /* for */
  pihb->SampleDistance.V = 1.0;
  pihb->WaveLength.V = 1.0;
  for (i=0;i<4;i++) {
    pihb->DetRot[i].V = 0.0;
    } /* for */
  pihb->Title.V[0] = '\0';
  pihb->Time.V[0] = '\0';
  pihb->Orientation.V = 1l; // default 1
  pihb->Projection.V = IO_ProSaxs; // default Saxs 

} /* init_header_block */

/*---------------------------------------------------------------------------
NAME
  AlreadyOpened

DESCRIPTION
  Returns 1 if the file is already opened otherwise 0.
  ---------------------------------------------------------------------------*/
int AlreadyOpened( CmdBlk *pcb, ImgBlk ib[], int blkno,
                   const char *fname, long int FileNumber, int *pstatus )
{ char fname_pattern[IO_len];

  *pstatus = Success;
  if (ib[blkno].Stream>=0) { // file is opened
    filename_parameter (fname_pattern,IO_len,fname,0);
    filename_pattern (fname_pattern,IO_len,fname_pattern,FileNumber);
    if ( filename_compare( fname_pattern, ib[blkno].FileName ) )
      return(0); else return(1);
  } else { // file is closed
  return(0);
  }

} // AlreadyOpened

/*----------------------------------------------------------------------------
 Private include files
----------------------------------------------------------------------------*/
/* include ImageLoop */
# include "SaxsImageLoop.inc"

/*+++------------------------------------------------------------------------
1 alloc_image_memory

2 DESCRIPTION
  Allocates Image memory

  ImgHeadBlk *pihb      (i/o)   : image header block
  pihb->LoopCount	(o)	: resetted to 0
  pihb->ActualLoopCount (o)     : resetted to 0
  pihb->Data[3]	 	(o)	: pointer to allocated memory
  pihb->Dim		(o)	: size of allocated memory

  float initdval        (i)     : initialization value of data array
  int initdflag		(i)	: TRUE initialization of data array
                                  FALSE no initialization
  int var               (i)     : TRUE allocate variance array
                                  FALSE do not allocate variance array
  float initvval        (i)     : initialization value of variance array
  int initvflag		(i)	: TRUE initialization of variance array
                                  FALSE no initialization

2 HISTORY
  15-Feb-1995   Peter Boesecke creation
  02-Mar-1995   PB LoopCount
  08-Mar-1995   PB initialization
  28-Feb-1996   PB ImgByteLen==0 returns Data<-NULL and Dim[0], [1], [2]
  12-Jul-2001   PB ActualLoopCount
  28-Aug-2001   PB pihb->Write
  05-Aug-2005   PB VarDat

----------------------------------------------------------------------------*/
void alloc_image_memory ( CmdBlk *pcb, long int Dim[], ImgHeadBlk *pihb,
                          float initdval, int initdflag, int var,
                          float initvval, int initvflag, int *pstatus )
{ register long int i;
  float *pData;

  *pstatus=Success;

  pihb->LoopCount = 0l;
  pihb->ActualLoopCount = 0l;
  pihb->ImgByteLen = sizeof(float) * Dim[1] * Dim[2];
  if (pihb->ImgByteLen) {
    /* allocate data array */
    pihb->Data = (float *) malloc (pihb->ImgByteLen);

    if (pihb->Data==NULL) {
      *pstatus=NotEnoughMemoryAvailable; return; 
    }

    pihb->Dim[0] = 2; pihb->Dim[1] = Dim[1]; pihb->Dim[2] = Dim[2];

    if (initdflag) {
      /* initialize data array */
      pData = pihb->Data;
      for (i=0;i<(Dim[1]*Dim[2]);i++) *(pData++)=initdval;
    }

    if (var) {
      /* allocate variance array */
      pihb->VarDat = (float *) malloc (pihb->ImgByteLen);
 
      if (pihb->VarDat==NULL) {
        *pstatus=NotEnoughMemoryAvailable; return;
      }
      if (initvflag) {
        /* initialize variance array */
        pData = pihb->VarDat;
        for (i=0;i<(Dim[1]*Dim[2]);i++) *(pData++)=initvval;
      }
    }

  } else {
    /* no memory to allocate */
    pihb->Data = (float *) NULL;
    pihb->Dim[0] = 2; pihb->Dim[1] = Dim[1]; pihb->Dim[2] = Dim[2]; 
    pihb->VarDat = (float *) NULL;
  }

  *pstatus = Success;

} /* alloc_image_memory */

/*+++------------------------------------------------------------------------
1 write_image_header

2 DESCRIPTION
  Writes the values of image header block *pihb

  return (void)
  ImgHeadBlk *pihb (i)          : image header block to be written
  long int pihb->ImNum (i)      : image number
  int *pErrorValue (o)          : error value
  int *pstatus (o)              : SAXS status

2 HISTORY
  15-Feb-1995   Peter Boesecke
  18-Mar-2000   PB ImageChain as parameter
  19-Aug-2001   ImageNumber removed from arglist because it 
                is given by pihb->ImNum 
  17-Sep-2001   Orientation: pihb->Header.I
  29-Sep-2004   Projection
  01-Dec-2007   AxisType
  ---------------------------------------------------------------------------*/
void write_image_header ( int stream, int ImageChain,
                          ImgHeadBlk * pihb, int *pErrorValue,
			  int * pstatus )
{       long int ImageNumber = pihb->ImNum;   /* Image Number */
 
        *pstatus = Success; /* edf_ routines */

        set_output_header_defaults( pihb );

        /* SaxsDataVersion */
        edf_write_header_line( stream, ImageNumber, ImageChain,
          KSaxsDataVersion, SAXSDATAVERSION, pErrorValue , pstatus );
        if (*pstatus!=Success) return;

        /* Dummy */
        if (pihb->Dummy.I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
                   KDummy, (pihb->Dummy.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* DDummy */
        if (pihb->DDummy.I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
                 KDDummy, (pihb->DDummy.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Offset_1 */
        if (pihb->Offset[1].I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KOffset_1, (pihb->Offset[1].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Offset_2 */
        if (pihb->Offset[2].I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KOffset_2, (pihb->Offset[2].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Center_1 */
        if (pihb->Center[1].I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KCenter_1, (pihb->Center[1].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Center_2 */
        if (pihb->Center[2].I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KCenter_2, (pihb->Center[2].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* BinSiz_1 */
        if ((pihb->BinSiz[1].I)||(pihb->BinSiz[1].V!=1)||
            (pihb->BinSiz[2].V!=1)) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KBinSiz_1, (pihb->BinSiz[1].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }
 
        /* BinSiz_2 */
        if ((pihb->BinSiz[2].I)||(pihb->BinSiz[2].V!=1)||
            (pihb->BinSiz[1].V!=1)) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KBinSiz_2, (pihb->BinSiz[2].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* PixSiz_1 */
        if (pihb->PixSiz[1].I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KPixSiz_1, (pihb->PixSiz[1].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* PixSiz_2 */
        if (pihb->PixSiz[2].I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KPixSiz_2, (pihb->PixSiz[2].V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* SampleDistance */
        if (pihb->SampleDistance.I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KSampleDistance, (pihb->SampleDistance.V),
                   pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* WaveLength */
        if (pihb->WaveLength.I) {
          edf_write_header_float( stream, ImageNumber, ImageChain,
            KWaveLength, (pihb->WaveLength.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* DetRot_1 */
        if (pihb->DetRot[1].I) {
          edf_write_header_unit( stream, ImageNumber, ImageChain,
            KDetRot_1, (pihb->DetRot[1].V), "deg", pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* DetRot_2 */
        if (pihb->DetRot[2].I) {
          edf_write_header_unit( stream, ImageNumber, ImageChain,
            KDetRot_2, (pihb->DetRot[2].V), "deg", pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* DetRot_3 */
        if (pihb->DetRot[3].I) { 
          edf_write_header_unit( stream, ImageNumber, ImageChain,
            KDetRot_3, (pihb->DetRot[3].V), "deg", pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Title */
        if (pihb->Title.I) {
          edf_write_header_line( stream, ImageNumber, ImageChain,
            KTitle, (pihb->Title.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Time */
        if (pihb->Time.I) {
          edf_write_header_line( stream, ImageNumber, ImageChain,
            KTime, (pihb->Time.V), pErrorValue, pstatus );
          if (*pstatus!=Success) return; }

        /* Orientation */
        if (pihb->Orientation.I) 
          if ( (pihb->Orientation.V > 1l) || (pihb->Header.I) ) { 
            edf_write_header_long( stream, ImageNumber, ImageChain,
              KOrientation, (pihb->Orientation.V), pErrorValue, pstatus );
            if (*pstatus!=Success) return; }

        /* Projection */
        if (pihb->Projection.I)
          if ( (pihb->Projection.V > IO_ProSaxs) || (pihb->Header.I) ) {
            edf_write_header_line( stream, ImageNumber, ImageChain,
              KProjection, protostr(pihb->Projection.V), pErrorValue, pstatus );
            if (*pstatus!=Success) return; }

        /* AxisType_1 */
        if (pihb->AxisType[1].I)
          if ( (pihb->AxisType[1].V>IO_AxisTypeDistance)||(pihb->Header.I) ) {
            edf_write_header_line( stream, ImageNumber, ImageChain,
              KAxisType_1,axistostr(pihb->AxisType[1].V),pErrorValue,pstatus );
            if (*pstatus!=Success) return; }

        /* AxisType_2 */
        if (pihb->AxisType[2].I)
          if ( (pihb->AxisType[2].V>IO_AxisTypeDistance)||(pihb->Header.I) ) {
            edf_write_header_line( stream, ImageNumber, ImageChain,
              KAxisType_2,axistostr(pihb->AxisType[2].V),pErrorValue,pstatus );
            if (*pstatus!=Success) return; }
 
        /* History */
        edf_write_header_history ( stream, ImageNumber, ImageChain, 
                                SaxsHistoryKey, pErrorValue, pstatus );

	*pstatus = Success;

} /* End of write_image_header */

/*----------------------------------------------------------------------------
read_saxs_image

DESCRIPTION
The routine returns only FALSE if the image header was not found. In 
all other cases the return value is TRUE.
DontRead	TRUE  : works as ReadImageHeader (ImgByteLen -> 0)
		FALSE : works as ReadImage
IO_DontReadWrite : header will not be read (only balancing with image block)
IO_DontReadData  : binary data will not be read (ImgByteLen -> 0)
IO_Dummy         : header will not be read (only balancing with image block),
                   binary data will be allocated if DontRead is not set

HISTORY
2000-03-18  PB ImageChain = ib[blkno].Memory.V; 
2000-07-30  PB orientation added
2001-07-10  PB returns also 0 if header was not found due to error 
               FileMustBeOpened 
2001-09-16  PB PassHeader 
2002-02-02  PB ImageNumber -> num[blkno]
----------------------------------------------------------------------------*/
int	read_saxs_image		( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  long num[], ImgHeadBlk ihb[], 
                                  int DontRead, int *pstatus)
{
  ImgBlk * pib;
  ImgHeadBlk * pihb;
  int retval;
  int ErrorValue=0;
  int stream;
  int ImageChain, VarChain;
  int read_variance=0;
  long ImageNumber;
  int simulate = TRUE; // do not access data arrays
  long *TmpDim, VarDim[4];
  size_t VarByteLen;
  char tmp_line[IO_len];

  *pstatus = Success; /* edf_ routines */

  pib  = &ib[blkno];
  pihb = &ihb[blkno];
  stream = ib[blkno].Stream;
  ImageChain = ib[blkno].Memory.V;
  VarChain = -ImageChain;
  ImageNumber = num[blkno];

  /* read image header and balance with image block,
     if (image header not found) return(FALSE) */

  init_header_block( ihb, blkno );
  if (pcb->TestBit>1) { printf(" -- init_header_block (blkno=%d)\n",blkno);
    PrintImageHeaderBlock(&ihb[blkno]); }

  /* Image Number */
  pihb->ImNum = ImageNumber;

  /* ImgByteLen */
  pihb->ImgByteLen=(size_t) NULL; 
  pihb->Data=(float*) NULL; /* no data available */

  // Set interpolation mode
  if (pcb->AntiAliased.V) Ipolmode ( IPOL_ANTIALIASED  );
  else Ipolmode ( IPOL_NORMAL );

  // Set weight method
  if (pcb->VarianceWeight.V) Ipolweight ( IPOL_WEIGHTED  );
  else Ipolweight ( IPOL_EQUAL );

  if (pcb->TestBit>1) 
    printf("Interpolation mode is set to %s.\n",Ipolmode2str(Ipolmode(0)));

  /* --- Read header */
  if ( !(( IO_DontReadWrite | IO_Dummy ) & pib->StreamOpenMode)  ) {

    /* test stream */
    if ( stream < 0 ) {
      *pstatus = FileMustBeOpened;
      return(FALSE); // image header cannot be read because file is not opened
      }

    if (pcb->PassHeader.V) {
      /* create header key */ 
      sprintf(pihb->Header.V,"i%u",blkno);
      /* read full header */
      edf_new_header( pihb->Header.V );
      pihb->Header.I = TRUE;
      edf_read_header( stream, ImageNumber, ImageChain, 
                       pihb->Header.V, &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( pihb->Header.V );
        return(FALSE);
      }
    }

    /* Dim */
    TmpDim=pihb->Dim; TmpDim[0]=2l;
    retval = edf_read_data_dimension( stream, ImageNumber, ImageChain,
                                     &TmpDim, &pihb->ImgByteLen,
                                     &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { /* image not found or other error*/
      pihb->Dim[0]=pihb->Dim[1]=pihb->Dim[2]=0;
      return(retval);
    }

    /* Dummy */
    if (edf_read_header_float( stream, ImageNumber, ImageChain,
          KDummy, &(pihb->Dummy.V), &ErrorValue, pstatus ) ) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KDummy ); 
        return(TRUE); 
      }
      pihb->Dummy.I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* DDummy */
    if (edf_read_header_float( stream, ImageNumber, ImageChain,
          KDDummy, &(pihb->DDummy.V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KDDummy );
        return(TRUE); 
      }
      pihb->DDummy.I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* Offset_1 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KOffset_1,
          &(pihb->Offset[1].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KOffset_1 );
        return(TRUE); 
      }
      pihb->Offset[1].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* Offset_2 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KOffset_2,
          &(pihb->Offset[2].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KOffset_2 );
        return(TRUE); 
      }
      pihb->Offset[2].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* Center_1 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KCenter_1,
          &(pihb->Center[1].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KCenter_1 );
        return(TRUE); 
      }
      pihb->Center[1].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* Center_2 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KCenter_2,
          &(pihb->Center[2].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KCenter_2 );
        return(TRUE); 
      }
      pihb->Center[2].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* BinSiz_1 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KBinSiz_1,
          &(pihb->BinSiz[1].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KBinSiz_1 );
        return(TRUE); 
      }
      pihb->BinSiz[1].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */
 
    /* BinSiz_2 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KBinSiz_2,
          &(pihb->BinSiz[2].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KBinSiz_2 );
        return(TRUE); 
      }
      pihb->BinSiz[2].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* PixSiz_1 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KPixSiz_1,
          &(pihb->PixSiz[1].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KPixSiz_1 );
        return(TRUE); 
      }
      pihb->PixSiz[1].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* PixSiz_2 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KPixSiz_2,
          &(pihb->PixSiz[2].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KPixSiz_2 );
        return(TRUE); 
      }
      pihb->PixSiz[2].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* SampleDistance */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KSampleDistance,
          &(pihb->SampleDistance.V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KSampleDistance );
        return(TRUE); 
      }
      pihb->SampleDistance.I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* WaveLength */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KWaveLength,
          &(pihb->WaveLength.V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KWaveLength );
        return(TRUE); 
      }
      pihb->WaveLength.I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* DetRot_1 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KDetRot_1,
          &(pihb->DetRot[1].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KDetRot_1 );
        return(TRUE); 
      }
      pihb->DetRot[1].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* DetRot_2 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KDetRot_2,
          &(pihb->DetRot[2].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KDetRot_2 );
        return(TRUE); 
      }
      pihb->DetRot[2].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* DetRot_3 */
    if (edf_read_header_float( stream, ImageNumber, ImageChain, KDetRot_3,
          &(pihb->DetRot[3].V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KDetRot_3 );
        return(TRUE); 
      }
      pihb->DetRot[3].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* Title */
    if (edf_read_header_line( stream, ImageNumber, ImageChain, KTitle,
         (pihb->Title.V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KTitle );
        return(TRUE); 
      }
      pihb->Title.I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* Time */
    if (edf_read_header_line( stream, ImageNumber, ImageChain, KTime,
         (pihb->Time.V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KTime );
        return(TRUE); 
      }
      pihb->Time.I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* Orientation */
    // edf_read_header_line + raster_str2number +++++++++++++++ 
    if (edf_read_header_long( stream, ImageNumber, ImageChain, KOrientation,
          &(pihb->Orientation.V), &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KOrientation );
        return(TRUE); 
      }
      pihb->Orientation.I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* Projection */
    if (edf_read_header_line( stream, ImageNumber, ImageChain, KProjection,
         tmp_line, &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { 
        SetSaxsErrorMessage( KProjection );
        return(TRUE); 
      }
      pihb->Projection.V = strtopro ( tmp_line );
      pihb->Projection.I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* AxisType_1 */
    if (edf_read_header_line( stream, ImageNumber, ImageChain, KAxisType_1,
         tmp_line, &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) {
        SetSaxsErrorMessage( KAxisType_1 );
        return(TRUE);
      }
      pihb->AxisType[1].V = strtoaxis ( tmp_line );
      pihb->AxisType[1].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* AxisType_2 */
    if (edf_read_header_line( stream, ImageNumber, ImageChain, KAxisType_2,
         tmp_line, &ErrorValue, pstatus )) {
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) {
        SetSaxsErrorMessage( KAxisType_2 );
        return(TRUE);
      }
      pihb->AxisType[2].V = strtoaxis ( tmp_line );
      pihb->AxisType[2].I = TRUE;
    } else *pstatus = Success; /* edf_ routines */

    /* History (only for blkno==1) */
    if (blkno==1) {
      if (edf_read_header_history( stream, ImageNumber, ImageChain, 
           SaxsHistoryKey, &ErrorValue, pstatus )) {
        SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
        if (*pstatus!=Success) return(TRUE);
      } else *pstatus = Success; /* edf_ routines */
    }

    *pstatus = Success;
    } /* if IO_DontReadWrite | IO_Dummy*/

  if (pcb->TestBit>1) { printf(" -- edf_read_header_... (blkno=%d)\n",blkno);
    PrintImageHeaderBlock(pihb); }

  balance_header_block ( pcb, blkno, num, ib, ihb, pstatus );
  if (*pstatus!=Success) return(TRUE);

  if (pcb->TestBit>1) { 
    printf(" -- balance_header_block (blkno=%d)\n",blkno);
    PrintImageBlock(pib);
    PrintImageHeaderBlock(pihb); 
  }

  /* Check parameters */
  if ( (pihb->Orientation.V<=0l)||(16l<pihb->Orientation.V) ) {
    *pstatus = InvalidOrientation;
    return(TRUE);
  }

  /* --- Read binary data */
  if ( !( (DontRead) || (IO_DontReadData & pib->StreamOpenMode) || 
          (IO_DontReadWrite  & pib->StreamOpenMode) ) ) {
    /* read image data */
    if (pcb->TestBit>1) 
      printf(" --- read image data ... (blkno=%d)\n",blkno);
    if (IO_Dummy  & pib->StreamOpenMode) { // allocate dummy image 
      if (pcb->TestBit>1) 
        printf(" -- allocate dummy image (blkno=%d)\n",blkno);  
      alloc_image_memory(pcb, pihb->Dim, pihb, pihb->Dummy.V, TRUE, 
                              pcb->CreateVariance.V, 0.0, TRUE, pstatus);
      if (*pstatus!=Success) return(TRUE);
    } else {
      read_variance = 0;
      if ( (pcb->CreateVariance.V)&&(VarChain<0) ) {
        if (pcb->TestBit>1) 
          printf(" -- check for variance block (blkno=%d,img=%ld,mem=%d)\n", 
            blkno,ImageNumber,VarChain);
        read_variance = edf_test_header( ib[blkno].Stream, ImageNumber, 
                        VarChain, &ErrorValue, pstatus );
        SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );

        if (read_variance) { // Check dimension
          /* VarDim */
          TmpDim=VarDim; TmpDim[0]=2l;
          edf_read_data_dimension( stream, ImageNumber, VarChain,
                                   &TmpDim, &VarByteLen,
                                   &ErrorValue, pstatus );
          SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
          if (*pstatus!=Success) { /* error*/
            return(TRUE);
          }
          if ( (VarDim[0]!=2l)||(VarDim[1]!=pihb->Dim[1])||
               (VarDim[2]!=pihb->Dim[2]) ) {
            fprintf(ERROUT,"ERROR: Dimensions of data array and variance array are different\n");
            fprintf(ERROUT,"           %ldd data array[ %ld, %ld]\n",
              pihb->Dim[0],pihb->Dim[1],pihb->Dim[2]);
            fprintf(ERROUT,"       %ldd variance array[ %ld, %ld]\n",
              VarDim[0],VarDim[1],VarDim[2]);
            *pstatus = Failed;
            return(TRUE);
          }
        }
      }

      TmpDim=pihb->Dim; TmpDim[0]=2l;

      if (pcb->TestBit>1) printf(" -- allocate image (blkno=%d)\n",blkno);
      alloc_image_memory(pcb, TmpDim, pihb, 0.0, FALSE, 
                         pcb->CreateVariance.V, 0.0, !(read_variance), pstatus);
      if (*pstatus!=Success) return(TRUE);

      if (pcb->TestBit>1) printf(" -- edf_read_data (blkno=%d)\n",blkno);
      edf_read_data ( stream, ImageNumber, ImageChain,
                      &TmpDim, &pihb->ImgByteLen, 
                      (void**) &(pihb->Data), MFloat, 
                      &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) return(TRUE);

      if ( read_variance ) {
        if (pcb->TestBit>1) 
          printf(" -- edf_read_data variance array (blkno=%d)\n",blkno);
        TmpDim=VarDim; TmpDim[0]=2l;
        edf_read_data ( stream, ImageNumber, VarChain,
                        &TmpDim, &pihb->ImgByteLen,
                        (void**) &(pihb->VarDat), MFloat,
                        &ErrorValue, pstatus );
        SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
        if (*pstatus!=Success) return(TRUE); 
      }
    }
    simulate = FALSE;
  } /* if (!(IO_DontReadData ... ) */

  /* clip, gnaw, rebin, orient and transform image */
  if (pcb->TestBit>1) printf(" --  read_saxs_image/condition_2d (blkno=%d)\n",blkno);
  condition_2d( pihb,
                pihb->MinClip, pihb->MaxClip,
                pihb->Gnaw[1], pihb->Gnaw[2],
                pihb->Bin[1],  pihb->Bin[2],
                pihb->Val,     pihb->ErrVal,
                pihb->Factor,  pihb->Const,
//                pib->Ave,      pihb->Orientation.V,
                pib->Ave,      0l,  // no orientation transformation
                simulate, pcb->TestBit, pstatus );
  if (*pstatus!=Success) return(TRUE);

  pihb->I = TRUE; /* header is valid */

  *pstatus = Success;

  return(TRUE);

} /* read_saxs_image */

/*+++------------------------------------------------------------------------
1 SearchMinMaxImage

2 DESCRIPTION
  Searches the minimum and maximum image number

  return( int ) : (o)   TRUE    : successful
                        FALSE   : not successful
  long int * pMinImage (o)      : minimum image number
  long int * pMaxImage (o)      : maximum image number
  int *pExternalError (o)       : external error value
  int *pstatus (o)		: SAXS status

  If IO_DontReadWrite or IO_Dummy is set, no read is done and the return 
  status is Success and the return value TRUE, *pMinImage -> 1 and 
  *pMaxImage -> 0.
---------------------------------------------------------------------------*/
int SearchMinMaxImage ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                          long int * pMinImage, long int *pMaxImage,
                          int *pstatus)
 { const char * RoutineName = "SearchMinMaxImage";
   int ImageChain = ib[blkno].Memory.V;

   int retval, ErrorValue=0;

   *pstatus = Success; /* edf_ routines */

   if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
   SetSaxsError( ModuleName, RoutineName, NULL );

   if ( ((IO_Dummy | IO_DontReadWrite) & ib[blkno].StreamOpenMode)  ) {
     *pMinImage = 1l; *pMaxImage = 0l;
     retval=TRUE;
   } else {
     
     retval = edf_search_minmax_number(ib[blkno].Stream,ImageChain,
       pMinImage,pMaxImage, &ErrorValue, pstatus);
     SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
   }
   return ( retval );

 } /* SearchMinMaxImage */

/*+++------------------------------------------------------------------------
1 ReadImageDimension

2 DESCRIPTION
  Returns the image dimensions Dim_1 and Dim_2.
  The routine test whether these dimensions are already available in the
  image block ((ib[blkno].Dim[1].V and ib[blkno].Dim[2].V). If yes, no read
  is done and these values are returned. If at least one of the dimensions
  is missing the missing one is read from the file. 

  return( int ) : (o)   TRUE    : dimensions are returned
                        FALSE   : no dimensions are returned
  CmdBlk * pcb (i)              : command block
  ImgHeadBlk ihb[] (i)          : image header blocks
  int blkno (i)                 : number of current block
  long int * pDim_1 (o)         : dimension 1
  long int * pDim_2 (o)         : dimension 2
  int *pstatus (o)              : SAXS status

  Returns TRUE if Dim_1 and Dim_2 have been read (either from file or
  image block) or if an error has occurred. In all other cases the
  return value is FALSE. If in case of FALSE Dim_1 and Dim_2 are not used
  the *pstatus value can be ignored.

  If IO_DontReadWrite or IO_Dummy is set, no read is done, the return 
  status is Success and the return value FALSE : Dim_1 --> -1, Dim_2 --> -1 .

  To allow correct replacements of key expressions in ib.Dim[] this function
  must only be used when the headers of all input image blocks are opened and
  read, i.e. it should only be used after an ReadImage or ReadImageHeader
  to all input blocks. 

2 HISTORY
  2003-02-08  PB Parameter list changed and updated for key values
---------------------------------------------------------------------------*/
int ReadImageDimension ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                         long int * pDim_1, long int *pDim_2,
                         int * pstatus )
{ int retval = FALSE;
  long Tmp[4], *Dim; 
  size_t Ibl;
  const char * RoutineName = "ReadImageDimension";
  ImgBlk *ib;
  long ImageNumber, *num;
  int ImageChain;
  int j;
  int ErrorValue=0;
  size_t size;

  *pstatus = Success;

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, NULL ); 

  ib = pcb->ib;
  ImageNumber = ihb[blkno].ImNum;
  ImageChain  = ib[blkno].Memory.V;

  size = sizeof(long)*pcb->ImgBlkLen;
  num = (long *) malloc( size );
  if (!num) {
    *pstatus = NotEnoughMemoryAvailable;
//    sprintf(SaxsImageMessage,"ERROR: Failed to allocate %zu bytes\n", size);
    sprintf(SaxsImageMessage,"ERROR: Failed to allocate %zu | %lu bytes\n", size, size);
    SetSaxsErrorMessage( SaxsImageMessage );
    return(TRUE);
  }

  for ( j=0; j < pcb->ImgBlkLen; j++ ) num[j] = ihb[j].ImNum;

  *pDim_1 = -1l; *pDim_2 = -1l;
  if ( !((ib[blkno].Dim[1].I) && (ib[blkno].Dim[2].I)) ) {

    if ( !((IO_DontReadWrite | IO_Dummy) & ib[blkno].StreamOpenMode)  ) {
      /* read from file */
      Dim=Tmp; Dim[0]=2l;
      retval = edf_read_data_dimension( ib[blkno].Stream,
        ImageNumber, ImageChain, &Dim, &Ibl, &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if ( retval ) {
        *pDim_1 = Dim[1]; *pDim_2 = Dim[2];
      }
    }
  } else retval = TRUE;

  if (ib[blkno].Dim[1].I) *pDim_1 = longkeyvalue( ib[blkno].Dim[1].V, 
                                      pcb, ib, num, blkno, pstatus );
  if (*pstatus!=Success) { free(num); return( FALSE ); }
  if (ib[blkno].Dim[2].I) *pDim_2 = longkeyvalue( ib[blkno].Dim[2].V,
                                      pcb, ib, num, blkno, pstatus );
  if (*pstatus!=Success) { free(num); return( FALSE ); }

  free(num);
  return(retval);

} /* ReadImageDimension */

/*+++------------------------------------------------------------------------
1 ReadHeaderLine

2 DESCRIPTON
  Read a keyword value
  Return value TRUE if found or error, if not found return FALSE and the
  corresponding error message. In case of FALSE the error message can
  be ignored. The routines returnes only Success, if the keyword has been
  found or IO_DontReadWrite or IO_Dummy is set.
  Can only be used after ReadImage, ReadImageHeader, NewImage.
 
  return( int )			:		
  CmdBlk *pcb (i/o)		: command block
    pcb->ib[blkno].Memory.V (i) : memory number
  int blkno (i)			: image block no
  ImgHeadBlk ihb[]		: image header block
     ihb[blkno].Stream (i)	: i/o stream
     ihb[blkno].ImNum (i)	: image, from which the value is read
  char * keyword (i)		: keyword
  char * Value (o)		: output value (line)
  int *pstatus (o)		: SAXS status

  If IO_DontReadWrite or IO_Dummy is set, no read is done and the return 
  status is Success and the return value FALSE (not found).
  ---------------------------------------------------------------------------*/
int ReadHeaderLine          ( CmdBlk *pcb, int blkno,
                              ImgHeadBlk ihb[], char * keyword,
                              char * Value, int * pstatus )
{ const char * RoutineName = "ReadHeaderLine";
  int ImageChain = pcb->ib[blkno].Memory.V;
  int retval;
  int ErrorValue=0;

  *pstatus = Success; 

  if (pcb->TestBit) 
    printf(" %s (blkno = %d, keyword = %s)\n",RoutineName,blkno,keyword);

  SetSaxsError( ModuleName, RoutineName, keyword );

  if ( ((IO_Dummy | IO_DontReadWrite) & ihb[blkno].StreamOpenMode)  ) {
    retval = FALSE;
  } else {
    retval = edf_read_header_line ( ihb[blkno].Stream, ihb[blkno].ImNum, 
              ImageChain, keyword, Value, &ErrorValue, pstatus ) ;
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (pcb->TestBit) if (retval) printf(" %s = %s\n", keyword, Value);
  }

  return( retval );

} /* ReadHeaderLine */

/*+++------------------------------------------------------------------------
1 ReadHeaderFloat

2 DESCRIPTON
  Read a keyword value
  Return value TRUE if found or error, if not found return FALSE and the
  corresponding error message. In case of FALSE the error message can
  be ignored. The routines returnes only Success, if the keyword has been
  found or IO_DontReadWrite or IO_Dummy is set.
  Can only used after ReadImage, ReadImageHeader, NewImage.

  return( int )			:
  CmdBlk *pcb (i/o)             : command block
    pcb->ib[blkno].Memory.V (i) : memory number
  int blkno (i)                 : image block no
  ImgHeadBlk ihb[]              : image header block
     ihb[blkno].Stream (i)      : i/o stream
     ibh[blkno].ImNum (i)       : image, from which the value is read
  char * keyword (i)            : keyword
  float * Value (o)             : output value (float)
  int *pstatus (o)              : SAXS status

  If IO_DontReadWrite or IO_Dummy is set, no read is done and the return 
  status is Success and the return value FALSE (not found).
  ---------------------------------------------------------------------------*/
int ReadHeaderFloat ( CmdBlk *pcb, int blkno,
                      ImgHeadBlk ihb[], char * keyword,
                      float * Value, int * pstatus )
{ const char * RoutineName = "ReadHeaderFloat";
  int ImageChain = pcb->ib[blkno].Memory.V;
  int retval, ErrorValue=0;

  *pstatus = Success; /* edf_ routines */

  if (pcb->TestBit) 
    printf(" %s (blkno = %d, keyword = %s)\n",RoutineName,blkno,keyword);

  SetSaxsError( ModuleName, RoutineName, keyword );

  if ( ((IO_Dummy | IO_DontReadWrite) & ihb[blkno].StreamOpenMode)  ) {
    retval = FALSE;
  } else {
    retval = edf_read_header_float ( ihb[blkno].Stream, ihb[blkno].ImNum, 
               ImageChain, keyword, Value, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (pcb->TestBit) if (retval) printf(" %s = %g\n", keyword, *Value);
  }

  return( retval );

} /* ReadHeaderFloat */

/*+++------------------------------------------------------------------------
1 ReadHeaderLong

2 DESCRIPTON
  Read a keyword value
  Return value TRUE if found or error, if not found return FALSE and the
  corresponding error message. In case of FALSE the error message can
  be ignored. The routines returnes only Success, if the keyword has been
  found or IO_DontReadWrite or IO_Dummy is set.
  Can only used after ReadImage, ReadImageHeader, NewImage.

  return( int )
  CmdBlk *pcb (i/o)             : command block
    pcb->ib[blkno].Memory.V (i) : memory number
  int blkno (i)                 : image block no
  ImgHeadBlk ihb[]              : image header block
     ihb[blkno].Stream (i)      : i/o stream
     ihb[blkno].ImNum (i)       : image, from which the value is read
  char * keyword (i)            : keyword
  long int * Value (o)          : output value (long int)
  int *pstatus (o)              : SAXS status

  If IO_DontReadWrite or IO_Dummy is set, no read is done and the return 
  status is Success and the return value FALSE (not found).
  ---------------------------------------------------------------------------*/
int ReadHeaderLong  ( CmdBlk *pcb, int blkno,
                      ImgHeadBlk ihb[], char * keyword,
                      long int * Value, int * pstatus )
{ const char * RoutineName = "ReadHeaderLong";
  int ImageChain = pcb->ib[blkno].Memory.V;
  int retval, ErrorValue=0;

  *pstatus = Success; /* edf_ routines */

  if (pcb->TestBit) 
    printf(" %s (blkno = %d, keyword = %s)\n",RoutineName,blkno,keyword);

  SetSaxsError( ModuleName, RoutineName, keyword );

  if ( ((IO_Dummy | IO_DontReadWrite) & ihb[blkno].StreamOpenMode)  ) {
    retval = FALSE;
  } else {
    retval = edf_read_header_long ( ihb[blkno].Stream, ihb[blkno].ImNum, 
               ImageChain, keyword, Value, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (pcb->TestBit) if (retval) printf(" %s = %ld\n", keyword, *Value);
  }

  return( retval );

} /* ReadHeaderLong */

/*---------------------------------------------------------------------------
Name
   reopen_image_file --- tries to reopen an existing stream

DESCRIPTON
  Tries first to reopen an old stream, if it does not exist it opens a
  new stream.
  int *pstatus           : (o) SAXS status

HISTORY
  06-Jul-2001           PB
  ---------------------------------------------------------------------------*/
int reopen_image_file(const char * fname_full, const char * mode,
                    int * pstatus)
{ // const char * RoutineName = "reopen_image_file"; // unused
  int ErrorValue=0;
  int stream=-1;

  stream = edf_search_stream (fname_full, mode, &ErrorValue, pstatus);
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );

  if ( ( *pstatus == Success ) && ( stream < 0 ) ) { 
    // open new stream
    stream = edf_open_data_file(fname_full, mode, &ErrorValue, pstatus);
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  }

  return( stream );

} // reopen_image_file

/*+++------------------------------------------------------------------------
1 OpenImageFile

2 DESCRIPTON
  Opens the data file
  return( void ):
  char * fname           : (i)   file name pattern terminated by comma, e.g.
                                 file%%%%.edf, 1,3,10. Only the first part
                                 before the comma is used. The placeholders
                                 %%%% are replaced by FileNumber. 
  int * ib[blkno].Stream : (o)   stream
  long int * ib[blkno].StreamOpenMode (o) : OpenMode
  long int OpenMode      : (i)   file opening mode, no action for IO_DontOpen
                                 and IO_Dummy
                             IO_New: open file and truncate, stream at start,
                             IO_Old: open file, stream at end
  int *pstatus           : (o) SAXS status
2 HISTORY
  12-Jan-1998           PB open_data_file opens now with three
                           different modi: IO_New, IO_Old and nothing of both 
                           (Any)
  21-Dec-1999           PB if the filename is "dummy", do not read or
                           write any data to it
  03-Jul-2001           PB char *fname -> const char *fname
  08-Jul-2001           PB FileNumber, full name is written to image block
  03-Mar-2005           PB fname_full initialised with fnam. It is otherwise
                           possible that, in case of an error, an already 
                           opened file is reopened, 
  ---------------------------------------------------------------------------*/
void OpenImageFile ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                     const char *fname, long int FileNumber,
                     long int OpenMode, int *pstatus)
{ int * pstream;
  static char fname_full[IO_len], fname_pattern[IO_len];
  const char * RoutineName = "OpenImageFile";

  *pstatus = Success;

  /* preset fname_full with fname */
  strncpy(fname_full,fname,IO_len);
  fname_full[IO_len-1]='\0';

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, fname_full ); 

  /* set debug mode */
  if (pcb->TestBit>2) edfio_debug( True );

  /* get full file name */
  filename_parameter (fname_pattern,IO_len,fname,0);
  filename_pattern (fname_pattern,IO_len,fname_pattern,FileNumber);
  filename_full(fname_full,IO_len,fname_pattern);

  /* set general block or not */
  edf_general_block( pcb->GeneralBlock.V ); 

  /* set output data value offset type */
  edf_set_datavalueoffset( pcb->OutputDataValueOffset.V );

  /* set output data type */
  edf_set_datatype( pcb->OutputDataType.V );

  /* set byte order of all bsl input files */
  if (pcb->BslInputByteOrder.I) 
    switch ( pcb->BslInputByteOrder.V ) {
      case 1: edf_set_bsl_input_byteorder ( LowByteFirst ); break;
      case 2: edf_set_bsl_input_byteorder ( HighByteFirst ); break;
      default: break;
      }

  /* set maximum line width */
  if (pcb->MaxLineWidth.I)
     edf_set_max_line_width( (unsigned long) MAX2(0,pcb->MaxLineWidth.V) );

  /* set minimum header size */
  if (pcb->MinHeaderSize.I)
     edf_set_minimumheadersize( (unsigned long) MAX2(0,pcb->MinHeaderSize.V) );

  if ( strcmp(fname_full,DummyFileName)==0 ) // dont read/write anything
     OpenMode = OpenMode | IO_Dummy;

  pstream = &ib[blkno].Stream;
  ib[blkno].StreamOpenMode = OpenMode;

  if (*pstream>=0) { ;
    *pstatus = FileAlreadyOpened; return;
    }

  // Save actual file number and file name in image block
  ib[blkno].FileNumber = FileNumber;
  CopyLine(fname_pattern,ib[blkno].FileName,IO_len,0);

  if ((IO_Dummy | IO_DontOpen) & OpenMode) { *pstatus = Success; return; }

  // open file
  if (IO_Old & OpenMode) {
    if (!filename_exists(fname_full)) {
      *pstatus=FileNotFound; return; 
      } else {
      if (IO_FileProtect & OpenMode) 
        *pstream = reopen_image_file(fname_full, "read", pstatus);
       else
        *pstream = reopen_image_file(fname_full, "old", pstatus);
      if (*pstatus!=Success) return;
      }
    } /* IO_Old */

  else if (IO_New & OpenMode) {
    if (filename_exists(fname_full)) {
      ; /* Rename old file to older version */
      } 
    *pstream = reopen_image_file(fname_full, "new", pstatus);
    if (*pstatus!=Success) return;
    } /* IO_New */
  
  else {
    if (IO_FileProtect & OpenMode)
      *pstream = reopen_image_file(fname_full, "read", pstatus);
     else
      *pstream = reopen_image_file(fname_full, "any", pstatus);
    if (*pstatus!=Success) return;
    } /* Any */

} /* OpenImageFile */

/*---------------------------------------------------------------------------
NAME
  StreamCanBeClosed

DESCRIPTION
  Returns 1 if no other image block uses the same Stream as ib[blkno].
  and if it can be closed, otherwise 0. 
  ---------------------------------------------------------------------------*/
int StreamCanBeClosed( CmdBlk *pcb, ImgBlk ib[], int blkno, int * pstatus)
{ int stream=ib[blkno].Stream;
  int value=1;
  int i;

  if (stream<0) return(0);

  for (i=0;(i<pcb->ImgBlkLen)&&(value);i++) {
    if (i!=blkno) {
      if (stream==ib[i].Stream) value=0;
    }
  }

  return(value);

} // StreamCanBeClosed

/*+++------------------------------------------------------------------------
1 CloseImageFile

2 DESCRIPTION
  Closes the data file
  return( void ) :
  long int ib[blkno].StreamOpenMode: (i) file opening mode, no action for
                                     IO_DontOpen or IO_Dummy
  int *pErrorValue  :   error value
  ---------------------------------------------------------------------------*/
void CloseImageFile ( CmdBlk *pcb, ImgBlk ib[], int blkno, int * pstatus)
{ int ErrorValue=0;
  const char * RoutineName = "CloseImageFile";

  *pstatus = Success; /* edf_ routines */

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, NULL );

  if (!((IO_Dummy | IO_DontOpen) & ib[blkno].StreamOpenMode)) {
    // close if no other image blocks uses Stream
    if ( StreamCanBeClosed( pcb, ib, blkno, pstatus) ) {
      if (*pstatus!=Success) return;
      (void) edf_close_data_file(ib[blkno].Stream, &ErrorValue, pstatus);
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) return;
    }
  } /* if ((!IO_DontOpen) ... */

  ib[blkno].Stream = -1;
  ib[blkno].FileName[0] = '\0';

  *pstatus = Success;

} /* CloseImageFile */

/*+++------------------------------------------------------------------------
1 ReadImage

2 DESCRIPTION
  Reads header and data of image number ImageNumber.

  Read the image data
  return( int )			FALSE : image header not found,
  				  *pstatus != Success
  				TRUE  : image header found or error,
  				  *pstatus = <SaxsStatus>
  CmdBlk *pcb (i)		: command block
  ImgBlk ib[blkno] (i)		: image block
    ib[blkno].StreamOpenMode (i): IO_DontReadData: only image header is read,
                                               no memory allocated,
                                               no image data available
    ib[blkno].Val (i)   	: value (program)
    ib[blkno].Factor (i)	: multiplication factor
    ib[blkno].Const (i)		: addition constant
    ib[blkno].Bin (i)		: rebin factors
  int blkno (i)			: block no of image block and header block
  long num[](i)	                : number of requested image 
                                  (ImageNumber=num[blkno])
  ImgHeadBlk ihb[blkno] (o)	: image header block
    ihb[blkno].I (o)		: TRUE  = header read from file
				  FALSE = no header read
    ihb[blkno].ImgByteLen (o)   : number of bytes allocated
    ihb[blkno].Stream (o)	: copy from Image Block
    ihb[blkno].StreamOpenMode(o): copy from Image Block
    ihb[blkno].ImNum (o)        : current image number (set to num[blkno])
  int *pstatus (o)		: exit status

2 HISTORY
  27-Nov-1995
  ---------------------------------------------------------------------------*/
int	ReadImage 		( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  long num[], ImgHeadBlk ihb[],
				  int *pstatus)
{ const char * RoutineName = "ReadImage";

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, NULL );

  *pstatus = Success;

  return(read_saxs_image(pcb,ib,blkno,num,ihb,FALSE,pstatus));

} /* ReadImage */

/*+++------------------------------------------------------------------------
1 ReadImageHeader

2 DESCRIPTION
  Reads header of image number ImageNumber.

  return( int )			FALSE : image header not found,
  				        *pstatus != Success
    				TRUE  : image header found,
  				        *pstatus = <SaxsStatus>
  CmdBlk *pcb (i)		: command block
  ImgBlk ib[blkno] (i)		: image block
    ib[blkno].Bin (i)		: rebin factors
  int blkno (i)			: block no of image block and header block
  long num[](i)                 : number of requested image
                                  (ImageNumber=num[blkno])
  ImgHeadBlk ihb[blkno] (o)	: image header block
    ihb[blkno].I                : TRUE  = header read from file
                                  FALSE = no header read
    ihb[blkno].ImgByteLen (o)   : 0 no bytes available
    ihb[blkno].Stream (o)       : copy from Image Block
    ihb[blkno].StreamOpenMode(o): copy from Image Block
    ihb[blkno].ImNum (o)        : current image number (set to num[blkno])
  int *pstatus (o)              : exit status

2 HISTORY
  27-Nov-1995
  ---------------------------------------------------------------------------*/
int	ReadImageHeader		( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  long num[], ImgHeadBlk ihb[],
				  int *pstatus)
{ const char * RoutineName = "ReadImageHeader";

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, NULL ); 

  *pstatus = Success;

  return(read_saxs_image(pcb,ib,blkno,num,ihb,TRUE,pstatus));

} /* ReadImageHeader */

/*+++------------------------------------------------------------------------
1 WriteHeader

2 DESCRIPTION
  Writes an external header into the header of the specified image.

  Read the image data
  return( void )                :
  CmdBlk *pcb                   : command block
  long int ImageNumber (i)      : output image number
  ImgHeadBlk ihb[blkno]         : image header of current image
            .Stream (i)         : output stream
            .StreamOpenMode (i) :  no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain
  const char * header_keyword   : header key 
  int * pstatus                 : exit status

2 HISTORY
  2004-10-01 PB  creation
----------------------------------------------------------------------------*/
void  WriteHeader               ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  const char * header_key, int * pstatus )
{ ImgBlk * pib;
  ImgHeadBlk * pihb;

  int ErrorValue=0;
  int stream;
  int ImageChain;
  long int ImageNumber;
  const char * RoutineName = "WriteHeaderLine";

  *pstatus = Success;

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, NULL ); 

  pib  = &((pcb->ib)[blkno]);
  pihb = &ihb[blkno];
  if (!pihb->I) { // is header initialized?
    *pstatus = HeaderNotInitialized; return; }

  stream = pihb->Stream;
  ImageChain = pib->Memory.V;
  ImageNumber = pihb->ImNum;

  if ((IO_Dummy | IO_DontReadWrite) & pib->StreamOpenMode) return;

  (void) edf_write_header ( stream, ImageNumber, ImageChain,
                header_key, &ErrorValue, pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) return;

} // WriteHeader

/*+++------------------------------------------------------------------------
1 WriteHeaderLine 
 
2 DESCRIPTION
  Writes a keyword and a line value into the header of the specified image.
 
  Read the image data
  return( void )                :
  CmdBlk *pcb                   : command block
  long int ImageNumber (i)      : output image number
  ImgHeadBlk ihb[blkno]         : image header of current image
            .Stream (i)         : output stream
            .StreamOpenMode (i) :  no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain
  char * keyword                : keyword to write
  char * Value                  : Value to write
  int * pstatus                 : exit status
 
2 HISTORY
  2001-08-19 PB  creation 
----------------------------------------------------------------------------*/ 
void    WriteHeaderLine         ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[], 
                                  char * keyword, char * Value, int * pstatus )
{ ImgBlk * pib;
  ImgHeadBlk * pihb;
 
  int ErrorValue=0;
  int stream;
  int ImageChain;
  long int ImageNumber;
  const char * RoutineName = "WriteHeaderLine";
 
  *pstatus = Success;

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, keyword ); 
 
  pib  = &((pcb->ib)[blkno]);
  pihb = &ihb[blkno];
  if (!pihb->I) { // is header initialized?
    *pstatus = HeaderNotInitialized; return; }

  stream = pihb->Stream;
  ImageChain = pib->Memory.V;
  ImageNumber = pihb->ImNum;
 
  if ((IO_Dummy | IO_DontReadWrite) & pib->StreamOpenMode) return;
 
  (void) edf_write_header_line( stream, ImageNumber, ImageChain,
                keyword, Value, &ErrorValue , pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) return;

} // WriteHeaderLine 

/*+++------------------------------------------------------------------------
1 WriteHeaderFloat
 
2 DESCRIPTION
  Writes a keyword and a float value into the header of the specified image. 
 
  Read the image data
  return( void )                :
  CmdBlk *pcb                   : command block
  ImgHeadBlk ihb[blkno]         : image header of current image
            .Stream (i)         : output stream
            .StreamOpenMode (i) :  no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain
  char * keyword                : keyword to write
  float Value                   : float value to write
  int * pstatus                 : exit status
 
2 HISTORY
  2001-08-19 PB  creation
----------------------------------------------------------------------------*/
void    WriteHeaderFloat        ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  char * keyword, float Value, int * pstatus )
{ ImgBlk * pib;
  ImgHeadBlk * pihb;
 
  int ErrorValue=0;
  int stream;
  int ImageChain;
  long int ImageNumber;
  const char * RoutineName = "WriteHeaderFloat"; 
 
  *pstatus = Success;
 
  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, keyword ); 
 
  pib  = &((pcb->ib)[blkno]);
  pihb = &ihb[blkno];
  if (!pihb->I) { // is header initialized?
    *pstatus = HeaderNotInitialized; return; }

  stream = pihb->Stream;
  ImageChain = pib->Memory.V;
  ImageNumber = pihb->ImNum;
 
  if ((IO_Dummy | IO_DontReadWrite) & pihb->StreamOpenMode) return;
 
  (void) edf_write_header_float( stream, ImageNumber, ImageChain,
                keyword, Value, &ErrorValue , pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) return;
 
} // WriteHeaderFloat

/*+++------------------------------------------------------------------------
1 WriteHeaderLong
 
2 DESCRIPTION
  Writes a keyword and a long int value into the header of the specified image.
 
  Read the image data
  return( void )                :
  CmdBlk *pcb                   : command block
  ImgHeadBlk ihb[blkno]         : image header of current image
            .Stream (i)         : output stream
            .StreamOpenMode (i) :  no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain
  char * keyword                : keyword to write
  long int Value                : long int value to write
  int * pstatus                 : exit status
 
2 HISTORY
  2001-08-19 PB  creation
----------------------------------------------------------------------------*/
void    WriteHeaderLong         ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
                                  char * keyword, long int Value, int * pstatus )
{ ImgBlk * pib;
  ImgHeadBlk * pihb;
 
  int ErrorValue=0;
  int stream;
  int ImageChain;
  long int ImageNumber;
  const char * RoutineName = "WriteHeaderLong";

  *pstatus = Success;
 
  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, keyword ); 
 
  pib  = &((pcb->ib)[blkno]);
  pihb = &ihb[blkno];
  if (!pihb->I) { // is header initialized?
    *pstatus = HeaderNotInitialized; return; }

  stream = pihb->Stream;
  ImageChain = pib->Memory.V;
  ImageNumber = pihb->ImNum;
 
  if ((IO_Dummy | IO_DontReadWrite) & pihb->StreamOpenMode) return;
 
  (void) edf_write_header_long( stream, ImageNumber, ImageChain,
                keyword, Value, &ErrorValue , pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) return; 
 
} // WriteHeaderLong

/*+++------------------------------------------------------------------------
1 WriteHeaderUnit

2 DESCRIPTION
  Writes a keyword and a float value with a unit into the header of the 
  specified image.

  Read the image data
  return( void )                :
  CmdBlk *pcb                   : command block
  ImgHeadBlk ihb[blkno]         : image header of current image
            .Stream (i)         : output stream
            .StreamOpenMode (i) :  no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain
  const char * keyword          : keyword to write
  float Value                   : float value to write (in basic units)
  const char * unit             : unit to write. The value to write is 
                                  recalculated in this unit.
  int * pstatus                 : exit status

2 EXAMPLE
  Value=10, unit="mm": written value is 10000_mm

2 HISTORY
  2004-10-04 PB  creation
----------------------------------------------------------------------------*/
void    WriteHeaderUnit        ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[],
      const char * keyword, float Value, const char * unit, int * pstatus )
{ ImgBlk * pib;
  ImgHeadBlk * pihb;

  int ErrorValue=0;
  int stream;
  int ImageChain;
  long int ImageNumber;
  const char * RoutineName = "WriteHeaderUnit";

  *pstatus = Success;

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, keyword ); 

  pib  = &((pcb->ib)[blkno]);
  pihb = &ihb[blkno];
  if (!pihb->I) { // is header initialized?
    *pstatus = HeaderNotInitialized; return; }

  stream = pihb->Stream;
  ImageChain = pib->Memory.V;
  ImageNumber = pihb->ImNum;

  if ((IO_Dummy | IO_DontReadWrite) & pihb->StreamOpenMode) return;

  (void) edf_write_header_unit ( stream, ImageNumber, ImageChain,
                keyword, Value, unit, &ErrorValue, pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) return;

} // WriteHeaderUnit

/*+++------------------------------------------------------------------------
1 WriteImage

2 DESCRIPTION
  Writes header and float data pdata[pihb->Dim[1],pihb->Dim[2]]
  In case of success pihb->ImNum is set to the ImageNumber.

  Read the image data
  return( void ) 		:

  CmdBlk *pcb                   : command block
  ImgHeadBlk ihb[blkno]         : image header of current image
                .Dim (i)        : Size of array (only [0],[1] and [2])
                .<any other header value> (i)
                .ImNum (o)      : image number to write
                .Data (i)       : data to write
            .Stream (i)         : output stream
            .StreamOpenMode (i) : no action, if IO_DontReadWrite or
                                  IO_Dummy is set
  ib[blkno].Memory.V (i)        : ImageChain

  int *pstatus  		: exit status

2 HISTORY
  2000-03-18 PB   ImageChain = pib->Memory.V;
  2000-07-30 PB   orientation added
  2001-08-19 PB   ImageNumber removed, taken from ibh[blkno].ImNum
  2003-05-29 PB   pib->Orientation.V -> pihb->Orientation.V
  2011-01-05 PB   output compression 
----------------------------------------------------------------------------*/
void    WriteImage              ( CmdBlk *pcb, ImgBlk ib[], int blkno,
				  ImgHeadBlk ihb[], int *pstatus)
{ int ErrorValue=0;
  ImgBlk * pib;
  ImgHeadBlk * pihb;
  int stream;
  int ImageChain, VarChain;
  long int ImageNumber;

  const char * RoutineName = "WriteImage";

  *pstatus = Success;

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, NULL ); 

  pib  = &ib[blkno];
  pihb = &ihb[blkno];
  if (!pihb->I) {
    *pstatus = HeaderNotInitialized; return; }

  stream = pihb->Stream;
  ImageNumber = pihb->ImNum;
  ImageChain = pib->Memory.V;
  VarChain = -ImageChain;

  if ((IO_Dummy | IO_DontReadWrite) & pihb->StreamOpenMode) return;

  /* clip, gnaw, rebin, orient and transform image */
  if (pcb->TestBit>1) printf(" --  WriteImage/condition_2d (blkno=%d)\n",blkno);
  condition_2d( pihb,
                pihb->MinClip, pihb->MaxClip,
                pihb->Gnaw[1], pihb->Gnaw[2],
                pihb->Bin[1],  pihb->Bin[2],
                pihb->Val,     pihb->ErrVal,
                pihb->Factor,  pihb->Const,
              //  pib->Ave,     -pihb->rori,
                pib->Ave,     0l,  // no orientation transformation
                FALSE, pcb->TestBit, pstatus );
  if (*pstatus!=Success) return;

  if (DUMMYDEFINED(pihb->Dummy.V,pihb->DDummy.V))
    mark_float_2d(pihb->Data,pihb->Dim[1],pihb->Dim[2],
                  pihb->Dummy.V,pihb->DDummy.V);

  if (pihb->VarDat)
    mark_float_2d(pihb->VarDat,pihb->Dim[1],pihb->Dim[2],
                  VarDummy,DDSET(VarDummy));

  /* write image header */
  if (pcb->TestBit>1) printf(" --  write_image_header (blkno=%d)\n",blkno);
  write_image_header ( stream, ImageChain, pihb, &ErrorValue, pstatus );
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) return;

  /* write image array */
  if (pcb->TestBit>1) printf(" --  edf_write_data (blkno=%d)\n",blkno);
  edf_set_datacompression( pcb->OutputCompression.V ); // set output compression
  edf_write_data ( stream, ImageNumber, ImageChain, 
                   pihb->Dim, pihb->Data, MFloat, 
                   &ErrorValue, pstatus );
  edf_set_datacompression( UnCompressed ); // reset output compression
  SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
  if (*pstatus!=Success) return;

  /* write variance array */
  if ( (pihb->VarDat)&&(VarChain<0) ) {
    if (pcb->TestBit>1) 
      printf(" --  edf_write_data variance array (blkno=%d)\n",blkno);
    /* write variance dummy value (info only, unchangeable value)*/
    edf_write_header_float( stream, ImageNumber, VarChain,
              KDummy, VarDummy, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return;

    edf_set_datacompression( pcb->OutputCompression.V ); // set output compression
    edf_write_data ( stream, ImageNumber, VarChain,
                     pihb->Dim, pihb->VarDat, MFloat,
                     &ErrorValue, pstatus );
    edf_set_datacompression( UnCompressed ); // reset output compression
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return;
  }

} /* WriteImage */

/*+++------------------------------------------------------------------------
1 ReportSaxsImageError

2 DESCRIPTION

2 HISTORY
----------------------------------------------------------------------------*/
char * ReportSaxsImageError ( int ExternalError )
{ char * ErrorMsg;
  ErrorMsg = edf_report_data_error(ExternalError);
  return (ErrorMsg);
} /* ReportSaxsImageError */

/*---------------------------------------------------------------------------
1 NewImageHeader

2 DESCRIPTION
  Creates image header with ImageNumber. The file does not need 
  to be opened.

  return( void )
  CmdBlk *pcb (i)		: command block
  ImgBlk ib[blkno] (i)		: image block
                                  only image header will be created, 
                                  no memory will be allocated 
  int blkno (i)			: block no
  long num (i) 	                : array with the current image number of all blocks
                                  The image of block blkno (ImageNumber=num[blkno])
                                  will be created.
  ImgHeadBlk ihb[blkno] (o)	: image header block
				  The header is created from the template
                                  and the image block.
    ihb[blkno].I (o)            : TRUE  = image header is created
                                  FALSE = no image header is not created
    ihb[blkno].Stream (o)       : copy from Image Block
    ihb[blkno].StreamOpenMode(o): copy from Image Block
  ImgHeadBlk * template (i)	: template for header block,
                                  not used if it is the NULL pointer
  void (*InitFunction) ...      : called after balance_header_block,
                                  not used if it is the NULL pointer
  int *pstatus (o)		: exit status

2 HISTORY
  2001-01-11   PB from NewImage
  2003-02-02   PB parameter list ImageNumber -> num[blkno]
  2005-07-14   PB splitted into new_image_header (internal)
                  and NewImageHeader (external)
---------------------------------------------------------------------------*/
void  new_image_header          ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                long num[], ImgHeadBlk ihb[], ImgHeadBlk * template, 
                void (*InitFunction) (CmdBlk *, long [], ImgHeadBlk [], int *),
                int *pstatus)
{ int ImageChain;
  long ImageNumber;
  int header_found;
  int ErrorValue=0;

  *pstatus = Success;

  filename_full(SaxsImageMessage,IO_len,ib[blkno].FileName); 

  ImageChain = ib[blkno].Memory.V;
  ImageNumber = num[blkno];

  init_header_block ( ihb, blkno );
  if (pcb->TestBit>1) { printf(" -- init_header_block (blkno=%d)\n",blkno);
    PrintImageHeaderBlock(&ihb[blkno]); }

  if (template) {
    template_header_block ( ImageNumber, 0, 0,
                            (size_t) NULL, template->Dim, (float *) NULL,
                            template, &ihb[blkno] );
    if (pcb->TestBit>1) { 
      printf(" -- template_header_block (blkno=%d)\n",blkno);
      PrintImageHeaderBlock(&ihb[blkno]); }
    }

  balance_header_block( pcb, blkno, num, ib, ihb, pstatus );
  if (*pstatus!=Success) return;

  // New Image has always orientation 1
  // ihb[blkno].Orientation.V = 1l; +++++++++++++++++++++ no longer

  if (pcb->TestBit>1) {
    printf(" -- balance_header_block (blkno=%d)\n",blkno);
    PrintImageBlock(&ib[blkno]);
    PrintImageHeaderBlock(&ihb[blkno]); 
  }

  // Modify standard header, if necessary 
  if (InitFunction) {
    InitFunction (pcb, num, ihb, pstatus);
    if (*pstatus!=Success) return;

    if (pcb->TestBit>1) { 
      printf(" -- InitFunction (blkno=%d)\n",blkno);
      PrintImageHeaderBlock(&ihb[blkno]); 
    }
  }

  /* Check parameters */
  if ( (ihb[blkno].Orientation.V<=0l)||(16l<ihb[blkno].Orientation.V) ) {
    *pstatus = InvalidOrientation;
    return;
  }

  // copy Stream and StreamOpenMode
  ihb[blkno].Stream = ib[blkno].Stream;
  ihb[blkno].StreamOpenMode = ib[blkno].StreamOpenMode;

  /* error, if file is protected */
  if (ihb[blkno].StreamOpenMode & IO_FileProtect) {
    *pstatus = FileProtected; return;}

  if ((IO_Dummy | IO_DontReadWrite) & ihb[blkno].StreamOpenMode) {
    ihb[blkno].I = TRUE; /* header is valid */
    return; 
  }                                               

  /* error, if image exists and is protected */
  if (ib[blkno].Stream>=0) { // do this test only if file is opened
    header_found = edf_test_header( ib[blkno].Stream, ImageNumber, ImageChain,
           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return;
 
    if ( header_found && (ihb[blkno].StreamOpenMode & IO_ImageProtect) ) {
         *pstatus = ImageProtected; return;
      } /* edf_test_header */

    ihb[blkno].I = TRUE; /* header is valid for I/O*/

    if (template) {
      if (template->Header.I) {
        /* write full header of template header to output header (PassHeader)*/
        edf_write_header( ihb[blkno].Stream, ImageNumber, ImageChain,
                          template->Header.V, &ErrorValue, pstatus );
        SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
        if (*pstatus!=Success) return;
      }
    }

  } // if (ib[blkno].Stream>=0)

} /* new_image_header */

void    NewImageHeader           ( CmdBlk *pcb, ImgBlk ib[], int blkno,
                   long num[], ImgHeadBlk ihb[], ImgHeadBlk * template, 
                   void (*InitFunction) (CmdBlk *,long [],ImgHeadBlk [],int *),
                   int *pstatus)
{ const char * RoutineName = "NewImageHeader";
  
  *pstatus = Success;

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  filename_full(SaxsImageMessage,IO_len,ib[blkno].FileName);
  SetSaxsError( ModuleName, RoutineName, SaxsImageMessage );

  new_image_header ( pcb, ib, blkno, num, ihb, template, InitFunction, pstatus);
  
} /* NewImageHeader */

/*---------------------------------------------------------------------------
1 NewImage

2 DESCRIPTION
  Creates image ImageNumber=num[blkno].
  *ppdata[pihb->Dim[1],pihb->Dim[2]].

  Read the image data
  return( void )
  CmdBlk *pcb (i)               : command block
  ImgBlk ib[blkno] (i)          : image block
    ib[blkno].StreamOpenMode (i): IO_DontReadData: only image header will
                                  be created, no memory will be allocated
  int blkno (i)                 : block no
  long num (i)                  : array with the current image number of all blocks
                                  The image of block blkno (ImageNumber=num[blkno])
                                  will be created.
  ImgHeadBlk ihb[blkno] (o)     : image header block
                                  The header is created from the template
                                  and the image block.
    ihb[blkno].I (o)            : TRUE  = image header created
                                  FALSE = no image header created
    ihb[blkno].ImgByteLen (o)   : number of bytes allocated
    ihb[blkno].Stream (o)       : copy from Image Block
    ihb[blkno].StreamOpenMode(o): copy from Image Block
  ImgHeadBlk * template (i)     : template for header block
  void (*InitFunction) ...      : initialization routine for the header
  ib[blkno].StreamOpenMode (i)  :
  int *pstatus (o)              : exit status

  Variance array is initialized with -1.0

2 HISTORY
  2002-02-02   PB parameter list ImageNumber -> num[blkno]
  2007-06-10   PB InitFunction
---------------------------------------------------------------------------*/
void    NewImage                ( CmdBlk *pcb, ImgBlk ib[], int blkno,
              long num[], ImgHeadBlk ihb[],ImgHeadBlk * template, 
              void (*InitFunction) (CmdBlk *, long [], ImgHeadBlk [], int *), 
              int *pstatus)
{ const char * RoutineName = "NewImage";
  int var;

  *pstatus = Success;

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  filename_full(SaxsImageMessage,IO_len,ib[blkno].FileName);
  SetSaxsError( ModuleName, RoutineName, SaxsImageMessage ); 

  // Set interpolation mode
  if (pcb->AntiAliased.V) Ipolmode ( IPOL_ANTIALIASED  );
  else Ipolmode ( IPOL_NORMAL );

  // Set weight method
  if (pcb->VarianceWeight.V) Ipolweight ( IPOL_WEIGHTED  );
  else Ipolweight ( IPOL_EQUAL );

  if (pcb->TestBit>1)
    printf("Interpolation mode is set to %s.\n",Ipolmode2str(Ipolmode(0)));

  // Create variance array only if image is not a variance array 
  var = ( ( pcb->CreateVariance.V ) && (ib[blkno].Memory.V>0) );

  new_image_header ( pcb, ib, blkno, num, ihb, template, InitFunction, pstatus);
  if (*pstatus!=Success) return;

  if (!(IO_DontReadData & ib[blkno].StreamOpenMode)) {
    if (pcb->TestBit>1) printf(" -- alloc_image_memory (blkno=%d)\n",blkno);
    alloc_image_memory(pcb,ihb[blkno].Dim,&ihb[blkno], 
      ihb[blkno].Dummy.V,TRUE, var, VarDummy, TRUE, pstatus);
    if (*pstatus!=Success) return;
  }

} /* NewImage */

/*---------------------------------------------------------------------------
1 free_image_memory

2 DESCRIPTION
  Releases Image memory

  ImgHeadBlk *pihb      (i/o)   : image header block

2 HISTORY
  15-Feb-1995   Peter Boesecke creation
  28-Feb-1996   PB pihb->ImgByteLen = 0l; added
  12-Jan-1998   PB free(xyz) -> {free(xyz); xyz=NULL;}
  10-Jul-2001   PB pihb.I = FALSE; added
  12-Jul-2001   PB set LoopCount, ActualLoopCount to 0
  28-Aug-2001   PB pihb->Write
  05-Aug-2005   PB VarDat
----------------------------------------------------------------------------*/
void free_image_memory ( ImgHeadBlk *pihb )
{ int errval;
  if (pihb->Header.I) {
    /* free header */
    edf_free_header ( pihb->Header.V );
    pihb->Header.I = FALSE;
    }

  if (pihb->Data!=NULL) { free(pihb->Data); pihb->Data=NULL; }
  if (pihb->VarDat!=NULL) { free(pihb->VarDat); pihb->VarDat=NULL; }
  pihb->Dim[0] = 0; pihb->Dim[1] = 0; pihb->Dim[2] = 0;pihb->Data=(float*) NULL;
  pihb->ImgByteLen = (size_t) NULL;
  pihb->I = FALSE;
  pihb->LoopCount = 0;
  pihb->ActualLoopCount = 0;
  num_rmprog(pihb->Val.V,&errval);pihb->Val.V=NULL;
  num_rmprog(pihb->ErrVal.V,&errval);pihb->ErrVal.V=NULL;

} /* free_image_memory */

/*---------------------------------------------------------------------------
1 FreeImage

2 DESCRIPTION
  Releases Image memory

  ImgHeadBlk ihb[blkno]  (i/o)   : image header block

2 HISTORY
  24-Nov-1995   Peter Boesecke creation

----------------------------------------------------------------------------*/
void    FreeImage               ( CmdBlk *pcb, int blkno, ImgHeadBlk ihb[] )

{ const char * RoutineName = "FreeImage";

  if (pcb->TestBit) printf(" %s (blkno = %d)\n",RoutineName,blkno);
  SetSaxsError( ModuleName, RoutineName, NULL );

  free_image_memory ( &(ihb[blkno]) );
} /* FreeImage */

/*---------------------------------------------------------------------------
1 GeoHeaderCopy

2 DESCRIPTION
  Copies geometry keywords from ihb[in] to ihb[out]

  ImgHeadBlk ihb[in]      (i)   : image header block
  ImgHeadBlk ihb[in]      (i)   : image header block

2 HISTORY
  07-Dec-1995   Peter Boesecke creation
----------------------------------------------------------------------------*/
void    GeoHeaderCopy ( ImgHeadBlk ihb[], int blkin, int blkout )
{ geo_header_copy(&ihb[blkin],&ihb[blkout]);
} /* GeoHeaderCopy */

/*---------------------------------------------------------------------------
1 DoNotScaleWith 

2 DESCRIPTION
  Marks image header block ihb[blk] so that it will not be used for scaling

2 HISTORY
  2001-07-07   Peter Boesecke
----------------------------------------------------------------------------*/
void DoNotScaleWith   ( ImgHeadBlk ihb[], int blk )
{ ihb[blk].DoNotScale = TRUE;
} // DoNotScaleWith

/*--------------------------------------------------------------------------
   PrintImageHeaderBlock
  --------------------------------------------------------------------------*/
void PrintImageHeaderBlock ( ImgHeadBlk * pihb )
{ char VName[IO_len];

  (void) sprintf(VName,"ihb[% d]",pihb->blkno);
  printf("Image Header Block (blkno = %d)\n",pihb->blkno);
  printf("%s.I              = %d\n", VName,pihb->I);
  printf("%s.ImNum          = %ld\n", VName,pihb->ImNum);
  printf("%s.StreamOpenMode = %#lx\n", VName,pihb->StreamOpenMode);
  printf("%s.LoopCount      = %ld\n", VName,pihb->LoopCount);
  printf("%s.ActualLoopCount= %ld\n", VName,pihb->ActualLoopCount);
//  printf("%s.ImgByteLen     = %zu\n", VName,pihb->ImgByteLen);
  printf("%s.ImgByteLen     = %zu | %lu\n",
    VName,pihb->ImgByteLen,pihb->ImgByteLen);
  printf("%s.Dim[1]         = %ld\n", VName,pihb->Dim[1]);
  printf("%s.Dim[2]         = %ld\n", VName,pihb->Dim[2]);
  printf("%s.Data           = %p\n", VName,pihb->Data);
  printf("%s.VarDat         = %p\n", VName,pihb->VarDat);
  printf("%s.DoNotScale     = %d\n", VName,pihb->DoNotScale);
  if (pihb->Header.I)         printf("%s.Header.V       = %s\n",
    VName,pihb->Header.V);
  if (pihb->MinClip.I)        printf("%s.MinClip.V      = % e\n",
    VName,pihb->MinClip.V);
  if (pihb->MaxClip.I)        printf("%s.MaxClip.V      = % e\n",
    VName,pihb->MaxClip.V);
  if (pihb->Const.I)          printf("%s.Const.V        = % e\n",
    VName,pihb->Const.V);
  if (pihb->Factor.I)         printf("%s.Factor.V       = % e\n",
    VName,pihb->Factor.V);
  if (pihb->Val.I) printf(
//    "%s.Val.V.Name     = %s(_iNval) (%zu bytes:%ld vrs/%ld ins/%ld acc)\n",
    "%s.Val.V.Name     = %s(_iNval) (%zu|%lu bytes:%ld vrs/%ld ins/%ld acc)\n",
      VName,(pihb->Val.V)->Name, 
      num_prog_size (pihb->Val.V), num_prog_size (pihb->Val.V), 
      num_prog_variables (pihb->Val.V),
      num_prog_instructions(pihb->Val.V,2), 
      num_prog_accumulators(pihb->Val.V) );
  if (pihb->ErrVal.I) printf(
//    "%s.ErrVal.V.Name  = %s(_iNval,_iNerr) (%zu bytes:%ld vrs/%ld ins/%ld acc)\n",
    "%s.ErrVal.V.Name  = %s(_iNval,_iNerr) (%zu|%lu bytes:%ld vrs/%ld ins/%ld acc)\n",
      VName,(pihb->ErrVal.V)->Name, 
      num_prog_size (pihb->ErrVal.V), num_prog_size (pihb->ErrVal.V),
      num_prog_variables (pihb->ErrVal.V),
      num_prog_instructions(pihb->ErrVal.V,2), 
      num_prog_accumulators(pihb->ErrVal.V) );
  if (pihb->Bin[1].I)         printf("%s.Bin[1].V       = %ld\n",
    VName,pihb->Bin[1].V);
  if (pihb->Bin[2].I)         printf("%s.Bin[2].V       = %ld\n",
    VName,pihb->Bin[2].V);
  if (pihb->Gnaw[1].I)        printf("%s.Gnaw[1].V      = %ld\n",
    VName,pihb->Gnaw[1].V);
  if (pihb->Gnaw[2].I)        printf("%s.Gnaw[2].V      = % e\n",
    VName,pihb->Shift[2].V);
  if (pihb->Shift[1].I)       printf("%s.Shift[1].V     = % e\n",
    VName,pihb->Shift[1].V);
  if (pihb->Shift[2].I)       printf("%s.Shift[2].V     = % e\n",
    VName,pihb->Shift[2].V);
  if (pihb->Dummy.I)          printf("%s.Dummy.V        = % e\n",
    VName,pihb->Dummy.V);
  if (pihb->DDummy.I)         printf("%s.DDummy.V       = % e\n",
    VName,pihb->DDummy.V);
  if (pihb->Offset[1].I)      printf("%s.Offset[1].V    = % e\n",
    VName,pihb->Offset[1].V);
  if (pihb->Offset[2].I)      printf("%s.Offset[2].V    = % e\n",
    VName,pihb->Offset[2].V);
  if (pihb->Center[1].I)      printf("%s.Center[1].V    = % e\n",
    VName,pihb->Center[1].V);
  if (pihb->Center[2].I)      printf("%s.Center[2].V    = % e\n",
    VName,pihb->Center[2].V);
                              printf("%s.BinSiz[1].V    = % e, %s.BinSiz[1].I = %d\n",
    VName,pihb->BinSiz[1].V,VName,pihb->BinSiz[1].I);
                              printf("%s.BinSiz[2].V    = % e, %s.BinSiz[2].I = %d\n",
    VName,pihb->BinSiz[2].V,VName,pihb->BinSiz[2].I);
  if (pihb->PixSiz[1].I)      printf("%s.PixSiz[1].V    = % e\n",
    VName,pihb->PixSiz[1].V);
  if (pihb->PixSiz[2].I)      printf("%s.PixSiz[2].V    = % e\n",
    VName,pihb->PixSiz[2].V);
  if (pihb->SampleDistance.I) printf("%s.SampleDistance.V    = % e\n",
    VName,pihb->SampleDistance.V);
  if (pihb->WaveLength.I) printf("%s.WaveLength.V        = % e\n",
    VName,pihb->WaveLength.V);
                              printf("%s.DetRot[1].V    = % e, %s.DetRot[1].I = %d\n",
    VName,pihb->DetRot[1].V,VName,pihb->DetRot[1].I);
                              printf("%s.DetRot[2].V    = % e, %s.DetRot[2].I = %d\n",
    VName,pihb->DetRot[2].V,VName,pihb->DetRot[2].I);
                              printf("%s.DetRot[3].V    = % e, %s.DetRot[3].I = %d\n",
    VName,pihb->DetRot[3].V,VName,pihb->DetRot[3].I);
  if (pihb->Title.I)          printf("%s.Title.V        = %s, %s.Title.I = %d\n",
    VName,pihb->Title.V,VName,pihb->Title.I);
  if (pihb->Time.I)           printf("%s.Time.V         = %s, %s.Time.I = %d\n",
    VName,pihb->Time.V,VName,pihb->Time.I);
  if (pihb->Orientation.I)    printf("%s.Orientation.V  = %ld, %s.Orientation.I = %d\n",
    VName,pihb->Orientation.V,VName,pihb->Orientation.I);
  if (pihb->Projection.I)     printf("%s.Projection.V   = %s, %s.Projection.I = %d\n",
    VName,protostr(pihb->Projection.V),VName,pihb->Projection.I);
  if (pihb->AxisType[1].I)    printf("%s.AxisType[1].V  = %s, %s.AxisType[1].I = %d\n",
    VName,axistostr(pihb->AxisType[1].V),VName,pihb->AxisType[1].I);
  if (pihb->AxisType[2].I)    printf("%s.AxisType[2].V  = %s, %s.AxisType[2].I = %d\n",
    VName,axistostr(pihb->AxisType[2].V),VName,pihb->AxisType[2].I);

} /* PrintImageHeaderBlock */

/*---------------------------------------------------------------------------*/
