/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 binary2saxs
  Conversion of a binary file into ESRF data format.

2 PURPOSE
  Conversion of a binary file into ESRF data format. 

  Arguments:
  binary2saxs [options] <fnam> <onam> <i1dim1> <i1dim2> <fskp> <ftyp> <odum>"

  Defaults:
  <binary file name>  : datafile 
  <output file name>  : output.edf
  <bytes to skip>     : 0 
  <binary type>       : float
  <input dim1>        : <input dimension 1>
  <input dim2>        : <input dimension 2>
  <dummy>             : <dummy value>

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  26-Jan-1996 PB
  27-Jan-1996 PB ReadImageDimension
  28-Jan-1996 PB Binning, Transform
  29-Mar-1996 PB Clip, foff
  22-May-1996 PB Gnaw
  12-Dec-1997 PB byte
  19-Dec-1997 PB Small corrections V3.32
  03-Jan-1998 PB char -> unsigned char (correction) V3.33
  12-Jan-1998 PB binary2saxs: data_size = MAX2(t_size, ...
                 -> data_size = MAX2(t_length, ...  V3.34
  12-Jan-1998 PB All free() calls have been changed to
                 free(xyz);xyz=NULL;
  15-Jan-1998 PB +-fend: endian correction of the input data (byte swap) V3.35
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput 
  09-Nov-1999 PB V3.36 DUMMYDEFINED updated
  2000-08-04  PB %f -> %g
  2001-07-03  PB V3.37 filename_exists(..), filename_full( )
  2001-07-08  PB V3.38 FFirst
  2001-07-09  PB V3.39 (pcb->ib)[1].FileNumber
  2001-07-09  PB V4.00 new loop, no repetition
  2002-02-12  PB V4.01 using standard conversion routines, new options 
                       +/-fhbf for HighByteFirst (default: machine byte order)
                               overrides +/-fend, 
                       -fras   <DataRasterConfiguration> (default: 1),
                       -fdvo   <DataValueOffset> (default: 0)
  2002-04-07  PB V4.02 accepts type styles of version binary2saxs <= V4.00
                       e.g. "unsigned long int" and all other basic standard
                       types e.g. "char", "unsigned char"
                       fend, fras and fdvo in argument list
  2002-05-18  PB V4.03 loop DUMMYDEFINED removed
  2002-05-28  PB V4.04 new call to condition_2d, ERROUT
  2002-06-01  PB V4.50 ave, GetImageRange etc.
  2003-02-02  PB V4.51 ReadImage... ImageNumber -> num[blkno]
  2003-02-04  PB V4.52 condition_2d parameters adapted 
                       (Factor, Const, MinClip, MaxClip)
  2003-02-07  PB V4.53 condition_2d parameters adapted
                       (Bin, Gnaw)
  2003-02-08  PB V4.54 user_io: Dummy -> IO_flexp
  2003-05-30  PB V4.55 ImageLoop: new parameter list
  2003-06-03  PB V4.56 option_parameter_search: Fname, Ftype, Fskip, Fras, Fend
  2003-08-11  PB V4.57 option_define, all option numbers removed
  2003-11-30  PB V4.58 BinSiz
  2004-04-10  PB V4.59 corrections for file name patterns
  2004-10-30  PB V4.60 MAKE_FUNCTION flag added, error exit replaced by return
  2005-07-11  PB V4.61 condition_2d with argument Val
  2005-07-15  PB V4.62 ReportSaxsStatus parameter updated
  2005-08-04  PB V4.63 PrintImageHeaderBlock paramters updated
  2005-08-06  PB V4.64 condition_2d with argument ErrVal
  2005-10-22  PB V4.65 SetSaxsErrorMessage added
  2007-06-10  PB V4.66 ImageLoop parameters updated
  2008-05-07  PB V4.67 BalanceHeaderBlock added, otherwise -i1ori xxx has
                       only effect on the header, not on the data
  2012-01-31  PB V4.68 ImageLoop parameters updated
---*/
# define Version "binary2saxs V4.68 2012-01-31 Peter Boesecke"
# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define SFname "fnam"      /* line option */
# define SFtype "ftyp"

# define SFskip "fskp"      /* long int option */
# define SFras  "fras"
# define SFdvo  "fdvo"

# define SFend "fend"       /* Special options */
# define SFhbf "fhbf"

# define Default_opfn      DefaultOutput
# define Default_inc       1
# define Default_dim_1     512
# define Default_dim_2     512

# define Usage "[options] \n\
    <fnam> <onam> <i1dim1> <i1dim2> <fskp> <ftyp> <fend> <fdvo> <fras> <odum>\n\
    The following types are available : \n\
    ftyp UnsignedByte, SignedByte, UnsignedShort, SignedShort,\n\
         UnsignedInteger, SignedInteger, UnsignedLong, SignedLong,\n\
         FloatValue, DoubleValue"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* types kept for backward compatibility */
const char * STypeFloat    = "float";
const char * STypeDouble   = "double";
const char * STypeChar     = "char";
const char * STypeByte     = "byte";
const char * STypeShort    = "short integer";
const char * STypeInteger  = "integer";
const char * STypeLong     = "long integer";
const char * STypeUChar    = "unsigned char";
const char * STypeUByte    = "unsigned byte";
const char * STypeUShort   = "unsigned short integer";
const char * STypeUInteger = "unsigned integer";
const char * STypeULong    = "unsigned long integer";

/*---------------------------------------------------------------------------
1 string2mtype 

2 PURPOSE
  Returns the type number that corresponds to the data type string, 
  0 if no match found.

  return value	: number of matches 
  char * typestring (i) : string to be analyzed
  long * datatype (o)   : binary data type number 
---------------------------------------------------------------------------*/
int string2mtype ( char * typestring, long * mtype )
{ int cnt;
  
  cnt = 0;
  *mtype = InValidMType;

  if (STypeFloat == strstr(STypeFloat,typestring)) {
     cnt++;*mtype = (long) MFloat;}
  if (STypeDouble == strstr(STypeDouble,typestring)) {
     cnt++;*mtype = (long) MDouble;}
  if (STypeChar == strstr(STypeChar,typestring)) {
     cnt++;*mtype = (long) MChar;}
  if (STypeByte == strstr(STypeByte,typestring)) {
     cnt++;*mtype = (long) MChar;}
  if (STypeShort == strstr(STypeShort,typestring)) {
     cnt++;*mtype = (long) MShort;}
  if (STypeInteger == strstr(STypeInteger,typestring)) {
     cnt++;*mtype = (long) MInteger;}
  if (STypeLong == strstr(STypeLong,typestring)) {
     cnt++;*mtype = (long) MLong;}
  if (STypeUChar == strstr(STypeUChar,typestring)) {
     cnt++;*mtype = (long) MUnsignedChar;}
  if (STypeUByte == strstr(STypeUByte,typestring)) {
     cnt++;*mtype = (long) MUnsignedChar;}
  if (STypeUShort == strstr(STypeUShort,typestring)) {
     cnt++;*mtype = (long) MUnsignedShort;}
  if (STypeUInteger == strstr(STypeUInteger,typestring)) {
     cnt++;*mtype = (long) MUnsignedInteger;}
  if (STypeULong == strstr(STypeULong,typestring)) {
     cnt++;*mtype = (long) MUnsignedLong;}

  return(cnt);

} /* string2mtype */

/*---------------------------------------------------------------------------
1 binary2saxs

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void binary2saxs (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const char * B2S_Error = "ERROR in binary2saxs: ";
  int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  int   I1Ave;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  char fname_full[IO_len];

  IO_line Fname;
  IO_line Ftype;
  IO_long Fskip, Fras, Fdvo, Fend;

  FILE * InputBinaryID;
  size_t t_length, t_size, data_size;
  long FDType;
  long FMType;
  long Dim_1, Dim_2;
  long data_dim[3];
  void *tmp = (void *) NULL; 
  int  tmpI = FALSE; // TRUE, if allocated

  *pstatus = Success;

 /* Check the number of image blocks */
  if (pcb->ImgBlkLen!=2) {
    fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
            pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* Rebalance header block with image block */
    BalanceHeaderBlock  ( pcb, pcb->ib, 1, num, ihb, pstatus );
    if (*pstatus!=Success) return;

 /* --- read binary file and convert to float */

     Fname = option_line ( SFname, pcb, pcb->ib, num, 1, pstatus );
     if (*pstatus!=Success) return;
     Ftype = option_line ( SFtype, pcb, pcb->ib, num, 1, pstatus );
     if (*pstatus!=Success) return;
     Fskip = option_long ( SFskip, pcb, pcb->ib, num, 1, pstatus );
     if (*pstatus!=Success) return;
     Fras = option_long ( SFras, pcb, pcb->ib, num, 1, pstatus );
     if (*pstatus!=Success) return;
     Fdvo = option_long ( SFdvo, pcb, pcb->ib, num, 1, pstatus );
     if (*pstatus!=Success) return;
     Fend = option_flag ( SFend, pcb, pstatus );
     if (*pstatus!=Success) return;

     /* find corresponding machine type FMType */
     /* check first old type style */
     string2mtype ( Ftype.V, &FMType );
     if (!FMType) {
       /* check new type style */
       FDType = edf_string2datatype( Ftype.V );
       FMType = edf_datatype2machinetype( FDType );
     }

     if (!FMType) {
       fprintf(ERROUT,"Type : %s\n",Ftype.V);
       *pstatus = UnsupportedDataType; return; 
       }

     data_dim[0] = 2l;
     (void) ReadImageDimension ( pcb, 1, ihb, 
                                 &data_dim[1], &data_dim[2], pstatus );
     if (*pstatus!=Success) return;

     if (!(t_size = edf_machine_sizeof ( FMType ))) {
       *pstatus = UnsupportedDataType; return; }

     t_length = t_size*edf_dim_product(data_dim);
     data_size = MAX2(t_length, sizeof(float)*edf_dim_product(data_dim)); 

     if (pcb->TestBit) {
       printf("Bytes to skip (bytes)      = %d\n",Fskip.V);
       printf("Data type                  = %s\n", 
                             edf_datatype2string( edf_machinetype2datatype( FMType ) ));
       printf("Data size (bytes)          = %d\n",t_size);
       printf("Binary data length (bytes) = %d\n",t_length);
       printf("Allocated memory (bytes)   = %d\n",data_size);
       printf("Dim_1                      = %d, Dim_2 = %d\n",data_dim[1], data_dim[2]);
       }

     /* Generate filename of binary file */
     filename_parameter (fname_full,IO_len,Fname.V,0);
     filename_pattern (fname_full,IO_len,fname_full,(pcb->ib)[1].FileNumber);
     filename_full (fname_full,IO_len,fname_full);

     /* --- read binary data */
     printf("Reading %s\n",fname_full);
     if ( (InputBinaryID=fopen(fname_full,"rb")) == NULL ) {
       SetSaxsErrorMessage( fname_full );
       *pstatus = FileOpeningError; return; }

     if ( fseek(InputBinaryID, Fskip.V, SEEK_SET) ) {
       SetSaxsErrorMessage( fname_full );
       *pstatus = FilePositioningError; return; }

     if ( (ihb[1].Data = (float *) malloc(data_size) ) == NULL ) {
       *pstatus = NotEnoughMemoryAvailable; return;}

       (void) fread(ihb[1].Data,t_length,1,InputBinaryID);
       if (ferror(InputBinaryID)) {
         SetSaxsErrorMessage( fname_full );
         *pstatus = FileReadError; free(ihb[1].Data); ihb[1].Data=NULL;
         ihb[ 1].ImgByteLen = (long) NULL; return;
         }

     fclose(InputBinaryID);

     /* --- convert binary data */

     /*    --- endian correction */
     if (Fend.V) {
       if (pcb->TestBit) printf(" Swapping bytes \n");
       edf_bswap ( ihb[1].Data,ihb[1].Data,t_size,edf_dim_product(data_dim) );
       }

     /*    --- raster order normalization */
     if (Fras.V!=1l) {
       /* input and output memory must be different, 
          allocate tmp memory for result */
       if (!(tmp = malloc( t_length ))) {
         *pstatus = NotEnoughMemoryAvailable;
         fprintf(ERROUT,"%s malloc of %u bytes failed",B2S_Error,t_length);return; }
       tmpI = TRUE;
   
       if (edf_raster_normalization ( tmp, ihb[1].Data, data_dim, Fras.V, t_size)) {
         *pstatus = Failed;
         free(tmp); return; }
       } else tmp = (void *) ihb[1].Data; 

     /*    --- conversion to float */
     if (pcb->TestBit) printf(" Converting to float \n");

     if ( edf_machine2machine ( ihb[1].Data, MFloat,
                                tmp, Fdvo.V,
                                FMType, edf_dim_product(data_dim) ) ) {
       if (tmpI) free (tmp); 
       *pstatus = TypeConversionFailed; 
       return; }

     if (tmpI) free (tmp);

     /* --- condition data */

     ihb[1].ImgByteLen = (long) t_length;
     ihb[1].Dim[0] = data_dim[0]; 
     ihb[1].Dim[1] = data_dim[1]; ihb[1].Dim[2] = data_dim[2];

     condition_2d            ( &ihb[1],
                               ihb[1].MinClip,     ihb[1].MaxClip,
                               ihb[1].Gnaw[1],     ihb[1].Gnaw[2],
                               ihb[1].Bin[1],      ihb[1].Bin[2],
                               ihb[1].Val,         ihb[1].ErrVal,
                               ihb[1].Factor,      ihb[1].Const,
                               pcb->ib[1].Ave,     ihb[1].Orientation.V,
                               FALSE, pcb->TestBit, pstatus );
     if (*pstatus!=Success) return;

     if (pcb->TestBit) PrintImageHeaderBlock   ( &ihb[1] );

  /* binary file is read */

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");


  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) { 
      free(ihb[1].Data); ihb[1].Data=NULL;
      ihb[ 1].ImgByteLen = (long) NULL; return;
      }

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Ave   = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* loop over the output array  */
  W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {
      if (Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
        f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], &I1Sum, &I1Weight)) {

        /* then do something with the data */
        I1Value = I1Sum; if (I1Ave)  I1Value = I1Sum/I1Weight;
 
        UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );

        } /* if (Ipol2d ... */
      pI0Data++;
      W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; } 
      } /* for i_1 ... */
    W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
    } /* for i_2 ... */

  free(ihb[1].Data); ihb[1].Data = NULL;
  ihb[ 1].ImgByteLen = (long) NULL;

} /* binary2saxs*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];
  long num[BlockNum];
  char fname_pattern[IO_len];
  char tmp_line[IO_len];

  long  image_1st, image_lst;
  long  i1dim_1, i1dim_2;

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  long datatype;
  long FMType, FDType;

  IO_line *pFname, *pFtype;
  IO_long *pFend;
  IO_lvexp *pFskip, *pFras, *pFdvo;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,pcb->MainHelp);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,pcb->MainVersion);
  printf("\n");

  pFname = option_parameter_search  ( SFname, pcb, pstatus );
  if (*pstatus!=Success) return;
  pFtype = option_parameter_search  ( SFtype, pcb, pstatus );
  if (*pstatus!=Success) return;
  pFskip = option_parameter_search  ( SFskip, pcb, pstatus );
  if (*pstatus!=Success) return;
  pFras = option_parameter_search  ( SFras, pcb, pstatus );
  if (*pstatus!=Success) return;
  pFend = option_parameter_search  ( SFend, pcb, pstatus );
  if (*pstatus!=Success) return;
  pFdvo = option_parameter_search  ( SFdvo, pcb, pstatus );
  if (*pstatus!=Success) return;

  /*--- OpenMode of first input file */
  if (!ib[1].OpenMode.I) {
    ib[1].OpenMode.V = IO_Old | IO_FileProtect | IO_DontReadWrite | IO_DontOpen;
    ib[1].OpenMode.I = TRUE;
    }
  /* Under no circumstances read any data */
  ib[1].OpenMode.V = ib[1].OpenMode.V | IO_DontReadData; 

  /*--- Argument  : binary file name */
  if (!pFname->I) {
    /* Determine input file name without directory and without extension */
    filename_name(pFname->V,IO_len,ib[1].Name.V);
    filename_body(pFname->V,IO_len,pFname->V);
    }

  argv_line(pcb,"Binary file name",pFname,pFname->V, pstatus);
  /* Exit in all cases after an abort from the first argument */
  if (*pstatus!=Success) return;

//++++++++++++++ added for file name patterns 2004-04-10
  if (!ib[1].Name.I) {
    // copy binary name to filename
    CopyLine( pFname->V, ib[1].Name.V, IO_len , 0 );
  }

  /* extract filenumbers from binary file name */
  extract_filenumbers( &ib[1], pFname, TRUE, pstatus );
  if (*pstatus!=Success) return;

  /* Generate filename of first binary file and test, whether it exists */
  filename_parameter (fname_pattern,IO_len,pFname->V,0);
  filename_pattern (fname_pattern,IO_len,fname_pattern,ib[1].FFirst.V);
  if (!filename_exists(fname_pattern)) { 
    SetSaxsErrorMessage(fname_pattern);
    *pstatus = FileNotFound; return; 
  }

  OpenImageFile( pcb,ib,1,ib[1].Name.V, ib[1].FFirst.V,
      ib[1].OpenMode.V | IO_Old | IO_FileProtect | IO_DontReadData, pstatus);
  if (*pstatus!=Success) {
    SetSaxsErrorMessage(ib[1].Name.V); return;
  }

  /*--- OpenMode of output file */
  if (!ib[0].OpenMode.I) {
    ib[0].OpenMode.V = IO_ImageProtect;
    ib[0].OpenMode.I = TRUE;
    }

  /*--- Argument  : output file name */
  argv_filename ( pcb, "Output file name", ib, 0, Default_opfn, pstatus);
  if (*pstatus!=Success) return;

  /* Search for minimum and maximum image number */
  (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
  if (*pstatus!=Success) return;

  /*--- number of first image */
  if (!(ib[1].First.I)) ib[1].First.V = image_1st;

  /*--- number of last image */
  if (!(ib[1].Last.I)) ib[1].Last.V = image_lst;

  /*--- Argument : increment */
  if (!(ib[1].Inc.I)) ib[1].Inc.V = Default_inc;

  num[0]=ib[0].First.V;
  num[1]=ib[1].First.V;
  (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
  if (*pstatus!=Success) return;
  if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

  (void) ReadImageDimension ( pcb, 1, ihb, &i1dim_1, &i1dim_2, pstatus );
  if (*pstatus!=Success) return; 

  if (i1dim_1<0) i1dim_1 = Default_dim_1; 
  if (i1dim_2<0) i1dim_2 = Default_dim_2;

  /*--- Argument  : input dimension 1 */
  sprintf(tmp_line,"%d",i1dim_1);
  argv_lvexp( pcb,"Input dimension 1", &ib[1].Dim[1],tmp_line, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : input dimension 2 */
  sprintf(tmp_line,"%d",i1dim_2);
  argv_lvexp( pcb,"Input dimension 2", &ib[1].Dim[2],tmp_line, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : bytes to skip */
  argv_lvexp(pcb,"Bytes to skip (bytes)",pFskip, pFskip->V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : input data type */
  edf_showdatatypes ( TRUE );
  argv_line(pcb,"Input data type",pFtype,pFtype->V, pstatus);
  if (*pstatus!=Success) return;

  /* check type */

  /* check first old type style */
  string2mtype ( pFtype->V, &FMType );
  if (!FMType) {
    /* check new type style */
    FDType = edf_string2datatype(pFtype->V);
    FMType = edf_datatype2machinetype( FDType );
  }

  if (!FMType) {
    fprintf(ERROUT,"Type : %s\n",pFtype->V);
    *pstatus = UnsupportedDataType; return;
    }

  if (pcb->TestBit) 
    printf("TypeString = %s, data type = %s\n",pFtype->V, 
      edf_datatype2string( edf_machinetype2datatype( FMType ) ) );

  /*--- Argument  : input data byte order */
  argv_flag(pcb,"Input endian correction",pFend, pFend->V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : input data value offset */
  argv_lvexp(pcb,"Input data value offset",pFdvo, pFdvo->V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : data raster configuration */
  argv_lvexp(pcb,"Input data raster configuration", pFras, pFras->V, pstatus);
  if (*pstatus!=Success) return;

  /* Read header again */
  (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
  if (*pstatus!=Success) return;
  if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);
     
  /*--- Argument  : output dummy */
  sprintf(tmp_line,"%g",ihb[1].Dummy.V);
  argv_flexp( pcb,"Output dummy", &ib[0].Dummy,tmp_line, pstatus);
  if (*pstatus!=Success) return;

  /*--- output dimension 1 */
  sprintf(tmp_line,"%d",ihb[1].Dim[1]);
  if (!(ib[0].Dim[1].I)) CopyLine(tmp_line,ib[0].Dim[1].V,IO_len,0);

  /*--- output dimension 2 */
  sprintf(tmp_line,"%d",ihb[1].Dim[2]);
  if (!(ib[0].Dim[2].I)) CopyLine(tmp_line,ib[0].Dim[2].V,IO_len,0);

  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  CloseImageFile( pcb, ib, 1, pstatus) ;
  if (*pstatus!=Success) return;

  printf("\n");
  if (pFname->I) 
                      printf("binary file           : %s\n",pFname->V);
  if (ib[0].Name.I)   printf("output file           : %s\n",ib[0].Name.V);
  if (ib[1].Dim[1].I) printf("input dimension 1     : %s\n",ib[1].Dim[1].V);
  if (ib[1].Dim[2].I) printf("input dimension 2     : %s\n",ib[1].Dim[2].V);
  if (pFskip->I)
                      printf("bytes to skip (bytes) : %s\n",pFskip->V);
  if (pFtype->I)
                      printf("data type             : %s\n",pFtype->V);
  if (pFend->I)
                      printf("endian correction     : %s\n",
                                                 pFend->V?"yes":"no");
  if (pFras->I)
                      printf("raster configuration  : %s\n",pFras->V);
  if (pFdvo->I)
                      printf("data value offset     : %s\n",pFdvo->V);

  if (ib[0].Dummy.I)  printf("output dummy          : %s\n",ib[0].Dummy.V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_binary2saxs
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

  IO_long Fhbf, Fend;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  option_define   ( IO_tpline, SFname, "datafile", &cb );
  option_define   ( IO_tplvexp, SFskip, "0", &cb );
  option_define   ( IO_tpflag, SFend, "FALSE", &cb );
  option_define   ( IO_tpflag, SFhbf, "FALSE", &cb );
  option_define   ( IO_tplvexp, SFras, "1", &cb );
  option_define   ( IO_tpline, SFtype, edf_datatype2string(FloatIEEE32), &cb );
  option_define   ( IO_tplvexp, SFdvo, "0", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* Fhbf overrides Fend */
  if (status==Success)
    Fhbf = option_flag  ( SFhbf, &cb, &status );
  
  if (status==Success)
    if (Fhbf.I) {
      Fend.I = TRUE;
      if (edf_byteorder()==HighByteFirst) Fend.V = Fhbf.V?0:1;
        else Fend.V = Fhbf.V?1:0;
      option_parameter_update  ( Fend.V?"1":"0", Fend.I, SFend, &cb, &status );
    }

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, binary2saxs, NULL, NULL, TRUE, &status );

  return ( ReportSaxsStatus( status, 0 ) );

} /* main */
