/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_grid
  Generates an image with a grid pattern

2 PURPOSE
  Generates an image with a grid pattern

  Arguments:
  saxs_grid [options] <onam> <odum> <dim 1> <dim 2> <ocon>"

  Defaults:
  output file name    : <onam>              : output.edf
  output dummy        : <odum>              : 0.0
  dimension 1         : <odim 1>            : 1
  dimension 2         : <odim 2>            : 1
  output constant     : <ocon>              : 0.0 (only used if dummy = 0)

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2001-09-10  PB V4.00 from saxs_new V4.00 
  2002-06-01  PB V4.50 ave
  2003-02-09  PB V4.51 line args for Dummy, DDummy, Dim
  2003-05-30  PB V4.52 ImageLoop: Function call changed
  2003-08-13  PB V4.53 option_define, all option numbers removed
  2003-11-30  PB V4.54 BinSiz
  2004-10-31  PB V4.55 MAKE_FUNCTION
  2005-07-15  PB V4.56 ReportSaxsStatus parameter list updated
  2007-06-11  PB V4.57 ImageLoop parameters updated
  2012-01-31  PB V4.58 ImageLoop parameters updated

---*/
# define Version  "saxs_grid V4.58 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
                <onam> <odum> <odim1> <odim2>\n\
                <anc1> <anc2> <gri1> <gri2)\n\
                <diam> <mval[0.0]> <pval[1.0]>\n\
                Generates an image with a grid pattern"

# define BlockNum 1       /* 1 output sequence */

/* Float options */
# define SAnch_1  "anc1"
# define SAnch_2  "anc2"
# define SPitch_1 "gri1"
# define SPitch_2 "gri2"
# define SDiam    "diam"
# define SMatVal  "mval"
# define SPeakVal "pval"

/*---------------------------------------------------------------------------
 internal functions 
---------------------------------------------------------------------------*/

static const float grid_eps = 1e-32;
/*---------------------------------------------------------------------------
NAME
  grid --- returns 1 if pixel (x_1, x_2) is inside a grid point 

SYNOPSIS 
  int isgridpoint( float anchor_1, float anchor_2, 
                   float pitch_1, float pitch_2,
                   float radius, 
                   float x_1, float x_2 );

DESCRIPTION
  The function tests whether (N = ..., -2, -1, 0, 1, 2, ...)
      ( sqr( x_1 - ( anchor_1 + N * pitch_1 ) ) +
        sqr( x_2 - ( anchor_2 + N * pitch_2 ) )) <= sqr(radius)
  where N is a positive or negative integer number
  It returns 1, if it is the case, otherwise 0, -1 in case of an error.
---------------------------------------------------------------------------*/
int isgridpoint( float anchor_1, float anchor_2, 
                 float pitch_1, float pitch_2, 
                 float radius, 
                 float x_1, float x_2 )
{ double x_1d, x_2d;
  double N_1, N_2;
  double invpitch_1, invpitch_2; 
  double tmp_1, tmp_2, dist2, radius2;

  if ( ( fabs(pitch_1)<grid_eps )||( fabs(pitch_2)<grid_eps ) ) {
    fprintf(ERROUT,"ERROR: Division by zero \n");return (-1); }

  invpitch_1 = 1.0/pitch_1;
  invpitch_2 = 1.0/pitch_2;

  x_1d = (double) x_1;
  x_2d = (double) x_2;
  radius2 = (double) radius * (double) radius;

  // round N_1 and N_2
  N_1 = floor( (x_1d - anchor_1) * invpitch_1 + (double) 0.5 );   
  N_2 = floor( (x_2d - anchor_2) * invpitch_2 + (double) 0.5 );
  tmp_1 = x_1d - ( N_1 * pitch_1 + anchor_1 );
  tmp_2 = x_2d - ( N_2 * pitch_2 + anchor_2 );
  dist2 = tmp_1*tmp_1 + tmp_2*tmp_2;

  if (dist2 <= radius2 ) 
    return(1); // it's a grid point
   else return(0); // it's a matrix point

} // grid

/*---------------------------------------------------------------------------
1 saxs_grid

2 PURPOSE
  Generates an image with a grid pattern
---------------------------------------------------------------------------*/
void saxs_grid (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float Value;

  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  int   itis;

  IO_float Anch_1, Anch_2, Pitch_1, Pitch_2, Diam, MatVal, PeakVal;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=1) {
    fprintf(ERROUT,"%d images found, 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

 /* 1 output */

  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

 /* parameters */
  Anch_1 = option_float ( SAnch_1, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Anch_2 = option_float ( SAnch_2, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Pitch_1 = option_float ( SPitch_1, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Pitch_2 = option_float ( SPitch_2, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Diam = option_float ( SDiam, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  MatVal = option_float ( SMatVal, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  PeakVal = option_float ( SPeakVal, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  /* loop over the output array */
  W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
    W_2 = WORLD( i_2, Off_2[0], Ps_2[0] );
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {
      W_1 = WORLD( i_1, Off_1[0], Ps_1[0] );
      /* do something with the data */

      itis = isgridpoint( Anch_1.V, Anch_2.V,
             Pitch_1.V, Pitch_2.V, Diam.V*0.5, W_1, W_2 );
      if (itis==0) Value = MatVal.V;
        else if (itis>0) Value = PeakVal.V;
          else { *pstatus = DivByZero; return; } // error
 
      UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
 
      pI0Data++;
      W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */
 
    W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
  } /* for i_2 ... */ 

} /* saxs_grid*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  char tmp_line[IO_len];

  IO_flexp *pAnch_1, *pAnch_2, *pPitch_1, *pPitch_2, *pDiam;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,0,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,0,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument : world coordinate 1 of anchor */
  sprintf(tmp_line,"Position 1 of grid anchor (%s coordinates)",
          reftostr(pcb->RSys.V));
  pAnch_1 = (IO_flexp*) option_parameter_search( SAnch_1, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,tmp_line,pAnch_1,pAnch_1->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : world coordinate 2 of anchor */
  sprintf(tmp_line,"Position 2 of grid anchor (%s coordinates)",
          reftostr(pcb->RSys.V));
  pAnch_2 = (IO_flexp*) option_parameter_search( SAnch_2, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,tmp_line,pAnch_2,pAnch_2->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : world pitch 1 of grid */
  sprintf(tmp_line,"Pitch 1 of grid (%s coordinates)",
          reftostr(pcb->RSys.V));
  pPitch_1 = (IO_flexp*) option_parameter_search( SPitch_1, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,tmp_line,pPitch_1,pPitch_1->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : world pitch 2 of grid */
  sprintf(tmp_line,"Pitch 2 of grid (%s coordinates)",
          reftostr(pcb->RSys.V));
  pPitch_2 = (IO_flexp*) option_parameter_search( SPitch_2, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,tmp_line,pPitch_2,pPitch_2->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : diameter of points */
  sprintf(tmp_line,"Diameter of grid points (%s coordinates)",
          reftostr(pcb->RSys.V));
  pDiam = (IO_flexp*) option_parameter_search( SDiam, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,tmp_line,pDiam,pDiam->V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
                      printf("reference system   : %s\n",reftostr(pcb->RSys.V));
  if (pAnch_1->I)      printf("grid anchor 1      : %s (reference system %s)\n",
                       pAnch_1->V, reftostr(pcb->RSys.V));
  if (pAnch_2->I)      printf("grid anchor 2      : %s (reference system %s)\n",
                       pAnch_2->V, reftostr(pcb->RSys.V));
  if (pPitch_1->I)     printf("pitch  1           : %s (reference system %s)\n",
                       pPitch_1->V, reftostr(pcb->RSys.V));
  if (pPitch_2->I)     printf("pitch  2           : %s (reference system %s)\n",
                       pPitch_2->V, reftostr(pcb->RSys.V));
  if (pDiam->I)        printf("Point diameter     : %s (reference system %s)\n",
                       pDiam->V, reftostr(pcb->RSys.V));

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_grid
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

  sprintf(ib[0].Dim[1].V,"%d",512);
  sprintf(ib[0].Dim[2].V,"%d",512);

 /* Define special options */
  option_define ( IO_tpflexp, SAnch_1,  "0.0", &cb );
  option_define ( IO_tpflexp, SAnch_2,  "0.0", &cb );
  option_define ( IO_tpflexp, SPitch_1, "50.0", &cb );
  option_define ( IO_tpflexp, SPitch_2, "50.0", &cb );
  option_define ( IO_tpflexp, SDiam,    "5.0", &cb );
  /* No prompting for the following parameters */
  option_define ( IO_tpflexp, SMatVal,  "0.0", &cb );
  option_define ( IO_tpflexp, SPeakVal, "1.0", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, saxs_grid, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

