/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_power
  Taking the n-th power of positive values out of the image.

2 PURPOSE
  Taking the n-th power of positive values out of the image.

  Calculation:       I0 = pow(I1,n)
  Error propagation: V0 = V1*n*pow(I1,n-1)

  Arguments:
  saxs_power [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <power>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <power>             : 2.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  05-Dec-1996 PB V1.0 creation
  06-Dec-1996 PB V1.01 no checks for negative powers in pow(xxx,power)
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  2000-08-04  PB %f->%g
  2001-07-09  PB V4.00 new loop, no repetition
  2002-06-01  PB V4.50 ave 
  2003-02-09  PB V4.51 line args for Dummy, DDummy, Dim
  2003-05-30  PB V4.52 ImageLoop: Function call changed
  2003-08-14  PB V4.53 option_define, all option numbers removed
  2003-11-30  PB V4.54 BinSiz
  2004-10-31  PB V4.55 MAKE_FUNCTION
  2004-12-07  PB V4.56 special condition for integer exponents
  2005-07-15  PB V4.57 ReportSaxsStatus parameter list updated
  2005-09-17  PB V4.58 VarDat
  2007-06-11  PB V4.59 ImageLoop parameters updated
  2012-01-31  PB V4.60 ImageLoop parameters updated

---*/
# define Version  "saxs_power V4.60 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <math.h>

# include "SaxsPrograms.h"

# define SPower       "pow"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <power> "

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
1 saxs_power

2 PURPOSE
  Taking the n-th power of positive values out of the image.
 ---------------------------------------------------------------------------*/
void saxs_power (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy;
  float Value, E0Value;

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  /* Do not change the following definitions,
     they are automatically changed with BlockNum */
  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
 
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
 
  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];
 
  int Imin_1, Imin_2, Imax_1, Imax_2;
 
  float Wmin_1, Wmax_1, W_1, DW_1;
  float Wmin_2, Wmax_2, W_2, DW_2;

  IO_float Power;
  int posonly = TRUE;
  const float eps=1e-32;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  Power = option_float ( SPower, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);

 /* 1 input, 1 output */
  I0Data   = ihb[0].Data;
  E0Data   = ihb[0].VarDat;
  I0Dummy  = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave    = pcb->ib[1].Ave.V;
  I1Data   = ihb[1].Data;
  E1Data   = ihb[1].VarDat;
  I1Dummy  = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* test, whether the exponent is an integer number */
  if (fabs(Power.V - (int) Power.V)<eps) {
    posonly=FALSE; Power.V = (int) Power.V;
  } else posonly=TRUE;

  /* loop over the output array */
  W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
 
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {

      if ( E0Data ) {
        VarDDummy = DDSET(VarDummy);
        if ( Isum2ldwE (I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
               f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], 
               &I1Sum, &I1Weight, &E1Sum, &E1Weight) ) {
 
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight;
 
          if (posonly) { // accept only positive values
            if (I1Value>0.0) {
              /* no zeros and negative values */
              Value = pow(I1Value,Power.V);
              UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
              if (E1Sum >= 0.0) {
                //  V0 = V1*n*pow(I1,n-1)
                E1Value = E1Sum; if (I1Ave) E1Value /= E1Weight*E1Weight;
                E0Value = E1Value*Power.V*Value/I1Value;
                UPDATE( *pE0Data, E0Value, -1.0, 0.1 );
              }
            }
          } else { // accept all values
            if (Power.V==0.0) {
              // Value = 1.0;
              UPDATE( *pI0Data, 1.0, I0Dummy, I0DDummy );
              // E0Value = 0.0;
              UPDATE( *pE0Data, 0.0, VarDummy, VarDDummy );
            } else {
              Value = pow(I1Value,Power.V);
              UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
              if (E1Sum >= 0.0) {
                //  V0 = V1*n*pow(I1,n-1)=V1*n*Value/I1
                E1Value = E1Sum; if (I1Ave) E1Value /= E1Weight*E1Weight;
                if (fabs(I1Value)>eps)
                  E0Value = E1Value*Power.V*Value/I1Value;
                else E0Value = 0.0;
                UPDATE( *pE0Data, E0Value, VarDummy, VarDDummy );
              }
            }
          }
 
        } /* if (Isum2ldwE ... */

      } else {
 
        if (Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
              f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], &I1Sum, &I1Weight)) {
 
          I1Value = I1Sum; if (I1Ave) I1Value /= I1Weight; 

          if (posonly) { // accept only positive values
            if (I1Value>0.0) {
              /* no zeros and negative values */
              Value = pow(I1Value,Power.V);
              UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
            }
          } else { // accept all values
            if (Power.V==0.0) Value = 1.0;
            else Value = pow(I1Value,Power.V);
            UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
          }
 
        } /* if (Isum2ldw ... */
      }
 
      pI0Data++;
      pE0Data++;
      for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */
 
    for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
  } /* for i_2 ... */

} /* saxs_power*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_flexp *pPower;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : power */
  printf("<output image> = <input image>^power\n");
  pPower = (IO_flexp*) option_parameter_search( SPower, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb,"Power",pPower,pPower->V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (pPower->I)       printf("power              : %s\n",pPower->V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_power
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Define special options */
  option_define ( IO_tpflexp, SPower,    "2.0", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_power, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

