/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_pol
  Conversion of a normalized scattering image from scattered photons per
  sterad per incident photons per meter thickness to electron units 
  (polarization correction).
  The output units is the number of equivalently scattering electrons per
  cubic nanometer (-mul) or the inverse (+mul):
  The input image will either be divided or multiplied with the 
  polarization factor. The default is a division:

  -mul (default):

        input: (scattered_photons/sterad)/incident_photons/thickness
    -> output: (scattering_electrons/nm3)/(thickness/m)

  +mul 

        input: (scattering_electrons/nm3)/(thickness/m)
    -> output: (scattered_photons/sterad)/incident_photons/thickness

  To correct only the angular dependence of the polarization factor without
  taking the scattering cross section of electrons into account set the factor
  ff to 1 (option -ff 1).
  If the projection type of the image is Saxs (default), the polarization
  correction is applied to an image that was taken by a rotated detector.
  In this case the header parameters DetectorRotation_1, _2 and _3 are
  used to define the detector rotation angels (default is 0 for all angles).
  The switch +mul allows to multiply an image with the polarization factor
  instead of dividing it.
  If the projection type of the image is Waxs it is an Ewald-sphere 
  projection. In this case the polarization factor is calculated without 
  rotation from the SAXS-coordinates sp of the projection.

  Pol   : degree of polarization (0<=P<=1)
  PChi  : ellipticity (after Poincar) (-pi/4<=PChi<=+pi/4)
          PChi=-pi/4 left hand (cw) circular polarization
          PChi<0 left hand polarization
          PChi==0 linear polarization
          PChi>0 right hand polarization
          PChi=pi/4 right hand (ccw) circular polarization
  PPsi  : polarization direction (after Poincar) (0<=PPsi<pi)
          PPsi is the angle between axis x_1 and the polarization direction
          measured ccw around axis x_1

  The scattering vectors are calculated in saxs-coordinates (-rsys saxs).

2 PURPOSE
  Multiplication with the polarization factor

  Calculation:       I0 = I1*polfac
  Error propagation: V0 = V1*polfac*polfac

  Arguments:
  saxs_pol [options] [-mul]
             <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <pol[1.0]> <pchi[0.0]> <ppsi[0.0]>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <pol>               : degree of polarization (0<=pol<=1), default 1, totally
                        polarized
  <pchi>              : ellipticity (-p/4<=pchi<=p/4), default 0, linear 
                        polarization
  <ppsi>              : polarization direction (0<=ppsi<pi), default 0,
                        x_1 is the polarization axis

2 HISTORY
  2004-07-25  Peter Boesecke V4.50 from saxs_waxs V4.09
  2004-09-26  PB V4.51 reading W0type from ihb[0].Projection.V, 
                       option saxs removed 
  2004-10-14  PB V4.52 options rot1, rot2, rot3 removed (use i1rot)
  2004-10-31  PB V4.53 MAKE_FUNCTION
  2004-10-31  PB V4.54 option ff, default "re*re/nm3", the result
                       of Invert=TRUE is electrons/nm3
  2005-02-11  PB V4.55 return, if reference system is not saxs
                       error messages to ERROUT
  2005-07-15  PB V4.56 ReportSaxsStatus parameter list updated
  2005-09-17  PB V4.57 VarDat
  2007-06-11  PB V4.58 ImageLoop parameters updated
  2010-05-31  PB V4.59 update for polarization.c V1.4
  2011-06-22  PB V4.60 update for polarization.c V1.6 (Orientation)
  2012-01-31  PB V4.61 ImageLoop parameters updated

---*/
# define Version  "saxs_pol V4.61 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "waxs.h"
# include "polarization.h"

# define Usage "[-mul] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> \n\
                <pol[1.0]> <pchi[0.0]> <ppsi[0.0]>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Special options */
# define SPol  "pol"    /* float */
# define SPChi "pchi"   /* float */
# define SPPsi "ppsi"   /* float */
# define SFf   "ff"     /* float */
# define SMul  "mul"    /* flag */

/*---------------------------------------------------------------------------
1 saxs_pol

2 PURPOSE
  Projection of an inclined detector surface
---------------------------------------------------------------------------*/
void saxs_pol (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;
  const char * RoutineName = "saxs_pol";
  int   i, imax; 

  float *I0Data, *E0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  PParams I1pparams;

  /* Do not change the following definitions,
     they are automatically changed with BlockNum */
  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float f1_1[BlockNum], f3_1[BlockNum], Df_1[BlockNum];
  float f1_2[BlockNum], f3_2[BlockNum], Df_2[BlockNum];

  int Imin_1, Imin_2, Imax_1, Imax_2;

  float Wmin_1, Wmax_1, DW_1;
  float Wmin_2, Wmax_2, DW_2;

  WaxsCoord W0;
  int W0type;
  float polfac;

  long ori;
  double rot1, rot2, rot3;
  double K, pol, pchi, ppsi;
  double ff;

  IO_long  Mul, Saxs;
  IO_float Pol, PChi, PPsi;
  IO_float Ff;

  float I0Off_1, I0Ps_1;
  float I0Off_2, I0Ps_2;
  float I1Off_1, I1Ps_1;
  float I1Off_2, I1Ps_2;

  SetSaxsError( Version, RoutineName, NULL );
 
  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* --- Get option values */ 
  Mul = option_flag( SMul, pcb, pstatus);
  if (*pstatus!=Success) return;
  Ff = option_float ( SFf, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Pol = option_float ( SPol, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  PChi = option_float ( SPChi, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  PPsi = option_float ( SPPsi, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

 /* 1 input, 1 output */
  I0Data   = ihb[0].Data;
  E0Data   = ihb[0].VarDat;
  I0Dummy  = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Ave    = pcb->ib[1].Ave.V;
  I1Data   = ihb[1].Data;
  E1Data   = ihb[1].VarDat;
  I1Dummy  = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* check the reference system: Saxs required */
  if (pcb->RSys.V != IO_Saxs) {
    fprintf(ERROUT,"ERROR: Reference system %s required\n",
      reftostr(IO_Saxs));
    fprintf(ERROUT,"       Start program with option -rsys %s\n",
      reftostr(IO_Saxs));
    *pstatus=Failed; return; 
    }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  GetImageRange         ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2,
                          f1_1, f3_1, Df_1, f1_2, f3_2, Df_2,
                          &Imin_1, &Imax_1, &Imin_2, &Imax_2,
                          &Wmin_1, &Wmax_1, &DW_1, &Wmin_2, &Wmax_2, &DW_2, 1);
 
  ori  = ihb[1].Orientation.V;
  K    = (double) WAVENUMBER(ihb[1].WaveLength.V);
  rot1 = (double) ihb[1].DetRot[1].V;
  rot2 = (double) ihb[1].DetRot[2].V;
  rot3 = (double) ihb[1].DetRot[3].V;
  pol  = (double) Pol.V;
  pchi = (double) PChi.V;
  ppsi = (double) PPsi.V;
  ff   = (double) Ff.V;

  W0type = (int) ihb[0].Projection.V;

  if (polarization_Init(&I1pparams, ori,
                        K, rot1, rot2, rot3, pol, pchi, -ppsi, ff, Mul.V?0:1)) {
    fprintf(ERROUT,"Polarization initialization error\n") ; 
    *pstatus=OutOfRange; return; 
  }
 
  SetSaxsError( Version, RoutineName, NULL );

  if (pcb->TestBit) {
    polarization_PrintParams ( stdout, I1pparams );
    printf("\n"); }

  /* loop over the output array  */ 
  W0.s_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=f1_2[i];
  for (i_2=Imin_2;i_2<=Imax_2;i_2++) {
  
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    W0.s_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=f1_1[i];
    for (i_1=Imin_1;i_1<=Imax_1;i_1++) {

      if ( E0Data ) {
        // V0 = V1*polfac*polfac
        if ( Isum2ldwE(I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
               f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1],
               &I1Sum, &I1Weight, &E1Sum, &E1Weight) ) {

          polfac = (float) polarization_factor ( &I1pparams, W0, W0type );

          if ( polfac >= 0.0 ) {
            I1Value = I1Sum*polfac; if (I1Ave) I1Value /= I1Weight;
            UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
            if (E1Sum >= 0.0) {
              E1Value = E1Sum*polfac*polfac; 
              if (I1Ave) E1Value /= E1Weight*E1Weight;
              UPDATE( *pE0Data, E1Value, VarDummy, DDSET(VarDummy) );
            }
          }

        }
      } else {
        if ( Isum2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
               f_1[1], f_2[1], f_1[1]+Df_1[1], f_2[1]+Df_2[1], 
               &I1Sum, &I1Weight) ) {
  
          polfac = (float) polarization_factor ( &I1pparams, W0, W0type );

          if ( polfac >= 0.0 ) {
            I1Value = I1Sum*polfac; if (I1Ave) I1Value /= I1Weight;
            UPDATE( *pI0Data, I1Value, I0Dummy, I0DDummy );
          }

        }
      }
      pI0Data++;
      pE0Data++;
      W0.s_1+=DW_1; for (i=0;i<imax;i++) f_1[i]+=Df_1[i];

    } /* for i_1 ... */

    W0.s_2+=DW_2; for (i=0;i<imax;i++) f_2[i]+=Df_2[i];
  } /* for i_2 ... */

} /* saxs_pol*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_len];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_flexp *pPol, *pPChi, *pPPsi;

  /*--- Determine program name without directory ---*/
  filename_name ( progfn, IO_len,  pcb->argv[0] );

  /*--- Show usage if no arguments are given ---*/
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*---             Description of polarization */
  if (pcb->Prompt.V) {
    printf("\n");
    printf("Division/multiplication by the polarization factor.\n");
    printf("Angles must be given in radians or must be followed\n");
    printf("by a unit, e.g 30_deg.\n");
    printf("\n");
    printf("Description of the polarization by the Poincar sphere.\n");
    printf("The polarization state of the primary beam is defined by \n");
    printf("three parameters\n");
    printf("  Pol : degree of polarization (0<=Pol<=1)\n");
    printf("  PChi: ellipticity (-45_deg<=PChi<=+45_deg)\n");
    printf("        PChi=-45_deg left hand (cw) circular polarization\n");
    printf("        PChi<0 left hand polarization\n");
    printf("        PChi==0 linear polarization\n");
    printf("        PChi>0 right hand polarization\n");
    printf("        PChi=45_deg right hand (ccw) circular polarization\n");
    printf("  PPsi: rotation of the polarization ellipse/axis (0<=PPsi<180_deg)\n");
    printf("        The angle PPsi is measured ccw around axis x_3 starting\n");
    printf("        at axis x_1.\n");
    printf("\n");
    printf("The program can be started with the following option:\n");
    printf(" +mul : Multiply input image with polarization factor.\n"); 
    printf("\n");
  }

  /*--- Argument  : Polarization */
  pPol = (IO_flexp*) option_parameter_search( SPol, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb," degree of polarization (0<=Pol<=1)",
             pPol,pPol->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : Ellipticity */
  pPChi = (IO_flexp*) option_parameter_search( SPChi, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb," ellipticity of polarization ellipse (-45_deg<=PChi<=+45_deg)",
             pPChi,pPChi->V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : Rotation angle of polarization ellipse around axis 3 */
  pPPsi = (IO_flexp*) option_parameter_search( SPPsi, pcb, pstatus );
  if (*pstatus!=Success) return;
  argv_flexp(pcb," rotation angle of polarization ellipse (0<=PPsi<180_deg)",
             pPPsi,pPPsi->V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file               : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file               : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image            : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image             : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment              : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy           : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1     : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2     : %s\n",ib[0].Dim[2].V);
  if (pPol->I)         printf("degree of polarization : %s\n",pPol->V);
  if (pPChi->I)        printf("ellipticity angle pchi : %s\n",pPChi->V);
  if (pPPsi->I)        printf("orientation angle ppsi : %s\n",pPPsi->V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_pol
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Default reference system is SAXS */
  cb.RSys.V = IO_Saxs; /* default */

 /* Define special options */
  option_define ( IO_tpflag,  SMul,    "FALSE",    &cb );

  option_define ( IO_tpflexp, SPol ,     "1.0",    &cb );
  option_define ( IO_tpflexp, SPChi, "0.0_deg",    &cb );
  option_define ( IO_tpflexp, SPPsi, "0.0_deg",    &cb );

  option_define ( IO_tpflexp, SFf ,"re*re/nm3",    &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_pol, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

