/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_average
  Creates an average and variances from a series of images.

2 PURPOSE
  Creates an average and variances from a series of images.
  All images must have the same geometries:

  Dim_1, Dim_2, 
  Offset_1, Offset_2
  Dummy, Ddummy

  No reference system other than array is possible.

  -add <#naverage> chooses the number of images to average 
                   (default 0, all)

  Calculation:       I0 = average(I1)  == average of Add.V images
  Error propagation: V0 = variance(I1) == variance of Add.V images
                                          (input variances are ignored)

  Arguments:
  saxs_average [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <ofac> <ocon>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <output const>      : 0.0
  <output factor>     : 1.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2012-02-09  PB 4.00
  2012-06-26  PB 4.01 strlib.h included, 
                       4) calculate variance of the average
                         --> variance calculation corrected
  2012-06-27  PB 4.02 percentchar added to avoid lcc format string bug

---*/
# define Version  "saxs_average V4.02 2012-06-27, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "statfunc.h"
# include "strlib.h"
# include "SaxsPrograms.h"

# define FREE( a) { if ( a) { free( a); a = NULL; } }

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <ofac> <ocon> \n\n\
                Calculation of the average and the variance of a series\n\
                of images. All images that are to average will be kept\n\
                in memory. If the option +var is set also the variance with \n\
                respect to the calculated average is calculated and written \n\
                into the error image (-omem -1).\n\
                options: -add <number of images to average>\n\
                              (default 0 to average all)\n\
                         -amod   <mean|median|quantil> average function\n\
                                   (default: mean)\n\
                         -fmod   <mean|median|quantil> filter function\n\
                                   (default: none)\n\
                         +/-n-1  <calculate empiric variance>\n\
                                   (default +n-1)\n\
                         -sig-   <lowfac> to filter values < <lowfac>*sigma\n\
                                   (default 5)\n\
                         -sig+   <upfac> to filter values > <upfac>*sigma\n\
                                   (default 5)\n\
                         -mins   <minimum sigma> for filter\n\
                                   (default 0.5)\n\
                         -pq     <quantil value p> [1-p..1] (0.5<=p<1)\n\
                                   (default 0.9)"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

# define SFramename   "fnam"  /* line */  // frame-file name
# define SAMode       "amod"  /* line */  // averaging mode 
# define SFMode       "fmod"  /* line */  // filter mode 
# define SNM1         "n-1"   /* flag */  // calculate variance with n-1
# define SSIGM        "sig-"  /* double */  // lower sigma limit factor
# define SSIGP        "sig+"  /* double */  // upper sigma limit factor
# define SMINS        "mins"  /* double */  // minimum sigma for filter 
# define SPQ          "pq"    /* double */  // quantil value (0.5<=pq<1) 

const float eps=1e-32;
const char percentchar='%';

/***************************************************************************
* Averaging Mode Translation Tables                                        *
***************************************************************************/
enum AMode { InValidAMode, Mean=1, Median, Quantil, EndAMode };

static const char * AModeStrings[] = { "Invalid", "mean", "median", 
                                       "quantil", (const char *) NULL };
/*---------------------------------------------------------------------------
NAME
  String2AMode --- converts a string to an averaging mode 

SYNOPSIS
  (AMode) int String2AMode( const char * string );

DESCRIPTION

RETURN VALUE
  AMode == 0 : error, e.g. cannot convert
  AMode >  0 : valid averaging mode value 

  -------------------------------------------------------------------------*/
int string2amode( const char *string )
{ int  NE=TRUE;
  int i, value=InValidAMode;
  size_t n;

  n = strlen(string);

  for (i=0;i<EndAMode;i++) {
    if (!(strlib_ncasecmp( string, AModeStrings[i], n ))) {
      value = i;
      break;
    };
  }

  return( value );

} // string2amode

/*---------------------------------------------------------------------------
NAME
  amode2string --- converts an averaging mode to a string

SYNOPSIS
  const char *amode2string( (AMode) int amode );

DESCRIPTION

RETURN VALUE
  "Invalid" : error, e.g. cannot convert
  averaging mode strings (in case of no errors)

  -------------------------------------------------------------------------*/
const char *amode2string( int amode )
{ if ((amode<0)||(amode>=EndAMode)) amode = InValidAMode;
   return( AModeStrings[amode] );
} // amode2string

/*---------------------------------------------------------------------------
NAME

  new_main_data, free_main_data --- Creates and releases common variables 

PURPOSE
  Creates and releases variables used by all functions.
  Must be declared in Command Block with InitOptions.

---------------------------------------------------------------------------*/
typedef struct {
  float *data;        // stacked input images
  size_t data_size;   // size of data
  float *dummy;       // stacked dummy values, one per frame 
  size_t dummy_size;  // size of ddummy
  float *ddummy;      // stacked dummy values, one per frame 
  size_t ddummy_size; // size of ddummy
  long   counter;     // counter of stacked frames

} MainData;

MainData * new_main_data( void )
{ return( (MainData *) calloc( 1, sizeof ( MainData ) ) );
} // new_main_data

MainData * free_main_data ( MainData * main_data )
{
  if (main_data) {
    if (main_data->data) free( main_data->data );
    if (main_data->dummy) free (main_data->dummy);
    if (main_data->ddummy) free (main_data->ddummy);
    free (main_data);
    main_data = NULL;
  }
  return( main_data );
} // free_main_data

/*---------------------------------------------------------------------------
NAME
  average --- Calculate average and variance of pixel (i_1,i_2)

PURPOSE
  Calculate average and variance for pixel (i_1,i_2). The type of average
  (mean, median) can be chosen. The routine takes into account dummy values.

ARGUMENTS
  const long dim[]        : input dimension dim = { 3, dim_1, dim_2, dim_3 }
  const float *frames (i) : input frames
  const float *dummy  (i) : input dummies, one value for each frame
  const float *ddummy (i) : input ddummies, one value for each frame
  int i_1, int i_2 (i) : pixel indices 
                         (i_1 from 1 to dim_1, i_2 from 1 to dim_2)
  (AMode) int amode   (i) : 1: mean, 2: median
  (AMode) int fmode   (i) : 1: mean, 2: median
  int vmode           (i) : 0: calculate variance using n
                            1: calculate variance using n-1
  int dovar           (i) : 0: do not return variance
                            1: return variance
  float lfac          (i) : minimum sigma factor
  float ufac          (i) : maximum sigma factor
  float mins          (i) : minimum sigma

  int *pstatus     (o) : return status 0==OK, otherwise error

---------------------------------------------------------------------------*/
typedef struct {
  float val;  // average value
  float var;  // variance
  long  n;    // number of averaged values
  long  i;    // number of ignored values
} AveValue;

AveValue average( const long dim[], const float *frames, 
                  const float *dummy, const float *ddummy,
                  long i_1, long i_2, int amode, int fmode,
                  int vmode, int dovar, float lfac, float ufac,
                  float mins, float pq, int *pstatus )
{ AveValue ave;
  double *values=NULL;  // table of valid values (dim_3 elements) 
  double value;

  const float *src;
  long k, pitch;
  double *dest;
  long nvalues=0, ivalues=0 ;    // number of valid values 
  double lower, upper;
  double dlfac, dufac, dmins, dpq;

  int status=Failed; 
  
  dlfac = (double) lfac;
  dufac = (double) ufac;
  dmins = (double) mins;
  dpq   = (double) pq;

  ave.val = 0.0;
  ave.var = VarDummy;
  ave.n = 0;

  // 1) extract pixel values (i_1, i_2) from frames and copy them to values
  if (!( values = (double *) malloc( sizeof(double)*dim[3] ) )) {
    status = NotEnoughMemoryAvailable; 
    goto average_error;
  }

  src     = frames+(i_1-1)+(i_2-1)*dim[1];
  dest    = values;
  pitch   = dim[1]*dim[2];
  nvalues = 0;

  for (k=0;k<dim[3];k++) {
    value = (double) *src;

    // UPDATE( *dest, value, dummy[k], ddummy[k] );
    if ( NODUMMY(value, dummy[k], ddummy[k]) ) {
      *dest = value;
      dest++;
      nvalues++;
    }

    src+=pitch; // next frame
  }

  // 2) filter values
  if (fmode) {
    double sigma, val, var;
    double upq, loq, hdif;

    // 2a) calculate average and deviation from average
    switch (fmode ) {
      case Mean:      
        val = mean( values, nvalues );
        var = variance ( values, nvalues, val );
        sigma = sqrt(var);
        break;

      case Quantil:
        upq = dquantil ( values, nvalues, 1.0-dpq );
        loq = dquantil ( values, nvalues, dpq );
        hdif = fabs((upq-loq)*0.5); // half distance beween quantils
        val = (upq+loq)*0.5; // average of upper and lower quantil
        sigma = hdif;
        break;

      case Median:
      default:
        val = dmedian( values, nvalues);
        var = dmediance ( values, nvalues, val);
        sigma = sqrt(var);
    }

    if (sigma<dmins) sigma=dmins;

    // 2b) calculate lower and upper limits
    if (lfac<0.0)
      lower = minimum ( values, nvalues ); // accept all below
    else
      lower = val - sigma * dlfac;

    if (ufac<0.0)
      upper = maximum ( values, nvalues ); // accept all above
    else
      upper = val + sigma * dufac;

    // 2c) remove table values outside lower and upper limits
    ivalues = nvalues;
    nvalues = minmaxfilter ( values, nvalues, lower, upper );
    ivalues -= nvalues; // ignored values
  }

  // 3) calculate average and variance

  switch (amode) {
    double upq, loq, hdif;
    case Median: 
      ave.val = dmedian ( values, nvalues );
      if (dovar) 
        ave.var = dmediance( values, nvalues, value );
      break;
     case Quantil:
        upq = dquantil ( values, nvalues, 1.0-dpq );
        loq = dquantil ( values, nvalues, dpq );
        hdif = fabs(upq-loq)*0.5; // half distance between quantils
        ave.val = (upq+loq)*0.5; // average of upper and lower quantil
        if (dovar)
          ave.var = hdif*hdif;
        break;
    default: 
    case Mean: 
      ave.val = mean ( values, nvalues );
      if (dovar)
        ave.var = variance( values, nvalues, value );
  }
  ave.n   = nvalues;
  ave.i   = ivalues;

  // 4) calculate variance of the average
  
  if (dovar) {
    if (vmode==0) { // calculate with n values
       if (nvalues>0)
         ave.var /= nvalues;
     }  else { // calculate with n-1 values
       if (nvalues>1) 
         ave.var /= nvalues-1;
     }
  }

  FREE(values);

  status = Success;
  if (pstatus) *pstatus=status;
  return( ave ); 

average_error:

  FREE(values);

  if (pstatus) *pstatus=status;
  return( ave ); 
  
} // average 

/*---------------------------------------------------------------------------
NAME
  saxs_average --- Creates average and variance from a series of images.

PURPOSE
  Creates average and variance from a series of images.

saxs_average_init : initialization of ihb variables in first image
saxs_average      : called every loop
saxs_average_term : finalization (final calculation, releasing variables ...)
---------------------------------------------------------------------------*/
void saxs_average_init (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], 
                        int * pstatus)
{ MainData * frames = pcb->MainData;

  long  dim_1=0, dim_2=1, dim_3=1;
  int status=Failed;

  if (pcb->TestBit>1)
    printf("saxs_average_init\n");

  SetSaxsError( "saxs_average", "saxs_average_init", NULL );

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0l;

  // I0Dim is initialized with I1Dim
  dim_1 = ihb[0].Dim[1];
  dim_2 = ihb[0].Dim[2];

  /* avoid realloc, if possible allocate all frames indicated by Add.V */
  if ( (pcb->Add.I)&&(pcb->Add.V>0) )
    dim_3=pcb->Add.V;
  else dim_3=1;

  frames->data_size = dim_1*dim_2*dim_3*sizeof(float);
  if (pcb->TestBit>1)
    printf("alloc %zu bytes data\n",frames->data_size);
  if (!(frames->data= (float*) malloc( frames->data_size ))) {
    fprintf(ERROUT,
      "ERROR: Cannot allocate %zu bytes for data (%dx%dx%d floats)\n",
      frames->data_size, dim_1,dim_2,dim_3);
    goto saxs_average_init_error;
  }

  frames->dummy_size = dim_3*sizeof(float);
  if (pcb->TestBit>1)
    printf("alloc %zu bytes data\n",frames->dummy_size);
  if (!(frames->dummy= (float*) malloc( frames->dummy_size ))) {
    fprintf(ERROUT,
      "ERROR: Cannot allocate %zu bytes for dummies (%d floats)\n",
      frames->dummy_size, dim_3);
    goto saxs_average_init_error;
  }

  frames->ddummy_size = dim_3*sizeof(float);
  if (pcb->TestBit>1)
    printf("alloc %zu bytes data\n",frames->ddummy_size);
  if (!(frames->ddummy= (float*) malloc( frames->ddummy_size ))) {
    fprintf(ERROUT,
      "ERROR: Cannot allocate %zu bytes for ddummies (%d floats)\n",
      frames->ddummy_size, dim_3);
    goto saxs_average_init_error;
  }

  if (pstatus) *pstatus = Success;

  return;

saxs_average_init_error:

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0;

  if (pstatus) *pstatus = status;

  return;

} /* saxs_average_init */

void saxs_average (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{
  MainData * frames = pcb->MainData;

  int i,imax;

  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  float *I1Data, *E1Data;
  long  I1Dim_1, I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value, I1Sum, I1Weight;
  float E1Value, E1Sum, E1Weight;

  float *dest;
  long  nelements=0;
  size_t required_data_size;
  float *pdummy;
  size_t required_dummy_size;
  float *pddummy;
  size_t required_ddummy_size;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  long  dim_1=0, dim_2=1, dim_3=1;
  float dummy = 0.0, ddummy = DDSET(dummy);

  int status=Failed;

  // Increment frame counter 
  frames->counter+=1l;

  imax = pcb->ImgBlkLen;
  if (!ihb[0].ActualLoopCount) {
    printf("\n Calculating ihb[0,%ld] = Function(",ihb[0].ImNum);
    for(i=1;i<imax;i++) {
      printf("ihb[%d,%ld] ", i, ihb[i].ImNum); }
    printf(")\n\n");
  }

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
    fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); 
    goto saxs_average_error; 
  }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, &status );
  if (status!=Success)
    goto saxs_average_error;

  SetSaxsError( "saxs_average", "saxs_average", NULL );

  if ( ( fabs(Off_1[0]-Off_1[1]) > eps ) ||
       ( fabs(Off_2[0]-Off_2[1]) > eps ) ||
       ( fabs(Ps_1[0]-Ps_1[1]) > eps ) ||
       ( fabs(Ps_2[0]-Ps_2[1]) > eps ) ) {
    
    status=ReferenceNotSupported;
    fprintf(ERROUT,"ERROR: Input and output images do not have the same geometry. To avoid\n");
    fprintf(ERROUT,"       this error choose the reference system array (option -rsys array).\n");
    goto saxs_average_error;
  }

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = ihb[1].Dim[1];
  I1Dim_2  = ihb[1].Dim[2];

  dim_1 = I0Dim_1;
  dim_2 = I0Dim_2;
  dim_3 = frames->counter;

  // check memory size and realloc if necessary 
  nelements = dim_1*dim_2;
  required_data_size = nelements*dim_3*sizeof(float);
  if ( frames->data_size < required_data_size ) {
    float * new;
    if (pcb->TestBit>1)
      printf("realloc %zu bytes data\n",required_data_size);
    if (!(new=
        (float *) realloc(frames->data, required_data_size ))) {
      fprintf(ERROUT,
        "ERROR: Failed to resize volume to %zu bytes (%dx%d%d floats)\n",
        required_data_size, dim_1, dim_2, dim_3);
      goto saxs_average_error;
    }
    frames->data_size = required_data_size;
    frames->data = new;
  }
  dest = frames->data + (dim_3-1)*nelements;

  required_dummy_size = dim_3*sizeof(float);
  if ( frames->dummy_size < required_dummy_size ) {
    float * new;
    if (pcb->TestBit>1)
      printf("realloc %zu bytes for dummies\n",required_dummy_size);
    if (!(new=
        (float *) realloc(frames->dummy, required_dummy_size ))) {
      fprintf(ERROUT,
        "ERROR: Failed to resize dummies to %zu bytes (%d floats)\n",
        required_dummy_size, dim_3);
      goto saxs_average_error;
    }
    frames->dummy_size = required_dummy_size;
    frames->dummy = new;
  }
  pdummy = frames->dummy + (dim_3-1);

  required_ddummy_size = dim_3*sizeof(float);
  if ( frames->ddummy_size < required_ddummy_size ) {
    float * new;
    if (pcb->TestBit>1)
      printf("realloc %zu bytes for ddummies\n",required_ddummy_size);
    if (!(new=
        (float *) realloc(frames->ddummy, required_ddummy_size ))) {
      fprintf(ERROUT,
        "ERROR: Failed to resize ddummies to %zu bytes (%d floats)\n",
        required_ddummy_size, dim_3);
      goto saxs_average_error;
    }
    frames->ddummy_size = required_ddummy_size;
    frames->ddummy = new;
  }
  pddummy = frames->ddummy + (dim_3-1);

  /*
   * Keep input images in 3d frame buffer until end of loop
   */
  if ( ( dim_1 == I1Dim_1 ) && ( dim_2 == I1Dim_2  ) ) {
    // copy I1Data to new frame
    memmove(dest,I1Data,nelements*sizeof(float));
    *pdummy  = I1Dummy;
    *pddummy = I1DDummy;
  } else {
    status = Failed;
    fprintf(ERROUT,"ERROR: The dimensions %ldx%ld of %s:%d\n",
      I1Dim_1,I1Dim_2,pcb->ib[1].FileName,ihb[1].ImNum);
    fprintf(ERROUT,"       are different from the output dimensions %ldx%ld.\n",
      dim_1,dim_2);
    goto saxs_average_error;
  }

  if (pstatus) *pstatus = Success;

  return;

saxs_average_error:

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0l;

  if (pstatus) *pstatus = status;

  return;

} /* saxs_average */

void saxs_average_term (CmdBlk * pcb, long num[], ImgHeadBlk ihb[],     
                        int * pstatus)
{ MainData * frames = pcb->MainData;

  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float VarDDummy=DDSET(VarDummy);

  int  errval;
  int  stream;
  IO_line Framename;
  IO_line AMode;
  IO_line FMode;
  IO_long NM1;
  IO_float SIGM;
  IO_float SIGP;
  IO_float MINS;
  IO_float PQ;
  int  amode;
  int  fmode;
  float lfac=-1.0, ufac=-1.0;

  long  dim_1=0, dim_2=1, dim_3=1;

  long used=0, ignored=0;

  int i_1, i_2;
  int status=Failed;

  if (pcb->TestBit>1)
    printf("saxs_average_term\n");

  SetSaxsError( "saxs_average", "saxs_average_term", NULL );

  Framename = option_line ( SFramename, pcb, pcb->ib, num, 0, &status );
  if (status!=Success)
    goto saxs_average_term_error;
  AMode = option_line ( SAMode, pcb, pcb->ib, num, 0, &status );
  if (status!=Success)
    goto saxs_average_term_error;
  FMode = option_line ( SFMode, pcb, pcb->ib, num, 0, &status );
  if (status!=Success)
    goto saxs_average_term_error;
  NM1 = option_flag(SNM1,pcb,pstatus);
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  SIGM = option_float ( SSIGM, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  SIGP = option_float ( SSIGP, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  MINS = option_float ( SMINS, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;
  PQ = option_float ( SPQ, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success)
    goto saxs_average_term_error;

  amode = string2amode( AMode.V );
  fmode = string2amode( FMode.V );
  if (pcb->TestBit>1) {
    printf("AMode.V = %s; amode %d = %s\n",AMode.V,amode,amode2string(amode));
    printf("FMode.V = %s; fmode %d = %s\n",FMode.V,fmode,amode2string(fmode));
    printf("  NM.V = %d\n",NM1.V);
    printf("SIGM.V = %g\n",SIGM.V);
    printf("SIGP.V = %g\n",SIGP.V);
    printf("MINS.V = %g\n",MINS.V);
    printf("  PQ.V = %g\n",PQ.V);
  }

  printf("\n Averaging %d images (%s)\n",frames->counter,amode2string(amode));

  I0Data  = ihb[0].Data;
  E0Data = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

  dim_1  = (int) ihb[0].Dim[1];
  dim_2  = (int) ihb[0].Dim[2]; 
  dim_3  = frames->counter;

  if  (!(ImageIsNew( 0, ihb ))) {
    long dim[4];
    dim[0]=3;dim[1]=dim_1;dim[2]=dim_2;dim[3]=dim_3;
    if (Framename.I) {
      /* 
       * write 3d frame buffer
       */

      if (pcb->TestBit)
        printf("    Writing file \'%s\'\n", Framename.V);

      // open edf file
      stream=edf_open_data_file  ( Framename.V, "new", &errval, &status );
      if (status) {
        fprintf(ERROUT,"Error opening \'%s\'\n",Framename.V);
        goto saxs_average_term_error;
      }

      // write edf file
      edf_write_data ( stream, 1, 1, dim, frames->data, MFloat,
                       &errval, &status );
      if (status) {
        fprintf(ERROUT,"Error writing data to \'%s\'\n",Framename.V);
        goto saxs_average_term_error;
      }

      // close edf file
      edf_close_data_file ( stream, &errval, &status );
      if (status) {
        fprintf(ERROUT,"Error closing \'%s\'\n",Framename.V);
        goto saxs_average_term_error;
      }
    }

    /* 
     * Calculate averages, variances and 
     * copy to I0Data and E0Data
     */

    pI0Data = I0Data;
    pE0Data = E0Data;
    for (i_2=1;i_2<=I0Dim_2;i_2++) {
      AveValue ave;
      for (i_1=1;i_1<=I0Dim_1;i_1++) {

        if ( E0Data ) {
          // value and variance (NM1.V==0: n values, NM1.V!=0: n-1 values)
          ave = average( dim, frames->data, frames->dummy, frames->ddummy,
                         i_1, i_2, amode, fmode, NM1.V, 1,
                         SIGM.V, SIGP.V, MINS.V, PQ.V, &status );
          if (status) {
            fprintf(ERROUT,"Error averaging frames at pixel %dx%d\n",i_1,i_2);
            goto saxs_average_term_error;
          }
          if (ave.n>0) {
            *pI0Data = ave.val;
            *pE0Data = ave.var;
          }
          used+=ave.n;
          ignored+=ave.i;
        } else {
          // value
          ave = average( dim, frames->data, frames->dummy, frames->ddummy,
                         i_1, i_2, amode, fmode, NM1.V, 0,
                         SIGM.V, SIGP.V, MINS.V, PQ.V, &status );
          if (status) {
            fprintf(ERROUT,"Error averaging frames at pixel %dx%d\n",i_1,i_2);
            goto saxs_average_term_error;
          }
          if (ave.n>0) {
            *pI0Data = ave.val;
          }
          used+=ave.n;
          ignored+=ave.i;
        }
  
        pI0Data++;
        pE0Data++;
  
      } /* for i_1 ... */

    } /* for i_2 ... */

  } /* if !ImageIsNew */

  if (fmode) {
    long total = used+ignored;
    double percent=0.0;
    if (total>0)
      percent = 100.0 * (((double) ignored)/((double) total));
    else percent = 100.0;
    printf("   %ld of %ld pixel values ignored (%3.1lg%c) \n",
      ignored,total,percent,percentchar);
  }
  printf("\n");

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0l;

  if (pstatus) *pstatus = Success;

  return;

saxs_average_term_error:

  FREE(frames->data);
  frames->data_size=0;
  FREE(frames->dummy);
  frames->dummy_size=0;
  FREE(frames->ddummy);
  frames->ddummy_size=0;
  frames->counter=0l;

  if (pstatus) *pstatus = status;

  return;

} /* saxs_average_term */

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/
void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  IO_line AMode;
  IO_line FMode;
  IO_long NM1;
  IO_flexp *pSIGM;
  IO_flexp *pSIGP;
  IO_flexp *pMINS;
  IO_flexp *pPQ;

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_flexp(pcb,"Multiplication factor",&ib[0].Factor,"1.0",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_flexp(pcb,"Addition constant",&ib[0].Const,"0.0",pstatus);
  if (*pstatus!=Success) return;

  AMode = option_line(SAMode,pcb,NULL,NULL,0,pstatus);
  if (*pstatus!=Success) return;

  FMode = option_line(SFMode,pcb,NULL,NULL,0,pstatus);
  if (*pstatus!=Success) return;

  NM1 = option_flag(SNM1,pcb,pstatus);
  if (*pstatus!=Success) return;

  pSIGM = (IO_flexp*) option_parameter_search( SSIGM, pcb, pstatus );
  if (*pstatus!=Success) return;

  pSIGP = (IO_flexp*) option_parameter_search( SSIGP, pcb, pstatus );
  if (*pstatus!=Success) return;

  pMINS = (IO_flexp*) option_parameter_search( SMINS, pcb, pstatus );
  if (*pstatus!=Success) return;

  pPQ = (IO_flexp*) option_parameter_search( SPQ, pcb, pstatus );
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file             : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file             : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image          : %ld\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image           : %ld\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment            : %ld\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy         : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1   : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2   : %s\n",ib[0].Dim[2].V);
  if (ib[0].Factor.I)  printf("factor               : %s\n",ib[0].Factor.V);
  if (ib[0].Const.I)   printf("constant             : %s\n",ib[0].Const.V);
                       printf("average mode         : %s\n", amode2string(string2amode(AMode.V)));
                       printf("filter mode          : %s\n", amode2string(string2amode(FMode.V)));
                       printf("create variances     : %s\n", pcb->CreateVariance.V?"yes":"no");
                       printf("variance calculation : %s\n", NM1.V?"n-1":"n");
                       printf("sigma-               : %s\n", pSIGM->V);
                       printf("sigma+               : %s\n", pSIGP->V);
                       printf("minimum sigma        : %s\n", pMINS->V);
                       printf("quantil value p      : %s\n", pPQ->V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_average
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, new_main_data(), TRUE, &cb, ib, BlockNum );

  option_define ( IO_tpline,  SFramename, "framebuffer.edf", &cb );
  option_define ( IO_tpline,  SAMode, "mean", &cb );
  option_define ( IO_tpline,  SFMode, "none", &cb );
  option_define ( IO_tpflag,  SNM1, "TRUE", &cb ); // get variance from n-measurements
  option_define ( IO_tpflexp, SSIGM, "5", &cb ); // lower sigma factor
  option_define ( IO_tpflexp, SSIGP, "5", &cb ); // upper sigma factor
  option_define ( IO_tpflexp, SMINS, "0.5", &cb ); // minimum sigma per pixel
  option_define ( IO_tpflexp, SPQ, "0.9", &cb ); // quantil value [1-p..p]
 
 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

 /* Default: Combine all images of the sequence (Add.V=0)*/
  if (status==Success) {
    if (!cb.Add.I) {
      cb.Add.V = 0; cb.Add.I = TRUE;
    }
  }

 /* Default: Create variance array (Var.V=1)*/
  if (status==Success) {
    if (!cb.CreateVariance.I) {
      cb.CreateVariance.V = 1; cb.CreateVariance.I = TRUE;
    }
  }

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_average, saxs_average_init, saxs_average_term, TRUE, &status );

  cb.MainData=free_main_data(cb.MainData);

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */
