/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_arc
  Transformation from cartesian coordinates to polar coordinates

2 PURPOSE
  -arc: Transformation from cartesian coordinates to polar coordinates
        with pixel interpolation and azimuthal averaging.
        axis_1: radius, axis_2: azimuth_angle
  +arc: Transformation from cartesian coordinates to arcs, i.e.
        axis_1: radius, axis_2: arc=azimuth_angle*radius

  Calculation:       I0 = I1
  Error propagation: V0 = V1

  The angle coordinate (x_2) uses always the real reference system (Offset,
  PixSiz). WaveLength and SampleDistance are not used.

  The AxisType of axis_2 is set to Angle.

  The output dimensions are automatically calculated. The option
  -odim = <number of azimuths/arc> sets the number of azimuthal sections
  (arc sections), -odim = 1 regroups the input image into a single line.

  If the input and output projection types are different and if the reference
  system is IO_Saxs the output image is calculated taking into account the
  detector rotations.

  Arguments:
  saxs_arc [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> <odum>
                <i1cen_1> <i1cen_2> <odim_1> <a0> <da> <odim_2>"

  Defaults:
  <input file name>   : input.edf
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <i1cen_1> (image)   : <image center coordinate along dimension 1>
  <r0>      (image)   : <minimum radius>
  <i1cen_2> (image)   : <image center coordinate along dimension 2>
  <odim_1>            : <number of radial intervals> (output dimension 1)
  <odim_2>            : <number of angular intervals> (output dimension 2)
  <a0>      (radian)  : <minimum angle>
  <da>      (radian)  : <angular interval>

2 HISTORY
  2009-10-02  PB  V4.26 Extracted from saxs_angle,
                        SaxsArc -> arc
  2010-03-05  PB  V4.27 Center[2] corrected, description updated
  2010-03-18  PB  V4.28 Option waxs added, +waxs is default, 
                        waxs takes the detector rotations into 
                        account. +xy creates combined displacement 
                        files for spd to correct a detector tilt 
                        and to regroup the data at the same time.
                        Displacement files are created for 
                        +arc and -arc (default). To take 
                        the tilts into account all parameters for
                        the reference system SAXS must be 
                        supplied.
  2010-05-31  PB  V4.29 update for waxs.c V1.10
  2010-09-29  PB  V4.30 masking of the input image before regrouping
                        The file given with -i2nam is used for masking.
                        To include binned ROIs the reference system region 
                        is used.
                        Restrictions: Because the input data array is 
                        modified it works only correctly, if the mask 
                        image is not changed while the input image is
                        not. For the mask file the interpolation function 
                        Ipol2ldw is used because Isum2ldw creates some
                        artefacts at the borders (to be checked).
  2010-11-08  PB  V4.31 Default of i2nam is set to [MaskFileName]
                        When +mask is set i2nam is available, either
                        with a filename or with [MaskFileName].
  2010-11-18  PB  V4.32 Default of i2nam is set to [FPATH][MaskFileName].
                        The default path of the mask file is the path
                        of the input file.
  2010-11-22  PB  V4.33 Ipol2ldw is shifted and replaced with Isum2ldw.
                        Problem observed on 2010-09-29 needs to be checked. 
  2011-06-07  PB  V4.34 strncat->strncpy
  2011-06-30  PB  V4.35 update for waxs V1.11, concat->strlib_concat
  2012-01-31  PB  V4.36 ImageLoop parameters updated

---*/
# define Version  "saxs_arc V4.36 2012-01-31, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <string.h>

# include "SaxsPrograms.h"
# include "arc.h"
# include "waxs.h"
# include "strlib.h"

# define FREE( a) { if ( a) { free( a); a = NULL; } }

/* Defaults */
# define Default_da deg2rad  // 1_deg

/* Special options */
# define SR0       "r0"    /* float */
# define SA0       "a0"    /* float */
# define SAsym     "asym"  /* float */

# define SXyname   "xyna"  /* line */  // xy-file name
# define SXy       "xy"    /* flag */  // xy-file flag

# define SS2       "s2"    /* flag */
# define SCsym     "csym"  /* flag */
# define SVsum     "vsum"  /* flag */
# define SDoArc    "arc"   /* flag */  // radius/arc coordinate

# define SWaxs     "waxs"  /* flag */ // take detector rotations into acount
# define SMask     "mask"  /* flag */ // mask input image before regrouping
# define SDbg      "dbg"   /* flag */ // write mask file for debugging 

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> <odum> \n\
                <i1cen1> <i1cen2> <r0> <dim1> <a0> <da> <dim2> \n\n\
                For masking add the option \"-i2nam <maskfile\">"

# define BlockNum 3       /* max 2 input sequences + 1 output sequence */

const float rad2deg = 180.0/SAXS_PI;
const float deg2rad = SAXS_PI/180.0;

/*---------------------------------------------------------------------------
 NAME

    mask_region --- masks data of image header block i_inp with i_mask

 SYNOPSIS

    void mask_region( CmdBlk * pcb, ImgHeadBlk ihb[], int i_inp, int i_mask,
                      float *buffer, size_t buffersize, int * pstatus );


 DESCRIPTION

    masks data of image header block i_inp with i_mask and writes result
    to buffer. The pointer to the result is returned. The input and 
    output buffers can be identical.


 RETURN VALUE

    none 

---------------------------------------------------------------------------*/
float *mask_region( CmdBlk * pcb, ImgHeadBlk ihb[], int i_inp, int i_mask,
                    float *buffer, size_t buffersize, int * pstatus )
{ float *out=NULL, *outend=NULL;

  float *Data[3], *pData[3];

  float Offset_1[3], Offset_2[3];
  float BinSiz_1[3], BinSiz_2[3];

  float Off_1[3], Ps_1[3];
  float Off_2[3], Ps_2[3];

  int   Dim_1[3], Dim_2[3];

  float Dummy[3];
  float DDummy[3];

  float W1_1, W1_2, W3_1, W3_2;
  float f1_1, f1_2, f3_1, f3_2;
  float Df_1, Df_2;
  float f_1, f_2;

  float Sum, Weight;

  int i_1, i_2;

  int status=Failed;

  IO_long Dbg;

  int ErrorValue;
  int stream;

  if (pcb->TestBit) {
    printf("\nmask_region START\n\n");
  }

  Dbg  = option_flag( SDbg, pcb, pstatus);
  if (*pstatus!=Success) goto mask_region_error;

  if (!buffer) goto mask_region_error;

  out    = &(buffer[0]);
  outend = out + buffersize;

  // uninitialzed elements have already been set to defaults by init_header
  Data[1]     = ihb[i_inp].Data;
  Dim_1[1]    = ihb[i_inp].Dim[1];
  Dim_2[1]    = ihb[i_inp].Dim[2];
  Offset_1[1] = ihb[i_inp].Offset[1].V; 
  Offset_2[1] = ihb[i_inp].Offset[2].V;
  BinSiz_1[1] = ihb[i_inp].BinSiz[1].V;
  BinSiz_2[1] = ihb[i_inp].BinSiz[2].V;

  Data[2]     = ihb[i_mask].Data;
  Dim_1[2]    = ihb[i_mask].Dim[1];
  Dim_2[2]    = ihb[i_mask].Dim[2];
  Offset_1[2] = ihb[i_mask].Offset[1].V;
  Offset_2[2] = ihb[i_mask].Offset[2].V;
  BinSiz_1[2] = ihb[i_mask].BinSiz[1].V;
  BinSiz_2[2] = ihb[i_mask].BinSiz[2].V;

  // output image has the same shape as input image 
  Data[0]  = out;
  Dim_1[0] = Dim_1[1];
  Dim_2[0] = Dim_2[1];
  Offset_1[0] = Offset_1[1];
  Offset_2[0] = Offset_2[1];
  BinSiz_1[0] = BinSiz_1[1];
  BinSiz_2[0] = BinSiz_2[1];

  // use region reference system
  REALREF(Off_1[1],Ps_1[1],Offset_1[1],BinSiz_1[1]);
  REALREF(Off_2[1],Ps_2[1],Offset_2[1],BinSiz_2[1]);

  REALREF(Off_1[2],Ps_1[2],Offset_1[2],BinSiz_1[2]);
  REALREF(Off_2[2],Ps_2[2],Offset_2[2],BinSiz_2[2]);

  Off_1[0] = Off_1[1];
  Off_2[0] = Off_2[1];
  Ps_1[0]  = Ps_1[1];
  Ps_2[0]  = Ps_2[1];

  if (pcb->TestBit) {
    printf("  Off_1[%d] = % f, Ps_1[%d] = % f\n", 1,Off_1[1],1,Ps_1[1]);
    printf("  Off_2[%d] = % f, Ps_2[%d] = % f\n", 1,Off_2[1],1,Ps_2[1]);
    printf("  Off_1[%d] = % f, Ps_1[%d] = % f\n", 2,Off_1[2],2,Ps_1[2]);
    printf("  Off_2[%d] = % f, Ps_2[%d] = % f\n", 2,Off_2[2],2,Ps_2[2]);
  }

  // calculate world coordinates of lower and upper edges 
  W1_1 = WORLD(LOWERBORDER,Off_1[0],Ps_1[0]);
  W1_2 = WORLD(LOWERBORDER,Off_2[0],Ps_2[0]);

  W3_1 = WORLD(Dim_1[0]+LOWERBORDER,Off_1[0],Ps_1[0]);
  W3_2 = WORLD(Dim_2[0]+LOWERBORDER,Off_2[0],Ps_2[0]);

  if (pcb->TestBit) {
    printf("  W1_1 = % f, W1_2 = % f\n", W1_1, W1_2);
    printf("  W3_1 = % f, W3_2 = % f\n", W3_1, W3_2);
  }

  // calculate corresponding indices (f1 and f3) of mask image
  f1_1 = INDEX(W1_1, Off_1[2], Ps_1[2]);
  f1_2 = INDEX(W1_2, Off_2[2], Ps_2[2]);

  f3_1 = INDEX(W3_1, Off_1[2], Ps_1[2]);
  f3_2 = INDEX(W3_2, Off_2[2], Ps_2[2]);

  if (pcb->TestBit) {
    printf("  f1_1 = % f, f1_2 = % f\n", f1_1, f1_2);
    printf("  f3_1 = % f, f3_2 = % f\n", f3_1, f3_2);
  }

  // calculate increments of mask image
  Df_1 = (Dim_1[0]>0)?(f3_1-f1_1)/(float) Dim_1[0]:0.0;
  Df_2 = (Dim_2[0]>0)?(f3_2-f1_2)/(float) Dim_2[0]:0.0;

  if (pcb->TestBit) {
    printf("  Df_1 = % f, Df_2 = % f\n", Df_1, Df_2);
  }

  Dummy[1] = ihb[i_inp].Dummy.V;  DDummy[1] = ihb[i_inp].DDummy.V;
  Dummy[2] = ihb[i_mask].Dummy.V; DDummy[2] = ihb[i_mask].DDummy.V;
  Dummy[0] = Dummy[1];            DDummy[0] = DDummy[1];

  f_2=f1_2;
  for (i_2=0;i_2<Dim_2[0];i_2++) {
    pData[0] = ABSPTR(Data[0],Dim_1[0],Dim_2[0],0,i_2);
    pData[1] = ABSPTR(Data[1],Dim_1[1],Dim_2[1],0,i_2);
    f_1=f1_1;
    for (i_1=0;i_1<Dim_1[0];i_1++) {

      if ( DUMMY( *(pData[1]), Dummy[1], DDummy[1]) ) {
        *(pData[0]) = Dummy[0];
      } else if ( Isum2ldw (Data[2],Dim_1[2],Dim_2[2],Dummy[2],DDummy[2],
              f_1, f_2, f_1+Df_1, f_2+Df_2,
              &Sum, &Weight) ) {
        *(pData[0]) = *(pData[1]);
      } else *(pData[0]) = Dummy[0];


      (pData[0])++;
      (pData[1])++;
      f_1 += Df_1;

      if (pData[0]>=outend) break;
    }
    f_2 += Df_2;
    if (pData[0]>=outend) break;
  }

  if (Dbg.V) {
    printf("\nDEBUG: Writing masked input file to \"mask_region.edf\"\n\n");
    /* open debug-file */
    stream = edf_search_stream( "mask_region.edf",  "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error;
    if (stream>=0) { *pstatus = FileAlreadyOpened; goto mask_region_error; }
    stream = edf_open_data_file ( "mask_region.edf", "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error; 

    edf_write_header_float(stream,ihb[1].ImNum,1l,KOffset_1,Offset_1[0],
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error;

    edf_write_header_float(stream,ihb[1].ImNum,1l,KOffset_2,Offset_2[0],
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error;

    edf_write_header_float(stream,ihb[1].ImNum,1l,KBinSiz_1,BinSiz_1[0],
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error;

    edf_write_header_float(stream,ihb[1].ImNum,1l,KBinSiz_2,BinSiz_2[0],
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error;

    edf_write_header_float(stream,ihb[1].ImNum,1l,KDummy,Dummy[0],
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error;

    edf_write_header_float(stream,ihb[1].ImNum,1l,KDDummy,DDummy[0],
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error;

    /* write masked array */
    edf_write_data( stream,ihb[1].ImNum,1l,ihb[1].Dim,Data[0],MFloat,
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error;
    /* close debug-file */
    edf_close_data_file ( stream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) goto mask_region_error;
  }

  status = Success;

  if (pstatus) *pstatus=status;

  if (pcb->TestBit) {
    printf("\nmask_region END\n\n");
  }

  return( Data[0] );

mask_region_error:

  if (pstatus) *pstatus=status;

  if (pcb->TestBit) {
    printf("\nmask_region END (status=%d)\n\n",status);
  }

  return( NULL );

} /* mask_region */

/*---------------------------------------------------------------------------
SetImageDefaults

 Set the defaults of the output image as required for saxs_arc_init
---------------------------------------------------------------------------*/
void SetImageDefaults( CmdBlk * pcb, ImgHeadBlk ihb[], int blkno,
                        IO_float R0, IO_float A0, IO_long DoArc, 
                        int * pstatus )
{ ImgHeadBlk *pihb;
  float off[3], cen[3], pix[3], dis=1, wvl=WaveLength0;
  long  dim[3];
  int   proin, proout;
  int   rsys;
  float minrad, maxrad, minradN, maxradN, minang, maxang, minarc, maxarc;
  float r0_real;
  ImgBlk *ib;

  float rot[4];
  WaxsCoord W1min, W1max, A1min, A1max;

  int transform;

  *pstatus = Success;

  pihb = &ihb[blkno];
  ib   = pcb->ib;

  if (pcb->TestBit) printf("SetImageDefaults START\n");

  /* Check and get input parameters */
  rsys=pcb->RSys.V;
  dim[1]=pihb->Dim[1]; dim[2]=pihb->Dim[2];
  off[1]=pihb->Offset[1].V; off[2]=pihb->Offset[2].V;
  if (!( (pihb->Center[1].I) && (pihb->Center[2].I) ) ) {
    *pstatus = KeywordMissing; 
    fprintf(ERROUT,"Missing center definitions\n");
    goto SetImageDefaultsError;
  } else {
    cen[1] = pihb->Center[1].V; cen[2] = pihb->Center[2].V;
  }
  if (!( (pihb->PixSiz[1].I) && (pihb->PixSiz[2].I) ) ) {
    *pstatus = KeywordMissing;
    fprintf(ERROUT,"Missing pixel size\n");
    goto SetImageDefaultsError;
  } else {
    pix[1] = pihb->PixSiz[1].V; pix[2] = pihb->PixSiz[2].V;
  }
  switch (rsys) {
    case IO_Saxs:
      if (!( (pihb->SampleDistance.I) && (pihb->WaveLength.I) ) ) {
        *pstatus = KeywordMissing;
        fprintf(ERROUT,"Missing SampleDistance and/or WaveLenght\n");
        goto SetImageDefaultsError;
      } else {
        dis = pihb->SampleDistance.V; wvl = pihb->WaveLength.V;
      }
      break;
    case IO_Normal:
      break;
    default:
      fprintf(ERROUT,"ERROR: Reference system is %s, but %s or %s is required\n",
        reftostr(rsys), reftostr(IO_Normal), reftostr(IO_Saxs));
      SetSaxsErrorMessage(NULL);
      *pstatus = ReferenceNotSupported;
      goto SetImageDefaultsError;
  }

  if (pcb->TestBit) printf("Reference system is %s\n",reftostr(rsys));

  rot[1] = pihb->DetRot[1].V;
  rot[2] = pihb->DetRot[2].V;
  rot[3] = pihb->DetRot[3].V;

  if (rsys == IO_Saxs) {
    /* projection default */
    if (ihb[0].Projection.I>=0) {
      ihb[0].Projection.V = IO_ProWaxs;
      ihb[0].Projection.I = TRUE;
    }
  } // if (rsys == IO_Saxs)

  proin = pihb->Projection.V;
  proout = ihb[0].Projection.V;

  /* Get world coordinate range of input image */
  transform=ang_range ( rsys, proin, proout,
                        dim[1], dim[2],
                        off[1], pix[1], cen[1],
                        off[2], pix[2], cen[2],
                        dis, wvl,
                        rot[1], rot[2], rot[3],
                        &W1min, &W1max, pstatus);
  if (*pstatus!=Success) goto SetImageDefaultsError;

  if (pcb->TestBit) {
    printf("W1min_1=%g, W1max_1=%g\n", W1min.s_1,W1max.s_1);
    printf("W1min_2=%g, W1max_2=%g\n", W1min.s_2,W1max.s_2);
    printf("transform=%d\n", transform);
  }

  /* Set output defaults */

  /* Set offset of output image to zero, if it is not explicitely set */
  if (ihb[0].Offset[1].I>=0) ihb[0].Offset[1].V = 0.0;
  if (ihb[0].Offset[2].I>=0) ihb[0].Offset[2].V = 0.0;

  if (pcb->TestBit) printf("Offset[1]=%g, Offset[2]=%g\n",
    ihb[0].Offset[1].V, ihb[0].Offset[2].V);

  /* Set BinSiz of output image to 1, if it is not explicitely set,
     because it is no longer well defined */
  if (ihb[0].BinSiz[1].I>=0) ihb[0].BinSiz[1].V = 1.0;
  if (ihb[0].BinSiz[2].I>=0) ihb[0].BinSiz[2].V = 1.0;

  if (pcb->TestBit) printf("BinSiz[1]=%g, BinSiz[2]=%g\n",
    ihb[0].BinSiz[1].V, ihb[0].BinSiz[2].V);

  /* Set PixSiz of output image */
  if (ihb[0].PixSiz[1].I>=0) {
    ihb[0].PixSiz[1].V = MAX2(pihb->PixSiz[1].V,pihb->PixSiz[2].V);
    ihb[0].PixSiz[1].I = TRUE;
  }
  if (ihb[0].PixSiz[2].I>=0) {
    if (DoArc.V) ihb[0].PixSiz[2].V = ihb[0].PixSiz[1].V;
    else ihb[0].PixSiz[2].V = Default_da;
    ihb[0].PixSiz[2].I = TRUE;
  }

  if (pcb->TestBit) printf("PixSiz[1]=%g, PixSiz[2]=%g\n",
    ihb[0].PixSiz[1].V, ihb[0].PixSiz[2].V);

  /* Radial range */
  ang_limits( W1min, W1max, &A1min, &A1max, pstatus);
  if (*pstatus) goto SetImageDefaultsError;

  if (pcb->TestBit) {
    printf("A1min.s_1=%lg    , A1max.s_1=%lg\n",A1min.s_1,A1max.s_1);
    printf("A1min.s_2=%lg_deg, A1max.s_2=%lg_deg\n",A1min.s_2*rad2deg,A1max.s_2*rad2deg);
  }

  if (R0.I) minrad=R0.V; 
    else minrad = A1min.s_1;
  if (ib[0].Dim[1].I) maxrad=minrad+ihb[0].Dim[1]*ihb[0].PixSiz[1].V;
    else maxrad = A1max.s_1;

  if (pcb->TestBit) printf("minrad=%g, maxrad=%g\n",minrad,maxrad);

  /* Radial range in Normal coordinates */
  switch (rsys) {
    case IO_Saxs:
      minradN = S2N(minrad,ihb[0].SampleDistance.V,ihb[0].WaveLength.V);
      maxradN = S2N(maxrad,ihb[0].SampleDistance.V,ihb[0].WaveLength.V);
      break;
    case IO_Normal:
      minradN = minrad;
      maxradN = maxrad;
      break;
    default:
      *pstatus = Failed;
      goto SetImageDefaultsError;
  }

  /* Radial Dimension */
  if (!(ib[0].Dim[1].I)) { // calculate Dim[1]
    if ( (minrad!=-999) && (minrad<maxrad) )
    switch (rsys) {
      case IO_Saxs:   
        ihb[0].Dim[1] = (long int) (
          S2PSIZE(maxrad,ihb[0].SampleDistance.V,ihb[0].WaveLength.V)/ihb[0].PixSiz[1].V -
          S2PSIZE(minrad,ihb[0].SampleDistance.V,ihb[0].WaveLength.V)/ihb[0].PixSiz[1].V);
          break;
      case IO_Normal: 
        ihb[0].Dim[1] = (long int) (
          N2PSIZE(maxrad)/ihb[0].PixSiz[1].V -
          N2PSIZE(minrad)/ihb[0].PixSiz[1].V);
        break;
      default:
        *pstatus = Failed;
        goto SetImageDefaultsError;
    }
  }
  if (pcb->TestBit) {
    printf("minrad=%g, maxrad=%g, minradN=%g, maxradN=%g\n",
      minrad,maxrad,minradN,maxradN);
    printf("dim_1=%ld\n",ihb[0].Dim[1]);
  }

  /* Angular Range */
  if (A0.I) { // start at A0, use angular stepsize, keep it constant
    minang=A0.V; 
    if (ib[0].Dim[2].I) maxang=A0.V+ihb[0].Dim[2]*ihb[0].PixSiz[2].V;
    else maxang=A0.V+SAXS_PI*2.0;
    minarc=minang*minradN;
    maxarc=maxang*maxradN;
    if (!ib[0].Dim[2].I) { // adjust Dim[2]
      if (DoArc.V) ihb[0].Dim[2]=(long int) ((maxarc-minarc)/ihb[0].PixSiz[2].V);
        else ihb[0].Dim[2]=(long int) ((maxang-minang)/ihb[0].PixSiz[2].V);
    }
  } else { // adjust PixSiz[2]
    minang = A1min.s_2;
    maxang = A1max.s_2;
    minarc=minang*minradN;
    maxarc=maxang*maxradN;
    if (ib[0].Dim[2].I) { // calculate new PixSiz
      if (DoArc.V) {
        if (ihb[0].PixSiz[2].I>=0) ihb[0].PixSiz[2].V = (maxarc-minarc)/ihb[0].Dim[2];
      } else if (ihb[0].PixSiz[2].I>=0) ihb[0].PixSiz[2].V = (maxang-minang)/ihb[0].Dim[2];
    } else { // calculate Dim[2]
      if (DoArc.V) {
        ihb[0].Dim[2]=(long int) ((maxarc-minarc)/ihb[0].PixSiz[2].V);
      } else ihb[0].Dim[2]=(long int) ((maxang-minang)/ihb[0].PixSiz[2].V);
    }
  }

  if (pcb->TestBit) {
    printf("minang=%g_deg, maxang=%g_deg, minarc=%g_deg, maxarc=%g_deg\n",
      minang*rad2deg,maxang*rad2deg,minarc*rad2deg,maxarc*rad2deg);
    printf("dim_2=%ld\n",ihb[0].Dim[2]);
  }

  /* Set Center of output image */
  if (ihb[0].Center[1].I>=0) {
    /* Get new Center, use Real coordinates to ignore an existing Center */
    switch (rsys) {
      case IO_Saxs:
        r0_real=S2PSIZE(minrad,ihb[0].SampleDistance.V,ihb[0].WaveLength.V);
        break;
      case IO_Normal:
        r0_real=minrad;
        break;
      default:
        *pstatus = Failed;
        goto SetImageDefaultsError;
    }
    ihb[0].Center[1].V = -R2CENTER(r0_real,ihb[0].PixSiz[1].V);
    ihb[0].Center[1].I = TRUE;
  }
  if (ihb[0].Center[2].I>=0) {
    /* Get new Center, use Real coordinates to ignore an existing Center 
       The angular/arc reference system is always Normal */
    if (DoArc.V) {
      ihb[0].Center[2].V = -R2CENTER(N2PSIZE(minarc),ihb[0].PixSiz[2].V);
    } else { ihb[0].Center[2].V = -R2CENTER(N2PSIZE(minang),ihb[0].PixSiz[2].V);
    }
    ihb[0].Center[2].I = TRUE;
  }

  if (pcb->TestBit) printf("Center[1]=%g, Center[2]=%g\n",\
      ihb[0].Center[1].V,ihb[0].Center[2].V);

  /* Set AxisType of output image */
  if (ihb[0].AxisType[2].I>=0) {
    if (!DoArc.V) {
      ihb[0].AxisType[2].V = IO_AxisTypeAngle;
      ihb[0].AxisType[2].I = TRUE;
    }
  }

  if (pcb->TestBit) {
    printf("dim_1=%ld, dim_2=%ld\n",ihb[0].Dim[1],ihb[0].Dim[2]);
  }

SetImageDefaultsError:

  if (pcb->TestBit) {
    printf("SetImageDefaults END (%s)\n",(*pstatus)?"Failed":"Success");
  }

  return;

} /* SetImageDefaults */

/*---------------------------------------------------------------------------
 saxs_arc_init

 Init output block (blkno=0), allocate image memory and set header parameters 
 for polar coordinates (x_1 = radius, x_2 = angle) or arc coordinates
 (x_1 = radius, x_2 = arc).

 All geometrical parameters of the output image that have not been set 
 explicitely by the image block, i.e. with .I flags >= 0, are changed 
 according to the following list.

 The coordinate range of the input image ihb[1] is determined with
 ang_limits (minimum radius, maximum radius, minimum angle, maximum
 angle).

 In case that the center of the input image lies inside the image, the 
 maximum angle is restricted to a0+360_deg.

 Coordinate 1 of the output image supports the following reference systems:
 Center, Normal, Saxs. In case of -waxs the radial projection is Saxs, in
 case of +waxs Waxs. 
 Coordinate 2 of the output image supports only the reference system Normal.

 In case of +arc the angle is calculated from the normal coordinates of 
 the axes 1 and 2: 
 angle=Normal(coordinate_2)/Normal(coordinate_1).

 The origin of the output (decomposed) image is Center[1] for the radius
 and Center[2] for the angle or arc.

 Offset[1]: set to 0
 Offset[2]: set to 0
 BSize[1]: set to 1 
 BSize[2]: set to 1 

 r0: set to minimum radius
 a0: set to minimum angle
 PixSiz[1]: minimum of horizontal and vertical pixel size of input image
 Dim[1]: set to cover r0 to maximum radius with PixSiz[1]
 PixSiz[2]: in case of -arc set to 1_deg, in case of +arc set to PixSiz[1].
 Dim[2]: in case of -arc set to cover a0 to maximum angle with PixSiz[2],
         If Dim[2] is explicitly set PixSiz[2] is recalculated to cover 
         a0 to maximum angle with Dim[2] sectors. 
         in case of +arc set to cover minimum arc=a0*minrad to maximum
         arc=maxang*maxrad with PixSiz[2].
         If Dim[2] is explicitly set PixSiz[2] is recalculated to cover 
         a0 to maximum angle with Dim[2] sectors. 
 Center[1]: -r0 (converted from rsys to image coordinates)
 Center[2]: -a0 (converted from normal to image coordinates)

 return value            :	void
 CmdBlk * pcb	         :   
 ImgHeadBlk ihb[0] (i/o) :
 ImgHeadBlk ihb[1] (i)   :
 int * pstatus     (o)   : return status

---------------------------------------------------------------------------*/
void saxs_arc_init( CmdBlk * pcb, long num [], ImgHeadBlk ihb[], 
                    int * pstatus )
{ IO_float R0, A0;
  IO_long  S2, DoArc;

  *pstatus = Success;

  if (pcb->TestBit) printf("saxs_arc_init START\n");

  // Get options
  R0 = option_float ( SR0, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) goto saxs_arc_init_end;
  A0 = option_float ( SA0, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) goto saxs_arc_init_end;

  S2 = option_flag ( SS2, pcb, pstatus );
  if (*pstatus!=Success) goto saxs_arc_init_end;
  DoArc = option_flag ( SDoArc, pcb, pstatus );
  if (*pstatus!=Success) goto saxs_arc_init_end;

  /* calculate defaults for ihb[0] from ihb[1] */
  SetImageDefaults( pcb, ihb, 1, R0, A0, DoArc, pstatus );

  if (*pstatus!=Success) goto saxs_arc_init_end;

  if (pcb->TestBit) {
    printf("%s: \n", DoArc.V?"arc":"angle");
    PrintImageHeaderBlock(&ihb[0]);
  }

saxs_arc_init_end:

  if (pcb->TestBit)
     printf("saxs_arc_init END (%s)\n",(*pstatus)?"Failed":"Success");

  return;

} /* saxs_arc_init */

/*---------------------------------------------------------------------------
1 saxs_arc

2 PURPOSE
  Transformation from cartesian coordinates to polar coordinates
---------------------------------------------------------------------------*/
void saxs_arc (CmdBlk * pcb, long num [], ImgHeadBlk ihb[], int * pstatus)
{ const float radius_eps = 1e-30;

  const char * RoutineName = "saxs_arc";

  int i,imax;

  int   rsys;

  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float I0Dummy, I0DDummy;

  float I0Offset_1, I0Offset_2;
  float I0PixSiz_1, I0PixSiz_2;
  float I0Cen_1, I0Cen_2;
  float I0Wvl, I0Dis;
  int   I0Pro;
  int   I0Axis_1, I0Axis_2;

  int   I1Ave;
  float *I1Data, *E1Data;
  int   I1Dim_1, I1Dim_2;
  float *pI1Data;
  float I1Dummy, I1DDummy;

  float I1Offset_1, I1Offset_2;
  float I1PixSiz_1, I1PixSiz_2;
  float I1Cen_1, I1Cen_2;
  float I1Wvl, I1Dis;
  float I1DetRot1,  I1DetRot2,  I1DetRot3;
  int   I1Pro;

  WParams I1params, I0params;
  double K=1.0, rot1=0.0, rot2=0.0, rot3=0.0;
  double Kout=1.0, rot1out=0.0, rot2out=0.0, rot3out=0.0;
  int transform=0;

  float Radius2, Radius, Angle;

  float Radius_1, Radius_2;
  float SinAngle, CosAngle; 

  int i_1, i_2;

  float f0_1, f0_2;
  WaxsCoord W0, W1, W2;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  IO_float A0, Asym;
  IO_long  S2, Csym, Vsum, DoArc;

  float Radius0, Radius1;
  float RadiusMin, RadiusMax;

  float Angle0, Angle1;
  float AngleMin, AngleMax;

 /* xyfile parameters */
  char xypath[IO_len], xyname[IO_len];
  char xy_pattern[IO_len], xyfull[IO_len];
  char xfile[IO_len], yfile[IO_len];
  int xstream, ystream;
  float *xdata, *ydata;
  float *pxdata, *pydata;
  int idata;
  long xynum;
  size_t xysize;
  int ErrorValue;

  IO_long Xy;
  IO_line Xyname;

  IO_long Waxs;
  IO_long Mask;

  char key[IO_len];

  float I0Off_1, I0Ps_1;
  float I0Off_2, I0Ps_2;
  float I1Off_1, I1Ps_1;
  float I1Off_2, I1Ps_2;

  float xypix_1, xypix_2;
  float xycen_1, xycen_2;
  float xydis, xywvl;
  float xyrot_1, xyrot_2, xyrot_3;

  SetSaxsError( Version, RoutineName, NULL );

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,%ld] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,%ld] ", i, ihb[i].ImNum); }

  printf(")\n\n");

// unnecessary check!!!
// /* Check the number of images */
//  if (pcb->ImgBlkLen!=no_blocks) {
//    fprintf(ERROUT,"%d images found, 1 input and 1 output image required\n",
//            pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* --- Get other option values */
  A0 = option_float ( SA0, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  Asym = option_float ( SAsym, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;
  S2   = option_flag ( SS2, pcb, pstatus );
  if (*pstatus!=Success) return;
  Csym = option_flag ( SCsym, pcb, pstatus );
  if (*pstatus!=Success) return;
  Vsum = option_flag ( SVsum, pcb, pstatus );
  if (*pstatus!=Success) return;

  DoArc = option_flag ( SDoArc, pcb, pstatus );
  if (*pstatus!=Success) return;

  Xy = option_flag( SXy, pcb, pstatus);
  if (*pstatus!=Success) return;
  Xyname = option_line ( SXyname, pcb, pcb->ib, num, 1, pstatus );
  if (*pstatus!=Success) return;

  Waxs = option_flag( SWaxs, pcb, pstatus);
  if (*pstatus!=Success) return;

  Mask = option_flag( SMask, pcb, pstatus);
  if (*pstatus!=Success) return;


  /* mask input image */
  if (Mask.V) {
    if ( ( ImageIsNew(1,ihb) ) || ( ImageIsNew(2,ihb) ) ) {
      // mask ihb[1] with ihb[2]
      if (pcb->TestBit) {
        printf("Masking input image\n");
      }

      mask_region( pcb, ihb, 1, 2, ihb[1].Data,
        (size_t) ihb[1].Dim[1]*(size_t) ihb[1].Dim[2], pstatus );
      if (*pstatus!=Success) return;
    }
  }

  /* check the reference system: Normal or Saxs required */
  if ( (pcb->RSys.V != IO_Normal) && (pcb->RSys.V != IO_Saxs) ) {
    fprintf(ERROUT,"ERROR: Reference system is %s, but %s or %s is required\n",
      reftostr(pcb->RSys.V), reftostr(IO_Normal), reftostr(IO_Saxs));
    SetSaxsErrorMessage(NULL);
    *pstatus = ReferenceNotSupported;
    return;
  }

  /* input block 1 only */
  GetReferenceParameters( pcb, ihb, 1, 1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  /* radial output block reference system is Normal or Saxs */
  GetReferenceParametersC( pcb->RSys.V, pcb, ihb, 1, 0, 0,
                           Off_1, Ps_1, pstatus );
  if (DoArc.V) /* angular output reference system is Normal or Saxs */
    GetReferenceParametersC( pcb->RSys.V, pcb, ihb, 2, 0, 0,
                             Off_2, Ps_2, pstatus );
  else /* angular output reference system always is Normal */
    GetReferenceParametersC( IO_Normal, pcb, ihb, 2, 0, 0,
                             Off_2, Ps_2, pstatus );

  /* Subtract output shift for calculation */
  if (ihb[0].Shift[1].I) Off_1[0] = Off_1[0]-ihb[0].Shift[1].V;
  if (ihb[0].Shift[2].I) Off_2[0] = Off_2[0]-ihb[0].Shift[2].V;

  if (pcb->TestBit) {
    printf("Off_1[%d] = % f, Ps_1[%d] = % f\n", 0,Off_1[0],0,Ps_1[0]);
    printf("Off_2[%d] = % f, Ps_2[%d] = % f\n", 0,Off_2[0],0,Ps_2[0]);
  }

  /* 1 input, 1 output */

  rsys        = pcb->RSys.V;

  I0Data      = ihb[0].Data;
  E0Data      = ihb[0].VarDat;
  I0Dummy     = ihb[0].Dummy.V;
  I0DDummy    = ihb[0].DDummy.V;
  I0Dim_1     = (int) ihb[0].Dim[1];
  I0Dim_2     = (int) ihb[0].Dim[2];

  I0Offset_1  = ihb[0].Offset[1].V;
  I0Offset_2  = ihb[0].Offset[2].V;
  I0PixSiz_1  = ihb[0].PixSiz[1].V;
  I0PixSiz_2  = ihb[0].PixSiz[2].V;
  I0Cen_1     = ihb[0].Center[1].V;
  I0Cen_2     = ihb[0].Center[2].V;
  I0Dis       = ihb[0].SampleDistance.V;
  I0Wvl       = ihb[0].WaveLength.V;
  I0Pro       = ihb[0].Projection.V;
  I0Axis_1    = ihb[0].AxisType[1].V;
  I0Axis_2    = ihb[0].AxisType[2].V;

  I1Ave   = pcb->ib[1].Ave.V;
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  I1Offset_1  = ihb[1].Offset[1].V;
  I1Offset_2  = ihb[1].Offset[2].V;
  I1PixSiz_1  = ihb[1].PixSiz[1].V;
  I1PixSiz_2  = ihb[1].PixSiz[2].V;
  I1Cen_1     = ihb[1].Center[1].V;
  I1Cen_2     = ihb[1].Center[2].V;
  I1Dis       = ihb[1].SampleDistance.V;
  I1Wvl       = ihb[1].WaveLength.V;
  I1DetRot1   = ihb[1].DetRot[1].V;
  I1DetRot2   = ihb[1].DetRot[2].V;
  I1DetRot3   = ihb[1].DetRot[3].V;
  I1Pro       = ihb[1].Projection.V;

  if (pcb->TestBit) {
    printf("Asym.V    = % g_deg\n", Asym.V * rad2deg);
  }

  /* Multiply input image with geometry factors */
  if ( ImageIsNew(1,ihb) ) {
    arc_lfactor ( I1Data,   E1Data, I1Dim_1, I1Dim_2,
                  I1Offset_1, I1PixSiz_1, I1Cen_1,
                  I1Offset_2, I1PixSiz_2, I1Cen_2,
                  I1Dis, I1Wvl, I1Dummy, I1DDummy,
                  Asym.V, pcb->RSys.V,
                  Csym.V, S2.V, pcb->TestBit, pstatus );
    if (*pstatus!=Success) return;
  }

  Radius0=INDEX2N(LOWERBORDER,I0Offset_1,I0PixSiz_1,I0Cen_1);
  Radius1=INDEX2N(LOWERBORDER+I0Dim_1,I0Offset_1,I0PixSiz_1,I0Cen_1);

  RadiusMin = MIN2(Radius0,Radius1);
  RadiusMax = MAX2(Radius0,Radius1);

  Angle0=INDEX2N(LOWERBORDER,I0Offset_2,I0PixSiz_2,I0Cen_2);
  Angle1=INDEX2N(LOWERBORDER+I0Dim_2,I0Offset_2,I0PixSiz_2,I0Cen_2);

  AngleMin = MIN2(Angle0,Angle1);
  AngleMax = MAX2(Angle0,Angle1);

  if (pcb->TestBit) {
    printf("Radius0   = % g,     Radius1   = % g\n", Radius0, Radius1);
    printf("Angle0    = % g_deg,  Angle1   = % g_deg\n", Angle0*rad2deg, Angle1*rad2deg);
    printf("RadiusMin = % g    , RadiusMax = % g\n", RadiusMin, RadiusMax);
  }

  if (DoArc.V) {
    /* arc averaging */
    if (Radius0!=0) AngleMin/=Radius0; else AngleMin=0.0;
    if (Radius1!=0) AngleMax/=Radius1; else AngleMax=0.0;
    if (pcb->TestBit) 
      printf(" AngleMin = % g_deg,  AngleMax = % g_deg\n", 
        AngleMin*rad2deg, AngleMax*rad2deg);

    arc_sum ( rsys, I0Data, E0Data, I0Dim_1, I0Dim_2,
              I0Offset_1, I0PixSiz_1, I0Cen_1,
              I0Offset_2, I0PixSiz_2, I0Cen_2,
              I0Dis,      I0Wvl, 
              I0Pro,
              I0Dummy,    I0DDummy,
              I1Data,     E1Data,
              I1Dim_1,    I1Dim_2,
              I1Offset_1, I1PixSiz_1, I1Cen_1,
              I1Offset_2, I1PixSiz_2, I1Cen_2,
              I1Dis,      I1Wvl,
              I1DetRot1,  I1DetRot2,  I1DetRot3,
              I1Pro,
              I1Dummy,    I1DDummy,
              AngleMin,   AngleMax,
              ihb[0].Shift[1].V, ihb[0].Shift[2].V,
              Vsum.V, I1Ave, pcb->TestBit, pstatus );
    if (*pstatus!=Success) return;
  } else {
    /* angular averaging */
    if (pcb->TestBit) 
      printf(" AngleMin = % g_deg,  AngleMax = % g_deg\n", 
        AngleMin*rad2deg, AngleMax*rad2deg);

    ang_sum ( rsys, I0Data, E0Data, I0Dim_1, I0Dim_2,
              I0Offset_1, I0PixSiz_1, I0Cen_1,
              I0Offset_2, I0PixSiz_2, I0Cen_2,
              I0Dis,      I0Wvl, 
              I0Pro,
              I0Dummy,    I0DDummy,
              I1Data,     E1Data,
              I1Dim_1,    I1Dim_2,
              I1Offset_1, I1PixSiz_1, I1Cen_1,
              I1Offset_2, I1PixSiz_2, I1Cen_2,
              I1Dis,      I1Wvl,
              I1DetRot1,  I1DetRot2,  I1DetRot3,
              I1Pro,
              I1Dummy,    I1DDummy,
              AngleMin,   AngleMax,
              ihb[0].Shift[1].V, ihb[0].Shift[2].V,
              Vsum.V, I1Ave, pcb->TestBit, pstatus );
    if (*pstatus!=Success) return;
  }

  //++++++++++++++
  // if ( rsys == IO_Saxs ) && ( opro == IO_ProSaxs )
  // ODetRot1 = 0.0; ODetRot2 = 0.0; ODetRot3 = 0.0;

  /* open xy-files */
  xyname[0] = xypath[0] = '\0';
  xfile[0]  = yfile[0]  = '\0';
  xstream   = ystream   = -1;
  xdata     = ydata     = (float *) NULL;
  if (Xy.V) {
    filename_parameter (xy_pattern,IO_len,Xyname.V,0);
    filename_pattern (xy_pattern,IO_len,xy_pattern,pcb->ib[0].FileNumber);
    filename_full ( xyfull, IO_len, xy_pattern );
    filename_path ( xypath, IO_len, xyfull );
    filename_name ( xyname, IO_len, xyfull );
    xysize = edf_dim_product ( ihb[1].Dim ); // identical to input array
    /* open x-file */
    sprintf(xfile,"%s%s%s",xypath,"x",xyname);
    xstream = edf_search_stream( xfile,  "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return;
    if (xstream>=0) { *pstatus = FileAlreadyOpened; return; }
    xstream = edf_open_data_file ( xfile, "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return;
    if (xysize>0) {
      if (!(xdata = (float *) malloc(sizeof(float)*xysize))) {
        *pstatus = NotEnoughMemoryAvailable; return; }
      pxdata = xdata;
      for (idata=0;idata<xysize;idata++) *(pxdata++)=0.0;
    }
    /* open y-file */
    sprintf(yfile,"%s%s%s",xypath,"y",xyname);
    ystream = edf_search_stream( yfile,  "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) {  FREE(xdata); return; }
    if (ystream>=0) { FREE(xdata); *pstatus = FileAlreadyOpened; return; }
    ystream = edf_open_data_file ( yfile, "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) {  FREE(xdata); return; }
    if (xysize>0) {
      if (!(ydata = (float *) malloc(sizeof(float)*xysize))) {
        FREE(xdata); *pstatus = NotEnoughMemoryAvailable; return; }
      pydata = ydata;
      for (idata=0;idata<xysize;idata++) *(pydata++)=0.0;
    }
  }

  /* loop over the input array  */
  if (Xy.V) {

    if ( pcb->RSys.V == IO_Saxs ) {

      K    = (double) WAVENUMBER(I1Wvl);
      rot1 = (double) I1DetRot1;
      rot2 = (double) I1DetRot2;
      rot3 = (double) I1DetRot3;

      transform = waxs_get_transform(I1Pro,I0Pro);

      Kout = (double) WAVENUMBER(I0Wvl);
    } 

    waxs_Init ( &I1params, K, rot1, rot2, rot3 );
    waxs_Init ( &I0params, Kout, rot1out, rot2out, rot3out );

    /* The displacements are given in array coordinates */
    /* calculate I0 and I1 array coordinates */
    ARRAYREF(I0Off_1,I0Ps_1);
    ARRAYREF(I0Off_2,I0Ps_2);
    ARRAYREF(I1Off_1,I1Ps_1);
    ARRAYREF(I1Off_2,I1Ps_2);

    for (i_2=0;i_2<I1Dim_2;i_2++) {
      W1.s_2 =  WORLD( i_2, Off_2[1], Ps_2[1] ); 
      pI1Data = ABSPTR(I1Data,I1Dim_1,I1Dim_2,0,i_2);
      for (i_1=0;i_1<I1Dim_1;i_1++) {
        W1.s_1 =  WORLD( i_1, Off_1[1], Ps_1[1] );
        W2=waxs_Transform(&I1params, &I0params, -transform, W1);
        Radius_1=W2.s_1;
        Radius_2=W2.s_2;

        Radius2      = Radius_1*Radius_1+Radius_2*Radius_2;
        Radius       = sqrt(Radius2);

        if (fabs(Radius)>radius_eps) {
          CosAngle     = Radius_1/Radius;
          SinAngle     = Radius_2/Radius;
        } else {
          CosAngle = 1.0;
          SinAngle = 0.0;
        }

        Angle = (float) atan2( (double) SinAngle, (double) CosAngle );
        if (Angle<0.0) Angle += 2*SAXS_PI;

        if ( (AngleMin < Angle) && (Angle < AngleMax) ) { 

          W0.s_1 = Radius;
          if (DoArc.V) W0.s_2 = Radius*Angle;
          else W0.s_2 = Angle;

          /* calculate I0 indices */
          f0_1 = INDEX(W0.s_1, Off_1[0], Ps_1[0]);
          f0_2 = INDEX(W0.s_2, Off_2[0], Ps_2[0]);

          /* calculate x- and y-shifts */
          pxdata = xdata + (pI1Data - I1Data);
          *pxdata = WORLD(f0_1, I0Off_1, I0Ps_1) - WORLD(i_1, I1Off_1, I1Ps_1);
          pydata = ydata + (pI1Data - I1Data);
          *pydata = WORLD(f0_2, I0Off_2, I0Ps_2) - WORLD(i_2, I1Off_2, I1Ps_2);
        }

        pI1Data++;
      } /* for i_1 ... */
    } /* for i_2 ... */
  } /* if (Xy.V) */

  /* write and close xy-files */
  if (Xy.V) {
    /* xfile */
    xynum = ihb[0].ImNum;
    printf("Writing : %s%s, Image :  %ld\n","x",xyname,xynum);
    edf_write_header_history ( xstream, xynum, 1l,
                               SaxsHistoryKey, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    /* The xy-displacement files cover the same region as the input image,
       i.e. dimensions, offsets and binning sizes are the same. */
    edf_write_header_float(xstream,xynum,1l,KOffset_1,ihb[1].Offset[1].V,
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KOffset_2,ihb[1].Offset[2].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    if ( (ihb[1].BinSiz[1].I)||(ihb[1].BinSiz[2].I)||\
         (ihb[1].BinSiz[1].V!=1.0)||(ihb[1].BinSiz[2].V!=1.0) ) {
      edf_write_header_float(xstream,xynum,1l,KBinSiz_1,ihb[1].BinSiz[1].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
      edf_write_header_float(xstream,xynum,1l,KBinSiz_2,ihb[1].BinSiz[2].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    }
    xypix_1 = ihb[0].PixSiz[1].V; xypix_2 = ihb[0].PixSiz[2].V;
    xycen_1 = ihb[0].Center[1].V; xycen_2 = ihb[0].Center[2].V;
    xydis   = ihb[0].SampleDistance.V;
    xywvl   = ihb[0].WaveLength.V;
    xyrot_1 = ihb[0].DetRot[1].V; xyrot_2 = ihb[0].DetRot[2].V;
    xyrot_3 = ihb[0].DetRot[3].V;
    edf_write_header_float(xstream,xynum,1l,KPixSiz_1,xypix_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KPixSiz_2,xypix_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KCenter_1,xycen_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KCenter_2,xycen_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KSampleDistance,xydis,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KWaveLength,xywvl,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

    edf_write_header_unit(xstream,xynum,1l,KDetRot_1,xyrot_1,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_2,xyrot_2,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_3,xyrot_3,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

   /* Write parameters of transformed image to x-displacement file */
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,"Dim_1"),ihb[0].Dim[1],
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,"Dim_2"),ihb[0].Dim[2],
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_1),ihb[0].Offset[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_2),ihb[0].Offset[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KBinSiz_1),ihb[0].BinSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KBinSiz_2),ihb[0].BinSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_1),ihb[0].PixSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_2),ihb[0].PixSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_1),ihb[0].Center[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_2),ihb[0].Center[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

    if ( pcb->RSys.V == IO_Saxs ) {
      edf_write_header_float(xstream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KSampleDistance),ihb[0].SampleDistance.V,
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
      edf_write_header_float(xstream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KWaveLength),ihb[0].WaveLength.V,
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    }

    if ( I0Pro != IO_ProSaxs ) {
      edf_write_header_line(xstream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KProjection),protostr(I0Pro),
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    }

    if ( I0Axis_2 != IO_AxisTypeDistance ) {
      edf_write_header_line(xstream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KAxisType_2),axistostr(I0Axis_2),
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    }

    edf_write_data( xstream,xynum,1l,ihb[1].Dim,xdata,MFloat,
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    FREE(xdata);
    edf_close_data_file ( xstream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
   /* yfile */
    printf("Writing : %s%s, Image :  %ld\n","y",xyname,xynum);
    edf_write_header_history ( ystream, xynum, 1l,
                               SaxsHistoryKey, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KOffset_1,ihb[1].Offset[1].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KOffset_2,ihb[1].Offset[2].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    if ( (ihb[1].BinSiz[1].I)||(ihb[1].BinSiz[2].I)||\
         (ihb[1].BinSiz[1].V!=1.0)||(ihb[1].BinSiz[2].V!=1.0) ) {
      edf_write_header_float(ystream,xynum,1l,KBinSiz_1,ihb[1].BinSiz[1].V,
                             &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
      edf_write_header_float(ystream,xynum,1l,KBinSiz_2,ihb[1].BinSiz[2].V,
                             &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
    }
    edf_write_header_float(ystream,xynum,1l,KPixSiz_1,xypix_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KPixSiz_2,xypix_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KCenter_1,xycen_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KCenter_2,xycen_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KSampleDistance,xydis,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KWaveLength,xywvl,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

    edf_write_header_unit(ystream,xynum,1l,KDetRot_1,xyrot_1,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_2,xyrot_2,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_3,xyrot_3,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

   /* Write parameters of transformed image to y-displacement file */
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,"Dim_1"),ihb[0].Dim[1],
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,"Dim_2"),ihb[0].Dim[2],
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_1),ihb[0].Offset[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KOffset_2),ihb[0].Offset[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KBinSiz_1),ihb[0].BinSiz[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_len,KDisplaced,KBinSiz_2),ihb[0].BinSiz[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_1),ihb[0].PixSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KPixSiz_2),ihb[0].PixSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_1),ihb[0].Center[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_len,KDisplaced,KCenter_2),ihb[0].Center[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

    if ( pcb->RSys.V == IO_Saxs ) {
      edf_write_header_float(ystream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KSampleDistance),ihb[0].SampleDistance.V,
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
      edf_write_header_float(ystream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KWaveLength),ihb[0].WaveLength.V,
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
    }
    if ( I0Pro != IO_ProSaxs ) {
      edf_write_header_line(ystream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KProjection),protostr(I0Pro),
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
    }

    if ( I0Axis_2 != IO_AxisTypeDistance ) {
      edf_write_header_line(ystream,xynum,1l,
        strlib_concat(key,IO_len,KDisplaced,KAxisType_2),axistostr(I0Axis_2),
        &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
    }

    edf_write_data( ystream, xynum, 1l, ihb[1].Dim, ydata, MFloat,
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_close_data_file ( ystream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    FREE(ydata);
  }

} /* saxs_arc*/

/*---------------------------------------------------------------------------
ArgvAngleFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

The header block of the first image is read into ihb[1].
---------------------------------------------------------------------------*/
void ArgvAngleFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                          int * pstatus)
{ int blkno;
  long int image_1st, image_lst;
  char tmp_line[IO_len];
  float Bin_1, Bin_2;
  long num[BlockNum];
  
  /* ---  File names and open modes --- */

    blkno = 1;
    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
    }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(tmp_line,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,tmp_line, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,tmp_line,ib,blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile(pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                  IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    /* Search for minimum and maximum image number */
    (void) SearchMinMaxImage (pcb,ib,blkno,&image_1st,&image_lst,pstatus);
    if (*pstatus!=Success) return;
    if (!ib[blkno].First.I) ib[blkno].First.V = image_1st;
    if (!ib[blkno].Last.I)  ib[blkno].Last.V  = image_lst;
    num[blkno] = ib[blkno].First.V;

    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect;
      ib[0].OpenMode.I = TRUE;
    }
 
    /*--- File name ib[0].Name of output */
    argv_filename( pcb,"Output sequence",ib,0,DefaultOutput,pstatus);
    if (*pstatus!=Success) return;

  /* ---  Image numbers --- */

    /*--- Argument : number of first image */
    argv_long(pcb,"First image of input sequence 1",&ib[1].First,ib[1].First.V,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
      else image_lst = ib[1].Last.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last,image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    num[1] = ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

    /*--- Argument : image coordinate 1 of center */
    Bin_1 = MAX2( 1,ihb[1].Bin[1].V);
    sprintf(tmp_line,"Center 1 (image coordinate), e.g. %g",
      Bin_1*ihb[1].Center[1].V);
    argv_flexp(pcb,tmp_line,&ib[1].Center[1],"",pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : image coordinate 2 of center */
    Bin_2 = MAX2( 1,ihb[1].Bin[2].V);
    sprintf(tmp_line,"Center 2 (image coordinate), e.g. %g",
      Bin_2*ihb[1].Center[2].V);
    argv_flexp(pcb,tmp_line,&ib[1].Center[2],"",pstatus);
    if (*pstatus!=Success) return;

    // re-read image header
    num[1] = ib[1].First.V;
    (void) ReadImageHeader( pcb, ib, 1, num, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(&ihb[1]);

    /* Defaults for output file */
    sprintf(tmp_line,"%g",ihb[1].Dummy.V);
    if (!ib[0].Dummy.I) CopyLine(tmp_line,ib[0].Dummy.V,IO_len,0);

    /*--- Argument : output dummy */
    sprintf(tmp_line,"Output dummy, e.g. %s",ib[0].Dummy.V);
    argv_flexp( pcb, tmp_line, &ib[0].Dummy, "", pstatus);
    if (*pstatus!=Success) return;

    CloseImageFile( pcb, ib, 1, pstatus) ;
    if (*pstatus!=Success) return;

} /* ArgvAngleFilenames */

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{ 

  char  progfn[IO_len];
  ImgHeadBlk ihb[BlockNum];

  char tmp_line[IO_len];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  IO_flexp *pR0, *pA0;
  IO_long DoArc;

  long num[BlockNum];

  IO_long Mask;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  Mask = option_flag ( SMask, pcb, pstatus );
  if (*pstatus!=Success) return;

  /*--- Adjust total number of blocks for masking ---*/
  if (!Mask.V) pcb->ImgBlkLen = BlockNum-1;

  ArgvAngleFilenames ( pcb, ib, ihb, pstatus); // read first image as template
  if (*pstatus!=Success) return;
  // create a clean header ihb[0] and initialize it
  NewImageHeader ( pcb, ib, 0, num, ihb, &(ihb[1]),  // use first image as template
                   saxs_arc_init, pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  pR0 = (IO_flexp*) option_parameter_search( SR0, pcb, pstatus );
  if (*pstatus!=Success) return;
  pA0 = (IO_flexp*) option_parameter_search( SA0, pcb, pstatus );
  if (*pstatus!=Success) return;

  DoArc = option_flag ( SDoArc, pcb, pstatus );
  if (*pstatus!=Success) return;

  num[0] = ib[0].First.V;

  /*--- Argument : output dimension 1 */
  sprintf(tmp_line,"Number of radial intervals, e.g. %ld",ihb[0].Dim[1]);
  argv_lvexp(pcb,tmp_line,&ib[0].Dim[1],"",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : start radius */
  sprintf(tmp_line,"Start radius (%s)",reftostr(pcb->RSys.V));
  argv_flexp(pcb,tmp_line, pR0,"",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : output dimension 2 */
  sprintf(tmp_line,"%ld",ihb[0].Dim[2]);
  argv_lvexp(pcb,"Number of angular intervals",
              &ib[0].Dim[2],tmp_line, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : minimum angle*/
  argv_flexp(pcb,"Start angle (radian or unit _deg)", pA0, "", pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    
     printf("i/p file                : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    
     printf("o/p file                : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   
     printf("first image             : %ld\n",ib[1].First.V);
  if (ib[1].Last.I)    
     printf("last image              : %ld\n",ib[1].Last.V);
  if (ib[1].Inc.I)     
     printf("increment               : %ld\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   
     printf("output dummy            : %s\n",ib[0].Dummy.V);
  if (ib[1].Center[1].I) 
     printf("center 1 (image coord.) : %s\n",ib[1].Center[1].V);
  if (ib[1].Center[2].I)
     printf("center 2 (image coord.) : %s\n",ib[1].Center[2].V);

  if (ib[0].Dim[1].I)  
     printf("num. radial intervals   : %s\n",ib[0].Dim[1].V);
  if (pR0->I) 
     printf("minimum radius          : %s\n",pR0->V);
  if (ib[0].Dim[2].I)  
     printf("num. angular intervals  : %s\n",ib[0].Dim[2].V);
  if (pA0->I) 
     printf("minimum angle           : %s\n",pA0->V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_arc
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

  IO_long Xy;
  IO_line Xyname;

  IO_long Waxs;
  IO_long Mask;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Default reference system is Normal */
  cb.RSys.V = IO_Normal; /* default */

  option_define ( IO_tpflexp, SR0,  "0.0", &cb );
  option_define ( IO_tpflexp, SA0,  "0.0_deg", &cb );
  option_define ( IO_tpflexp, SAsym,"0.0_deg", &cb );

  option_define ( IO_tpflag, SS2,   "FALSE", &cb );
  option_define ( IO_tpflag, SCsym, "FALSE", &cb );
  option_define ( IO_tpflag, SVsum, "FALSE", &cb );

  option_define ( IO_tpflag, SDoArc, "FALSE", &cb );

  option_define ( IO_tpline,  SXyname, "file.edf", &cb );
  option_define ( IO_tpflag,  SXy,     "FALSE",    &cb );

  option_define ( IO_tpflag, SWaxs, "TRUE", &cb );
  option_define ( IO_tpflag, SMask, "FALSE", &cb );
  option_define ( IO_tpflag, SDbg,  "FALSE", &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

 /* Set xy-flag if xy-file name is given */
  if (status==Success)
    Xy = option_flag(SXy,&cb,&status );
  if (status==Success)
    if (!Xy.I) Xyname = option_line(SXyname,&cb,NULL,NULL,0,&status);
  if (status==Success)
    if (!Xy.I) option_flag_update(Xyname.I?TRUE:FALSE,Xy.I,SXy,&cb,&status);

 /* Set reference system to IO_Saxs if +waxs is set */
  if (status==Success)
    Waxs = option_flag(SWaxs,&cb,&status );
  if (status==Success)
    if ((Waxs.V)&&(!cb.RSys.I)) cb.RSys.V = IO_Saxs;
  if (status==Success)
    if ((Waxs.V)&&(cb.RSys.V!=IO_Saxs)) status=NotSaxsReference;

 /* Set +mask if i2nam has been set */
  if (status==Success)
    Mask = option_flag(SMask,&cb,&status );
  if (status==Success) {
    if (ib[2].Name.I) {
      if (!Mask.I) {
        Mask.V = TRUE;
        option_flag_update       ( Mask.V, Mask.I, SMask, &cb, &status );
      }
      /* Mask file is readonly */
      if (!ib[2].OpenMode.I) {
        ib[2].OpenMode.V = IO_Old | IO_FileProtect;
        ib[2].OpenMode.I = TRUE;
      }
    } else {
      if (Mask.V) { // Use the path of the input file
        CopyLine("[FPATH][MaskFileName]",ib[2].Name.V,IO_len,0);
        if (!ib[2].OpenMode.I) {
          ib[2].OpenMode.V = IO_Old | IO_FileProtect;
          ib[2].OpenMode.I = TRUE;
        }
      }
    }
  }

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success)
    IMAGELOOP( &cb, ib, saxs_arc, saxs_arc_init, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */
