/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
NAME

   schulz.c --- Schulz distribution

SYNOPSIS

   double schulz( double R, double Z, double X ); // schulz distribution
   double sigma_schulz( double R, double Z ); // sqrt(<R^2>-<R>^2)
   double R2_schulz( double R, double Z );    // <R^2>
   double R3_schulz( double R, double Z );    // <R^3>
   double f2_schulz( double Z, double qR ); // averaged form factor of spheres

HISTORY
  2000-12-27 V1.0 Peter Boesecke
  2007-02-14 V1.1 PB schultz -> schulz
  2007-04-19 V1.2 PB f2_schulz: added ad hoc expression for zp1atan2da<=eps2,
                     -Wall compiler warnings resolved
---*/

/****************************************************************************
*  Include                                                                  *
****************************************************************************/

# include "schulz.h"

/*+++------------------------------------------------------------------------
NAME
   schulz --- schulz distribution

SYNOPSIS

   double schulz( double R, double Z, double X );            

DESCRIPTION
  The Schulz distribution is a non-symmetric distribution function with the
  mean value <X> = R and sigma = sqrt(<X^2> - <X>^2) = <X>/sqrt(Z+1):

    schulz(X,R,Z) = ((Z+1)/R)^(Z+1) * X^Z * exp( -((Z+1)/R)*X )/gamma(Z+1)

  The function is skewed to larger sizes, tending to a Gaussian form at large
  values of Z. The distribution approaches a delta function delta(X-R) as 
  Z approaches infinity. For a description of the Schulz distribution see:
  M. Kotlarchyk and S.H. Chen, J. Chem. Phys. Vol. 79, No.5, 1983, p 2453 

  The maximum of the function is at X = R * Z/(Z+1) for Z>=0

RETURN VALUE
 
  >=0 schulz distribution value
  <0  error, Z <= -1:

----------------------------------------------------------------------------*/
PUBLIC extern double schulz( double R, double Z, double X )
{
  const double eps = FLT_MIN;
  double Zp1, Zp1dR;
  double value;

  if (Z<=-1) return(-1.0);

  Zp1 = Z+1;
  if (fabs(R)<eps) return(-1.0);
  Zp1dR = Zp1/R;

  if (X>0) value = exp( log(Zp1dR)*Zp1+log(X)*Z-Zp1dR*X-loggamma(Zp1) );
    else value = 0.0;

  return( value );

} /* schulz */

/*+++------------------------------------------------------------------------
NAME
   f2_schulz --- Average form factor of spheres with a Schulz distribution

SYNOPSIS

   double f2_schulz( double Z, double qR ); 

DESCRIPTION
   f2_schulz returns the average form factor of spheres following
   a Schulz distribution. Z is the width parameter of the distribution.

   The absolute scattering intensity of diluted spherical particles with
   average radius R and averaged particle density n is given by

   1/A*(dSigma/dOmega) = n * re^2 * DeltaRhoElektron^2 * Vs^2 * f2_schulz * d

   For qR values less than 0.04 a constant value is returned.

   The calculation is described in the article:
   M. Kotlarchyk and S.H. Chen, J. Chem. Phys. Vol. 79, No.5, 1983, p 2453

RETURN VALUE

  >=0 f2_schulz value
  <0  error, e.g. Z <= -1 or qR<=0:

----------------------------------------------------------------------------*/
PUBLIC extern double f2_schulz( double Z, double qR )
{
  const double eps1 = 0.1;   // replace cos(x) by 1-2*sin(x/2)^2 for x<eps1 
  const double eps2 = 0.05;  // use power series for A+B
  const double qRmin = 0.04; // replace qR by qRmin
//  const double Zmax = 10000; // fixed value for center if qR<qRmin and Z>Zmax

  double Zp1, Zp2, Zp3, Zp5;
  double invaa, alpha;
  double zp1atan2da, zp3atan2da, zp2atan2da;
  double A, B, C, D, E;
  double AB, ABC;
  double qR2, qR4, qR6;
  double G1malphahzp1;
  double arg;
  double value;

  if (Z<=-1) return(-1.0);

  Zp1 = Z+1;
  Zp2 = Z+2;
  Zp3 = Z+3;
  Zp5 = Z+5;

  // calculate 9/2 * G1 * pow(alpha,Zp1) / qR^6

  if ((Z<=10000)||(qR>=qRmin)) {
    if (qR<qRmin) qR=qRmin;

    alpha = Zp1/qR;
    qR2 = qR*qR;
    qR4 = qR2*qR2;
    qR6 = qR4*qR2;

    arg = atan(qR*2.0/Zp1);
    zp1atan2da = Zp1*arg;
    zp3atan2da = Zp3*arg;
    zp2atan2da = Zp2*arg;

    if (alpha<1000) {
      A = 1.0;
      B =          -exp( log(alpha)*Zp1 - log(4.0+alpha*alpha)*Zp1*0.5 );
      C = (Zp2/Zp1)*qR2;
      D = (Zp1*Zp2)*exp( log(alpha)*Zp1 - log(4.0+alpha*alpha)*Zp3*0.5 );
      E =  -2.0*Zp1*exp( log(alpha)*Zp1 - log(4.0+alpha*alpha)*Zp2*0.5 );

      if (zp1atan2da>eps1) { // normal form
        ABC = A+B*cos(zp1atan2da);

      } else {
        if (zp1atan2da>eps2) { // special expression for cos
          AB  = A+B;
          ABC = (AB)-B*2*sin(zp1atan2da*0.5)*sin(zp1atan2da*0.5);
        } else { // added ad hoc for the case zp1atan2da<=eps2 (2007-04-19)
          AB  = A+B;
          ABC = (AB)-B*2*qR2;
        }
      }

    } else { // special cases for large alpha
      A = 1.0;
      B =                -exp( -log(4.0/(alpha*alpha)+1.0) * Zp1*0.5 );
      C = (Zp2/Zp1)*qR2;
      D = (Zp2/Zp1)*qR2 * exp( -log(4.0/(alpha*alpha)+1.0) * Zp3*0.5 );
      E =       -2.0*qR * exp( -log(4.0/(alpha*alpha)+1.0) * Zp2*0.5 ); 

      if (zp1atan2da>eps1) { // normal form
        ABC = A+B*cos(zp1atan2da);

      } else {
        if (zp1atan2da>eps2) { // special expression for cos
          AB  = A+B;
          ABC = (AB)-B*2*sin(zp1atan2da*0.5)*sin(zp1atan2da*0.5);

        } else { // approximate A+B by a power series of invaa
          invaa = 1.0/(alpha*alpha);
          AB  = ((Zp1*2.0)-
                 ((Zp1*Zp3*2.0)+(Zp1+Zp3+Zp5)*(8.0/3.0)*invaa)*invaa)*invaa;
          ABC = (AB)-B*2*sin(zp1atan2da*0.5)*sin(zp1atan2da*0.5);
        }
      }
    }

    G1malphahzp1=ABC+C+D*cos(zp3atan2da)+E*sin(zp2atan2da);

    value = 4.5 * G1malphahzp1/qR6; 

  } else { // used fixed value in center
    value = (2.0/5.0+3.0*(Zp2/Zp1)*pow(Zp3/Zp1,4)-(12.0/5.0)*pow(Zp2/Zp1,5));
  }

  return( value );

} /* f2_schulz */

/*+++------------------------------------------------------------------------
NAME
   sigma_schulz --- sqrt(<R^2>-<R>^2)

SYNOPSIS

   double sigma_schulz( double R, double Z );

DESCRIPTION
Returns sigma = sqrt(<R^2>-<R>^2) of the schulz distribution.

RETURN VALUE
   sqrt(<R^2>-<R>^2) for Z>-1 otherwise -1

----------------------------------------------------------------------------*/
PUBLIC extern double sigma_schulz( double R, double Z )
{ double value;

  if (Z<=-1) return(-1.0);

  value = R*sqrt(1.0/(Z+1.0));
  
  return(value);

} /* sigma_schulz */

/*+++------------------------------------------------------------------------
NAME
   R2_schulz --- sqrt(<R^2>) 

SYNOPSIS

   double R2_schulz( double R, double Z );

DESCRIPTION
Returns the expectation value of <R^2> 

RETURN VALUE
   sqrt(<R^2>) for Z>-1 otherwise -1

----------------------------------------------------------------------------*/
PUBLIC extern double R2_schulz( double R, double Z )
{ double value;

  if (Z<=-1) return(-1.0);
  
  value = R*R*((Z+2.0)/(Z+1.0));

  return(value);

} /* R2_schulz */

/*+++------------------------------------------------------------------------
NAME
   R3_schulz --- <R^3>

SYNOPSIS

   double R3_schulz( double R, double Z );

DESCRIPTION
Returns the expectation value <R^3>.

Lacking an exact expression for pow(<R^3>,1/3) the formula was 
empirically developed and was tested between Z=-0.99 and large Z. The
difference to the numerically calculated values are less than the 
expected calculation errors (<1%). 

The accuracy of the
calculated values are better than the accuracy

RETURN VALUE
   <R^3> for Z>-1 otherwise -1

----------------------------------------------------------------------------*/
PUBLIC extern double R3_schulz( double R, double Z )
{ double value;

  if (Z<=-1) return(-1.0);

  value = (R*R*R)*((Z+2.0)*(Z+3.0)/(Z+1.0)/(Z+1.0));

  return(value);

} /* R3_schulz */

