/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
# define MPA_VERSION      "mpa : V0.31 Peter Boesecke 2011-06-30"
/*+++------------------------------------------------------------------------
NAME
   mpa --- routines to read mpa files 

SYNOPSIS

   # include mpa.h

HISTORY
  2004-07-29 V0.0 
  2007-04-19 V0.2 sizeof is always first operand,
                  -Wall compiler warnings resolved
  2011-05-26 V0.3 mpa_Read: strncat replaced with strncpy
  2011-06-30 V0.31 mpa_Read: strncpy corrected

DESCRIPTION

----------------------------------------------------------------------------*/
/******************************************************************************
* Include Files                                                               *
******************************************************************************/

# include "mpa.h"
# include "SaxsDefinition.h"

/******************************************************************************
* Private Defines                                                             *
******************************************************************************/

# define MPA_MAXLEN 1024
# define MPA_BUFLEN 1024

# define MPA_SECTION_START  '['
# define MPA_SECTION_END    ']'
# define MPA_SEPARATOR      ','  /* separator between parameters */

# ifndef MAX
#   define MAX(A,B) ((A)>(B))?(A):(B)
#   define MIN(A,B) ((A)<(B))?(A):(B)
# endif

/******************************************************************************
* Private Type Defs                                                           *
******************************************************************************/

  enum MpaSections { MpaInvalid, MpaHeader, MpaAdc, MpaMap, MpaDat, MpaCDat,
                     MpaEnd };

  const char * MpaSectionStrings[7] =
      { "INVALID",
        "HEADER", "ADC", "MAP", "DATA", "CDAT", 
        (const char *) NULL };

/******************************************************************************
* Private Constants                                                           *
******************************************************************************/
// PRIVATE const char * filename_default = "data_001.mpa"; //unused
// PRIVATE const char * mpa_header_key = "mpa_header"; //unused
PRIVATE const char end_of_line[3] = { '\r', '\n', '\0' };

/******************************************************************************
* Private Variables                                                           *
******************************************************************************/
PRIVATE int      MPA_debug = 0;

/******************************************************************************
* Routines                                                                    *
******************************************************************************/

/*---------------------------------------------------------------------------
NAME
 
SYNOPSIS
 
DESCRIPTION
 
RETURN VALUE
---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------
NAME

   newstr_mpa --- allocate memory and copy a character string into it

SYNOPSIS

   char * newstr_mpa( const char * string );

DESCRIPTION
  Allocates strlen(string)+1 bytes of memory and copies string into it.
  If string is the NULL pointer an empty string with strlen 0 is created.
  In case of success the pointer to the allocated memory is returned. The
  null pointer is returned in case of an error.

RETURN VALUE
  Returns the pointer to the allocated string or (char *) NULL in case
  of an error.
---------------------------------------------------------------------------*/
char * newstr_mpa( const char * string )
{ char * newstr_mpaing;

  if (string) {
    if (!(newstr_mpaing=(char*) malloc(sizeof(char)*(strlen(string)+1))))
      return((char*) NULL);
    (void) strcpy(newstr_mpaing,string);
  } else {
    if (!(newstr_mpaing=(char*) malloc(sizeof(char)))) return((char*) NULL);
    newstr_mpaing[0]='\0';
  }

  return( newstr_mpaing );

} /* newstr_mpa */

/* removes end_of_line from line */
char * rmeoln_mpa ( char * line )
{ char * pc = line;


  if (pc)
    while (*pc) {
      if (strchr( end_of_line, (int) *pc )) *pc='\0';
      pc++;
      }
  return( line );

} /* rmeoln_mpa */

/*---------------------------------------------------------------------------
NAME

   ltrim_mpa --- return a pointer to the first non-space character

SYNOPSIS

   char * ltrim_mpa( char * string );

RETURN VALUE
   Returns the pointer to the first non-space character in string

---------------------------------------------------------------------------*/
char * ltrim_mpa( char * string )
{
  if (!string) return( (char *) NULL);

  while (*string==' ') string++;

  return( string );

} // ltrim_mpa

/*---------------------------------------------------------------------------
NAME

   strsp_mpa --- search the end of the next delimiter

SYNOPSIS

   strsp_mpa( char * string );

DESCRIPTION
   Returns the position of the end of the next group of spaces before a
   non-space character. If the next non-space character is a back-slash or
   if the it is no spaces are found the return value is NULL

RETURN VALUE
   Returns the position of the end of the next group of spaces.
---------------------------------------------------------------------------*/
const char * strsp_mpa( const char * string )
{ const char * stop, *start;

  if (!string) return( (const char *) NULL );

  start = string;
  stop = strchr(start,' '); // searching next space as delimiter

  if (stop) { // skip next spaces
    while (*stop==' ') stop++;
    if (*stop!='\\') stop--;
  }

//++++++++  if (*stop=='\\') stop = (const char *) NULL;
  return ( stop );

} // strsp_mpa

/*---------------------------------------------------------------------------
NAME

  str2section --- converts a string to a section 

SYNOPSIS

  int str2section( const char * string );

DESCRIPTION

RETURN VALUE
   0 : error, e.g. cannot convert
  >0 : valid section value
  -------------------------------------------------------------------------*/
int str2section( const char * string )
{ int  NE=True;
  long i = 0;
  const char * ps;

  while ( (NE && MpaSectionStrings[i]) ) {
    ps=MpaSectionStrings[i++];
    NE=strncmp(string,ps,strlen(ps));
  }

  i = MAX(0,i-1);

  if (NE) return( MpaInvalid );
    else return( i );

} /* str2section */

/*---------------------------------------------------------------------------
NAME

  str2sectionnum --- reads section number from string

SYNOPSIS

  int str2sectionnum( const char * string );

DESCRIPTION

RETURN VALUE
   -1 : error, e.g. no valid section string 
  >=0 : section number 
  -------------------------------------------------------------------------*/
int str2sectionnum( const char * string )
{ int  NE=True;
  long i = 0;
  const char *ps;
  char *pe;

  while ( (NE && MpaSectionStrings[i]) ) {
    ps=MpaSectionStrings[i++];
    NE=strncmp(string,ps, strlen(ps));
  }

  i = MAX(0,i-1);

  if (NE) return( -1 );

  // read section number
  ps=string+strlen(MpaSectionStrings[i]);

  return( strtol(ps,&pe,10) );

} /* str2sectionnum */

/*---------------------------------------------------------------------------
NAME
  mpa_New --- create new mpa object with key mpa_key

SYNOPSIS
  MpaData *  mpa_New( const char * mpa_key );

DESCRIPTION
  Creates a new mpa object with key mpa_key. 

RETURN VALUE
  Pointer to created mpa object or NULL in case of an error 
---------------------------------------------------------------------------*/
MpaData * mpa_New( const char * mpa_key )
{ const char * funcnam = "mpa_New";
  MpaData * mpa_data;

  if (MPA_debug>0) printf("(%s):\n", funcnam);

  if (!(mpa_data = (MpaData *) malloc( sizeof(MpaData) )))
    return( mpa_data );

  // create header
  if ( !(edf_new_header( mpa_key )) ) {
    fprintf(stderr,"ERROR (%s): Cannot create header.\n", funcnam);
    free( mpa_data );
    return( (MpaData *) NULL );
  }

  mpa_data->header_key = newstr_mpa( mpa_key );

  mpa_data->data0    = (long *) NULL;
  mpa_data->data0len = 0l;            // buffer length

  mpa_data->data1    = (long *) NULL;
  mpa_data->data1len = 0l;            // buffer length

  mpa_data->cdat0    = (long *) NULL;
  mpa_data->cdat0len = 0l;            // buffer length

  return( mpa_data );

} // mpa_New

/*---------------------------------------------------------------------------
NAME
  mpa_Free --- release mpa object

SYNOPSIS
  int mpa_Free( MpaData * mpa_data )

DESCRIPTION
  Releases mpa_data

RETURN VALUE
  0: no error, otherwise error
---------------------------------------------------------------------------*/
int mpa_Free( MpaData * mpa_data )
{ const char * funcnam = "mpa_Free";

  if (MPA_debug>0) printf("(%s):\n", funcnam);

  if (mpa_data) {
    if (mpa_data->header_key) {
      edf_free_header (mpa_data->header_key);
      free(mpa_data->header_key);
    }
    if (mpa_data->data0) free(mpa_data->data0);
    if (mpa_data->data1) free(mpa_data->data1);
    if (mpa_data->cdat0) free(mpa_data->cdat0);

    free(mpa_data);
  }

  return(0);

} // mpa_Free

/*+++------------------------------------------------------------------------
NAME
  mpa_Read --- read mpa header and data

SYNOPSIS

  int mpa_Read( MpaData * data, const char * filename,
                int readdata );

DESCRIPTION

  Read mpa header and data from file filename

The mpa file consists of different sections with key value pairs separated 
by equal signs. A section starts with an identifier in square brackets.
The header starts immediately with key value pairs. 

<key-value pairs>
[ADC1]
<key-value pairs>
[ADC2]
<key-value pairs>
[MAP0] 1A x 1B
<key-value pairs>
[DATA0,1024 ]
1024 lines with a single integer number follow
[DATA1,1024 ]
1024 lines with a single integer number follow
[CDAT0,1048576 ]
1048576 lines with a single integer number follow

The idea is to write all key-value pairs preceeded by the section identifier
into the edf header.
Binary data arrays (DATA0, DATA1, CDATA0) will be written into separate
memories.

PARAMETERS

  MpaData * data        : Mpa data object into which the data is loaded
  const char * filename : name of mpa file
  int readdata          : 0: read header only,
                          1: read header and data

RETURN VALUES

  0 : successful or eof
 -1 : error

----------------------------------------------------------------------------*/
int mpa_Read( MpaData * mpadata, const char * filename, int readdata )
{ const char * funcnam = "mpa_Read";
  FILE *input;
  char buffer[MPA_BUFLEN];
  char line[MPA_MAXLEN];
  char keybuffer[MPA_MAXLEN];

  long i;

  char sectionkey[MPA_MAXLEN]="";
  char sectionval[MPA_MAXLEN]="";
  long sectionnum=0;
  long sectionlen=0;

  char * key, * value, * len;

  char * comment_marker = "#";

  size_t comment_len;
  char *pe;

  int  section;

  int  status, errorvalue;

  if (MPA_debug>0) printf("(%s):\n", funcnam);

  comment_len = strlen(comment_marker);

  if (MPA_debug>0) printf("(%s): \"%s\": fopen\n", funcnam, filename);

  input = fopen( filename, "r");
  if (!input) {
    sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
    perror(buffer); return(-1);
  }

  section = MpaHeader;
  /* --- read header until data section */
  if (!(feof(input))) {
    if (MPA_debug>0) 
      printf("(%s): \"%s\" %d: fgets", funcnam, filename, section);

    fgets (line, MPA_MAXLEN, input);
    if (ferror(input)) {
      sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
      perror(buffer); fclose( input ); return(-1);
    }
    if (MPA_debug>0) printf(" \"%s\"\n", line);
  }

  while (!feof(input)) { 
    if (MPA_debug>0) 
      printf("(%s): \"%s\" %d:\n", funcnam, filename, section);
    switch (section) { // read header and data
      case MpaHeader:
      case MpaAdc:
      case MpaMap:
        // read key value pairs from section
        while (!(feof(input)||(line[0]==MPA_SECTION_START))) {
          if (!strncmp(line,comment_marker,comment_len)) {
            // split line into key and value
            rmeoln_mpa ( line ); // remove end of line characters
            key = ltrim_mpa( line ); // skip leading white spaces
            value = strchr(key,'='); // searching first '=' char as delimiter
            if (value) *value++ = '\0'; // split line into key and value
    
            if (value) {
              strcpy(keybuffer,sectionkey);
              // +++++++++++ strncat(keybuffer, key, MIN(strlen(key)+1,MPA_MAXLEN));
              // // +++++++++++ char *strncpy(char *restrict s1, const char *restrict s2, size_t n);
              // +++++++++++++ need to add strlen(keybuffer)!!!! 
              // ++++++++++++ strncpy(keybuffer, key, MIN(strlen(key)+1,MPA_MAXLEN));
              strncpy(keybuffer+strlen(keybuffer), key, 
                MIN(strlen(key)+1,MPA_MAXLEN-strlen(keybuffer)));
              if ( !(edf_add_header_element ( mpadata->header_key,
                  keybuffer, value, &errorvalue, &status )) ) {
                fprintf(stderr,"ERROR (%s): \"%s\": %s.\n",
                  funcnam, filename, edf_report_data_error( errorvalue ));
                fclose( input ); return(-1);
              }
            }
          }
          if (MPA_debug>1) 
            printf("(%s): \"%s\" %d: fgets", funcnam, filename, section);

          fgets (line, MPA_MAXLEN, input);
            if (ferror(input)) {
              sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
              perror(buffer); fclose( input ); return(-1);
            }
          if (MPA_debug>1) printf(" \"%s\"\n", line);
        }
        break;

      case MpaDat:
        // read datalen lines with single values
        switch (sectionnum) {
          case 0:
            if (readdata) {
              mpadata->data0 = (long*) malloc ( sizeof(long)*sectionlen );
              if (!mpadata->data0) return(-1);
            } else mpadata->data0 = (long*) NULL;
            mpadata->data0len = sectionlen;
            for (i=0;(i<sectionlen)&&(!feof(input));i++) {
              if (mpadata->data0) mpadata->data0[i]=strtol(line,&pe,10);
              if (MPA_debug>1) 
                printf("(%s): \"%s\" %d: fgets", funcnam, filename, section);

              fgets (line, MPA_MAXLEN, input);
              if (ferror(input)) {
                sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
                perror(buffer); fclose( input ); return(-1);
              }
              if (MPA_debug>1) printf(" \"%s\"\n", line);
            }
            break;
          case 1:
            if (readdata) {
              mpadata->data1 = (long*) malloc ( sizeof(long)*sectionlen );
              if (!mpadata->data1) return(-1);
            } else mpadata->data1 = (long*) NULL;
            mpadata->data1len = sectionlen;
            for (i=0;(i<sectionlen)&&(!feof(input));i++) {
              if (mpadata->data1) mpadata->data1[i]=strtol(line,&pe,10);
              if (MPA_debug>1) 
                printf("(%s): \"%s\" %d: fgets", funcnam, filename, section);
              fgets (line, MPA_MAXLEN, input);
              if (ferror(input)) {
                sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
                perror(buffer); fclose( input ); return(-1);
              }
              if (MPA_debug>1) printf(" \"%s\"\n", line);
            }
            break;
          default:
            printf("WARNING (%s): \"%s\": skip data of section %s.\n",
              funcnam, filename, sectionkey );
            for (i=0;(i<sectionlen)&&(!feof(input));i++) {
              if (MPA_debug>1) 
                printf("(%s): \"%s\" %d: fgets", funcnam, filename, section);

              fgets (line, MPA_MAXLEN, input);
              if (ferror(input)) {
                sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
                perror(buffer); fclose( input ); return(-1);
              }
              if (MPA_debug>1) printf(" \"%s\"\n", line);
            }
        }
        break;

      case MpaCDat:
        switch (sectionnum) {
          case 0:
            if (readdata) {
              mpadata->cdat0 = (long*) malloc ( sizeof(long)*sectionlen );
              if (!mpadata->cdat0) return(-1);
            } else mpadata->cdat0 = (long*) NULL;
            mpadata->cdat0len = sectionlen;
            for (i=0;(i<sectionlen)&&(!feof(input));i++) {
              if (mpadata->cdat0) mpadata->cdat0[i]=strtol(line,&pe,10);
              if (MPA_debug>1) 
                printf("(%s): \"%s\" %d: fgets", funcnam, filename, section);
              fgets (line, MPA_MAXLEN, input);
              if (ferror(input)) {
                sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
                perror(buffer); fclose( input ); return(-1);
              }
              if (MPA_debug>1) printf(" \"%s\"\n", line);
            }
            break;
          default:
            printf("WARNING (%s): \"%s\": skip data of section %s.\n",
              funcnam, filename, sectionkey );
            for (i=0;(i<sectionlen)&&(!feof(input));i++) {
              if (MPA_debug>1) 
                printf("(%s): \"%s\" %d: fgets", funcnam, filename, section);
              fgets (line, MPA_MAXLEN, input);
              if (ferror(input)) {
                sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
                perror(buffer); fclose( input ); return(-1);
              }
              if (MPA_debug>1) printf(" \"%s\"\n", line);
            }
        }

        break;

      default: // error
        sprintf(buffer,"ERROR (%s): \"%s\" undefined section key %s", 
          funcnam, filename, sectionkey);
        fclose( input ); return(-1);

    } // switch

    /* --- read new section key and value [<key>,<len>]<value> */
    if (!(feof(input))&&(line[0]==MPA_SECTION_START)) {
      if (MPA_debug>0) {
        printf("(%s): \"%s\": new section key found: %s\n", 
          funcnam, filename, line);
      }

      rmeoln_mpa ( line );
      key = ltrim_mpa( line+1 ); // skip MPA_SECTION_START and white spaces
      value = strchr(key,MPA_SECTION_END); 
      if (value) *value++ = '\0'; // split into key and value
  
      // does key contain a length?
      len = strchr(key,MPA_SEPARATOR);
      if (len) *len++ = '\0'; // split into key and length
  
      strncpy(sectionkey,key,MIN(strlen(key)+1,MPA_MAXLEN));
  
      section = str2section( sectionkey );
      if (!section) {
        fprintf(stderr,"ERROR (%s): \"%s\": undefined %s.\n",
         funcnam, filename, sectionkey );

        return(-1);
      }
      sectionnum = str2sectionnum ( sectionkey );
      if (len) {if (*len!='\0') sectionlen = strtol(len,&pe,10);}
      else sectionlen=0;

      if (value) {
        strncpy(sectionval,value,MIN(strlen(value)+1,MPA_MAXLEN));
        if ( !(edf_add_header_element ( mpadata->header_key,
          sectionkey, sectionval, &errorvalue, &status )) ) {
          fprintf(stderr,"ERROR (%s): \"%s\": %s.\n",
          funcnam, filename, edf_report_data_error( errorvalue ));
          fclose( input ); return(-1);
        }
      } else sectionkey[0]='\0';

      if (MPA_debug>0) 
        printf("(%s): \"%s\" %d: fgets", funcnam, filename, section);
      fgets (line, MPA_MAXLEN, input);
        if (ferror(input)) {
          sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
          perror(buffer); fclose( input ); return(-1);
        }
      if (MPA_debug>0) printf(" \"%s\"\n", line);
    }

  } // while !eof

  fclose( input );

  return(0);

} // mpa_Read

/*+++--------------------------------------------------------------------------
NAME

  mpa_debug --- set / reset module into debug mode

SYNOPSIS

  void mpa_debug ( int debug );

----------------------------------------------------------------------------+*/
void mpa_debug ( int debug )
{ MPA_debug = debug;
} /* mpa_debug */

