/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/****************************************************************************

 File:      	GetCCDFrame.c

 Description:	This is a sub-program to extract given frame from a large file.

                int ReadCCDData (FILE *stream, int FrameNumber, void **Start)
                int GetMaxCCDNumber (FILE *stream, int *Maximum)
                int ReadCCDType (FILE *stream, int *X, int *Y, int *DataLength,
                                  int * OffsetX, int * OffsetY)
                int ReadCCDScalars (FILE *stream,
                           char **HmDldVersion,
                           float Scalars[HScalerLen],
                           char *ScalarName[HScalerLen],
                           float ScalarZero[HScalerLen],
                           float ScalarCalib[HScalerLen],
                           float *ScalarDepth,
                           int *ScalarI0, int *ScalarI1,
                           int *ScalarAnode, int *ScalarTime,
                           float *HmDeltaTime,
                           char *HmStartTime,
                           int FrameNumber)
                ReadCCDInfo(FILE * stream,
                 char * DetectorType[],
                 char * MachineInfo[],
                 char * OpticsInfo[],
                 char * StationInfo[],
                 char * ProposalInfo),
                ReadCCDExperiment(FILE * stream,
                 char * Title[], char * ExperimentInfo[],
                 float * Center_1, float * Center_2, int * ICenter,
                 float * PixSize_1, float * PixSize_2, int * IPixSize,
                 float * WaveLength, int * IWaveLength,
                 float * SampleDistance, int * ISampleDistance,
	         float * DetectorPosition, int * IDetectorPosition,
                 float * DataValueOffset, int * IDataValueOffset,
                 float * Dummy, int * IDummy);

                All return 0 for success or a positive integer for some error.

                There are two separate file to look at to get information
                about the image. These files are called xxx and xxxhm, where
                xxx is your file name.

                The first file contains information about scalar data, the
                image size, the pixel length and the number of images in the
                other file xxxhm. 

                The file xxxhm contains the images in sequence and a 512
                byte header. We skip this header to get at the actual data.

                For the first file the full description is as follows:

                Version 0.0;
                General Header: 3, run number and separator (-1).
                Version 1.0;    
                General Header: 4, run number, version number and separator(-1)

                Scalar header:  Length of header (including separator),
                                scalar memory ID (2 for scalar, 3 for
                                                  histogramming memory),
                                first channel,
                                last channel,
                                first scalar,
                                last scalar,
                                two zero for padding,
                                length of scalar data in bytes,
                                separator (-1),
                Scalar data:    should be (last scalar - first scalar) * 
                                          (last channel - first channel)
                                separator (-1),
		Version 2.1
                Scalar header:  Length of header (including separator),
                                scalar memory ID (2 for scalar, 3 for
                                                  histogramming memory),
                                first channel,
                                last channel,
                                first scalar,
                                last scalar,
                                two zero for padding,
                                length of scalar data in bytes,
                                separator (-1),
                Scalar data:    Length of scalar data block (including sep.),
                                scalar data ID (5 for scalar data)
                                (last scalar - first scalar) *
                                  (last channel - first channel) lines of data
                                separator (-1),

                Version 0.0
                Image header:   length of header (including separator),
                                3 (presumably signifying hist. memory ???),
                                data length in bits (16 or 32),
                                x size,
                                y size,
                                number of images,
                                two zero for padding,
                                size of each image in bytes,
                                separator (-1)
                Version 1.0
                Image header:   length of header (including separator),
                                3 (presumably signifying hist. memory ???),
                                data length in bits (16 or 32),
                                x size,
                                y size,
                                number of images,
                                two zero for padding,
                                size of each image in bytes,
                                x offset
                                y offset
                                separator (-1)
                Version 1.0
                Timing Header:  length of header (including separator),
                                "TIMING"
				DLD_HMSTARTTIME
				for (i=first_frameno;i<=last_frameno;i++) {
				  DLD_HMDELTATIME
				  }
                                separator (-1)

                Version 1.0 and 1.1
                Scaler Calibration Header:
                                length of header (including separator),
				"SCALERCALIB"
				DLD_SCALER_DEPTH
				DLD_SCALER_I0
				DLD_SCALER_I1
				DLD_SCALER_ANODE
				DLD_SCALER_TIME
				first_scalerno
				last_scalerno
				for (i=first_scalerno-1;i<last_scalerno;i++) {	
				  DLD_SCALER_NAME
				  DLD_SCALER_ZERO
				  DLD_SCALER_CALIB
				  }
				DLD_SCALER_I0S
				DLD_SCALER_I1S
				DLD_SCALER_ANODES
				DLD_SCALER_TIMES
                                separator (-1)

                Version 1.0
                Experiment Header:
                                length of header (including separator),
				"EXPERIMENT"
				DLD_CENTER_1
				DLD_CENTER_2
				DLD_PIXSIZE_1
				DLD_PIXSIZE_2
				DLD_WAVELENGTH
				DLD_SAMPLEDISTANCE
				DLD_TITLE
				DLD_ExperimentInfo
                                separator (-1)
                Version 2.2 
                Experiment Header:
                                length of header (including separator),
                                "EXPERIMENT"
                                DLD_CENTER_1
                                DLD_CENTER_2
                                DLD_PIXSIZE_1
                                DLD_PIXSIZE_2
                                DLD_WAVELENGTH
                                DLD_SAMPLEDISTANCE
                                DLD_TITLE
                                DLD_ExperimentInfo
                                DLD_DETECTORPOSITION
                                separator (-1)
                Version 2.3
                Experiment Header:
                                length of header (including separator),
                                "EXPERIMENT"
                                DLD_CENTER_1
                                DLD_CENTER_2
                                DLD_PIXSIZE_1
                                DLD_PIXSIZE_2
                                DLD_WAVELENGTH
                                DLD_SAMPLEDISTANCE
                                DLD_TITLE
                                DLD_ExperimentInfo
                                DLD_DETECTORPOSITION
                                DataValueOffset
                                separator (-1)
                Version 2.4
                Experiment Header:
                                length of header (including separator),
                                "EXPERIMENT"
                                DLD_CENTER_1
                                DLD_CENTER_2
                                DLD_PIXSIZE_1
                                DLD_PIXSIZE_2
                                DLD_WAVELENGTH
                                DLD_SAMPLEDISTANCE
                                DLD_TITLE
                                DLD_ExperimentInfo
                                DLD_DETECTORPOSITION
                                DataValueOffset
                                DUMMY
                                separator (-1)
                Version 1.0
                Info Header:
                                length of header (including separator),
				"INFO"
				DLD_DETECTORTYPE
				DLD_MACHINEINFO
				DLD_OPTICSINFO
				DLD_STATIONINFO
				DLD_PROPOSALINFO
                                separator (-1)

		Histogramming Memory File (HM)
		for version < 2.0 512 bytes header

                Version 2.0
		Header file not changed but data file has no header any more.


 Compatibility:	(PB) The header file of BL1 does not contain a scalar data 
                block. There is no straightforward way to distinguish both 
		header automatically. For the time beeing a second version 
		of this transformation program is required which skips this 
		block. -->
		Remove corresponding line in ParseScalarHeaderFile or add
                manually an additional separator (-1) in the header file 
                after the scalar header block. 
		PB 20-May-1995

 Author(s): 	Peter J. Daly (GetFrame.c)
 Modified:	17-Feb-1995 Peter Boesecke (change of names)
                19-Feb-1995 PB (expanded)
                19-May-1995 PB V1.0 100 -> BufLen
                09-Jun-1995 PB V1.1
		28-Oct-1995 PB V1.1 ScalarDepth -> float, Scaler[] -> float
		08-Dec-1995 PB V2.0 binary file without any header
                               ReadCCDDataV1 to read old files
		20-Jan-1995 PB V2.1 scalar data block modified
		21-Jan-1995 PB V2.2 DetectorPosition in experiment block
		20-Jul-1997 PB V2.3 renamed for CCD, skips esrf header
		22-Jul-1997 PB V2.31 skipping works now also with padded
                                     spaces
                30-Apr-1999 PB V2.32 DetectorType is now an allocated string
                                     that can be free'd without seg fault.
                16-Jun-1999 PB V2.2 Empty scaler calibration cells filled
                                    with dummies
                23-Feb-2000 PB V2.21 Read edf header with TmpLen 1024,
                                     otherwise problems with padded
                                     ccd files.
                05-Sep-2000 PB V2.4 Scaler file version 2.3 
                                    with DataValueOffset
                07-Sep-2000 PB V2.5 Scaler file version 2.4
                                    with Dummy
                08-Aug-2001 PB V2.6 EDF file can also start with \n{\r\n
                24-Feb-2005 PB V2.7 stdlib.h included
                19-Apr-2007 PB V2.8 string.h included, 
                                    -Wall compiler warnings resolved
                23-Apr-2007 PB V2.9 GetCCDFrame.h error in parameter list:
                                    type of IDataValueOffset was undefined,
                                    unused h-- removed

 Original:   	February 1995

****************************************************************************/

/***************************************************************************
 * Include Files                                                           *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "GetCCDFrame.h"       

/***************************************************************************
 * Defines                                                                 *
 ***************************************************************************/

#define IMAGE_HEADERV1 512L /* image file has IMAGE_HEADER bytes before data */
#define IMAGE_HEADER   0L /* Version > V2 image files have no header bytes */

#define ATOI(s) (int) strtol (s, (char **) NULL, 10)
#define ATOF(s) strtod (s, (char **) NULL)

#define PRIVATE static        /* used to declare variables of private type */
#define PUBLIC                 /* used to declare variables of public type */

#define DLD_SCANERROR  555

#define BufLen 128

#ifndef True
# define True 1
#endif
#ifndef False
# define False 0
#endif

/***************************************************************************
 * Global Initialised Variables                                            *
 ***************************************************************************/

PRIVATE int DataLen = 0, Xsize = 0, Ysize = 0, NumberOfFrames = 0;
PRIVATE int XOff = 0, YOff = 0;
PRIVATE char DldVersion[BufLen]; 
PRIVATE char HScalarNameBuffer[HScalerLen*HScalerNameLen];
PRIVATE char * CCDDetector = "X-ray Image Intensifier with FRELON CCD\n";

/***************************************************************************

 @(#) $Header: parse header in the SPEC generated file                     $

 ***************************************************************************/

PRIVATE int ParseScalarHeaderFile (FILE *stream)
{
  int i;
  char str[BufLen]; char * pbuf;

  if (stream == (FILE *) NULL) return 1;               /* no file open */

  if (fseek (stream, 0L, SEEK_SET)) return 2;          /* can't seek start */

                                        /* read version from global header */
  if (( (i = ATOI (fgets (str, BufLen, stream))) )!=-1) ; /* length of header */
  if (( (i = ATOI (fgets (str, BufLen, stream))) )!=-1) ;       /* run number */
  if (( (i = ATOI (fgets (str, BufLen, stream))) )!=-1) {     
    pbuf=DldVersion; (void) strcpy(DldVersion,str);             /* DldVersion */
    while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0';      /* remove CRs */
    while ((i = ATOI (fgets (str, BufLen, stream))) != -1) ;     /* skip rest */
    } else { 
    strcpy(DldVersion,"0.0");
    }
  while ((i = ATOI (fgets (str, BufLen, stream))) != -1) ;   /* scalar header */
  while ((i = ATOI (fgets (str, BufLen, stream))) != -1) ;     /* scalar data */

  i = ATOI (fgets (str, BufLen, stream));                 /* length of header */
  i = ATOI (fgets (str, BufLen, stream));             /* histogramming memory */
  DataLen = ATOI (fgets (str, BufLen, stream)) / 8;            /* data length */
  Xsize = ATOI (fgets (str, BufLen, stream));                       /* x size */
  Ysize = ATOI (fgets (str, BufLen, stream));                       /* y size */
  NumberOfFrames = ATOI (fgets (str, BufLen, stream));    /* number of images */
  i = ATOI (fgets (str, BufLen, stream));                             /* zero */
  i = ATOI (fgets (str, BufLen, stream));                             /* zero */
  i = ATOI (fgets (str, BufLen, stream));                       /* image size */
  if (strcmp(DldVersion,"1.0")>=0) {
     XOff = ATOI (fgets (str, BufLen, stream));                    /* XOffset */ 
     YOff = ATOI (fgets (str, BufLen, stream));                    /* YOffset */ 
     }
  i = ATOI (fgets (str, BufLen, stream));                        /* separator */
     
  return 0;
}

/***************************************************************************

 @(#) $Header: Read the image given by the FrameNumber and set the pointer $
 @(#) $Header: to point at the read data (dld header versions < V2.0)      $

 ***************************************************************************/

PUBLIC int ReadCCDDataV1 (FILE *stream, int FrameNumber, void **Start)
{
  int Read = 0, BufferSize = 0;

  if (stream == (FILE *) NULL) return 1;                   /* no file open */

  if (fseek (stream, IMAGE_HEADERV1, SEEK_SET)) return 2;/* can't seek start */

  BufferSize = DataLen * Xsize * Ysize;
  if (BufferSize <= 0) {
    printf("BAD SIZE : (Header %s) DataLen * Xsize * Ysize<=0\n",
            DldVersion);
    return 3;                               /* bad size */
    }

  if ((*Start = (void *) malloc (BufferSize)) == (void *) NULL)
    return 4;                                              /* can't malloc */

  if (FrameNumber > 1 &&
      fseek (stream, (long) ((FrameNumber - 1) * BufferSize), SEEK_CUR))
    return 2;                                                /* can't seek */

  Read = fread (*Start, 1, BufferSize, stream);
  if (Read != BufferSize)  return 5;                           /* bad read */

  return 0;
}

/***************************************************************************

 @(#) $Header: Read the image given by the FrameNumber and set the pointer $
 @(#) $Header: to point at the read data                                   $

***************************************************************************/

PUBLIC int ReadCCDData (FILE *stream, int FrameNumber, void **Start)
{
# define TmpLen 1024
  char str[TmpLen]; 
  char *ps;
  int Read = 0, BufferSize = 0;

  if (stream == (FILE *) NULL) return 1;                   /* no file open */

  /* Skip esrf data format header 
     The header must start with \n'{'\n and end with [' ']{\n .
     The data starts in the next byte after \n.
     If the first byte of the file is not equal to \n, it is assumed
     that no header follows, and the file pointer is placed at the start */
 
  /* seek start */
  if (fseek (stream, IMAGE_HEADER, SEEK_SET)) return 2;/* can't seek start */
  (void) fgets (str, TmpLen, stream);
  if (*str=='\n') (void) fgets (str, TmpLen, stream);
  if ( (strcmp(str,"{\n")==0)||(strcmp(str,"{\r\n")==0) ) { 
    /* skip esrf header */
    ps=str;
    while (strcmp(ps,"}\n")!=0) {
      (void) fgets (str, TmpLen, stream);
      /* skip spaces */
      ps=str;
      while (*ps==' ') ps++;
      } /* while */
    } else {
    /* seek start */
    printf("WARNING: FIRST CHARACTERS NOT [\\n]'{'[\\r]\\n, NO ESRF HEADER\n");
    if (fseek (stream, IMAGE_HEADER, SEEK_SET)) return 2;/* can't seek start */
    } /* if esrf header */

  /* Skip esrf data end */

  BufferSize = DataLen * Xsize * Ysize;
  if (BufferSize <= 0) {
    printf("BAD SIZE : (Header %s) DataLen * Xsize * Ysize<=0\n",
            DldVersion);
    return 3;                               /* bad size */
    }
  if ((*Start = (void *) malloc (BufferSize)) == (void *) NULL)
    return 4;                                              /* can't malloc */

  if (FrameNumber > 1 &&
      fseek (stream, (long) ((FrameNumber - 1) * BufferSize), SEEK_CUR))
    return 2;                                                /* can't seek */

  Read = fread (*Start, 1, BufferSize, stream);
  if (Read != BufferSize)  return 5;                           /* bad read */

  return 0;
}

/***************************************************************************

 @(#) $Header: get the number of images in the file                        $

 ***************************************************************************/

PUBLIC int GetMaxCCDNumber (FILE *stream, int *Maximum)
{
  if (! ParseScalarHeaderFile (stream)) {
    *Maximum = NumberOfFrames;
    return 0;
  } else {
    return 1;
  }
}

/***************************************************************************

 @(#) $Header: return the x and y dimensions of the image ,                $
 @(#) $Header: the x and y offset and the number                           $
 @(#) $Header: of bytes for each pixel (2 or 4)                            $

 ***************************************************************************/

PUBLIC int ReadCCDType (FILE *stream, int *X, int *Y, int *DataLength,
                         int * OffsetX, int * OffsetY)
{
  if (! ParseScalarHeaderFile (stream)) {
    *X = Xsize;
    *Y = Ysize;
    *OffsetX = XOff;
    *OffsetY = YOff;
    *DataLength = DataLen;
    return 0;
  } else {
    return 1;
  }
}
/***************************************************************************

 @(#) $Header: return various texts about the setup (as char **)           $
 @(#) $Header: DetectorType, MachineInfo, OpticsInfo, StationInfo,         $ 
 @(#) $Header: FramingInfo                                                 $
 @(#) $Header: allocated memory must be released                           $

 ***************************************************************************/

int ReadCCDInfo(FILE * stream,
         char ** DetectorType,
         char ** MachineInfo,
         char ** OpticsInfo,
         char ** StationInfo,
         char ** ProposalInfo)

{  int i,h;
   char str[BufLen]; char * pbuf;
   
   *DetectorType = (char *) NULL;
   *MachineInfo  = (char *) NULL; 
   *OpticsInfo   = (char *) NULL;
   *StationInfo  = (char *) NULL;
   *ProposalInfo = (char *) NULL;

  if (ParseScalarHeaderFile (stream)) return 1;

  /* ==================  Version 1.0 ================== */

  if (strcmp(DldVersion,"1.0")<0) return 0; /* stop here if too old version */

  /* --- timing header */
  h = ATOI (fgets (str, BufLen, stream));               /* length of header */
  for (i=1;i<h;i++) (void) fgets(str,BufLen,stream);      /* skip all lines */

  /* --- scaler calibration header */
  h = ATOI (fgets (str, BufLen, stream));               /* length of header */
  for (i=1;i<h;i++) (void) fgets(str,BufLen,stream);      /* skip all lines */

  /* --- experiment header */
  h = ATOI (fgets (str, BufLen, stream));               /* length of header */
  for (i=1;i<h;i++) (void) fgets(str,BufLen,stream);      /* skip all lines */

  /* --- info header */
   h = ATOI (fgets (str, BufLen, stream));               /* length of header */
  (void) fgets (str, BufLen, stream);h--;                              /* Id */
  if (strcmp(str,"INFO\n")!=0) {
    printf("SCANERROR : Header version %s (%s)\n",DldVersion,"INFO");
    exit(-1); }

/* --- replaced by fixed string 
  *DetectorType=(char *) malloc(strlen(fgets(str,BufLen,stream))+1); h--;
  pbuf=*DetectorType; (void) strcpy(pbuf,str);
  while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0'; 
*/

  (void) fgets(str,BufLen,stream);  /* skip this line */
  *DetectorType=(char *) malloc(strlen(CCDDetector)+1);h--;
  pbuf=*DetectorType; (void) strcpy(pbuf,CCDDetector);
  while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0';

  *MachineInfo=(char *) malloc(strlen(fgets(str,BufLen,stream))+1);h--; 
                                                             /* MachineInfo */
  pbuf=*MachineInfo; (void) strcpy(pbuf,str);
  while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0';

  *OpticsInfo=(char *) malloc(strlen(fgets(str,BufLen,stream))+1);h--; 
                                                              /* OpticsInfo */
  pbuf=*OpticsInfo; (void) strcpy(pbuf,str);
  while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0'; 

  *StationInfo=(char *) malloc(strlen(fgets(str,BufLen,stream))+1);h--; 
                                                             /* StationInfo */
  pbuf=*StationInfo; (void) strcpy(pbuf,str);
  while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0'; 

  *ProposalInfo=(char *) malloc(strlen(fgets(str,BufLen,stream))+1);h--; 
                                                           /* ProposalInfo */
  pbuf=*ProposalInfo; (void) strcpy(pbuf,str);
  while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0'; 

   return(0);
}
 /***************************************************************************

 @(#) $Header: return various numbers about the experimental setup:        $
 @(#) $Header: char * Title (title given by the user)                      $
 @(#) $Header: Center_1, Center_2 (position of beam [pixel values]         $
 @(#) $Header: PixSize_1, PixSize_2 (size of of a pixels) [meters]         $
 @(#) $Header: WaveLength (current wavelength) [meters]                    $
 @(#) $Header: SampleDistance (distance from sample to center) [meters]    $ 

 HmStartTime is the time where the the acquisition has been started.
 The actual starting time of frame n is in the simplest case:
     HmStartTime + HmWaitTime + n * (HmWaitTime+HmFrameTime)
 Generally an acquisition can consist of a set of different HmWaitTimes and
 HmFrameTimes. The values for HmWaitTime and HmFrameTime are written into
 FramingInfo. 
 
 ***************************************************************************/

int ReadCCDExperiment(FILE * stream,
         char ** Title, char ** ExperimentInfo,
         float * Center_1, float * Center_2, int * ICenter,
         float * PixSize_1, float * PixSize_2, int * IPixSize,
         float * WaveLength, int * IWaveLength,
         float * SampleDistance, int * ISampleDistance,
	 float * DetectorPosition, int * IDetectorPosition,
         float * DataValueOffset, int * IDataValueOffset,
         float * Dummy, int * IDummy)
{ int i,h;
   char str[BufLen]; char * pbuf;

  *Title = (char *) NULL;
  *ExperimentInfo = (char *) NULL;
  *Center_1 = 0; *Center_2 = 0; *ICenter = False;
  *PixSize_1 = 1.0; *PixSize_2 = 1.0; *IPixSize = False;
  *WaveLength = 1e-10; *IWaveLength = False;
  *SampleDistance = 0.0; *ISampleDistance = False;
  *DetectorPosition = 0.0; *IDetectorPosition = False;
  *DataValueOffset = 0.0; *IDataValueOffset = False;
  *Dummy = 0.0; *IDummy = False;


  if (ParseScalarHeaderFile (stream)) return 1; 

  /* ==================  Version 1.0 ================== */

  if (strcmp(DldVersion,"1.0")<0) return 0; /* stop here if too old version */

  /* --- timing header */
  h = ATOI (fgets (str, BufLen, stream));               /* length of header */
  for (i=1;i<h;i++) (void) fgets(str,BufLen,stream);      /* skip all lines */

  /* --- scaler calibration header */
  h = ATOI (fgets (str, BufLen, stream));               /* length of header */
  for (i=1;i<h;i++) (void) fgets(str,BufLen,stream);      /* skip all lines */

  /* --- experiment header */
   h = ATOI (fgets (str, BufLen, stream));               /* length of header */

  (void) fgets (str, BufLen, stream);h--;                             /* Id */
  if (strcmp(str,"EXPERIMENT\n")!=0) {
     printf("SCANERROR : Header version %s (%s)\n",DldVersion,"EXPERIMENT");
     exit(-1); }

  (void) sscanf (fgets(str,BufLen,stream),"%g",Center_1);h--;     /* Center_1 */
  (void) sscanf (fgets(str,BufLen,stream),"%g",Center_2);h--;     /* Center_2 */
  *ICenter = True;
  (void) sscanf (fgets(str,BufLen,stream),"%g",PixSize_1);h--;   /* PixSize_1 */
  (void) sscanf (fgets(str,BufLen,stream),"%g",PixSize_2);h--;   /* PixSize_2 */
  *IPixSize = True;
  (void) sscanf (fgets(str,BufLen,stream),"%g",WaveLength);h--; /* WaveLength */
  *IWaveLength = True;
  (void) sscanf (fgets(str,BufLen,stream),"%g",SampleDistance);h--;/*SampDist.*/
  *ISampleDistance = True;

  *Title=(char *) malloc(strlen(fgets(str,BufLen,stream))+1);h--; /* Title */
  pbuf=*Title; (void) strcpy(pbuf,str);     
  while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0';    /* remove cr's */

  *ExperimentInfo=
     (char *) malloc(strlen(fgets(str,BufLen,stream))+1);h--;/*ExperimentInfo*/
  pbuf = *ExperimentInfo; (void) strcpy(pbuf,str);
  while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0';    /* remove cr's */

  /* ==================  Version 2.2 ================== */

  if (strcmp(DldVersion,"2.2")<0) return(0); /* stop here if too old version */

  (void) sscanf (fgets(str,BufLen,stream),"%g",DetectorPosition);h--;/*DetPos.*/
  *IDetectorPosition = True;

  /* ==================  Version 2.3 ================== */

  if (strcmp(DldVersion,"2.3")<0) return(0); /* stop here if too old version */

  (void) sscanf (fgets(str,BufLen,stream),"%g",DataValueOffset);h--;
  *IDataValueOffset = True;

  /* ==================  Version 2.4 ================== */
 
  if (strcmp(DldVersion,"2.4")<0) return(0); /* stop here if too old version */
 
  (void) sscanf (fgets(str,BufLen,stream),"%g",Dummy);h--;
  *IDummy = True;

  return(0);
}

/***************************************************************************

 @(#) $Header: Get the scalar data for a particular channel number         $

 ***************************************************************************/

PUBLIC int ReadCCDScalars (FILE *stream, 
                           char **HmDldVersion,
                           float Scalars[HScalerLen],
                           char *ScalarName[HScalerLen],
                           float ScalarZero[HScalerLen],
                           float ScalarCalib[HScalerLen],
                           float *ScalarDepth,
                           int *ScalarI0, int *ScalarI1,
                           int *ScalarAnode, int *ScalarTime,
                           int *ScalarI0S, int *ScalarI1S,
                           int *ScalarAnodeS, int *ScalarTimeS,
                           float *HmDeltaTime, 
                           char *HmStartTime,
                           int FrameNumber)
{
//  const char *TIMEEXAMPLE = "00-Jan-0000 00:00:00.000"; /* example */

  int i, j, k, h, FirstScalar, LastScalar, FirstFrame, LastFrame;
  char str[BufLen]; 
  char * pbuf;

  /* reset all arrays */ 
  for (i=0;i<HScalerLen;i++) {
    Scalars[i] = 0.0; ScalarName[i] = (char *) NULL;
    ScalarZero[i] = 0.0; ScalarCalib[i] = 1.0;
    }
    *HmDldVersion = DldVersion;
    *ScalarDepth = 24.0; 
    *ScalarI0 = *ScalarI1 = *ScalarAnode = *ScalarTime = 0;
    *ScalarI0S = *ScalarI1S = *ScalarAnodeS = *ScalarTimeS = 0;
    *HmDeltaTime = 0.0; strcpy(HmStartTime,""); 

  if (stream == (FILE *) NULL) return 1;                   /* no file open */

  if (fseek (stream, 0L, SEEK_SET)) return 2;          /* can't seek start */

                                        /* read version from global header */
  if (( (i = ATOI (fgets (str, BufLen, stream))) )!=-1) ; /* length of header */
  if (( (i = ATOI (fgets (str, BufLen, stream))) )!=-1) ;       /* run number */
  if (( (i = ATOI (fgets (str, BufLen, stream))) )!=-1) {
    pbuf=DldVersion; (void) strcpy(DldVersion,str);             /* DldVersion */
    while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0';      /* remove CRs */
    while ((i = ATOI (fgets (str, BufLen, stream))) != -1) ;     /* skip rest */
    } else {
    strcpy(DldVersion,"0.0");
    }

  i = ATOI (fgets (str, BufLen, stream));                 /* length of header */
  i = ATOI (fgets (str, BufLen, stream));                      /* scalar data */
  FirstScalar = ATOI (fgets (str, BufLen, stream));           /* first scalar */
  LastScalar = ATOI (fgets (str, BufLen, stream));             /* last scalar */
  FirstFrame = ATOI (fgets (str, BufLen, stream));             /* First Frame */
  LastFrame = ATOI (fgets (str, BufLen, stream));               /* Last Frame */
  i = ATOI (fgets (str, BufLen, stream));                             /* zero */
  i = ATOI (fgets (str, BufLen, stream));                             /* zero */
  i = ATOI (fgets (str, BufLen, stream));                 /* scalar data size */
  i = ATOI (fgets (str, BufLen, stream));                        /* separator */

  if (FrameNumber < FirstFrame || FrameNumber > LastFrame) {
    printf("BAD SIZE : (Header %s) FrameNumber %d out of range (%d .. %d)\n",\
    DldVersion,FrameNumber,FirstFrame,LastFrame);
    return 3;
    }

  if ((LastScalar-FirstScalar+1)>HScalerLen) {
    printf("BAD SIZE : (Header %s)\n",DldVersion);
    printf("LastScalar %d and FirstScalar %d out of range (1..%d)\n",\
            LastScalar, FirstScalar, HScalerLen);
    /* ##### The following lines check only for an error in 1.0 */
    if (strcmp(DldVersion,"1.0")!=0) return 3; /* bad size */
    /* In Version 1.0 of the header file FirstScalar can be incorrect */
    printf("Lengths adjusted, scaler 32 lost (Header %s)\n",DldVersion); 
    i = ATOI (fgets (str, BufLen, stream));           /* read incorrect value */
    FirstScalar = 1; LastScalar = HScalerLen-1; 
    }

  /* ========  Two lines added in Version >= 2.1 ========= */
  if (strcmp(DldVersion,"2.1")>=0) {
    i = ATOI (fgets (str, BufLen, stream));  /* read block length */
    i = ATOI (fgets (str, BufLen, stream));  /* read block type */
    } /* Version >=2.1 */

  for (k = 0, i = FirstScalar; i <= LastScalar; i++) { /* read scalar data */
    for (j = FirstFrame; j <= LastFrame; j++) {
       if (j == FrameNumber) {
         (void) sscanf (fgets(str,BufLen,stream),"%g",&Scalars[k++]);
       } else
         (void) fgets (str, BufLen, stream);
     }
  }

 /* ==================  Version 1.0 ================== */
 if (strcmp(DldVersion,"1.0")<0) return 0; /* stop here if too old version */

 while ((i = ATOI (fgets (str, BufLen, stream))) != -1) ;       /* skip rest */

 /* --- image header */
 h = ATOI (fgets (str, BufLen, stream));                  /* length of header */
 for (i=1;i<h;i++) (void) fgets(str,BufLen,stream);         /* skip all lines */

 /* --- timing header */
  h = ATOI (fgets (str, BufLen, stream));               /* length of header */
  (void) fgets (str, BufLen, stream);h--;                              /* Id */
  if (strcmp(str,"TIMING\n")!=0) {
     printf("SCANERROR : Header version %s (%s)\n",DldVersion,"TIMING");
     exit(-1); }

  pbuf = HmStartTime;
  (void) strncpy(pbuf,fgets(str,BufLen,stream),BufLen-1);h--;  /* HmStartTime */ 
  while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0';    /* remove cr's */

  if (FrameNumber < FirstFrame || FrameNumber > LastFrame) {
    printf("BAD SIZE : (Header %s) FrameNumber %d out of range (%d .. %d)\n",\
    DldVersion,FrameNumber,FirstFrame,LastFrame);
    return 3;
    }
  for (i=FirstFrame;i<FrameNumber;i++) { 
      (void) fgets (str, BufLen, stream);h--; }              /* skip before */
  (void) sscanf (fgets(str,BufLen, stream),"%g",HmDeltaTime);h--;/*HmDeltaTime*/
  for (;h>1;h--) (void) fgets (str, BufLen, stream);          /* skip all lines */
  
 /* scaler calibration header */
  h = ATOI (fgets (str, BufLen, stream));               /* length of header */
  (void) fgets (str, BufLen, stream);h--;                             /* Id */
  if (strcmp(str,"SCALERCALIB\n")!=0) {
     printf("SCANERROR : Header version %s (%s)\n",DldVersion,"SCALERCALIB");
     exit(-1); }
  (void) sscanf (fgets(str,BufLen,stream),"%g",ScalarDepth);h--;
                                                           /* HmScalarDepth */
  *ScalarI0 = ATOI (fgets (str, BufLen, stream));h--;          /* I0 scaler */
  *ScalarI1 = ATOI (fgets (str, BufLen, stream));h--;          /* I1 scaler */ 
  *ScalarAnode = ATOI (fgets (str, BufLen, stream));h--;    /* anode scaler */
  *ScalarTime = ATOI (fgets (str, BufLen, stream));h--;      /* time scaler */
  FirstScalar = ATOI (fgets (str, BufLen, stream));h--;     /* first scalar */
  LastScalar = ATOI (fgets (str, BufLen, stream));h--;       /* last scalar */


  if ((LastScalar-FirstScalar+1)>HScalerLen) {
    printf("BAD SIZE : (Header %s)\n",DldVersion);
    printf("LastScalar %d and FirstScalar %d out of range (1..%d)\n",\
            LastScalar, FirstScalar, HScalerLen);
    /* ##### The following lines check for an error in V 1.0 */
    if (strcmp(DldVersion,"1.0")!=0) return 3; /* bad size */
    /* In Version 1.0 of the header file FirstScalar can be incorrect */
    printf("Lengths adjusted, scaler 32 lost (Header %s)\n",DldVersion); 
    i = ATOI (fgets (str, BufLen, stream));           /* read incorrect value */
    i = ATOI (fgets (str, BufLen, stream));           /* read incorrect value */
    i = ATOI (fgets (str, BufLen, stream));           /* read incorrect value */
    FirstScalar = 1; LastScalar = HScalerLen-1;
    }

  pbuf = HScalarNameBuffer;
  for (k = 0, i = FirstScalar; i <= LastScalar; i++,k++) {/*read scalar cal.*/
    ScalarName[k] = pbuf; 
    (void) strncpy(pbuf,fgets(str,BufLen,stream),HScalerNameLen-1);h--; /*Name*/
    while ((*pbuf) && (*pbuf!='\n')) pbuf++; *pbuf++='\0'; /* remove cr's */
    (void) sscanf (fgets(str,BufLen,stream),"%g",&ScalarZero[k]);h--;   /*Zero*/
    (void) sscanf (fgets(str,BufLen,stream),"%g",&ScalarCalib[k]);h--; /*Calib*/
    }

  for (k=k ;k<HScalerLen;k++) { /* fill with none */
    ScalarName[k] = "none"; /*Name*/
    ScalarZero[k] = 0.0 ; /*Zero*/
    ScalarCalib[k] = 1.0 ; /*Calib*/
    }

 /* ==================  Version 1.1 ================== */
  if (strcmp(DldVersion,"1.1")<0) return 0; /* stop here if too old version */

  *ScalarI0S = ATOI (fgets (str, BufLen, stream));h--;       /* 2nd I0 scaler */
  *ScalarI1S = ATOI (fgets (str, BufLen, stream));h--;       /* 2nd I1 scaler */
  *ScalarAnodeS = ATOI (fgets (str, BufLen, stream));h--; /* 2nd anode scaler */
  *ScalarTimeS = ATOI (fgets (str, BufLen, stream));h--;   /* 2nd time scaler */


//  for (;h>1;h--) (void) fgets (str, BufLen, stream);      /* skip all lines */

  return 0;
}

#ifdef INCLUDE_MAIN
main (int argc, char *argv[])
{
  void *Start;
  FILE *in, *inhm;
  int j, X, Y, OX, OY, Data, Max, Frame;
  float S[HScalerLen];
  char  Name[512];
  char *Number;

  if (argc>1) (void) strcpy(Name,argv[1]); else (void) strcpy(Name,"gasdet001");
  if (argc>2) Number=argv[2]; else Number="1";

  if ((in = fopen (Name, "r")) != (FILE *) NULL) {
    Frame = ATOI (Number);
    j = GetMaxCCDNumber (in, &Max);
    (void) printf ("Returned %d Maximum = %d\n", j, Max);
    j = ReadCCDType (in, &X, &Y, &Data, &OX, &OY);
    (void) printf ("Returned %d X = %d Y = %d Data = %d\n", j, X, Y, Data);
    j = ReadCCDScalars (in, S, Frame);
    (void) printf ("Returned %g Scalars = ", j);
    for (j = 0; j < HScalerLen; j++) (void) printf ("% d ", S[j]);
    (void) strcat(Name,"hm");
    if ((inhm = fopen (Name, "r")) != (FILE *) NULL) {

       j = ReadCCDData(inhm, Max, &Start); 

       printf("\n");
       (void) printf ("Returned %d ReadCCDData ", j);
       free(Start);
       }

    (void) printf ("\n");
  }
}
#endif
