/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
# define MCA_VERSION      "mca : V0.2 Peter Boesecke 2007-04-19"
/*+++------------------------------------------------------------------------
NAME
   mca --- routines to read mca spec files 

SYNOPSIS

   # include mca.h

HISTORY
  2003-04-25 V0.0 Peter Boesecke
  2004-10-01 V0.1 PB
  2007-04-19 V0.2 PB -Wall compiler warnings resolved    

DESCRIPTION

----------------------------------------------------------------------------*/
/******************************************************************************
* Include Files                                                               *
******************************************************************************/

# include "mca.h"
# include "SaxsDefinition.h"

/******************************************************************************
* Private Defines                                                             *
******************************************************************************/

# define MCA_MAXLEN 1024
# define MCA_BUFLEN 1024

/******************************************************************************
* Private Type Defs                                                           *
******************************************************************************/

  enum McaMode { NoMode, SpecHeader, SpecData, SpecMcaData, EndMode };

/******************************************************************************
* Private Variables                                                           *
******************************************************************************/

/******************************************************************************
* Private Constants                                                           *
******************************************************************************/
// PRIVATE const char * filename_default = "data_001.mca"; // unused
PRIVATE const char * mca_header_key = "mca_header";
PRIVATE const char end_of_line[3] = { '\r', '\n', '\0' };

/******************************************************************************
* Routines                                                                    *
******************************************************************************/

/*---------------------------------------------------------------------------
NAME
 
   newstr_mca --- allocate memory and copy a character string into it
 
SYNOPSIS
 
   char * newstr_mca( const char * string );
 
DESCRIPTION
  Allocates strlen(string)+1 bytes of memory and copies string into it.
  In case of success the pointer to the allocated memory is returned. The
  null pointer is returned in case of an error.
 
RETURN VALUE
  Returns the pointer to the allocated string or (char *) NULL in case
  of an error.
---------------------------------------------------------------------------*/
char * newstr_mca( const char * string )
{ char * newstr_mcaing;
 
  if (!(newstr_mcaing = (char *) malloc(strlen(string)+1))) return((char *) NULL);
  (void) strcpy(newstr_mcaing,string);
 
  return( newstr_mcaing );
 
} /* newstr_mca */

/* removes end_of_line from line */
char * rmeoln_mca ( char * line )
{ char * pc = line;
 
 
  if (pc)
    while (*pc) {
      if (strchr( end_of_line, (int) *pc )) *pc='\0';
      pc++;
      }
  return( line );
 
} /* rmeoln_mca */ 

/*---------------------------------------------------------------------------
NAME
 
   ltrim --- return a pointer to the first non-space character 
 
SYNOPSIS
 
   const char * ltrim( const char * string );
 
RETURN VALUE
   Returns the pointer to the first non-space character in string
 
---------------------------------------------------------------------------*/
const char * ltrim( const char * string )
{ 
  if (!string) return( (const char *) NULL);

  while (*string==' ') string++;

  return( string );
  
} // ltrim

/*---------------------------------------------------------------------------
NAME

   ltrim1 --- Remove leading and trailing spaces. The string is modified.

SYNOPSIS

   const char * ltrim1( char * string );

RETURN VALUE
   Returns the pointer to the first non-space character in string and removes
   trailing spaces. The string is modified.

---------------------------------------------------------------------------*/
const char * ltrim1( char * string )
{ char * value;

  if (!string) return( (char *) NULL);

  string = (char *) ltrim( string );
  value = string;

  /* remove trailing spaces */
  while ((*string)&&(*string!=' ')) string++;
  *string='\0';

  return( value );

} // ltrim1

/*---------------------------------------------------------------------------
NAME
 
   strsp_mca --- search the end of the next group of spaces 
 
SYNOPSIS
 
   strsp_mca( char * string );
 
DESCRIPTION
   Returns the pointer to the end of the next group of spaces before a 
   non-space character. If the next non-space character is a back-slash 
   a pointer to the back-slash is returned. If no spaces and no 
   back-slashes are found the NULL-pointer is returned. 
 
RETURN VALUE
   Returns a pointer to the end of the next group of spaces.
---------------------------------------------------------------------------*/
const char * strsp_mca( const char * string )
{ const char * stop, *start;

  if (!string) return( (const char *) NULL );

  start = string;
  stop = strchr(start,' '); // searching next space as delimiter

  if (stop) { // skip next spaces
    while (*stop==' ') stop++;
    if (*stop!='\\') stop--;
  } 
  
  return ( stop );

} // strsp_mca

/*---------------------------------------------------------------------------
NAME
 
   appendvalue --- copy string a to buffer and append strings b and c 
 
SYNOPSIS
 
   char * appendvalue( char * buffer, size_t buflen, 
                       const char * a, const char * b, const char *c );
 
DESCRIPTION
   The strings a, b and c are concatenated and copied to buffer. The size of 
   buffer is buflen. If one of the strings is a null-pointer, it is ignored.
   String b is used as delimiter and is only used if string a is not ignored.
   The return value is a pointer to buffer or NULL in case of an error
 
RETURN VALUE
  Returns the pointer to buffer or (char *) NULL in case of an error.
---------------------------------------------------------------------------*/
char * appendvalue( char * buffer, size_t buflen,
                    const char * a, const char * b, const char *c )
{ long len, len_a = (size_t) NULL, len_b = (size_t) NULL; 
  char * pb;

  if (!buffer) return( buffer );

  pb = buffer;
  len = buflen;

  if ((a)&&(len)) { 
    len_a = strlen(a); strncpy(pb, a, len);
    pb += len_a;
    if (len_a<len) len -= len_a; else len = (size_t) NULL;

    if ((b)&&(len)) { // copy b only, if a was copied
      len_b = strlen(b); strncpy(pb, b, len);
      pb += len_b;
      if (len_b<len) len -= len_b; else len = (size_t) NULL;
    }
  }

  if ((c)&&(len)) strncpy(pb, c, len);

  buffer[buflen-1] = '\0';

  return( buffer );
  
} // appendvalue


/*---------------------------------------------------------------------------
NAME
 
   anotherline --- check, whether last character is a back-slash
 
SYNOPSIS
 
   int anotherline ( const char * line );
 
RETURN VALUE
   Returns 1, if the last character of line is a bask-slash, otherwise 0
 
---------------------------------------------------------------------------*/
int anotherline( const char * line )
{ if (!line) return(0);
  return((line[strlen(line)-1]=='\\')?1:0);
} // anotherline

/*---------------------------------------------------------------------------
NAME
 
   spec_counterread --- read all counters from line 
 
SYNOPSIS
 
   int spec_counterread( McaData * mcadata, int counterno, const char * line,
                         int flag, int * pErrorValue, int * pstatus  );

DESCRIPTION

   Writes subsequently all values of input line to header counter values,
   starting with counter number counterno. The uptdated counter number
   is returned.

   flag==0 : read counter names from line
   flag==1 : read counter values from line
 
RETURN VALUES
 
 >= 0 : successful, the incremented counter number (counterno) is returned
   -1 : error  
---------------------------------------------------------------------------*/
int spec_counterread( McaData * mcadata, int counterno, const char * line,
                      int flag, int * pErrorValue, int * pstatus )
{ char  key[MCA_MAXLEN];
  char  param[MCA_MAXLEN], newvalue[MCA_MAXLEN];
  const char *start, *stop;
  const char *readvalue, *value;
  int   next=1;
  long  len;

  *pErrorValue = RoutineSucceeded;
  *pstatus = status_error;

  if ((!mcadata)||(counterno<0)||(!line)) return(-1);

  start = ltrim(line);

  while (next) {
    stop = strsp_mca(start); // searching next space as delimiter 
    if (!stop) {
      stop = start+strlen(start);
      if (*(stop-1)=='\\') stop-=1;
      next = 0; // no other value in this line
    }

    len = stop-start;

    if ((MCA_MAXLEN)<=len) return(-1);

    if (len>0) {
      strncpy(param,start,len);
      param[len] = '\0';
      value=ltrim1(param);

      if (flag)
        sprintf(key,"%s%s%02u",KHScaler,KHC,counterno++);
      else
        sprintf(key,"%s%s%02u",KHScaler,KHN,counterno++);

      // read back value of existing key
      edf_search_header_element( mcadata->header_key,
        key, &readvalue,  pErrorValue, pstatus ); // do not check for errors

      appendvalue( newvalue, MCA_MAXLEN, readvalue, " ", value);

      if ( !(edf_add_header_element ( mcadata->header_key,
          key, newvalue, pErrorValue, pstatus )) ) {
        return(-1);
      }
    }

    start = stop+1;
  }

  *pErrorValue = RoutineSucceeded;
  *pstatus = status_success;

  return ( counterno );

} // spec_counterread

/*---------------------------------------------------------------------------
NAME
 
   spec_mcaread --- read all mca values from line
 
SYNOPSIS
 
   int spec_mcaread( McaData * mcadata, int mcano, const char * line );
 
DESCRIPTION
 
   Writes subsequently all values of input line to the elements of
   mcadata->data, starting with mca number mcano. The new mcano is returned.
   If the buffer is too short the subsequent values are skipped. 

   The array is numbered from 0 to len-1;
 
RETURN VALUES
 
 >= 0 : successful, the incremented mca number (mcano) is returned
   -1 : error
---------------------------------------------------------------------------*/
int spec_mcaread( McaData * mcadata, int mcano, const char * line )
{ char  value[MCA_MAXLEN];
  const char *start, *stop;
  int   next=1;
  float *data, tmp;
  long len;

  if ((!mcadata)||(mcano<0)||(!line)) return(-1);

  data = mcadata->data;

  if (!data) return(-1);

  start = ltrim(line);
 
  while (next) {
    stop = strsp_mca(start); // searching next space as delimiter
    if (!stop) {
      stop = start+strlen(start);
      if (*(stop-1)=='\\') stop-=1;
      next = 0; // no other value in this line
    }
 
    len = stop-start;

    if ((MCA_MAXLEN)<=len) return(-1);

    if (len>0) {
      strncpy(value,start,len);
      value[len] = '\0';
      if (sscanf(value,"%g", &tmp ) < 1 ) {
        return(-1);
      }
      if (mcadata->buflen>mcano)
        data[mcano] = tmp;
      mcano++;
    } 

    start = stop+1;
  }
  mcadata->len = (mcadata->buflen>mcano)?mcano:mcadata->buflen;
  mcadata->dim = mcano;

  return( mcano );

} // spec_mcaread

/*---------------------------------------------------------------------------
NAME
  mca_New --- initialize mca data
---------------------------------------------------------------------------*/
McaData *  mca_New( long buflen )
{ const char * funcnam = "mca_New"; 
  char buffer[MCA_BUFLEN];
  McaData * mca_data;

  if (!(mca_data = (McaData *) malloc( sizeof(McaData) )))
    return( mca_data );

  // create header
  if ( !(edf_new_header( mca_header_key )) ) {
    fprintf(stderr,"ERROR (%s): Cannot create header.\n", funcnam);
    free( mca_data );
    return( (McaData *) NULL );
  }
    
  mca_data->header_key = newstr_mca( mca_header_key ); 

  if (!(mca_data->data = (float *) malloc( sizeof(float) * buflen ))) {
    sprintf(buffer,"ERROR (%s): mca data buffer",funcnam);
    perror(buffer); 
    free( mca_data );
    return( (McaData *) NULL );
  }

  mca_data->dim    = 0l; 
  mca_data->len    = 0l; 
  mca_data->buflen = buflen; 

  return( mca_data );

} // mca_New

/*---------------------------------------------------------------------------
NAME
  mca_Free --- release mca_data
---------------------------------------------------------------------------*/
int mca_Free( McaData * mca_data )
{ 

  if (mca_data) {
    if (mca_data->header_key) {
      edf_free_header (mca_data->header_key);
      free(mca_data->header_key);
    }
    if (mca_data->data) free(mca_data->data);

    free(mca_data);
  }

  return(0);

} // mca_Free

/*+++------------------------------------------------------------------------
NAME
  mca_Read --- read mca data
 
SYNOPSIS
 
  int mca_Read( McaData * data, const char * filename,
                long scanno, long skip, int readdata ) );
 
DESCRIPTION
 
  Read mca data of scanno. skipping the first skip occurences of scanno.

PARAMETERS

  McaData * data        : output data
  const char * filename : name of spec scan file
  long scanno           : scan number
  long skip             : number of scan numbers to ignore
  int readdata          : 0: read header only, 
                          1: read header and data 
 
RETURN VALUES
 
  0 : successful or eof
 -1 : error
 
----------------------------------------------------------------------------*/
int mca_Read( McaData * mcadata, const char * filename,
              long scanno, long skip, int readdata )
{ const char * funcnam = "mca_Read";
  FILE *input;
  char buffer[MCA_BUFLEN];
  char line[MCA_MAXLEN];
  char newvalue[MCA_MAXLEN];
  char keybuffer[MCA_MAXLEN];
  char * key, * value;
  const char * readvalue;
  char start_marker[MCA_MAXLEN];
  char * end_marker = "#S";
  char * comment_marker = "#";
  char * mca_marker = "@";
  char * spec_key = "L";

  size_t start_len, end_len;
  size_t comment_len, mca_len, spec_len;
  int  mode;
  int  counterno;
  int  mcano=0;

  long k;
  int  status, errorvalue;

  sprintf(start_marker,"#S %lu ",scanno);

  start_len = strlen(start_marker);
  end_len = strlen(end_marker);
  comment_len = strlen(comment_marker);
  mca_len = strlen(mca_marker);
  spec_len = strlen(spec_key);
 
  input = fopen( filename, "r");
  if (!input) { 
    sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
    perror(buffer); return(-1); 
  }

  /* --- read general header until first end_marker */
  mode = SpecHeader;
  if (!(feof(input))) {
    fgets (line, MCA_MAXLEN, input);
    if (ferror(input)) {
      sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
      perror(buffer); fclose( input ); return(-1);
    } 
  }

  while (!((feof(input))||(!strncmp(line,end_marker,end_len)))) {

    if (!strncmp(line,comment_marker,comment_len)) {
      // read header
      rmeoln_mca ( line ); // remove end of line characters
      key = line + comment_len; // skip comment character
 
      value = (char *) strsp_mca(key); // searching first space as delimiter
      if (value) *value++ = '\0'; // split line into key and value
 
      if (value) {
        if ( !(edf_add_header_element ( mcadata->header_key,
            key, value, &errorvalue, &status )) ) {
          fprintf(stderr,"ERROR (%s): \"%s\": %s.\n",
            funcnam, filename, edf_report_data_error( errorvalue ));
          fclose( input ); return(-1);
        }
      }
    }

    fgets (line, MCA_MAXLEN, input);
    if (ferror(input)) {
      sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
      perror(buffer); fclose( input ); return(-1);
    }
  }

  /* --- skip all lines until start_marker */
  for (k=skip;k>=0;k--) {
 
    while (!((feof(input))||(!strncmp(line,start_marker,start_len)))) {
      /*  printf("skipping %s\r\n",rmeoln(line)); */
      fgets (line, MCA_MAXLEN, input);
      if (ferror(input)) {
        sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
        perror(buffer); fclose( input ); return(-1); 
      }
    }

    if (k>0) {
      if (!(feof(input))) {
        fgets (line, MCA_MAXLEN, input);
        if (ferror(input)) {
          sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
          perror(buffer); fclose( input ); return(-1);
        }
      }
    }

  } // for (k=skip;k>=0;k--)

  if (feof(input)) {
    fprintf(stderr,"ERROR (%s): \"%s\": Start marker %s not found (skip=%ld).\n",
      funcnam, filename, start_marker, skip);
    fclose( input ); return(-1); 
  }
 
  /* --- start_marker found, read lines until end_marker */
  mode = SpecHeader;
  fgets (line, MCA_MAXLEN, input);
  if (ferror(input)) { 
    sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
    perror(buffer); fclose( input ); return(-1); 
  }

  while (!((feof(input))||(!strncmp(line,end_marker,end_len)))) {

    rmeoln_mca ( line ); // remove end of line characters

    if (!strncmp(line,comment_marker,comment_len)) {
      // read header
      key = line + comment_len; // skip comment character

      value = (char *) strsp_mca(key); // searching first space as delimiter
      if (value) *value++ = '\0'; // split line into key and value

      if (value) {
        // read back value of existing key
        edf_search_header_element( mcadata->header_key,
            key, &readvalue, &errorvalue, &status ); // do not check for errors

        value = appendvalue( newvalue, MCA_MAXLEN, readvalue, ", ", value);
        if (!value) {
          fprintf(stderr,"ERROR (%s): \"%s\" : Error appending value.\n",
            funcnam, filename );
          fclose( input ); return(-1);
        }

        if ( !(edf_add_header_element ( mcadata->header_key,
            key, value, &errorvalue, &status )) ) {
          fprintf(stderr,"ERROR (%s): \"%s\": %s.\n", 
            funcnam, filename, edf_report_data_error( errorvalue ));
          fclose( input ); return(-1);
        }
      }

      if (!strncmp(key,spec_key,spec_len)) { // spec data follows
        mode = SpecData; // next line has data
        key = appendvalue(keybuffer,MCA_MAXLEN,(char*)NULL,(char*)NULL,"HS32");

        // read counter names from line
        if ((counterno = spec_counterread( mcadata, 1, value , 0,
                           &errorvalue, &status )) <0) {
          fprintf(stderr,"ERROR (%s): \"%s\": %s.\n",
            funcnam, filename, edf_report_data_error( errorvalue ));
          fclose( input ); return(-1);
        }

      }
    } else {
      if (!strncmp(line,mca_marker,mca_len)) {
        mode = SpecMcaData;
        key = line; 
        mcano = 0;

        value = (char*) strsp_mca(key); //searching first space after mca_marker
        if (value) *value++ = '\0'; //split line into key and value

        key = appendvalue(keybuffer,MCA_MAXLEN,(char*)NULL,(char*)NULL,key);

      } else { 
        value = line;
      }

      switch (mode) {
        case SpecData : // append values to keys HS32Cnn

           counterno = 1;

           // read lines until it does not end with back-slash
           if ((counterno = spec_counterread( mcadata, counterno, value , 1,
                              &errorvalue, &status )) <0) { 
             fprintf(stderr,"ERROR (%s): \"%s\": %s.\n",
               funcnam, filename, edf_report_data_error( errorvalue ));
             fclose( input ); return(-1);
           }
           
           while ( anotherline( value ) ) {
             fgets (line, MCA_MAXLEN, input);
             if (ferror(input)) {
               sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
               perror(buffer); fclose( input ); return(-1);
             }
             value = rmeoln_mca ( line );
 
             // read lines until it does not end with back-slash
             if ((counterno = spec_counterread( mcadata, counterno, value , 1,
                                &errorvalue, &status )) <0) {
               fprintf(stderr,"ERROR (%s): \"%s\": %s.\n",
                 funcnam, filename, edf_report_data_error( errorvalue ));
               fclose( input ); return(-1);
             }
 
           } // anotherline
           break ;
        case SpecMcaData : //read values into mcadata->data,->dim,->len,->buflen
           /* read values subsequently from line and read a new line
              until it does not end with back-slash */

           if ((mcano = spec_mcaread( mcadata, mcano, value )) <0) {
             fprintf(stderr,"ERROR (%s): \"%s\": %s.\n",
               funcnam, filename, "Mca data conversion failed");
             fclose( input ); return(-1);
           }

           while ( anotherline( value ) ) { 
             fgets (line, MCA_MAXLEN, input);
             if (ferror(input)) {
               sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
               perror(buffer); fclose( input ); return(-1);
             }
             value = rmeoln_mca ( line );

             if ((mcano = spec_mcaread( mcadata, mcano, value )) <0) {
               fprintf(stderr,"ERROR (%s): \"%s\": %s.\n",
                 funcnam, filename, "Mca data conversion failed");
               fclose( input ); return(-1);
             }

           } // anotherline
           break ;
        default : 
           break ;
      }
    }
 
    fgets (line, MCA_MAXLEN, input);
    if (ferror(input)) { 
      sprintf(buffer,"ERROR (%s): \"%s\"", funcnam, filename);
      perror(buffer); fclose( input ); return(-1); 
    }

  }
 
  fclose( input );
 
  return(0);

} // mca_Read
