# Copyright (c) 2000 Phil Thompson <phil@river-bank.demon.co.uk>
# Copyright (c) 2002 Detlev Offenbach <detlev@die-offenbachs.de>

import os
import os.path
import sys

from qt import *

from Icons import *
from Scanner import Scanner
from SourceViewer import SourceViewer
from Explorer import Explorer
from VariablesViewer import VariablesViewer
from Shell import Shell
from Info import *
from Config import ConfigVarDefaultFilter
from FilterDialog import *
from BreakpointDialog import *
from UnittestDialog import UnittestDialog
from HelpWindow import HelpWindow


class UserInterface(QMainWindow):
    """UserInterface(dbs,prog=None)

    Create the user interface.  dbs is the DebugServer instance.  prog is the
    name of the optional program to debug.

    """
    def __init__(self,dbs,prog=None):
        QMainWindow.__init__(self)

        self.resize(qApp.desktop().size())
        self.dbs = dbs
        self.lastFN = ''

        self.setCaption(Program)
        
        self.setCentralWidget(QWidget(self))
        appWindowLayout = QHBoxLayout(self.centralWidget(),0,6)

        # Connect the signals emitted by the debug-server
        self.connect(dbs,PYSIGNAL('clientGone'),self.handleClientGone)
        self.connect(dbs,PYSIGNAL('clientLine'),self.handleLineChange)
        self.connect(dbs,PYSIGNAL('clientExit'),self.handleExit)
        self.connect(dbs,PYSIGNAL('clientSyntaxError'),self.handleSyntaxError)
        self.connect(dbs,PYSIGNAL('clientException'),self.handleException)
        self.connect(dbs,PYSIGNAL('clientVariables'),self.handleClientVariables)
        
        # Create the workspace now so that we can connect QActions to it.  This
        # is done to match the appearence of Qt Designer.
        self.hSplitter = QSplitter(self.centralWidget(),"hSplitter")
        self.hSplitter.setOrientation(QSplitter.Horizontal)

        self.mdi = QWorkspace(self.hSplitter)

        self.vSplitter = QSplitter(self.hSplitter)
        self.vSplitter.setOrientation(QSplitter.Vertical)

        # The initial user interface is the Explorer and the Shell.
        # The variables viewer are started hidden.
        self.shell = Shell(dbs,self.vSplitter)
        self.shell.show()
        
        self.globalsViewer = VariablesViewer(self.vSplitter, 1, dbs)
        self.globalsViewer.hide()
        self.localsViewer = VariablesViewer(self.vSplitter, 0, dbs)
        self.localsViewer.hide()

        self.explorer = Explorer(self.vSplitter)
        self.connect(self,PYSIGNAL('debuggedProgramChanged'),
                     self.explorer.handleProgramChange)
        self.connect(self.explorer,PYSIGNAL('pythonFile'),
                     self.handlePythonFile)
        self.connect(self.explorer,PYSIGNAL('pythonFileOpen'),
                     self.handleOpen)
        self.connect(self.explorer,PYSIGNAL('designerFile'),
                     self.handleDesigner)
        self.connect(self.explorer,PYSIGNAL('linguistFile'),
                     self.handleLinguist)
        self.connect(self.explorer,PYSIGNAL('unittestOpen'),
                     self.handleUnittest)
        self.connect(self,PYSIGNAL('debuggedProgramChanged'),
                     self.dbs.clearBreakpoints)
        self.explorer.show()
        appWindowLayout.addWidget(self.hSplitter)
        
        # Set the sizes of the splitters
        width = self.width()
        height = self.height()
        hSizes = [2*width/3, width/3]
        self.hSplitter.setSizes(hSizes)
        
        # Generate the variables filter dialog
        self.dbgFilterDialog = FilterDialog(self, 'Filter Dialog', 1)

        # Generate the unittest dialog
        self.unittestDialog = UnittestDialog(None, 1, self)
        self.connect(self.unittestDialog,PYSIGNAL('unittestFile'),
                     self.setFileLine)
        
        # Initialize the actions, menues, toolbar and statusbar
        self.initActions()
        self.initMenus()
        self.initToolbars()
        self.initStatusbar()

        # Initialise the instance variables.
        self.scanners = []
        self.srcWindows = []
        self.currentProg = QFileInfo()
        self.isProg = 0
        self.currentScanner = None
        self.argvHistory = QStringList()
        self.evalHistory = QStringList()
        self.execHistory = QStringList()
        self.lastAction = -1
        self.actions = [self.handleContinue, self.handleStep,\
                        self.handleStepOver, self.handleStepOut,\
                        self.handleStepQuit]
        self.localsVarFilter  = ConfigVarDefaultFilter
        self.globalsVarFilter = ConfigVarDefaultFilter

        if prog is not None:
            self.handleOpen(prog)

    def initActions(self):
        """
        Define the user interface actions.
        """
        self.openAct = QAction(self.trUtf8('Open'),
                QIconSet(IconOpen),
                self.trUtf8('&Open...'),
                QKeySequence(self.trUtf8("CTRL+O","File|Open")),self)
        self.openAct.setStatusTip(self.trUtf8('Open a Python program to debug'))
        self.openAct.setWhatsThis(self.trUtf8(
            """<b>Open a Python program</b>"""
            """<p>You will be asked for the name of a Python program to be debugged."""
            """ Any Python program currently debugged will then be stopped.</p>"""
                            ))
        self.connect(self.openAct,SIGNAL('activated()'),self.handleOpen)
        
        self.closeAct = QAction(self.trUtf8('Close'),
                QIconSet(IconClose),
                self.trUtf8('&Close'),
                QKeySequence(self.trUtf8("CTRL+W","File|Close")),self)
        self.closeAct.setStatusTip(self.trUtf8('Close the current window'))
        self.closeAct.setWhatsThis(self.trUtf8(
            """<b>Close Window</b>"""
            """<p>Close the current window. If the window to be closed"""
            """ is the one of the program being debugged, all windows"""
            """ will be closed.</p>"""
                            ))
        self.connect(self.closeAct,SIGNAL('activated()'),self.handleClose)
        self.closeAct.setEnabled(0)

        self.exitAct = QAction(self.trUtf8('Exit'),
                QIconSet(IconExit),
                self.trUtf8('E&xit'),
                QKeySequence(self.trUtf8("CTRL+X","File|Exit")),self)
        self.exitAct.setStatusTip(self.trUtf8('Exit the debugger'))
        self.exitAct.setWhatsThis(self.trUtf8(
            """<b>Exit the debugger</b>"""
            """<p>Any Python program being debugged will be stopped.</p>"""
                            ))
        self.connect(self.exitAct,SIGNAL('activated()'),
                     qApp,SLOT('closeAllWindows()'))

        self.tileAct = QAction(self.trUtf8('Tile'),
                self.trUtf8('&Tile'),0,self)
        self.tileAct.setStatusTip(self.trUtf8('Tile the windows'))
        self.tileAct.setWhatsThis(self.trUtf8(
            """<b>Tile the windows</b>"""
            """<p>Rearrange and resize the windows so that they are tiled.</p>"""
                                 ))
        self.connect(self.tileAct,SIGNAL('activated()'),self.mdi.tile)

        self.cascadeAct = QAction(self.trUtf8('Cascade'),
                self.trUtf8('&Cascade'),0,self)
        self.cascadeAct.setStatusTip(self.trUtf8('Cascade the windows'))
        self.cascadeAct.setWhatsThis(self.trUtf8(
            """<b>Cascade the windows</b>"""
            """<p>Rearrange and resize the windows so that they are cascaded.</p>"""
                                    ))
        self.connect(self.cascadeAct,SIGNAL('activated()'),self.mdi.cascade)

        self.shellAct = QAction(self.trUtf8('Shell'),
                self.trUtf8('&Shell'),0,self,None,1)
        self.shellAct.setStatusTip(self.trUtf8('Toggle the Shell window'))
        self.shellAct.setWhatsThis(self.trUtf8(
            """<b>Toggle the Shell window</b>"""
            """<p>If the Shell window is hidden then display it. It is displayed"""
            """ then close it.</p>"""
                                  ))
        self.connect(self.shellAct,SIGNAL('activated()'),self.handleShell)

        self.localsAct = QAction(self.trUtf8('Local Variables'),
                self.trUtf8('&Local Variables'),0,self,None,1)
        self.localsAct.setStatusTip(self.trUtf8('Toggle the Local Variables window'))
        self.localsAct.setWhatsThis(self.trUtf8(
            """<b>Toggle the Local Variables window</b>"""
            """<p>If the Local Variables window is hidden then display it."""
            """ If it is displayed then close it.</p>"""
                                  ))
        self.connect(self.localsAct,SIGNAL('activated()'),self.handleLocalsViewer)

        self.globalsAct = QAction(self.trUtf8('Global Variables'),
                self.trUtf8('&Global Variables'),0,self,None,1)
        self.globalsAct.setStatusTip(self.trUtf8('Toggle the Global Variables window'))
        self.globalsAct.setWhatsThis(self.trUtf8(
            """<b>Toggle the Global Variables window</b>"""
            """<p>If the Global Variables window is hidden then display it."""
            """ If it is displayed then close it.</p>"""
                                  ))
        self.connect(self.globalsAct,SIGNAL('activated()'),self.handleGlobalsViewer)

        self.explorerAct = QAction(self.trUtf8('Explorer'),
                self.trUtf8('&Explorer'),0,self,None,1)
        self.explorerAct.setStatusTip(self.trUtf8('Toggle the Explorer window'))
        self.explorerAct.setWhatsThis(self.trUtf8(
            """<b>Toggle the Explorer window</b>"""
            """<p>If the Explorer window is hidden then display it. It is displayed"""
            """ then close it.</p>"""
                                     ))
        self.connect(self.explorerAct,SIGNAL('activated()'),self.handleExplorer)

        self.whatsThisAct = QAction(self.trUtf8('What\'s This?'),
                QIconSet(IconWhatsThis),
                self.trUtf8('What\'s This?'), Qt.SHIFT+Qt.Key_F1,self)
        self.whatsThisAct.setStatusTip(self.trUtf8('Context sensitive help'))
        self.whatsThisAct.setWhatsThis(self.trUtf8(
            """<b>Display context sensitive help</b>"""
            """<p>In What's This? mode, the mouse cursor shows an arrow with a question"""
            """ mark, and you can click on the interface elements to get a short"""
            """ description of what they do and how to use them. In dialogs, this"""
            """ feature can be accessed using the context help button in the"""
            """ titlebar.</p>"""
                                 ))
        self.connect(self.whatsThisAct,SIGNAL('activated()'),self.whatsThis)

        self.helpviewerAct = QAction(self.trUtf8('Helpviewer'),
                self.trUtf8('&Helpviewer...'), Qt.Key_F1,self)
        self.helpviewerAct.setStatusTip(self.trUtf8('Open the helpviewer window'))
        self.helpviewerAct.setWhatsThis(self.trUtf8(
            """<b>Helpviewer</b>"""
            """<p>Display the helpviewer window. This window will show"""
            """ HTML help files. It has the capability to navigate to"""
            """ links, set bookmarks, print the displayed help and some"""
            """ more features.</p>"""
                             ))
        self.connect(self.helpviewerAct,SIGNAL('activated()'),self.handleHelpViewer)
        
        self.aboutAct = QAction(self.trUtf8('About'),
                self.trUtf8('&About'),0,self)
        self.aboutAct.setStatusTip(self.trUtf8('Display information about this software'))
        self.aboutAct.setWhatsThis(self.trUtf8(
            """<b>About %1</b>"""
            """<p>Display some information about this software.</p>"""
                             ).arg(Program))
        self.connect(self.aboutAct,SIGNAL('activated()'),self.handleAbout)

        self.aboutQtAct = QAction(self.trUtf8('About Qt'),
                self.trUtf8('About &Qt'),0,self)
        self.aboutQtAct.setStatusTip(self.trUtf8('Display information about the Qt toolkit'))
        self.aboutQtAct.setWhatsThis(self.trUtf8(
            """<b>About Qt</b>"""
            """<p>Display some information about the Qt toolkit.</p>"""
                               ))
        self.connect(self.aboutQtAct,SIGNAL('activated()'),self.handleAboutQt)

        self.runAct = QAction(self.trUtf8('Run'),
                QIconSet(IconRun),
                self.trUtf8('&Run...'),Qt.Key_F5,self)
        self.runAct.setStatusTip(self.trUtf8('Run the program'))
        self.runAct.setWhatsThis(self.trUtf8(
            """<b>Run</b>"""
            """<p>Set the command line arguments set the current line to be the first"""
            """ executable Python statement of the program.</p>"""
                                ))
        self.connect(self.runAct,SIGNAL('activated()'),self.handleRun)

        self.debugActGrp = QActionGroup(self)

        act = QAction(self.trUtf8('Continue'),
                QIconSet(IconContinue),
                self.trUtf8('&Continue'),Qt.Key_F6,
                self.debugActGrp)
        act.setStatusTip(self.trUtf8('Continue running the program from the current line'))
        act.setWhatsThis(self.trUtf8(
            """<b>Continue</b>"""
            """<p>Continue running the program from the current line. The program will"""
            """ stop when it terminates or when a breakpoint is reached.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'),self.handleContinue)

        act = QAction(self.trUtf8('Single Step'),
                QIconSet(IconStep),
                self.trUtf8('Single &Step'),Qt.Key_F7,
                self.debugActGrp)
        act.setStatusTip(self.trUtf8('Execute a single Python statement'))
        act.setWhatsThis(self.trUtf8(
            """<b>Single Step</b>"""
            """<p>Execute a single Python statement. If the statement"""
            """ is an <tt>import</tt> statement, a class constructor, or a"""
            """ method or function call then control is returned to the debugger at"""
            """ the next statement.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'),self.handleStep)

        act = QAction(self.trUtf8('Step Over'),
                QIconSet(IconStepOver),
                self.trUtf8('Step &Over'),Qt.Key_F8,
                self.debugActGrp)
        act.setStatusTip(self.trUtf8('Execute a single Python statement staying in the current frame'))
        act.setWhatsThis(self.trUtf8(
            """<b>Step Over</b>"""
            """<p>Execute a single Python statement staying in the same frame. If"""
            """ the statement is an <tt>import</tt> statement, a class constructor, or a"""
            """ method or function call then control is returned to the debugger after"""
            """ the statement has completed.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'),self.handleStepOver)

        act = QAction(self.trUtf8('Step Out'),
                QIconSet(IconStepOut),
                self.trUtf8('Step Ou&t'),Qt.Key_F9,
                self.debugActGrp)
        act.setStatusTip(self.trUtf8('Execute Python statements until leaving the current frame'))
        act.setWhatsThis(self.trUtf8(
            """<b>Step Out</b>"""
            """<p>Execute Python statements until leaving the current frame. If"""
            """ the statements are inside an <tt>import</tt> statement, a class constructor, or a"""
            """ method or function call then control is returned to the debugger after"""
            """ the current frame has been left.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'),self.handleStepOut)

        act = QAction(self.trUtf8('Stop'),
                QIconSet(IconStepQuit),
                self.trUtf8('Sto&p'),Qt.Key_F10,
                self.debugActGrp)
        act.setStatusTip(self.trUtf8('Stop debugging'))
        act.setWhatsThis(self.trUtf8(
            """<b>Stop</b>"""
            """<p>Stop the running debugging session.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'),self.handleStepQuit)
        
        self.debugActGrp2 = QActionGroup(self)

        act = QAction(self.trUtf8('Evaluate...'),
                self.trUtf8('&Evaluate...'),
                0,self.debugActGrp2)
        act.setStatusTip(self.trUtf8('Evaluate in current context'))
        act.setWhatsThis(self.trUtf8(
            """<b>Evaluate</b>"""
            """<p>Evaluate an expresion in the current context of the"""
            """ debugged program. The result is displayed in the"""
            """ shell window.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'),self.handleEval)
        
        act = QAction(self.trUtf8('Execute...'),
                self.trUtf8('&Execute...'),
                0,self.debugActGrp2)
        act.setStatusTip(self.trUtf8('Execute a one line statement in the current context'))
        act.setWhatsThis(self.trUtf8(
            """<b>Execute</b>"""
            """<p>Execute a one line statement in the current context"""
            """ of the debugged program.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'),self.handleExec)
        
        self.dbgFilterAct = QAction(self.trUtf8('Variables Filter...'),
                self.trUtf8('&Variables Filter...'), 0, self)
        self.dbgFilterAct.setStatusTip(self.trUtf8('Configure variables filter'))
        self.dbgFilterAct.setWhatsThis(self.trUtf8(
            """<b>Variables Filter</b>"""
            """<p>Configure the variables filter. Only variable types that are not"""
            """ selected are displayed in the global or local variables window"""
            """ during a debugging session.</p>"""
                        ))
        self.connect(self.dbgFilterAct,SIGNAL('activated()'), self.handleConfigFilters)

        self.dbgAllBpActGrp = QActionGroup(self)

        act = QAction(self.trUtf8('File Breakpoints...'),
                self.trUtf8('&File Breakpoints...'), 0, self.dbgAllBpActGrp)
        act.setStatusTip(self.trUtf8('Show breakpoints of the current file'))
        act.setWhatsThis(self.trUtf8(
            """<b>File Breakpoints</b>"""
            """<p>Show all breakpoints of the current file. The dialog allows to"""
            """ delete or change these breakpoints.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'), self.handleAllFileBp)
        
        act = QAction(self.trUtf8('All Breakpoints...'),
                self.trUtf8('&All Breakpoints...'), 0, self.dbgAllBpActGrp)
        act.setStatusTip(self.trUtf8('Show all breakpoints'))
        act.setWhatsThis(self.trUtf8(
            """<b>All Breakpoints</b>"""
            """<p>Show all breakpoints. The dialog allows to"""
            """ delete or change these breakpoints.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'), self.handleAllBp)
        
        self.debugActGrp.setEnabled(0)
        self.debugActGrp2.setEnabled(0)
        self.dbgAllBpActGrp.setEnabled(0)
        self.runAct.setEnabled(0)
        
        self.toolsActGrp = QActionGroup(self)
        
        act = QAction(self.trUtf8('Unittest'), 
                QIconSet(IconUnittest),
                self.trUtf8('&Unittest...'),
                0, self.toolsActGrp)
        act.setStatusTip(self.trUtf8('Start unittest dialog'))
        act.setWhatsThis(self.trUtf8(
            """<b>Unittest</b>"""
            """<p>Perform unit tests. The dialog gives you the"""
            """ ability to select and run a unittest suite. """
            """ The python file implementing the suite must have"""
            """ a function called 'suite()' returning a"""
            """ unittest.TestSuite object.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'), self.handleUnittest)

        self.qtActGrp = QActionGroup(self)
        
        act = QAction(self.trUtf8('Qt-Designer'), 
                QIconSet(IconDesigner),
                self.trUtf8('&Designer...'), 0, self.qtActGrp)
        act.setStatusTip(self.trUtf8('Start the Qt-Designer'))
        act.setWhatsThis(self.trUtf8(
            """<b>Qt-Designer</b>"""
            """<p>Start the Qt-Designer.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'), self.handleDesigner)

        act = QAction(self.trUtf8('Qt-Linguist'), 
                QIconSet(IconLinguist), 
                self.trUtf8('&Linguist...'), 0, self.qtActGrp)
        act.setStatusTip(self.trUtf8('Start the Qt-Linguist'))
        act.setWhatsThis(self.trUtf8(
            """<b>Qt-Linguist</b>"""
            """<p>Start the Qt-Linguist.</p>"""
                        ))
        self.connect(act,SIGNAL('activated()'), self.handleLinguist)

    def initMenus(self):
        """
        Create the menus.
        """
        mb = self.menuBar()

        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&File'),menu)
        self.openAct.addTo(menu)
        self.closeAct.addTo(menu)
        menu.insertSeparator()
        self.exitAct.addTo(menu)

        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&Debug'),menu)
        self.runAct.addTo(menu)
        self.debugActGrp.addTo(menu)
        menu.insertSeparator()
        self.debugActGrp2.addTo(menu)
        menu.insertSeparator()
        self.dbgAllBpActGrp.addTo(menu)
        menu.insertSeparator()
        self.dbgFilterAct.addTo(menu)

        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&Window'),menu)
        self.connect(menu,SIGNAL('aboutToShow()'),self.handleShowWindowMenu)
        self.windowMenu = menu
        
        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&Tools'),menu)
        self.toolsActGrp.addTo(menu)
        menu.insertSeparator()
        self.qtActGrp.addTo(menu)

        mb.insertSeparator()

        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&Help'),menu)
        self.helpviewerAct.addTo(menu)
        self.aboutAct.addTo(menu)
        self.aboutQtAct.addTo(menu)
        menu.insertSeparator()
        self.whatsThisAct.addTo(menu)

    def initToolbars(self):
        """
        Create the toolbar.
        """
        filetb = QToolBar(self)
        debugtb = QToolBar(self)
        toolstb = QToolBar(self)
        helptb = QToolBar(self)

        self.openAct.addTo(filetb)
        self.closeAct.addTo(filetb)
        self.exitAct.addTo(filetb)
        
        self.runAct.addTo(debugtb)
        self.debugActGrp.addTo(debugtb)
        
        self.toolsActGrp.addTo(toolstb)
        toolstb.addSeparator()
        self.qtActGrp.addTo(toolstb)
        
        self.whatsThisAct.addTo(helptb)

    def initStatusbar(self):
        """
        Set up the status bar.
        """
        sb = self.statusBar()

        self.sbFile = QLabel(sb)
        sb.addWidget(self.sbFile,0,1)
        QWhatsThis.add(self.sbFile,self.trUtf8(
            """<p>This part of the status bar displays the name of file containing the"""
            """ Python statement that is about to be executed.</p>"""
            """<p>It is blank if there is no program currently being debugged.</p>"""
                      ))

        self.sbLine = QLabel(sb)
        sb.addWidget(self.sbLine,0,1)
        QWhatsThis.add(self.sbLine,self.trUtf8(
            """<p>This part of the status bar displays the line number of the Python"""
            """ statement that is about to be executed.</p>"""
            """<p>It is blank if there is no program currently being debugged.</p>"""
                      ))

        self.setSbFile()

    def handleAbout(self):
        """
        Private slot to handle the About dialog.
        """
        QMessageBox.information(self,Program,self.trUtf8(
            """<h3> About %1</h3>"""
            """<p>%1 is a graphical debugger for the Python"""
            """ programming language. It is written using the PyQt Python bindings for"""
            """ the Qt GUI toolkit.</p>"""
            """<p>This version is %2.</p>"""
            """<p>For more information see"""
            """ <tt>http://www.riverbankcomputing.co.uk/pyqt/</tt> or"""
            """ <tt>http://www.die-offenbachs.de/detlev</tt>.</p>"""
            """<p>%3</p>"""
                               )
            .arg(Program)
            .arg(Program)
            .arg(Version)
            .arg(Copyright))

    def handleAboutQt(self):
        """
        Private slot to handle the About Qt dialog.
        """
        QMessageBox.aboutQt(self,Program)

    def handleOpen(self,prog=None):
        """
        Public slot to open a new Python program to debug.
        """
        # Get the program name if one wasn't specified.
        if prog is None:
            prog = QFileDialog.getOpenFileName(self.currentProg.absFilePath(),
                        self.trUtf8('Python Files (*.py)'),self)

            if prog.isNull():
                return

        self.currentProg.setFile(prog)
        prog = self.currentProg.absFilePath()

        # Close any existing program.
        self.closeProgram()

        # Open up the new program.
        try:
            sv = self.displayPythonFile(prog)
        except IOError:
            return

        self.setCaption(QString(Program + ' - %1').arg(prog))

        # Signal the change in program.
        self.emit(PYSIGNAL('debuggedProgramChanged'),(self.currentProg,))

        self.runAct.setEnabled(1)
        self.dbgAllBpActGrp.setEnabled(1)
        self.isProg = 1
        self.lastFN = prog
        sv.show()
        self.closeAct.setEnabled(1)

    def closeProgram(self):
        """
        Private method to close the current program.
        """
        if self.isProg:
            for sv in self.srcWindows:
                sv.close()

            self.srcWindows = []
            self.scanners = []
            self.isProg = 0
            self.resetUserInterface()
            self.runAct.setEnabled(0)
            self.debugActGrp.setEnabled(0)
            self.debugActGrp2.setEnabled(0)
            self.dbgAllBpActGrp.setEnabled(0)
            self.closeAct.setEnabled(0)
            self.lastFN = ''
            self.lastAction = -1
            
    def handleClose(self):
        """
        Private method to close the current window.
        """
        aw = self.mdi.activeWindow()
        if QString.compare(aw.scn.getFileName(),self.currentProg.absFilePath()) == 0:
            self.closeProgram()
            self.dbs.clearBreakpoints()
        else:
            aw.close()
            self.scanners.remove(aw.scn)
            self.srcWindows.remove(aw)
            if not len(self.srcWindows):
                self.closeAct.setEnabled(0)

    def handleLineChange(self,fn,line):
        """
        Private method to handle a change to the current line.
        """
        self.setFileLine(fn,line)
        self.getClientVariables()
        self.debugActGrp.setEnabled(1)
        self.debugActGrp2.setEnabled(1)

    def handleExit(self,status):
        """
        Private method to handle the debugged program terminating.
        """
        if self.currentScanner is not None:
            self.currentScanner.highlight()
            self.currentScanner = None

        self.setSbFile()
        self.resetUserInterface()

        QMessageBox.information(self,Program,
            self.trUtf8('<b>%1</b> has terminated with an exit status of %2.')
                .arg(self.currentProg.absFilePath())
                .arg(status))

    def handleSyntaxError(self,status):
        """
        Private method to handle a syntax error in the debugged program
        """
        self.resetUserInterface()
        
        try:
            message, (fn, ln, cn) = eval(status)
        except:
            QMessageBox.critical(self,Program,
                self.trUtf8('The program being debugged contains an unspecified syntax error.'))
            return
            
        self.setFileLine(fn,ln,1)
        QMessageBox.critical(self,Program,
            self.trUtf8('The file <b>%1</b> contains the syntax error <b>%2</b> at line <b>%3</b>.')
                .arg(fn)
                .arg(message)
                .arg(ln))
        
    def handleException(self,status):
        """
        Private method to handle an unhandled exception of the debugged program
        """
        try:
            exclist = eval(status)
        except:
            QMessageBox.critical(self,Program,
                self.trUtf8('An unhandled exception occured. See the shell window for details.'))
            return
            
        exctype, msg, (fn, ln) = exclist[:3]
        self.setFileLine(fn,ln,1)
        QMessageBox.critical(self,Program,
                self.trUtf8('The debugged program raised the exception <b>%1</b><br>"<b>%2</b>"<br>File: <b>%3</b>, Line: <b>%4</b>')
                    .arg(str(exctype))
                    .arg(msg)
                    .arg(fn)
                    .arg(ln))
        
        if self.lastAction != -1:
            self.actions[self.lastAction]()
                    
    def handleClientGone(self,unplanned):
        """
        Private method to handle the disconnection of the debugger client.
        """
        if unplanned:
            self.resetUserInterface()
            QMessageBox.information(self,Program,
                self.trUtf8('The program being debugged has terminated unexpectedly.'))

    def handlePythonFile(self,pyfn):
        """
        Private method to handle the user selecting a Python file with
        Explorer.
        """
        try:
            self.displayPythonFile(pyfn).show()
        except IOError:
            pass

    def displayPythonFile(self,fn):
        """
        Private method to display a file in a Source Viewer.  An exception will
        be raised if the file could not be read.
        """
        scn = self.getScanner(fn)
        # if there is already a viewer than make sure it isn't hidden
        for sv in self.srcWindows:
            if sv.canvas() is scn:
                break
        else:
            sv = SourceViewer(scn,self.mdi)
            self.srcWindows.append(sv)
            self.closeAct.setEnabled(1)

        if self.lastFN != fn:
            self.lastFN = fn
            # QWorkspace is a little bit tricky
            sv.hide()
            sv.show()
            
        return sv

    def handleShowWindowMenu(self):
        """
        Private slot to set up the Window menu.
        """
        self.windowMenu.clear()

        self.tileAct.addTo(self.windowMenu)
        self.cascadeAct.addTo(self.windowMenu)

        self.windowMenu.insertSeparator()

        # Set the rest of the options according to what is being displayed.
        self.shellAct.addTo(self.windowMenu)
        self.shellAct.setOn(not self.shell.isHidden())

        self.localsAct.addTo(self.windowMenu)
        self.localsAct.setOn(not self.localsViewer.isHidden())

        self.globalsAct.addTo(self.windowMenu)
        self.globalsAct.setOn(not self.globalsViewer.isHidden())

        self.explorerAct.addTo(self.windowMenu)
        self.explorerAct.setOn(not self.explorer.isHidden())

        # Now do any Source Viewer windows.
        idx = 0

        for sv in self.srcWindows:
            if idx == 0:
                self.windowMenu.insertSeparator()

            id = self.windowMenu.insertItem(sv.getFileName(),
                                            self.handleSVWindow)
            self.windowMenu.setItemParameter(id,idx)
            self.windowMenu.setItemChecked(id,not sv.isHidden())

            idx = idx + 1

    def handleExplorer(self):
        """
        Private slot to handle the toggle of the Explorer window.
        """
        self.toggleWindow(self.explorer)

    def handleShell(self):
        """
        Private slot to handle the toggle of the Shell window.
        """
        self.toggleWindow(self.shell)

    def handleLocalsViewer(self):
        """
        Private slot to handle the toggle of the Local Variables window.
        """
        self.toggleWindow(self.localsViewer)

    def handleGlobalsViewer(self):
        """
        Private slot to handle the toggle of the Global Variables window.
        """
        self.toggleWindow(self.globalsViewer)

    def handleSVWindow(self,idx):
        """
        Private method to handle to toggle of a Source Viewer window.
        """
        self.toggleWindow(self.srcWindows[idx])

    def toggleWindow(self,w):
        """
        Private method to toggle a workspace window.
        """
        if w.isHidden():
            w.show()
        else:
            w.hide()

    def setFileLine(self,fn,line,error=0):
        """
        Private method to update the user interface when the current program or
        line changes.
        """
        self.setSbFile(fn,line)

        try:
            self.currentScanner = self.getScanner(fn)
        except IOError:
            return

        # Change the highlighted line.
        self.currentScanner.highlight(line,error)

        # If there is already a viewer then make sure it isn't hidden.
        for sv in self.srcWindows:
            if sv.canvas() is self.currentScanner:
                break
        else:
            sv = SourceViewer(self.currentScanner,self.mdi)
            self.srcWindows.append(sv)
            self.closeAct.setEnabled(1)

        if self.lastFN != fn:
            self.lastFN = fn
            sv.hide()
        sv.highlightVisible()

    def setSbFile(self,fn=None,line=None):
        """
        Private method to set the file in the status bar.
        """
        if fn is None:
            line = None
            fn = ''

        self.sbFile.setText(self.trUtf8('File: %1').arg(fn,-50))

        if line is None:
            line = ''

        self.sbLine.setText(self.trUtf8('Line: %1').arg(line,5))

    def handleRun(self):
        """
        Private method to handle the Run action.
        """
        # Get the command line arguments.
        if len(self.argvHistory) > 0:
            curr = 0
        else:
            curr = -1

        argv, ok = QInputDialog.getItem(
                        self.trUtf8('Run'),
                        self.trUtf8('Enter the command line arguments'),
                        self.argvHistory,curr,1,self)

        if ok:
            if argv.isNull():
                argv = QString('')

            # This moves any previous occurence of these arguments to the head
            # of the list.
            self.argvHistory.remove(argv)
            self.argvHistory.prepend(argv)

            # Hide all error highlights
            for scn in self.scanners:
                scn.highlight()
                
            # Show the variable viewers, hide the explorer
            self.globalsViewer.show()
            self.localsViewer.show()
            self.explorer.hide()
                
            # Ask the client to open the new program.
            self.dbs.remoteLoad(self.currentProg,argv)
            
    def handleContinue(self):
        """
        Private method to handle the Continue action.
        """
        self.lastAction=0
        self.enterRemote()
        self.dbs.remoteContinue()

    def handleStep(self):
        """
        Private method to handle the Step action.
        """
        self.lastAction=1
        self.enterRemote()
        self.dbs.remoteStep()

    def handleStepOver(self):
        """
        Private method to handle the Step Over action.
        """
        self.lastAction=2
        self.enterRemote()
        self.dbs.remoteStepOver()

    def handleStepOut(self):
        """
        Private method to handle the Step Out action.
        """
        self.lastAction=3
        self.enterRemote()
        self.dbs.remoteStepOut()

    def handleStepQuit(self):
        """
        Private method to handle the Step Quit action.
        """
        self.lastAction=4
        self.enterRemote()
        self.dbs.remoteStepQuit()
        self.resetUserInterface()

    def handleEval(self):
        """
        Private method to handle the Eval action.
        """
        # Get the command line arguments.
        if len(self.evalHistory) > 0:
            curr = 0
        else:
            curr = -1

        arg, ok = QInputDialog.getItem(
                        self.trUtf8('Evaluate'),
                        self.trUtf8('Enter the statement to evaluate'),
                        self.evalHistory,curr,1,self)

        if ok:
            if arg.isNull():
                return

            # This moves any previous occurence of this expression to the head
            # of the list.
            self.evalHistory.remove(arg)
            self.evalHistory.prepend(arg)
            
            self.dbs.remoteEval(arg)
            
    def handleExec(self):
        """
        Private method to handle the Exec action.
        """
        # Get the command line arguments.
        if len(self.execHistory) > 0:
            curr = 0
        else:
            curr = -1

        stmt, ok = QInputDialog.getItem(
                        self.trUtf8('Execute'),
                        self.trUtf8('Enter the statement to execute'),
                        self.execHistory,curr,1,self)

        if ok:
            if stmt.isNull():
                return

            # This moves any previous occurence of this statement to the head
            # of the list.
            self.execHistory.remove(stmt)
            self.execHistory.prepend(stmt)
            
            self.dbs.remoteExec(stmt)
            
    def enterRemote(self):
        """
        Private method to update the user interface prior to executing some of
        the program being debugged.
        """
        # Disable further debug commands from the user.
        self.debugActGrp.setEnabled(0)
        self.debugActGrp2.setEnabled(0)

        if self.currentScanner is not None:
            self.currentScanner.highlight()

    def getScanner(self,fn):
        """
        Private method to return the scanner displaying the given file creating
        a new one if needed.
        """
        for scn in self.scanners:
            if QString.compare(fn,scn.getFileName()) == 0:
                break
        else:
            scn = Scanner(self.dbs,fn)
            self.scanners.append(scn)

        return scn

    def getClientVariables(self):
        """
        Private method to request the global and local variables of the
        debugged program.
        """
        # get globals first
        self.dbs.remoteClientVariables(1, self.globalsVarFilter)
        # the local variables are requested once we have received the globals
        
    def handleClientVariables(self, vars):
        """
        Private method to write the clients variables to the user interface
        """
        vlist = eval(vars)
        scope = vlist[0]
        try:
            vlist = vlist[1:]
        except:
            vlist = []
            
        if scope == 1:
            self.globalsViewer.showVariables(vlist)
            # now get the local variables
            self.dbs.remoteClientVariables(0, self.localsVarFilter)
        elif scope == 0:
            self.localsViewer.showVariables(vlist)
        elif scope == -1:
            vlist = [('None','','')]
            self.localsViewer.showVariables(vlist)
        
    def resetUserInterface(self):
        """
        Private slot to reset the user interface.
        """
        self.globalsViewer.hide()
        self.globalsViewer.clear()
        self.localsViewer.hide()
        self.localsViewer.clear()
        self.explorer.show()
        self.debugActGrp.setEnabled(0)
        self.debugActGrp2.setEnabled(0)
        self.runAct.setEnabled(1)

    def handleConfigFilters(self):
        """
        Private slot for displaying the variables filter configuration dialog.
        """
        result = self.dbgFilterDialog.exec_loop()
        if result == QDialog.Accepted:
            (self.localsVarFilter, self.globalsVarFilter) = \
                self.dbgFilterDialog.getSelection()
        else:
            self.dbgFilterDialog.setSelection(
                self.localsVarFilter, self.globalsVarFilter)

    def handleAllFileBp(self):
        """
        Private slot for displaying the breakpoints of the current file.
        """
        aw = self.mdi.activeWindow()
        if aw:
            BreakpointDialog(self, 'BPDialog', 1, Qt.WDestructiveClose,\
                aw.scn.fileName).exec_loop()
        else:
            QMessageBox.information(self, Program, 
                self.trUtf8('No source window active'), QMessageBox.Ok)
        
    def handleAllBp(self):
        """
        Private slot for displaying the all breakpoints.
        """
        BreakpointDialog(self, 'BPDialog', 1, Qt.WDestructiveClose).exec_loop()

    def handleUnittest(self,prog=None):
        """
        Private slot for displaying the unittest dialog.
        """
        if prog:
            self.unittestDialog.insertProg(prog)
        self.unittestDialog.show() 

    def handleDesigner(self,fn=None):
        """
        Private slot to start the Qt-Designer executable.
        """
        qtdir = os.getenv('QTDIR')
        if qtdir is None:
            QMessageBox.critical(self, 
                self.trUtf8('Qt Problem'),
                self.trUtf8('The environment variable QTDIR is not set.'
                ' Cannot start Qt-Designer.'))
            return
        designer = qtdir + os.sep + 'bin' + os.sep + 'designer'
        if sys.platform == "win32":
            designer = designer + '.exe'
        args = [designer]
        if fn is not None:
            args += [str(fn)]
        os.spawnv(os.P_NOWAIT,designer,args)
        
    def handleLinguist(self,fn=None):
        """
        Private slot to start the Qt-Linguist executable.
        """
        qtdir = os.getenv('QTDIR')
        if qtdir is None:
            QMessageBox.critical(self, 
                self.trUtf8('Qt Problem'),
                self.trUtf8('The environment variable QTDIR is not set.'
                ' Cannot start Qt-Linguist.'))
            return
        linguist = qtdir + os.sep + 'bin' + os.sep + 'linguist'
        if sys.platform == "win32":
            linguist = linguist + '.exe'
        args = [linguist]
        if fn is not None:
            args += [str(fn)]
        os.spawnv(os.P_NOWAIT,linguist,args)

    def handleHelpViewer(self):
        qtdir = os.getenv('QTDIR')
        
        if qtdir is None:
            home = None
        else:
            home = os.path.normpath(qtdir + '/doc/html/index.html')
    
        help = HelpWindow(home, '.', None, 'help viewer')
        help.setCaption(self.trUtf8('Helpviewer'))

        if QApplication.desktop().width() > 400 and \
           QApplication.desktop().height() > 500:
            help.show()
        else:
            help.showMaximized()
