"""
    Transf2DFilter.py
    2D Transformations  
"""    

from PyDVT import __version__,__date__,__author__


from Filter import *
import Numeric



class Transf2DFilter(Filter):
    """
    Filter derived class, implement flip, transpose and rotate operations
    over 2d sources.
    Requires the  following keys from it's source (GetOutputmethod):
        "data": 2D NumPy array
        "xdata":(optional) 1d NumPy array with the x values
        "ydata":(optional) 1d NumPy array with the y values

    If this source is connected to a view with x and y labels, in case of rotation
    the labels have to be changed at application level.
    Example:
        if my_transf in ('Transpose','Rotate90','Rotate270'):
            my_view.SetLabels(x_label="(y)",y_label="(x)")
        else:
            my_view.SetLabels(x_label="(x)",y_label="(y)")
        my_filter.SetTransformation(my_transf,refresh=1)

    """
    def __init__(self,*args,**keys):
        """
        Constructor
        See Filter.__init__
        """
        Filter.__init__(self,*args,**keys)
        self.sizex=self.sizey=0
        self.transformation=None
        
        
    def GetOutput(self):
        """
        See Filter.GetOutput        
        """
        sel=self.GetInput()
        if "data" not in sel.keys(): return {}

        self.sizex=sel["data"].shape[1]
        self.sizey=sel["data"].shape[0]
        if self.transformation=='Transpose' or self.transformation=='Rotate90' or self.transformation=='Rotate270':
            olddata=sel["data"]
            if "xdata" in sel.keys(): oldxdata=sel["xdata"]
            else: oldxdata=None
            if "ydata" in sel.keys(): oldydata=sel["ydata"]
            else: oldydata=None
            ###Doing copy make array contiguous and flat to work
            newdata=Numeric.transpose(olddata).copy()            
            sel["data"],sel["xdata"],sel["ydata"]=newdata,oldydata,oldxdata
        if self.transformation=='FlipY' or self.transformation=='Rotate270':
            if "ydata" in sel.keys() and sel["ydata"] is not None: sel["ydata"]=sel["ydata"][::-1]
            sel["data"]=sel["data"][::-1,:].copy()
        elif self.transformation=='FlipX'  or self.transformation=='Rotate90':
            if "xdata" in sel.keys() and sel["xdata"] is not None: sel["xdata"]=sel["xdata"][::-1]
            sel["data"]=sel["data"][:,::-1].copy()
        elif self.transformation=='FlipXY' or self.transformation=='Rotate180':
            if "xdata" in sel.keys() and sel["xdata"] is not None: sel["xdata"]=sel["xdata"][::-1]
            if "ydata" in sel.keys() and sel["ydata"] is not None: sel["ydata"]=sel["ydata"][::-1]
            sel["data"]=sel["data"][::-1,::-1].copy()
        return sel
        

    def DataCoord2SelectionCoord(self,data_coord):
        """
        Overriden to corret coords convertion acording to the transformation.
        """
        source=self.GetSource()
        if source is None: return (-1,-1)
        coord=source.DataCoord2SelectionCoord(data_coord)
        if self.transformation=='Transpose': return (coord[1],coord[0])
        elif self.transformation=='FlipY': return (coord[0],max(self.sizey-1-coord[1],0))
        elif self.transformation=='FlipX': return (max(self.sizex-1-coord[0],0),coord[1])
        elif self.transformation=='FlipXY' or self.transformation=='Rotate180': return (max(self.sizex-1-coord[0],0),max(self.sizey-1-coord[1],0))
        elif self.transformation=='Rotate90': return (max(self.sizey-1-coord[1],0),coord[0])
        elif self.transformation=='Rotate270': return (coord[1],max(self.sizex-1-coord[0],0))
        else: return coord
            
        


    def SelectionCoord2DataCoord(self,selection_coord):
        """
        Overriden to corret coords convertion acording to the transformation.
        """
        source=self.GetSource()
        if source is None: return DataPosition(-1,(-1,-1))
        if self.transformation=='Transpose': coord=(selection_coord[1],selection_coord[0],)
        elif self.transformation=='FlipY': coord= (selection_coord[0],max(self.sizey-1-selection_coord[1],0))
        elif self.transformation=='FlipX': coord= (max(self.sizex-1-selection_coord[0],0),selection_coord[1])
        elif self.transformation=='FlipXY' or self.transformation=='Rotate180': coord= (max(self.sizex-1-selection_coord[0],0),max(self.sizey-1-selection_coord[1],0))
        elif self.transformation=='Rotate90':  coord= (selection_coord[1],max(self.sizey-1-selection_coord[0],0))
        elif self.transformation=='Rotate270': coord= (max(self.sizex-1-selection_coord[1],0),selection_coord[0]) 
        else: coord=selection_coord
        return source.SelectionCoord2DataCoord(coord)
        


    def SetTransformation(self,transformation=None,refresh=0):
        """
        Sets the 2d transformation
        Parameters:
          transformation:
              None or 'Normal': No transformation
              'Transpose'
              'FlipY'
              'FlipX'
              'FlipXY'
              'Rotate90'
              'Rotate180'
              'Rotate270'
            refresh:
              If non-zero, calls raises refresh event (redraws linked windows)  
        """
        self.transformation=transformation
        if refresh: self.Refresh()

        
        

