#include <stdio.h>      

#include <Python.h>        
#include <Numeric/arrayobject.h>   
#include <sps_lut.h>

static PyObject *SPSLUTError;

PyObject *new_pyimage(const char *mode, unsigned xsize, unsigned ysize, 
		      void *data)
{
  PyObject *outarr;
  return PyString_FromStringAndSize ((const char *)data, 
				     strlen(mode) * xsize * ysize); 
}
   
static int natbyteorder()
{
  union {
    struct {
      unsigned char b1;
      unsigned char b2;
      unsigned char b3;
      unsigned char b4;
    } c;
    unsigned long p;
  } val;

  val.p = 1;
  if (val.c.b4 == 1) {
    return SPS_MSB;
  } else {
    return SPS_LSB;
  }
}

static PyObject *spslut_transform(self, args)
     PyObject *self, *args;
     
{
  char *key, *value;
  void *data;
  int type, cols, rows, reduc, fastreduc, meth, autoscale, mapmin, mapmax;
  int palette_code;
  double gamma, min, max;
  XServer_Info Xservinfo;
  void *palette;
  int prows, pcols, pal_entries;
  int byteorder, pixelsize;
  unsigned long redmask, greenmask, bluemask;
  void *r, *res;
  char *mode;
  PyArrayObject *src;
  PyObject *in_src;

  if (!PyArg_ParseTuple(args, "O(ii)(id)sii(dd)", &in_src, &reduc, 
			&fastreduc, &meth, &gamma, &mode, &palette_code,
			&autoscale, &min, &max)) 
    return NULL;
  
  if (strcmp(mode, "RGB") == 0) {
    Xservinfo.red_mask = 0x0000ff;
    Xservinfo.green_mask = 0x00ff00;
    Xservinfo.blue_mask = 0xff0000;
    Xservinfo.pixel_size = 3;
    Xservinfo.byte_order = natbyteorder();
  } else if (strcmp(mode, "RGBX") == 0) {
    Xservinfo.red_mask = 0x0000ff;
    Xservinfo.green_mask = 0x00ff00;
    Xservinfo.blue_mask = 0xff0000;
    Xservinfo.pixel_size = 4;
    Xservinfo.byte_order = natbyteorder();
  } else if (strcmp(mode, "L") == 0 || strcmp(mode, "P") == 0  ) {
    Xservinfo.pixel_size = 1;
    Xservinfo.byte_order = natbyteorder();
    mapmin = 0;
    mapmax = 255;
  } else {
    PyErr_SetString(SPSLUTError, "Mode must be RGB, RGBX, L or P");
    return NULL;
  }

  if (!(src = (PyArrayObject*) PyArray_ContiguousFromObject(in_src, 
   			    PyArray_NOTYPE, 2, 2))) {
    PyErr_SetString(SPSLUTError, "Input Array could is not a 2x2 array");
    return NULL;
  }

  switch (src->descr->type_num) {
  case PyArray_LONG: 
  case PyArray_INT: 
    type = SPS_LONG; break;
  case PyArray_SHORT: 
    type = SPS_SHORT; break;
  case PyArray_UBYTE: 
    type = SPS_UCHAR; break;
  case PyArray_SBYTE: 
    type = SPS_CHAR; break;
  case PyArray_FLOAT: 
    type = SPS_FLOAT; break;
  case PyArray_DOUBLE: 
    type = SPS_DOUBLE; break;
  default:
    PyErr_SetString(SPSLUTError, "Input Array type not supported");
    return NULL;
  }
  
  data = src->data;
  cols = src->dimensions[0];  /* FIX THIS cols and rows are turned around */
  rows = src->dimensions[1];
  
  r = SPS_PaletteArray (data, type, cols, rows, reduc, fastreduc, meth, gamma, 
			autoscale, mapmin, mapmax, Xservinfo, palette_code,
			&min, &max, &pcols, &prows, &palette, &pal_entries);
  if (r == 0) {
    PyErr_SetString(SPSLUTError, "Error while trying to calculate the image");
    return NULL;
  }
  
  res = Py_BuildValue("(O(i,i)(d,d))",  new_pyimage(mode, (unsigned) pcols, 
		(unsigned) prows, r), pcols, prows, min, max);
  free(r);
  return res;
}

/* The simple palette always returns 4 bytes per entry */
static PyObject *spslut_palette(self, args)
     PyObject *self, *args;
     
{
  int entries, palette_code;
  XServer_Info Xservinfo;
  void *r;
  char *mode;
  PyObject *res;

  if (!PyArg_ParseTuple(args, "ii", &entries, &palette_code)) 
    return NULL;
  
  mode = "RGBX";
  Xservinfo.red_mask = 0x0000ff;
  Xservinfo.green_mask = 0x00ff00;
  Xservinfo.blue_mask = 0xff0000;
  Xservinfo.pixel_size = 4;
  Xservinfo.byte_order = natbyteorder();

  r = SPS_SimplePalette ( 0, entries - 1, Xservinfo, palette_code); 
  
  if (r == 0) {
    PyErr_SetString(SPSLUTError, "Error calculating the palette");
    return NULL;
  }
  
  return new_pyimage(mode, 1, entries, r);
}


                                                                               
static PyMethodDef SPSLUTMethods[] = {
  { "transform", spslut_transform, METH_VARARGS},
  { "palette", spslut_palette, METH_VARARGS},
  { NULL, NULL}
};
  
void initspslut()
{
  PyObject *d, *m;
  /* Create the module and add the functions */
  m = Py_InitModule ("spslut", SPSLUTMethods);
  
  /* Add some symbolic constants to the module */
  d = PyModule_GetDict(m);

  PyDict_SetItemString(d, "LINEAR", PyInt_FromLong(SPS_LINEAR));
  PyDict_SetItemString(d, "LOG", PyInt_FromLong(SPS_LOG));
  PyDict_SetItemString(d, "GAMMA", PyInt_FromLong(SPS_GAMMA));

  PyDict_SetItemString(d, "GREYSCALE", PyInt_FromLong(SPS_GREYSCALE));
  PyDict_SetItemString(d, "TEMP", PyInt_FromLong(SPS_TEMP));
  PyDict_SetItemString(d, "RED", PyInt_FromLong(SPS_RED));
  PyDict_SetItemString(d, "GREEN", PyInt_FromLong(SPS_GREEN));
  PyDict_SetItemString(d, "BLUE", PyInt_FromLong(SPS_BLUE));
  PyDict_SetItemString(d, "REVERSEGREY", PyInt_FromLong(SPS_REVERSEGREY));
  PyDict_SetItemString(d, "MANY", PyInt_FromLong(SPS_MANY));

  SPSLUTError = PyErr_NewException("spslut.error", NULL, NULL);
  PyDict_SetItemString(d, "error", SPSLUTError);
  import_array();
}













