"""
    SpecFileData.py
    Data derived class to access spec files
"""

from PyDVT import __version__,__date__,__author__


################################################################################  
from Data import *
import specfile
import Numeric
import string
################################################################################

SOURCE_TYPE = "SpecFile"   


class SpecFileData(Data):
    """
    Specializes Data class to access Spec files.
    Interface: Data class interface.
    """
    def __init__(self,refresh_interval=None,info={}):
        """
        See Data.__init__
        """
        info["Class"]="SpecFileData"
        Data.__init__(self,refresh_interval,info)


    def SetSource (self,source_name=None):
        """
        Sets a new source for data retrieving, an specfile.
        If the file exists, self.Source will be the Specfile
        object associated to this file.
        Parameters:
        source_name: name of the specfile 
        """        
        if source_name==self.SourceName: return 1
        if (source_name != None): 
            try:
                self.Source = specfile.Specfile(source_name)
            except:
                self.SourceName=None
                self.Source=None
                return 0
        else:
            self.Source=None
        self.SourceName=source_name
        return 1


    def GetSourceInfo (self, key=None):
        """
        Returns information about the Specfile object created by
        SetSource, to give application possibility to know about
        it before loading.
        Returns a dictionary with the keys "Size" (number of possible
        keys(to this source) and "KeyList" (list of all available keys
        in this source). Each element in "KeyList" is an string in
        the format "x.y" where x is the number of scan and y is
        the order. "x.y" works as the key to retrieve the information
        of this scan.
        There's also the key "NumMca" in the returned dictionary,
        which value is a list of numbers of mcas, for each value of
        "KeyList".
        If key given returns information of a perticular key instead.
        """        
        if self.SourceName == None: return None

        if key is None: return self.__GetSourceInfo()
        else:
            key_type= self.__GetKeyType(key)
            if key_type=="scan": scan_key= key
            elif key_type=="mca": (scan_key, mca_no)=self.__GetMcaPars(key)
            return self.__GetScanInfo(scan_key)
    
    
    def LoadSource(self,key_list="ALL",append=0,invalidate=1,row="ALL",col="ALL"):
        """
        Creates a given number of pages, getting data from the actual
        source (set by SetSource)
        Parameters:
        key_list: list of all keys to be read from source. It is a list of
                 strings in the following formats:
                 "x.y": defines a scan data, x= number, y=order
                 "x.y.z": defines a mca, mca number z of the scan "x.y"
                 "x.y.z.w": defines a mca, row z, mca w of the scan "x.y"
                 It can be also one string, if only one array is to be read.
        append: If non-zero appends to the end of page list.
                Otherwise, initializes the page list                
        invalidate: if non-zero performas an invalidade call after
                    loading

        row: If set to an integer, loads a single row (not considered for mca) (0-based indexed)
        col: If set to an integer, loads a single column (not considered for mca) (0-based indexed)                     
        """
        if append==0: Data.Delete(self)
        if key_list == "ALL": key_list=self.__GetScanList
        elif type(key_list) is types.StringType: key_list=[key_list]

        file_info= self.__GetFileInfo()

        for key in key_list:
            key_type= self.__GetKeyType(key)
            if key_type=="scan": scan_key=key
            elif key_type=="mca": (scan_key, mca_no)=self.__GetMcaPars(key)

            scandata= self.Source.select(scan_key)
            info= self.__GetScanInfo(scan_key, scandata)
            info["Key"]= key
            info["FileInfo"]=file_info
            info["KeyType"]=key_type
            if key_type == "scan":
                info["row"]= row
                info["col"]= col
                try:
                    if info["row"]!="ALL":
                        if type(info["row"])==types.StringType: array= scandata.dataline(info["row"])
                        else: array= scandata.dataline(info["row"]+1)
                        array=Numeric.reshape(array,(1,array.shape[0]))
                    elif info["col"]!="ALL":
                        if type(info["col"])==types.StringType: array= scandata.datacol(info["col"])
                        else: array= scandata.datacol(info["col"]+1)
                        array=Numeric.reshape(array,(array.shape[0],1))
                    else: array=Numeric.transpose(scandata.data()).copy()
                except: array=None
            elif key_type == "mca":
                try: array=scandata.mca(mca_no)
                except: array=None
                info.update(self.__GetMcaInfo(mca_no, scandata, info))
            self.AppendPage(info,array)
        if invalidate: self.Invalidate()


    def __GetFileInfo(self):
        file_info={}
        #try: file_info["AllMotors"] = self.Source.allmotors()
        #except: file_info["AllMotors"] = None
        #---> read it in scan info together with motor positions
        try: file_info["Title"] = self.Source.title()
        except: file_info["Title"] = None
        try: file_info["User"] = self.Source.user()
        except: file_info["User"] = None
        try: file_info["Date"] = self.Source.date()
        except: file_info["Date"] = None
        try: file_info["Epoch"] = self.Source.epoch()    
        except: file_info["Epoch"] = None
        try: file_info["ScanNo"] = self.Source.scanno()
        except: file_info["ScanNo"] = None
        try: file_info["List"] = list
        except: file_info["List"] = None
        return file_info

    def __GetSourceInfo(self):
        scanlist=self.__GetScanList()
        source_info={}
        source_info["Size"]=len(scanlist)
        source_info["KeyList"]=scanlist

        num_mca=[]
        num_pts=[]
        commands=[]
        for i in scanlist:
            sel=self.Source.select(i)
            try: n = sel.nbmca()
            except: n = 0
            num_mca.append(n)
            try: n = sel.lines()
            except: n = 0
            num_pts.append(n)
            try: n = sel.command()
            except: n = ""
            commands.append(n)
        source_info["NumMca"]=num_mca
        source_info["NumPts"]=num_pts
        source_info["Commands"]= commands
        return source_info


    def __GetScanList(self):
        aux= string.split(self.Source.list(),",")
        newlistcount=[]
        newlist=[]
        for i in aux:
            if string.find(i,":")== -1:  start_index=end_index=int(i)
            else:
                s= string.split(i,":")
                start_index=int(s[0])
                end_index=int(s[1])
            for j in range(start_index,end_index+1):
                newlist.append(j)
                newlistcount.append(newlist.count(j))
        for i in range(len(newlist)):
            newlist[i]="%d.%d" % (newlist[i],newlistcount[i])
        return newlist
    

    def __GetKeyType (self,key):
        count = string.count(key, '.')
        if (count==1): return "scan"
        elif (count==2) or (count==3): return "mca"
        else: raise "SpecFileData: Invalid key"


    def __GetScanInfo(self, scankey, scandata=None):
        if scandata is None: scandata= self.Source.select(scankey)

        info={}
        info["SourceType"]= SOURCE_TYPE
        info["SourceName"]=self.SourceName
        info["Key"]=scankey
        info["Source"]=self.Source

        try: info["Number"] = scandata.number()
        except: info["Number"] = None
        try: info["Order"] = scandata.order()
        except: info["Order"] = None
        try: info["Cols"] = scandata.cols()
        except: info["Cols"] = 0
        try: info["Lines"] = scandata.lines()
        except: info["Lines"] = 0
        try: info["Date"] = scandata.date()
        except: info["Date"] = None
        try: info["AllMotor"] = self.Source.allmotors()
        except: info["AllMotor"] = None
        try: info["AllMotorPos"] = scandata.allmotorpos()
        except: info["AllMotorPos"] = None
        try: info["AllLabels"] = scandata.alllabels()
        except: info["AllLabels"] = None
        try: info["Command"] = scandata.command()
        except: info["Command"] = None
        try: info["Header"] = scandata.header("")
        except: info["Header"] = None
        try: info["NbMca"] = scandata.nbmca()
        except: info["NbMca"] = 0
        try: info["Hkl"] =  scandata.hkl()
        except: info["Hkl"] =  None
        if info["NbMca"]:
            if info["Lines"]>0 and info["NbMca"]%info["Lines"]==0:
                info["NbMcaDet"]= info["NbMca"]/info["Lines"]
            else:
                info["NbMcaDet"]= info["NbMca"]
        return info


    def __GetMcaInfo(self, mcano, scandata, info={}):
        mcainfo= {}
        if info.has_key("NbMcaDet"):
            det= info["NbMcaDet"]
            if info["Lines"]>0:
                mcainfo["McaPoint"]= int(mcano/info["NbMcaDet"])+(mcano%info["NbMcaDet"]>0)
                mcainfo["McaDet"]= mcano-((mcainfo["McaPoint"]-1)*info["NbMcaDet"])
                try: mcainfo["Labels"]= scandata.dataline(mcainfo["McaPoint"])
                except: mcainfo["Labels"]= None
            else:
                mcainfo["McaPoint"]= 0
                mcainfo["McaDet"]= mcano
                mcainfo["Labels"]= None
        return mcainfo


    def __GetMcaPars(self,key):
        nums= string.split(key,'.')
        size = len(nums)
        sel_key = nums[0] + "." + nums[1] 
        if size==3:
            mca_no=int(nums[2])
        elif size==4:
            sel=self.Source.select(sel_key)
            try: lines = sel.lines()
            except: lines=0
            if nums[3]==0: mca_no=int(nums[2])
            else:          mca_no=((int(nums[3])-1)*lines)+int(nums[2])
        else: raise "SpecFileData: Invalid key"
        return (sel_key,mca_no)     



################################################################################
#EXEMPLE CODE:
    
if __name__ == "__main__":
    import sys,time
    try:    
        Filename=sys.argv[1]
        Scan=sys.argv[2]
        obj=SpecFileData()
        obj.SetSource(Filename)
        obj.LoadSource(Scan)
    except:
        print "Usage: SpecFileData.py <filename> <key>"
        sys.exit()        
    
    for i in range (obj.GetNumberPages()):
        info= obj.GetPageInfo(i)
        data= obj.GetPageArray(i)
        print info
        print data
    
