"""
    TableView.py
    View derived class for table data representation
"""

from PyDVT import __version__,__date__,__author__

    
from View import *
from   Filter import Filter
from Data import Data
from DataSelection import RectSelection
import Numeric 
import types





class TableFilter(Filter):
    """
    View's standard filter.
    """
    def __init__(self,name=None,source=None,synchronized=1,buffer_input=0,xdata=None,ydata=None):
        """
        Parameters:
            name: Optional name of the function
            source: Source Filter/DataSelection
            synchronized: non-zero if on-line
            buffer_input: see Filter.__init__
            xdata: Optional NumPy array defining horizontal header
                   (can be received by source  as well)
            ydata: Optional NumPy array defining vertical header
                   (can be received by source  as well)
        """
        Filter.__init__(self,source,synchronized,buffer_input)
        self.ydata=ydata
        self.xdata=xdata        
        if name==None: self.name=str(self)
        else: self.name=name
        self.Function={}

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys:
            "name": string: name of function or None
            "data": NumPy array
            "xdata" 1d NumPy array or None
            "ydata" 1d NumPy array or None
        """
        sel=self.GetInput()
        if "data" not in sel.keys(): return {"name":self.name}
        if self.xdata is not None: ax= self.xdata
        elif "xdata" in sel.keys(): ax=sel["xdata"]
        else: ax=None
        if self.ydata is not None: ay= self.ydata
        elif "ydata" in sel.keys(): ay=sel["ydata"]
        else: ay=None
        return {"name":self.name,"data":sel["data"],"xdata":ax,"ydata":ay}
        


class Table(TableFilter):
    """
    View's simplified filter for direct NumPy interface.
    (creates and hides it's own data object)
    """
    def __init__(self,data=None,xdata=None,ydata=None):
        """
        Parameters:
            data: NumPy 1d or 2d array to be displayed
            xdata: Optional NumPy array defining x coordinates
            ydata: Optional NumPy array defining y coordinates 
        """
        self.data=Data()
        if data is not None: self.data.AppendPage(array=data)                
        TableFilter.__init__(self,None,RectSelection(self.data),1,0,xdata,ydata)

    def Destroy(self,source=None):
        """
        If filter gets out of scope, this should be called to destroy
        internally created data object.
        """
        if self.Alive==0: return
        self.Alive=0
        self.data.Destroy()



class TableView(View):
    """
    View derived class for table (numeric) representation of data.
    This class accepts just one source object. If the SetData
    is called with multiple sources, only the first one is considered.    
    TableView expects the following keys from it's source (GetOutput
    method):
    "data": 1d or 2d NumPy array with the information to be displayed on the table.
    "name": Optional. Not used for the moment.
    "xdata": Optional. 1d NumPy array, sets the horizontal header.
    "ydata": Optional. 1d NumPy array, sets the vertical header (if "data" is 2d).

    Interface:
    ===========================
    View interface
    """

    def __init__(self, parent=None, pars={}, **kw):       
        """
        See View.__init__
        Parameters:
          parent: Parent window
          pars:   Dictionary with View initialization options
                  New options defined by this class (in addiction to base class options):
                    No new option
          kw:     keywords to Container initializatregisterion                
        """
        View.__init__(self, parent, pars, **kw)
        self.CreateMenu()


    def CreateDrawable(self):
        """
        Virtual: Implements creation of  a Drawable object (which implements the
        Drawable interface)
        """        
        self.Drawable= Drawable_Table(self)
        self.Drawable.Show()
        

    def CreateMenu(self):
        """
        Can be overwritten by derived classes to create a
        different popup menu
        """
        pass        

       
    def DataChanged(self,source=None):
        """
        Virtual: See View.DataChanged
        """
        self.Function={}
        if self.Source != ():
            self.Function=self.Source[0].GetOutput()
            data=xdata=ydata=None
            if "data" in self.Function.keys():                
                data=self.Function["data"]
                if len(data.shape)==1:
                    data=Numeric.reshape(data,(1,data.shape[0]))
                    if "xdata" in self.Function.keys(): xdata=self.Function["xdata"]
                elif len(data.shape)==2:                        
                    if "xdata" in self.Function.keys(): xdata=self.Function["xdata"]
                    if "ydata" in self.Function.keys(): ydata=self.Function["ydata"]
                else:
                    data=None
            self.Drawable.SetData(data,xdata,ydata)
        else:
            self.Drawable.SetData(None)

    def Redraw(self):   
        """
        Virtual: See View.Redraw
        """
        pass
        
        
    def Save(self,filename,format="CSV"):
        try:        
            if format != "CSV": return
            file=open(filename, 'w+')
            if "data" in self.Function.keys():
                data=self.Function["data"]
                if len(data.shape)==1:
                    if "xdata" in self.Function.keys():
                        self._SaveRow(file,self.Function["xdata"])
                    self._SaveRow(file,data)
                elif len(data.shape)==2:
                    if "xdata" in self.Function.keys():
                        if "ydata" in self.Function.keys(): file.write(' ;')
                        self._SaveRow(file,self.Function["xdata"])
                    index=0

                    if "ydata" in self.Function.keys():
                        index=0
                        for row in range(data.shape[0]):
                            file.write(str(self.Function["ydata"][index]) + ';')
                            self._SaveRow(file,data[row])
                            index=index+1                        
                    else:
                        for row in range(data.shape[0]): self._SaveRow(file,data[row])
            file.close()
            return 1
        except:
            return 0
        
        
    def _SaveRow(self,file,row):
        for item in row: file.write(str(item) + ';')
        file.write(str(item) + '\n')

    def GetSaveFormats(self):
        """
        Returns tuple with supported save formats
        """
        return ("CSV",)
            
