"""
    MeshView.py
    View derived class for 3d image displaying

"""

from PyDVT import __version__,__date__,__author__


from View import *
from Filter import Filter
from Data import Data
from DataSelection import RectSelection

import Numeric
import Command


class MeshFilter(Filter):
    """
    View's standard filter.
    """
    def __init__(self,name=None,source=None,synchronized=1,buffer_input=0,xdata=None,ydata=None,pen=Pen((128,128,128),1,"solid")):
        """
        Parameters:
            name: Optional name of the function
            source: Source Filter/DataSelection
            synchronized: non-zero if on-line
            buffer_input: see Filter.__init__
            xdata: Optional NumPy array defining x coordinates, increasing values
                   (can be received by source  as well)
            ydata: Optional NumPy array defining y coordinates, increasing values 
                   (can be received by source  as well)
            pen: Optional Pen for drawing (it can be set afterwards with MeshView's SetPen)
        """
        Filter.__init__(self,source,synchronized,buffer_input)
        if xdata is not None and type(xdata) != Numeric.arraytype: self.xdata=Numeric.array(xdata)
        else:self.xdata=xdata
        if ydata is not None and type(ydata) != Numeric.arraytype: self.ydata=Numeric.array(ydata)
        else:self.ydata=ydata
        self.pen=pen
        if name==None: self.name=str(self)
        else: self.name=name

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys:
            "name": string: name of function or None
            "data": NumPy array
            "xdata" 1d NumPy array or None
            "ydata" 1d NumPy array or None
            "pen"  Pen object      
        """
        sel=self.GetInput()
        if "data" not in sel.keys(): return {"name":self.name}

        if self.xdata is not None: ax= self.xdata
        elif "xdata" in sel.keys(): ax=sel["xdata"]
        else: ax=ax=Numeric.arrayrange(sel["data"].shape[1])

        if self.ydata is not None: ay= self.ydata
        elif "ydata" in sel.keys(): ay=sel["ydata"]
        else: ay=Numeric.arrayrange(sel["data"].shape[0])
        
        return {"name":self.name,"data":sel["data"],"xdata":ax,"ydata":ay,"pen":self.pen}
        


class Mesh(MeshFilter):
    """
    View's simplified filter for direct NumPy interface.
    (creates and hides it's own data object)
    """
    def __init__(self,data=None,xdata=None,ydata=None,pen=Pen((128,128,128),1,"solid")):
        """
        Parameters:
            data: NumPy 2d array to be displayed
            xdata: Optional NumPy array defining x coordinates, increasing values
            ydata: Optional NumPy array defining y coordinates , increasing values
            pen: Optional Pen to the drawing
        """
        self.data=Data()
        if data is not None: self.data.AppendPage(array=data)                
        MeshFilter.__init__(self,None,RectSelection(self.data),1,0,xdata,ydata,pen)

    def Destroy(self,source=None):
        """
        If filter gets out of scope, this should be called to destroy
        internally created data object.
        """
        if self.Alive==0: return
        self.Alive=0
        self.data.Destroy()


class MeshView(View):
    """
    View derived class for 3d representation of 2d arrays.
    MeshView expects the following keys from it's source (GetOutput
    method):
    "data": 2d NumPy array to be drawn
    "xdata":1d NumPy array with the x values. If None, take's [0 .. x dimention of "data"]
            (have to be ascending)
    "ydata":1d NumPy array with the y values. If None, take's [0 .. y dimention of "data"]
            (have to be ascending)
    "name": Optional. Not used for the moment.
    "pen":  Optional. Pen for drawing current source.

    Interface:
    ===========================
    View interface
    SetStyle
    SetMeshPlottingMode
    SetScaleLinear
    SetScaleLog
    SetLabels
    SetPen
    SetXAxis
    SetYAxis
    SetZAxis
    """

    def __init__(self, parent=None, pars={}, **kw):
        """
        See View.__init__
        Parameters:
          parent: Parent window
          pars:   Dictionary with View initialization options
                  New options defined by this class (in addiction to base class options):
                    "TrackScrollbars": If non-zero, redraws for every 
                        Default 0
          kw:     keywords to Container initializatregisterion                
        """
        View.__init__(self, parent,pars,**kw)
        self.CreateMenu()
        self.XAxis=self.YAxis=self.ZAxis=None
        self.SetPointer("cross")


    def CreateDrawable(self):
        """
        Virtual: Implements creation of  a Drawable object (which implements the
        Drawable interface)
        """        
        self.Drawable= Drawable_3D(self)
        self.Drawable.Show()
        self.Drawable.SetTrackScrollbars(1)
        self.Drawable.SetTrackScrollbars(0)
        

    def CreateMenu(self):
        """
        Can be overwritten by derived classes to create a
        different popup menu
        """
        if self.MenuPopup is not None:self.AddMenuSeparator()
        else: self.MenuPopup = Menu(self.Drawable)
        popup=Menu(self.MenuPopup)          
        popup.AddCommand("3D Surface",Command.Command(self.SetStyle,style="3dSurface"),"radiobutton")
        popup.AddCommand("Surface Mesh",Command.Command(self.SetStyle,style="SurfaceMesh"),"radiobutton")
        popup.AddCommand("Shaded 3D Surface",Command.Command(self.SetStyle,style="ShadedSurface"),"radiobutton")
        popup.SetCheckedRadio("3D Surface")
        self.AddMenuPopupCascade("Style",popup)
        popup=Menu(self.MenuPopup)          
        popup.AddCommand("X Lines",Command.Command(self.SetMeshPlottingMode,mode="XLines"),"radiobutton")
        popup.AddCommand("Y Lines",Command.Command(self.SetMeshPlottingMode,mode="YLines"),"radiobutton")
        popup.AddCommand("Both",Command.Command(self.SetMeshPlottingMode,mode="Both"),"radiobutton")
        popup.SetCheckedRadio("Y Lines")
        self.plotmodechk=self.AddMenuPopupCascade("Plot Mode",popup)
        popup=Menu(self.MenuPopup)          
        popup.AddCommand("Linear",self.SetScaleLinear,"radiobutton")
        popup.AddCommand("Logarithmic",Command.Command(self.SetScaleLog),"radiobutton")
        popup.SetCheckedRadio("Linear")
        self.AddMenuPopupCascade("Scale",popup)
        self.trackscrollchk=self.AddMenuPopupItem("Track Scrollbars",self._CheckTrackScrollbars,"checkbutton")

    def _CheckTrackScrollbars(self):
        if self.MenuPopup.IsItemChecked(self.trackscrollchk):
            self.MenuPopup.CheckItem(self.trackscrollchk,0)
            self.SetTrackScrollbars(0)
        else:
            self.MenuPopup.CheckItem(self.trackscrollchk,1)
            self.SetTrackScrollbars(1)

    def SetTrackScrollbars(self,value):
        """
        Sets scrollbar mode
        Parameters:
          value: if non-zero, scrollbars generate continuous redrawing
                 default:0
        """
        self.Drawable.SetTrackScrollbars(value)
        

    def SetStyle(self,style="3dSurface"):
        """
        Sets drawing style
        Parameters:
          style: "3dSurface", "SurfaceMesh" or "ShadedSurface"
                 default:"3dSurface"
        """
        self.Drawable.SetStyle(style)
        if hasattr (self,"plotmodechk"):
            if style=="ShadedSurface": self.MenuPopup.DisableItem(self.plotmodechk)
            else: self.MenuPopup.EnableItem(self.plotmodechk)
        

    def SetMeshPlottingMode(self,mode=""):
        """
        Sets mesh plotting mode
        Parameters:
          mode: "XLines", "YLines" or "Both"
                 default:"YLines" (Z=f(x) for each y)
        """
        self.Drawable.SetMeshPlottingMode(mode)

    def SetScaleLinear(self):
        """
        Sets linear scale for the drawing (default)
        """
        self.Drawable.Linear()

    def SetScaleLog(self,minval=0.000001):
        """
        Sets log scale for the drawing 
        Parameters:
           minvalue: all values less than or equals to 0 are set to it
        """
        self.Drawable.Log(minval)


    def SetLabels(self,title_label="",x_label="",y_label="",z_label=""):
        """
        Sets drawing labels
        Parameters:
          title_label: Title of the drawing
          x_label: Label of x axis
          y_label: Label of y axis
          z_label: Label of z axis
        """
        self.Drawable.SetLabels(title_label,x_label,y_label,z_label)

    def SetPen(self,name,pen):
        """
        Sets pen for a source plotting
        Parameters:
          name: name of the source
          pen: Pen object
        """
        for sel in self.Source:
            if sel.name==name:
                sel.pen=pen                
                self.Drawable.SetFunctionItem(name,"pen",pen)
                self.Update()

    def SetXAxis(self,x_Axis=None):
        """
        Fixes the range of x axis.
        Parameters:
          x_Axis: tuple (min_value,maxvalue)
                  If x_Axis==None (default) range definition is automatic.
        """
        self.XAxis=x_Axis

    def SetYAxis(self,y_Axis=None):
        """
        Fixes the range of y axis.
        Parameters:
          y_Axis: tuple (min_value,maxvalue)
                  If y_Axis==None (default) range definition is automatic.
        """
        self.YAxis=y_Axis

    def SetZAxis(self,z_Axis=None):
        """
        Fixes the range of z axis.
        Parameters:
          z_Axis: tuple (min_value,maxvalue)
                  If z_Axis==None (default) range definition is automatic.
        """
        self.ZAxis=z_Axis

    def _DrawAxis(self,sels):
        minxarr=[]
        maxxarr=[]
        minyarr=[]
        maxyarr=[]
        minzarr=[]
        maxzarr=[]
        
        for sel in sels.values():
            if "data" in sel.keys():
                func=sel["data"]
                if "xdata" not in sel.keys() or  sel["xdata"] is None or len(sel["xdata"])==0:
                    sel["xdata"]=Numeric.arrayrange(func.shape[1])
                    minx,maxx=0,func.shape[1]-1
                else:
                    minx,maxx=sel["xdata"][0],sel["xdata"][-1]                    
                minxarr.append(minx)
                maxxarr.append(maxx)
                
                if "ydata" not in sel.keys() or  sel["ydata"] is None or len(sel["ydata"])==0:
                    sel["ydata"]=Numeric.arrayrange(func.shape[0])
                    miny,maxy=0,func.shape[0]-1
                else:
                    miny,maxy=sel["ydata"][0],sel["ydata"][-1]
                if "name" not in sel.keys():
                    sel["name"]="1"
                minyarr.append(miny)
                maxyarr.append(maxy)

                if func.shape[0] and func.shape[1]:
                    flat=func.flat
                    minz,maxz=min(flat),max(flat)
                else:
                    minz,maxz=0,1
                minzarr.append(minz)
                maxzarr.append(maxz)
        try:
            minx,maxx,miny,maxy,minz,maxz=min(minxarr),max(maxxarr),min(minyarr),max(maxyarr),min(minzarr),max(maxzarr)
        except:
            minx,maxx,miny,maxy,minz,maxz=0,1,0,1,0,1
        if maxz <= minz: maxz=minz+0.000001
        if maxx <= minx: maxx=minx+0.000001
        if maxy <= miny: maxy=miny+0.000001
        if self.XAxis is not None: (minx,maxx)=self.XAxis
        if self.YAxis is not None: (miny,maxy)=self.YAxis
        if self.ZAxis is not None: (minz,maxz)=self.ZAxis
        if self.FlagLockPosition and hasattr(self,"minz"):minz,maxz=self.minz,self.maxz
        self.Drawable.SetEnv(minx,maxx,miny,maxy,minz,maxz)
        self.minz,self.maxz=minz,maxz
        

    def DataChanged(self,source=None):
        """
        Virtual: See View.DataChanged
        """
        self.Drawable.RemoveFunction()
        sels={}
        if self.Source != ():
            for i in self.Source:
                sel=i.GetOutput()
                if "name" not in sel.keys() or sel["name"]==None: sel["name"] = str(i)
                sels[sel["name"]]=sel
            self._DrawAxis(sels)
            for i in sels.keys():
                if "data" in sels[i].keys():
                    self.Drawable.SetFunction(i,sels[i])                
        #else:
        #    self.Drawable.RemoveGraphicObjects()
        self.Update()


    def Redraw(self):
        """
        Virtual: See View.Redraw
        """
        self.Drawable.Redraw()


    def GetSaveFormats(self):
        """
        Returns tuple with supported save formats
        """
        return ("PNG","BMP","PS","JPG")
