"""
    ContourView.py
    View derived class for contour and shading plots
"""

from PyDVT import __version__,__date__,__author__

from View import *
from Filter import Filter
from Data import Data
from DataSelection import RectSelection

import Numeric
import Command


class ContourFilter(Filter):
    """
    View's standard filter.
    """
    def __init__(self,name=None,source=None,synchronized=1,buffer_input=0,xdata=None,ydata=None):
        """
        Parameters:
            name: Optional name of the function
            source: Source Filter/DataSelection
            synchronized: non-zero if on-line
            buffer_input: see Filter.__init__
            xdata: Optional NumPy array defining x coordinates
                   (can be received by source  as well)
            ydata: Optional NumPy array defining y coordinates 
                   (can be received by source  as well)
        """
        Filter.__init__(self,source,synchronized,buffer_input)
        if xdata is not None and type(xdata) != Numeric.arraytype: self.xdata=Numeric.array(xdata)
        else:self.xdata=xdata
        if ydata is not None and type(ydata) != Numeric.arraytype: self.ydata=Numeric.array(ydata)
        else:self.ydata=ydata
        if name==None: self.name=str(self)
        else: self.name=name

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys:
            "name": string: name of function or None
            "data": NumPy array
            "xdata" 1d NumPy array
            "ydata" 1d NumPy array
        """
        sel=self.GetInput()
        if "data" not in sel.keys(): return {"name":self.name}

        if self.xdata is not None: ax= self.xdata
        elif "xdata" in sel.keys(): ax=sel["xdata"]
        else: ax=ax=Numeric.arrayrange(sel["data"].shape[1])

        if self.ydata is not None: ay= self.ydata
        elif "ydata" in sel.keys(): ay=sel["ydata"]
        else: ay=Numeric.arrayrange(sel["data"].shape[0])
        return {"name":self.name,"data":sel["data"],"xdata":ax,"ydata":ay}
        


class Contour(ContourFilter):
    """
    View's simplified filter for direct NumPy interface.
    (creates and hides it's own data object)
    """
    def __init__(self,data=None,xdata=None,ydata=None):
        """
        Parameters:
            data: NumPy 2d array to be displayed
            xdata: Optional NumPy array defining x coordinates
            ydata: Optional NumPy array defining y coordinates 
        """
        self.data=Data()
        if data is not None: self.data.AppendPage(array=data)                
        ContourFilter.__init__(self,None,RectSelection(self.data),1,0,xdata,ydata)

    def Destroy(self,source=None):
        """
        If filter gets out of scope, this should be called to destroy
        internally created data object.
        """
        if self.Alive==0: return
        self.Alive=0
        self.data.Destroy()


class ContourView(View):
    """
    View derived class for contour/shading representation of 2d arrays.
    This class accepts just one source object. If the SetData
    is called with multiple sources, only the first one is considered
    (this because the shading plot overrites existing plots. As DrawableContour
    manages multiple functions, other class can be written to display multiple
    contour plots.    
    MeshView expects the following keys from it's source (GetOutput
    method):
    "data": 2d NumPy array to be drawn
    "xdata":1d NumPy array with the x values. If None, take's [0 .. x dimention of "data"]
    "ydata":1d NumPy array with the y values. If None, take's [0 .. y dimention of "data"]
    "name": Optional. Not used for the moment.

    Interface:
    ===========================
    View interface
    SetStyle
    SetContourPen
    ShowContourLabels
    SetContourLevels
    SetShadeLevels
    SetShadeColormap
    SetScaleLinear
    SetScaleLog
    SetLabels
    ToggleGridlines
    """

    def __init__(self, parent=None, pars={}, **kw):
        """
        See View.__init__
        Parameters:
          parent: Parent window
          pars:   Dictionary with View initialization options
                  New options defined by this class (in addiction to base class options):
                    No new option
          kw:     keywords to Container initializatregisterion                
        """
        View.__init__(self, parent,pars,**kw)
        self.contour_levels_menu_dict,self.shade_levels_menu_dict={},{}
        self.CreateMenu()
        self.SetContourLevels(8)
        self.SetShadeLevels(8)
        self.ContourPen=Pen((0,255,0),1,"solid")
        self.ContourPenDlg=None
        self.FunctionName=None
        self.eh.register("graphElementStyleChangedEvent",self._ChangeGraphElementStyle)
        self.SetPointer("cross")        

    def CreateDrawable(self):
        """
        Virtual: Implements creation of  a Drawable object (which implements the
        Drawable interface)
        """        
        self.Drawable= Drawable_Contour(self)
        self.Drawable.Show()        
        

    def CreateMenu(self):
        """
        Can be overwritten by derived classes to create a
        different popup menu
        """
        if self.MenuPopup is not None:self.AddMenuSeparator()
        else: self.MenuPopup = Menu(self.Drawable)
        popup=Menu(self.MenuPopup)
        for style in ("Contour","Shade","Both"):
            popup.AddCommand(style,Command.Command(self.SetStyle,style=style),"radiobutton")
        popup.SetCheckedRadio("Contour")
        self.AddMenuPopupCascade("Style",popup)
        self.contour_menu=Menu(self.MenuPopup)
        popup_levels=Menu(self.contour_menu)
        for n in (2,4,6,8,12,16,20,32):
            popup_levels.AddCommand(str(n),Command.Command(self.SetContourLevels,n),"radiobutton")
        popup_levels.SetCheckedRadio("8")
        self.contour_menu.AddCascade("Levels",popup_levels)
        self.contour_menu.AddCommand("Pen",self._CmdSetContourPen)
        self.contour_labels_menu=self.contour_menu.AddCommand("Labels",self._SetContourLabels,"checkbutton")
        self.AddMenuPopupCascade("Contour Proprieties",self.contour_menu)

        popup=Menu(self.MenuPopup)
        popup_levels=Menu(popup)
        for n in (4,8,16,32,64,128):
            popup_levels.AddCommand(str(n),Command.Command(self.SetShadeLevels,n),'radiobutton')
        popup_levels.SetCheckedRadio("8")
        popup.AddCascade("Levels",popup_levels)
        colormappopup=Menu(popup)
        for n in ("GrayScale","Color","Temperature","Red","Green","Blue","RevGrey"):
            colormappopup.AddCommand(n,Command.Command(self.SetShadeColormap,n),'radiobutton')
        colormappopup.SetCheckedRadio("GrayScale")
        popup.AddCascade("Colormap",colormappopup)
        self.AddMenuPopupCascade("Shade Proprieties",popup)

        popup=Menu(self.MenuPopup)          
        popup.AddCommand("Linear",self.SetScaleLinear,'radiobutton')
        popup.AddCommand("Logarithmic",self.SetScaleLog,'radiobutton')
        popup.SetCheckedRadio("Linear")
        self.AddMenuPopupCascade("Scale",popup)

        self.AddMenuPopupItem("Toggle Gridlines",self.ToggleGridlines)

    def _CmdSetContourPen(self):
        if self.ContourPenDlg==None or self.ContourPenDlg.IsDestroyed():
            Cnf={'style':self.ContourPen.style,'width':self.ContourPen.width,'color':self.ContourPen.color}            
            self.ContourPenDlg=GraphStyleDialog(self.parent,self.eh,"Contour Pen",Cnf)
        self.ContourPenDlg.Show()

    def _ChangeGraphElementStyle(self,graphName,graphCnf):
        newpen=Pen(graphCnf['color'],int(str(graphCnf['width'])),str(graphCnf['style']))
        self.SetContourPen(newpen)
        
    def SetContourPen(self,pen):
        """
        Sets pen of the contour plot
        Parameters:
          pen: Pen object
        """
        self.ContourPen=pen
        if self.FunctionName != None:
            self.Drawable.SetFunctionItem(self.FunctionName,"pen",pen)
            self.Update()


    def _SetContourLabels(self):
        if self.contour_menu.IsItemChecked(self.contour_labels_menu):
            self.ShowContourLabels(0)
        else:
            self.ShowContourLabels(1)

    def ShowContourLabels(self,show_labels):
        """
        Show/hides countour labels
        Parameters:
          show_labels: shows if non-zero (default = 0)
        """
        self.Drawable.SetContourLabels(show_labels)
        if hasattr(self,"contour_labels_menu"): self.contour_menu.CheckItem(self.contour_labels_menu,show_labels)

    def SetContourLevels(self,levels):
        """
        Parameters:
          levels: number of contour levels
        """
        self.Drawable.SetContourLevels(levels)        
        

    def SetShadeLevels(self,levels):
        """
        Parameters:
          levels: number of shade levels
        """
        self.Drawable.SetShadeLevels(levels)

    def SetShadeColormap(self,colormap):
        """
        Parameters:
          colormap: "GrayScale","Color","Temperature","Red","Green","Blue" or "RevGrey"
        """
        self.Drawable.SetShadeColormap(colormap)

    def SetScaleLinear(self):
        """
        Sets linear scale for the drawing (default)
        """
        self.Drawable.Linear()

    def SetScaleLog(self):
        """
        Sets log scale for the drawing 
        """
        self.Drawable.Log()

    def SetStyle(self,style="Contour"):
        """
        Sets drawing style 
        Parameters:
          style: "Contour", "Shade" or "ContourShade"
                 Default: "Contour"
        """
        self.Drawable.SetStyle(style)


    def SetLabels(self,title_label="",x_label="",y_label=""):
        """
        Sets drawing labels
        Parameters:
          title_label: Title of the drawing
          x_label: Label of x axis
          y_label: Label of y axis
        """
        self.Drawable.SetLabels(title_label,x_label,y_label)

    def ToggleGridlines(self):
        """
        Toggles gridlines on the plotting
        """
        self.Drawable.ToggleGridlines()


    def DataChanged(self,source=None):
        """
        Virtual: See View.DataChanged
        """
        self.Drawable.RemoveFunction()
        if self.Source != ():
            sel=self.Source[0].GetOutput()
            if "data" in sel.keys():
                func=sel["data"]
                if func.shape[0]>1 and func.shape[1]>1:
                    if "xdata" not in sel.keys() or  sel["xdata"] is None:
                        sel["xdata"]=Numeric.arrayrange(func.shape[1])
                        minx,maxx=0,func.shape[1]-1
                    else:
                        minx,maxx=sel["xdata"][0],sel["xdata"][-1]
                        
                    if "ydata" not in sel.keys() or  sel["ydata"] is None:
                        sel["ydata"]=Numeric.arrayrange(func.shape[0])
                        miny,maxy=0,func.shape[0]-1
                    else:
                        miny,maxy=sel["ydata"][0],sel["ydata"][-1]
                    if "name" not in sel.keys() or sel["name"]==None:
                        sel["name"]=str(self.Source[0])
                    self.FunctionName=sel["name"]
                    
                    self.Drawable.SetEnv(minx,maxx,miny,maxy)
                    sel["pen"]=self.ContourPen
                    self.Drawable.SetFunction(sel["name"],sel)
                
        self.Update()

    def Destroy(self,source=None):
        """
        See View.Destroy
        """
        self.eh.unregister("graphElementStyleChangedEvent",self._ChangeGraphElementStyle)
        View.Destroy(self)

    def Redraw(self):
        """
        Virtual: See View.Redraw
        """
        self.Drawable.Redraw()


    def GetSaveFormats(self):
        """
        Returns tuple with supported save formats
        """
        return ("PNG","BMP","PS","JPG")
