/****************************************************************************

                     uMatrix C++ Matrix Library

    Copyright (C) 1996  David Weber, Michael Sipe and Rajesh Shenoy

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    David Weber can be contacted at weber@ece.cmu.edu or 
    http://www.ece.cmu.edu/afs/ece/usr/weber/.home-page.html

****************************************************************************/

/*
$Id: nVector.h,v 1.1.1.1 2001/06/27 17:07:55 mirone Exp $
$Author: mirone $
$Source: /scisoft/users/mirone/SOURCESCVS//nMatrix/nVector.h,v $
$Revision: 1.1.1.1 $
$Log: nVector.h,v $
Revision 1.1.1.1  2001/06/27 17:07:55  mirone
nMatrix now on cvs


 * Revision 1.5  1996/06/17  15:36:38  weber
 * Added Gnu library public license header
 *
 * Revision 1.4  1996/01/30  18:09:25  weber
 * Fixed bug caused by doing a shallow copy instead of a deep copy
 * when assigning a uMatrix to a uMatrix constructed from a pointer.
 * The bug caused the LHS to become detached from its intended memory
 * destination and caused some really weird errors.
 *
 * Revision 1.2  1995/11/06  17:23:12  weber
 * Changed many "int"'s to "unsigned int" to satisfy the whim of g++
 * 2.7.0.
 *
*/

#ifndef uVector_included
#define uVector_included
#include <stdlib.h>
#include <iostream.h>

#ifdef Debug_on
#define DBG(Stuff) Stuff
#else
#define DBG(Stuff) /*don't do this */
#endif


template<class T>
class uVector
{
      // We need this so that we can have multiple references to the same 
      // piece of memory. Destructors only blow the storage if there are no
      // more references to the memory. Cost: one extra level of indirection. 
      // The "noDestructorDelete" is needed if we want to construct a vector
      // from some memory allocated elsewhere. If this is set to 1, then the
      // destructors and shallow stuff will not delete the memory. This is
      // to be used carefully.

      struct Store
      {
	    T *s;
	    unsigned int M;  // Size of vector
	    unsigned int RefCount; 
	    int noDestructorDelete;
      };

      Store *Storage;

   public:
      inline void inizializza();
 
      // Constructors
      inline uVector( const int s );
      inline uVector();  // Special blank (zero length) constructor

   // Construct from pre allocated memory
      inline uVector( const int s, T[] ); 

      // Construct from another uVector
      inline uVector( const uVector& data ); 

      // Destructor
      inline ~uVector();

      // Operators
      // This one has no index check!!
      inline T& operator[](const unsigned int m) const {return Storage->s[m];}
      T& operator()(const unsigned int m) const; // index checked

      inline void operator=(const uVector&); // Deep Copy for the sane
      inline void operator=( T[] ); // assign to pointer. Carefull here.
      inline void operator=(const T&); // assign to scalar value
    
      // Member functions
      unsigned int size() const { return Storage->M; }
      void resize( const unsigned int n );  // resize vector and trash old data
      void resize( const unsigned int n, T data[] ); // resize to pointer 
      inline T* address() const { return Storage->s; } // return data address
      void shallow( const uVector& );   // shallow copy

      // true if more than one vector shares the allocated memory
      int hasMultipleReferences()
      {
	 return ( Storage->RefCount > 1 );
      }

      // Returns true if we allocated the memory, false if not. 
      int ownsAllocatedMemory()
      {
	 return ( Storage->noDestructorDelete == 0 );
      }
      
      
};

template<class T> // Deep copy of data
inline void uVector<T>::operator=( const uVector& data )
{
   DBG(cerr << "uVector<T>::operator=(const uVector<T>): "
       << "this=" << this << " *param=" << &data << "\n"; );
   
   // Note that resize will choke if the memory was not allocated by
   // uVector and the new size is different from the old size.
   (*this).resize( data.size() );

   // We HAVE to use the assignment operator here if we have vectors of
   // something like strings or matrices. No memcpy weenies here.
   T *s = Storage->s;
   T *d = data.address();
   unsigned int M4= Storage->M/4;
   for ( unsigned int i = 0; i <M4; i++ )
   {
      *s = *d;
      *(s+1)=*(d+1);
      *(s+2)=*(d+2);
      *(s+3)=*(d+3);
      s+=4;
      d+=4;
   }
   for ( unsigned int i = M4*4; i <Storage->M ; i++ )
   {
      *(s++) = *(d++);
   }
}

/*
    Copy a scalar value to all the elements of "this" vector
*/
template<class T>
inline void uVector<T>::operator=( const T& data)
{
   T *d = Storage->s;
   DBG(cerr << "uVector<T>::operator=(const T&): "
       << "this=" << this << " param=" << data << "\n"; );

   for ( unsigned int i = 0; i < Storage->M; i++ ) 
   {
      *d++ = data;
   }
}

// We HAVE to assume that the pointer points to something sensible with
// enough allocated memory else.... Long legal disclaimer goes here.

template<class T>
inline void uVector<T>::operator=( T dubiousDataPointer[] )
{
   // Chuck the existing storage (if any) if we can
   int m = Storage->M; 
   if ( --Storage->RefCount == 0 )
   {
      if  ( Storage->noDestructorDelete == 0 )
      {
	 delete [] Storage->s;
      }
      delete Storage;
   }

   Storage = new Store;
   Storage->M = m;
   Storage->RefCount = 1; // These things have references too!
   Storage->s = dubiousDataPointer;
   // We were created from a pointer so don't allow a delete[] in a destructor
   Storage->noDestructorDelete = 1;
}

template<class T>
inline uVector<T>::uVector( const int m, T dubiousDataPointer[] )
{
   Storage = new Store;
   Storage->s = dubiousDataPointer;
   Storage->RefCount = 1;
   Storage->M = m;
   // We were created from a pointer so don't allow a delete[] in a destructor
   Storage->noDestructorDelete = 1; 
}

template<class T>
inline uVector<T>::uVector( const int m )
{
   Storage = new Store;
   Storage->s = new T[Storage->M=m]; 
   Storage->RefCount = 1;
   Storage->noDestructorDelete = 0; // We were not created from a pointer
   DBG(cerr << "uVector(const int) constructor called: "
       << "this=" << this << " RefCount=" << Storage->RefCount 
       << "\n"; );   
}

template<class T>
inline uVector<T>::uVector()  // create a blank zero length vector
{
   int m = 0;
   Storage = new Store;
   Storage->s = new T[Storage->M = m];
   Storage->RefCount = 1;
   Storage->noDestructorDelete = 0; // We were not created from a pointer
   DBG(cerr << "uVector() constructor called: "
       << "this=" << this << " RefCount=" << Storage->RefCount 
       << "\n"; );
}

template<class T>
inline void uVector<T>::inizializza()  // create a blank zero length vector
{
   int m = 0;
   Storage = new Store;
   Storage->s = new T[Storage->M = m];
   Storage->RefCount = 1;
   Storage->noDestructorDelete = 0; // We were not created from a pointer
   DBG(cerr << "uVector() constructor called: "
       << "this=" << this << " RefCount=" << Storage->RefCount 
       << "\n"; );
}


template<class T>  // Deep copy of argument
inline uVector<T>::uVector( const uVector& data )
{
   Storage = new Store;
   Storage->RefCount = 1;     // First reference to this   
   Storage->s = new T[Storage->M = data.size()];
   Storage->noDestructorDelete = 0; // we own this because we allocated it!
   
   T *s = Storage->s;
   T *d = data.address();

   for ( unsigned int i = 0; i < Storage->M; i++ )
   {
      *s++ = *d++;
   }
   DBG(cerr << "uVector(const Vector) constructor called: "
       << "this=" << this << " *param=" << &data 
       << " RefCount=" << Storage->RefCount << "\n"; );
}

/*
   Destructor. Only blow the data storage if there are no other instances
   of Vector using the data
*/
template<class T>
inline uVector<T>::~uVector()
{
   // Only dump Storage if there are no more references to it
   DBG(cerr << "~uVector() destructor called: "
       << "this=" << this << " RefCount=" << Storage->RefCount; );
   
   if ( --Storage->RefCount == 0 )
   {
      // Check if we allocated this, then we are clear to delete it
      if ( Storage->noDestructorDelete == 0 ) 
      {
	 delete [] Storage->s;  
      }
      delete Storage;
      DBG(cerr << " Storage deleted"; );
   }
   DBG(cerr << "\n";);
}


/*
   Simple index operator with bounds checking. Use the [] operator if you
   do not like the bounds check.
*/
template<class T>
inline T& uVector<T>::operator()(const unsigned int m) const // index check
{
   if ( m < Storage->M )
   { 
      return Storage->s[m];
   }
   else // my version of a bounds checking error
   {
      cerr << "Bounds error in uVector::operator() at element " << m << endl;
      abort();
   }
}

/*
   Resize the vector by deleting the old data and reinitializing it
   with the new required size.
*/
template<class T>
void uVector<T>::resize( const unsigned int m )  // trashes existing data
{
   if ( m != Storage->M ) // Only resize if we must
   {

      // If we constructed using a pointer, it means that we do not own the
      // memory so we may not delete or resize it. 
      if ( Storage->noDestructorDelete == 1  ) 
      {
	 cerr << "uVector<T>::resize(int) called for uVector constructed from a pointer\n";
	 abort();
      }

      if ( --Storage->RefCount == 0 )
      {
	 delete [] Storage->s;
	 delete Storage;
      }
      Storage = new Store;  // Blow pointer to old storage structure
      Storage->M = m;
      Storage->RefCount = 1;
      Storage->noDestructorDelete = 0;
      Storage->s = new T[Storage->M]; // Assign required memory
   }
}

// Resize a vector to an existing pointer. Delete any existing storage
template<class T>
void uVector<T>::resize( const unsigned int m, T data[] )
{
   if ( --Storage->RefCount == 0 )
   {
      // Check if we allocated this, then we are clear to delete it
      if ( Storage->noDestructorDelete == 0 ) 
      {
	 delete [] Storage->s;  
      }
      delete Storage;
   }
   
   Storage = new Store;  // Blow pointer to old storage structure
   Storage->M = m;
   Storage->RefCount = 1;
   Storage->noDestructorDelete = 1;
   Storage->s = data; // Assign required memory
}

/*
   Do a shallow copy by sharing the Storage structure between one or more
   instances of a vector. It may be a good idea to prevent writing to
   a Vector when more than one instance of a Vector shares the data, 
   i.e. when Storage->RefCount > 1.
*/
template<class T>
void uVector<T>::shallow( const uVector& data )
{
   // delete the existing storage if needed
   if ( --Storage->RefCount == 0 )
   {
      if ( Storage->noDestructorDelete == 0 )
      {
	 delete [] Storage->s;
      }
      delete Storage;
   }
   Storage = data.Storage; // Just copy pointer to storage
   Storage->RefCount++;  // Pop up ref count and cheat const decl. for data
}
#endif









