from qt import *
from qttable import *
from mdi_icons import IconDict
from mdi_help import HelpDict

__version__="1.1"
__author__="A.Gobbo (gobbo@esrf.fr), E.Papillon (papillon@esrf.fr)"

class MDIApp(QApplication):
	#Defined to control the use of child windows not redrawables during resizing
	def __init__(self,*args,**key):
		QApplication.__init__(self,*args,**key)
		self.ToplevelMouseDown = 0
		self.ToplevelIgnoreResize = 0
		self.ToplevelSaveObject = None
		self.installEventFilter(self)

	def eventFilter(self,o,e):
		type=e.type()

		if type==QEvent.MouseButtonPress:
			self.ToplevelMouseDown = 1
			self.ToplevelIgnoreResize = 0
			self.ToplevelSaveObject = None
			self.ToplevelSaveEvent = None

        	elif type==QEvent.MouseButtonRelease:
			self.ToplevelMouseDown = 0
			if self.ToplevelIgnoreResize and self.ToplevelSaveObject is not None:
				self.ToplevelIgnoreResize = 0
				QApplication.postEvent(self.ToplevelSaveObject, self.ToplevelSaveEvent)
				self.ToplevelIgnoreResize = 0
				self.ToplevelSaveObject=None
		elif type==QEvent.Resize:
			if self.ToplevelMouseDown and isinstance(o,ChildWindow) and o.RepaintsWhileResizing==0:
				self.ToplevelIgnoreResize = 1
				self.ToplevelSaveObject=o
				self.ToplevelSaveEvent = QResizeEvent(e.size(),e.oldSize())
				return 1
		return QApplication.eventFilter(self, o, e)


class ApplicationWindow(QMainWindow):
	def __init__(self, parent=None, name="ApplicationWindow", fl=Qt.WDestructiveClose, options={}):
		QMainWindow.__init__(self, parent, name, fl)
		self.setCaption(name)

		# --- load application icons
		try:
			exec("from %s_icons import IconDict as AppIconDict"%name.replace(" ","_"))
			IconDict.update(AppIconDict)
		except:
			pass

		# --- load application help
		try:
			exec("from %s_help import HelpDict as AppHelpDict"%name.replace(" ","_"))
			HelpDict.update(AppHelpDict)
		except:
			pass

		self._options= {}
		self._options["FileToolBar"]= options.get("FileToolBar", 1)
		self._options["WinToolBar"]= options.get("WinToolBar", 1)
		self._options["MenuFile"]= options.get("MenuFile", 1)
		self._options["MenuTools"]= options.get("MenuTools", 1)
		self._options["MenuWindow"]= options.get("MenuWindow", 1)
		self._options["MenuHelp"]= options.get("MenuHelp", 1)
		self._options["MenuPlugins"]= options.get("MenuPlugins", 1)

		self.printer= QPrinter()
		if qVersion()>="3.0.0":
			self.mdi= QWorkspace(self)
			self.mdi.setScrollBarsEnabled(1)
			self.setCentralWidget(self.mdi)
		else:
			self.mdi= QWorkspace(self)
			self.mdi.setBackgroundColor(QColor("gainsboro"))
			self.setCentralWidget(self.mdi)

		self.setDockEnabled(Qt.DockTop, 0)

		self.initIcons()
		self.initMenuBar()
		self.initToolBar()

		self.followActiveWindow= 0

		self.mdi.show()
		self.resize(600,400)

	def initIcons(self):
		self.Icons= {}
		for (name, icon) in IconDict.items():
			pixmap= QPixmap(icon)
			if qVersion()<"3.0.0": self.Icons[name]= pixmap
			else: self.Icons[name]= QIconSet(pixmap)

	def initToolBar(self):
		if self._options["FileToolBar"]:
			self.fileToolBar= QToolBar(self, "filetoolbar")
			self.fileToolBar.setLabel("File Operations")
			self.onInitFileToolBar(self.fileToolBar)
			fileOpen= QToolButton(self.Icons["fileopen"], "Open", None, self.onOpen, self.fileToolBar, "open")
			QWhatsThis.add(fileOpen, HelpDict["fileopen"])
			fileSave= QToolButton(self.Icons["filesave"], "Save", None, self.onSave, self.fileToolBar, "save")
			QWhatsThis.add(fileSave, HelpDict["filesave"])
			filePrint= QToolButton(self.Icons["fileprint"], "Print", None, self.onPrint, self.fileToolBar, "print")
			QWhatsThis.add(filePrint, HelpDict["fileprint"])
			self.fileToolBar.addSeparator()
			QWhatsThis.whatsThisButton(self.fileToolBar)

		self.onInitToolBar()

		if self._options["WinToolBar"]:
			self.winToolBar= QToolBar(self, "wintoolbar")
			self.winToolBar.setLabel("Window resize")
			self.onInitWinToolBar(self.winToolBar)
			FullScreen= QToolButton(self.Icons["window_fullscreen"], "Full Screen", None, 
					self.windowFullScreen, self.winToolBar, "fullscreen")
			QWhatsThis.add(FullScreen, HelpDict["fullscreen"])
			self.winToolButton= QToolButton(self.Icons["window_nofullscreen"], "Tile",
							None, self.onWinToolAction, self.winToolBar, "wintile")
			QWhatsThis.add(self.winToolButton, HelpDict["nofullscreen"])
			self.winToolMenu= QPopupMenu(self.winToolButton)
			self.winToolMenu.setCheckable(1)
			self.winToolMenuText= ["Cascade", "Tile", "Tile Horizontally", "Tile Vertically"]
			self.winToolMenuIndex= []
			for text in self.winToolMenuText:
				self.winToolMenuIndex.append(self.winToolMenu.insertItem(text, self.onWinToolMenu))

			self.winToolMenu.setItemChecked(self.winToolMenuIndex[1], 1)
			self.winToolMenuAction= self.windowTile

			self.winToolButton.setPopup(self.winToolMenu)
			self.winToolButton.setPopupDelay(0)

			
	def onInitPluginsMenu(self, PluginsMenu):
		pass
	
	def onWinToolMenu(self, idx):
		for midx in self.winToolMenuIndex:
			self.winToolMenu.setItemChecked(midx, midx==idx)
		act= self.winToolMenuIndex.index(idx)
		self.winToolButton.setTextLabel(self.winToolMenuText[act])
		if act==0:	self.winToolMenuAction= self.windowCascade
		elif act==1:	self.winToolMenuAction= self.windowTile
		elif act==2:	self.winToolMenuAction= self.windowHorizontal
		elif act==3:	self.winToolMenuAction= self.windowVertical
		self.onWinToolAction()

	def onWinToolAction(self):
		apply(self.winToolMenuAction, ())

	def initMenuBar(self):
		if self._options["MenuFile"]:
		    self.menuFile= QPopupMenu(self.menuBar())
		    idx= self.menuFile.insertItem(self.Icons["fileopen"], "&Open", self.onOpen, Qt.CTRL+Qt.Key_O)
		    self.menuFile.setWhatsThis(idx, HelpDict["fileopen"])
		    idx= self.menuFile.insertItem(self.Icons["filesave"], "&Save", self.onSave, Qt.CTRL+Qt.Key_S)
		    self.menuFile.setWhatsThis(idx, HelpDict["filesave"])
		    
		    self.menuFile.insertItem("Load in &View", self.onLoadInView)
		    
		    self.menuFile.insertItem("Save &as", self.onSaveAs)
		    self.menuFile.insertSeparator()
		    idx= self.menuFile.insertItem(self.Icons["fileprint"], "&Print", self.onPrint, Qt.CTRL+Qt.Key_P)
		    self.menuFile.setWhatsThis(idx, HelpDict["fileprint"])
		    self.menuFile.insertSeparator()
		    self.menuFile.insertItem("&Quit", qApp, SLOT("closeAllWindows()"), Qt.CTRL+Qt.Key_Q)
		    self.menuBar().insertItem('&File',self.menuFile)

		self.onInitMenuBar(self.menuBar())

		if self._options["MenuTools"]:
		    self.menuTools= QPopupMenu()
		    self.menuTools.setCheckable(1)
		    self.connect(self.menuTools, SIGNAL("aboutToShow()"), self.menuToolsAboutToShow)
		    self.menuBar().insertItem("&Tools", self.menuTools)

		if self._options["MenuWindow"]:
		    self.menuWindow= QPopupMenu()
		    self.menuWindow.setCheckable(1)
		    self.connect(self.menuWindow, SIGNAL("aboutToShow()"), self.menuWindowAboutToShow)
		    self.menuBar().insertItem("&Window", self.menuWindow)

		if self._options["MenuHelp"]:
		    self.menuHelp= QPopupMenu()
		    self.menuHelp.insertItem("&About", self.onAbout)
		    self.menuHelp.insertItem("About &Qt",self.onAboutQt)
		    self.menuBar().insertSeparator()
		    self.menuBar().insertItem("&Help", self.menuHelp)

		if self._options["MenuPlugins"]:
		        self.PluginsMenu= QPopupMenu(self.menuBar())
			self.menuBar().insertItem('&Plugins',self.PluginsMenu)
			self.onInitPluginsMenu(self.PluginsMenu)

	def menuWindowAboutToShow(self):
		self.menuWindow.clear()
		if len(self.mdi.windowList())==0: return
		self.menuWindow.insertItem("&Cascade", self.windowCascade)
		self.menuWindow.insertItem("&Tile", self.windowTile)
		self.menuWindow.insertItem("&Tile Horizontally", self.windowHorizontal)
		self.menuWindow.insertItem("&Tile Vertically", self.windowVertical)
		self.menuWindow.insertSeparator()

		num= 0
		self.menuWindowMap= {}
		for window in self.mdi.windowList():
			idx= self.menuWindow.insertItem("&%d %s"%(num, str(window.caption())), self.menuWindowActivated)
			self.menuWindowMap[idx]= window
			num += 1
			if window==self.mdi.activeWindow():
				self.menuWindow.setItemChecked(idx, 1)

	def menuWindowActivated(self, idx):
		self.menuWindowMap[idx].setFocus()

	def menuToolsAboutToShow(self):
		self.menuTools.clear()
		self.menuToolsMap= {}
		for win in self.dockWindows():
			if isinstance(win, DockWindow):
				idx= self.menuTools.insertItem("%s"%str(win.caption()), self.menuToolsActivated)
				self.menuToolsMap[idx]= win
				self.menuTools.setItemChecked(idx, not win.isHidden())
		if len(self.menuToolsMap.keys()):
			self.menuTools.insertSeparator()
			self.menuTools.insertItem("Customize", self.customize)

	def menuToolsActivated(self, idx):
		if self.menuTools.isItemChecked(idx):
			self.menuToolsMap[idx].hide()
		else:
			self.menuToolsMap[idx].show()

	#
	# Mdi windows geometry
	# 
	def windowCascade(self):
		if self.followActiveWindow: self.__disconnectFollow()
		self.mdi.cascade()
		for window in self.mdi.windowList():
			window.resize(0.7*self.mdi.width(),0.7*self.mdi.height())
		if self.followActiveWindow: self.__connectFollow()

	def windowTile(self):
		if self.followActiveWindow: self.__disconnectFollow()
		self.mdi.tile()
		if self.followActiveWindow: self.__connectFollow()

	def windowHorizontal(self):
		#if self.followActiveWindow: self.__disconnectFollow()
		if not len(self.mdi.windowList()): return
		windowheight=float(self.mdi.height())/len(self.mdi.windowList())
		i=0
		for window in self.mdi.windowList():
			window.parentWidget().showNormal()
			window.parentWidget().setGeometry(0, int(windowheight*i),
						self.mdi.width(),int(windowheight))
			window.parentWidget().raiseW()
			i+=1
		self.mdi.update()
		self.update()
		#if self.followActiveWindow: self.__connectFollow()

	def windowVertical(self):
		#if self.followActiveWindow: self.__disconnectFollow()
		if not len(self.mdi.windowList()): return
		windowwidth=float(self.mdi.width())/len(self.mdi.windowList())
		i=0
		for window in self.mdi.windowList():
			window.parentWidget().showNormal()
			window.parentWidget().setGeometry(int(windowwidth*i),0,
						int(windowwidth),self.mdi.height())
			window.parentWidget().raiseW()
			i+=1
		self.mdi.update()
		self.update()
		#if self.followActiveWindow: self.__connectFollow()

	def windowFullScreen(self):
		if len(self.mdi.windowList()):
			self.mdi.activeWindow().showMaximized()

	def __connectFollow(self):
		self.connect(self.mdi, SIGNAL("windowActivated(QWidget*)"), self.onWindowActivated)

	def __disconnectFollow(self):
		self.disconnect(self.mdi, SIGNAL("windowActivated(QWidget*)"), self.onWindowActivated)

	def setFollowActiveWindow(self, follow):
		if follow!=self.followActiveWindow:
			if not follow: self.__disconnectFollow()
			else: self.__connectFollow()
			self.followActiveWindow= follow

	def onWindowActivated(self, win):
		pass

	#
	# Dock windows
	#
	def isCustomizable(self):
		nb= 0
		for win in self.dockWindows():
			nb += isinstance(win, DockWindow)
		return (nb>0)
	
	def customize(self, *args):
		dg= DockPlaceDialog(self, window=self, title="Tool Places")
		dg.exec_loop()

	#
	# Menus customization
	#
	def onInitMenuBar(self, menubar):
		pass

	def onInitFileToolBar(self, toolbar):
		pass

	def onInitToolBar(self):
		pass

	def onInitWinToolBar(self, toolbar):
		pass

	#
	# Menus callback
	#
	def onAboutQt(self):
		QMessageBox.aboutQt(self, "About Qt")

	def onAbout(self):
		QMessageBox.about(self, "MDI",
			"MDI Application Framework\nVersion: "+__version__)

	def onOpen(self):
		QMessageBox.about(self, "Open", "Not implemented")

	def onSave(self):
		QMessageBox.about(self, "Save", "Not implemented")

	def onSaveAs(self):
		QMessageBox.about(self, "SaveAs", "Not implemented")

	def onPrint(self):
		QMessageBox.about(self, "Print", "Not implemented")

#
# Child window (on the MDI canvas)
#
class ChildWindow(QMainWindow):
	def __init__(self, parent=None, name=None, master=None, flags=Qt.WDestructiveClose):
		QMainWindow.__init__(self, parent, name, flags)
		self.parentWindow= parent
		self.masterWindow= master
		if self.parentWindow is not None:
			self.isMdiChild= (self.parentWindow.__class__ is QWorkspace)
		else:
			self.isMdiChild= 0
		self.RepaintsWhileResizing= 0
		self.name= name
		self.setTitle()
		if self.isMdiChild:
			self.resize(0.5*self.parentWindow.width(), 0.5*self.parentWindow.height())
		self.show()

	def setTitle(self):
		if self.masterWindow is not None:
			self.title= str(self.masterWindow.getTitle()) + "-" + self.name
		else:
			self.title= self.name
		self.setCaption(self.title)

	def getTitle(self):
		return self.title


##     def closeEvent(self,ce):
##         if self.Changed==0:
##             ce.accept()
##             if self.parent is not None: self.parent.windows.remove (self)
##             return
##         """
##         rc = qt.QMessageBox.information(self,'Close',
##                     'The document has been changed since the last save.',
##                     'Save Now','Cancel','Leave Anyway',0,1)
##         if rc == 0:
##             self.save()            
##             ce.accept()
##             if self.parent is not None: self.parent.windows.remove (self)
##         elif rc == 2:
##             ce.accept()
##             if self.parent is not None: self.parent.windows.remove (self)
##         else:
##             ce.ignore()
##         """



	def closeEvent(self, ce):
		if self.isMdiChild:
			for window in self.parentWindow.windowList():
				if window.masterWindow is self: window.close(1)
		ce.accept()

	def repaintWhileResizing(self, flag):
		self.RepaintsWhileResizing= flag


#
# DockWindow
#
class DockWindow(QDockWindow):
	def __init__(self, parent, name, defaultdock=Qt.DockLeft):
		QDockWindow.__init__(self, QDockWindow.InDock, parent, name)
		self.mainWindow= parent
		self.visibility= 0
		self.defaultDock= defaultdock
		self.setCloseMode(QDockWindow.Always)
		self.setResizeEnabled(1)
		self.setCaption(name)
		self.connect(self, SIGNAL("visibilityChanged(bool)"), self.__visibilityChanged)
		self.moveTo(self.defaultDock)

	def __visibilityChanged(self, visibility):
		self.visibility= visibility

	def isHidden(self):
		return (self.visibility==0)

	def getDefaultDock(self):
		return self.defaultDock

	def setOrientation(self, o):
		if hasattr(self.widget(), "setOrientation"):
			self.widget().setOrientation(o)
		else:
			QDockWindow.setOrientation(self,o)

	def moveToDefault(self):
		self.moveTo(self.defaultDock)

	def moveTo(self, dockside):
		self.mainWindow.moveDockWindow(self, dockside)
		if dockside==Qt.DockBottom or dockside==Qt.DockTop:
			o= Qt.Horizontal
		else:
			o= Qt.Vertical
		self.setOrientation(o)

#
# Dock Place Dialog
#

# mapping DockPosition / String description
DockAreaName= { Qt.DockTop:"top", Qt.DockLeft:"left", Qt.DockBottom:"bottom", Qt.DockRight:"right", 
		Qt.DockTornOff:"undocked", Qt.DockMinimized:"minimized" }
DockAreaValue= {}
for val,name in DockAreaName.items():
	DockAreaValue[name]= val

class DockPlaceDialog(QDialog):
	def __init__(self, parent=None, name="DockPlaceDialog", title=None, window=None):
		QDialog.__init__(self, parent, name, 0, Qt.WDestructiveClose)
		self.setCaption(title)

		gridLayout= QGridLayout(self, 2, 3)

		self.table= DockPlaceTable(self, window=window)
		gridLayout.addMultiCellWidget(self.table, 0, 0, 0, 2)
		gridLayout.addRowSpacing(0, 10)

		default= QPushButton(self)
		default.setText("Default Place")
		gridLayout.addWidget(default, 1, 0)
		self.connect(default, SIGNAL('clicked()'), self.table.reset)

		apply= QPushButton(self)
		apply.setText("Apply")
		gridLayout.addWidget(apply, 1, 1)
		self.connect(apply, SIGNAL('clicked()'), self.table.update)

		close= QPushButton(self)
		close.setText("Close")
		gridLayout.addWidget(close, 1, 2)
		self.connect(close, SIGNAL('clicked()'), self.close)

		
class DockPlaceTable(QWidget):
	def __init__(self, parent=None, name="DockPlaceTable", fl=0, window=None):
		QWidget.__init__(self, parent, name, 0)

		self.mainWindow= window
		self.dockWindows= []

		layout= QVBoxLayout(self)
		self.table= Table(0, 3, self)
		self.table.verticalHeader().hide()
		self.table.horizontalHeader().setLabel(0, "Tool")
		self.table.horizontalHeader().setLabel(1, "Place")
		self.table.horizontalHeader().setLabel(2, "Hidden")
		self.table.setColumnReadOnly(0, 1)
		layout.addWidget(self.table)

		self.refresh()

	def refresh(self):
		if self.mainWindow is None: return

		nb= 0
		for dw in self.mainWindow.dockWindows():
			nb += isinstance(dw, DockWindow)
		if nb!=self.table.numRows(): self.table.setNumRows(nb)
		if not nb: return

		self.dockWindows= []
		names= DockAreaName.values()
		for da,name in DockAreaName.items():
			for dw in self.mainWindow.dockWindows(da):
				if isinstance(dw, DockWindow):
					nb= len(self.dockWindows)
					self.table.setText(nb, 0, dw.caption())
					self.table.setComboItem(nb, 1, names, name)
					self.table.setCheckItem(nb, 2, None, dw.isHidden())
					self.dockWindows.append(dw)

		self.table.adjustColumn(0)

	def reset(self):
		for dw in self.dockWindows:
			dw.moveToDefault()
		self.refresh()

	def update(self):
		for idx in range(len(self.dockWindows)):
			dw= self.dockWindows[idx]
			hide= self.table.item(idx, 2)
			if hide.isChecked() and not dw.isHidden(): 
				dw.hide()
			else:
				if dw.isHidden(): dw.show()
				place= self.table.item(idx, 1)
				da= DockAreaValue[str(place.currentText())]
				dw.moveTo(da)
		self.refresh()

#
# Table with utility for checkbox/combobox
#
class Table(QTable):
	def setCheckItem(self, row, col, text=None, checked=0):
		item= QCheckTableItem(self, text)
		item.setChecked(checked)
		self.setItem(row, col, item)

	def setComboItem(self, row, col, data, selected=None):
		item= QComboTableItem(self, self.__getStringList(data))
		if selected is not None:
			item.setCurrentItem(str(selected))
		self.setItem(row, col, item)
 
	def __getStringList(self, plist):
		strlist= QStringList()
		for d in plist:
			strlist.append(d)
		return strlist

	def adjustTable(self):
		for col in range(self.numCols()):
			self.adjustColumn(col)

def testmdi():
	class MyApp(ApplicationWindow):
		def onInitMenuBar(self, menubar):
			self.menuFile.insertSeparator(0)
			self.menuFile.insertItem("&New Tool", self.onNewTool, 0, -1, 0)
			self.menuFile.insertItem("&New Window", self.onNewWindow, 0, -1, 0)
			self.menuFile.insertItem("&New Sub-Window", self.onNewSubWindow, 0, -1, 0)

		def onNewSubWindow(self):
			window= self.mdi.activeWindow()
			if window is not None:
				nt= ChildWindow(self.mdi, "sub-window (%d)"%self.nbWin, window)
				self.nbWin+=1

		def onNewWindow(self):
			if not hasattr(self, "nbWin"): self.nbWin= 0
			nt= ChildWindow(self.mdi, "window (%d)"%self.nbWin)
			self.nbWin+=1

		def onNewTool(self):
			if not hasattr(self, "nbTool"): self.nbTool= 0
			nt= DockWindow(self, "tool (%d)"%self.nbTool)
			nt.show()
			self.nbTool+=1
	import sys
	a= MDIApp(sys.argv)
	w= MyApp(name="example")
	a.setMainWidget(w)
	w.show()
	a.connect(a, SIGNAL('lastWindowClosed()'), a, SLOT('quit()'))
	a.exec_loop()

if __name__=='__main__':
	testmdi()
