from PyDVT.DataSelection import OrthoLineSelection, RectSelection, DataSelection
import os.path
import Numeric

class SpecFileMcaSelection(OrthoLineSelection):
	def __init__(self, source, name=None, key=None, limits=None, synchronized=1):
		"""
		Initialization parameters:
			- source: Data Object
			- name: name of the specfile (=source.SourceName if None)
			- key: mca key index (like \"1.1.1\" or \"3.1.5\")
			- limits: a tuple to limit the size of the returned array
				( (0,1024) for example)
		"""
		if name is None: 
			name= source.SourceName
		if key is None: 
			loaded= source.GetSourceLoadedKeys(name)
			if len(loaded): key= loaded[0]
		page= {"SourceName":name, "Key":key}
		if limits is not None:
			xmin= int(min(limits[0], limits[1]))
			xmax= int(max(limits[0], limits[1]))
			position= (xmin,)
			size= (xmax-xmin,)
			limits= (xmin, xmax)
		else:
			position= None
			size= None
		OrthoLineSelection.__init__(self, source, position, size, page, synchronized)
		self.Limits= limits


	def GetOutput(self):
		"""
		Return a dictionnary with:
			- data: numeric array for y data
			- xdata: numeric array for x data
		"""
		ret= OrthoLineSelection.GetOutput(self)
		if ret.has_key("data"):
			length= ret["data"].shape[0]
			xdata= Numeric.arrayrange(length)
			if self.Limits is not None:
				xdata= xdata + self.Limits[0]
			ret["xdata"]= xdata
		return ret


	def GetInfo(self):
		"""
		Retrieve SpecFileMcaSelection information
		"""
		ret= OrthoLineSelection.GetInfo(self)
		ret["limits"]= self.Limits
		ret["legend"]= "%s:%s"%(os.path.basename(self.IndexList[0]["SourceName"]),
					self.IndexList[0]["Key"])
		ret["type"]= "SpecFile"
		return ret


	def Reconfig(self, name=None, key=None, limits=None):
		"""
		Reconfigure the current selection with new parameters.
		A Refresh is called so that linked object are informed of the changes.
		"""
		if name is not None:
			self.IndexList[0]["SourceName"]= name
		if key is not None:
			self.IndexList[0]["Key"]= key
		if limits is not None:
			xmin= int(min(limits[0], limits[1]))
			xmax= int(max(limits[0], limits[1]))
			self.Limits= (xmin, xmax)
			self.Position= (xmin,)
			self.Size= (xmax-xmin,)
		else:
			self.Limits= None
			self.Position= None
			self.Size= None
		self.Refresh()

class SpecFileScanSelection(OrthoLineSelection):
	def __init__(self, source, name=None, key=None, xcounter=None, ycounter=None, synchronized=1):
		if name is None:
			name= source.SourceName
		if key is None:
			loaded= source.GetSourceLoadedKeys(name)
			if len(loaded): key= loaded[0]
		page= {"SourceName": name, "Key": key}

		info= source.GetPageInfo(page)
		cnts= info.get("LabelNames", [])

		(xidx, xcnt)= self.__GetCounter(cnts, xcounter, 0)
		(yidx, ycnt)= self.__GetCounter(cnts, ycounter, 1)

		OrthoLineSelection.__init__(self, source, (yidx,0), (1,"ALL"), page, synchronized)
		self.xSelection= OrthoLineSelection(source, (xidx,0), (1,"ALL"), page, 0)

		self.xcounter= xcnt
		self.ycounter= ycnt

	def __GetCounter(self, cnts, cnt, idx):
		if cnt in cnts:
			return (cnts.index(cnt), cnt)
		else:	
			if idx<len(cnts):
				return (idx, cnts[idx])
			else:	return (idx, None)


	def GetOutput(self):
		ysel= OrthoLineSelection.GetOutput(self)
		if not ysel.has_key("data"): return {}

		if self.xSelection is not None:
			xsel= self.xSelection.GetOutput()
		else:	xsel= {}

		if not xsel.has_key("data"):
			ysel["xdata"]= Numeric.arrayrange(ysel["data"].shape[0])
		else:	ysel["xdata"]= xsel["data"]

		return ysel

	def GetInfo(self):
		info= OrthoLineSelection.GetInfo(self)
		info["xcounter"]= self.xcounter
		info["ycounter"]= self.ycounter
		info["legend"]= "%s:%s:%s:%s"%(self.IndexList[0]["SourceName"],
			self.IndexList[0]["Key"], self.xcounter, self.ycounter)
		return info

	def Reconfig(self, *args):
		print "Not yet implemented"

class SpecFileMeshSelection(RectSelection):
	def __init__(self, source, name=None, key=None, zcounter=None, synchronized=1):
		if name is None:
			name= source.SourceName
		if key is None: 
			loaded= source.GetSourceLoadedKeys(name)
			if len(loaded): key= loaded[0]
		page= {"SourceName": name, "Key": key}

		info= source.GetPageInfo(page)
		cnts= info.get("LabelNames", [])

		(xidx, xcnt)= self.__GetCounter(cnts, None, 0)
		(yidx, ycnt)= self.__GetCounter(cnts, None, 1)
		(zidx, zcnt)= self.__GetCounter(cnts, zcounter, 2)

		RectSelection.__init__(self, source, (0,0,zidx), ("ALL","ALL",1), page, synchronized)
		self.xSelection= OrthoLineSelection(source, (0,0,xidx), ("ALL",1,1), page, 0)
		self.ySelection= OrthoLineSelection(source, (0,0,yidx), (1,"ALL",1), page, 0)

		self.xcounter= xcnt
		self.ycounter= ycnt
		self.zcounter= zcnt

	def __GetCounter(self, cnts, cnt, idx):
		if cnt in cnts:
			return (cnts.index(cnt), cnt)
		else:
			if idx<len(cnts):
				return (idx, cnts[idx])
			else:	return (idx, None)
		
	def GetOutput(self):
		ret= RectSelection.GetOutput(self)
		if self.xSelection is not None:
			xsel= self.xSelection.GetOutput()
			if xsel.has_key("data"):
				ret["xdata"]= xsel["data"]
		if self.ySelection is not None:
			ysel= self.ySelection.GetOutput()
			if ysel.has_key("data"):
				ret["ydata"]= ysel["data"]
		return ret

	def GetInfo(self):
		info= RectSelection.GetInfo(self)
		info["xcounter"]= self.xcounter
		info["ycounter"]= self.ycounter
		info["zcounter"]= self.zcounter
		info["legend"]= "%s:%s:%s:%s:%s"%(os.path.basename(self.IndexList[0]["SourceName"]),
				self.IndexList[0]["Key"], self.xcounter, self.ycounter, self.zcounter)
		info["type"]= "SpecFile"
		return info

	def Reconfig(self, *args):
		print "Not yet implemented"

class SpecFileMcaMeshRoiSelection(RectSelection):
	def __init__(self, source, name=None, key=None, chmin=None, chmax=None, synchronized=1):
		if name is None:
			name= source.SourceName
		if key is None: 
			loaded= source.GetSourceLoadedKeys(name)
			if len(loaded): key= loaded[0]
		page= {"SourceName": name, "Key": key}

		if chmin is None:	chbeg=0
		else:			chbeg= chmin
		if chmax is None:	chend="ALL"
		else:			chend= chmax-chmin

		DataSelection.__init__(self, source, (chbeg,0,0), (chend,"ALL","ALL"), page, 3, synchronized)

		self.chmin= chmin
		self.chmax= chmax

	def GetOutput(self):
		sel= DataSelection.GetOutput(self)
		if not sel.has_key("data"): return {}

		size= sel["data"].shape
		data= Numeric.sum(sel["data"], 2)

		return {"data": data}
		
	def GetInfo(self):
		info= DataSelection.GetInfo(self)
		info["chmin"]= self.chmin
		info["chmax"]= self.chmax
		info["legend"]= "%s:%s:roi:%s:%s"%(os.path.basename(self.IndexList[0]["SourceName"]),
				self.IndexList[0]["Key"], self.chmin, self.chmax)
		info["type"]= "SpecFile"
		return info

	def Reconfig(self, *args):
		print "Not yet implemented"


def test():
	from PyDVT.SpecFileData import SpecFileData

	f= SpecFileData()
	print f.SetSource("/bliss/users/papillon/data/02021201.dat")
	f.LoadSource("1.1.1")
	f.LoadSource("2.1.1", append=1)
	print f.GetSourceLoadedKeys(f.SourceName)

	s= SpecFileMcaSelection(f, key="1.1.1")
	r= s.GetOutput()
	print "XDATA:",r["xdata"][0:20]
	print "YDATA:",r["data"][0:20]

	s.Reconfig(limits=(10,30))
	r= s.GetOutput()
	print "XDATA:",r["xdata"]
	print "YDATA:",r["data"]

	s.Reconfig(key="2.1.1", limits=(0,50))
	r= s.GetOutput()
	print "XDATA:",r["xdata"]
	print "YDATA:",r["data"]

	print "Selection.GetInfo:"
	print s.GetInfo()

if __name__=="__main__":
	test()
