"""
    MeshView.py
    View derived class for 3d image displaying

"""
#XXX: need this for test function
import qt
#XXX
from PyDVT import __version__,__date__,__author__


from View import *
import ImageViewSelect
from Filter import Filter
from Data import Data
from DataSelection import RectSelection

import Numeric
import Command
import spslut

colormap_table={}
NUM_COLORS_8BITS=100
def SetColormapTable():
    for colormap in (spslut.GREYSCALE,spslut.TEMP,spslut.RED,spslut.GREEN,spslut.BLUE,spslut.REVERSEGREY):
        table=Numeric.fromstring(spslut.palette(NUM_COLORS_8BITS,colormap),'l')        
        if GetDisplayProperties()["SPSLUT_MODE"]=="BGRX": table= (Numeric.bitwise_and(table.byteswapped(),0xFFFFFF00)/0x100)
        aux=Numeric.zeros((256,),'l')
        aux[50:(50+NUM_COLORS_8BITS)]=table
        colormap_table[colormap]=list(aux)

################################################################################


class ColormapFilter(Filter):
    """
    View's standard filter 
    """
    def __init__(self,name=None,source=None,synchronized=1,buffer_input=0):
        """
        Parameters:
            name: optional name of the function
            source: Source Filter/DataSelection
            synchronized: non-zero if on-line
            buffer_input: see Filter.__init__
        """
        Filter.__init__(self,source,synchronized,buffer_input)
        self.MinMax=(0,0)
        self.AbsoluteMinMax=(0,0)
	self.Size= (0,0)
        self.Scale="Linear"
        self.Colormap="Temperature"
        self.SpslutScale=spslut.LINEAR
        self.SpslutColormap=spslut.TEMP
        self.AutoScale=1
        self.ColormapChange=0
        self.colormap_table=None
        if name==None: self.name=str(self)
        else: self.name=name
        self.ColormapSelection=None
        self.reduc=1
        self.fastreduc=0
        self.Gamma=3.0

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys:
            "name" : string
            "data" : 2d-NumPy array with the original data
            "image" : String with RGB representation of data
            "size": Size of the image
            "depth": 8 or 32
            "colormap_table": if "depth"==8, palette table, otherwize None           
            "filterchange" : flag indicating just a change in
                             filter parameters has taken place
                             (not the actual data).
                             ImageView always keeps Zoom/Position
                             if this flag is set.
        """        
        sel=self.GetInput()
        if "data" not in sel.keys(): return {"name":self.name}
        if sel["data"] is not None:
            disp_props=GetDisplayProperties()
            if disp_props["DISPLAY_DEPTH"]==8:
                global NUM_COLORS_8BITS
                if sel["data"].typecode() in ("b","1","s"): data=sel["data"].astype("l")
                else: data=sel["data"]
                (image,size,self.MinMax)=spslut.transform(data , (self.reduc,self.fastreduc), 
                                                    (self.SpslutScale, self.Gamma), 'L',
                                                    self.SpslutColormap, self.AutoScale, self.MinMax,(50,49+NUM_COLORS_8BITS))
                                
                #All this pad mess is done because one qt expects 4-byte aligned data.
                #I think one of the constructors of QImage could solve the problem
                #but it is not implemented in PyQt 
                if disp_props["ALIGN_8BIT_IMAGES"]:
                    pad=(4-(size[0] % 4))%4
                    if pad:
                        img_arr=Numeric.reshape(Numeric.fromstring(image,"b"),(size[1],size[0]))
                        image=Numeric.concatenate((img_arr,(Numeric.zeros((size[1],pad),"b"))),1).tostring()
                
                global colormap_table
                if colormap_table=={}: SetColormapTable()
                self.colormap_table=colormap_table[self.SpslutColormap]
            else:
		data= sel["data"]
                (image,size,self.MinMax)=spslut.transform(data, (self.reduc,self.fastreduc), 
                                                    (self.SpslutScale, self.Gamma), disp_props["SPSLUT_MODE"], 
                                                    self.SpslutColormap, self.AutoScale, self.MinMax)            
	    self.Size= size
            if self.AutoScale: self.AbsoluteMinMax=self.MinMax
            ret = {"name":self.name,"image":image,"size":size,"datachanged":not self.ColormapChange,"data":sel["data"]}
            if "xdata" in sel.keys(): ret["xdata"]=sel["xdata"]
            if "ydata" in sel.keys(): ret["ydata"]=sel["ydata"]            
	    ret["zdata"]= data
            ret["depth"]=disp_props["DISPLAY_DEPTH"]
            ret["colormap_table"]=self.colormap_table
	    #XXX For testing only
	    ret["pen"]= Pen((128,128,128),1,"solid")
            return ret
        
        self.AbsoluteMinMax=(0,0)
        return {"name":self.name}


    def GetColormapOutput(self):
	ret= {}
	if self.MinMax==(0,0) or self.Size==(0,0): return ret

	step= float(self.MinMax[1]-self.MinMax[0])/float(self.Size[0])
	line= Numeric.arrayrange(self.MinMax[0], self.MinMax[1], step)
	data= Numeric.array([line for idx in range(10)])

	disp_props= GetDisplayProperties()
        if disp_props["DISPLAY_DEPTH"]==8:
        	global NUM_COLORS_8BITS
		data= data.astype("l")
		(image,size,minmax)= spslut.transform(data, (1,0), (self.SpslutScale, self.Gamma), "L",
						self.SpslutColormap, self.AutoScale, self.MinMax,
						(50,49+NUM_COLORS_8BITS))
		if disp_props["ALIGN_8BIT_IMAGES"]:
			pad=(4-(size[0] % 4))%4
                    	if pad:
                        	img_arr=Numeric.reshape(Numeric.fromstring(image,"b"),(size[1],size[0]))
                        	image=Numeric.concatenate((img_arr,(Numeric.zeros((size[1],pad),"b"))),1).tostring()
	else:
		(image,size,minmax)=spslut.transform(data, (1,0), (self.SpslutScale, self.Gamma), 
					disp_props["SPSLUT_MODE"], self.SpslutColormap, self.AutoScale, self.MinMax)            
	return {"image":image, "size":size, "depth":disp_props["DISPLAY_DEPTH"], 
			"colormap_table":self.colormap_table, "minmax":minmax, "scale":self.Scale}


    def ConnectColormap(self,colormap_sel):
        """
        Connects the ColormapFilter object to a ColormapSelect object. The view object
        will answer to ColormapChange events from Colormap object
        Parameter:
        colormap: colormap object
        """
        self.DisconnectColormap()
        self.ColormapSelection=colormap_sel
        self.ColormapSelection.eh.register("ColormapChange", self.SetColormapParameters)
        

    def DisconnectColormap(self):
        """
        Disconnects the ColormapFilter object from a ColormapSelect object.
        Parameter:
        colormap: colormap object
        """
        if self.ColormapSelection is not None:
            if self.ColormapSelection.eh is not None:
                self.ColormapSelection.eh.unregister("ColormapChange", self.SetColormapParameters)
            self.ColormapSelection=None
        


    def GetColormapParameters(self):
        """
        Gets colormap parameters of the view object, a dictionary, with the keys:
            AbsoluteMinMax: tuple (min,max) with the lowest and highest values of the data
            MinMax: tuple (min,max) with the lowest and highest values used for the colormap
            Scale: "Linear", "Logarithmic", "Gamma"
            Colormap: "GrayScale","Temperature","Red","Green","Blue" or "RevGrey"
            Gamma: Factor for gamma scale
            AutoScale: if set, generates MinMax according to the displayed data
                       (takes highest and lowest values of the data)
        """
        return ({"AbsoluteMinMax":self.AbsoluteMinMax,"MinMax":self.MinMax,"Scale":self.Scale,"Colormap":self.Colormap,"AutoScale":self.AutoScale,"Gamma":self.Gamma})


    def SetColormapParameters(self,colormap_pars):
        """
        Sets colormap parameters of the view object
        Parameters
        colormap_pars: dictionary with the following keys (as described in GetColormapParameters):
            MinMax
            Scale
            Colormap
            AutoScale
            Gamma
            
            AbsoluteMinMax is not considered
        """
        colormap_table_change=0
        if "MinMax"    in colormap_pars.keys(): self.MinMax=colormap_pars["MinMax"]
        if "Scale"     in colormap_pars.keys(): self.Scale=colormap_pars["Scale"]
        if "Colormap"  in colormap_pars.keys(): self.Colormap=colormap_pars["Colormap"]
        if "AutoScale" in colormap_pars.keys(): self.AutoScale=colormap_pars["AutoScale"]
        if "Gamma" in colormap_pars.keys(): self.Gamma=colormap_pars["Gamma"]

        if self.Colormap=="GrayScale":self.SpslutColormap=spslut.GREYSCALE
        elif self.Colormap=="Temperature":self.SpslutColormap=spslut.TEMP
        elif self.Colormap=="Red":self.SpslutColormap=spslut.RED
        elif self.Colormap=="Green":self.SpslutColormap=spslut.GREEN
        elif self.Colormap=="Blue":self.SpslutColormap=spslut.BLUE
        elif self.Colormap=="RevGrey":self.SpslutColormap=spslut.REVERSEGREY
        else: self.SpslutColormap=spslut.TEMP

        if self.Scale=="Linear":self.SpslutScale=spslut.LINEAR
        elif self.Scale=="Logarithmic":self.SpslutScale=spslut.LOG
        elif self.Scale=="Gamma":self.SpslutScale=spslut.GAMMA
        else: self.SpslutScale=spslut.LINEAR                
        
        self.ColormapChange=1
        self.Refresh()
        self.ColormapChange=0


        


    def SetReducParameters(self,reduc,fastreduc):
        """
        Sets reduction parameters 
        Parameters:
        reduc: scale to the reduction
        fastreduc: if non-zero just eliminates rows/cols 
        """
        self.reduc=reduc
        self.fastreduc=fastreduc
        self.Refresh()
    
    def Destroy(self,source=None):
        """
        Virtual: See View.Destroy
        """
        self.DisconnectColormap()
        Filter.Destroy(self,source)


    def DataCoord2SelectionCoord(self,data_coord):
        """
        Overriden to corret coords convertion based on reduction applied.
        """
        source=self.GetSource()
        if source is not None:
            coord=source.DataCoord2SelectionCoord(data_coord)
            return (coord[0]/self.reduc,coord[1]/self.reduc)
        return (-1,-1)


    def SelectionCoord2DataCoord(self,selection_coord):
        """
        Overriden to corret coords convertion based on reduction applied.
        """
        source=self.GetSource()
        if source is not None:
            coord=(selection_coord[0]*self.reduc,selection_coord[1]*self.reduc,)
            return source.SelectionCoord2DataCoord(coord)
        return DataPosition(-1,(-1,-1))


class Image(ColormapFilter):
    """
    View's simplified filter for direct NumPy interface.
    (creates and hides it's own data object)
    """
    def __init__(self,data=None,name=None):
        """
        Parameters:
            data: NumPy 2d array to be displayed
        """
        self.data=Data()
        if data is not None: self.data.AppendPage(array=data)                
        ColormapFilter.__init__(self,name,RectSelection(self.data))
            
    def Destroy(self,source=None):
        """
        If filter gets out of scope, this should be called to destroy
        internally created data object.
        """
        if self.Alive==0: return
        self.Alive=0
        self.data.Destroy()


class MeshImageView(View):
    """
    View derived class for 3d representation of 2d arrays.
    MeshView expects the following keys from it's source (GetOutput
    method):
    "data": 2d NumPy array to be drawn
    "xdata":1d NumPy array with the x values. If None, take's [0 .. x dimention of "data"]
            (have to be ascending)
    "ydata":1d NumPy array with the y values. If None, take's [0 .. y dimention of "data"]
            (have to be ascending)
    "name": Optional. 
    "pen":  Optional. Pen for drawing current source.

    Interface:
    ===========================
    View interface
    SetStyle
    SetMeshPlottingMode
    SetTrackScrollbars
    SetScaleLinear
    SetScaleLog
    SetLabels
    SetPen
    SetXAxis
    SetYAxis
    SetZAxis
    """

    def __init__(self, parent=None, pars={}, **kw):
        """
        See View.__init__
        Parameters:
          parent: Parent window
          pars:   Dictionary with View initialization options
          kw:     keywords to Container initializatregisterion                
        """
        View.__init__(self, parent,pars,**kw)
        self.CreateMenu()
        self.XAxis=self.YAxis=self.ZAxis=None
        self.SetPointer("cross")


    def CreateDrawable(self):
        """
        Virtual: Implements creation of  a Drawable object (which implements the
        Drawable interface)
        """        
        self.Drawable= Drawable_3D(self)
        self.Drawable.Show()
        

    def CreateMenu(self):
        """
        Can be overwritten by derived classes to create a
        different popup menu
        """
        if self.MenuPopup is not None:self.AddMenuSeparator()
        else: self.MenuPopup = Menu(self.Drawable)
        popup=Menu(self.MenuPopup)          
        popup.AddCommand("3D Surface",Command.Command(self.SetStyle,style="3dSurface"),"radiobutton")
        popup.AddCommand("Surface Mesh",Command.Command(self.SetStyle,style="SurfaceMesh"),"radiobutton")
        popup.AddCommand("Shaded 3D Surface",Command.Command(self.SetStyle,style="ShadedSurface"),"radiobutton")
        popup.SetCheckedRadio("3D Surface")
        self.AddMenuPopupCascade("Style",popup)
        popup=Menu(self.MenuPopup)          
        popup.AddCommand("X Lines",Command.Command(self.SetMeshPlottingMode,mode="XLines"),"radiobutton")
        popup.AddCommand("Y Lines",Command.Command(self.SetMeshPlottingMode,mode="YLines"),"radiobutton")
        popup.AddCommand("Both",Command.Command(self.SetMeshPlottingMode,mode="Both"),"radiobutton")
        popup.SetCheckedRadio("Y Lines")
        self.plotmodechk=self.AddMenuPopupCascade("Plot Mode",popup)
        popup=Menu(self.MenuPopup)          
        popup.AddCommand("Linear",self.SetScaleLinear,"radiobutton")
        popup.AddCommand("Logarithmic",Command.Command(self.SetScaleLog),"radiobutton")
        popup.SetCheckedRadio("Linear")
        self.AddMenuPopupCascade("Scale",popup)
        self.trackscrollchk=self.AddMenuPopupItem("Track Scrollbars",self._CheckTrackScrollbars,"checkbutton")

    def _CheckTrackScrollbars(self):
        if self.MenuPopup.IsItemChecked(self.trackscrollchk):
            self.MenuPopup.CheckItem(self.trackscrollchk,0)
            self.SetTrackScrollbars(0)
        else:
            self.MenuPopup.CheckItem(self.trackscrollchk,1)
            self.SetTrackScrollbars(1)

    def SetTrackScrollbars(self,value):
        """
        Sets scrollbar mode
        Parameters:
          value: if non-zero, scrollbars generate continuous redrawing
                 default:0
        """
        self.Drawable.SetTrackScrollbars(value)
        

    def SetStyle(self,style="3dSurface"):
        """
        Sets drawing style
        Parameters:
          style: "3dSurface", "SurfaceMesh" or "ShadedSurface"
                 default:"3dSurface"
        """
        self.Drawable.SetStyle(style)
        if hasattr (self,"plotmodechk"):
            if style=="ShadedSurface": self.MenuPopup.DisableItem(self.plotmodechk)
            else: self.MenuPopup.EnableItem(self.plotmodechk)
        

    def SetMeshPlottingMode(self,mode=""):
        """
        Sets mesh plotting mode
        Parameters:
          mode: "XLines", "YLines" or "Both"
                 default:"YLines" (Z=f(x) for each y)
        """
        self.Drawable.SetMeshPlottingMode(mode)

    def SetScaleLinear(self):
        """
        Sets linear scale for the drawing (default)
        """
        self.Drawable.Linear()

    def SetScaleLog(self,minval=0.000001):
        """
        Sets log scale for the drawing 
        Parameters:
           minvalue: all values less than or equals to 0 are set to it
        """
        self.Drawable.Log(minval)


    def SetLabels(self,title_label="",x_label="",y_label="",z_label=""):
        """
        Sets drawing labels
        Parameters:
          title_label: Title of the drawing
          x_label: Label of x axis
          y_label: Label of y axis
          z_label: Label of z axis
        """
        self.Drawable.SetLabels(title_label,x_label,y_label,z_label)

    def SetPen(self,name,pen):
        """
        Sets pen for a source plotting
        Parameters:
          name: name of the source
          pen: Pen object
        """
        for sel in self.Source:
            if (hasattr(sel,"name") and sel.name==name) or name==str(sel):
                sel.pen=pen                
                self.Drawable.SetFunctionItem(name,"pen",pen)
                self.Update()

    def SetXAxis(self,x_Axis=None):
        """
        Fixes the range of x axis.
        Parameters:
          x_Axis: tuple (min_value,maxvalue)
                  If x_Axis==None (default) range definition is automatic.
        """
        self.XAxis=x_Axis

    def SetYAxis(self,y_Axis=None):
        """
        Fixes the range of y axis.
        Parameters:
          y_Axis: tuple (min_value,maxvalue)
                  If y_Axis==None (default) range definition is automatic.
        """
        self.YAxis=y_Axis

    def SetZAxis(self,z_Axis=None):
        """
        Fixes the range of z axis.
        Parameters:
          z_Axis: tuple (min_value,maxvalue)
                  If z_Axis==None (default) range definition is automatic.
        """
        self.ZAxis=z_Axis

    def _DrawAxis(self,sels):
        minxarr=[]
        maxxarr=[]
        minyarr=[]
        maxyarr=[]
        minzarr=[]
        maxzarr=[]
        
        for sel in sels.values():
            if "data" in sel.keys():
                func=sel["data"]
                if "xdata" not in sel.keys() or  sel["xdata"] is None or len(sel["xdata"])==0:
                    sel["xdata"]=Numeric.arrayrange(func.shape[1])
                    minx,maxx=0,func.shape[1]-1
                else:
                    minx,maxx=sel["xdata"][0],sel["xdata"][-1]                    
                minxarr.append(minx)
                maxxarr.append(maxx)
                
                if "ydata" not in sel.keys() or  sel["ydata"] is None or len(sel["ydata"])==0:
                    sel["ydata"]=Numeric.arrayrange(func.shape[0])
                    miny,maxy=0,func.shape[0]-1
                else:
                    miny,maxy=sel["ydata"][0],sel["ydata"][-1]
                if "name" not in sel.keys():
                    sel["name"]="1"
                minyarr.append(miny)
                maxyarr.append(maxy)

                if func.shape[0] and func.shape[1]:
                    flat=func.flat
                    minz,maxz=min(flat),max(flat)
                else:
                    minz,maxz=0,1
                minzarr.append(minz)
                maxzarr.append(maxz)
        try:
            minx,maxx,miny,maxy,minz,maxz=min(minxarr),max(maxxarr),min(minyarr),max(maxyarr),min(minzarr),max(maxzarr)
        except:
            minx,maxx,miny,maxy,minz,maxz=0,1,0,1,0,1
        if maxz <= minz: maxz=minz+0.000001
        if maxx <= minx: maxx=minx+0.000001
        if maxy <= miny: maxy=miny+0.000001
        if self.XAxis is not None: (minx,maxx)=self.XAxis
        if self.YAxis is not None: (miny,maxy)=self.YAxis
        if self.ZAxis is not None: (minz,maxz)=self.ZAxis
        if self.FlagLockPosition and hasattr(self,"minz"):minz,maxz=self.minz,self.maxz
        self.Drawable.SetEnv(minx,maxx,miny,maxy,minz,maxz)
        self.minz,self.maxz=minz,maxz
        

    def DataChanged(self,source=None):
        """
        Virtual: See View.DataChanged
        """
        self.Drawable.RemoveFunction()
        sels={}
        if self.Source != ():
            for i in self.Source:
                sel=i.GetOutput()
                if "name" not in sel.keys() or sel["name"]==None: sel["name"] = str(i)
                sels[sel["name"]]=sel
            self._DrawAxis(sels)
            for i in sels.keys():
                if "data" in sels[i].keys():
                    self.Drawable.SetFunction(i,sels[i])                
        #else:
        #    self.Drawable.RemoveGraphicObjects()
        self.Update()


    def Redraw(self):
        """
        Virtual: See View.Redraw
        """
        self.Drawable.Redraw()


    def GetSaveFormats(self):
        """
        Returns tuple with supported save formats
        """
        return ("PNG","BMP","PS","JPG")


def test():
	import sys
	import qt

	a= qt.QApplication(sys.argv)
	a.connect(a, qt.SIGNAL("lastWindowClosed()"), a.quit)

	view= MeshImageView(None)

	arr= Numeric.zeros ((500,500))
	for i in range(500):
		for j in range(500):
			arr[i][j]= i+j

	view.SetSource(Image(arr))
	view.SetSize(500,500)
	view.Show()

	a.setMainWidget(view)
	a.exec_loop()

if __name__=="__main__":
	test()
