"""
    DataSelection.py
    Filter derived, basic selections on Data object

"""

from PyDVT import __version__,__date__,__author__


from Filter import *

from Data import DataPosition
import math
import copy
import Numeric
import types




class DataSelection(Filter):
    """
    DataSelection base class.
    This is a Filter derived class that has an Data object as source. 
    This base class implements GetOutput that manages retrieving and assembling
    data from different pages, and can be used by derived classes to perform more
    complex operations.
    """
    def __init__(self,source=None,position=None,size=None,index_list=None,dimentions=None,synchronized=1):
        """
        Parameters:
            source= source Data object 
            position= initial coordinates (a tuple (x,),(x,y) or (x,y,z), according to the dimention of the page)
            size= size of the slice (a tuple (x,),(x,y) or (x,y,z), according to the dimention of the page)
            index_list= page or list of the pages to take the selection on (if None, takes page=0)
                        Each index can be either an integer meaning the sequencial position of the page
                        or a dictionary that logically index the page based on keys of
                        the page's Info dictionary.            
            dimentions= If declared, try to force the number of dimentions of the returned selection 
            synchronized: See Filter.__init__
        """
        Filter.__init__(self,source,synchronized)
        
        self.Position=position
        self.Size=size
        self.IndexList=index_list
        self.EvalSize=self.EvalSize=None
                
        if self.IndexList is None:
            self.IndexList=[0]
        elif self.IndexList == "ALL":
            self.IndexList=range(source.GetNumberPages())
        elif type(self.IndexList) is types.IntType or type(self.IndexList) is types.DictType:
            self.IndexList=[self.IndexList]
            
        self.Dimentions=dimentions
        
        

    def Reconfig(self,position=None,size=None,index_list=None):
        """
        Change position parameters and generates a refresh
        Parameters:
            position: if not None changes position property (see __init__)
            size= if not None changes size property (see __init__)
            index_list= if not None changes index_list property (see __init__) 
        """
        if position is not None: self.Position=position
        if size is not None: self.Size=size
        if index_list is not None:
            self.IndexList=index_list
            if self.IndexList == "ALL":
                self.IndexList=range(self.GetSource().GetNumberPages())
            elif type(self.IndexList) is types.IntType or type(self.IndexList) is types.DictType:
                self.IndexList=[self.IndexList]
        self.Refresh()


            
#################################
#       OVERRIDABLES
#################################
            

    def DataCoord2SelectionCoord(self,data_coord, trim=1):
        """
        Implements convertion from data coordinates to selection coordinates.
        Must be overriden, since data selections are filters with Data object as sources.
        (default implementation doesn't work)
        Up to derived classes.
        """
        return None


    def SelectionCoord2DataCoord(self,selection_coord):        
        """
        Implements convertion from selection coordinates to data coordinates.
        Must be overriden, since data selections are filters with Data object as sources.
        (default implementation doesn't work)
        Up to derived classes.
        """
        return None
            

    def DataChanged(self,(source,page_list)):
        """
        Data change event.
        Verifies if pages the selection have been changed and if so calls derived DataChange.
        If page_list is a dictionary having keyword "reconfiguremoithis", method
        reconfig is called, to point to the page  index_list=page_list["reconfiguremoithis"]
        """
        if source!=self.GetSource(): return
        changed=0
        if   page_list=="ALL": changed=1
        elif (type(page_list)==type({})   and page_list.has_key("reconfiguremoithis")  ):
            if( [ page_list["reconfiguremoithis"] ]  != self.IndexList and  page_list["reconfiguremoithis"] != self.IndexList ):
                 self.Reconfig( index_list=page_list["reconfiguremoithis"]  )
        elif (self.IndexList is None) or (self.IndexList=="ALL"): changed=1
        else:
            for i in self.IndexList:
                if source.GetPageListIndex(i) in page_list:
                    changed=1
                    break
        
        if changed==1: self.Refresh()




#################################
#            VIRTUALS
#################################
    def GetOutput(self):
        """
        Returns a dictionary, selection over source Data object, based on initialization parameters:
        position,size,index_list and dimentions.
        This method takes an multi-dimentional orthogonal slice from a given number of pages.
        Keys:
            "data": NumPy array
        If no selection or error happens returns {}
        """
        source=self.GetSource()
        if source is None: return {}
        try:
            self.EvalPos = Pos=self.Position        
            if self.Size is None: self.EvalSize=None
            else: self.EvalSize = list(copy.copy(self.Size))
            shape_ret=[]
            
            FirstPageDim=source.GetPageDimention(self.IndexList[0])
            FirstPageSize=source.GetPageSize(self.IndexList[0])
            if (FirstPageDim==1):                
                if self.EvalPos is None: self.EvalPos=(0,)                
                if self.EvalSize is None:
                    self.EvalSize=[FirstPageSize[0]-self.EvalPos[0],]
                else:
                    if self.EvalSize[0]=="ALL": self.EvalSize[0]=FirstPageSize[0]-self.EvalPos[0]
            elif (FirstPageDim==2):
                if self.EvalPos is None: self.EvalPos=(0,0)
                if self.EvalSize is None:
                    self.EvalSize=[FirstPageSize[0]-self.EvalPos[0],FirstPageSize[1]-self.EvalPos[1]]
                else:
                    if self.EvalSize[0]=="ALL": self.EvalSize[0]=FirstPageSize[0]-self.EvalPos[0]
                    if self.EvalSize[1]=="ALL": self.EvalSize[1]=FirstPageSize[1]-self.EvalPos[1]
            elif (FirstPageDim==3):
                if self.EvalPos is None: self.EvalPos=(0,0,0)
                if self.EvalSize is None:
                    self.EvalSize=(FirstPageSize[0]-self.EvalPos[0],FirstPageSize[1]-self.EvalPos[1],FirstPageSize[2]-self.EvalPos[2])
                else:
                    if self.EvalSize[0]=="ALL": self.EvalSize[0]=FirstPageSize[0]-self.EvalPos[0]
                    if self.EvalSize[1]=="ALL": self.EvalSize[1]=FirstPageSize[1]-self.EvalPos[1]
                    if self.EvalSize[2]=="ALL": self.EvalSize[2]=FirstPageSize[2]-self.EvalPos[2]
            if self.Dimentions is None: self.Dimentions=FirstPageDim
            

            if self.Dimentions==4:
                if len(self.IndexList)>1:
                    shape_ret.append(len(self.IndexList))                
                shape_ret.append(self.EvalSize[2])
                shape_ret.append(self.EvalSize[1])
                shape_ret.append(self.EvalSize[0])

            if self.Dimentions==3:
                if len(self.IndexList)>1:
                    if self.EvalSize[2]==1:  self.EvalSize[2] = len(self.IndexList)
                    if self.EvalSize[1]==1:  self.EvalSize[1] = len(self.IndexList)
                    else: self.EvalSize[0] = self.EvalSize[0]*len(self.IndexList)
                shape_ret.append(self.EvalSize[2])
                shape_ret.append(self.EvalSize[1])
                shape_ret.append(self.EvalSize[0])

            elif self.Dimentions==2:
                if len(self.IndexList)>1:
                    if self.EvalSize[1]==1:  self.EvalSize[1] = len(self.IndexList)
                    else: self.EvalSize[0] = self.EvalSize[0]*len(self.IndexList)
                shape_ret.append(self.EvalSize[1])
                shape_ret.append(self.EvalSize[0])
            else:
                size=1
                for x in self.EvalSize: size=size*x
                shape_ret.append(size*len(self.IndexList))
            ret=None
            for i in self.IndexList:
                arr=source.GetPageArrayRegion(Pos,self.Size,i,self.Dimentions)
                if ret is None: ret=arr
                else: ret=Numeric.concatenate((ret,arr))
            ret=Numeric.reshape(ret,(shape_ret))
        except:
            return {}
        
        return {"data":ret}
            


    def GetInfo(self):
        """
        Returns a dictionary with initialization parameters for this data selection.
        Can Used by views to retrieve data page's info (through "index_list")
        Can be overriden if other information is needed.
        """
        return {"position":self.Position,"size":self.Size,"index_list":self.IndexList,"dim":self.Dimentions}


class RectSelection(DataSelection):
    """
    Simple rectangle data selection
    """
    #TODO: Generalize Coord Convertions (works just for a 2d inside a page)
    def __init__(self,source=None,position=None,size=None,page=None,synchronized=1):
        """
        See DataSelection.__init__
        """
        DataSelection.__init__(self,source,position,size,page,2,synchronized)            
        
    def DataCoord2SelectionCoord(self,data_coord, trim=1):
        """
        See DataSelection.DataCoord2SelectionCoord
        """
        page=data_coord.PageIndex
        pos=data_coord.PageCoord
        if page != self.IndexList[0]: return (-1,-1)
        if self.Position is None: position=[0,0]
        else:   position=self.Position
        if(trim):
            xval=max(pos[0]-position[0],0)
            yval=max(pos[1]-position[1],0)
        else:
            xval=(pos[0]-position[0])
            yval=(pos[1]-position[1])
        return ((xval,yval))


    def SelectionCoord2DataCoord(self,selection_coord):
        """
        See DataSelection.SelectionCoord2DataCoord
        """
        ret=DataPosition()
        if self.Position is None: position=[0,0]
        else:   position=self.Position
        xval=selection_coord[0]+position[0]
        yval=selection_coord[1]+position[1]
        ret.PageIndex=self.IndexList[0]
        ret.PageCoord=(xval,yval)        
        return ret



class XYRectSelection(RectSelection):
    """
    Rectangle derived class thar includes "xdata" and "ydata" in output
    based in position and size of the selection.
    """
    def GetOutput(self):
        """
        See DataSelection.GetOutput
        """
        sel=RectSelection.GetOutput(self)
        if sel=={}: return {}
        if self.EvalPos is not None and self.EvalSize is not None:
            sel["xdata"]=Numeric.arrayrange(self.EvalPos[0],self.EvalPos[0]+self.EvalSize[0])
            sel["ydata"]=Numeric.arrayrange(self.EvalPos[1],self.EvalPos[1]+self.EvalSize[1])        
        return sel


class MultiPageRectSelection(RectSelection):
    """
    Simple rectangle data selection .
    This RectSelection derived class allows ViewSelect object
    to draw over views displaying different pages
    (doesn't mind page to perform DataCoord2SelectionCoord)
    """
    def DataCoord2SelectionCoord(self,data_coord, trim=1):
        """
        See DataSelection.DataCoord2SelectionCoord
        """
        page=data_coord.PageIndex
        pos=data_coord.PageCoord
        if self.Position is None: position=[0,0]
        else:   position=self.Position
        if(trim):
            xval=max(pos[0]-position[0],0)
            yval=max(pos[1]-position[1],0)
        else:
            xval=(pos[0]-position[0])
            yval=(pos[1]-position[1])
        return ((xval,yval))

class PointSelection(DataSelection):
    """
    Simple point selection
    """
    def __init__(self,source=None,position=None):
        """
        See DataSelection.__init__
        """
        DataSelection.__init__(self,source,position.PageCoord,None,position.PageIndex,1)        


    def GetOutput(self):
        """
        See DataSelection.GetOutput
        """
        source=self.GetSource()
        if source is None: return {}
        if self.Position is None: return {}
        return {"value":source.GetCoordValue(DataPosition(self.IndexList[0],self.Position))}


class OrthoLineSelection(DataSelection):
    """
    Simple orthogonal line data selection
    """
    def __init__(self,source=None,position=None,size=None,page=None,synchronized=1):
        """
        See DataSelection.__init__
        """
        DataSelection.__init__(self,source,position,size,page,1,synchronized)        

    
    #TODO: Coord Convertions




class XYOrthoLineSelection(OrthoLineSelection):
    """
    Orthogonal line data selection with xdata.
    Data changed events are rised just if ydata change.
    If xdata selection is not valid, GetOutput returns {}
    """
    def __init__(self,source=None,position=None,size=None,page=None,xposition=None,xsize=None,xpage=None,synchronized=1):
        """
        See DataSelection.__init__
        """
        OrthoLineSelection.__init__(self,source,position,size,page,synchronized)
        self.XAxis=DataSelection(source,xposition,xsize,xpage,1,synchronized)


    def GetOutput(self):
        """
        See DataSelection.GetOutput
        """
        sel=OrthoLineSelection.GetOutput(self)
        if sel=={}: return {}
        xdata=self.XAxis.GetOutput()
        if "data" not in xdata.keys(): return {}
        sel["xdata"]=xdata["data"]
        return sel


class LineSelection(DataSelection):
    """
    Generic line data selection
    """
    #TODO: Works just inside a page
    def __init__(self,source=None,begin=None,end=None,page=None,synchronized=1):
        """
        See DataSelection.__init__
        """
        self.Begin=begin
        self.End=end
        self.Position=begin
        size=[abs(self.Begin[0]-self.End[0]),abs(self.Begin[1]-self.End[1])]
        DataSelection.__init__(self,source,begin,size,page,1,synchronized)        


    def Reconfig(self,begin=None,end=None,index_list=None):
        """
        Change position parameters and generates a refresh
        Parameters:
            position: if not None changes position property (see __init__)
            size= if not None changes size property (see __init__)
            index_list= if not None changes index_list property (see __init__) 
        """
        if begin is not None: self.Begin=begin
        if end is not None: self.End=end
        self.Position=begin
        self.Size=[abs(self.Begin[0]-self.End[0]),abs(self.Begin[1]-self.End[1])]
        if index_list is not None:
            self.IndexList=[index_list]
        self.Refresh()


    def GetOutput(self):        
        """
        See DataSelection.GetOutput
        """
        source=self.GetSource()
        if source is None: return {}
        if self.Position is None: return {}
        
        sizearray=math.hypot(self.Size[0],self.Size[1])
        
        if (sizearray):
            array=source.GetPageArray(self.IndexList[0])
            ret=Numeric.array(range(sizearray),array.typecode())
            senang=(self.End[0]-self.Position[0])/sizearray
            cosang=(self.End[1]-self.Position[1])/sizearray
            
            for i in range(sizearray):
              offx=i*senang+0.5
              offy=i*cosang+0.5
              ret[i]=array[int(self.Position[1] + offy)][int(self.Position[0] + offx)]
        else:
            return {}
        return {"data":ret}

    #TODO: Coord Convertions
            
