from qt import *
from qttable import *

from GraphWindow import PenStyles, PenColors, getColorName, getColorRgb, YScales
from PyDVT.GraphView import Pen

import EventHandler

class McaFilterInfoTable(QWidget):
	def __init__(self, parent=None, name="PlotList", fl=0):
		QWidget.__init__(self, parent, name, fl)

		self.name	= None
		self.graphNames	= []
	
		self.eh	= EventHandler.EventHandler()
		self.addFilterEvent= self.eh.create("addFilterEvent")
		self.delFilterEvent= self.eh.create("delFilterEvent")

		self.mainLayout= QHBoxLayout(self)

		# --- table with plot list
		self.table= QTable(self)
		self.table.setSelectionMode(QTable.MultiRow)
		self.table.setFocusStyle(QTable.FollowStyle)

		self.table.setNumCols(7)
		Hheader= self.table.horizontalHeader()
		Hheader.setLabel(0, "Type")
		self.table.setColumnReadOnly(0, 1)
		Hheader.setLabel(1, "Name")
		self.table.setColumnReadOnly(0, 1)
		Hheader.setLabel(2, "PlotColor")
		self.table.setColumnReadOnly(0, 0)
		Hheader.setLabel(3, "PlotLine")
		self.table.setColumnReadOnly(0, 0)
		Hheader.setLabel(4, "YScale")
		self.table.setColumnReadOnly(0, 0)
		Hheader.setLabel(5, "YFactor")
		self.table.setColumnReadOnly(0, 0)
		Hheader.setLabel(6, "Normalization")
		self.table.setColumnReadOnly(0, 0)

		self.connect(self.table, SIGNAL("valueChanged(int,int)"), self.__valueChanged)
		self.mainLayout.addWidget(self.table)

		# --- buttons
		butWidget	= QWidget(self)
		butLayout	= QVBoxLayout(butWidget)
		removeButton	= QPushButton("Remove", butWidget)
		movetoButton	= QToolButton(butWidget)
		movetoButton.setSizePolicy(QSizePolicy(QSizePolicy.Minimum, QSizePolicy.Fixed))
		movetoButton.setText("Move To")
		self.graphMenu	= QPopupMenu(movetoButton)
		self.connect(self.graphMenu, SIGNAL("aboutToShow()"), self.__graphMenu)
		movetoButton.setPopup(self.graphMenu)
		movetoButton.setPopupDelay(0)

		butLayout.addWidget(removeButton)
		butLayout.addWidget(movetoButton)

		self.mainLayout.addWidget(butWidget)

		self.connect(removeButton, SIGNAL("clicked()"), self.__removeClicked)


	def setGraphNames(self, graphNames):
		if graphNames != self.graphNames[:-1]:
			self.graphNames= graphNames + ["New"]

	def connectTo(self, graph, name):
		self.graph= graph
		self.name = name
		graph.eh.register("SourceChange", self.setGraphFilters)

	def setGraphFilters(self, graphfilters):
		self.table.setNumRows(0)
		self.names= []

		if graphfilters is not None:
			for filter in graphfilters:
				self.addRow(filter)
			self.__adjustTable()

	def addRow(self, filter):
		graphinfo= filter.GetInfo()
		datainfo= filter.GetDataSelection().GetInfo()

		self.table.setNumRows(self.table.numRows()+1)	
		row= self.table.numRows()-1
		self.names.append(graphinfo["name"])

		self.table.setText(row, 0, str(datainfo.get("SourceType", "-")))
		self.table.setText(row, 1, str(datainfo.get("Legend", "-")))
		self.table.setItem(row, 2, 
			self.__getComboItem([col[0] for col in PenColors], getColorName(graphinfo["pen"].color)))
		self.table.setItem(row, 3, self.__getComboItem(PenStyles, graphinfo["pen"].style))
		self.table.setItem(row, 4, self.__getComboItem(YScales, YScales[graphinfo["yscale"]]))
		self.table.setText(row, 5, str(graphinfo.get("yfactor", 1)))
		self.table.setItem(row, 6, self.__getComboItem(graphinfo["norm"].keys()+["none"], 
						(graphinfo.get("active_norm") or "none")))

	def __getComboItem(self, data, selected=None):
		item= QComboTableItem(self.table, self.__getStringList(data))
		if selected is not None:
			item.setCurrentItem(data.index(selected))
		return item

	def __getStringList(self, plist):
		strlist= QStringList()
		for d in plist:
			strlist.append(d)
		return strlist

	def __adjustTable(self):
	 	for col in range(self.table.numCols()):
			self.table.adjustColumn(col)

	def __removeClicked(self):
		filters= ()
		names= ()
		for row in range(self.table.numRows()):
			if self.table.isRowSelected(row):
				names += (self.names[row],)
				filters += (self.graph.GetFilter(self.names[row]),)
		if len(filters):
			self.graph.delFilter(names)
			self.eh.event(self.delFilterEvent, filters)

	def __valueChanged(self, row, col):
		if col==2 or col==3:	self.__penChanged(row)
		elif col==4:		self.__yscaleChanged(row)
		elif col==5:		self.__yfactorChanged(row)
		elif col==6:		self.__normChanged(row)
		else: print "__valueChanged not handled", row, col

	def __penChanged(self, row):
		name= self.names[row]
		color= getColorRgb(str(self.table.item(row, 2).currentText()))
		style= str(self.table.item(row, 3).currentText())
		self.graph.SetPen(name, Pen(color, 1, style))

	def __yscaleChanged(self, row):
		name= self.names[row]
		yscale= YScales.index(str(self.table.item(row, 4).currentText()))
		self.graph.SetYScale(name, yscale)

	def __yfactorChanged(self, row):
		name= self.names[row]
		sfactor= str(self.table.text(row, 5))
		filter= self.graph.GetFilter(name)
		try:
			yfactor= float(sfactor)
			filter.SetYFactor(yfactor)
			filter.Refresh()
		except:
			QMessageBox.critical(self, "ERROR in YFactor", \
				"YFactor should be a float number.\nReset it to old value")
			self.table.setText(row, 5, str(filter.yscale))
		
	def __normChanged(self, row):
		name= self.names[row]
		norm= str(self.table.item(row, 6).currentText())
		if norm=="none": norm=None
		filter= self.graph.GetFilter(name)
		filter.SetActiveNorm(norm)
		filter.Refresh()

	def __graphMenu(self):
		self.graphMenu.clear()
		self.graphMenuIndex= {}
		for graph in self.graphNames:
			if graph!=self.name:
				idx= self.graphMenu.insertItem(graph, self.__movetoGraph)
				self.graphMenuIndex[idx]= graph

	def __movetoGraph(self, idx):
		graph= self.graphMenuIndex[idx]
		filters= ()
		names= ()
		for row in range(self.table.numRows()):
			if self.table.isRowSelected(row):
				names += (self.names[row],)
				filters += (self.graph.GetFilter(self.names[row]),)
		if len(filters):
			self.graph.delFilter(names)
			self.eh.event(self.addFilterEvent, filters, graph)
