"""numarray: The big enchilada numeric module

"""
import sys as _sys
import types, math, os.path
import operator as _operator
import copy as _copy
import warnings as _warnings
from math import pi, e

import memory
import generic as _gen
import _bytes
import _numarray
import _ufunc
import _sort
import numerictypes as _nt

# rename built-in function type so not to conflict with keyword
_type = type

MAX_LINE_WIDTH = None
PRECISION = None
SUPPRESS_SMALL = None

PyINT_TYPES = {
            types.IntType: 1,
            types.LongType: 1,
            }
PyREAL_TYPES = {
            types.IntType: 1,
            types.LongType: 1,
            types.FloatType: 1,
            }

# Python numeric types with values indicating level in type hierarchy
PyNUMERIC_TYPES = {
            types.IntType: 0,
            types.LongType: 1,
            types.FloatType: 2,
            types.ComplexType: 3
            }

# Mapping back from level to type
PyLevel2Type = {}
for key, value in PyNUMERIC_TYPES.items():
    PyLevel2Type[value] = key
del key, value

# Mapping from Python to Numeric types
Py2NumType = {
            types.IntType:   _nt.Long,
            types.LongType:  _nt.Long,
            types.FloatType: _nt.Float,
            types.ComplexType: _nt.Complex
            }

def array2list(arr):
    return arr.tolist()

# array factory functions
def _maxtype(args):
    """Find the maximum scalar numeric type in the arguments.

    An exception is raised if the types are not python numeric types.
    """
    if not len(args):
        return None
    return PyLevel2Type[_numarray._maxtype(args)]

def _storePyValueInBuffer(buffer, Ctype, index, value):
    """Store a python value in a buffer, index is in element units, not bytes"""
    # Do not use for complex scalars!
    Ctype._conv.fromPyValue(value, buffer._data,
                            index*Ctype.bytes, Ctype.bytes, 0)

def _storePyValueListInBuffer(buffer, Ctype, valuelist):
    # Do not use for complex values!
    for i in xrange(len(valuelist)):
        _storePyValueInBuffer(buffer, Ctype, i, valuelist[i])

def _fillarray(size, start, delta, type=None):
    ptype = _maxtype((start, delta))
    if PyINT_TYPES.has_key(ptype):
        ptype = _nt.Long
    elif PyREAL_TYPES.has_key(ptype):
        ptype = _nt.Float
    else:
        ptype = _nt.Complex
    if type:
        outtype = getType(type)
        if _isComplexType(ptype) and not _isComplexType(outtype):
            raise TypeError("outtype must be a complex type")
    else:
        outtype = ptype
        
    if outtype > ptype: # Hack for Int64/UInt64 on 32-bit platforms.
        ptype = outtype
        
    if _isComplexType(outtype):
        # Not memory efficient at the moment
        real = _fillarray(size, complex(start).real, complex(delta).real,
                type = _realtype(ptype))
        imag = _fillarray(size, complex(start).imag, complex(delta).imag,
                type = _realtype(ptype))
        outarr = ComplexArray((size,), outtype)
        outarr.getreal()[:] = real
        outarr.getimag()[:] = imag
    else:
        # save parameters in a buffer
        parbuffer = ufunc._bufferPool.getBuffer()
        _storePyValueListInBuffer(parbuffer, ptype, [start, delta])
        cfunction = _sort.functionDict[repr((ptype.name, 'fillarray'))]
        outarr = NumArray((size,), outtype)
        if ptype == outtype:
            # no conversion necessary, simple case
            _ufunc.CheckFPErrors()
            cfunction(size, 1, 1, ((outarr._data, 0), (parbuffer._data, 0)))
            errorstatus = _ufunc.CheckFPErrors()
            if errorstatus:
                ufunc.handleError(errorstatus, " in fillarray")
        else:
            # use buffer loop
            convbuffer = ufunc._bufferPool.getBuffer()
            convfunction = ptype._conv.astype[outtype.name]
            bsize = len(convbuffer._data)/ptype.bytes
            iters, lastbsize = divmod(size, bsize)
            _ufunc.CheckFPErrors()

            outoff = 0
            for i in xrange(iters + (lastbsize>0)):
                if i == iters:
                    bsize = lastbsize
                cfunction(bsize, 1, 1,
                          ((convbuffer._data, 0), (parbuffer._data, 0)))
                convfunction(bsize, 1, 1, 
                             ((convbuffer._data, 0), (outarr._data, outoff)))
                outoff += bsize*outtype.bytes
                start += delta * bsize
                _storePyValueListInBuffer(parbuffer, ptype, [start, delta])
            errorstatus = _ufunc.CheckFPErrors()
            if errorstatus:
                ufunc.handleError(errorstatus, " in fillarray")
    return outarr

def NewArray(shape=None, type=None, *args, **keys):
    """NewArray() creates a new array of arbitrary (real or complex) class."""
    if _isComplexType(type):
        return ComplexArray(shape=shape, type=type, *args, **keys)
    else:
        return NumArray(shape=shape, type=type, *args, **keys)

def _frontseqshape(seq):
    """Find the length of all the first elements, return as a list"""
    if not len(seq):
        return (0,)
    try:
        shape = []
        while 1:
            shape.append(len(seq))
            seq = seq[0]
    except TypeError:
        return shape

def fromlist(seq, type=None, shape=None):
    """fromlist(seq, type=None, shape=None) creates a NumArray from
    the sequence 'seq' which must be a list or tuple of python numeric
    types.  If type is specified, it is as the type of the resulting
    NumArray.  If shape is specified, it becomes the shape of the result
    and must have the same number of elements as seq.
    """
    for x in seq:
        if not isinstance(x, _gen.NDArray):
            break
    else:                
        if len(seq): # sequence of NDArray
            l = list(seq)
            for i in range(len(l)):
                l[i] = l[i][_gen.NewAxis,...]
            arr = _gen.concatenate(l)
            if type is not None:
                arr = arr.astype(type)
            if shape is not None:
                arr.setshape(shape)
            return arr

    if not len(seq) and type is None:
        type = _nt.Long
        
    highest_type = _maxtype(seq)
    tshape = _frontseqshape(seq)
    if shape is not None and _gen.product(shape) != _gen.product(tshape):
        raise ValueError("shape incompatible with sequence")
    ndim = len(tshape)
    if ndim <= 0:
        raise TypeError("Argument must be a sequence")
    if type is None:
        type = Py2NumType.get(highest_type)
    if type is None:
        raise TypeError("Cannot create array of type %s" % highest_type.__name__)
    tshape = tuple(tshape)
    arr = NewArray(shape=tshape, type=type)
    arr.fromlist(seq)
    # _numarray._setarray(arr, seq)
    if shape is not None:
        arr.setshape(shape)
    return arr

def getTypeObject(sequence, type, typecode):
    """getTypeObject computes the typeObject for 'sequence' if both 'type' and
    'typecode' are unspecified.  Otherwise,  it returns the typeObject specified by
    'type' or 'typecode'.
    """
    if type is not None and typecode is not None and type != typecode:
        raise ValueError("Can't define both 'type' and 'typecode' for an array.")
    elif type is not None:              # Still might be a string or typecode
        return getType(type)
    elif typecode is not None:
        return getType(typecode)
    elif isinstance(sequence, _gen.NDArray):  # handle array([])
        return sequence.type()
    elif isinstance(sequence, (types.ListType, types.TupleType)) and len(sequence) == 0:
        return _nt.Long
    else:
        if isinstance(sequence, (types.IntType, types.LongType,
                                 types.FloatType, types.ComplexType)):
            sequence = [sequence]
        return Py2NumType[ _maxtype(sequence) ]

def array(sequence=None, typecode=None, copy=1, savespace=0,
          type=None, shape=None, buffer=None):
    """array() constructs a NumArray by calling NumArray, one of its
    factory functions (fromstring, fromfile, fromlist), or by making a
    copy of an existing array.  If copy=0, array() will create a new
    array only if

    sequence and buffer  are interchangeable and specify the contents
                         or storage for the array; buffer is
                         deprecated.

    type and typecode    are interchangeable and define the array type
                         using either a type object or string.

    copy                  when sequence is an array, copy determines
                          whether a copy or the original is returned.

    savespace            is ignored by numarray.
    
    shape                defines the shape of the array object and is
                         necessary when not implied by the sequence
                         parameter.
    """
    # buffer interchangeable with sequence
    if buffer is not None and sequence is not None and (buffer != sequence):
        raise ValueError("Cannot specify both buffer and sequence")

    if buffer is not None:
        _warnings.warn("The 'buffer' keyword is deprecated.  Use 'sequence' instead.",
                       DeprecationWarning, stacklevel=2)
        sequence = buffer
        
    if buffer is None and sequence is None and shape is None:
        return None

    if isinstance(shape, types.IntType):
        shape = (shape,)
        
    if sequence is None and (shape is None or type is None):
        raise ValueError("Must define shape and type if sequence is None")
    
    if isinstance(sequence, _gen.NDArray):
        if type is None and typecode is None:
            if copy:
                a = sequence.copy()
            else:
                a = sequence
        else:
            type = getTypeObject(sequence, type, typecode) 
            a = sequence.astype(type) # make a re-typed copy
        if shape is not None:
            a.setshape(shape)
        return a

    type = getTypeObject(sequence, type, typecode) 
        
    if sequence is None or _gen.SuitableBuffer(sequence):
        return NewArray(buffer=sequence, shape=shape, type=type)
    elif isinstance(sequence, types.StringType):
        return fromstring(sequence, shape=shape, type=type)
    elif isinstance(sequence, (types.ListType, types.TupleType)):
        return fromlist(sequence, type, shape)
    elif PyNUMERIC_TYPES.has_key(_type(sequence)):
        if shape and shape != ():
            raise ValueError("Only shape () is valid for a rank 0 array.")
        return fromlist([sequence], shape=(), type=type or
                        Py2NumType[_type(sequence)])
    elif isinstance(sequence, types.FileType):
        return fromfile(sequence, type=type, shape=shape)
    else:
        try:
            return sequence.__array__(type)
        except AttributeError:
            raise ValueError("Unknown input type")
    
def inputarray(seq, type=None, typecode=None):
    """inputarray(seq, type=None)  converts scalars, lists and tuples to
    numarray if possible, passes NumArrays thru making copies only to convert
    types, and raises a TypeError for everything else.
    """
    return array(seq, type=type, typecode=typecode, copy=0)

asarray = inputarray   # Alias for Numeric compatability

def getType(type):
    """Return the numeric type object for type

    type may be the name of a type object or the actual object
    """
    if isinstance(type, _nt.NumericType):
        return type
    try:
        return _nt.typeDict[type]
    except KeyError:
        raise TypeError("Not a numeric type")

def fromstring(datastring, type, shape=None):
    """Create an array from binary data contained in a string (by copying)"""
    type = getType(type)
    if not shape:
        size, rem = divmod(len(datastring), type.bytes)
        if rem:
            raise ValueError("Type size inconsistent with string length")
        else:
            shape = (size,) # default to a 1-d array
    elif _type(shape) is types.IntType:
        shape = (shape,)
    if len(datastring) != (_gen.product(shape)*type.bytes):
        raise ValueError("Specified shape and type inconsistent with string length")
    arr = NewArray(shape=shape, type=type)
    strbuff = buffer(datastring)
    nelements = arr.nelements()
    # Currently uses only the byte-by-byte copy, should be optimized to use
    # larger element copies when possible.
    cfunc = _bytes.functionDict["copyNbytes"]
    cfunc((nelements,), strbuff, 0, (type.bytes,),
          arr._data, 0, (type.bytes,), type.bytes)
    if arr._type == _nt.Bool:
        arr = ufunc.not_equal(arr, 0)
    return arr

def fromfile(file, type, shape=None):
    """Create an array from binary file data

    If file is a string then that file is opened, else it is assumed
    to be a file object. No options at the moment, all file positioning
    must be done prior to this function call with a file object

    """
    type = getType(type)
    name =  0
    if _type(file) == _type(""):
        name = 1
        file = open(file, 'rb')
    size = os.path.getsize(file.name) - file.tell()
    if not shape:
        nelements, rem = divmod(size, type.bytes)
        if rem:
            raise ValueError(
                "Type size inconsistent with shape or remaining bytes in file")
        shape = (nelements,)
    elif _type(shape) is types.IntType:
        shape=(shape,)
    nbytes = _gen.product(shape)*type.bytes
    if nbytes > size:
        raise ValueError(
                "Not enough bytes left in file for specified shape and type")
    # create the array
    arr = NewArray(shape=shape, type=type)
    # Use of undocumented file method! XXX
    nbytesread = memory.file_readinto(file, arr._data)
    if nbytesread != nbytes:
        raise IOError("Didn't read as many bytes as expected")
    if name:
        file.close()
    if arr._type == _nt.Bool:
        arr = ufunc.not_equal(arr, 0)
    return arr

class NumArray(_numarray._numarray, _gen.NDArray):
    """Fundamental Numeric Array
    
    type       The type of each data element, e.g. Int32  
    byteorder  The actual ordering of bytes in buffer: "big" or "little".

    def __init__(self, shape=(), type=_nt.Any, buffer=None,
                 byteoffset=0, bytestride=None, byteorder=_sys.byteorder,
                 aligned=1):

        # need to add a whole bunch of checking here (or then again,
        # maybe not.)

        type = getType(type)
        itemsize = type.bytes
        _gen.NDArray.__init__(self, shape, itemsize, buffer, byteoffset, bytestride)

        self._type = type
        if byteorder in ["little", "big"]:
            self._byteorder = byteorder
        else:
            raise ValueError("byteorder must be 'little' or 'big'")

    """

    def __getstate__(self):
        """returns state of NumArray for pickling."""
        # assert not hasattr(self, "_shadows") # Not a good idea for pickling.
        state = _gen.NDArray.__getstate__(self)
        state["_byteorder"] = self._byteorder
        state["_type"]      = self._type
        return state

    def __setstate__(self, state):
        """sets state of NumArray after unpickling."""
        _gen.NDArray.__setstate__(self, state)
        self._byteorder = state["_byteorder"]
        self._type      = state["_type"]

    def getreal(self):
        if isinstance(self._type, _nt.FloatingType):
            return self
        else:
            return self.astype(_nt.Float64)

    def setreal(self, value):
        a = self.getreal()
        if a is self:
            a[:] = value
        else:
            raise TypeError("Can't setreal() on a non-floating-point array")
        
    real = property(getreal, setreal,
                    doc="real component of a non-complex numarray")

    def togglebyteorder(self):
        "reverses the state of the _byteorder attribute:  little <-> big."""
        self._byteorder = {"little":"big","big":"little"}[self._byteorder]

    def byteswap(self):
        """Byteswap data in place, leaving the _byteorder attribute untouched.
        """
        if self._itemsize != 1:
            fname = "byteswap"+str(self._itemsize)+"bytes"
            cfunc = _bytes.functionDict[fname]
            cfunc(self._shape, 
                  self._data, self._byteoffset, self._strides,
                  self._data, self._byteoffset, self._strides)

    _byteswap = byteswap  # alias to keep old code working.
        
    def info(self):
        """info() prints out the key attributes of a numarray."""
        _gen.NDArray.info(self)
        print "byteorder:", self._byteorder
        print "byteswap:", self.isbyteswapped()
        print "type:", repr(self._type)

    def astype(self, type=None):
        """Return a copy of the array converted to the given type"""
        if type is None:
            type = self._type
        type = getType(type)
        if type == self._type:
            # always return a copy even if type is same
            return self.copy()
        if type._conv:
            retarr = NewArray(buffer=None, shape=self._shape, type=type)
            if self.is_c_array():
                _ufunc.CheckFPErrors()
                cfunc = self._type._conv.astype[type.name]
                cfunc(self.nelements(), 1, 1,
                      ((self._data, 0), (retarr._data, 0)))
                errorstatus = _ufunc.CheckFPErrors()
                if errorstatus:
                    ufunc.handleError(errorstatus, " during type conversion")
            else:
                ufunc._copyFromAndConvert(self, retarr)
        elif type.fromtype:
            retarr = type.fromtype(self)
        else:
            raise TypeError("Don't know how to convert from %s to %s" %
                            (self._type.name, type.name))
        return retarr

    def is_c_array(self):
        """returns 1 iff an array is c-contiguous, aligned, and not
        byteswapped."""
        return (self.iscontiguous() and self.isaligned() and not
                self.isbyteswapped())

    def is_f_array(self):
        """returns 1 iff an array is fortan-contiguous, aligned, and not
        byteswapped."""
        return (self.is_fortran_contiguous() and self.isaligned() and not
                self.isbyteswapped())

    def new(self, type):
        """Return a new array of given type with same shape as this array

        Note this only creates the array; it does not copy the data.
        """
        # return NewArray(buffer=None, shape=self._shape, type=type)
        if _isComplexType(type):
            return ComplexArray(buffer=None, shape=self._shape, type=type)
        else:
            return NumArray(buffer=None, shape=self._shape, type=type)
        
    def type(self):
        """Return the type object for the array"""
        return self._type

    def _prototype_copyFrom(self, arr):
        """Copy elements from another array.

        This overrides the _generic NDArray version
        """
        # Test for simple case first
        if isinstance(arr, NumArray):
            if (arr._type == self._type and
                self._shape == arr._shape and
                arr._byteorder == self._byteorder and
                _gen.product(arr._strides) != 0 and
                arr.isaligned() and self.isaligned()):                
                name = 'copy'+`self._itemsize`+'bytes'
                cfunc = ( _bytes.functionDict.get(name) or
                          _bytes.functionDict["copyNbytes"])
                cfunc(self._shape, arr._data,  arr._byteoffset,  arr._strides,
                      self._data, self._byteoffset, self._strides,
                      self._itemsize)
                return
        elif PyNUMERIC_TYPES.has_key(_type(arr)):
            # Convert scalar to a one element array for broadcasting
            arr = array([arr])
        else:
            raise TypeError('argument is not array or number')
        barr = self._broadcast(arr, raiseOnFail=1)
        ufunc._copyFromAndConvert(barr, self)

    def copy(self):
        c = _gen.NDArray.copy(self)
        c._byteorder = self._byteorder
        c._type = self._type
        if self.isbyteswapped():
            c.byteswap()
            c.togglebyteorder()
        return c

    def _prototype_view(self):
        v = _gen.NDArray.view(self)
        v._type = self._type
        v._byteorder = self._byteorder
        return v
 
    def __str__(self):
        return arrayprint.array2string(self,
                MAX_LINE_WIDTH, PRECISION, SUPPRESS_SMALL, ' ', 0)

    def __repr__(self):
        return arrayprint.array2string(self,
                MAX_LINE_WIDTH, PRECISION, SUPPRESS_SMALL, ', ', 1)

    def __int__(self):
        if len(self._shape) == 0:
            return int(self[()])
        else:
            raise TypeError, "Only rank-0 numarray can be cast to integers."

    def __float__(self):
        if len(self._shape) == 0:
            return float(self[()])
        else:
            raise TypeError, "Only rank-0 numarray can be cast to floats."

    def __complex__(self):
        if len(self._shape) == 0:
            return complex(self[()])
        else:
            raise TypeError, "Only rank-0 numarray can be cast to complex."

    def __neg__(self): return ufunc.minus(self)
    def __add__(self, operand): return ufunc.add(self, operand)
    def __radd__(self, operand): return ufunc.add(operand, self)
    def __sub__(self, operand): return ufunc.subtract(self, operand)
    def __rsub__(self, operand): return ufunc.subtract(operand, self)
    def __mul__(self, operand): return ufunc.multiply(self, operand)
    def __rmul__(self, operand): return ufunc.multiply(operand, self)
    def __div__(self, operand): return ufunc.divide(self, operand)
    def __rdiv__(self, operand): return ufunc.divide(operand, self)
    def __mod__(self, operand): return ufunc.remainder(self, operand)
    def __rmod__(self, operand): return ufunc.remainder(operand, self)
    def __pow__(self, operand): return ufunc.power(self, operand)
    def __rpow__(self, operand): return ufunc.power(operand, self)

    def __and__(self, operand): return ufunc.bitwise_and(self, operand)
    def __rand__(self, operand): return ufunc.bitwise_and(operand, self)
    def __or__(self, operand): return ufunc.bitwise_or(self, operand)
    def __ror__(self, operand): return ufunc.bitwise_or(operand, self)
    def __xor__(self, operand): return ufunc.bitwise_xor(self, operand)
    def __rxor__(self, operand): return ufunc.bitwise_xor(operand, self)
    def __invert__(self): return ufunc.bitwise_not(self)
    def __rshift__(self, operand): return ufunc.rshift(self, operand)
    def __rrshift__(self, operand): return ufunc.rshift(operand, self)
    def __lshift__(self, operand): return ufunc.lshift(self, operand)
    def __rlshift__(self, operand): return ufunc.lshift(operand, self)
    def __pos__(self): return self

    # augmented assignment operators

    def __iadd__(self, operand):
        ufunc.add(self, operand, self)
        return self
    
    def __isub__(self, operand):
        ufunc.subtract(self, operand, self)
        return self
    
    def __imul__(self, operand):
        ufunc.multiply(self, operand, self)
        return self
    
    def __idiv__(self, operand):
        ufunc.divide(self, operand, self)
        return self
    
    def __imod__(self, operand):
        ufunc.remainder(self, operand, self)
        return self
    
    def __ipow__(self, operand):
        ufunc.power(self, operand, self)
        return self
    
    def __iand__(self, operand):
        ufunc.bitwise_and(self, operand, self)
        return self
    
    def __ior__(self, operand):
        ufunc.bitwise_or(self, operand, self)
        return self
    
    def __ixor__(self, operand):
        ufunc.bitwise_xor(self, operand, self)
        return self
    
    def __irshift(self, operand):
        ufunc.rshift(self, operand, self)
        return self
    
    def __ilshift(self, operand):
        ufunc.lshift(self, operand, self)
        return self

    # rich comparisons (only works in Python 2.1 and later)

    def __lt__(self, operand): return ufunc.less(self, operand)
    def __gt__(self, operand): return ufunc.greater(self, operand)
    def __le__(self, operand): return ufunc.less_equal(self, operand)
    def __ge__(self, operand): return ufunc.greater_equal(self, operand)
    def __eq__(self, operand):
        if operand is None:
            return 0
        else:
            return ufunc.equal(self, operand)
    def __ne__(self, operand):
        if operand is None:
            return 1
        else:
            return ufunc.not_equal(self, operand)

    def __nonzero__(self):
        _warnings.warn("Using a NumArray as a truth value is deprecated",
                       DeprecationWarning, stacklevel=2)
        b = ufunc.logical_or.areduce(self)
        b.ravel()
        return ufunc.logical_or.reduce(b)

    def sort(self, axis=-1):
        if axis==-1:
            ufunc._sortN(self)
        else:
            self.swapaxes(axis,-1)
            ufunc._sortN(self)
            self.swapaxes(axis,-1)

    def _argsort(self, axis=-1):
        if axis==-1:
            ashape = self.getshape()
            w = array(shape=ashape, type=_nt.Long)
            w[...,:] = arange(ashape[-1], type=_nt.Long)
            ufunc._argsortN(self,w)
            return w
        else:
            self.swapaxes(axis,-1)
            return self._argsort()

    def argsort(self, axis=-1):
        return self.copy()._argsort(axis)

    def argmax(self, axis=-1):
        return self.argsort(axis)[...,-1]

    def argmin(self, axis=-1):
        return self.argsort(axis)[...,0]

    def diagonal(self, *args, **keywords):
        return diagonal(self, *args, **keywords)

    def trace(self, *args, **keywords): 
        return trace(self, *args, **keywords)

    def typecode(self):
        return _nt.typecode[self._type]

    def min(self):
        """Returns the minimum element in the array."""
        return ufunc.minimum.reduce(ufunc.minimum.areduce(self).flat)

    def max(self):
        """Returns the maximum element in the array."""
        return ufunc.maximum.reduce(ufunc.maximum.areduce(self).flat)

    def sum(self):
        """Returns the sum of all elements in the array."""
        return ufunc.add.reduce(ufunc.add.areduce(self).flat)

    def mean(self):
        """Returns the average of all elements in the array."""
        return self.sum()/(self.nelements()*1.0)

class ComplexArray(NumArray):
    """Complex Arrays  are derived from NumArray, and supported
    in C with Complex Ufuncs.  A Complex array consists of
    interleaved real and imaginary numarray.
    """
    __safe_for_unpickling__ = 1

    def __init__(self, shape=(), type=_nt.Any, buffer=None,
                 byteoffset=0, bytestride=None,
                 byteorder=_sys.byteorder, aligned=1, real=None, imag=None):

        type = getType(type)
        if not _isComplexType(type) and type is not Any:
            raise  TypeError("Type must be a Complex number type")
        itemsize = type.bytes
        NumArray.__init__(self, shape, type, buffer=buffer,
                          byteoffset=byteoffset,
                          bytestride=bytestride,
                          byteorder=byteorder,
                          aligned=aligned)
        if real is not None:
            self.getreal()[:] = real
        if imag is not None:
            self.getimag()[:] = imag

    def byteswap(self):
        """Byteswap the data in place; customized for Complex."""
        if self._itemsize != 1:
            fname = "byteswap" + self._type.name
            cfunc = _bytes.functionDict[fname]
            cfunc(self._shape, 
                  self._data, self._byteoffset, self._strides,
                  self._data, self._byteoffset, self._strides)
        
    def itemsize(self):
        return 2*_gen.NDArray.itemsize(self)

    def getreal(self):
        t = _realtype(self._type)
        arr = NumArray(self._shape, t, buffer=self._data,
                       byteoffset=self._byteoffset,
                       bytestride=self._bytestride,
                       byteorder=self._byteorder)
        arr._strides = self._strides[:]
        return arr

    def setreal(self, value):
        self.getreal()[:] = value
        
    def getimag(self):
        t = _realtype(self._type)
        arr = NumArray(self._shape, t, buffer=self._data,
                       byteoffset=self._byteoffset+t.bytes,
                       bytestride=self._bytestride,
                       byteorder=self._byteorder)
        arr._strides = self._strides[:]
        return arr

    def setimag(self, value):
        self.getimag()[:] = value

    imag = property(getimag, setimag,
                        doc="imaginary component of complex array")

    real = property(getreal, setreal,
                        doc="real component of complex array")

    setimaginary = setimag
    getimaginary = getimag
    imaginary = imag

    def _getitem(self, key):
        """Returns value for single element in array using byte offset
        
        Does no validity checking on byteoffset (assumes it is valid).
        """
        return complex(self.getreal()._getitem(key),
                       self.getimag()._getitem(key))
        
    def _setitem(self, key, value):
        """Sets a single value in an array using byte offset
        
        Does no validity checking on byteoffset (assumes it is valid).
        """
        cvalue = complex(value)
        self.getreal()._setitem(key, cvalue.real)
        self.getimag()._setitem(key, cvalue.imag)

    def _copyFrom(self, arr):
        """Copy elements from another array.

        This overrides the NumArray version
        """
        if   isinstance(arr, ComplexArray):
            real, imag = arr.getreal(), arr.getimag()
        elif _type(arr)==types.ComplexType:
            real, imag = arr.real,  arr.imag
        else:
            real, imag = arr,       0.
        self.getreal()._copyFrom(real)
        self.getimag()._copyFrom(imag)

    def conjugate(self):
        ufunc.minus(self.getimag(), self.getimag())

    def __lt__(self, operand):
        raise TypeError("Complex numarray don't support < comparison")
    def __gt__(self, operand):
        raise TypeError("Complex numarray don't support > comparison")
    def __le__(self, operand):
        raise TypeError("Complex numarray don't support <= comparison")
    def __ge__(self, operand):
        raise TypeError("Complex numarray don't support >= comparison")



def Complex32_fromtype(arr):
    """Used for converting other types to Complex32.

    This is used to set an fromtype attribute of the ComplexType objects"""
    rarr = arr.astype(Float32)
    retarr = ComplexArray(arr._shape, _nt.Complex32)
    retarr.getreal()[:] = rarr
    retarr.getimag()[:] = 0.
    return retarr

def Complex64_fromtype(arr):
    """Used for converting other types to Complex64.

    This is used to set an fromtype attribute of the ComplexType objects"""
    rarr = arr.astype(Float64)
    retarr = ComplexArray(arr._shape, _nt.Complex64)
    retarr.getreal()[:] = rarr
    retarr.getimag()[:] = 0.
    return retarr

# Check whether byte order is big endian or little endian.

from sys import byteorder
isBigEndian = (byteorder == "big")
del byteorder

# Add fromtype function to Complex types

_nt.Complex32.fromtype = Complex32_fromtype
_nt.Complex64.fromtype = Complex64_fromtype

# Return type of complex type components

def _isComplexType(type):
    return type in [_nt.Complex32, _nt.Complex64]

def _realtype(complextype):
    if complextype == _nt.Complex32:
        return _nt.Float32
    else:
        return _nt.Float64


def conjugate(a):
    """conjugate(a) returns the complex conjugate of 'a'"""
    a = array(a)
    if not _isComplexType(a._type):
        a = a.astype(_nt.Complex64)
    a.conjugate()
    return a


def zeros(shape, type=None):
    shape = _gen.getShape(shape)
    retarr = _fillarray(_gen.product(shape), 0, 0, type)
    retarr.setshape(shape)
    return retarr

def ones(shape, type=None):
    shape = _gen.getShape(shape)
    retarr = _fillarray(_gen.product(shape), 1, 0, type)
    retarr.setshape(shape)
    return retarr

def arange(a1, a2=None, stride=1, type=None, shape=None):
    # Return empty range of correct type for single negative paramter.
    if not isinstance(a1, types.ComplexType) and a1 < 0 and a2 is None:
        t = __builtins__["type"](a1)
        return zeros(shape=(0,), type=Py2NumType[t])
    if a2 == None:
        start = 0 + (0*a1) # to make it same type as stop
        stop  = a1
    else:
        start = a1 +(0*a2)
        stop  = a2
    delta = (stop-start)/stride ## xxx int divide issue
    if _type(delta) == types.ComplexType:
        # xxx What make sense here?
        size = int(math.ceil(delta.real))
    else:
        size = int(math.ceil((stop-start)/float(stride)))
    if size < 0:
        size = 0
    r = _fillarray(size, start, stride, type)
    if shape is not None:
        r.setshape(shape)
    return r

arrayrange = arange  # alias arange as arrayrange.

def identity(n, type=None):
    a = zeros(shape=(n,n), type=type)
    i = arange(n)
    a.put(i, i, 1)
    return a

def dot(a,b):
    """dot(a,b) matrix-multiplies a by b.
    """
    return ufunc.innerproduct(a, _gen.swapaxes(inputarray(b), -1, -2))

matrixmultiply = dot  # Deprecated in Numeric

def outerproduct(a,b):
    """outerproduct(a,b) computes the NxM outerproduct of N vector 'a' and
    M vector 'b', where result[i,j] = a[i]*b[j].
    """
    a=_gen.reshape(inputarray(a), (-1,1))  # ravel a into an Nx1
    b=_gen.reshape(inputarray(b), (1,-1))  # ravel b into a 1xM
    return matrixmultiply(a,b)   # return NxM result

def allclose (a, b, rtol=1.e-5, atol=1.e-8): # From Numeric 20.3
    """ allclose(a,b,rtol=1.e-5,atol=1.e-8)
        Returns true if all components of a and b are equal
        subject to given tolerances.
        The relative error rtol must be positive and << 1.0
        The absolute error atol comes into play for those elements
        of y that are very small or zero; it says how small x must be also.
    """
    x, y = inputarray(a), inputarray(b)
    d = ufunc.less(ufunc.abs(x-y), atol + rtol * ufunc.abs(y))
    return alltrue(_gen.ravel(d))


# From Numeric-21.0
def diagonal(a, offset= 0, axis1=0, axis2=1):
    """diagonal(a, offset=0, axis1=0, axis2=1) returns the given diagonals
    defined by the last two dimensions of the array.
    """
    a = inputarray(a)
    if axis2 < axis1: axis1, axis2 = axis2, axis1
    if axis2 > 1:
        new_axes = range (len (a._shape))
        del new_axes [axis2]; del new_axes [axis1]
        new_axes [0:0] = [axis1, axis2]
        a = _gen.transpose (a, new_axes)
    s = a._shape
    if len (s) == 2:
        n1 = s [0]
        n2 = s [1]
        n = n1 * n2
        s = (n,)
        a = _gen.reshape (a, s)
        if offset < 0:
            return _gen.take (a, range(- n2 * offset, min(n2, n1+offset) * (n2+1) - n2 * offset, n2+1), axis=0)
        else:
            return _gen.take (a, range (offset,         min(n1, n2-offset) * (n2+1) + offset,      n2+1), axis=0)
    else :
        my_diagonal = []
        for i in range (s [0]) :
            my_diagonal.append (diagonal (a [i], offset))
        return array(my_diagonal)

# From Numeric-21.0
def trace(a, offset=0, axis1=0, axis2=1):
    """trace(a,offset=0, axis1=0, axis2=1) returns the sum along diagonals
    (defined by the last two dimenions) of the array.
    """
    return ufunc.add.reduce(diagonal(a, offset, axis1, axis2))

def rank(a):
    """rank(a) returns the number of dimensions in 'a'"""
    return inputarray(a).getrank()

def shape(a):
    """rank(a) returns the shape tuple of 'a'"""
    return inputarray(a).getshape()

def size(a):
    """size(a) returns the size in bytes of array 'a'"""
    return inputarray(a).nelements()

def array_str(a):
    return str(inputarray(a))

def array_repr(a):
    return repr(inputarray(a))

def around(a, digits=0, output=None):
    """rounds array 'a' to 'digits' of precision, storing the result in
    'output', or returning the result as new array if output=None"""
    a = inputarray(a)
    scale = 10.0**digits
    if output is None:
        wout = a.copy()
    else:
        wout = output
    if digits != 0:
        wout *= scale  # bug in 2.2.1 and earlier causes fail as bad sequence op
    ufunc._round(wout, wout)
    if digits != 0:
        wout /= scale
    if output is None:
        return wout

def round(*args, **keys):
    _warnings.warn("round() is deprecated.  Switch to around().",
                   DeprecationWarning)
    return ufunc._round(*args, **keys)

def explicit_type(x):
    """explicit_type(x) returns a view of x which will always show it's type in it's repr.
    This is useful when the same test is run in two places, one where the type used *is* the
    default and hence not normally repr'ed, and one where the type used *is not* the default
    and so is displayed.
    """
    y = x.view()
    y._explicit_type = 1
    return y

ArrayType = NumArray  # Alias for backwards compatability with Numeric

def array_equiv(a, b):

    """array_equiv returns True if 'a' and 'b' are shape consistent
    (mutually broadcastable) and have all elements equal and False
    otherwise."""
    
    try:
        a, b = inputarray(a), inputarray(b)
    except TypeError:
        return 0
    if not isinstance(a, NumArray) or not isinstance(b, NumArray):
        return 0
    a, b = a._dualbroadcast(b)
    if a is None or b is None:
        return 0
    return ufunc.logical_and.reduce(_gen.ravel(a == b))

def array_equal(a, b):
    
    """array_equal returns True if 'a' and 'b' have identical shapes
    and all elements equal and False otherwise."""

    try:
        a, b = inputarray(a), inputarray(b)
    except TypeError:
        return 0
    if not isinstance(a, NumArray) or not isinstance(b, NumArray):
        return 0
    if a._shape != b._shape:
        return 0
    return ufunc.logical_and.reduce(_gen.ravel(a == b))


class _UBuffer(NumArray):
    """_UBuffer is used to hold a single "block" of ufunc data during
    the block-wise processing of all elements in an array.

    Subclassing the buffer object from numnumarray simplifies (and speeds!)
    their usage at the C level.  They are not intended to be used as
    public array objects, hence they are private!
    """

    def __init__(self, pybuffer):
        NumArray.__init__(self, (len(pybuffer),), _nt.Int8, pybuffer)
        self._strides    = None   # how it is distinguished from a real array

    def isbyteswapped(self):           return 0    
    def isaligned(self):               return 1
    def iscontiguous(self):            return 1
    def is_c_array(self):              return 1
    def _getByteOffset(self, shape):   return 0
    
    def __del__(self):
        """On deletion return the data to the buffer pool"""
        if self._data is not None:
            if ufunc is not None and ufunc._bufferPool is not None:
                ufunc._bufferPool.buffers.append(self._data)

import ufunc
import arrayprint

sum = ufunc.add.reduce
cumsum = ufunc.add.accumulate
product = ufunc.multiply.reduce
cumproduct = ufunc.multiply.accumulate
alltrue = ufunc.logical_and.reduce
sometrue = ufunc.logical_or.reduce
absolute = ufunc.abs  
negative = ufunc.minus
fmod = ufunc.remainder


