/***********************************************************
Copyright 1991-1995 by Stichting Mathematisch Centrum, Amsterdam,
The Netherlands.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of Stichting Mathematisch
Centrum or CWI or Corporation for National Research Initiatives or
CNRI not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission.

While CWI is the initial source for this software, a modified version
is made available by the Corporation for National Research Initiatives
(CNRI) at the Internet address ftp://ftp.python.org.

STICHTING MATHEMATISCH CENTRUM AND CNRI DISCLAIM ALL WARRANTIES WITH
REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL STICHTING MATHEMATISCH
CENTRUM OR CNRI BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.

******************************************************************/

/* Use this file as a template to start implementing a module that
   also declares objects types. All occurrences of 'SpecfitFunso' should be changed
   to something reasonable for your objects. After that, all other
   occurrences of 'SpecfitFuns' should be changed to something reasonable for your
   module. If your module is named foo your sourcefile should be named
   foomodule.c.
   
   You will probably want to delete all references to 'x_attr' and add
   your own types of attributes instead.  Maybe you want to name your
   local variables other than 'self'.  If your object type is needed in
   other files, you'll have to create a file "foobarobject.h"; see
   intobject.h for an example. */

/* SpecfitFunso objects */

#include <Python.h>
#include <./Numeric/arrayobject.h>
#include <math.h>
#define isARRAY(a) ((a) && PyArray_Check((PyArrayObject *)a))
#define A_SIZE(a) PyArray_Size((PyObject *) a)
#define isARRAY(a) ((a) && PyArray_Check((PyArrayObject *)a))
# if sun || linux
#define MIN(x, y) (((x) < (y)) ? (x) : (y))
#define MAX(x, y) (((x) > (y)) ? (x) : (y))
#endif


static PyObject *ErrorObject;

typedef struct {
	PyObject_HEAD
	PyObject	*x_attr;	/* Attributes dictionary */
} SpecfitFunsoObject;

staticforward PyTypeObject SpecfitFunso_Type;

/*
 * Function prototypes
 */
static SpecfitFunsoObject *newSpecfitFunsoObject (PyObject *arg);
static void                SpecfitFunso_dealloc  (SpecfitFunsoObject *self);

#define SpecfitFunsoObject_Check(v)	((v)->ob_type == &SpecfitFunso_Type)

static SpecfitFunsoObject *
newSpecfitFunsoObject(arg)
	PyObject *arg;
{
	SpecfitFunsoObject *self;
	self = PyObject_NEW(SpecfitFunsoObject, &SpecfitFunso_Type);
	if (self == NULL)
		return NULL;
	self->x_attr = NULL;
	return self;
}

/* SpecfitFunso methods */

static void
SpecfitFunso_dealloc(self)
	SpecfitFunsoObject *self;
{
	Py_XDECREF(self->x_attr);
	PyMem_DEL(self);
}

/*
static PyObject *
SpecfitFunso_demo(self, args)
	SpecfitFunsoObject *self;
	PyObject *args;
*/
static PyObject *
SpecfitFunso_demo(SpecfitFunsoObject *self,
	PyObject *args)
{
	if (!PyArg_ParseTuple(args, ":demo"))
		return NULL;
	Py_INCREF(Py_None);
	return Py_None;
}

static PyMethodDef SpecfitFunso_methods[] = {
	{"demo",	(PyCFunction)SpecfitFunso_demo,	1},
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
SpecfitFunso_getattr(SpecfitFunsoObject *self,
	char *name)
{
	if (self->x_attr != NULL) {
		PyObject *v = PyDict_GetItemString(self->x_attr, name);
		if (v != NULL) {
			Py_INCREF(v);
			return v;
		}
	}
	return Py_FindMethod(SpecfitFunso_methods, (PyObject *)self, name);
}

static int
SpecfitFunso_setattr(SpecfitFunsoObject *self, char *name,
	PyObject *v)
{
	if (self->x_attr == NULL) {
		self->x_attr = PyDict_New();
		if (self->x_attr == NULL)
			return -1;
	}
	if (v == NULL) {
		int rv = PyDict_DelItemString(self->x_attr, name);
		if (rv < 0)
			PyErr_SetString(PyExc_AttributeError,
			        "delete non-existing SpecfitFunso attribute");
		return rv;
	}
	else
		return PyDict_SetItemString(self->x_attr, name, v);
}

statichere PyTypeObject SpecfitFunso_Type = {
	/* The ob_type field must be initialized in the module init function
	 * to be portable to Windows without using C++. */
	PyObject_HEAD_INIT(NULL)
	0,			/*ob_size*/
	"SpecfitFunso",			/*tp_name*/
	sizeof(SpecfitFunsoObject),	/*tp_basicsize*/
	0,			/*tp_itemsize*/
	/* methods */
	(destructor)SpecfitFunso_dealloc, /*tp_dealloc*/
	0,			/*tp_print*/
	(getattrfunc)SpecfitFunso_getattr, /*tp_getattr*/
	(setattrfunc)SpecfitFunso_setattr, /*tp_setattr*/
	0,			/*tp_compare*/
	0,			/*tp_repr*/
	0,			/*tp_as_number*/
	0,			/*tp_as_sequence*/
	0,			/*tp_as_mapping*/
	0,			/*tp_hash*/
};
/* --------------------------------------------------------------------- */

/* Function SUBAC returning smoothed array */

static PyObject *
SpecfitFuns_subac(PyObject *self, PyObject *args)
{
    PyObject *input;
    PyArrayObject   *array, *ret;
    int n, dimensions[1];
    int i, j, niter = 5000;
    double  t_old, t_mean, c = 1.0001;
    double  *data;
    
    if (!PyArg_ParseTuple(args, "O|di", &input, &c, &niter))
        return NULL;
    array = (PyArrayObject *)
             PyArray_CopyFromObject(input, PyArray_DOUBLE,1,1);
    if (array == NULL)
        return NULL;
    n = array->dimensions[0];
    dimensions[0] = array->dimensions[0];
    ret = (PyArrayObject *)
        PyArray_FromDims(1, dimensions, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(array);
        return NULL;
    }

    /* Do the job */
    data = (double *) array->data;
    for (i=0;i<niter;i++){
        t_old = *(data);
        for (j=1;j<n-1;j++) {
            t_mean = 0.5 * (t_old + *(data+j+1));
            t_old = *(data+j);
            if (t_old > (t_mean * c))
                *(data+j) = t_mean;
       }
    /*   t_mean = 0.5 * (t_old + *(data+n-1));
       t_old = *(data+n-1);
       if (t_old > (t_mean * c))
                *(data+n-1) = t_mean;*/
    }
    ret = (PyArrayObject *) PyArray_Copy(array);
    Py_DECREF(array);
    if (ret == NULL)
        return NULL;
    return PyArray_Return(ret);  

}

static PyObject *
SpecfitFuns_gauss(PyObject *self, PyObject *args)
{
    PyObject *input1, *input2;
    int debug=0;
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp, log2;
    double  *px, *pret;
    char *tpe;
    typedef struct {
        double  height;
        double  centroid;
        double  fwhm;
    } gaussian;
    gaussian *pgauss;

    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|i", &input1,&input2,&debug))
        return NULL;

    if (debug == 1){
        tpe = input1->ob_type->tp_name;
            printf("C(iotest): input1 type of object = %s\n",tpe);
       /* j = PyObject_Length (input1);
        printf("Length = %d\n",j);
        for (i=0;i<j;i++){
            
            printf("Element %d = %ld\n",i,
                            PyInt_AsLong(PyList_GetItem(input1,i)));
        }
        */
    
    }

    param = (PyArrayObject *)
             PyArray_ContiguousFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }

    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%3) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    log2 = 0.69314718055994529;
    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
       *pret = 0;
        pgauss = (gaussian *) param->data;
        for (i=0;i<(npars/3);i++){
            dhelp = pgauss[i].fwhm/(2.0*sqrt(2.0*log2));
            dhelp = (*px - pgauss[i].centroid)/dhelp;
            if (dhelp <= 35) {
                *pret += pgauss[i].height * exp (-0.5 * dhelp * dhelp);
            }
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            pgauss = (gaussian *) param->data;
            for (i=0;i<(npars/3);i++){
                dhelp = pgauss[i].fwhm/(2.0*sqrt(2.0*log2));
                dhelp = (*px - pgauss[i].centroid)/dhelp;
                if (dhelp <= 35) {
                *pret += pgauss[i].height * exp (-0.5 * dhelp * dhelp);
                }
            }
            pret++;
            px++;
        }
    }
    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}

static PyObject *
SpecfitFuns_agauss(PyObject *self, PyObject *args)
{
    PyObject *input1, *input2;
    int debug=0;
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp, log2, sqrt2PI,sigma,tosigma;
    double  *px, *pret;
    typedef struct {
        double  area;
        double  centroid;
        double  fwhm;
    } gaussian;
    gaussian *pgauss;

    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|i", &input1,&input2,&debug))
        return NULL;

    param = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }

    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%3) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    log2 = 0.69314718055994529;
    sqrt2PI= sqrt(2.0*M_PI);
    tosigma=1.0/(2.0*sqrt(2.0*log2));

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
       *pret = 0;
        pgauss = (gaussian *) param->data;
        for (i=0;i<(npars/3);i++){
            sigma = pgauss[i].fwhm*tosigma;
            dhelp = (*px - pgauss[i].centroid)/sigma;
            if (dhelp <= 35){
                *pret += (pgauss[i].area/(sigma*sqrt2PI))* exp (-0.5 * dhelp * dhelp);
            }
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            pgauss = (gaussian *) param->data;
            for (i=0;i<(npars/3);i++){
                sigma = pgauss[i].fwhm*tosigma;                
                dhelp = (*px - pgauss[i].centroid)/sigma;
                if (dhelp <= 35){
                    *pret += (pgauss[i].area/(sigma*sqrt2PI)) * exp (-0.5 * dhelp * dhelp);
                }
            }
            pret++;
            px++;
        }
    }
    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}

static PyObject *
SpecfitFuns_apvoigt(PyObject *self, PyObject *args)
{
    PyObject *input1, *input2;
    int debug=0;
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp, log2, sqrt2PI,sigma,tosigma;
    double  *px, *pret;
    typedef struct {
        double  area;
        double  centroid;
        double  fwhm;
        double  eta;
    } pvoigtian;
    pvoigtian *ppvoigt;

    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|i", &input1,&input2,&debug))
        return NULL;

    param = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }

    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%4) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
       *pret = 0;
        ppvoigt = (pvoigtian *) param->data;
        for (i=0;i<(npars/4);i++){
            dhelp = (*px - ppvoigt[i].centroid) / (0.5 * ppvoigt[i].fwhm);
            dhelp = 1.0 + (dhelp * dhelp);
            *pret += ppvoigt[i].eta * \
                (ppvoigt[i].area / (0.5 * M_PI * ppvoigt[i].fwhm * dhelp));
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            ppvoigt = (pvoigtian *) param->data;
            for (i=0;i<(npars/4);i++){
            dhelp = (*px - ppvoigt[i].centroid) / (0.5 * ppvoigt[i].fwhm);
            dhelp = 1.0 + (dhelp * dhelp);
            *pret += ppvoigt[i].eta * \
                (ppvoigt[i].area / (0.5 * M_PI * ppvoigt[i].fwhm * dhelp));
            }
            pret++;
            px++;
        }
    }
    
    /* The lorentzian term is calculated */
    /* Now it has to calculate the gaussian term */
    log2 = 0.69314718055994529;
    sqrt2PI= sqrt(2.0*M_PI);
    tosigma=1.0/(2.0*sqrt(2.0*log2));

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
        ppvoigt = (pvoigtian *) param->data;
        for (i=0;i<(npars/3);i++){
            sigma = ppvoigt[i].fwhm * tosigma;
            dhelp = (*px - ppvoigt[i].centroid)/sigma;
            if (dhelp <= 35) {
                *pret += (1.0 - ppvoigt[i].eta) * \
                        (ppvoigt[i].area/(sigma*sqrt2PI)) \
                        * exp (-0.5 * dhelp * dhelp);
            }
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            ppvoigt = (pvoigtian *) param->data;
            for (i=0;i<(npars/3);i++){
                sigma = ppvoigt[i].fwhm * tosigma;
                dhelp = (*px - ppvoigt[i].centroid)/sigma;
                if (dhelp <= 35) {
                    *pret += (1.0 - ppvoigt[i].eta) * \
                        (ppvoigt[i].area/(sigma*sqrt2PI)) \
                        * exp (-0.5 * dhelp * dhelp);
                }
        }
            pret++;
            px++;
        }
    }
    
    

    /* word done */    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}



static PyObject *
SpecfitFuns_pvoigt(PyObject *self, PyObject *args)
{
    PyObject *input1, *input2;
    int debug=0;
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp, log2;
    double  *px, *pret;
    typedef struct {
        double  height;
        double  centroid;
        double  fwhm;
        double  eta;
    } pvoigtian;
    pvoigtian *ppvoigt;

    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|i", &input1,&input2,&debug))
        return NULL;

    param = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }

    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%4) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
       *pret = 0;
        ppvoigt = (pvoigtian *) param->data;
        for (i=0;i<(npars/4);i++){
            dhelp = (*px - ppvoigt[i].centroid) / (0.5 * ppvoigt[i].fwhm);
            dhelp = 1.0 + (dhelp * dhelp);
            *pret += ppvoigt[i].eta * (ppvoigt[i].height / dhelp);
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            ppvoigt = (pvoigtian *) param->data;
            for (i=0;i<(npars/4);i++){
            dhelp = (*px - ppvoigt[i].centroid) / (0.5 * ppvoigt[i].fwhm);
            dhelp = 1.0 + (dhelp * dhelp);
            *pret += ppvoigt[i].eta * (ppvoigt[i].height / dhelp);
            }
            pret++;
            px++;
        }
    }
    
    /* The lorentzian term is calculated */
    /* Now it has to calculate the gaussian term */
    log2 = 0.69314718055994529;

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
        ppvoigt = (pvoigtian *) param->data;
        for (i=0;i<(npars/3);i++){
            dhelp = ppvoigt[i].fwhm/(2.0*sqrt(2.0*log2));
            dhelp = (*px - ppvoigt[i].centroid)/dhelp;
            if (dhelp <= 35) {
                *pret += (1.0 - ppvoigt[i].eta) * ppvoigt[i].height \
                        * exp (-0.5 * dhelp * dhelp);
            }
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            ppvoigt = (pvoigtian *) param->data;
            for (i=0;i<(npars/3);i++){
                dhelp = ppvoigt[i].fwhm/(2.0*sqrt(2.0*log2));
                dhelp = (*px - ppvoigt[i].centroid)/dhelp;
                if (dhelp <= 35) {
                *pret += (1.0 - ppvoigt[i].eta) * ppvoigt[i].height \
                        * exp (-0.5 * dhelp * dhelp);
                }
        }
            pret++;
            px++;
        }
    }
    
    

    /* word done */    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}

static PyObject *
SpecfitFuns_lorentz(PyObject *self, PyObject *args)
{
    PyObject *input1, *input2;
    int debug=0;
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp;
    double  *px, *pret;
    typedef struct {
        double  height;
        double  centroid;
        double  fwhm;
    } lorentzian;
    lorentzian *plorentz;

    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|i", &input1,&input2,&debug))
        return NULL;

    param = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }

    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%3) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
       *pret = 0;
        plorentz = (lorentzian *) param->data;
        for (i=0;i<(npars/3);i++){
            dhelp = (*px - plorentz[i].centroid) / (0.5 * plorentz[i].fwhm);
            dhelp = 1.0 + (dhelp * dhelp);
            *pret += (plorentz[i].height / dhelp);
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            plorentz = (lorentzian *) param->data;
            for (i=0;i<(npars/3);i++){
            dhelp = (*px - plorentz[i].centroid) / (0.5 * plorentz[i].fwhm);
            dhelp = 1.0 + (dhelp * dhelp);
            *pret += (plorentz[i].height / dhelp);
            }
            pret++;
            px++;
        }
    }
    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}


static PyObject *
SpecfitFuns_alorentz(PyObject *self, PyObject *args)
{
    PyObject *input1, *input2;
    int debug=0;
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp;
    double  *px, *pret;
    typedef struct {
        double  area;
        double  centroid;
        double  fwhm;
    } lorentzian;
    lorentzian *plorentz;

    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|i", &input1,&input2,&debug))
        return NULL;

    param = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }

    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%3) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
       *pret = 0;
        plorentz = (lorentzian *) param->data;
        for (i=0;i<(npars/3);i++){
            dhelp = (*px - plorentz[i].centroid) / (0.5 * plorentz[i].fwhm);
            dhelp = 1.0 + (dhelp * dhelp);
            *pret += plorentz[i].area /(0.5 * M_PI * plorentz[i].fwhm * dhelp);
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            plorentz = (lorentzian *) param->data;
            for (i=0;i<(npars/3);i++){
            dhelp = (*px - plorentz[i].centroid) / (0.5 * plorentz[i].fwhm);
            dhelp = 1.0 + (dhelp * dhelp);
            *pret += plorentz[i].area /(0.5 * M_PI * plorentz[i].fwhm * dhelp);
            }
            pret++;
            px++;
        }
    }
    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}

static PyObject *
SpecfitFuns_downstep(PyObject *self, PyObject *args)
{
    double erfc(double);
    PyObject *input1, *input2;
    int debug=0;
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp, tosigma;
    double  *px, *pret;
    typedef struct {
        double  height;
        double  centroid;
        double  fwhm;
    } errorfc;
    errorfc *perrorfc;


    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|i", &input1,&input2,&debug))
        return NULL;

    param = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }

    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%3) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    tosigma=1.0/(2.0*sqrt(2.0*log(2.0)));


    if (nd_x == 0){
       *pret = 0;
        perrorfc = (errorfc *) param->data;
        for (i=0;i<(npars/3);i++){        
            dhelp = perrorfc[i].fwhm * tosigma;
            dhelp = (*px - perrorfc[i].centroid) / (sqrt(2)*dhelp);
            *pret += perrorfc[i].height * 0.5 * erfc(dhelp);
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            perrorfc = (errorfc *) param->data;
            for (i=0;i<(npars/3);i++){
            dhelp = perrorfc[i].fwhm * tosigma;
            dhelp = (*px - perrorfc[i].centroid) / (sqrt(2)*dhelp);
            *pret += perrorfc[i].height * 0.5 * erfc(dhelp);
            }
            pret++;
            px++;
        }
    }
    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}

static PyObject *
SpecfitFuns_upstep(PyObject *self, PyObject *args)
{
    double erf(double);
    PyObject *input1, *input2;
    int debug=0;
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp, tosigma;
    double  *px, *pret;
    typedef struct {
        double  height;
        double  centroid;
        double  fwhm;
    } errorf;
    errorf *perrorf;


    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|i", &input1,&input2,&debug))
        return NULL;

    param = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }

    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%3) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    tosigma=1.0/(2.0*sqrt(2.0*log(2.0)));

    if (nd_x == 0){
       *pret = 0;
        perrorf = (errorf *) param->data;
        for (i=0;i<(npars/3);i++){        
            dhelp = perrorf[i].fwhm * tosigma;
            dhelp = (*px - perrorf[i].centroid) / (sqrt(2)*dhelp);
            *pret += perrorf[i].height * 0.5 * (1.0 + erf(dhelp));
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            perrorf = (errorf *) param->data;
            for (i=0;i<(npars/3);i++){
            dhelp = perrorf[i].fwhm * tosigma;
            dhelp = (*px - perrorf[i].centroid) / (sqrt(2)*dhelp);
            *pret += perrorf[i].height * 0.5 * (1.0 + erf(dhelp));
            }
            pret++;
            px++;
        }
    }
    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}
static PyObject *
SpecfitFuns_slit(PyObject *self, PyObject *args)
{
    double erf(double);
    PyObject *input1, *input2;
    int debug=0;
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp, dhelp1,dhelp2,centroid1,centroid2,tosigma;
    double  *px, *pret;
    typedef struct {
        double  height;
        double  position;
        double  fwhm;
        double  beamfwhm;
    } errorf;
    errorf *perrorf;


    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|i", &input1,&input2,&debug))
        return NULL;

    param = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }

    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%4) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    tosigma=1.0/(2.0*sqrt(2.0*log(2.0)));

    if (nd_x == 0){
       *pret = 0;
        perrorf = (errorf *) param->data;
        for (i=0;i<(npars/4);i++){        
            dhelp = perrorf[i].beamfwhm * tosigma;
            centroid1=perrorf[i].position - 0.5 * perrorf[i].fwhm;
            centroid2=perrorf[i].position + 0.5 * perrorf[i].fwhm;            
            dhelp1 = (*px - centroid1) / (sqrt(2)*dhelp);
            dhelp2 = (*px - centroid2) / (sqrt(2)*dhelp);
            *pret += perrorf[i].height * 0.5 * (1.0 + erf(dhelp1))*erfc(dhelp2);
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            perrorf = (errorf *) param->data;
            for (i=0;i<(npars/4);i++){
            dhelp = perrorf[i].beamfwhm * tosigma;
            centroid1=perrorf[i].position - 0.5 * perrorf[i].fwhm;
            centroid2=perrorf[i].position + 0.5 * perrorf[i].fwhm;            
            dhelp1 = (*px - centroid1) / (sqrt(2)*dhelp);
            dhelp2 = (*px - centroid2) / (sqrt(2)*dhelp);
            *pret += perrorf[i].height * 0.5 * (1.0 + erf(dhelp1))*erfc(dhelp2);
            }
            pret++;
            px++;
        }
    }
    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}

static PyObject *
SpecfitFuns_erfc(PyObject *self, PyObject *args)
{
    double erfc(double);
    PyObject *input1;
    int debug=0;
    PyArrayObject   *x;
    PyArrayObject   *ret;
    int nd_x;
    int dim_x[2];
    int j, k;
    double  dhelp;
    double  *px, *pret;


    /** statements **/
    if (!PyArg_ParseTuple(args, "O|i", &input1,&debug))
        return NULL;
        
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (x == NULL){
        return NULL;
    }    

    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_x = %d\n",nd_x);
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if(debug !=0) {
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(x);
        return NULL;    
    }

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
       dhelp = *px;
       *pret = erfc(dhelp);
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            dhelp = *px;
            *pret = erfc(dhelp);
            pret++;
            px++;
        }
    }
    
    Py_DECREF(x);
    return PyArray_Return(ret);  
}

static PyObject *
SpecfitFuns_erf(PyObject *self, PyObject *args)
{
    double erfc(double);
    PyObject *input1;
    int debug=0;
    PyArrayObject   *x;
    PyArrayObject   *ret;
    int nd_x;
    int dim_x[2];
    int j, k;
    double  dhelp;
    double  *px, *pret;


    /** statements **/
    if (!PyArg_ParseTuple(args, "O|i", &input1,&debug))
        return NULL;
        
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (x == NULL){
        return NULL;
    }    

    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_x = %d\n",nd_x);
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    if(debug !=0) {
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(x);
        return NULL;    
    }

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;

    if (nd_x == 0){
       dhelp = *px;
       *pret = erf(dhelp);
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            dhelp = *px;
            *pret = erf(dhelp);
            pret++;
            px++;
        }
    }
    
    Py_DECREF(x);
    return PyArray_Return(ret);  
}

static PyObject *
SpecfitFuns_ahypermet(PyObject *self, PyObject *args)
{
    double erfc(double);
    PyObject *input1, *input2;
    int debug=0;
    int tails=15;
    int expected_pars;
    int g_term_flag, st_term_flag, lt_term_flag, step_term_flag;
    /*double g_term, st_term, lt_term, step_term;*/
    PyArrayObject   *param, *x;
    PyArrayObject   *ret;
    int nd_param, nd_x, npars;
    int dim_param[2];
    int dim_x[2];
    int i, j, k;
    double  dhelp, log2, sqrt2PI,tosigma;
    double x1, x2, x3, x4, x5, x6, x7, x8;
    double z0, z1, z2;
    double  *px, *pret;
    typedef struct {
        double  area;
        double  position;
        double  fwhm;
        double  st_area_r;
        double  st_slope_r;
        double  lt_area_r;
        double  lt_slope_r;
        double  step_height_r;
    } hypermet;
    hypermet *phyper;

    /** statements **/
    if (!PyArg_ParseTuple(args, "OO|ii", &input1,&input2,&tails,&debug))
        return NULL;

    param = (PyArrayObject *)
             PyArray_CopyFromObject(input1, PyArray_DOUBLE,0,0);
    if (param == NULL)
        return NULL;
    x = (PyArrayObject *)
             PyArray_CopyFromObject(input2, PyArray_DOUBLE,0,0);
    if (x == NULL){
        Py_DECREF(param);
        return NULL;
    }    

    nd_param = param->nd;
    nd_x = x->nd;
    if(debug !=0) {
        printf("nd_param = %d nd_x = %d\n",nd_param,nd_x);
    }
    if (nd_param == 1) {
        dim_param [0] = param->dimensions[0];
        dim_param [1] = 0;
    }else{
        dim_param [0] = param->dimensions[0];
        dim_param [1] = param->dimensions[1];
    }

    if (nd_x == 1) {
        dim_x [0] = x->dimensions[0];
        dim_x [1] = 0;
    }else{
        if (nd_x == 0) {
            dim_x [0] = 0;
            dim_x [1] = 0;        
        }else{
            dim_x [0] = x->dimensions[0];
            dim_x [1] = x->dimensions[1];
        }
    }

    /* The gaussian terms must always be there */
    if(tails <= 0){
        /* I give back a matrix filled with zeros */
        ret = (PyArrayObject *)
        PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
        if (ret == NULL){
            Py_DECREF(param);
            Py_DECREF(x);
            return NULL;    
        }else{
            Py_DECREF(param);
            Py_DECREF(x);
            return PyArray_Return(ret);
        }
    }else{
        g_term_flag    = tails & 1;
        st_term_flag   = (tails>>1) & 1;
        lt_term_flag   = (tails>>2) & 1;
        step_term_flag = (tails>>3) & 1;
    }
    if (debug){
        printf("flags g = %d st = %d lt = %d step = %d\n",\
               g_term_flag,st_term_flag,lt_term_flag,step_term_flag);
    }
    expected_pars = 3 + st_term_flag * 2+lt_term_flag * 2+step_term_flag * 1;
    expected_pars = 8;    
    if (nd_param == 1) {
        npars = dim_param[0];
    }else{
        npars = dim_param[0] * dim_param[1];
    }
    if ((npars%expected_pars) != 0) {
        printf("Incorrect number of parameters\n");
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;
    }

    if(debug !=0) {
        printf("parameters %d raws and %d cols\n",dim_param[0],dim_param[1]); 
        printf("nparameters = %d\n",npars);
        printf("x %d raws and %d cols\n",dim_x[0],dim_x[1]); 
    }

    /* Create the output array */
    ret = (PyArrayObject *)
    PyArray_FromDims(nd_x, dim_x, PyArray_DOUBLE);
    if (ret == NULL){
        Py_DECREF(param);
        Py_DECREF(x);
        return NULL;    
    }

    log2 = 0.69314718055994529;
    sqrt2PI= sqrt(2.0*M_PI);
    tosigma=1.0/(2.0*sqrt(2.0*log2));

    /* the pointer to the starting position of par data */
    px = (double *) x->data;
    pret = (double *) ret->data;
    phyper = (hypermet *) param->data;
if(0){
if(debug !=0){
    for (i=0;i<(npars/expected_pars);i++){       
printf("Area%d=%f,Pos%d=%f,FWHM%d=%f\n",i,phyper[i].area,i,phyper[i].position,i,phyper[i].fwhm);
    }
}
}
    if (nd_x == 0){
       *pret = 0;
        phyper = (hypermet *) param->data;
        for (i=0;i<(npars/expected_pars);i++){
            /* g_term = st_term = lt_term = step_term = 0; */
            x1 = phyper[i].area;
            x2 = phyper[i].position;
            x3 = phyper[i].fwhm*tosigma;
            /* some intermediate variables */
            z0 = *px - x2;
            z1 = x3 * 1.4142135623730950488;
            /*I should check for sigma = 0 */
            if (x3 != 0) {
                z2 = (0.5 * z0 * z0) / (x3 * x3);
            }else{
                /* I should raise an exception */
                printf("Linear Algebra Error: Division by zero\n");
printf("Area=%f,Position=%f,FWHM=%f\n",x1,x2,phyper[i].fwhm);
printf("ST_Area=%f,ST_Slope=%f\n",phyper[i].st_area_r,phyper[i].st_slope_r);
printf("LT_Area=%f,LT_Slope=%f\n",phyper[i].lt_area_r,phyper[i].lt_slope_r);
                Py_DECREF(param);
                Py_DECREF(x);
                Py_DECREF(ret);
                return NULL;
            }
            if (g_term_flag){
                if (z2 < 612) {
                    *pret += exp (-z2) * (x1/(x3*sqrt2PI));
                }
            }
            if (st_term_flag){
                x4 = phyper[i].st_area_r;
                x5 = phyper[i].st_slope_r;
                if ((x5 != 0) && (x4 != 0)){
                    dhelp = x4 * 0.5 * erfc((z0/z1) + 0.5 * z1/x5);
                    if (dhelp != 0.0){
                    if (fabs(z0/x5) <= 612){
                *pret += ((x1 * dhelp)/x5) * exp(0.5 * (x3/x5) * (x3/x5)) \
                                      * exp(z0/x5);   
                    }
                    }
                } 
            }
            if (lt_term_flag){
                x6 = phyper[i].lt_area_r;
                x7 = phyper[i].lt_slope_r;
                if ((x7 != 0) && (x6 != 0)){
                    dhelp = x6 * 0.5 * erfc((z0/z1) + 0.5 * z1/x7);
                    if (fabs(z0/x7) <= 612){
                *pret += ((x1 * dhelp)/x7) * exp(0.5 * (x3/x7) * (x3/x7)) \
                                      * exp(z0/x7);  
                    }
                } 
            }
            if (step_term_flag){
                x8 = phyper[i].step_height_r;
                if ((x8 != 0) && (x3 != 0)){
                *pret +=  x8 * (x1/(x3*sqrt2PI)) * 0.5 * erfc(z0/z1);
                }
            }            
        }
    }else{
        k = 1;
        for (j=0;j<nd_x;j++){
            k = dim_x [j] * k;
        }
        for (j=0;j<k;j++){
            *pret = 0;
            phyper = (hypermet *) param->data;
         for (i=0;i<(npars/expected_pars);i++){
            /* g_term = st_term = lt_term = step_term = 0; */
            x1 = phyper[i].area;
            x2 = phyper[i].position;
            x3 = phyper[i].fwhm * tosigma;
            /* some intermediate variables */
            z0 = *px - x2;
            z1 = x3 * 1.4142135623730950488;
            /*I should check for sigma = 0 */
            if (x3 != 0) {
                z2 = (0.5 * z0 * z0) / (x3 * x3);
            }else{
                /* I should raise an exception */
                printf("Linear Algebra Error: Division by zero\n");
printf("Area=%f,Position=%f,FWHM=%f\n",x1,x2,phyper[i].fwhm);
printf("ST_Area=%f,ST_Slope=%f\n",phyper[i].st_area_r,phyper[i].st_slope_r);
printf("LT_Area=%f,LT_Slope=%f\n",phyper[i].lt_area_r,phyper[i].lt_slope_r);
                Py_DECREF(param);
                Py_DECREF(x);
                Py_DECREF(ret);
                return NULL;            
            }
            if (g_term_flag){
                if (z2 < 612) {
                    *pret += exp (-z2) * (x1/(x3*sqrt2PI));
                }
            }
            if (st_term_flag){
                x4 = phyper[i].st_area_r;
                x5 = phyper[i].st_slope_r;
                if ((x5 != 0) && (x4 != 0)){
                    dhelp = x4 * 0.5 * erfc((z0/z1) + 0.5 * z1/x5);
                    if (dhelp != 0){
                    if (fabs(z0/x5) <= 612){
                *pret += ((x1 * dhelp)/x5) * exp(0.5 * (x3/x5) * (x3/x5)) \
                                      * exp(z0/x5);   
                    }
                    }
                } 
            }
            if (lt_term_flag){
                x6 = phyper[i].lt_area_r;
                x7 = phyper[i].lt_slope_r;
                if ((x7 != 0) && (x6 != 0)){
                    dhelp = x6 * 0.5 * erfc((z0/z1) + 0.5 * z1/x7);
                    if (fabs(z0/x7) <= 612){
                *pret += ((x1 * dhelp)/x7) * exp(0.5 * (x3/x7) * (x3/x7)) \
                                      * exp(z0/x7);  
                    }
                } 
            }
            if (step_term_flag){
                x8 = phyper[i].step_height_r;
                if ((x8 != 0) && (x3 != 0)){
                *pret +=  x8 * (x1/(x3*sqrt2PI)) * 0.5 * erfc(z0/z1);
                }
            }            
         }
            pret++;
            px++;
        }
    }
    
    Py_DECREF(param);
    Py_DECREF(x);
    return PyArray_Return(ret);  
}



static PyObject *
SpecfitFuns_seek(PyObject *self, PyObject *args)
{    
    long SpecfitFuns_seek2(long , long, long,
                  double, double, double, long , long,
                  long,
                  double, double, long,
                  double *, long, long, double *, long *, double *);
    
    /* required input parameters */              
    PyObject *input;        /* The array containing the y values */
    long    BeginChannel;   /* The first channel to start search */
    long    EndChannel;     /* The last channel of the search */
    double  FWHM;           /* The estimated FWHM in channels */

    /* optional input parameters */
    double  Sensitivity = 3.5;
    double  debug_info = 0;
    
    /* some other variables required by the fortran function */
    long    FixRegion = 1;  /* Set to 1 if the program cannot adjust
                               the fitting region */
    double  LowDistance = 5.0;
    double  HighDistance = 3.0;
    long    AddInEmpty  = 0;
    long    channel1;
    long    channel2;
    long    npeaks;
    long    Ecal = 0;
    double  E[2];

    /* local variables */
    PyArrayObject    *yspec, *result;
    long        i, nchannels; 
    long        NMAX_PEAKS = 40;
    double      peaks[40];
    long        seek_result;    
    double      *pvalues;
    long        nd;
    int         dimensions[1];
    
    /* statements */        
    if (!PyArg_ParseTuple(args, "Olld|dd", &input, &BeginChannel,
                                    &EndChannel, &FWHM, &Sensitivity,
                                    &debug_info ))
        return NULL;
    yspec = (PyArrayObject *)
             PyArray_CopyFromObject(input, PyArray_DOUBLE,0,0);
    if (yspec == NULL)
        return NULL;
    if (Sensitivity < 0.1) {
        Sensitivity = 3.25;
    }

    nd = yspec->nd;
    if (nd == 0) {
        printf("I need at least a vector!\n");
        Py_DECREF(yspec);
        return NULL;
    }

    nchannels = yspec->dimensions[0];

    if (nd > 1) {
        if (nchannels == 1){
            nchannels = yspec->dimensions[0];
        }
    }

    pvalues = (double *) yspec->data;

    seek_result=SpecfitFuns_seek2(BeginChannel, EndChannel, nchannels,
                    FWHM, Sensitivity, debug_info,channel1, channel2,
                    FixRegion,
                    LowDistance, HighDistance, NMAX_PEAKS,
                    pvalues, AddInEmpty, Ecal,
                    E, &npeaks, peaks);



    Py_DECREF(yspec);
    
    if(seek_result != 0) {
        return NULL;
    }
    dimensions [0] = npeaks;
    result = (PyArrayObject *) 
             PyArray_FromDims(1,dimensions,PyArray_DOUBLE);
    for (i=0;i<npeaks;i++){
       /*printf("Peak %ld found at %g\n",i+1,peaks[i]);*/
       *((double *) (result->data +i*result->strides[0])) =  peaks[i];
    }
    return PyArray_Return(result);  
}


long SpecfitFuns_seek2(long BeginChannel, long EndChannel,
      long nchannels, double FWHM, double Sensitivity,double debug_info,
      long channel1, long channel2, long FixRegion,
      double LowDistance, double HighDistance,long max_npeaks,
      double *yspec, long AddInEmpty, long Ecal,double *E,
      long *n_peaks, double *peaks)
{
    /* local variables */
    double  sigma, sigma2, sigma4;
    long    max_gfactor = 100;
    double  gfactor[100];
    long    nr_factor;
    double  sum_factors;
    double  lowthreshold;
    double  yspec2[2];
    double  nom;
    double  den2;
    long    begincalc, endcalc;
    long    lld;
    long    cch;
    long    cfac, cfac2;
    double  distance;
    long    i, j;
    double  peakstarted = 0;

    /* statements */
    distance=MAX(LowDistance, HighDistance);
    
    /* Make sure the peaks matrix is filled with zeros */
    for (i=0;i<max_npeaks;i++){
        peaks[i] = 0.0;
    }    

    /* prepare the calculation of the Gaussian scaling factors */
    
    sigma = FWHM / 2.35482;
    sigma2 = sigma * sigma;
    sigma4 = sigma2 * sigma2;
    lowthreshold = 0.01 / sigma2;
    sum_factors = 0.0;
    
    /* calculate the factors until lower threshold reached */
    j = MIN(max_gfactor, ((EndChannel - BeginChannel -2)/2)-1);
    for (cfac=1;cfac<j+1;cfac++) {
        cfac2 = cfac * cfac;
        gfactor[cfac-1] = (sigma2 - cfac2) * exp (-cfac2/(sigma2*2.0)) / sigma4;
        sum_factors += gfactor[cfac-1];
        /*printf("gfactor[%ld] = % g\n",cfac,gfactor[cfac-1]);*/
        if ((gfactor[cfac-1] < lowthreshold) 
        && (gfactor[cfac-1] > (-lowthreshold))){
            break;
        }  
    }
    /*printf("sum_factors = %g\n",sum_factors);*/
    nr_factor = cfac;
    
    /* What comes now is specific to MCA spectra ... */
    /*lld = 7;*/
    lld = 0;
    while (yspec [lld] == 0) {
        lld++;
    }
    lld = lld + 0.5 * FWHM;
    
    channel1 = BeginChannel - nr_factor - 1;
    channel1 = MAX (channel1, lld);
    begincalc = channel1+nr_factor+1;
    endcalc = MIN (EndChannel+nr_factor+1, nchannels-nr_factor-1);
    *n_peaks = 0;
    cch = begincalc;
    
    /* calculates second derivative at begincalc */
    nom = yspec[cch] / sigma2;
    den2 = yspec[cch] / sigma4;
    
    for (cfac = 1; cfac < nr_factor; cfac++){
        nom += gfactor[cfac-1] * (yspec[cch+cfac] + yspec [cch-cfac]);
        den2 += gfactor[cfac-1] * gfactor[cfac-1] *
                 (yspec[cch+cfac] + yspec [cch-cfac]);    
    }
    
    /* now calculate the 2nd derivative out of the smoothed value */
    if (den2 <= 0.0) {
        yspec2[1] = 0.0;
    }else{
        yspec2[1] = nom / sqrt(den2);
    }
    yspec[0] = yspec[1];
    
    while (cch <= endcalc){
        /* calculate gaussian smoothed values */
        yspec2[0] = yspec2[1];
        cch++;
        nom = yspec[cch]/sigma2;
        den2 = yspec[cch] / sigma4;
        for (cfac = 1; cfac < nr_factor;cfac++){
            nom += gfactor[cfac-1] * (yspec[cch+cfac] + yspec [cch-cfac]);
            den2 += gfactor[cfac-1] * gfactor[cfac-1] *
                 (yspec[cch+cfac] + yspec [cch-cfac]);    
        }
        
        /* now calculate the second derivative out of the smoothed values */
        if (den2 <= 0) {
            yspec2[1] = 0;
        }else{
            yspec2[1] = nom / sqrt(den2);
        }
        if (cch == endcalc)
            yspec2[1] = 0.0;
 
                
        /* look if the current point falls in a peak */
        if (yspec2[1] > Sensitivity) {
            if(peakstarted == 0){
                peakstarted=1;
            }
            /* there is a peak */
            if (debug_info){
                printf("At cch = %ld y[cch] = %g ",cch,yspec[cch]);
                printf("yspec2[0] = %g ",yspec2[0]);
                printf("yspec2[1] = %g ",yspec2[1]);
                printf("Sensitivity = %g\n",Sensitivity);
            }
            if(peakstarted == 1){
                /* look for the top of the peak */
                if (yspec2[1] < yspec2 [0]) {                
                    /* we are close to the top of the peak */
                    if (debug_info){
                        printf("we are close to the top of the peak\n");
                    }
                    if (*n_peaks < max_npeaks) {
                        peaks [*n_peaks] = cch-1;
                        (*n_peaks)++;
                        peakstarted=2;
                    }else{
                        printf("Found too many peaks\n");
                        return (-2);
                    }
                }
            }
            /* Doublet case */
            if(peakstarted == 2){
                if ((cch-peaks[(*n_peaks)-1]) > FWHM) {
                    if (yspec2[1] > yspec2 [0]){
                        if(debug_info){
                            printf("We may have a doublet\n");
                        }
                        peakstarted=1;
                    }
                }
            }
        }else{
            if (peakstarted==1){
            /* We were on a peak but we did not find the top */
                if(debug_info){
                    printf("We were on a peak but we did not find the top\n");
                }
            }
            peakstarted=0;       
        }
    }
    if(debug_info){
      for (i=0;i< *n_peaks;i++){
        printf("Peak %ld found at ",i+1);
        printf("index %g with y = %g\n",peaks[i],yspec[(long ) peaks[i]]);
      }
    }

    return (0);
}

double erfc(double x)
{
    double z;
    double t;
    double r;
    
    z=fabs(x);
    t=1.0/(1.0+0.5*z);
    r=t * exp(-z * z - 1.26551223 + t * (1.00002368 + t * (0.3740916 +
      t * (0.09678418 + t * (-0.18628806 + t * (0.27886807 + t * (-1.13520398 +
      t * (1.48851587 + t * (-0.82215223+t*0.17087277)))))))));
    if (x<0) 
       r=2.0-r;
    return (r);
}

double erf(double x)
{
    double z;
    double t;
    double r;
    
    z=fabs(x);
    t=1.0/(1.0+0.5*z);
    r=t * exp(-z * z - 1.26551223 + t * (1.00002368 + t * (0.3740916 +
      t * (0.09678418 + t * (-0.18628806 + t * (0.27886807 + t * (-1.13520398 +
      t * (1.48851587 + t * (-0.82215223+t*0.17087277)))))))));
    if (x<0) 
       r=2.0-r;
    return (1.0-r);
}


/* List of functions defined in the module */

static PyMethodDef SpecfitFuns_methods[] = {
	{"subac",		SpecfitFuns_subac,		METH_VARARGS},
    {"gauss",       SpecfitFuns_gauss,      METH_VARARGS},
    {"agauss",      SpecfitFuns_agauss,     METH_VARARGS},
    {"alorentz",    SpecfitFuns_alorentz,   METH_VARARGS},
    {"lorentz",     SpecfitFuns_lorentz,    METH_VARARGS},
    {"apvoigt",     SpecfitFuns_apvoigt,    METH_VARARGS},
    {"pvoigt",      SpecfitFuns_pvoigt,     METH_VARARGS},
    {"downstep",    SpecfitFuns_downstep,   METH_VARARGS},
    {"upstep",      SpecfitFuns_upstep,     METH_VARARGS},
    {"slit",        SpecfitFuns_slit,       METH_VARARGS},
    {"ahypermet",   SpecfitFuns_ahypermet,  METH_VARARGS},
    {"erfc",        SpecfitFuns_erfc,       METH_VARARGS},
    {"erf",         SpecfitFuns_erf,        METH_VARARGS},
    {"seek",        SpecfitFuns_seek,       METH_VARARGS},
	{NULL,		NULL}		/* sentinel */
};


/* Initialization function for the module (*must* be called initSpecfitFuns) */

DL_EXPORT(void)
initSpecfitFuns(void)
{
	PyObject *m, *d;

	/* Initialize the type of the new type object here; doing it here
	 * is required for portability to Windows without requiring C++. */
	SpecfitFunso_Type.ob_type = &PyType_Type;

	/* Create the module and add the functions */
	m = Py_InitModule("SpecfitFuns", SpecfitFuns_methods);
    import_array();

	/* Add some symbolic constants to the module */
	d = PyModule_GetDict(m);
	ErrorObject = PyErr_NewException("SpecfitFuns.error", NULL, NULL);
	PyDict_SetItemString(d, "error", ErrorObject);
}
