/* NAME
   specscan

   DESCRIOPTION
   Extraction of scans from spec files and conversion to PC readable text 
   files 

   HISTORY
   1999-04-23 V1.0 Peter Boesecke 
   1999-04-27 V1.1 PB argument counting corrected
   1999-04-27 V1.2 PB space added after number ("#S 4" != "#S 49")
*/
/*=========================================================================*/
/* Public                                                                  */
/*=========================================================================*/

# include <stdlib.h>
# include <errno.h>
# include <stdio.h>
# include <unistd.h>

# include <ctype.h>
# include <string.h>
# include <time.h>

# define BUFLEN 1024

# define STATUS_SUCCESS 0
# define STATUS_OPENERROR 1
# define STATUS_READERROR 2

# define SPECSCAN_VERSION "1.2 (PB 1999-04-27)"

char end_of_line[3] = { '\r', '\n', '\0' };

/* removes end_of_line from line */
char * rmeoln ( char * line )
{ char * pc = line;

  
  if (pc)
    while (*pc) {
      if (strchr( end_of_line, (int) *pc )) *pc='\0';
      pc++;
      }
  return( line );

} /* rmeoln */

/* spec2kal

   Description
   Extraction of a scan from a spec file and adding cr-lf at line end 

   Arguments
   const char * onam : outfput file name
   const char * fnam : spec file name
   const char * start_marker : marker string, where extraction should start 
                               (inclusive)
   const char * end_marker : marker string, where extraction ends (exclusive) 
   long int skip : number of start marker that should be skipped if the spec
                   file contains several scan files with the same start marker
*/

void spec2kal ( const char * onam, 
                const char * fnam, 
                const char * start_marker, 
                const char * end_marker, 
                long int skip,
                int *pstatus )
{ 
  FILE *input, *output;
  char buffer[BUFLEN];
  char line[BUFLEN];
  int  start_len = strlen(start_marker);
  int  end_len = strlen(end_marker);
  int  found = 0;

  if (*pstatus) return;

  sprintf(buffer,"function spec2kal, input file %s", fnam);
  input = fopen( fnam, "r");
  if (!input) { perror(buffer); *pstatus = STATUS_OPENERROR; return; }

  sprintf(buffer,"function spec2kal, output file %s", onam);
  output = fopen ( onam, "w" );
  if (!output) { perror(buffer); *pstatus = STATUS_OPENERROR; return; }

  /* search start line with start_marker and skip some */
  for (;skip>=0;skip--) {
    if (!(feof(input))) {
      sprintf(buffer,"function spec2kal, reading file %s", fnam);
      fgets (line, BUFLEN, input);
      if (ferror(input)) { 
        perror(buffer); *pstatus = STATUS_READERROR; return; }
      }

    while (!((feof(input))||(!strncmp(line,start_marker,start_len)))) {
      /*  printf("skipping %s\r\n",rmeoln(line)); */

      fgets (line, BUFLEN, input);
      if (ferror(input)) { 
        perror(buffer); *pstatus = STATUS_READERROR; return; }
      }
    }

  /* convert lines until next line starting with end_marker */
  
  if (!(feof(input))) {
    fprintf(output,"%s\r\n",rmeoln(line));

    fgets (line, BUFLEN, input);
    if (ferror(input)) { perror(buffer); *pstatus = STATUS_READERROR; return; }

    } else { 
    printf("Start marker %s not found\n",start_marker);
    fprintf(output,"Start marker %s not found\r\n",start_marker);
    }

  while (!((feof(input))||(!strncmp(line,end_marker,end_len)))) {
    fprintf(output,"%s\r\n",rmeoln(line));

    fgets (line, BUFLEN, input);
    if (ferror(input)) { perror(buffer); *pstatus = STATUS_READERROR; return; }

    }


  fclose( input );
  fclose( output );
 
  return;
} /* spec2kal */

main (int argc, char *argv[])
{ int status=STATUS_SUCCESS;
  char start_marker[BUFLEN];
  char end_marker[BUFLEN];
  char file_out[BUFLEN];
  long int skip = 0l;

  if (argc<=2) {
    printf("Extraction of a scan from a spec scan file\n");
    printf("%s version: %s\n", argv[0], SPECSCAN_VERSION);
    printf("Usage: %s fnam scanno [skip]\n",argv[0]);
    exit(-1);
    }

  printf("\n");
  printf(" fnam   : %s\n",argv[1]);
  printf(" scanno : %s\n",argv[2]);
  if (argc>3) printf(" skip   : %s\n",argv[3]);
  printf("\n");

  sprintf(file_out,"%s_%s.txt\0",argv[1],argv[2]);
  sprintf(start_marker,"#S %s \0",argv[2]);
  sprintf(end_marker,"#S\0");
  if (argc>3) sscanf(argv[3],"%d",&skip);
  if (skip>0) sprintf(file_out,"%s_%ss%u.txt\0",argv[1],argv[2],skip); 

  printf("output file           : %s\n", file_out);
  printf("start extraction at   : %s\n", start_marker);
  printf("end extraction at     : %s\n", end_marker);
  printf("start markers to skip : %d\n", skip);
  printf("\n");

  spec2kal ( file_out, argv[1], start_marker, end_marker, skip, &status );
  if (status) { 
    fprintf(stderr,"ERROR: status %d, exiting %s\n",status, argv[0]); 
    exit(-1); 
    }
  
}

