/*+++
1 saxs_new
  Generates a new image with constant pixel values

2 PURPOSE
  Generates a new image with constant pixel values

  Arguments:
  saxs_new [options] <onam> <odum> <dim 1> <dim 2> <ocon>"

  Defaults:
  output file name    : <onam>              : output.edf
  output dummy        : <odum>              : 0.0
  dimension 1         : <odim 1>            : 1
  dimension 2         : <odim 2>            : 1
  output constant     : <ocon>              : 0.0 (only used if dummy = 0)

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2000-11-24  PB V3.0
  2001-01-10  PB V3.1 GetReference with output image
  2001-07-09  PB V4.00 new loop, no repetition

---*/
# define Version  "saxs_new V4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
                <onam> <odum> <odim1> <odim2> <ocon>\n\
                Generates an image with constant pixel values"

# define BlockNum 1       /* 1 output sequence */

/*---------------------------------------------------------------------------
1 saxs_new

2 PURPOSE
  Generates a new image with constant pixel values
---------------------------------------------------------------------------*/
void saxs_new (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float Value;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1, W_2;
  float DW_1, DW_2;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=1) {
     printf("%d images found, 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

     GetImageOverlap       ( pcb, ihb, 1, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2,
                             fmin_1, fmax_1, fmin_2, fmax_2,
                             &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

 /* loop over 1 output image */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

  /* loop over the output array  */
  for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
  for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1[0],i_2);
    for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
    for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
        /* do something with the data */

        Value = 0.0;

        UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

      pI0Data++;
      for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
      } /* for i_1 ... */
    for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
    } /* for i_2 ... */

 } /* saxs_new*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,0,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,0,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : constant */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_float(pcb,"Constant (only if dummy = 0)",&ib[0].Const,0.0,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
  if (ib[0].Const.I)   printf("constant           : %g\n",ib[0].Const.V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_new, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

