/*+++
1 saxs_ascii
  Output of text files. One for each image. 

2 PURPOSE
  This program writes an ascii file that contains the value of
  each pixel. A header can preceed the file. The first column and/or
  row can contain the array, image, real or saxs coordinates of the file. 

  Arguments:
  saxs_ascii [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> 
             <header print> <swap output> <head 1> <head 2>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <output dummy>      : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <header print>      : on   * to print the header in the ascii file
  <swap output>       : off  * to transpose the output in the file
  <head 1>            : on   * to see the i_1 world coordinate at the top
  <head 2>            : on   * to see the i_2 world coordinate on the left

  Specific options:
  -pref 'prefix'      : specify the prefix of the ouptput file
  -ext 'extension'    : extension of the output file (default: "txt")
  -hdmk 'marker'      : first character in a header line (default: "#")
  -spc 'spacer'       : spacer string between columns (default: "\t")

  +-head              : enables/disables the header printing
  +-swap              : enables/disables the swap
  +-hd1               : enables/disables the i_1 coordinate printing
  +-hd2               : enables/disables the i_2 coordinate printing
  +-abs               : absolut values of coordinates
  -scf <factor>       : multiplication of coordinates with float <factor>
  +-hist              : writes/do not write history into header
  +-noi               : suppress/do not suppress image number extensions in output file

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  1995        Pierre Jourde
  01-Dec-1995 PB
  27-Nov-1996 PB error in i_1 header in non-swapped data corrected
  03-Dec-1996 PB option +abs (absolut values of coordinates) and option -scf
  01-Aug-1998 PB V3.21 Segmentation error in Linux version due to an error 
                       in writing info pcb->Lin[NPrefix]: .V extension
                       forgotten, now corrected.
  01-Aug-1998 PB V3.3  All new lines '\n' replaced by '\r\n'
  25-Feb-1999 PB V3.4  Header marker can be changed with -hdmk <marker>.
                       Default is "#".
  27-Mar-2000 PB V3.5  option for spacer between columns (-spc <spacer>)
  2000-08-04  PB V3.5  %f->%g
  2000-08-15  PB V3.6  first column header changed
  2000-09-15  PB V3.7  option -hedl for head line string
  2000-11-18  PB V3.8  WriteHistory, WriteHeader, Time keyword into header
                       new options: +/-hist (NHist), -len  (NHlen)
  2000-11-18  PB V3.9  default header line starts with #
  2001-03-02  PB V3.10 space before \r\n
  2001-03-30  PB V3.11a space not in label line
  2001-07-03  PB V3.12 filename_full
  2001-07-07  PB V3.13 IO_DontReadWrite
  2001-07-08  PB V3.14 FFirst
  2001-07-09  PB V3.15 ib[0].FileNumber
  2001-07-09  PB V4.00 new loop, no repetition
  2001-08-30  PB V4.01 
  2001-09-02  PB V4.02 option noi (switch on/off suppression of image numbers)

---*/
# define Version  "saxs_ascii V4.02 2001-09-02, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <math.h>

# include "SaxsPrograms.h"

// line
# define SPrefix "pref"
# define NPrefix 0
# define SExtension "ext"
# define NExtension 1
# define SHeadMark "hdmk"
# define NHeadMark 2
# define SSpacer "spc"
# define NSpacer 3
# define SHeadL  "hedl"
# define NHeadL  4
// flag 
# define SihbPrint "head" // write header
# define NihbPrint 0
# define SHist     "hist" // write history
# define NHist     1
# define Si1Head   "hd1"  // write header 1
# define Ni1Head   2
# define Si2Head   "hd2"  // write header 2
# define Ni2Head   3
# define SAbs      "abs"  // write absolute values of coordinates
# define NAbs      4
# define SSwap     "swap" // swap columns and lines in output file
# define NSwap     5
# define SNoImNum  "noi"  // suppress image number in output, if not required
# define NNoImNum  6 

//float
# define SScf    "scf"  // scale factor for header column
# define NScf    0
// long
# define SHlen   "len"  // minimum number of header lines (if +head)
# define NHlen   0

# define Default_extension ".txt"

# define Default_ihbPrint  1            /* for on */
# define Default_swap      0            /* for off */
# define Default_head1     1            /* for on */
# define Default_head2     1            /* for on */

# define DevSeparator ":"
# define DirSeparator "/"    
# define ExtSeparator "."

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc>\n\
                <odum> <odim1> <odim2>\n\
                <header> <swap> <head_1> <head_2>\n\
Purpose: Transformation into ASCII format\n\
Options: +-head, +-swap, +-hd1, +-hd2, -hdmk <marker>, -ext <extension>, +-noi"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

const char *delim = " ";
 
char * str_tok( char * string, const char * delim )
{ char * label=(char*) NULL;
  label=strtok(string, delim);

  return( label );
} /* str_tok */

/*---------------------------------------------------------------------------
SaxsAsciiArgvFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

If blkno >= 1 the header block of the first image is read into ihb[1].
The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void SaxsAsciiArgvFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long image_1st, image_lst;
  char linebuffer[1000];
  int nameblk;

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib, blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile( pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {

    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadWrite | IO_DontOpen;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- Argument  : text file prefix */
    if (ib[0].Name.I) nameblk = 0; else nameblk = 1;
    if (!pcb->Lin[NPrefix].I) {
      /* Determine prefix from input file name */
      (void) RightFR(ib[nameblk].Name.V,DevSeparator,pcb->Lin[NPrefix].V,
                     InputLineLength);
      (void) RightFR(pcb->Lin[NPrefix].V,DirSeparator,pcb->Lin[NPrefix].V,
                     InputLineLength);
      (void) LeftFL(pcb->Lin[NPrefix].V,ExtSeparator,pcb->Lin[NPrefix].V,
                     InputLineLength);
      pcb->Lin[NPrefix].I = TRUE;
      } 

    /*--- Prefix of output file name */
    argv_line( pcb,"Ascii file prefix",&pcb->Lin[NPrefix],pcb->Lin[NPrefix].V, 
               pstatus);
    if (*pstatus!=Success) return;

    /* extract output filenumbers from ASCII file name */
    extract_filenumbers( &ib[0], &(pcb->Lin[NPrefix]), TRUE, pstatus );
    if (*pstatus!=Success) return;

    /* Expand Prefix to the full file name */
    if (!filename_full (pcb->Lin[NPrefix].V,IO_len,pcb->Lin[NPrefix].V)) {

      printf("Input file name or path not allowed\n");exit(-1);}

    /*--- Take prefix as default for output filename */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",pcb->Lin[NPrefix].V);
      } 

    } /* output file */

  /* ---  Image numbers --- */

  if (first_block==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

    CloseImageFile( pcb, ib, 1, pstatus) ;
    if (*pstatus!=Success) return;

    /* Defaults for output file */
    if (!ib[0].Dummy.I) ib[0].Dummy.V  = ihb[1].Dummy.V;
    if (!ib[0].Dim[1].I) ib[0].Dim[1].V = ihb[1].Dim[1];
    if (!ib[0].Dim[2].I) ib[0].Dim[2].V = ihb[1].Dim[2];
  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;

    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;

  } /* all other input files */

  if (block_1st <= 0) {
    /*--- Argument : output dummy */
    argv_float( pcb, "Output dummy", &ib[0].Dummy, ib[0].Dummy.V, pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 1 */
    argv_long(pcb,"Output dimension 1", &ib[0].Dim[1],ib[0].Dim[1].V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 2 */
    argv_long(pcb,"Output dimension 2", &ib[0].Dim[2],ib[0].Dim[2].V, pstatus);
    if (*pstatus!=Success) return;

    }

} /* SaxsAsciiArgvFilenames */

/*----------------------------------------------------------------------------
NAME
  WriteHistory

DESCRIPTION
  Reads all history lines from the input file ihb[1] and writes them to 'out'

RETURN VALUE
  Number of written lines

HISTORY
  18-Nov-2000 PB
----------------------------------------------------------------------------*/
long WriteHistory( FILE * out, CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus )
{
   const char * history_key_prefix = "History-";
   char history_line[InputLineLength];
   char history_key_buffer[InputLineLength];
   char *hdmk = pcb->Lin[NHeadMark].V;  /* shortcut */

   long depth=1, cnt=0;

   sprintf(history_key_buffer,"%s%u",history_key_prefix,depth++);
   while (ReadHeaderLine(pcb,1,ihb,history_key_buffer,history_line,pstatus)) {
     if (*pstatus != Success) return(cnt);
     fprintf(out,"%s%s = %s \r\n", hdmk, history_key_buffer, history_line);
     cnt++;
     sprintf(history_key_buffer,"%s%u",history_key_prefix,depth++);
     }

   *pstatus = Success;
   return(cnt);

}  /* WriteHistory */

/*----------------------------------------------------------------------------
NAME
  WriteHeader 

DESCRIPTION
  Writes the header 

RETURN VALUE
  Number of written lines

HISTORY
  18-Nov-2000 PB
----------------------------------------------------------------------------*/
long WriteHeader( FILE * out, CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus )
{ long cnt=0, cntmax;
  char *hdmk = pcb->Lin[NHeadMark].V;  /* shortcut */

  cntmax  = pcb->Num[NHlen].V; // header line counter

  fprintf(out,"%s{\r\n",hdmk); cnt++;
  fprintf(out,"%s%s = % d \r\n", hdmk, KImage, ihb->ImNum); cnt++;
  fprintf(out,"%s%s = % d \r\n", hdmk, "Size", ihb->ImgByteLen); cnt++;
  fprintf(out,"%s%s = % d \r\n", hdmk, "Dim_1", ihb->Dim[1]); cnt++;
  fprintf(out,"%s%s = % d \r\n", hdmk, "Dim_2", ihb->Dim[2]); cnt++;
  if (ihb->Dummy.I)        { fprintf(out,"%s%s = % e \r\n",
    hdmk, KDummy, ihb->Dummy.V); cnt++; }
  if (ihb->DDummy.I)       { fprintf(out,"%s%s = % e \r\n",
    hdmk, KDDummy, ihb->DDummy.V); cnt++; }
  if (ihb->Offset[1].I)    { fprintf(out,"%s%s = % e \r\n",
    hdmk, KOffset_1, ihb->Offset[1].V); cnt++; }
  if (ihb->Offset[2].I)    { fprintf(out,"%s%s = % e \r\n",
    hdmk, KOffset_2, ihb->Offset[2].V); cnt++; }
  if (ihb->Center[1].I)    { fprintf(out,"%s%s = % e \r\n",
    hdmk, KCenter_1, ihb->Center[1].V); cnt++; }
  if (ihb->Center[2].I)    { fprintf(out,"%s%s = % e \r\n",
    hdmk, KCenter_2, ihb->Center[2].V); cnt++; }
  if (ihb->PixSiz[1].I)    { fprintf(out,"%s%s = % e \r\n",
    hdmk, KPixSiz_1, ihb->PixSiz[1].V); cnt++; }
  if (ihb->PixSiz[2].I)    { fprintf(out,"%s%s = % e \r\n",
    hdmk, KPixSiz_2, ihb->PixSiz[2].V); cnt++; }
  if (ihb->SampleDistance.I) { fprintf(out,"%s%s = % e \r\n",
    hdmk, KSampleDistance, ihb->SampleDistance.V); cnt++; }
  if (ihb->WaveLength.I)   { fprintf(out,"%s%s = % e \r\n",
    hdmk, KWaveLength, ihb->WaveLength.V); cnt++; }
  if (ihb->Title.I)        { fprintf(out,"%s%s = % s \r\n",
    hdmk, KTitle, ihb->Title.V); cnt++; }
  if (ihb->Time.I)         { fprintf(out,"%s%s = % s \r\n",
    hdmk, KTime, ihb->Time.V); cnt++; }
  if (pcb->Flg[NHist].V)
    cnt += WriteHistory( out, pcb, ihb, pstatus );
  if (*pstatus != Success) return( cnt );

  for (cnt;cnt<cntmax-1;cnt++) {
    fprintf(out,"%s\r\n", hdmk);
    }
  fprintf(out,"%s}\r\n", hdmk); cnt ++;

  return( cnt );

} /* WriteHeader */

/*---------------------------------------------------------------------------
1 saxs_ascii

2 PURPOSE
  Output of ascii data
---------------------------------------------------------------------------*/

void saxs_ascii (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy;
  float *I1Data;
  float *pI1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float Value;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1=0.0, W_2=0.0;
  float DW_1=0.0, DW_2=0.0;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  char OutputFileName[InputLineLength];
  FILE *OutputFileId;
  char DefExtension[InputLineLength]; 
  char *hdmk = pcb->Lin[NHeadMark].V;  /* shortcut */
  char *label=(char *) NULL, *nextlabel=(char *) NULL; 
  char fname_full[IO_len];

  /* boolean indicators */
  int SwapI, i1HeadI, i2HeadI, AbsI;

  /* output coordinates */
  float O_1, O_2;
  float scf;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\nCalculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(i,&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

     GetImageOverlap       ( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2,
                             fmin_1, fmax_1, fmin_2, fmax_2,
                             &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

 /* loop over 1 input and 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (pcb->TestBit)
    printf("maxloop_1 = % d, maxloop_2 = % d\n",maxloop_1,maxloop_2);

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  if (pcb->TestBit)
    printf("Wmin_1 = % g, DW_1 = % g, Wmin_2 = % g, DW_2 = % g.\n",
            Wmin_1,DW_1,Wmin_2,DW_2);

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

  /* loop over the output array */
  W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
  for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
    W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
    for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {

      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);

      if (Ipol2ld (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
        f_1[1], f_2[1], &Value)) {
        /* then do something with the data */
        *pI0Data = Value;
        } /* if (Ipol2ld ... */

      W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
      } /* for i_2 ... */
    W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */

  /*--- Get default extension */
  if (pcb->Lin[NExtension].I) {
    (void) sprintf(DefExtension,"%s",pcb->Lin[NExtension].V);
    } else {
    (void) sprintf(DefExtension,"%s",Default_extension);
    }

  /* building of the file name from pcb->Lin[NPrefix] and
    the number of the image in the sequence */
  
  /* Generate filename of ASCII file */
  filename_parameter (fname_full,IO_len,pcb->Lin[NPrefix].V,0);
  filename_pattern (fname_full,IO_len,fname_full,(pcb->ib)[0].FileNumber);
  filename_full (fname_full,IO_len,fname_full);

  if ((pcb->Flg[NNoImNum].V) && ImageIsFirst(pcb, 0, ihb ) && ImageIsLast(pcb, 0, ihb )) 
    /* suppress image number if not required */
    sprintf(OutputFileName,"%s%s",fname_full, DefExtension);
   else
    /* write image number always */
    sprintf(OutputFileName,"%s_%d%s",fname_full,ihb[0].ImNum, DefExtension);

  /* open ascii file "pcb->Lin[NPrefix]"_NNN.txt" */
  if ((OutputFileId=fopen(OutputFileName,"w")) == NULL) return;

  printf("Writing output ascii file : %s\n",OutputFileName);

  /* writing of the image header block into the ouput file */
  if (pcb->Flg[NihbPrint].V) {
    /* Writing header */
     WriteHeader( OutputFileId, pcb, ihb, pstatus );
     if (*pstatus!=Success) return;
    } 

  /* set the boolean indicators */
  SwapI = pcb->Flg[NSwap].V;
  i1HeadI = pcb->Flg[Ni1Head].V;
  i2HeadI = pcb->Flg[Ni2Head].V;
  AbsI = pcb->Flg[NAbs].V;
  if ( pcb->Arg[NScf].I) scf = pcb->Arg[NScf].V;
   else scf = 1.0; /* scale factor for coordinates */

  if (pcb->TestBit) 
    printf("\nFlag : Head = %d, Swap = %d, Hd1 = %d, Hd2 = %d, Abs = %d\n",
           pcb->Flg[NihbPrint].V,SwapI, i1HeadI, i2HeadI, AbsI);

  /* writing of the x (or y if swap) heading line if wished */
  W_1 = Wmin_1;
  W_2 = Wmin_2;

  if (AbsI) { O_1 = fabs(scf*W_1);  O_2 = fabs(scf*W_2); 
   } else { O_1 = scf*W_1; O_2 = scf*W_2; }

  /* Writing data */

  if (pcb->Lin[NHeadL].I) nextlabel = pcb->Lin[NHeadL].V;


  if (SwapI) {
    /* Swapped */
    if (i2HeadI) {
      if (nextlabel) label = str_tok(nextlabel, delim);

      if (i1HeadI) {

        if (label) {
          fprintf(OutputFileId,"%s%s",label,pcb->Lin[NSpacer].V);
         } else {
          fprintf(OutputFileId,"#i_1v/i_2>%s",pcb->Lin[NSpacer].V);
         }
        if (nextlabel) label = str_tok((char*) NULL, delim);

       } // if (i1HeadI)

      if (label) {
        fprintf(OutputFileId,"%s",label);
       } else {
        fprintf(OutputFileId,"%g",O_2);
       }

      for (i_2=1; i_2<I0Dim_2; i_2++) {
        W_2 += DW_2;
        if (AbsI) O_2 = fabs(scf*W_2); else O_2 = scf*W_2; 
  
        if (nextlabel) label = str_tok((char *) NULL, delim);
 
        if (label) {
          fprintf(OutputFileId,"%s%s",pcb->Lin[NSpacer].V,label);
         } else {
          fprintf(OutputFileId,"%s%g",pcb->Lin[NSpacer].V,O_2);
         }
       }
     fprintf(OutputFileId,"\r\n");
     } // if (i2HeadI)
      
   } else {
    /* not swapped */
    if (i1HeadI) {
      if (nextlabel) label = str_tok(nextlabel, delim);

      if (i2HeadI) {

        if (label) {
          fprintf(OutputFileId,"%s%s",label,pcb->Lin[NSpacer].V);
         } else {
          fprintf(OutputFileId,"i_2v/i_1>%s",pcb->Lin[NSpacer].V);
         }
        if (nextlabel) label = str_tok((char *) NULL, delim);
       }

      if (label) { 
        fprintf(OutputFileId,"%s",label);
       } else {
        fprintf(OutputFileId,"%g",O_1);
       }

    for (i_1=1; i_1<I0Dim_1; i_1++) {
      W_1 += DW_1;
      if (AbsI) O_1 = fabs(scf*W_1); else O_1 = scf*W_1; 
      if (nextlabel) label = str_tok((char *) NULL, delim);

      if (label) {
        fprintf(OutputFileId,"%s%s",pcb->Lin[NSpacer].V,label);
       } else {
        fprintf(OutputFileId,"%s%g",pcb->Lin[NSpacer].V,O_1);
       }
     }
    fprintf(OutputFileId,"\r\n");
    } // if (i1Head1)

  } // if not swapped 

  /* Write the array I0Data[I0Dim_1,I0Dim_2] to the file
     array is written by lines : each row of the line is separated
     from the next by a tab caracter (ASCII 9), each line by a RC */

  if (SwapI) { /* if swapped output */
    W_1= Wmin_1;
    if (AbsI) { O_1 = fabs(scf*W_1);  O_2 = fabs(scf*W_2);
      } else { O_1 = scf*W_1; O_2 = scf*W_2; }

    for (i_1=0; i_1<I0Dim_1; i_1++) {
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,0);

      if (i1HeadI) {
        fprintf(OutputFileId,"%g%s",O_1,pcb->Lin[NSpacer].V);
        W_1 += DW_1;
        if (AbsI) O_1 = fabs(scf*W_1); else O_1 = scf*W_1;
        }

      fprintf(OutputFileId,"%g",*pI0Data);
      for (i_2=1; i_2<I0Dim_2; i_2++) {
        pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);
        fprintf(OutputFileId,"%s%g",pcb->Lin[NSpacer].V,*pI0Data);
        }

      fprintf(OutputFileId," \r\n");
      }
    } else { /* if not swapped output */
    W_2= Wmin_2;
    if (AbsI) O_2 = fabs(scf*W_2); else O_2 = scf*W_2; 
    for (i_2=0; i_2<I0Dim_2; i_2++) {
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,0,i_2);
 
      if (i2HeadI) {
        fprintf(OutputFileId,"%g%s",O_2,pcb->Lin[NSpacer].V);
        W_2 += DW_2;
        if (AbsI) O_2 = fabs(scf*W_2); else O_2 = scf*W_2; 
        }

      fprintf(OutputFileId,"%g",*pI0Data);

      for (i_1=1; i_1<I0Dim_1; i_1++) {
        pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);
        fprintf(OutputFileId,"%s%g",pcb->Lin[NSpacer].V,*pI0Data);
        }

      fprintf(OutputFileId," \r\n");
      }
   }

 /* close ascii file */
 if (fclose(OutputFileId)!=0) return;

} /* saxs_ascii*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,
           InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  /* Expand Prefix to the full file name */
  if (!filename_full (pcb->Lin[NPrefix].V,IO_len,pcb->Lin[NPrefix].V)) {

     printf("% s. Input file name or path not allowed\n",progfn);exit(-1);}
 
  SaxsAsciiArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : image header block printing authorization */
  argv_flag(pcb,"Write header block (yes|no)",
            &pcb->Flg[NihbPrint],Default_ihbPrint,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : swapping of the output file */
  argv_flag(pcb,"Swap output (yes|no)",
            &pcb->Flg[NSwap],Default_swap,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : i_1 heading in the output file authorization */
  argv_flag(pcb,"Write column coordinates (yes|no)",
            &pcb->Flg[Ni1Head],Default_head1,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : i_2 heading in the output file authorization */
  argv_flag(pcb,"Write row coordinates (yes|no)",
            &pcb->Flg[Ni2Head],Default_head2,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)   printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)   printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)  printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)   printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)    printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)  printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I) printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I) printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
  if (pcb->Flg[NihbPrint].I)
                      printf("header print       : %d\n",pcb->Flg[NihbPrint].V);
  if (pcb->Flg[NSwap].I)
                      printf("swap               : %d\n",pcb->Flg[NSwap].V);
  if (pcb->Flg[Ni1Head].I)
                      printf("i_1 head           : %d\n",pcb->Flg[Ni1Head].V);
  if (pcb->Flg[Ni2Head].I)
                      printf("i_2 head           : %d\n",pcb->Flg[Ni2Head].V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

  DefLineOption ( &cb, SPrefix, NPrefix );
  DefLineOption ( &cb, SExtension, NExtension );
  DefLineOption ( &cb, SHeadMark, NHeadMark );
    (void) sprintf(cb.Lin[NHeadMark].V,"%s","#"); /* default is # */
  DefLineOption ( &cb, SSpacer, NSpacer);
    (void) sprintf(cb.Lin[NSpacer].V,"%s","\t"); /* default is tab */
  DefLineOption ( &cb, SHeadL, NHeadL);
    (void) sprintf(cb.Lin[NHeadL].V,"%s",""); /* default */

  DefFlgOption ( &cb, SihbPrint, NihbPrint);
  DefFlgOption ( &cb, SHist, NHist);
    cb.Flg[NHist].V = TRUE; 
  DefFlgOption ( &cb, Si1Head, Ni1Head);
  DefFlgOption ( &cb, Si2Head, Ni2Head);
  DefFlgOption ( &cb, SAbs, NAbs);
  DefFlgOption ( &cb, SSwap, NSwap);
  DefFlgOption ( &cb, SNoImNum, NNoImNum);
    cb.Flg[NNoImNum].V = FALSE;
  DefFloatOption ( &cb, SScf, NScf);
  DefIntOption ( &cb, SHlen, NHlen);
    cb.Num[NHlen].V = 25; /* at least 25 header lines */

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, BlockNum );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_ascii, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

