/*+++
1 saxs_angle
  Transformation from cartesian coordinates to polar coordinates 

2 PURPOSE
  Transformation from cartesian coordinates to polar coordinates
  with pixel interpolation and azimuthal averaging.

  x_1 of the output file corresponds to the radius
  x_2 of the output file corresponds to the angle

  The angle coordinate (x_2) uses always the real reference system (Offset, 
  PixSiz). Center, WaveLength and SampleDistance are not used.

  Arguments:
  saxs_angle [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> <odum> 
                <i1cen_1> <i1cen_2> <r0> <dr> <odim_1> <a0> <da> <odim_2>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <i1cen_1> (image)   : <image center coordinate along dimension 1>
  <i1cen_2> (image)   : <image center coordinate along dimension 2>
  <r0>      (rsys)    : <minimum radius coordinate>
  <dr>      (rsys)    : <radial interval>
  <odim_1>            : <number of radial intervals>
  <a0>      (rsys)    : <minimum angle coordinate>
  <da>      (rsys)    : <angular interval>
  <odim_2>            : <number of angular intervals>

  The image coordinate of the center is (<i1cen_1>, <i1cen_2>). The input 
  data array is radially cut into <odim_1> rings with the thicknesses <dr>.  
  The rings are defined in the following way:
  0 <= o_1 <= (odim_1-1) from <minimum radius> + o_1 * <radial interval> to
  <minimum radius> + (o_1+1) * <radial interval>
  The default of <dr> is chosen in that way that not more than one full pixel
  falls between two subsequent rings. 

  The input data array is azimuthally averaged along the center of the odim_1
  numbers of rings which are defined by 

  0 <= o_1 <= (odim_1-1),  
  <minimum radius> + (o_1+0.5) * <radial interval>.

  The azimuthal integration is done in odim_2 angular sections which are 
  defined by:

  0 <= o_2 <= (odim_2-1),
  <angle minimum> + o_2 * <angular interval>, 
  <angle minimum> + (o_2+1) * <angular interval>. 

  All pixel values are linearly interpolated between the four neighbours.

  The averaged value for ring o_1 and angle o_2 is written to element 
  (o_1,o_2) of the output array.

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  13-Mar-1996 PB  V3.0
  21-Aug-1996 PB  V3.1 
                  option -s2 for multiplication of the result with s2 and a
                  factor (-s2 <factor>)
                  (-rsys array, -rsys image : multiplication with radius^2,
                   -rsys real : multiplication with radius^2 [m],
                   -rsys saxs : mulitplication with s^2 [1/nm] )
  30-Aug-1996 PB  V3.22
                  option +s2 multiplication with radius^2.
                  This allows the direct calculation of re^2*Q/V from the 
                  data by summation over i_1.
                  (-rsys array, -rsys image : multiplication with radius^2,
                   -rsys real : multiplication with radius^2 [m^2],
                   -rsys saxs : mulitplication with s^2 [1/nm^2])
  21-Sep-1996 PB  V3.3
                  option +csym : multiplication of the input pattern with
                                 Pi/2*sin(Alpha-ASym), where Alpha is the 
                                 angle of the data point from the w_1 axis 
                                 and ASym the angle of the symmetry axis from
                                 the w_1 axis (default: no multiplication)
                  option -asym : specification of the symmetry axis in degrees
                                 from the w_1 axis (default: 0.0 deg)
  31-Dec-1999 PB  V3.4 new reference system BEAM 
  10-Jan-2000 PB  V3.5 new reference systems CENTER and NORMAL, 
                  NORMAL replaces BEAM
  22-Mar-2000 PB  V3.6 optional output of azimuthal sum 
                  option: +vsum
  2000-08-04  PB  V3.6 %f->%g
  2000-11-16  PB  V3.7 azimuthal sum calculated with weight
  2001-07-08  PB  V3.8 FFirst
  2001-07-09  PB  V4.00 new loop, no repetition

---*/
# define Version  "saxs_angle 4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

/* Defaults */
# define Default_dim1  300
# define Default_dim2  360
# define Default_a0    0.0
# define Default_da    1.0

/* Special options */
# define SR0       "r0" /* float */
# define NR0       0 
# define SDr       "dr" 
# define NDr       1
# define SA0       "a0"
# define NA0       2
# define SDa       "da"
# define NDa       3
# define SAsym     "asym"
# define NAsym     4
# define SS2       "s2" /* flg */
# define NS2       0
# define SCsym     "csym"
# define NCsym     1
# define SVsum     "vsum"
# define NVsum     2

// # define DevSeparator ":" +++++++++++++++++
// # define DirSeparator "/" +++++++++++++++++   
// # define ExtSeparator "." +++++++++++++++++

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> <odum> \n\
                <i1cen1> <i1cen2> <r0> <dr> <dim1> <a0> <da> <dim2> "

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
DefaultRadialInterval

 Returns a default interval for the radius depending on the chosen
 reference system.

---------------------------------------------------------------------------*/
float DefaultRadialInterval( CmdBlk * pcb, ImgHeadBlk ihb[], int blkno, 
                             int * pstatus )
{ float dr;
  ImgHeadBlk * pihb;

   *pstatus = Success;
   dr = 1.0;
   pihb = &ihb[blkno];

   switch (pcb->RSys.V) {
     case IO_Array : dr = 1.0; break;
     case IO_Image : dr = 1.0; break;
     case IO_Center: dr = 1.0; break;
     case IO_Real  : if (!( (pihb->PixSiz[1].I) && (pihb->PixSiz[2].I) ) ) {
                       *pstatus = Failed; break; }
                     dr = PSIZE2R( MIN2( fabs(pihb->PixSiz[1].V),
                                   fabs(pihb->PixSiz[2].V) ) ); break;
     case IO_Normal: if (!( (pihb->PixSiz[1].I) && (pihb->PixSiz[2].I) ) ) {
                       *pstatus = Failed; break; }
                     dr = PSIZE2N( MIN2( fabs(pihb->PixSiz[1].V),
                                   fabs(pihb->PixSiz[2].V) ) ); break;
     case IO_Saxs  : if (!( (pihb->PixSiz[1].I) && (pihb->PixSiz[2].I) &&
                       (pihb->SampleDistance.I) && (pihb->WaveLength.I) ) ) {
                       *pstatus = Failed; break; }
                     dr = PSIZE2S( MIN2( fabs(pihb->PixSiz[1].V),
                                   fabs(pihb->PixSiz[2].V) ),
                                   pihb->SampleDistance.V,
                                   pihb->WaveLength.V ); break;
     default       : *pstatus = Failed; break;
     }

  if (*pstatus!=Success) {
    printf("Missing parameters for the chosen reference system (% s)\n",
            reftostr(pcb->RSys.V)); return; }

  return ( dr );

} /* DefaultRadialInterval */

/*---------------------------------------------------------------------------
calc_dist
---------------------------------------------------------------------------*/
float calc_dist( float A[], float B[] )
{ return(sqrt((A[1]-B[1])*(A[1]-B[1])+(A[2]-B[2])*(A[2]-B[2])));
} /* calc_dist */

/*---------------------------------------------------------------------------
SetOutputDimension

 Sets the output dimension ihb[0].Dim[1] and ihb[0].Dim[2] to a default.

 If the flags ib[0].Dim[1].I or ib[0].Dim[2].I are set these values are used
 as defaults.

 If the flag ib[0].Dim[1].I is not set the maximum image distance 
 between the center and the 4 edges of the image ihb[blkno] is used as 
 default for Dim[1].

 If the flag ib[0].Dim[2].I is not set the default for Dim[2] is 
 Default_dim2.
---------------------------------------------------------------------------*/
float SetOutputDimension( CmdBlk * pcb, ImgHeadBlk ihb[], int blkno,
                          int * pstatus )
{ ImgHeadBlk * pihb;
  float Off_1, Off_2;
  float Ps_1, Ps_2;
  float edge1[3], edge2[3], edge3[3], edge4[3];
  float center[3];
  float distance;
  float i_10, i_11, i_20, i_21;
 
  *pstatus = Success;

  pihb = &ihb[blkno];

  if (!( (pihb->Center[1].I) && (pihb->Center[2].I) ) ) {
    *pstatus = Failed; 
    printf("Missing center definitions\n"); return;
    } else {
    center[1] = pihb->Center[1].V; center[2] = pihb->Center[2].V;
    }
 
  if (pcb->ib[0].Dim[1].I) ihb[0].Dim[1] = pcb->ib[0].Dim[1].V;
    else {
    IMAGEREF(Off_1,Ps_1,ihb[1].Offset[1].V);
    IMAGEREF(Off_2,Ps_2,ihb[1].Offset[2].V);

    i_10 = i_20 = A2INDEX(ARRAYSTART+LOWERBORDER);
    i_11 = A2INDEX(ARRAYSTART+LOWERBORDER+ihb[1].Dim[1]);
    i_21 = A2INDEX(ARRAYSTART+LOWERBORDER+ihb[1].Dim[2]);

    /* edge 1 */
    edge1[1] = WORLD(i_10,Off_1,Ps_1); 
    edge1[2] = WORLD(i_20,Off_2,Ps_2);

    edge2[1] = WORLD(i_11,Off_1,Ps_1);
    edge2[2] = edge1[2];

    edge3[1] = edge2[1];
    edge3[2] = WORLD(i_21,Off_2,Ps_2);

    edge4[1] = edge1[1];
    edge4[2] = edge3[2];

    distance = MAX4( calc_dist(edge1,center), calc_dist(edge2,center), 
                     calc_dist(edge3,center), calc_dist(edge4,center) );
 
    ihb[0].Dim[1] = (long int) distance; 

    }

  if (pcb->ib[0].Dim[2].I) ihb[0].Dim[2] = pcb->ib[0].Dim[2].V;
    else ihb[0].Dim[2] = Default_dim2;
    
} /* SetOutputDimension */

/*---------------------------------------------------------------------------
 PrepareOutputBlock

 Prepare output header parameters for polar coordinates (x_1 = radius, 
 x_2 = angle) and allocate memory for output block ihb[0].

 All geometrical parameters that have not been set by the image block, i.e.
 with .I flags >= 0, are changed according to the following list.

  Defaults:

  r0 = 0.0
  dr = DefaultRadialInterval;
  a0 = Default_a0
  da = Default_da

  i1cen_1 no default
  i1cen_2 no default

  ocsys = "polar"

  ocen_1 = 0.0;
  ocen_2 = 0.0;

  array,
  image : if ((i1pix_1.I) && (i2pix_2.I)) {
          opix_1   (Alpha1 != Alpha0) :
                 = sqrt( i1pix_1^2 + (i1pix_2^2 - i1pix_1^2) * 0.5 *
                     ( 1 - 0.5 * (sin(2 Alpha1) - sin(2 Alpha0)) /
                                 (Alpha1 - Alpha0) ) )
                   Alpha1 = Alpha0 :
                 = sqrt( i1pix_1^2 + (i1pix_2^2 - i1pix_1^2) * 0.5 *
                     ( 1 - cos(2 Alpha0) ) );
          }
          opix_2 = da;
          ooff_1 = I2OFFSET(r0);
          ooff_2 = I2OFFSET(a0);

  real :  opix_1 = R2PSIZE(dr);
          opix_2 = R2PSIZE(da);
          ooff_1 = R2OFFSET(r0,opix_1);
          ooff_2 = R2OFFSET(a0,opix_2);

  saxs :  opix_1 = S2PSIZE(dr,SampleDistance,WaveLength);
          opix_2 = R2PSIZE(da);
          ooff_1 = S2OFFSET(r0,opix_1,ocen_1,SampleDistance,WaveLength);
          ooff_2 = R2OFFSET(a0,opix_2) + ocen_2;

 return value            :	void
 CmdBlk * pcb	         :   
 ImgHeadBlk ihb[0] (i/o) :
 ImgHeadBlk ihb[1] (i)   :
 int * pstatus     (o)   : return status

---------------------------------------------------------------------------*/
void PrepareOutputBlock( CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus )
{ const float alpha_eps = 1e-30;
  const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;
  register long int i;
  float *pData;
  float initvalue;
  ImgHeadBlk * pihb;
  
  float Alpha0, Alpha1;
  float r0, dr, a0, da;
  float i1pix_1, i1pix_2;

  *pstatus = Success;

  if (pcb->TestBit) printf("PrepareOutputBlock\n");

  pihb = &ihb[0];

  /* calculate dimension for ihb[0] from ihb[1] or take from image block */
  SetOutputDimension( pcb, ihb, 1, pstatus );
  if (*pstatus!=Success) return;

  /* release memory if already allocated */
  if (pihb->Data!=NULL) free(pihb->Data);

  /* allocate memory */
  initvalue = pihb->Dummy.V;
  pihb->ImgByteLen = sizeof(float) * pihb->Dim[1] * pihb->Dim[2];
  if (pihb->ImgByteLen>0) {
    /* allocate memory */
    pihb->Data = (float *) malloc (pihb->ImgByteLen);

    if (pihb->Data==NULL) {
      *pstatus=NotEnoughMemoryAvailable; return; }

    pihb->Dim[0] = 2;

    pData = pihb->Data;
    for (i=0;i<(pihb->Dim[1]*pihb->Dim[2]);i++) *(pData++)=initvalue;
    } else {
    /* no memory to allocate */
    pihb->Data = (float *) NULL;
    pihb->Dim[0] = 2;
    }

  /* get the transformation parameters */

  if (!(pcb->Arg[NR0].I)) pcb->Arg[NR0].V = 0.0;
  if (!(pcb->Arg[NDr].I)) {
    pcb->Arg[NDr].V = DefaultRadialInterval(pcb,ihb,1,pstatus);
    if (*pstatus!=Success) return;    
    }
  if (!(pcb->Arg[NA0].I)) pcb->Arg[NA0].V = Default_a0;
  if (!(pcb->Arg[NDa].I)) pcb->Arg[NDa].V = Default_da;

  /* calculate geometrical header values for polar coordinates,
     replace all parameters that have not been set with an option, i.e.
     ....I>=0 */

  r0 = pcb->Arg[NR0].V;
  dr = pcb->Arg[NDr].V;
  a0 = pcb->Arg[NA0].V;
  da = pcb->Arg[NDa].V;

  Alpha0 = deg2rad * a0;
  Alpha1 = deg2rad * (a0 + da * pihb->Dim[2]);
  i1pix_1 = ihb[1].PixSiz[1].V; i1pix_2 = ihb[1].PixSiz[2].V;

  if (pihb->Center[1].I>=0) pihb->Center[1].V = 0.0; pihb->Center[1].I = TRUE;
  if (pihb->Center[2].I>=0) pihb->Center[2].V = 0.0; pihb->Center[2].I = TRUE;

  switch (pcb->RSys.V) {
    case IO_Array :  /* the same as IO_Image */;

    case IO_Image :  
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V = I2OFFSET(r0); pihb->Offset[1].I = TRUE; 
           }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = I2OFFSET(a0); pihb->Offset[2].I = TRUE;
           }
         if (pihb->PixSiz[1].I>=0) {
           if ((ihb[1].PixSiz[1].I) && (ihb[1].PixSiz[2].I)) {
             if (fabs(Alpha1-Alpha0)<alpha_eps) {
               pihb->PixSiz[1].V = sqrt( i1pix_1*i1pix_1 +
                (i1pix_2*i1pix_2 - i1pix_1*i1pix_1) * 0.5 *
                 (1 - cos(2*Alpha0)) );
               } else {
               pihb->PixSiz[1].V = sqrt( i1pix_1*i1pix_1 +
                (i1pix_2*i1pix_2 - i1pix_1*i1pix_1) * 0.5 *
                 (1 - 0.5*(sin(2*Alpha1) - sin(2*Alpha0)) / (Alpha1-Alpha0)) );
               }
             pihb->PixSiz[1].I = TRUE;
             }
           }
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = da; pihb->PixSiz[2].I = TRUE;
           }
         break;

    case IO_Center :
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V =
             C2OFFSET(r0,pihb->Center[1].V);
           pihb->Offset[1].I = TRUE;
           }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = I2OFFSET(a0); pihb->Offset[2].I = TRUE;
           }
         if (pihb->PixSiz[1].I>=0) {
           if ((ihb[1].PixSiz[1].I) && (ihb[1].PixSiz[2].I)) {
             if (fabs(Alpha1-Alpha0)<alpha_eps) {
               pihb->PixSiz[1].V = sqrt( i1pix_1*i1pix_1 +
                (i1pix_2*i1pix_2 - i1pix_1*i1pix_1) * 0.5 *
                 (1 - cos(2*Alpha0)) );
               } else {
               pihb->PixSiz[1].V = sqrt( i1pix_1*i1pix_1 +
                (i1pix_2*i1pix_2 - i1pix_1*i1pix_1) * 0.5 *
                 (1 - 0.5*(sin(2*Alpha1) - sin(2*Alpha0)) / (Alpha1-Alpha0)) );
               }
             pihb->PixSiz[1].I = TRUE;
             }
           }
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = da; pihb->PixSiz[2].I = TRUE;
           }
         break;

    case IO_Real  :
         if (pihb->PixSiz[1].I>=0) {
          pihb->PixSiz[1].V = R2PSIZE(dr); pihb->PixSiz[1].I = TRUE;
          }
         if (pihb->PixSiz[2].I>=0) {
          pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
          }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V = R2OFFSET(r0,pihb->PixSiz[1].V); 
           pihb->Offset[1].I = TRUE;
           }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = R2OFFSET(a0,pihb->PixSiz[2].V); 
           pihb->Offset[2].I = TRUE;
           }
         break;

    case IO_Normal :
         if (pihb->PixSiz[1].I>=0) {
          pihb->PixSiz[1].V = N2PSIZE(dr); pihb->PixSiz[1].I = TRUE;
          }
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
           }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V =
             N2OFFSET(r0,pihb->PixSiz[1].V,pihb->Center[1].V);
           pihb->Offset[1].I = TRUE;
           }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V =
             R2OFFSET(a0,pihb->PixSiz[2].V) + pihb->Center[2].V;
           pihb->Offset[2].I = TRUE;
           }
         break;

    case IO_Saxs  :
         if (pihb->PixSiz[1].I>=0) {
           pihb->PixSiz[1].V = 
             S2PSIZE(dr,pihb->SampleDistance.V,pihb->WaveLength.V);
           pihb->PixSiz[1].I = TRUE;
           }
         if (pihb->PixSiz[2].I>=0) {
           pihb->PixSiz[2].V = R2PSIZE(da); pihb->PixSiz[2].I = TRUE;
           }
         if (pihb->Offset[1].I>=0) {
           pihb->Offset[1].V = 
             S2OFFSET(r0,pihb->PixSiz[1].V,pihb->Center[1].V,
                      pihb->SampleDistance.V,pihb->WaveLength.V);
           pihb->Offset[1].I = TRUE;
           }
         if (pihb->Offset[2].I>=0) {
           pihb->Offset[2].V = 
             R2OFFSET(a0,pihb->PixSiz[2].V) + pihb->Center[2].V;
           pihb->Offset[2].I = TRUE;
           }
         break;

    default : 
         printf("Impossible reference system : %s\n",reftostr(pcb->RSys.V));
         *pstatus = Failed; return;

    } /* switch */

  if (pcb->TestBit) {
    printf("r0 = %g\n",r0);
    printf("dr = %g\n",dr);
    printf("a0 = %g\n",a0);
    printf("da = %g\n",da);
    PrintImageHeaderBlock(0,&ihb[0]);
    }

  if (*pstatus!=Success) {
    printf("Missing parameters for the chosen reference system (% s)\n",
            reftostr(pcb->RSys.V)); return; }

} /* PrepareOutputBlock */

/*---------------------------------------------------------------------------
1 saxs_angle

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void saxs_angle (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;
  const float csym_fac = SAXS_PI/2.0, radius_eps = 1e-30;

  const char * OutputBinaryName = "i1test.bin";
  FILE * OutputBinaryId;

  int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float *pI1Data;
  float I1Dummy, I1DDummy;
  float I1Off_1,I1Off_2;
  float I1Cen_1,I1Cen_2;
  float Value, Weight, Sum, SumWeight;
  float Wcenter_1, Wcenter_2;
  float Radius, Angle, DAngle, DDAngle, Angle0;
  int   NAngle, cnt, iangle;

  float Radius_1, Radius_2;
  float SinDiffAngle, SAngle, SinSAngle, CosSAngle, SinAngle, CosAngle; 

  int i_1, i_2;
  float W_1, W_2;
  float f_11, f_21;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     PrepareOutputBlock( pcb, ihb, pstatus );
     if (*pstatus!=Success) return;

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

   /* Set the angular reference system to real */
     REALREF(Off_2[0],Ps_2[0],ihb[0].Offset[2].V,ihb[0].PixSiz[2].V);
   /* Subtract output shift for calculation */
     if (pcb->Shift[2].I) Off_2[0] = Off_2[0]-pcb->Shift[2].V;
   /* Center for output angle axis is always 0.0 */
     ihb[0].Center[2].V = 0.0;

     if (pcb->TestBit) {
       printf("Off_1[%d] = % f, Ps_1[%d] = % f\n", 0,Off_1[0],0,Ps_1[0]);
       printf("Off_2[%d] = % f, Ps_2[%d] = % f\n", 0,Off_2[0],0,Ps_2[0]); 
       }

   /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];
  I1Off_1  = ihb[1].Offset[1].V;
  I1Off_2  = ihb[1].Offset[2].V;
  I1Cen_1  = ihb[1].Center[1].V;
  I1Cen_2  = ihb[1].Center[2].V;

  /* calculate actual center */
  Wcenter_1 = WORLD(I2INDEX(I1Cen_1,I1Off_1),Off_1[1],Ps_1[1]); 
  Wcenter_2 = WORLD(I2INDEX(I1Cen_2,I1Off_2),Off_2[1],Ps_2[1]);
  
  DDAngle = (Ps_2[0]/(MIN2(Ps_1[1],Ps_2[1]))) * deg2rad;
  SAngle  = pcb->Arg[NAsym].V * deg2rad;

  if (pcb->TestBit) {
    printf("Wcenter_1 = % g, Wcenter_2 = % g\n",Wcenter_1,Wcenter_2); 
    printf("DDAngle   = % g, SAngle    = % g\n",
            DDAngle * rad2deg, SAngle * rad2deg);
    }

  if (pcb->Flg[NS2].V) {
    printf(" Multiplication with radius^2\n");
    } 

  if (pcb->Flg[NCsym].V) {
    printf(" Multiplication with (Pi/2)*sin(Alpha-% g)\n",SAngle*rad2deg);

    SinSAngle = sin(SAngle); 
    CosSAngle = cos(SAngle); 
    pI1Data   = ABSPTR(I1Data,I1Dim_1,I1Dim_2,0,0);
    for (i_2=0;i_2<I1Dim_2;i_2++) {
      Radius_2     = WORLD(i_2,Off_2[1],Ps_2[1])-Wcenter_2;
      for (i_1=0;i_1<I1Dim_1;i_1++) {
        Radius_1     = WORLD(i_1,Off_1[1],Ps_1[1])-Wcenter_1;
        Radius       = sqrt(Radius_1*Radius_1+Radius_2*Radius_2);

        if (fabs(Radius)>radius_eps) {
          CosAngle     = Radius_1/Radius;
          SinAngle     = Radius_2/Radius; 
          SinDiffAngle = SinAngle*CosSAngle-CosAngle*SinSAngle;
          } else {
          SinDiffAngle = 0.0;
          }

        if NODUMMY(*pI1Data, I1Dummy, I1DDummy) {
          *pI1Data    *= csym_fac*fabs(SinDiffAngle);  
          }
        pI1Data++;
        }
      }

    if (pcb->TestBit>3) {
      /* write test file */
      printf("Writing %s\n",OutputBinaryName);
      if ((OutputBinaryId=fopen(OutputBinaryName,"w")) != NULL) {
        fwrite(I1Data,sizeof(float)*(I1Dim_1*I1Dim_2),1,OutputBinaryId);
        fclose(OutputBinaryId);
        }
      }

    }

  /* loop over the output array  */

    for (i_1=0;i_1<I0Dim_1;i_1++) {
      Radius = WORLD(i_1,Off_1[0],Ps_1[0]);
      /* number of angular intervals for averaging */
      NAngle  = MAX2(1,(int) (DDAngle * Radius) + 1 ); 
      DAngle  = (Ps_2[0]/(float) NAngle) * deg2rad;

/*        printf("NAngle = % d\n",NAngle); /* ++++++++++ */
/*        printf("DAngle = % g\n",DAngle*rad2deg);  /* ++++++++++ */

      for (i_2=0;i_2<I0Dim_2;i_2++) {

        Angle = WORLD((LOWERBORDER+(float)i_2),Off_2[0],Ps_2[0])*deg2rad + 
                  DAngle*0.5;
       
        /* angular averaging */ 
        cnt = 0;
        Sum = 0.0;
        SumWeight = 0.0;
        for (iangle = 0; iangle<NAngle; iangle++) {

/*          printf("iangle = %d, Angle = % g\n", iangle, Angle*rad2deg); /*+++++++++++++*/

          W_1 = Radius * cos( Angle  ) + Wcenter_1;  
          W_2 = Radius * sin( Angle  ) + Wcenter_2;

          f_11 = INDEX(W_1,Off_1[1],Ps_1[1]);
          f_21 = INDEX(W_2,Off_2[1],Ps_2[1]);

          if (Ipol2ldw (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
            f_11, f_21, &Value, &Weight)) {
            /* then do something with the data */
            Sum       += Value;
            SumWeight += Weight;
            cnt++;
            } /* if (Ipol2ld ... */

          Angle += DAngle;
          } /* for */

        if (cnt>0) {
          pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);
          *pI0Data = Sum/SumWeight;

          if (pcb->Flg[NVsum].V) {
            /* Multiply with number of covered pixels */
//+++++++++ *pI0Data *= (Radius * Ps_1[0] * NAngle * DAngle)/(Ps_1[1]*Ps_2[1]); 
            *pI0Data *= ( Radius*Ps_1[0]*SumWeight*DAngle )/( Ps_1[1]*Ps_2[1] );
            }
          /* Multiplication with Radius^2*dr */
          if (pcb->Flg[NS2].V) {*pI0Data *= Radius * Radius;}
          }

        /* end angular averaging */

        } /* for i_2 ... */

      } /* for i_1 ... */

 } /* saxs_angle*/

/*---------------------------------------------------------------------------
ArgvAngleFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

The header block of the first image is read into ihb[1].
---------------------------------------------------------------------------*/
void ArgvAngleFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                          int * pstatus)
{ int blkno;
  long int image_1st, image_lst;
  char linebuffer[1000];
  float Bin_1, Bin_2;
  
  /* ---  File names and open modes --- */

    blkno = 1;
    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib,blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile(pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                  IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    /* Search for minimum and maximum image number */
    (void) SearchMinMaxImage (pcb,ib,blkno,&image_1st,&image_lst,pstatus);
    if (*pstatus!=Success) return;
    if (!ib[blkno].First.I) ib[blkno].First.V = image_1st;
    if (!ib[blkno].Last.I)  ib[blkno].Last.V  = image_lst;


    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect;
      ib[0].OpenMode.I = TRUE;
      }
    ib[0].OpenMode.V = ib[0].OpenMode.V | IO_DontReadData;
 
    /*--- File name ib[0].Name of output */
    argv_filename( pcb,"Output sequence",ib,0,DefaultOutput,pstatus);
    if (*pstatus!=Success) return;

  /* ---  Image numbers --- */

    /*--- Argument : number of first image */
    argv_long(pcb,"First image of input sequence 1",&ib[1].First,ib[1].First.V,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
      else image_lst = ib[1].Last.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last,image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

    /*--- Argument : image coordinate 1 of center */
    Bin_1 = MAX2( 1,ib[1].Bin[1].V);
    argv_float(pcb,"Center 1 (image coordinate)",
                &ib[1].Center[1],Bin_1*ihb[1].Center[1].V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : image coordinate 2 of center */
    Bin_2 = MAX2( 1,ib[1].Bin[2].V);
    argv_float(pcb,"Center 2 (image coordinate)",
                &ib[1].Center[2],Bin_2*ihb[1].Center[2].V,pstatus);
    if (*pstatus!=Success) return;

    (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

    CloseImageFile( pcb, ib, 1, pstatus) ;
    if (*pstatus!=Success) return;

    /* Defaults for output file */
    if (!ib[0].Dummy.I) ib[0].Dummy.V  = ihb[1].Dummy.V;

    /*--- Argument : output dummy */
    argv_float( pcb, "Output dummy", &ib[0].Dummy, ib[0].Dummy.V, pstatus);
    if (*pstatus!=Success) return;

} /* ArgvAngleFilenames */

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  const char * InputRefSys  = "Input reference system is";

  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvAngleFilenames ( pcb, ib, ihb, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument : start radius */
  printf("%s %s\n",InputRefSys,reftostr(pcb->RSys.V));
  argv_float(pcb,"Start radius (RSys) ", &pcb->Arg[NR0],0.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : radial interval */
  printf("%s %s\n",InputRefSys,reftostr(pcb->RSys.V));
  if (!(pcb->Arg[NDr].I) )
    pcb->Arg[NDr].V = DefaultRadialInterval(pcb,ihb,1,pstatus);
  if (*pstatus!=Success) return;
  argv_float(pcb,"Radial interval (RSys)",
               &pcb->Arg[NDr],pcb->Arg[NDr].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : output dimension 1 */
  SetOutputDimension( pcb, ihb, 1, pstatus ); /* no check of pstatus */
  argv_long(pcb,"Number of radial intervals", 
              &ib[0].Dim[1],ihb[0].Dim[1],pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : minimum angle (default 0.0) */
  argv_float(pcb,"Start angle [deg]", &pcb->Arg[NA0],Default_a0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : angular interval */
  argv_float(pcb,"Angular interval [deg]", &pcb->Arg[NDa],Default_da,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : output dimension 2 */
  argv_long(pcb,"Number of angular intervals", 
              &ib[0].Dim[2],ihb[0].Dim[2], pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    
     printf("i/p file                : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    
     printf("o/p file                : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   
     printf("first image             : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    
     printf("last image              : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     
     printf("increment               : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   
     printf("output dummy            : %g\n",ib[0].Dummy.V);
  if (ib[1].Center[1].I) 
     printf("center 1 (image coord.) : %g\n",ib[1].Center[1].V);
  if (ib[1].Center[2].I)
     printf("center 2 (image coord.) : %g\n",ib[1].Center[2].V);

  if (pcb->Arg[NR0].I) 
     printf("minimum radius          : %g\n",pcb->Arg[NR0].V);
  if (pcb->Arg[NDr].I) 
     printf("radial interval         : %g\n",pcb->Arg[NDr].V);
  if (ib[0].Dim[1].I)  
     printf("num. radial intervals   : %d\n",ib[0].Dim[1].V);
  if (pcb->Arg[NA0].I) 
     printf("minimum angle [deg]     : %g\n",pcb->Arg[NA0].V);
  if (pcb->Arg[NDa].I) 
     printf("angular interval [deg]  : %g\n",pcb->Arg[NDa].V);
  if (ib[0].Dim[2].I)  
     printf("num. angular intervals  : %d\n",ib[0].Dim[2].V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

  DefFloatOption( &cb, SR0,       NR0);
  DefFloatOption( &cb, SDr,       NDr);
  DefFloatOption( &cb, SA0,       NA0);
  DefFloatOption( &cb, SDa,       NDa);
  DefFloatOption( &cb, SAsym,     NAsym);
  DefFlgOption  ( &cb, SS2,       NS2);
  DefFlgOption  ( &cb, SCsym,     NCsym);
  DefFlgOption  ( &cb, SVsum,     NVsum);

 /* Set defaults */
  cb.Flg[NVsum].V = FALSE;

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_angle, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

