/* SaxsOption.h V6.31 by Peter Boesecke 2001-09-16                        */
/**************************************************************************/
/* SaxsOption.h                                                           */
/**************************************************************************/

#ifndef _SAXS_OPTION
# define _SAXS_OPTION

# include "SaxsError.h"
# include "SaxsExpression.h"
# include "SaxsInput.h"
# include "SaxsDefinition.h"

/* History */
# define SaxsHistoryKey "i0" /* key to history */

/* Restrictions */
# define IO_len InputLineLength
# define IO_CodeLen      5 /* maximum length of option strings */
# define IO_MaxFloatOpt 30 /* maximum number of definable float options */	
# define IO_MaxIntOpt   10 /* maximum number of definable integer options */
# define IO_MaxLineOpt   5 /* maximum number of definable line options */
# define IO_MaxFlagOpt  10 /* maximum number of definable flags */

/* File opening modi */
# define IO_New             0x0001l   /* new file is opened */
# define IO_Old             0x0002l   /* old file is opened */
# define IO_FileProtect     0x0004l   /* file cannot be modified */
# define IO_ImageProtect    0x0008l   /* existing images cannot be modified */
# define IO_DontReadData    0x0010l   /* do not read/alloc binary data */
# define IO_DontReadWrite   0x0020l   /* do not rd/wr image header nor data */
# define IO_DontOpen        0x0040l   /* do not open file */
# define IO_Dummy           0x0080l   /* like IO_DontReadWrite, but create 
                                         dummy data */

/* Description
All files can be updated (read and write).

IO_New           : create a new output file, backup the old file, if possible.
IO_Old           : open an old file
IO_FileProtect   : file is protected (no write)
IO_ImageProtect  : all data blocks in the file are protected (no update of
                   existing data blocks, append only)
IO_DontReadData  : only the header will be read or created,
                   no binary data will be read or allocated
IO_DontReadWrite : file is not opened,
                   no read, no write (e.g. for read and write with separate
                   routines, format conversion etc)
IO_DontOpen      : file is not opened
IO_Dummy         : similar to IO_DontReadWrite, but reading routines 
                   create dummy data

/* Reference systems */
enum { IO_NoRSys=0, 
       IO_Array, IO_Image, IO_Center, IO_Real, IO_Normal, IO_Saxs };

/* Block structures */
typedef struct { char     V[IO_len]; int I; } IO_line;
typedef struct { long int V;         int I; } IO_long;
typedef struct { float    V;         int I; } IO_float;

typedef struct OptionDescriptor {
	char	   code[IO_CodeLen];
	int        ptyp;   /* parameter type */
	int	   npar;   /* number of parameters */
	} OptDsc;

typedef struct ImageBlock {
        /* file control variables, internal use */
        int       Stream;           /* I/O stream used internally */
	long int  StreamOpenMode;   /* actual open mode used internally */
        long int  FileNumber;       /* file number of actually opened file */
        char      FileName[IO_len]; /* file name of actually opened file */
        /* --- image option variables */
        IO_line   Name;          /* filename with pattern */
        IO_long   FFirst;        /* first file number extracted from Name */ 
        IO_long   FLast;         /* last file number extracted from Name */ 
        IO_long   FInc;          /* file number increment extracted from Name */
        IO_long   OpenMode;
        IO_long   Memory;
        IO_long   First;
        IO_long   Last;
        IO_long   Inc;
        IO_float  MinClip;
        IO_float  MaxClip;
        IO_float  Const;
        IO_float  Factor;
        IO_long   Bin[3];
        IO_long   Gnaw[3];

        /*     image header block variables */
        IO_line   Title;
        IO_line   Time;
        IO_long   Dim[4];  // only 3 should be used (PB 2000-11-16) 
        IO_float  Dummy;
        IO_float  DDummy;
        IO_float  Offset[3];
        IO_float  Center[3];
        IO_float  PixSiz[3];
        IO_float  SampleDistance;
        IO_float  WaveLength;
        IO_long   Orientation;
}       ImgBlk;

typedef struct CommandBlock {
	/* control variables, internal use */
	char *  MainName;    /* pointer to name of main program */
	const char * MainVersion; /* pointer to version string of main prog. */
	const char * MainHelp;    /* pointer to help string of main program */
	SaxsErrBlk seb;      /* SAXS error block */
	FILE *  infile;      /* input file */
	int     argc;        /* length of argument list */
	char ** argv;        /* pointer to argument list */
	int  *  parg_no;     /* pointer to argument number variable */
	OptDsc * co;         /* pointer to command option descriptor array,
                                NULL after call of ReadOption */
	OptDsc * io;	     /* pointer to image option descriptor array,
                                NULL after call of ReadOption */
	ImgBlk * ib;         /* pointer to image block array */
	int     ImgBlkLen;   /* number of defined image blocks */
	int     RejectNum;   /* if TRUE, numbers are rejected as options */
        /* --- command option variables */
	int     TestBit;     /* for program tests only: on/off */
	IO_long Help;        /* show help and exit: on */
	IO_long Prompt;      /* prompt mode: on/off */
	IO_long USys;        /* reference system for user input */
	IO_long RSys;        /* reference system for calculation */
	IO_long GeneralBlock;         /* write general block */ 
        IO_long PassHeader;           /* write the full input header into
                                         the output header */ 
        IO_long OutputDataType;       /* output data type */
        IO_long OutputDataValueOffset;/* output data value offset */
        IO_long BslInputByteOrder;    /* byte order of bsl input files */
        IO_long MaxLineWidth;         /* maximum width of text lines */
        IO_long Add;                  /* number of output images to add */
	IO_float Shift[3];            /* shift of the output image origin */
	IO_float Arg[IO_MaxFloatOpt]; /* definable float options, free use */
	IO_long Num[IO_MaxIntOpt];    /* definable long options, free use */
	IO_line Lin[IO_MaxLineOpt];   /* definable line options, free use */
	IO_long Flg[IO_MaxFlagOpt];   /* definable flags, free use */
}       CmdBlk;

extern void
    InitOptions    ( const char * mainhelp, const char * mainvers, int rjctnum,
		     CmdBlk * pcb, ImgBlk ib[], int ib_len ),
    DefFloatOption ( CmdBlk * pcb, char * name, int argopt_no ),
    DefIntOption   ( CmdBlk * pcb, char * name, int numopt_no ),
    DefLineOption  ( CmdBlk * pcb, char * name, int linopt_no ),
    DefFlgOption   ( CmdBlk * pcb, char * name, int flgopt_no ),
    ReadOptions    ( char * argv[], int * parg_no,
		     CmdBlk * pcb, ImgBlk ib[], int * status),
    PrintBlocks    ( CmdBlk * pcb, ImgBlk ib[] ),
    argv_end       ( CmdBlk * pcb ),
    argv_start     ( CmdBlk * pcb, int narg ),
    argv_filename  ( CmdBlk * pcb,
                     char * promptline, ImgBlk ib[], int blkno,
                     char * defval, int * pstatus),
    argv_line      ( CmdBlk * pcb,
                     char   * promptline, IO_line  * value, char * defval,
                     int    * pstatus),
    argv_float     ( CmdBlk * pcb,
                     char   * promptline, IO_float * value, float  defval,
                     int    * pstatus),
    argv_long      ( CmdBlk * pcb,
                     char   * promptline, IO_long  * value, long int defval,
                     int    * pstatus),
    argv_flag      ( CmdBlk * pcb,
                     char   * promptline, IO_long  * value, long int defval,
                     int    * pstatus),
    argv_coord     ( CmdBlk * pcb,
                     long int RSys, float ROff, float RPs,
                     long int USys, float UOff, float UPs,
                     char * promptline, IO_float * value, float defval,
                     int * pstatus),
    argv_dist      ( CmdBlk * pcb,
                     long int RSys, float ROff, float RPs,
                     long int USys, float UOff, float UPs,
                     char * promptline, IO_float * value, float defval,
                     int * pstatus);
extern char
    *reftostr      ( long int refsys );

extern int
    extract_filenumbers( ImgBlk *pib, IO_line *pName,
                         int opt_onoff, int * pstatus );

#endif /* _SAXS_OPTION */

/**************************************************************************/
/*                                                                        */
/**************************************************************************/
