"""
	TableWidget.py
	Class for the visualization of datas or for placing widgets in a scrolled table 

	Contact: Nicolas Pascal (pascal@esrf.fr)
	(Beta) Version: 1.1
	
	##  BUG FIX  ##
	10/10/01 : creation
	15/2/02   : column and row resizing bugs fixed
					add hide/show methods
	1/03/02   : fix scrolling bugs when column/row's titles are hided
	30/04/02 : fix scrollbars display bugs
	2/05/02   : add parametre in addRow/addColumn methods to set rowTitle/colTitle configuration		

	Interface:
	================
	class TableWidget:
		The scrolled table		  
		__init__(self,master,nbCol,nbrow,colNames,rowNames,rowTitleCnf,colTitleCnf,cellsCnf)
			builder
			PARAMETERS :
		 	master : 		the master window of the tableWidget
			nbCol : 		the number of columns
			colNames : 		the list of the column's titles
							Must be given in a dictionnary {'option':'OptionVal',...}
			nbRow : 		the number of rows
			rowNames : 		the list of the row's titles
			rowTitlesCnf : 	the default configuration of the row's titles labels. 
							Must be given in a dictionnary {'option':'OptionVal',...}
			cellsCnf : 		the default configuration of the table's cells. 
							Must be given in a dictionnary {'cellsType':'default_cell_type',
															'cellsOptions':{'option':'OptionVal',...}}
															
		addColumn(columnName=None, colTitleCnf={}, cellsCnf={})
		   append a column to the table widget.
		   This column will contain a title Label with the text 'columnName',and a list of cells which contain 
		   a widget corresponding to the configuration 'cellsCnf' 
		   (must be given in a dictionnary {'cellsType':'default_cell_type',
											'cellsOptions':{'option':'OptionVal',...}},

	  	addRow(rowName=None, rowTitleCnf={}, cellsCnf={})
		   append a row (with the title 'rowName') to the table widget
		   This row will contain a title Label with the text 'columnName' and a list of cells which contain 
		   a widget corresponding to the configuration cellsCnf 
		   (must be given in a dictionnary {'cellsType':'default_cell_type',
											'cellsOptions':{'option':'OptionVal',...}},

		configColTitle(col, key, newVal)
		   set the value of the option 'key' to 'newVal' for the title label of the column 'col'

	  	configRowTitle(row, key, newVal)
		   set the value of the option 'key' to 'newVal' for the title label of the row 'row'

	  	delColumn(col)
		   delete a column (number 'col') from the table widget

	  	delRow(row)
		   delete a row (number 'row') from the table widget

	 	getCell(col, row)
		   return the cell situed at the row 'row', and at the column 'col'

	  	getCellWidget(col, row)
		   return the widget contained in the cell ('col','row')

	  	getWidgetPos(widget)
		   return the position (col,row) of the widget 'widget'
				
		getNbrOfCols()
		   return the number of columns in this table

	  	getColumn(col)
		   return a dictionnary with {colName : [sequence of the column cells objects]}

	  	getColIndex(colName):
			return the index of the first column with the title 'colName' (None if no column has this title)
		
		getColumnLabel(col)
		   return the title label object of the column number 'col'

	  	getColumnName(col)
		   return the name of the column number 'col'

	  	getRow(row)
		   return a dictionnary with {rowName : [sequence of the Row cells objects]}

	  	getRowIndex(rowName):
			return the index of the first row with the title 'rowName' (None if no row has this title)
		
		getRowLabel(row)
		   return the title label object of the row number 'row'

	  	getRowName(row)
		   return the name of the Row number 'row'

	  	getNbrOfRows()
		   return the number of rows in this table

	  	getTableSize()
		   return the size of the table in a tuple (nbCol,nbRow)

	  	getValue(col, row)
		   return the value contained in the cell at ('col','row')

	  	hideColumnTitles()
		   hide the column title labels

	 	hideRowTitles()
		   hide the row title labels

	  	putWidgetInCell(col, row, obj)
		   put the widget 'obj', in the cell ('col','row')

	  	setCell(col,row, cellCnf={'cellOptions': {}, 'cellType': 'Text'})
		   change the content of the cell ('col','row') for an object of type cellCnf['cellType'],
		   with the options cellCnf['cellOptions']

	  	setColumnName(col, newName)
		   change the name of the column number 'col' to 'newName'

	  	setColumnWidth(col, newWidth)
		   change the width of the column number 'col' for 'newWidth'

	  	setRowHeight(row, newHeight)
		   change the height of the row number 'row' for 'newHeight'

	  	setRowName(row, newName)
		   change the name of the row 'row' to newName

	  	setValue(col, row, newVal)
		   set the value contained in the cell situed at ('col','row') to 'newVal'

	  	showColumnTitles()
		   show the column title labels

	  	showRowTitles()
		   show the row title labels

	  	uniformizeCellSize()
		   update the cells size to the labels size if the cell size is bigger than the 
		   label's one (and the cell size if the label size is bigger than the cell one)

	  	updateColSize(column, width, height)
		   update the table widget's cells and titles size when a resizing of ('width','height') the column label 
		   'column' occurs

	  	updateRowSize(row, width, height)
		   update the table widget's cells and titles size when a resizing of ('width','height') the row label 
		   'row' occurs
		   
		 show()
		 	show the table
		 
		 hide()
		 	hide the table

	class TableCell
		generic class :
	 	represents one cell of the TableWidget.
	 	To specify it, overwrite the methods : setValue,getValue
	  
	  	__init__(self, master, cellsCnf)
		   builder
		   PARAMETERS
		   master : the master frame of this cell
		   cellsCnf : the configuration of the widget contained in the cell
												   (must be given in a dictionnary {'cellsType':'default_cell_type',
																				   'cellsOptions':{'option':'OptionVal',...}}
	  	getValue()
		   return the value contained in the cell
		   must be overwritten depending of the object contained in the cell

	  	getWidget()
		   return the widget contained in this cell

	  	putWidgetInCell(obj)
		   put the already created widget 'obj' in the cell

	  	setCell(cellCnf={})
		   set the object contained in the cell become 'object'

	  	setValue(newVal)
		   set the cell value to newVal
		   must be overwritten depending of the object contained in the cell
"""
################################################################################  
from Tkinter import *
import tkFont
import Pmw
import tkMessageBox
import string
import os
import copy
################################################################################
# constants
DEBUG=0
TRACE=0
################################################################################

class TableWidget(Frame):
	"""
	create a table widget
	"""
	def __init__(self,master=None,nbCol=0,nbRow=0,colNames=[],rowNames=[],rowTitlesCnf={},colTitlesCnf={},cellsCnf={},scrollbarOn=1,**kw):
		"""
		builder of the class
		build a table with a number of columns of nbCol, where the column i is called colNames[i] 
		(IDEM for the rows)
		@param master : 		the master window of the tableWidget
		@param nbCol : 			the number of columns
		@param colNames : 		the list of the column's titles
		@param colTitlesCnf : 	the default configuration of the column's titles labels. 
								Must be given in a dictionnary {'option':'OptionVal',...}
		@param nbRow : 			the number of rows
		@param rowNames : 		the list of the row's titles
		@param rowTitlesCnf : 	the default configuration of the row's titles labels. 
								Must be given in a dictionnary {'option':'OptionVal',...}
		@param cellsCnf : 		the default configuration of the table's cells. 
								Must be given in a dictionnary {'cellsType':'default_cell_type',
																'cellsOptions':{'option':'OptionVal',...}},
								where cellsCnf['cellsType'] will be the default widget's type contained in each cell, 
								with a default configuration of cellsCnf['cellsOptions']
		@param scrollbarOn :  if set, display horizontal and vertical scrollbars if needed
		If the number of columns is greater than the length of the sequence colNames, the 
		non-explicitely given column names are called colx where x is the corresponding 
		column position (IDEM for the rows)
		if the length of the sequence colNames is greater than the specified number of columns, 
		nbCol will be set to the sequence's length
		If cellsCnf isn't given, the cells will contain nothing (a None object) 
		"""
		if DEBUG :
			print "init TableWidget"
		
		if master:
			self.master=master
		else:
			self.master=Tk()
		
		Frame.__init__(self,master,kw)

		self.content=Frame(self)
		
		self.nbCol=nbCol
		self.nbRow=nbRow

		# defining attributes
		self.colNames=colNames
		if (nbCol<=len(self.colNames)):
			# set the number of columns to the length of the column's names
			self.nbCol=len(self.colNames) # number of columns
		else :
			# give a default name to the uncalled columns
			for i in range(len(self.colNames),nbCol):
				self.colNames.append('col'+str(i))
		
		self.rowNames=rowNames
		if (nbRow<=len(self.rowNames)):
			# set the number of columns to the length of the row's names
			self.nbRow=len(self.rowNames) # number of rows			
		else :
			# give a default name to the uncalled rows
			for i in range(len(self.rowNames),nbRow):
				self.rowNames.append('row'+str(i))				

		# building components
		if DEBUG :
			print "start creating the components"

		# components creation :		
		# create and place the column titles labels
		self.colTitleCanvas=_colTitlesCanvas(self.content,self.colNames,colTitlesCnf)
		self.colMaxWidth=[]
		for i in range(self.nbCol):
			self.colMaxWidth.append(self.colTitleCanvas.colTitle[i].winfo_width()) # init list
		self.showColumnsTitles=1 # if set to 1, columns titles are displayed 		

		# create and place the row titles labels 
		self.rowTitleCanvas=_rowTitlesCanvas(self.content,self.rowNames,rowTitlesCnf)
		self.rowMaxHeight=[]
		for i in range(self.nbRow):
			self.rowMaxHeight.append(self.rowTitleCanvas.rowTitle[i].winfo_height()) # init list
		self.showRowsTitles=1 # if set to 1, rows titles are displayed

		# create and place the cells' table
		self.cellsTable=_tableCellsCanvas(self.content,self.colTitleCanvas,self.rowTitleCanvas,cellsCnf)

		self.scrollbarOn=scrollbarOn
		if self.scrollbarOn:
			# create the scrollbars
			self.xscroll = Scrollbar(self.content,name="xscroll",orient=HORIZONTAL,command=(self.__xmoveCB))
			self.cellsTable['xscrollcommand'] = self.xscroll.set
			self.colTitleCanvas['xscrollcommand'] = self.xscroll.set
			self.XScrollIsEnabled=0

			self.yscroll = Scrollbar(self.content,name="yscroll",orient=VERTICAL,command=(self.__ymoveCB))	   	
			self.cellsTable['yscrollcommand'] = self.yscroll.set
			self.rowTitleCanvas['yscrollcommand'] = self.yscroll.set
			self.YScrollIsEnabled=0

		# adding the components
		self.colTitleCanvas.grid(row=0,col=1,sticky='nsew')		
		self.rowTitleCanvas.grid(row=1,col=0,sticky='nsew')		
		self.cellsTable.grid(row=1,col=1,sticky='nsew')				
		
		# configure scrollbars placing :
		# enable table and scrollbars length resizing
		self.content.rowconfigure(1,weight=1)
		self.content.columnconfigure(1,weight=1)
		# fix scrollbars width
		self.content.rowconfigure(2,weight=0)
		self.content.columnconfigure(2,weight=0)

		# set the cell size to the corresponding labels one ( and the inverse too )
		if DEBUG:
			print "start uniformizing cells size"
		self.uniformizeCellSize()
		
		# pack the components together
		self.content.update()
		self.content.pack(expand=1,fill='both')
		
		#events
		self.bind("<Configure>",self.__resizingCB)
		
		if self.scrollbarOn:
			self.__setScrollbars()

	###########################################################################
	################################  METHODS  ################################
	###########################################################################
	
	def show(self):
		"""
		show the table widget
		"""
		#self.content.pack(expand=1,fill='both')
		#self.pack()
		self.content.pack(expand=1,fill='both')
		
	def hide(self):
		"""
		hide the table widget
		"""
		#self.pack_forget()
		self.content.pack_forget()
		
	def addColumn(self,columnName="",colTitleCnf={},cellsCnf={}):
		"""
		append a column to the table widget :
		This column will contain :
		a column's title Label with the text 'columnName'
		a list of cells which contain a widget corresponding to the configuration cellsCnf 
		(must be given in a dictionnary {'cellsType':'default_cell_type',
										'cellsOptions':{'option':'OptionVal',...}},
		where cellsCnf['cellsType'] will be the default widget's type contained in each cell, 
		with a default configuration of cellsCnf['cellsOptions'].If cellsCnf isn't given, a None object will be placed in each cell
		"""
		# create and add the objects
		if columnName=="":
			columnName="col"+str(self.nbCol)
		self.colTitleCanvas.addColumn(columnName,colTitleCnf)
		self.colTitleCanvas.content.update()
		self.cellsTable.addColumn(self.nbCol,self.nbRow,cellsCnf)
		self.cellsTable.content.update()
		self.nbCol+=1

		w,h=self.colTitleCanvas.colTitle[-1].winfo_width(),self.colTitleCanvas.colTitle[-1].winfo_height()
		self.colMaxWidth.append(w)
		if (self.colTitleCanvas.maxHeight<h):
			self.colTitleCanvas.maxHeight=h
			self.colTitleCanvas.setHeight(h)
			#self.content.rowconfigure(0,minsize=h)
		
		# resizing
		rowHeight=0
		maxWidth=w
		# config height for each row
		#for i in range(len(self.rowTitleCanvas.rowTitle)):
		for i in range(self.nbRow):
			try:
				#w,h=self.cellsTable.getWidget(len(self.colTitleCanvas.colTitle)-1,i).winfo_width(),self.cellsTable.getWidget(len(self.colTitleCanvas.colTitle)-1,i).winfo_height()
				w,h=self.cellsTable.getWidget(self.nbCol-1,i).winfo_width(),self.cellsTable.getWidget(self.nbCol-1,i).winfo_height()
			except AttributeError:	
				# no displayed widget in the table
				dummy=Label(self.content)
				dummy.grid(col=0,row=0)
				#self.content.update()
				#h=self.content.bbox(column=len(self.colTitleCanvas.colTitle)-1,row=i)[2]
				h=self.cellsTable.content.bbox(column=self.nbCol-1,row=i)[2]				
				dummy.destroy()

			if w>maxWidth:
				maxWidth=w
			rowHeight=self.rowTitleCanvas.rowTitle[i].winfo_height()
			if h>rowHeight:
				self.rowMaxHeight[i]=h
				self.rowTitleCanvas.setRowHeight(i,h)
				self.cellsTable.setRowHeight(i,h)
		
		# config column width
		if maxWidth>self.colTitleCanvas.colTitle[-1].winfo_width():
			#self.colTitleCanvas.setColumnWidth(len(self.colTitleCanvas.colTitle)-1,maxWidth)
			self.colTitleCanvas.setColumnWidth(self.nbCol-1,maxWidth)
		# uniformize cells width
		#self.cellsTable.setColumnWidth(len(self.colTitleCanvas.colTitle)-1,maxWidth)
		self.cellsTable.setColumnWidth(self.nbCol-1,maxWidth)
		
		# update the canvas
		self.colTitleCanvas.updateCanvas()
		self.rowTitleCanvas.updateCanvas()
		self.cellsTable.updateCanvas()
		
		# add scrollbars if needed
		if self.scrollbarOn:
			self.__setScrollbars()

	def addRow(self,rowName="",rowTitleCnf={},cellsCnf={}):
		"""
		append a row (with the title 'rowName') to the table widget
		This row will contain :
		_ a row's title Label with the text 'columnName'
		_ a list of cells which contain a widget corresponding to the configuration cellsCnf 
		(must be given in a dictionnary {'cellsType':'default_cell_type',
										'cellsOptions':{'option':'OptionVal',...}},
		where cellsCnf['cellsType'] will be the default widget's type contained in each cell, 
		with a default configuration of cellsCnf['cellsOptions'].If cellsCnf isn't given, a None object will be placed in each cell 
		"""
		# create and add the objects
		if rowName=="":
			rowName="row"+str(len(self.rowTitleCanvas.rowTitle))
		self.rowTitleCanvas.addRow(rowName,rowTitleCnf)
		self.rowTitleCanvas.content.update()
		self.cellsTable.addRow(self.nbCol,self.nbRow,cellsCnf)
		self.cellsTable.content.update()
		self.nbRow+=1
		
		w,h=self.rowTitleCanvas.rowTitle[-1].winfo_width(),self.rowTitleCanvas.rowTitle[-1].winfo_height()
		self.rowMaxHeight.append(h)
		#self.content.columnconfigure(0,minsize=w)
		
		# resizing
		colWidth=0
		maxHeight=h
		# config width for each col
		#for i in range(len(self.colTitleCanvas.colTitle)):
		for i in range(self.nbCol):
			try:
				#w,h=self.cellsTable.getWidget(i,len(self.rowTitleCanvas.rowTitle)-1).winfo_width(),self.cellsTable.getWidget(i,len(self.rowTitleCanvas.rowTitle)-1).winfo_height()
				w,h=self.cellsTable.getWidget(i,self.nbRow-1).winfo_width(),self.cellsTable.getWidget(i,self.nbRow-1).winfo_height()
			except:
				# no displayed widget in the table
				dummy=Label(self.content)
				dummy.grid(col=0,row=0)
				#self.content.update()
				#w=self.content.bbox(row=len(self.rowTitleCanvas.rowTitle)-1,column=i)[3]
				w=self.cellsTable.content.bbox(row=self.nbRow-1,column=i)[3]
				
				dummy.destroy()
				
			if h>maxHeight:
				maxHeight=h
			colWidth=self.colTitleCanvas.colTitle[i].winfo_width()
			if w>colWidth:
				self.colMaxWidth[i]=w
				self.colTitleCanvas.setColumnWidth(i,w)
				self.cellsTable.setColumnWidth(i,w)
		
		# config row height
		if maxHeight>self.rowTitleCanvas.rowTitle[-1].winfo_height():
			#self.rowTitleCanvas.setRowHeight(len(self.rowTitleCanvas.rowTitle)-1,maxHeight)
			self.rowTitleCanvas.setRowHeight(self.nbRow-1,maxHeight)
		#self.cellsTable.setRowHeight(len(self.rowTitleCanvas.rowTitle)-1,maxHeight)
		self.cellsTable.setRowHeight(self.nbRow-1,maxHeight)
			
		# update the canvas
		self.colTitleCanvas.updateCanvas()
		self.rowTitleCanvas.updateCanvas()
		self.cellsTable.updateCanvas()
		
		# add scrollbars if needed
		if self.scrollbarOn:
			self.__setScrollbars()

	def configColTitle(self,col=0,key='bd',newVal=2):
		"""
		set the value of the option 'key' to 'newVal' for the title label of the column 'col'
		To see the available options, please refer to the Tkinter Label widget options manual
		Be careful with the height and width options, their units are expressed in text ones. 
		To use pixel units, please use the setColumnHeight,setColumnWidth methods 
		"""
		try:
			exec("(self.getColumnLabel("+str(col)+")).config("+str(key)+"='"+str(newVal)+"')")
			if (key=='bd' or key=='relief' or key=='font' or key=='height' or key=='width'):
				(self.getColumnLabel(col)).update()
				self.updateColSize(col,self.getColumnLabel(col).winfo_width(),self.getColumnLabel(col).winfo_height())
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This column "+str(col)+" doesn't exist")
		except(TclError):
			tkMessageBox.showerror("UNKNOWN OPTION","This option "+str(key)+" doesn't exist for a Label")
		
	def configRowTitle(self,row=0,key='bd',newVal=2):
		"""
		set the value of the option 'key' to 'newVal' for the title label of the row 'row'
		To see the available options, please refer to the Tkinter Label widget options manual
		Be careful with the height and width options, their units are expressed in text ones. 
		To use pixel units, please use the setRowHeight,setRowWidth methods 
		"""
		try:
			exec("(self.getRowLabel("+str(row)+")).config("+str(key)+"='"+str(newVal)+"')")
			if (key=='bd' or key=='relief' or key=='font' or key=='height' or key=='width'):
				(self.getRowLabel(row)).update()
				self.updateRowSize(row,self.getRowLabel(row).winfo_width(),self.getRowLabel(row).winfo_height())
				
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This row "+str(row)+" doesn't exist")
		except(TclError):
			tkMessageBox.showerror("UNKNOWN OPTION","This option "+str(key)+" doesn't exist for a Label")	
	
	def delColumn(self,col=0):
		"""
		delete a column (number 'col') from the table widget 
		"""
		try:
			self.cellsTable.delColumn(col)
			self.colTitleCanvas.delColumn(col)
			del self.colMaxWidth[col]
			self.nbCol-=1

			# update the canvas
			self.colTitleCanvas.updateCanvas()
			self.cellsTable.updateCanvas()
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This column "+str(col)+" doesn't exist")
		
	def delRow(self,row=0):
		"""
		delete a row (number 'row') from the table widget
		(the column and row numbering starts at 0) 
		"""
		try:
			self.cellsTable.delRow(row)
			self.rowTitleCanvas.delRow(row)
			del self.rowMaxHeight[row]
			self.nbRow-=1

			# update the canvas
			self.rowTitleCanvas.updateCanvas()
			self.cellsTable.updateCanvas()
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This row "+str(row)+" doesn't exist")

	def getCell(self,col=0,row=0):
		"""
		return the cell situed at the row 'row', and at the column 'col'
		(the column and row numbering starts at 0) 
		"""
		try :
			return(self.cellsTable.getCell(col,row))
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This cell "+"("+str(col)+","+str(row)+") doesn't exist")

	def getCellWidget(self,col=0,row=0):
		"""
		return the widget contained in the cell ('col','row')
		(the column and row numbering starts at 0) 
		"""
		if (self.getCell(col,row)):
			return (self.getCell(col,row).getWidget())
		else:
			return None
	
	def getWidgetPos(self,widget=None):
		"""
		return the position (col,row) of the widget instance 'widget' in the table
		return None if the widget hasn't been found
		"""
		found=0
		i,j=0,0
		w=None
		while i<self.nbRow and w!=widget:
			j=0
			while j<self.nbCol and w!=widget:
				w=self.getCellWidget(j,i)
				j+=1
			i+=1
		if w==widget:
			return (j,i)
		else:
			return None
	
	def getNbrOfCols(self):
		"""
		return the number of columns in this table
		"""
		return(self.nbCol)
			
	def getColumn(self,col=0):
		"""
		return a dictionnary with {colName : [sequence of the column cells objects]}
		(the column and row numbering starts at 0) 
		"""
		try:
			dic={}
			dic[self.getColumnName(col)]=self.cellsTable.getColumn(col)
			return(dic)
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This column "+str(col)+" doesn't exist")

	def getColIndex(self,colName=""):
		"""
		return the index of the first column with the title 'colName'
		return None if no column has this title
		"""
		return(self.colTitleCanvas.getColIndex(colName))
	
	def getColumnLabel(self,col=0):
		"""
		return the title label object of the column number 'col'
		(the column and row numbering starts at 0) 
		"""
		try:
			return(self.colTitleCanvas.getColumnLabel(col))
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This column "+str(col)+" doesn't exist")		
	
	def getColumnName(self,col=0):
		"""
		return the name of the column number 'col'
		(the column and row numbering starts at 0)  
		"""
		try:
			return(self.colTitleCanvas.getColumnName(col))
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This column "+str(col)+" doesn't exist")

	def getRow(self,row=0):
		"""
		return a dictionnary with {rowName : [sequence of the Row cells objects]}
		(the column and row numbering starts at 0) 
		"""
		try:
			dic={}
			dic[self.getRowName(row)]=self.cellsTable.getRow(row)
			return(dic)
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This row "+str(row)+" doesn't exist")

	def getRowIndex(self,rowName=""):
		"""
		return the index of the first row with the title 'rowName'
		return None if no row has this title
		"""
		return(self.rowTitleCanvas.getRowIndex(rowName))
	
	def getRowLabel(self,row=0):
		"""
		return the title label object of the row number 'row'
		(the column and row numbering starts at 0) 
		"""
		try:
			return(self.rowTitleCanvas.getRowLabel(row))
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This row "+str(row)+" doesn't exist")

	def getRowName(self,row=0):
		"""
		return the name of the Row number 'row'
		(the column and row numbering starts at 0)  
		"""
		try:
			return(self.rowTitleCanvas.getRowName(row))
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This row "+str(row)+" doesn't exist")

	def getNbrOfRows(self):
		"""
		return the number of rows in this table
		"""
		return(self.nbRow)
	
	def getTableSize(self):
		"""
		return the size of the table in a tuple (nbCol,nbRow)
		"""
		return((self.nbCol,self.nbRow))
		
	def getValue(self,col,row):
		"""
		return the value contained in the cell situed at the intersection of the 
		column number 'col' and the row number 'row'
		(the column and row numbering starts at 0)
		"""
		try:
			return(((self.cellsTable.cell[int(col)])[int(row)]).getValue())
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This cell "+"("+str(col)+","+str(row)+") doesn't exist")

	def hideColumnTitles(self):
		"""
		hide the column title labels
		"""
		self.colTitleCanvas.grid_forget()
		self.showColumnsTitles=0
		self.content.update()
	
	def hideRowTitles(self):
		"""
		hide the row title labels
		"""
		self.rowTitleCanvas.grid_forget()	
		#self.content.columnconfigure(0,minsize=0,weight=0)	
		self.showRowsTitles=0	
		#self.content.pack(expand=1,fill='both')
		self.content.update()
	
	def putWidgetInCell(self,col,row,obj):
		"""
		put the widget 'obj', in the cell ('col','row')
		(the column and row numbering starts at 0) 
		"""
		try:
			self.cellsTable.putWidgetInCell(col,row,obj)
			cellHeight,cellWidth=self.getCellWidget(col,row).winfo_height(),self.getCellWidget(col,row).winfo_width()
			if (self.colMaxWidth[col]<cellWidth):
				self.colMaxWidth[col]=cellWidth
				self.colTitleCanvas.setColumnWidth(col,cellWidth)
				self.colTitleCanvas.updateCanvas()
			if (self.rowMaxHeight[row]<cellHeight):
				self.rowMaxHeight[row]=cellHeight
				self.rowTitleCanvas.setRowHeight(row,cellHeight)
				self.rowTitleCanvas.updateCanvas()
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This cell "+"("+str(col)+","+str(row)+") doesn't exist")

	def setCell(self,col,row,cellCnf={'cellType':'Text','cellOptions':{}}):
		"""
		change the content of the cell ('col','row') for an object of type cellCnf['cellType'],
		with the options cellCnf['cellOptions']
		(the column and row numbering starts at 0) 
		"""
		try:
			oldCellHeight,oldCellWidth=0,0
			if (self.getCellWidget(col,row)):
				# a widget is previously displayed at (col,row)
				oldCellHeight,oldCellWidth=self.getCellWidget(col,row).winfo_height(),self.getCellWidget(col,row).winfo_width()
			else :
				# no previously displayed widget
				oldCellHeight,oldCellWidth=self.getRowLabel(row).winfo_height(),self.getColumnLabel(col).winfo_width()
				
			self.cellsTable.setCell(col,row,cellCnf)
			self.cellsTable.update()
			cellHeight,cellWidth=self.getCellWidget(col,row).winfo_reqheight(),self.getCellWidget(col,row).winfo_reqwidth()
			if (self.colMaxWidth[col]<cellWidth):
				# the new cell has the biggest width
				self.colMaxWidth[col]=cellWidth
				self.colTitleCanvas.setColumnWidth(col,cellWidth)
				self.colTitleCanvas.updateCanvas()
				self.cellsTable.updateCanvas()
			elif (oldCellWidth>cellWidth):
				# the old cell width could be the biggest one. If it is, resize the column width, else nothing
				newBiggestWidth=0
				for i in range(self.nbCol):	
					if (self.getCellWidget(i,row) and self.getCellWidget(i,row).winfo_width()>newBiggestWidth):
						newBiggestWidth=self.getCellWidget(i,row).winfo_width()
				self.colMaxWidth[col]=newBiggestWidth
				if (newBiggestWidth!=oldCellWidth):
					# update column width
					self.colTitleCanvas.setColumnWidth(col,newBiggestWidth)
					self.cellsTable.setColumnWidth(col,newBiggestWidth)
					self.colTitleCanvas.updateCanvas()
					self.cellsTable.updateCanvas()
			if (self.rowMaxHeight[row]<cellHeight):
				self.rowMaxHeight[row]=cellHeight
				self.rowTitleCanvas.setRowHeight(row,cellHeight)
				self.rowTitleCanvas.updateCanvas()
				self.cellsTable.updateCanvas()
			elif(oldCellHeight>cellHeight):
				# the old cell height could be the biggest one. If it is, resize the row , else nothing
				newBiggestHeight=0
				for i in range(self.nbRow):	
					if (self.getCellWidget(col,i) and self.getCellWidget(col,i).winfo_height()>newBiggestHeight):
						newBiggestHeight=self.getCellWidget(col,i).winfo_height()
				self.rowMaxHeight[row]=newBiggestHeight
				if (newBiggestHeight!=oldCellHeight):
					# update row height	  
					self.rowTitleCanvas.setRowHeight(row,cellHeight)
					self.cellsTable.setRowHeight(col,cellHeight)
					self.rowTitleCanvas.updateCanvas()
					self.cellsTable.updateCanvas()
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This cell "+"("+str(col)+","+str(row)+") doesn't exist")

	def setColumnName(self,col=0,newName=""):
		"""
		change the name of the column number 'col' to newName
		(the column and row numbering starts at 0) 
		"""
		if newName=="":
			newName="col"+str(col)
		try:
			# create a dummy label with the titles labels options ; new implemented styling options have to be added 
			dummyLabel=Label(self.master,
							text=newName,
							bd=(self.colTitleCanvas.colTitle[col])['bd'],
							relief=(self.colTitleCanvas.colTitle[col])['relief'],
							borderwidth=(self.colTitleCanvas.colTitle[col])['borderwidth'],
							font=(self.colTitleCanvas.colTitle[col])['font'],
							)
					
			# change the column title to the new value
			self.colTitleCanvas.setColumnName(col,newName)

			# update column's width if the new title is bigger than the old one
			w,h=dummyLabel.winfo_reqwidth(),dummyLabel.winfo_reqheight()
			self.updateColSize(col,w,h)
			
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This column "+str(col)+" doesn't exist")

	def setColumnWidth(self,col=0,newWidth=0):
		"""
		change the width of the column number 'col' for newWidth.
		If newWidth is less than the minimum requested width to see	the whole 
		column's items, the column's width will be set to this minimum width
		(the column and row numbering starts at 0) 
		"""		
		try:
			colLabel=self.getColumnLabel(col)
			minWidth=colLabel.winfo_reqwidth()
			for cell in self.getColumn(col)[colLabel['text']]:

				cellWidth=0
				if (cell.getWidget()):
					cellWidth=cell.getWidget().winfo_reqwidth()

				if (cellWidth>minWidth):
					minWidth=cellWidth

			if (minWidth<=newWidth):
				self.colTitleCanvas.setColumnWidth(col,newWidth)		
				self.cellsTable.setColumnWidth(col,newWidth)
				self.colMaxWidth[col]=newWidth
			else :
				self.colTitleCanvas.setColumnWidth(col,minWidth)		
				self.cellsTable.setColumnWidth(col,minWidth)
				self.colMaxWidth[col]=minWidth

			# update the canvas
			self.cellsTable.updateCanvas()
			self.colTitleCanvas.updateCanvas()
		except(ValueError):
			tkMessageBox.showerror("VALUE ERROR","The new width must be expressed as an integer. "+str(newWidth)+" is not a valid integer")

	def setRowHeight(self,row=0,newHeight=0):
		"""
		change the height of the row number 'row' for newHeight
		If newHeight is less than the minimum requested height to see 
		the whole row's items, the row's height will be set to this minimum height
		(the column and row numbering starts at 0) 
		"""
		try:
			rowLabel=self.getRowLabel(row)
			minHeight=rowLabel.winfo_reqheight()
			for cell in self.getRow(row)[rowLabel['text']]:
				cellHeight=0
				if (cell.getWidget()):
					cellHeight=cell.getWidget().winfo_reqheight()

				if (cellHeight>minHeight):
					minHeight=cellHeight

			if (minHeight<=newHeight):
				self.rowTitleCanvas.setRowHeight(row,newHeight)		
				self.cellsTable.setRowHeight(row,newHeight)
				self.rowMaxHeight[row]=newHeight
			else :
				self.rowTitleCanvas.setRowHeight(row,minHeight)		
				self.cellsTable.setRowHeight(row,minHeight)
				self.rowMaxHeight[row]=minHeight

			# update the canvas
			self.cellsTable.updateCanvas()
			self.rowTitleCanvas.updateCanvas()
		except(ValueError):
			tkMessageBox.showerror("VALUE ERROR","The new height must be expressed as an integer. "+str(newHeight)+" is not a valid integer")
	
	def setRowName(self,row=0,newName=""):
		"""
		change the name of the row 'row' to newName
		if the name is bigger than the biggest of the old row titles, 
		the row titles width will be updated
		(the column and row numbering starts at 0) 
		"""
		if newName=="":
			newName="row"+str(row)
		try:
			# create a dummy label with the titles labels options ; new implemented styling options have to be added 
			dummyLabel=Label(self.master,
							text=newName,
							bd=(self.rowTitleCanvas.rowTitle[row])['bd'],
							relief=(self.rowTitleCanvas.rowTitle[row])['relief'],
							borderwidth=(self.rowTitleCanvas.rowTitle[row])['borderwidth'],
							font=(self.rowTitleCanvas.rowTitle[row])['font'],
							)

			# change the row title to the new value
			self.rowTitleCanvas.setRowName(row,newName)

			# update rows width if the new title is bigger than the biggest old one
			w,h=dummyLabel.winfo_reqwidth(),dummyLabel.winfo_reqheight()
			self.updateRowSize(row,w,h)
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This row "+str(row)+" doesn't exist")
		
	def setValue(self,col=0,row=0,newVal=""):
		"""
		set the value contained in the cell situed at the intersection 
		of the column number 'col' and the row number 'row'	to 'newVal' 
		(the column and row numbering starts at 0)
		"""
		try:
			((self.cellsTable.cell[int(col)])[int(row)]).setValue(newVal)
		except(IndexError):
			tkMessageBox.showerror("INDEX ERROR","This cell "+"("+str(col)+","+str(row)+") doesn't exist")

	def showColumnTitles(self):
		"""
		show the column title labels
		"""
		self.colTitleCanvas.grid(row=0,col=1,sticky='nsew')
		#self.content.rowconfigure(0,minsize=self.colTitleCanvas.maxHeight,weight=0)
		self.showColumnsTitles=1
		#self.content.pack(expand=1,fill='both')
		self.content.update()

	def showRowTitles(self):
		"""
		show the row title labels
		"""
		self.rowTitleCanvas.grid(row=1,col=0,sticky='nsew')
		#self.content.columnconfigure(0,minsize=self.rowTitleCanvas.maxWidth,weight=0)
		self.showRowsTitles=1
		#self.content.pack(expand=1,fill='both')
		self.content.update()

	def uniformizeCellSize(self):
		"""
		update the cells size to the labels size if the cell size is bigger than the 
		label's one (and the cell size if the label size is bigger than the cell one)
		"""
		if (len(self.colTitleCanvas.colTitle)==0):
			# only row titles labels exist
			for i in range(self.nbRow):				
				self.rowTitleCanvas.setRowHeight(i,self.rowMaxHeight[i])
				self.cellsTable.setRowHeight(i,self.rowMaxHeight[i])
			self.cellsTable.setColumnWidth(0,0)
			self.colTitleCanvas.setColumnWidth(0,0)
		elif(self.nbRow==0):
			# only col titles labels exist
			for i in range(self.nbCol):
				self.colTitleCanvas.setColumnWidth(i,self.colMaxWidth[i])
				self.cellsTable.setColumnWidth(i,self.colMaxWidth[i])
			self.cellsTable.setRowHeight(0,0)
			self.rowTitleCanvas.setRowHeight(0,0)
		else:
			# general case
			for i in range(self.nbCol):
				for j in range(self.nbRow):
					# config column titles height
					if (self.colTitleCanvas.colTitle[i].winfo_height()>self.colTitleCanvas.maxHeight):
						self.colTitleCanvas.setHeight(self.colTitleCanvas.colTitle[i].winfo_height())

					# config row titles width
					if (self.rowTitleCanvas.rowTitle[j].winfo_width()>self.rowTitleCanvas.maxWidth):
						self.rowTitleCanvas.setWidth(self.rowTitleCanvas.rowTitle[j].winfo_width())

					# config cells size
					if (self.cellsTable.getCell(i,j).getWidget()):
						w=self.cellsTable.getCell(i,j).getWidget().winfo_width()
						h=self.cellsTable.getCell(i,j).getWidget().winfo_height()
					else:
						w,h=0,0

					# set column width
					if w>self.colMaxWidth[i]:
						self.colMaxWidth[i]=w
						self.colTitleCanvas.setColumnWidth(i,w)
					elif(w<self.colMaxWidth[i]):
						self.cellsTable.setColumnWidth(i,self.colTitleCanvas.colTitle[i].winfo_width())
					if h>self.rowMaxHeight[j]:
						self.rowMaxHeight[j]=h
						self.rowTitleCanvas.setRowHeight(j,h)
					elif(h<self.rowTitleCanvas.rowTitle[j].winfo_height()):
						self.cellsTable.setRowHeight(j,self.rowTitleCanvas.rowTitle[j].winfo_height())

		# config titles and cells table new scrollregion
		self.colTitleCanvas.updateCanvas()
		self.rowTitleCanvas.updateCanvas()
		self.cellsTable.updateCanvas()
		
		self.content.pack(expand=1,fill='both')
		self.content.update()

	def updateColSize(self,column,width,height):
		"""
		update the table widget's cells and titles size when a resizing of ('width','height') the column label 
		'column' occurs
		"""
		if (self.colMaxWidth[column]<width):
			self.colMaxWidth[column]=width
			self.colTitleCanvas.setColumnWidth(column,width)
			self.cellsTable.setColumnWidth(column,width)
			# update the canvas
			self.cellsTable.updateCanvas()
		if (self.colTitleCanvas.maxHeight<height):
			self.colTitleCanvas.maxHeight=height
			self.colTitleCanvas.setHeight(height)
		self.colTitleCanvas.updateCanvas()

	def updateRowSize(self,row,width,height):
		"""
		update the table widget's cells and titles size when a resizing of ('width','height') the row label 
		'row' occurs
		"""
		if (self.rowMaxHeight[row]<height):
			self.rowMaxHeight[row]=height
			self.rowTitleCanvas.setRowHeight(row,height)
			self.cellsTable.setRowHeight(row,height)
			# update the canvas
			self.cellsTable.updateCanvas()
		if (self.rowTitleCanvas.maxWidth<width):
			self.rowTitleCanvas.maxWidth=width
			self.rowTitleCanvas.setWidth(width)
		self.rowTitleCanvas.updateCanvas()

	###########################################################################
	###############################  CALLBACKS  ###############################
	###########################################################################
	
	def __xmoveCB(self,a,b,c=None):
		"""
		callback : called by the horizontal scrollbar for scrolling the table on x axis 
		"""
		self.cellsTable.xview(a,b,c)
		if self.showColumnsTitles:
			self.colTitleCanvas.xview(a,b,c)
		
	def __ymoveCB(self,a,b,c=None):
		"""
		callback : called by the vertical scrollbar for scrolling the table on y axis 
		"""
		self.cellsTable.yview(a,b,c)
		if self.showRowsTitles:
			self.rowTitleCanvas.yview(a,b,c)
		
	def __setXScroll(self,XScrollSet=0):
		"""
		display the X scrollbar if XScrollSet is one (hide it if nul)
		"""
		if (self.XScrollIsEnabled!=XScrollSet):
			self.XScrollIsEnabled=XScrollSet
			if (self.XScrollIsEnabled):
				self.xscroll.grid(row=2,col=1,sticky='sew')
			else:
				self.xscroll.grid_forget()
	
	def __setYScroll(self,YScrollSet=0):
		"""
		display the Y scrollbar if YScrollSet is one (hide it if nul)
		"""
		if (self.YScrollIsEnabled!=YScrollSet):
			self.YScrollIsEnabled=YScrollSet
			if (self.YScrollIsEnabled):
				self.yscroll.grid(row=1,col=2,sticky='nse')
			else:
				self.yscroll.grid_forget()
				
	def __setScrollbars(self):
		"""
		display the scrollbars if they are useful, hide it if not
		"""

		self.content.update_idletasks()
		
		scrollbarWidth=self.yscroll.winfo_reqwidth()
		
		tableHeight=self.cellsTable.winfo_reqheight()
		tableWidth=self.cellsTable.winfo_reqwidth()

		if self.showColumnsTitles==1: tableHeight+=self.colTitleCanvas.winfo_reqheight()
		if self.showRowsTitles==1: tableWidth+=self.rowTitleCanvas.winfo_reqwidth()
		
		if self.xscroll.winfo_viewable() or self.content.winfo_width()<tableWidth: # x scrollbar present or will be added soon
			tableHeight+=scrollbarWidth
		if self.yscroll.winfo_viewable() or self.content.winfo_height()<tableHeight: # y scrollbar present or will be added soon
			tableWidth+=scrollbarWidth

		if (self.cellsTable.winfo_height()!=1 and self.content.winfo_height()<tableHeight):
			self.__setYScroll(1)
		else:
			self.__setYScroll(0)
		if (self.cellsTable.winfo_width()!=1 and self.content.winfo_width()<tableWidth):
			self.__setXScroll(1)
		else:
			self.__setXScroll(0)
			
	def __resizingCB(self,event):
		"""
		callback for a master window's resizing event
		"""
		if self.scrollbarOn:
			self.__setScrollbars()	
		
class _colTitlesCanvas(Canvas):
	"""
	canvas containing the column's title labels
	"""
	def __init__(self,master,colNames=[],colTitlesCnf={},**kw):
		"""
		builder
		"""
		Canvas.__init__(self,master,kw)
		
		self.master=master
		self.content=Frame(self)
		
		self.colTitle=[]
		self.maxHeight=0
		unknownWidth=0
		if (not colTitlesCnf.has_key('width')):
			unknownWidth=1
		if (not colTitlesCnf.has_key('height')):
			colTitlesCnf['height']=1
			
		for i in range(len(colNames)):
			if unknownWidth:
				colTitlesCnf['width']=len(colNames[i])
			exec("self.colTitle.append(Label(self.content,"+repr(colTitlesCnf)+",text=colNames[i]))")
			self.colTitle[i].grid(row=0,col=i,sticky='nsew')	
			if (self.colTitle[i].winfo_reqheight()>self.maxHeight):
				self.maxHeight=self.colTitle[i].winfo_reqheight()
		self.content.update()
		
	###########################################################################
	################################  METHODS  ################################
	###########################################################################
	
	def getColIndex(self,colName):
		"""
		return the index of the first column with the title 'colName'
		return None if no column has this title
		"""
		i=0
		while (i<len(self.colTitle) and self.colTitle[i]['text']!=colName):
			i+=1
		if (i!=len(self.colTitle)):
			return(i)
		else :
			return(None)
	
	def getColumnLabel(self,col):
		"""
		return the title label of the column 'col'
		"""
		return(self.colTitle[col])
	
	def delColumn(self,col):
		"""
		delete a column (number 'col') from the column titles (the first column has the number 0) 
		"""
		# delete the label of the column 'col'	
		self.colTitle[col].destroy()
		del self.colTitle[col]
		
		for i in range(col,len(self.colTitle)):
			self.setColumnWidth(i,self.colTitle[i].winfo_width())
			self.colTitle[i].grid(row=0,col=i,sticky='nsew')

		# update columnWidth values
		self.content.columnconfigure(len(self.colTitle),minsize=0)
	
	def updateCanvas(self):
		"""
		re-built the canvas
		"""
		self.delete(ALL)
		#self.content.pack(expand=1,fill='both')
		self.master.update()
		#self.content.update()
		#self.content.update_idletasks()
		#print "\ncolTitle canvas height : ",self.content.winfo_height()," reqheight : ",self.content.winfo_reqheight()
		#print "colTitle canvas width : ",self.content.winfo_width()," reqwidth : ",self.content.winfo_reqwidth()
		x,y=self.content.winfo_reqheight(),self.content.winfo_reqwidth()
		self.config(height=x,width=y)
		
		# update scrollRegion
		scrollbox=(0,0,y,x)
		self.config(scrollregion=scrollbox)
		
		self.contentid=self.create_window(0,0,anchor='nw',window=self.content)	
	
	def addColumn(self,columnName=None,colTitleCnf={}):
		"""
		append a column (with the title "columnName") 
		If columnName is None, a default name is given : 'col'+ the number of the label position
		"""
		# give a default column name if not defined
		if (columnName==None):
			columnName='col'+str(len(self.colTitle))
		# create and place the label	
		if colTitleCnf!={}:
			# a configuration is defined
			self.colTitle.append(Label(self.content,text=columnName,width=len(columnName),height=1,**colTitleCnf))
		elif len(self.colTitle)>0:
			# a column title label already exists : get its configuration 
			self.colTitle.append(Label(self.content,text=columnName,width=len(columnName),height=1,
									bg=self.colTitle[0]['bg'],bd=self.colTitle[0]['bd'],relief=self.colTitle[0]['relief']))
		else:
			# no already existing column title label and no configuration
			self.colTitle.append(Label(self.content,text=columnName,width=len(columnName),height=1))
								
		self.colTitle[-1].grid(row=0,col=(len(self.colTitle)-1),sticky='nsew')
		self.content.update()
		# update columnWidth values
		if (self.colTitle[-1].winfo_height()>self.maxHeight):
			self.maxHeight=self.colTitle[-1].winfo_height()
	
	def setColumnName(self,col,newName):
		"""
		change the name of the column 'col' to newName
		"""		
		self.colTitle[col].config(text=newName)
	
	def getColumnName(self,col):
		"""
		return the name of the column 'col'
		"""
		return(self.colTitle[col].cget('text'))

	def setHeight(self,newHeight=1):
		"""
		set the height of the column's title label,and set their height to 'newHeight'
		"""
		self.content.rowconfigure(0,minsize=newHeight)
					
	def setColumnWidth(self,column,newWidth):
		"""
		set the width of the column's title label number 'column',and set his width to 'newWidth'
		"""
		self.content.columnconfigure(column,minsize=newWidth)

class _rowTitlesCanvas(Canvas):
	"""
	canvas containing the row's title labels
	"""
	def __init__(self,master,rowNames=[],rowTitlesCnf={},**kw):
		"""
		builder
		"""
		Canvas.__init__(self,master,kw)
		
		self.master=master
		self.content=Frame(self)
		
		# set max width needed to display all the row labels
		maxWidth=0
		for i in range(len(rowNames)):
			if (len(rowNames[i])>maxWidth):
				maxWidth=len(rowNames[i])
		
		self.maxWidth=0
		self.rowTitle=[]
		if (not rowTitlesCnf.has_key('width')):
			rowTitlesCnf['width']=maxWidth
		if (not rowTitlesCnf.has_key('height')):
			rowTitlesCnf['height']=1
				
		for i in range(len(rowNames)):
			exec("self.rowTitle.append(Label(self.content,"+repr(rowTitlesCnf)+",text=rowNames[i]))")
			self.rowTitle[i].grid(col=0,row=i,sticky='nsew')
			if self.rowTitle[i].winfo_reqwidth()>self.maxWidth:
				self.maxWidth=self.rowTitle[i].winfo_reqwidth()
		self.content.update()

	###########################################################################
	################################  METHODS  ################################
	###########################################################################
	
	def getRowIndex(self,rowName):
		"""
		return the index of the first row with the title 'rowName'
		return None if no row has this title
		"""
		i=0
		while (i<len(self.rowTitle) and self.rowTitle[i]['text']!=rowName):
			i+=1
		if (i!=len(self.rowTitle)):
			return(i)
		else :
			return(None)
	
	def getRowLabel(self,row):
		"""
		return the title label object of the row number 'row'
		"""
		return(self.rowTitle[row])
	
	def delRow(self,row):
		"""
		delete a row (number 'row') from the row titles (the first row has the number 0) 
		"""
		# delete the label of the row 'row'	
		titleToDel=self.rowTitle[row]
		titleToDel.destroy()
		del self.rowTitle[row]
		
		for i in range(row,len(self.rowTitle)):
			self.setRowHeight(i,self.rowTitle[i].winfo_height())
			self.rowTitle[i].grid(row=i,col=0,sticky='nsew')

		# update RowWidth values
		self.content.rowconfigure(len(self.rowTitle),minsize=0)
		self.content.update()
	
	def updateCanvas(self):
		"""
		re-built the canvas
		"""
		self.delete(ALL)
		#self.content.pack(expand=1,fill='both')
		#self.content.update()
		self.master.update()
		#print "\nrowTitle canvas height : ",self.content.winfo_height()," reqheight : ",self.content.winfo_reqheight()
		#print "rowTitle canvas width : ",self.content.winfo_width()," reqwidth : ",self.content.winfo_reqwidth()
		x,y=self.content.winfo_reqheight(),self.content.winfo_reqwidth()
		self.config(height=x,width=y)
		
		# update scrollRegion
		scrollbox=(0,0,y,x)
		self.config(scrollregion=scrollbox)
		
		self.contentid=self.create_window(0,0,anchor='nw',window=self.content)	
	
	def addRow(self,rowName=None,rowTitleCnf={}):
		"""
		append a column (with the title "columnName") 
		If columnName is None, a default name is given : 'col'+ the number of the label position
		"""
		# give a default row name if not defined
		if (rowName==None):
			rowName='row'+str(len(self.rowTitle))
		# create and place the label	
		if rowTitleCnf!={}:
			# a label configuraton is given
			self.rowTitle.append(Label(self.content,text=rowName,width=len(rowName),height=1,**rowTitleCnf))
		elif len(self.rowTitle)>0:
			# a row title label already exists
			self.rowTitle.append(Label(self.content,text=rowName,width=len(rowName),height=1,
									bg=self.rowTitle[0]['bg'],bd=self.rowTitle[0]['bd'],relief=self.rowTitle[0]['relief']))
		else:
			# no row title label : use default label options
			self.rowTitle.append(Label(self.content,text=rowName,width=len(rowName),height=1))
		
		self.rowTitle[-1].grid(row=len(self.rowTitle)-1,col=0,sticky='nsew')
			
		# update rowHeight values
		self.rowTitle[-1].update()
		self.content.update()
		if (self.rowTitle[-1].winfo_width()>self.maxWidth):
			self.maxWidth=self.rowTitle[-1].winfo_width()
			self.setWidth(self.maxWidth)

	def setRowName(self,row,newName):
		"""
		change the name of the row 'row' to newName
		if the name is bigger than the biggest of the old row titles, the row titles width will be updated
		"""
		self.rowTitle[row].config(text=newName)

	def getRowName(self,row):
		"""
		return the name of the column 'row'
		"""
		return(self.rowTitle[row].cget('text'))

	def setRowHeight(self,row,newHeight=1):
		"""
		set the height of the row label's 'row',and set his height to 'newHeight'
		"""
		self.content.rowconfigure(row,minsize=newHeight)
	
	def setWidth(self,newWidth):
		"""
		set the width of the row title labels,and set their width to 'newWidth'
		"""
		self.content.columnconfigure(0,minsize=newWidth)	
	
	def getWidth(self):
		"""
		return the width of the row's titles label, in pixel unit
		"""
		return(self.maxWidth)
		
class _tableCellsCanvas(Canvas):
	"""
	canvas containing the table cells
	"""
	def __init__(self,master,colFrame,rowFrame,cellsCnf):
		"""
		builder
		"""
		Canvas.__init__(self,master)
		
		self.master=master
		self.content=Frame(self)
		
		self.colFrame=colFrame
		self.rowFrame=rowFrame

		self.cell=[]
		for i in range(len(colFrame.colTitle)):
			rowCell=[]
			for j in range(len(rowFrame.rowTitle)):
				cell=TableCell(self.content,cellsCnf)				
				rowCell.append(cell)
				if (cell.getWidget()!=None):
					cell.getWidget().grid(row=j,col=i,sticky='nsew')
				else :
					self.setRowHeight(j,self.rowFrame.rowTitle[j].winfo_height())
					self.setColumnWidth(i,self.colFrame.colTitle[i].winfo_width())
				
			self.cell.append(rowCell)
		self.content.pack(expand=1,fill='both')
		self.content.update()

	###########################################################################
	################################  METHODS  ################################
	###########################################################################
			
	def putWidgetInCell(self,col,row,obj):
		"""
		put the widget 'obj', in the cell ('col','row')
		"""
		cellToChange=self.getCell(col,row)
		cellToChange.putWidgetInCell(obj)
		cellToChange.getWidget().grid(col=col,row=row,sticky='nsew',in_=self.content)
		self.updateCanvas()
	
	def getWidget(self,col,row):
		return(self.getCell(col,row).widget)
	
	def setCell(self,col,row,cellCnf={'cellType':'Text','cellOptions':{'text':'blablabla'}}):
		"""
		change the content of the cell ('col','row') for the object 'cellType', with the options 'cellOptions'
		"""
		cellToChange=self.getCell(col,row)
		cellToChange.setCell(cellCnf)
		cellToChange.getWidget().grid(row=row,col=col,sticky='nsew',in_=self.content)
		#cellToChange.getWidget().update()
		#self.updateCanvas()
	
	def delRow(self,row):
		"""
		delete a Row (number 'row') from the Row titles (the first Row has the number 0) 
		"""
		# delete the label of the Row 'row'	
		for col in range(len(self.cell)):
			if ((self.cell[col])[row].getWidget()):
				(self.cell[col])[row].getWidget().destroy()
			del (self.cell[col])[row]
			
		# move the others cells in the free places
		nbRow=len(self.cell[0])
		for j in range(row,nbRow): # row
			self.setRowHeight(j,self.rowFrame.rowTitle[j].winfo_height())
			for i in range(len(self.cell)): # col
				if ((self.cell[i])[j].getWidget()):
					#((self.cell[i])[j]).getWidget().grid_forget()
					((self.cell[i])[j]).getWidget().grid(row=j,col=i,sticky='nsew')
		self.content.rowconfigure(nbRow,minsize=0)	

	def delColumn(self,col):
		"""
		delete a column (number 'col') from the column titles (the first column has the number 0) 
		"""
		# delete the label of the column 'col'	
		
		for row in range(len(self.cell[col])):
			if ((self.cell[col])[row].getWidget()):
				(self.cell[col])[row].getWidget().destroy()
		del self.cell[col]
		# move the others cells in the free places	
		for i in range(col,len(self.cell)): # col
			self.setColumnWidth(i,self.colFrame.colTitle[i].winfo_width())
			for j in range(len(self.cell[i])): # row
				if (((self.cell[i])[j]).getWidget()):
					#((self.cell[i])[j]).getWidget().grid_forget()
					((self.cell[i])[j]).getWidget().grid(row=j,col=i,sticky='nsew')
		self.content.columnconfigure(len(self.cell),minsize=0)

	def updateCanvas(self):
		"""
		re-built the canvas
		"""
		# erase old canvas
		self.delete(ALL)
		# pack the content
		#self.content.pack(fill='both',expand=1)
		#self.master.update()
		#self.content.update()
		
		#x,y=self.content.winfo_reqheight(),self.content.winfo_reqwidth()
		#if (x==1 and y==1):
			# no widget in the table. Must place a dummy one to catch the requested width and height # to be done only when needed. Not the case now!!!!!
		dummy=Label(self.content)
		dummy.grid(col=0,row=0)
		self.master.update()
		
		#print "\nTable canvas height : ",self.content.winfo_height()," reqheight : ",self.content.winfo_reqheight()
		#print "Table canvas width : ",self.content.winfo_width()," reqwidth : ",self.content.winfo_reqwidth()
		
		x,y=self.content.winfo_reqheight(),self.content.winfo_reqwidth()
		dummy.destroy()

		# config canvas size
		self.config(height=x,width=y)
		
		# update scrollRegion
		scrollbox=(0,0,y,x)
		self.config(scrollregion=scrollbox)
		
		self.contentid=self.create_window(0,0,anchor='nw',window=self.content)	
	
	def addRow(self,nbCol,nbRow,cellsCnf):
		"""
		append a row (with the title "rowName") 
		"""
		# add a table cell to each row 
		for i in range(nbCol):
			col=self.cell[i]
			cell=TableCell(self.content,cellsCnf=cellsCnf)
			col.append(cell)									
			if (cell.getWidget()):
				cell.getWidget().grid(row=nbRow,col=i,sticky='nsew')
			self.cell[i]=col

	def addColumn(self,nbCol,nbRow,cellsCnf):
		"""
		append a column (with the title "columnName") 
		"""
		# add a table cell to each row 
		col=[]
		for i in range(nbRow):
			cell=TableCell(self.content,cellsCnf=cellsCnf)
			col.append(cell)									
			if (cell.getWidget()):
				cell.getWidget().grid(row=i,col=nbCol,sticky='nsew')
		
		(self.cell).append(col)

	def getCell(self,col,row):
		"""
		returned the object contained in the cell situed at the row 'row', and at the column 'col'
		"""
		return((self.cell[col])[row])
	
	def getRow(self,row):
		"""
		return a sequence of the objects contained in the row number 'row'		
		"""
		seq=[]
		for i in range(len(self.cell)):
			seq.append((self.cell[i])[row])
		return(seq)
	
	def getColumn(self,col):
		"""
		return a sequence of the objects contained in the column number 'col'
		"""		
		return(self.cell[col])
	
	def setRowHeight(self,row,newHeight):
		"""
		set the height of the table's row number 'row',and set his height to 'newHeight'
		"""
		self.content.rowconfigure(row,minsize=newHeight)
	
	def setColumnWidth(self,col,newWidth):
		"""
		set the width of the table's column number 'col',and set his width to 'newWidth'
		"""
		self.content.columnconfigure(col,minsize=newWidth)

class TableCell:
	"""
	generic class :
	represents one cell of the TableWidget.
	To specify it, overwrite the methods : setValue,getValue
	
	"""	
	def __init__(self,master,cellsCnf):
		"""
		builder
		@param master : the master frame of this cell
		@param cellsCnf : the configuration of the widget contained in the cell
							(must be given in a dictionnary {'cellsType':'default_cell_type',
															'cellsOptions':{'option':'OptionVal',...}}
		"""
		self.master=master
		
		# set a default object in the cell, if it hasn't been given
		if cellsCnf=={}:
			#cellsCnf={'cellsType':'Frame','cellsOptions':{'relief':'ridge','bd':2}} # another way to set a default widget cell type
			self.type=None
			self.opts=None
			self.widget=None
		else:
			self.type=cellsCnf['cellsType']
			self.opts=cellsCnf['cellsOptions']

			exec("self.widget="+self.type+"(self.master,"+repr(self.opts)+")")

	###########################################################################
	################################  METHODS  ################################
	###########################################################################

	def putWidgetInCell(self,obj):
		"""
		put the already created widget 'obj' in the cell
		"""
		if (self.widget):
			self.widget.destroy()
		self.widget=obj		
		try:
			# type and options to be defined
			self.type=self.widget.__class__
			
			# catch options for a Tkinter widget
			self.opts={}
			for key in self.widget.keys():
					self.opts[key]=self.widget[key]
		except(AttributeError):
			# cath options for a other widget
			self.opts=self.widget.__dict__
	
	def getWidget(self):
		"""
		return the widget contained in this cell
		"""
		return(self.widget)
	
	def setCell(self,cellCnf={}):
		"""
		set the object contained in the cell become 'object'
		"""
		if (self.widget):
			self.widget.destroy()
		try :
			self.type=cellCnf['cellType']
			self.opts=cellCnf['cellOptions']
		
			exec("self.widget="+self.type+"(self.master,"+repr(self.opts)+")")
		except:
			pass
	
	def getValue(self):
		"""
		return the value contained in the cell
		must be overwritten depending of the object contained in the cell
		"""
		try:
			return(self.widget.get(1.0, END))
		except:
			try:
				return(self.widget['text'])
			except:
				pass
			
	def setValue(self,newVal):
		"""
		set the cell value to newVal
		must be overwritten depending of the object contained in the cell
		"""
		try:	
			self.widget.delete(1.0, END)
			self.widget.insert(END,newVal)
		except:
			try:	
				self.widget.config(text=newVal)
			except:
				pass

def test():
	if TRACE:
		print os.system('date')," : start exec "
	root=Tk()
	f=Frame(root)
	z=TableWidget(master=f,#colNames=['mycoltodestroy111111111111111111111','myco2','mycllllllllllll1','myc2'],
							#rowNames=['myrowtodestroy11111111','myrow2','myll111111row3','mylrow2','myllll11111row2','myrow3'],
							nbCol=5,nbRow=10,
							rowTitlesCnf={'bg':'blue','bd':4,'relief':'groove'},
							colTitlesCnf={'bg':'red','bd':2,'relief':'sunken'},
							#cellsCnf={'cellsType':'Button','cellsOptions':{'text':'blablabla','height':2,'width':9,'bd':5}},
							#cellsCnf={'cellsType':'Text','cellsOptions':{'height':1,'width':7,'bd':2}},
							)
	
	# test the titles hide function
	#z.hideColumnTitles() # OK
	#z.hideRowTitles() # OK

	# test the titles show function
	#z.showColumnTitles() # OK
	#z.showRowTitles() # OK

	# test the getValue function
	#print z.getValue(0,5) 

	# test the getColumn/getRow function
	#print z.getColumn(0) 
	#print z.getRow(1) 

	# test the setColumnName/setRowName function
	#z.setColumnName(0,'blablablablablablablablablablablabla1') # OK
	#z.setRowName(0,'ablablablablablablablablabla2') # OK 

	# test the getCell function
	#print (z.getCell(0,5)).getValue() 

	# test the addColumn function
	#z.addColumn('col++++++++++++++++++++',{'cellsType':'Text','cellsOptions':{'height':3,'width':7}}) # OK
	#z.addColumn('col+++++++++++++++++++++') # OK

	# test the delColumn function
	#z.delColumn(0) # OK 

	# test the addRow function
	#z.addRow('row+++++++++++++++++++++++',{'cellsType':'Button','cellsOptions':{}}) # OK
	#z.addRow('row++++++++++++++++++++++++') # OK

	# test the delRow function
	z.delRow() # OK 

	# test the configColTitle/configRowTitle function
	#z.configColTitle(2,'bg','white') # OK
	#z.configRowTitle(2,'bg','white') # OK
	#z.configColTitle(2,'bd',15) # OK
	#z.configRowTitle(2,'bd',20) # OK
	#z.configColTitle(2,'width',15) # OK
	#z.configColTitle(2,'height',3) # OK
	#z.configRowTitle(2,'height',5) # OK
	#z.configRowTitle(2,'width',10) # OK

	# test the setCell function
	#print z.getNbrOfCols()-1
	#print z.getNbrOfRows()-1
	#z.setCell(z.getNbrOfCols()-1,z.getNbrOfRows()-1,{'cellType':'Button','cellOptions':{'text':'ouais genial ca marche','bd':10,'relief':'groove'}}) # OK 

	# test the putWidgetInCell function
	#obj=Scale(z.cellsTable.content)# the master of the object must be the cellsTable content. In this case : OK
	obj=Scale(z.cellsTable.content)
	#z.putWidgetInCell(0,1,obj) 

	# test the setColumnWidth function
	#z.setColumnWidth(0,200) # OK

	# test the setRowHeight function
	#z.setRowHeight(1,50) # OK

	# test the getTableSize
	#print z.getTableSize() # OK
	#print z.getColIndex('col2') # OK
	#print z.getRowIndex('row3') # OK

	f.pack(expand=1,fill='both')
	if TRACE:
		print os.system('date')," : end exec "

	root.mainloop()
	
if __name__=='__main__':
	test()
