# define WAXS_VERSION      "waxs : V1.0 Peter Boesecke 2001-04-20"
/*+++------------------------------------------------------------------------
NAME
   waxs --- routines for waxs detector projections 

SYNOPSIS

   # include waxs.h

HISTORY
  2001-04-19 V1.0 

DESCRIPTION

These routines calculate the projection of the Ewald sphere for a scattering 
pattern that was taken with an inclined 2d-detector and project it to a plane 
surface. The radial distance from the center of this surface is the length of 
the scattering vector s. The azimuthal angle is proportional to the azimuthal 
angle of the scattered beam. 

In the following, 3d vectors are followed by '~', the length of a vector 
is just its name: e.g. kin = ||kin~||. Vectors with unit length are
followed by '^', e.g. kin^ = kin~/kin. 

      kin~              : wavevector of incident beam
(i)   kout~             : wavevector of scattered beam
      s~ = kout~ - kin~ : scattering vector

The scattering is elastic. The wavenumber k of kin~ and kout~ is:

(ii) k = 1/wavelength = kin = kout

Scattering Geometry and Orientation of Detector
The angles rot1, rot2 and rot3 define the orientation of the detector
in the lab coordinate system with orientation 1. The unit vectors along the 
three axes are: e1^, e2^, e3^. The rotations are applied sequentially. The 
first rotation is around axis 1, the second around axis 2 and the third around 
axis3. All rotations are counter clockwise. The coordinate system with 
orientation 1 is defined in the following way:

  origin (0,0) at lower left corner of the image
  axis 1: horizontally to the right
  axis 2: vertically up
  axis 3: against beam direction

             ^ 2 (vertical)
             |
             |
             |
             |
             +------> 1 (horizontal)
             \
              \
               \
                _| 3 (against primary beam)

ID01:  -x <=> axis 1,  y <=> axis 2, -z <=> axis 3
ID02:   x <=> axis 1, -y <=> axis 3,  z <=> axis 2

The primary beam (kin~) is antiparallel to axis3.

(iii) kin~ = -kin * e3^

The direction of the scattered beam relative to the coordinate system
of the inclined detector image is:

The direction of the scattered beam expressed in saxs-coordinates (see *)
of the inclined detector is:

            (  sc_1 )
(iv) sc~ =  |  sc_2 | 
            (   -k  ) 

Here, sc_1 and sc_2 are saxs-coordinates of the detector image with respect to
the point of normal incidence ("center"). k is the wavenumber.
The direction kout^ is calculated by

(v) kout^ = A * sc^ 

where A is the rotation matrix that describes the inclination of the detector.

With eq. i to eq. v the scattering vector becomes

(vi) s~ = k * kout^  + k * e3^

kout^ can also be expressed by the scattering angle 2Theta and the
azimuthal angle alpha:

                         ( sin(2Theta)*cos(alpha) )
(vii) kout^ = A * sc^ =  | sin(2Theta)*sin(alpha) |
                         (     -cos(2Theta)       )

The output image is the projection of s on a plane surface. The coordinates
are sp_1 and sp_2: 

(viii)  sp_1 = s * cos(alpha)
        sp_2 = s * sin(alpha)

From eq. viii s and cos(alpha) can be calculated:

(ix)   s = sqrt(sp_1^2+sp_2^2)
       cos(alpha) = sp_1/s, sin(alpha) = sp_2/s

Eq. vi allows to calculate sin(2Theta) and cos(2Theta) from s and k:

(x)                   s^2 = k^2 * (kout^ + e3^)^2
                          = k^2 * (1 + 2*kout^*e3^ + 1)
                          = k^2 * 2 * (1 + kout^*e3^)
                          = k^2 * 2 * (1 - cos(2Theta))
                          = k^2 * 2 * 2*sin(Theta)^2

(xi)       2*sin(Theta)^2 = s^2/(2*k^2)

(xii)      cos(2Theta) = 1-s^2/(2*k^2) 
           sin(2Theta) = sqrt( 2*s^2/(2*k^2) - (s^2/(2*k^2))^2 )  

Eq. ix and xii allow the calculation of 2Theta and alpha from the saxs-
coordinates (sp_1, sp_2) of the projection. kout can then be calculated with 
eq. vii. The saxs-coordinates (sc_1,sc_2) on the inclined detector are given by:

                                       ( sin(2Theta)*cos(alpha) )   ( sc^_1 )
(xiii)     sc^ = InvA * kout^ = InvA * | sin(2Theta)*sin(alpha) | = | sc^_2 |
                                       (     -cos(2Theta)       )   ( sc^_3 )

(xiv)            ( k * sc^_1/sc^_3 )   ( sc_1 )
           sc~ = | k * sc^_2/sc^_3 | = | sc_2 |
                 (      -k         )   ( sc_3 )

*) SAXS-Coordinates

     sc_1 =  k * ((x_1+off_1) - cen_1) * (pix_1/dis)    
     sc_2 =  k * ((x_2+off_2) - cen_2) * (pix_2/dis)

where x_1, x_2 are the pixel coordinates, off_1,off_2, the offsets, 
cen_1,cen_2 the point of normal incidence ("poni", "center"), pix_1,pix_2, 
the pixel sizes, dis the distance between the sample and the point of 
normal incidence and k the wavenumber (1/wavelength).
For small scattering angles (sqrt(sc_1^2+sc_2^2) << k) (sc_1,sc_2,-k) 
approximates the scattering vector.

The detector orientation is defined by three sequential ccw-rotations around
axis 1 (rot1), axis 2 (rot2) and axis 3 (rot3).
 
rotation around
 
 axis 1:
                 |         1.0         0.0         0.0 |
       ROT_1 =   |         0.0  cos(angle) -sin(angle) |
                 |         0.0  sin(angle)  cos(angle) |
 
 axis 2:
                 |  cos(angle)         0.0  sin(angle) |
       ROT_2 =   |        0.0          1.0         0.0 |
                 | -sin(angle)         0.0  cos(angle) |
 
 axis 3:
                 |  cos(angle) -sin(angle)         0.0 |
       ROT_3 =   |  sin(angle)  cos(angle)         0.0 |
                 |         0.0         0.0         1.0 |                                                                                      

----------------------------------------------------------------------------*/
/******************************************************************************
* Include Files                                                               *
******************************************************************************/

# include "waxs.h"

/******************************************************************************
* Private Constants                                                           *
******************************************************************************/

# define R_PI 3.1415926535897932384626

static const double deg2rad = R_PI/180.0;
static const double rad2deg = 180.0/R_PI;
static const double pi      = R_PI;
static const double halfpi  = R_PI*0.5;
static const double twopi   = R_PI*2.0;
static const double one     = 1.0;
static const double eps     = 1e-30;

/******************************************************************************
* Private Type Defs                                                           *
******************************************************************************/

typedef struct waxs_params {
  int    Init;
  double Rot[3][3];    // rotation matrix
  double InvRot[3][3]; // inverse rotation matrix
  double k;            // absolute value of k-vector
  double halfdk2;      // 1/k^2

} WParams;

typedef struct waxs_dir {
  int    status;
  double sinTwoTheta;
  double cosTwoTheta;
  double sinAlpha;
  double cosAlpha;
  } WaxsDir;

/******************************************************************************
* Private Variables                                                           *
******************************************************************************/

static WParams WaxsParams;

/******************************************************************************
* Routines                                                                    *
******************************************************************************/

void _fprint_mat ( FILE * out, double A[3][3] ) 
{ int i,j;
  for (j=0;j<3;j++) {
    for (i=0;i<3;i++) {
      fprintf(out," %15.3f", A[i][j]);
      }
    fprintf(out,"\n");
    }
} // _fprint_mat

void _fprint_vec ( FILE * out, double V[3] )
{ int i;
  for (i=0;i<3;i++)
    fprintf(out," %15g\n", V[i] );
} // _fprint_vec                                                                                                                              

void _fprint_dir ( FILE * out, WaxsDir Beam )
{
  fprintf(out," sinAlpha           = %g\n", Beam.sinAlpha );
  fprintf(out," cosAlpha           = %g   (%g deg)\n",
          Beam.cosAlpha, atan2(Beam.sinAlpha,Beam.cosAlpha)*rad2deg );
  fprintf(out," sinTwoTheta        = %g\n", Beam.sinTwoTheta );
  fprintf(out," cosTwoTheta        = %g   (%g deg)\n",
          Beam.cosTwoTheta, atan2(Beam.sinTwoTheta,Beam.cosTwoTheta)*rad2deg );
} // _fprint_dir                                                                                                                              

void waxs_PrintParams ( FILE * out )
{ WParams * pParams = &WaxsParams;
  if (!pParams->Init) return;
  fprintf(out," Init                 = %d\n", pParams->Init); 
  _fprint_mat ( out, pParams->Rot );
  _fprint_mat ( out, pParams->InvRot );
  fprintf(out," k                    = %g\n", pParams->k);   
  fprintf(out," halfdk2              = %g\n", pParams->halfdk2);
} //  waxs_PrintParams 

void waxs_PrintCoord ( FILE * out, WaxsCoord sp )
{ WaxsCoord *pCoord = &sp;
  fprintf(out," status               = %d\n", pCoord->status);
  fprintf(out," s_1                  = %g\n", pCoord->s_1);
  fprintf(out," s_2                  = %g\n", pCoord->s_2);
} // waxs_PrintCoord 

/*+++------------------------------------------------------------------------
NAME
  _beam_set --- set WaxsDir 

SYNOPSIS

  WaxsDir _beam_set( WaxsDir *pbeam, int status )

DESCRIPTION
 
  Changes error status in WaxsDir to status.
  Other parameters are not changed.

RETURN VALUE

  WaxsDir *pBeam with error status

----------------------------------------------------------------------------*/
WaxsDir _beam_set( WaxsDir *pbeam, int status )
{ 
  pbeam->status = status;
  return( *pbeam );

} // _beam_set 

/*+++------------------------------------------------------------------------
NAME
  _s_set --- set error status in WaxsCoord 

SYNOPSIS

  WaxsCoord _s_set( WaxsCoord *ps, int status )

DESCRIPTION

  Changes error status in WaxsCoord to status.
  Other parameters are not changed.

RETURN VALUE

  WaxsCoord *ps with error status

----------------------------------------------------------------------------*/
WaxsCoord _s_set( WaxsCoord *ps, int status )
{
  ps->status = status;
  return( *ps );

} // _s_set

/*+++------------------------------------------------------------------------
NAME
  sp2kdir --- calculates the angles of kout 

SYNOPSIS

  WaxsDir sp2kdir ( WaxsCoord sp )

DESCRIPTION

  Calculates the unit vector in lab coordinates of the scattered beam from 
the saxs-coordinates (sp_1, sp_2) of its Ewald-sphere projection.

RETURN VALUE
  
  .status==0 : sinTwoTheta, cosTwoTheta, sinAlpha, cosAlpha angles in rad
               (external angles)
  .status!=0 : error 

----------------------------------------------------------------------------*/
WaxsDir sp2kdir ( WaxsCoord sp )
{ 
  WaxsDir Beam;
  WParams * pParams = &WaxsParams;

  double s, s2, s2d2k2;
  double tmp;

  // pParams initialized
  if (!pParams->Init) return(_beam_set(&Beam,-1));

  s2     = sp.s_1*sp.s_1+sp.s_2*sp.s_2;
  s      = sqrt(s2);
  s2d2k2 = s2*pParams->halfdk2;

  Beam.cosTwoTheta = 1.0 - s2d2k2;
  tmp = 2.0*s2d2k2-s2d2k2*s2d2k2;
  if (tmp<0.0)
    if (tmp>-eps) tmp=0.0;
      else return(_beam_set(&Beam,-2));
  Beam.sinTwoTheta = sqrt(tmp);

  if (s>eps) {
   Beam.cosAlpha = sp.s_1/s;
   Beam.sinAlpha = sp.s_2/s;
   } else {
   Beam.cosAlpha = 0.0;
   Beam.sinAlpha = 0.0;
   }

  return( _beam_set( &Beam, 0 ) );

} // sp2kdir

/*+++------------------------------------------------------------------------
NAME
  rotation_matrix_3 --- calculates the 3-dimensional rotation matrix 
 
SYNOPSIS
 
  void rotation_matrix_3 ( double Rot[3][3], int axis, double angle )
 
DESCRIPTION
 
  Calculates the 3-dimensional rotation matrix for a ccw rotation of 
  angle degrees around axis (axis = 1 | 2 | 3, double Rot[3][3]).
 
RETURN VALUE
  none
 
----------------------------------------------------------------------------*/
void rotation_matrix_3 ( double Rot[3][3], int axis, double angle )
{ 
  switch (axis) {
    case 1:  
      Rot[0][0] =  1.0;        Rot[1][0] =  0.0;        Rot[2][0] =  0.0;
      Rot[0][1] =  0.0;        Rot[1][1] =  cos(angle); Rot[2][1] = -sin(angle);
      Rot[0][2] =  0.0;        Rot[1][2] =  sin(angle); Rot[2][2] =  cos(angle);
      break; 
    case 2: 
      Rot[0][0] =  cos(angle); Rot[1][0] =  0.0;        Rot[2][0] =  sin(angle);
      Rot[0][1] =  0.0;        Rot[1][1] =  1.0;        Rot[2][1] =  0.0;
      Rot[0][2] = -sin(angle); Rot[1][2] =  0.0;        Rot[2][2] =  cos(angle);
      break; 
    case 3: 
      Rot[0][0] =  cos(angle); Rot[1][0] = -sin(angle); Rot[2][0] =  0.0;
      Rot[0][1] =  sin(angle); Rot[1][1] =  cos(angle); Rot[2][1] =  0.0;
      Rot[0][2] =  0.0;        Rot[1][2] =  0.0;        Rot[2][2] =  1.0;
      break;
    default:
      printf("ERROR in rotation_matrix_3: axis = %d, 1<=axis<=3 required.\n",
         axis);
      exit(-1);
    }

  return;

} // rotation_matrix_3 

/*+++------------------------------------------------------------------------
NAME
  mat_mul_3 --- product of two 3-dimensional matrices
 
SYNOPSIS
 
  void mat_mul_3 ( double Out[3][3], double A[3][3], double B[3][3] )
 
DESCRIPTION
 
  Out[3][3] = A[3][3]*B[3][3]
 
RETURN VALUE
  none
 
----------------------------------------------------------------------------*/
void mat_mul_3 ( double Out[3][3], double A[3][3], double B[3][3] )
{ int i,j,k;
  for (j=0;j<3;j++)
    for (k=0;k<3;k++) {
      Out[j][k] = 0.0;
      for (i=0;i<3;i++) 
        Out[j][k] += A[i][k] * B[j][i];
      }
  return;
} // mat_mul_3

/*+++------------------------------------------------------------------------
NAME

  vec_mul --- multiplication of a 3x3 matrix with a 3d vector
 
SYNOPSIS
 
  void vec_mul ( double VOut[3], double A[3][3], double V[3] )
 
DESCRIPTION
 
  VOut[3] = A[3][3]*V[3]
 
RETURN VALUE
  none
 
----------------------------------------------------------------------------*/
void vec_mul ( double VOut[3], double A[3][3], double V[3] )
{ int i,j;
  for (j=0;j<3;j++) {
    VOut[j] = 0.0;
    for (i=0;i<3;i++)
      VOut[j] += A[i][j] * V[i];
    }
  return;
} // vec_mul

/*+++------------------------------------------------------------------------
NAME
  waxs_Init --- Initialisation of parameters

SYNOPSIS

  int waxs_Init ( double k, double rot_1, double rot_2, double rot_3 )

DESCRIPTION
It initializes all static parameters.

ARGUMENTS
k     : wavenumber                                                                                                                  
rot_1 : ccw rotation around axis 1
rot_2 : ccw rotation around axis 2
rot_3 : ccw rotation around axis 3

RETURN VALUE
  returns 0 if OK

----------------------------------------------------------------------------*/
int waxs_Init ( double k, double rot_1, double rot_2, double rot_3 )
{ WParams * pParams = &WaxsParams;

  double Rot_1[3][3], Rot_2[3][3], Rot_3[3][3];
  double tmp[3][3];

  pParams->Init   = 0;

  // rotation matrix
  rotation_matrix_3 ( Rot_1, 1, rot_1 );
  rotation_matrix_3 ( Rot_2, 2, rot_2 );
  rotation_matrix_3 ( Rot_3, 3, rot_3 );

  mat_mul_3 ( tmp, Rot_2, Rot_1 );
  mat_mul_3 ( pParams->Rot, Rot_3, tmp );

  // inverse rotation matrix
  rotation_matrix_3 ( Rot_1, 1, -rot_1 );
  rotation_matrix_3 ( Rot_2, 2, -rot_2 );
  rotation_matrix_3 ( Rot_3, 3, -rot_3 );

  mat_mul_3 ( tmp, Rot_2, Rot_3 );
  mat_mul_3 ( pParams->InvRot, Rot_1, tmp );

  // wavevector k
  pParams->k = k;
  pParams->halfdk2 = 0.5/(k*k);

  pParams->Init = 1;

  return( 0 );

} // waxs_Init

/*+++------------------------------------------------------------------------
NAME

  waxs_Saxs --- calculation of saxs coordinate from s-projection 

SYNOPSIS

  WaxsCoord waxs_Saxs ( WaxsCoord sp )

DESCRIPTION

  Calculates the saxs-coordinate s of the inclined detector image 
  from the saxs-coordinate sp of the Ewald sphere-projection.

ARGUMENT

  WaxsCoord sp : saxs-coordinate of the Ewald sphere projection

RETURN VALUE
  WaxsCoord s  : saxs-coordinate of the inclined detector image
      s.status!=0 in case of an error

----------------------------------------------------------------------------*/
WaxsCoord waxs_Saxs ( WaxsCoord sp ) 
{ WParams * pParams = &WaxsParams;
  WaxsCoord sout;
  WaxsDir kdir;
  double kvec[3];
  double kvecout[3];

  // pParams initialized
  if (!pParams->Init) return(_s_set( &sout,-1));
  
  kdir = sp2kdir ( sp );
  if (kdir.status) return( _s_set( &sout,kdir.status*10-2) );

  kvec[0] =  kdir.sinTwoTheta*kdir.cosAlpha; 
  kvec[1] =  kdir.sinTwoTheta*kdir.sinAlpha;
  kvec[2] = -kdir.cosTwoTheta;

  vec_mul ( kvecout, WaxsParams.InvRot, kvec );

  // no solution for positive kvecout[2]
  if (kvecout[2]>-eps) return(_s_set( &sout,-3));

  sout.s_1 = -(kvecout[0]/kvecout[2])*WaxsParams.k;
  sout.s_2 = -(kvecout[1]/kvecout[2])*WaxsParams.k; 
  
  return(_s_set( &sout,0));

} // waxs_Saxs
