/****************************************************************************

 File:      	GetGelFrame.c

 Description:	This is a sub-program to extract given informatione from a 
                .gel file.

	        The .gel files are created by ImageQuant detectors. 
                They are binary, using 16 bits to store each value 
                (2 bytes/pixel). The data is in the
		intel byte order (LO byte | HI byte). ImageQuant files have 
                a Tag Image File Format (width, length, file, size...). 
                Ten private TIFF fields are reserved for use with the 
                ImageQuant data files.
		
 Author(s): 	Peter J. Daly (GetFrame.c)
 Modified:	17-Feb-1995 Peter Boesecke (change of names)
                19-Feb-1995 (expanded)
                10-Apr-1995 Pierre Jourde, adaptation for .gel
			    image files and change of names
			    

 Original:   	February 1995

****************************************************************************/

/***************************************************************************
 * Include Files                                                           *
 ***************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include "GetGelFrame.h"       

/***************************************************************************
 * Defines                                                                 *
 ***************************************************************************/

#define ATOI(s) (int) strtol (s, (char **) NULL, 10)

#define PRIVATE static        /* used to declare variables of private type */
#define PUBLIC                /* used to declare variables of public type */

#ifndef True
# define True 1
#endif
#ifndef False
# define False 0
#endif

/***************************************************************************
 * Global Initialised Variables                                            *
 ***************************************************************************/

PRIVATE int InFileDataLen = 0, Xsize = 0, Ysize = 0;
PRIVATE int PictureStart,CompresCode,FactorDenom,FactorNumer;
PRIVATE int XPositionn,XPositiond,YPositionn,YPositiond;
PRIVATE int HeaderRead = False, OutFileDataLen = 0;

int pointtoint(buffer) unsigned char  *buffer;
  { return buffer[0] + 256 * buffer[1]; }

long int pointtolong(buffer) unsigned char  *buffer;
  { return buffer[0] + 256 * (buffer[1] + 256 * (buffer[2] + 256 * buffer[3])); }


/***************************************************************************

 @(#) $Header: parse header in the SPEC generated file                     $

 ***************************************************************************/

PRIVATE int ParseHeaderFile (FILE *stream)
{
  PUBLIC unsigned char WorkBuffer[256];
  PUBLIC long int NextDirOff, i;
  PUBLIC int infonumber, entries;
  PUBLIC char info[256][12];
 
  if ( HeaderRead ) return 0;                   /* header already analysed */

  if (stream == (FILE *) NULL) return 1;                   /* no file open */

  if (fseek (stream, 0L, SEEK_SET)) return 2;          /* can't seek start */

  /* reading of the file header */
  /* and saving data in the info array*/
  fread(WorkBuffer, 8, 1, stream);
  NextDirOff = pointtolong(&WorkBuffer[4]);
  infonumber = 0;
  do {
    fseek(stream , NextDirOff, 0);
    fread(WorkBuffer, 2, 1, stream);
    entries = pointtoint(&WorkBuffer[0]);
    for (i = 0; i < entries; i++) {
      fread(info[infonumber], 12, 1, stream);
      infonumber++;
      }
    fread(WorkBuffer, 4, 1, stream);
    NextDirOff = pointtolong(&WorkBuffer[0]);
    } while (NextDirOff != 0);

  /* analyse of the header from the info array */
  for (i = 0; i < infonumber; i++) {
    switch (pointtoint(info[i])) {
      case 256:
        if (pointtoint(&info[i][2]) == 3)
          Xsize = pointtoint(&info[i][8]);
        else
          Xsize = pointtolong(&info[i][8]);
        break;

      case 257:
        if (pointtoint(&info[i][2]) == 3)
          Ysize = pointtoint(&info[i][8]);
        else
          Ysize = pointtolong(&info[i][8]);
        break;

      case 273:
        PictureStart = pointtolong(&info[i][8]);
        break;

      case 286:
        fseek(stream, pointtolong(&info[i][8]), 0);
        fread(WorkBuffer, 4, 1, stream);
        XPositionn = pointtolong(WorkBuffer);
        fread(WorkBuffer, 4, 1, stream);
        XPositiond = pointtolong(WorkBuffer);
        break;

      case 287:
        fseek(stream, pointtolong(&info[i][8]), 0);
        fread(WorkBuffer, 4, 1, stream);
        YPositionn = pointtolong(WorkBuffer);
        fread(WorkBuffer, 4, 1, stream);
        YPositiond = pointtolong(WorkBuffer);
        break;

      case 33445:
        CompresCode = pointtolong(&info[i][8]);
        break;

      case 33446:
        fseek(stream, pointtolong(&info[i][8]), 0);
        fread(WorkBuffer, 4, 1, stream);
        FactorDenom = pointtolong(WorkBuffer);
        fread(WorkBuffer, 4, 1, stream);
        FactorNumer = pointtolong(WorkBuffer);
        break;

      default:
        break;
      }
    }

  InFileDataLen = 2;

  if (!((CompresCode == 2)||(CompresCode == 0))) {
    printf("Compresscode is not implemented in this program, ");
    printf("therefore program is halted!! \n");
    return 3;
    }
  OutFileDataLen = 4; /* length of the data after decompression */
                      /* like they will appear in the esrf file */
  HeaderRead = True;  /* indicates that the file header has already
			 been read, so the data are available */
  return 0;
}

/***************************************************************************

 @(#) $Header: Read the image and set the pointer                          $
 @(#) $Header: to point at the read data                                   $

 ***************************************************************************/

PUBLIC int ReadGelData (FILE *stream, int FrameNumber, void **Start)
{
  int Read = 0, BufferSize = 0, BlockSize = 0;
  float *FltImage;
  unsigned char *Image;
  long int WorkData, i, j;
 
  if (stream == (FILE *) NULL) return 1;                   /* no file open */

  if (fseek (stream, PictureStart, SEEK_SET)) return 2;/* can't seek start */

  BlockSize = Xsize * Ysize;
  BufferSize = InFileDataLen * Xsize * Ysize;
  if (BufferSize <= 0) return 3;                               /* bad size */

  if ((Image = (void *) malloc (BufferSize)) == (void *) NULL)
    return 4;                                              /* can't malloc */

  if (FrameNumber > 1 &&
      fseek (stream, (long) ((FrameNumber - 1) * BufferSize), SEEK_CUR))
    return 2;                                                /* can't seek */

  Read = fread (Image, 1, BufferSize, stream);
  if (Read != BufferSize)  return 5;                           /* bad read */

  /* creating the float image */
  if((FltImage=(float *)malloc(OutFileDataLen*BlockSize))==NULL)
    {
    printf("Not enough memory for flt-image. Program stopped.\n");
    return 4;
    }

  /* decompressing image into a float image */

  for (i=0; i< BlockSize; i++ ) {
    j=i*2;
    WorkData = Image[j] + 256 * Image[j+1];
    if ( CompresCode==0 ) 
      FltImage[i]=(float)WorkData; 
    else
      FltImage[i]=1.0 * WorkData * WorkData * FactorDenom / FactorNumer;
    }

  /* free allocated memory */
  free( Image );   
 
  /* return the pointer to the image data */ 
  *Start = FltImage;

   return 0;
}

/***************************************************************************

 @(#) $Header: return the x and y dimensions of the image ,                $
 @(#) $Header: the x and y offset and the number                           $
 @(#) $Header: of bytes for each pixel (2 or 4)                            $

 ***************************************************************************/

PUBLIC int ReadGelType (FILE *stream, int *pX, int *pY, int *pOutFileDataLen,
			int *pDataFloatI, int * pOffsetX, int * pOffsetY)
{
  if (! ParseHeaderFile (stream)) {
    *pX = Xsize;
    *pY = Ysize;
    *pOffsetX = 0;
    *pOffsetY = 0;
    *pOutFileDataLen = OutFileDataLen;/* after decompression, not 2 (before) */
    *pDataFloatI = True;              /* Data Float Indicator */ 
    return 0;
  } else {
    return 1;
  }
}
/***************************************************************************

 @(#) $Header: return various texts about the setup (as char **)           $
 @(#) $Header: DetectorType, MachineInfo, OpticsInfo, StationInfo,         $ 
 @(#) $Header: FramingInfo                                                 $
 @(#) $Header: allocated memory must be released                           $

 ***************************************************************************/

int ReadGelInfo(FILE * stream,
         char ** DetectorType,
         char ** MachineInfo,
         char ** OpticsInfo,
         char ** StationInfo,
         char ** ProposalInfo)
# define DETECTORTYPE "image plate detector"
{   
   *DetectorType = (char *) malloc(strlen(DETECTORTYPE)+1); 
     if (*DetectorType != (char *) NULL) strcpy(*DetectorType,DETECTORTYPE);
   *MachineInfo  = (char *) NULL; 
   *OpticsInfo   = (char *) NULL;
   *StationInfo  = (char *) NULL;
   *ProposalInfo = (char *) NULL;

   return(0);
}
 /***************************************************************************

 @(#) $Header: return various numbers about the experimental setup:        $
 @(#) $Header: char * Title (title given by the user)                      $
 @(#) $Header: Center_1, Center_2 (position of beam [pixel values]         $
 @(#) $Header: PixSize_1, PixSize_2 (size of of a pixels) [meters]         $
 @(#) $Header: WaveLength (current wavelength) [meters]                    $
 @(#) $Header: SampleDistance (distance from sample to center) [meters]    $ 

 HmStartTime is the time where the the acquisition has been started.
 The actual starting time of frame n is in the simplest case:
     HmStartTime + HmWaitTime + n * (HmWaitTime+HmFrameTime)
 Generally an acquisition can consist of a set of different HmWaitTimes and
 HmFrameTimes. The values for HmWaitTime and HmFrameTime are written into
 FramingInfo. 
 
 ***************************************************************************/

int ReadGelExperiment(FILE * stream,
         char ** Title,
         char ** ExperimentInfo,
         char ** HmFraming,
         char  * HmStartTime,
         float * Center_1, float * Center_2, int * ICenter,
         float * PixSize_1, float * PixSize_2, int * IPixSize,
         float * WaveLength, int * IWaveLength,
         float * SampleDistance, int * ISampleDistance)
{
  const char *TIMEEXAMPLE = "00-Jan-0000 00:00:00.000"; /* example */

  *Title = (char *) NULL;
  *ExperimentInfo = (char *) NULL;
  *HmFraming = (char *) NULL;
/*  *HmStartTime = (char *) malloc(strlen(TIMEEXAMPLE)+1);
    if (*HmStartTime != (char *) NULL) */
  strcpy(HmStartTime,TIMEEXAMPLE);

  *Center_1 = 0; *Center_2 = 0; *ICenter = False;
  *PixSize_1 = 1.0; *PixSize_2 = 1.0; *IPixSize = False;
  *WaveLength = 1e-10; *IWaveLength = False;
  *SampleDistance = 0.0; *ISampleDistance = False;

  return(0);
}

/***************************************************************************

 @(#) $Header: Get the scalar data for a particular channel number         $

 ***************************************************************************/

#ifdef INCLUDE_MAIN
main (int argc, char *argv[])
{
  void *Start;
  FILE *in, *inhm;
  int j, X, Y, OX, OY, Data, Max, Frame, DataFloat;
  char  Name[512];
  char *Number;

  if (argc>1) (void) strcpy(Name,argv[1]); else (void) strcpy(Name,"gasdet001");
  if (argc>2) Number=argv[2]; else Number="1";

  if ((in = fopen (Name, "r")) != (FILE *) NULL) {
    Frame = ATOI (Number);
    Max=1;
    (void) printf ("Returned Maximum = %d\n",Max);
    j = ReadGelType (in, &X, &Y, &Data, &DataFloat, &OX, &OY);
    (void) printf ("Returned %d X = %d Y = %d Data = %d FloatValue = %d\n",
		   j, X, Y, Data, DataFloat);
    (void) strcat(Name,"hm");
    if ((inhm = fopen (Name, "r")) != (FILE *) NULL) {

       j = ReadGelData(inhm, Max, &Start); 

       printf("\n");
       (void) printf ("Returned %d ReadGelData ", j);
       free(Start);
       }

    (void) printf ("\n");
  }
}
#endif
