"""
    EdfFileData.py
    Data derived class to access Edf files    
"""

from PyDVT import __version__,__date__,__author__


################################################################################  
from Data import *
import EdfFile
################################################################################

SOURCE_TYPE = "EdfFile"


class EdfFileData(Data):
    """
    Specializes Data class in order to access Edf files.    
    Interface: Data class interface.
    """
    def __init__(self,refresh_interval=None,info={}):
        """
        See Data.__init__
        """
        self.EdfObj=None
        info["Class"]="EdfFileData"        
        Data.__init__(self,refresh_interval,info)


    def SetSource (self,source_name=None):
        """
        Sets a new source for data retrieving, an edf file.
        If the file exists, self.Source will be the EdfFile
        object associated to this file.
        Parameters:
        source_name: name of the edf file 
        """
        if source_name==self.SourceName: return 1
        if (source_name != None): 
            try:
                self.Source = EdfFile.EdfFile(source_name)
            except:
                self.SourceName=None
                self.Source=None
                return 0
        else:
            self.Source=None
        self.SourceName=source_name
        return 1


    def GetSourceInfo (self):
        """
        Returns information about the EdfFile object created by
        SetSource, to give application possibility to know about
        it before loading.
        Returns a dictionary with the keys "Size" (number of possible
        keys to this source) and "KeyList" (list of all available keys
        in this source). Each element in "KeyList" is an integer
        meaning the index of the array in the file.
        """        
        if self.SourceName == None: return None
        NumImages=self.Source.GetNumImages()
        source_info={}
        source_info["Size"]=NumImages
        source_info["KeyList"]=range(NumImages)
        return source_info        

        
    
    def LoadSource(self,key_list="ALL",append=0,invalidate=1,pos=None,size=None):
        """
        Creates a given number of pages, getting data from the actual
        source (set by SetSource)
        Parameters:
        key_list: list of all keys to be read from source. It is a list of
                 integers, meaning the indexes to be read from the file.
                 It can be also one integer, if only one array is to be read.
        append: If non-zero appends to the end of page list.
                Otherwise, initializes the page list                
        invalidate: if non-zero performas an invalidade call after
                    loading
        pos and size: (x), (x,y) or (x,y,z) tuples defining a roi
                      If not defined, takes full array
                      Stored in page's info
        """
        if append==0: Data.Delete(self)
        numimages=self.Source.GetNumImages()
        if key_list == "ALL": key_list=range(numimages)
        elif type(key_list) is types.IntType: key_list=[key_list]

        if pos is not None:
            edf_pos=list(pos)
            for i in range(len(edf_pos)):
                if edf_pos[i]=="ALL":edf_pos[i]=0
        else: edf_pos=None
            
        if size is not None:
            edf_size=list(size)
            for i in range(len(edf_size)):
                if edf_size[i]=="ALL":edf_size[i]=0
        else: edf_size=None


        for key in key_list:
            i=int(key)
            if int(i) >= numimages: raise "EdfFileData: index out of bounds"
            info={}
            info["SourceType"]=SOURCE_TYPE
            info["SourceName"]=self.SourceName
            info["Key"]=i
            info["Source"]=self.Source

            info["pos"]=pos
            info["size"]=size

            info.update(self.Source.GetStaticHeader(i))
            info.update(self.Source.GetHeader(i))
            if info["DataType"]=="UnsignedShort":array=self.Source.GetData(i,"SignedLong",Pos=edf_pos,Size=edf_size)
            elif info["DataType"]=="UnsignedLong":array=self.Source.GetData(i,"DoubleValue",Pos=edf_pos,Size=edf_size)
            else: array=self.Source.GetData(i,Pos=edf_pos,Size=edf_size)
            self.AppendPage(info,array)        
        if invalidate: self.Invalidate()



################################################################################
#EXEMPLE CODE:
    
if __name__ == "__main__":
    import sys,time
    try:
        Filename=sys.argv[1]
        key=sys.argv[2]    
        obj=EdfFileData()
        obj.SetSource(Filename)
        obj.LoadSource(key)
    except:
        print "Usage: EdfFileData.py <filename> <image>"
        sys.exit()
    
    for i in range (obj.GetNumberPages()):
        print obj.GetPageInfo(i)
        print obj.GetPageArrayRegion((0,0),(2,2),i)
    

        