/****************************************************************************

                     uMatrix C++ Matrix Library

    Copyright (C) 1996  David Weber, Michael Sipe and Rajesh Shenoy

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    David Weber can be contacted at weber@ece.cmu.edu or 
    http://www.ece.cmu.edu/afs/ece/usr/weber/.home-page.html

****************************************************************************/

#include <stdlib.h>
#include <fftpack++.h>

/*
   FFT class interface to FFTPACK.

   Author: Dave Weber

   Revision Status:
   0.0: 1d FFT's implemented
   0.1: 2d FFT's implemented
   0.2: Real 2d FFT's implemented. NOTE: This is numerically poorly conditioned
   0.3: We finally implemented double precision FFT's. What a pain.   

   $Id: fftpack++.cc,v 1.1.1.1 2001/06/27 17:07:53 mirone Exp $
   $Author: mirone $
   $Date: 2001/06/27 17:07:53 $
   $Source: /scisoft/users/mirone/SOURCESCVS//nMatrix/fftpack++.cc,v $
   $Revision: 1.1.1.1 $
   $Log: fftpack++.cc,v $
   Revision 1.1.1.1  2001/06/27 17:07:53  mirone
   nMatrix now on cvs


// Revision 1.5  1996/06/17  15:36:04  weber
// Added Gnu library public license header
//
// Revision 1.4  1996/04/18  15:26:45  weber
// Fixes to allow libg++ 2.7.x with the standard template library complex
// class to be used if it is present. The code remains compatible with
// previous versions of libg++
//
   Revision 1.3  1996/04/17 17:07:46  weber
   Removed declarations of loop counters so that the code compiles
   independently of the -no-for-scope flag or gcc version

// Revision 1.2  96/02/02  14:49:43  weber
// Eliminated a warning about signed/unsigned integer comparisons
// 
// Revision 1.1  1995/11/08  18:52:17  weber
// Initial revision
//


*/

fComplexFFT1d::fComplexFFT1d( const int Size ) // Constructor
{
   N = Size;
   fftStorage = new float[4*N+15];
   cffti_( &N, fftStorage );   // Call fftpack initializer
}

void fComplexFFT1d::forward( uVector<fComplex>& data )
{
   if ( N != (int)data.size() ) // No range errors possible if false
   {
      cout << "uVector size is not equal to fft size in "
              "ComplexFFT1d::forward()\n";
   }
   cfftf_(&N, (float *)(data.address()), fftStorage ); // I know, Its ugly.
}

void fComplexFFT1d::inverse( uVector<fComplex>& data )
{
   if ( N != (int)data.size() ) // No range errors possible if false
   {
      cout << "uVector size is not equal to fft size in "
              "ComplexFFT1d::inverse()\n";
   }
   cfftb_(&N, (float *)(data.address()), fftStorage );
   for ( int i = 0; i < N; i++ ) // fftpack does not normalize the data
   {
      data[i] /= (float) N; // use unchecked index for speed.
   }
}

// 2d FFT stuff


fComplexFFT2d::fComplexFFT2d( const int SizeN, const int SizeM ) 
{
   N = SizeN; // No of rows
   M = SizeM; // No of columns

   fftStorageN = new float[4*N+15];
   fftStorageM = new float[4*M+15];
   cffti_( &N, fftStorageN );   // Call fftpack initializers
   cffti_( &M, fftStorageM ); 
}

void fComplexFFT2d::forward( uMatrix<fComplex>& data )
{
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "fComplexFFT2d::forward()\n";
   }

   for ( int n = 0; n < N; n++ ) // go down rows first 
   {   
      cfftf_(&M, (float *)(&data(n,0)), fftStorageM ); 
   }

   for ( int m = 0; m < M; m++ ) // go down columns second
   {
      // gather column into temporary vector
      uIndex I(0,1,N-1);
      uIndex J(m,1,m);
      uMatrix<fComplex> temp = data( I, J );
      // perform fft
      cfftf_(&N, (float *)(&temp(0,0)), fftStorageN ); 
      // scatter temporary vector back into column
      data.insert( I, J, temp );
   }
}

void fComplexFFT2d::inverse( uMatrix<fComplex>& data )
{
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "fComplexFFT2d::forward()\n";
   }

   int n;
   for ( n = 0; n < N; n++ ) // go down rows first 
   {   
      cfftb_(&M, (float *)(&data(n,0)), fftStorageM ); 
   }

   for ( int m = 0; m < M; m++ ) // do columns second
   {   
      // gather column into temporary vector
      uIndex I(0,1,N-1);
      uIndex J(m,1,m);
      uMatrix<fComplex> temp = data( I, J );
      // perform inverse fft
      cfftb_(&N, (float *)(&temp(0,0)), fftStorageN ); 
      // scatter temporary vector back into column
      data.insert( I, J, temp );
   }

   for ( n = 0; n < N; n++ ) // fftpack does not normalize the data
   {
      for ( int m = 0; m < M; m++ ) 
      {
	 data(n,m) /= (float) N*M; 
      }  
   }
}

// real 2d FFT stuff

RealFFT2d::RealFFT2d( const int SizeN, const int SizeM ) 
{
   N = SizeN; // No of rows
   M = SizeM; // No of columns

   fftStorageN = new float[4*N+15];
   fftStorageM = new float[4*M+15];
   cffti_( &N, fftStorageN );   // Call fftpack initializers
   cffti_( &M, fftStorageM ); 
}

/* 
   This method introduces some serious numerical errors due to round off
   in the fft pass through the rows. It is probable that this will
   only work with double precision complex types.

   All in all, this is not very efficient because the underlying
   uMatrix classes are not too efficient when manipulating the data
   the way that it does. Not much can be done about this until we
   get some form of profiling for linux.

   A speedup of 25% is achieved at a serious numerical cost.
*/

void RealFFT2d::forward( uMatrix<fComplex>& data )
{
   uVector<fComplex> Mtemp(M);
   int Mterm;
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "RealFFT2d::forward()\n";
   }
   int n;
   for ( n = 0; n < N - 1; n += 2 ) // go down rows first 
   {
      int m;
      // Pack two real vectors into one
      for ( m = 0; m < M; m++)
      {
	 Mtemp(m) = fComplex( data(n,m).real(), data(n+1,m).real() );
      }
      cfftf_(&M, (float *)(&Mtemp(0)), fftStorageM );
      data(n,0) = Mtemp(0).real();
      data(n+1,0) = Mtemp(0).imag();
      for ( m = 1; m < M; m++) // unravel data ** ROUND OFF ERRORS HERE
      {
	 data(n,m)   = ( Mtemp(m) + conj( Mtemp(M-m) ) ) * 0.5; 
	 data(n+1,m) = ( Mtemp(m) - conj( Mtemp(M-m) ) ) 
                        * fComplex(0,-0.5);  // make this even.
      }    
   }
   if ( ( N & 1 ) != 0 ) // uMatrix has an odd number of rows
   {
      cfftf_(&M, (float *)(&data(N-1,0)), fftStorageM ); // fft of odd line
   }

   data.transpose(); // This is not terribly efficient if M != N  
   if ( ( M & 1 ) == 0 ) // check if M is even or not
   {
      Mterm = M / 2;  // M even
   }
   else
   {
      Mterm = M / 2 + 1;
   }
 
   int m;
   for ( m = 0; m < M / 2 + 1; m++ ) // do rows 
   {   
      cfftf_(&N, (float *)(&data(m,0)), fftStorageN ); 
   }

   // Swap it all back again
   data.transpose();

   for ( n = 1; n < N ; n++ ) // exploit conjugate anti-symmetry 
   {   
      for ( m = 1; m < Mterm; m++ )
      {
	 data(N - n,M - m) = conj( data(n,m) );
      }
   } 
   for ( m = 1; m < Mterm; m++ )  // Fix up DC
   {
      data(0,M - m) = conj( data(0,m) );
   }
}

void RealFFT2d::inverse( uMatrix<fComplex>& data )
{
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "RealFFT2d::forward()\n";
   }
   int n;
   for ( n = 0; n < N; n++ ) // go down rows first 
   {   
      cfftb_(&M, (float *)(&data(n,0)), fftStorageM ); 
   }
   data.transpose();
   for ( int m = 0; m < M; m++ ) // go down rows first 
   {   
      cfftb_(&N, (float *)(&data(m,0)), fftStorageN ); 
   }
   // Swap it all back again
   data.transpose();
   for ( n = 0; n < N; n++ ) // fftpack does not normalize the data
   {
      for ( int m = 0; m < M; m++ ) 
      {
	 data(n,m) /= (float) N*M; 
      }  
   }
}

void fComplexFFT2d::flip( uMatrix<fComplex>& data )
{
   fComplex temp;

   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "fComplexFFT2d::flip()\n";
   }

   for ( int i = 0; i < N/2; i++ )
   {
      for ( int j = 0; j < M/2; j++ )
      {
         temp = data(i,j);
         data(i,j) = data(i+N/2,j+M/2);
         data(i+N/2,j+M/2) = temp;
         temp = data(i,j+M/2);
         data(i,j+M/2) = data(i+N/2,j);
         data(i+N/2,j) = temp;
      }
   }
}

void RealFFT2d::flip( uMatrix<fComplex>& data )
{
   fComplex temp;
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "RealFFT2d::flip()\n";
   }

   for ( int i = 0; i < N/2; i++ )
   {
      for ( int j = 0; j < M/2; j++ )
      {
         temp = data(i,j);
         data(i,j) = data(i+N/2,j+M/2);
         data(i+N/2,j+M/2) = temp;
         temp = data(i,j+M/2);
         data(i,j+M/2) = data(i+N/2,j);
         data(i+N/2,j) = temp;
      }
   }
}

/*
   Double precision stuff
*/

ComplexFFT1d::ComplexFFT1d( const int Size ) // Constructor
{
   N = Size;
   fftStorage = new double[4*N+15];
   dcffti_( &N, fftStorage );   // Call fftpack initializer
}

void ComplexFFT1d::forward( uVector<Complex>& data )
{
   if ( N != (int) data.size() ) // No range errors possible if false
   {
      cout << "Vector size is not equal to fft size in "
              "ComplexFFT1d::forward()\n";
   }
   dcfftf_(&N, (double *)(data.address()), fftStorage ); // I know, Its ugly.
}

void ComplexFFT1d::inverse( uVector<Complex>& data )
{
   if ( N != (int) data.size() ) // No range errors possible if false
   {
      cout << "Vector size is not equal to fft size in "
              "ComplexFFT1d::inverse()\n";
   }
   dcfftb_(&N, (double *)(data.address()), fftStorage );
   for ( int i = 0; i < N; i++ ) // fftpack does not normalize the data
   {
      data[i] /= (double) N; // use unchecked index for speed.
   }
}

// 2d FFT stuff


ComplexFFT2d::ComplexFFT2d( const int SizeN, const int SizeM ) 
{
   N = SizeN; // No of rows
   M = SizeM; // No of columns

   fftStorageN = new double[4*N+15];
   fftStorageM = new double[4*M+15];
   dcffti_( &N, fftStorageN );   // Call fftpack initializers
   dcffti_( &M, fftStorageM ); 
}

void ComplexFFT2d::forward( uMatrix<Complex>& data )
{
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "ComplexFFT2d::forward()\n";
   }

   for ( int n = 0; n < N; n++ ) // go down rows first 
   {   
      dcfftf_(&M, (double *)(&data(n,0)), fftStorageM ); 
   }

   for ( int m = 0; m < M; m++ ) // go down columns second
   {   
      // gather column into temporary vector
      uIndex I(0,1,N-1);
      uIndex J(m,1,m);
      uMatrix<Complex> temp = data( I, J );
      // perform fft
      dcfftf_(&N, (double *)(&temp(0,0)), fftStorageN ); 
      // scatter temporary vector back into column
      data.insert( I, J, temp );
   }
}

void ComplexFFT2d::inverse( uMatrix<Complex>& data )
{
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "ComplexFFT2d::forward()\n";
   }

   int n;
   for ( n = 0; n < N; n++ ) // go down rows first 
   {   
      dcfftb_(&M, (double *)(&data(n,0)), fftStorageM ); 
   }

   for ( int m = 0; m < M; m++ ) // do columns second
   {   
      // gather column into temporary vector
      uIndex I(0,1,N-1);
      uIndex J(m,1,m);
      uMatrix<Complex> temp = data( I, J );
      // perform inverse fft
      dcfftb_(&N, (double *)(&temp(0,0)), fftStorageN ); 
      // scatter temporary vector back into column
      data.insert( I, J, temp );
   }

   for ( n = 0; n < N; n++ ) // fftpack does not normalize the data
   {
      for ( int m = 0; m < M; m++ ) 
      {
	 data(n,m) /= (double) N*M; 
      }  
   }
}

// real 2d FFT stuff

doubleRealFFT2d::doubleRealFFT2d( const int SizeN, const int SizeM ) 
{
   N = SizeN; // No of rows
   M = SizeM; // No of columns

   fftStorageN = new double[4*N+15];
   fftStorageM = new double[4*M+15];
   dcffti_( &N, fftStorageN );   // Call fftpack initializers
   dcffti_( &M, fftStorageM ); 
}

/* 
   This method introduces some serious numerical errors due to round off
   in the fft pass through the rows. It is probable that this will
   only work with double precision complex types.

   All in all, this is not very efficient because the underlying
   uMatrix classes are not too efficient when manipulating the data
   the way that it does. Not much can be done about this until we
   get some form of profiling for linux.

   A speedup of 25% is achieved at a serious numerical cost.
*/

void doubleRealFFT2d::forward( uMatrix<Complex>& data )
{
   uVector<Complex> Mtemp(M);
   int Mterm;
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "RealFFT2d::forward()\n";
   }

   int n;
   for ( n = 0; n < N - 1; n += 2 ) // go down rows first 
   {
      // Pack two real vectors into one
      int m;
      for ( m = 0; m < M; m++)
      {
	 Mtemp(m) = Complex( data(n,m).real(), data(n+1,m).real() );
      }
      dcfftf_(&M, (double *)(&Mtemp(0)), fftStorageM );
      data(n,0) = Mtemp(0).real();
      data(n+1,0) = Mtemp(0).imag();
      for ( m = 1; m < M; m++) // unravel data ** ROUND OFF ERRORS HERE
      {
	 data(n,m)   = ( Mtemp(m) + conj( Mtemp(M-m) ) ) * 0.5; 
	 data(n+1,m) = ( Mtemp(m) - conj( Mtemp(M-m) ) ) 
                        * Complex(0,-0.5);  // make this even.
      }    
   }
   if ( ( N & 1 ) != 0 ) // uMatrix has an odd number of rows
   {
      dcfftf_(&M, (double *)(&data(N-1,0)), fftStorageM ); // fft of odd line
   }

   data.transpose(); // This is not terribly efficient if M != N  
   if ( ( M & 1 ) == 0 ) // check if M is even or not
   {
      Mterm = M / 2;  // M even
   }
   else
   {
      Mterm = M / 2 + 1;
   }
   int m;
   for ( m = 0; m < M / 2 + 1; m++ ) // do rows 
   {   
      dcfftf_(&N, (double *)(&data(m,0)), fftStorageN ); 
   }

   // Swap it all back again
   data.transpose();

   for ( n = 1; n < N ; n++ ) // exploit conjugate anti-symmetry 
   {   
      for ( m = 1; m < Mterm; m++ )
      {
	 data(N - n,M - m) = conj( data(n,m) );
      }
   } 
   for ( m = 1; m < Mterm; m++ )  // Fix up DC
   {
      data(0,M - m) = conj( data(0,m) );
   }
}

void doubleRealFFT2d::inverse( uMatrix<Complex>& data )
{
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "RealFFT2d::forward()\n";
   }

   int n;
   for ( n = 0; n < N; n++ ) // go down rows first 
   {   
      dcfftb_(&M, (double *)(&data(n,0)), fftStorageM ); 
   }
   data.transpose();
   for ( int m = 0; m < M; m++ ) // go down rows first 
   {   
      dcfftb_(&N, (double *)(&data(m,0)), fftStorageN ); 
   }
   // Swap it all back again
   data.transpose();
   for ( n = 0; n < N; n++ ) // fftpack does not normalize the data
   {
      for ( int m = 0; m < M; m++ ) 
      {
	 data(n,m) /= (double) N*M; 
      }  
   }
}

void ComplexFFT2d::flip( uMatrix<Complex>& data )
{
   Complex temp;

   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "ComplexFFT2d::flip()\n";
   }

   for ( int i = 0; i < N/2; i++ )
   {
      for ( int j = 0; j < M/2; j++ )
      {
         temp = data(i,j);
         data(i,j) = data(i+N/2,j+M/2);
         data(i+N/2,j+M/2) = temp;
         temp = data(i,j+M/2);
         data(i,j+M/2) = data(i+N/2,j);
         data(i+N/2,j) = temp;
      }
   }
}

void doubleRealFFT2d::flip( uMatrix<Complex>& data )
{
   Complex temp;
   if ( ( data.rows() != N ) || ( data.columns() !=M ) )
   {
      cout << "Matrix size is not equal to fft size in "
              "RealFFT2d::flip()\n";
   }

   for ( int i = 0; i < N/2; i++ )
   {
      for ( int j = 0; j < M/2; j++ )
      {
         temp = data(i,j);
         data(i,j) = data(i+N/2,j+M/2);
         data(i+N/2,j+M/2) = temp;
         temp = data(i,j+M/2);
         data(i,j+M/2) = data(i+N/2,j);
         data(i+N/2,j) = temp;
      }
   }
}

/*
   Convolution routines.

   Convolution is performed using FFT's. All stuff like quadrant shifting 
   etc. is taken care off. The convolution kernel/filter/image must
   be smaller than the image itself. 

   Parameters:
   
   image :  The image to be filtered. On return, this contains the filtered
            version of the image.
   filter : The convolution kernel. This is unchanged by the operation.

*/

void convolve( uMatrix<fComplex>& image, const uMatrix<fComplex>& filter )
{
   fComplexFFT2d FFT( image.rows(), image.columns() );
   uMatrix<fComplex> temp( image.rows(), image.columns() );

   zeroPad( temp, filter );
   FFT.forward( temp );
   FFT.forward( image );
   image = image.pointMul( temp );
   FFT.inverse( image );
   FFT.flip( image );
}

void convolve( uMatrix<Complex>& image, const uMatrix<Complex>& filter )
{
   ComplexFFT2d  FFT( image.rows(), image.columns() );
   uMatrix<Complex> temp( image.rows(), image.columns() );

   zeroPad( temp, filter );
   FFT.forward( temp );
   FFT.forward( image );
   image = image.pointMul( temp );
   FFT.inverse( image );
   FFT.flip( image );
}







