! FDMNES program, Yves Joly, Oana Bunau, 3 July 2009,
!                 Institut Neel, CNRS-UJF, Grenoble, France.

! Program performing calculations of x-ray spectroscopies, XANES, RXS, 
! dichroism.
! Work using the finite difference method or the multiple scattering
! theory. Monoelectronic approach.

! Main routines of the FDMNES package.
! Needs also :
!   clemf0.f, coabs.f, convolution.f, dirac.f, fdm.f, fprime.f,  
!   general.f, lecture.f, mat.f, metric.f, minim.f, not_mpi.f,
!   potential.f, scf.f selec.f, spgroup.f, sphere.f, tab_data.f,
!   tensor.f,  
!   and the LAPACK library.

!***********************************************************************

! Declarations imported in most routines by instruction "use".

      module declarations

      implicit none

      integer, parameter:: db = selected_real_kind(15)
      integer, parameter:: sg = selected_real_kind(6)

      integer, parameter:: nlfm = 52     ! Max number of (l,m) for the selection rules
      integer, parameter:: nnlm = 34     ! Max number of orbitals
      integer, parameter:: nopsm = 64    ! Number of symmetry operation 
      integer, parameter:: nrepm = 12    ! Max number of representation 
      integer, parameter:: nslapwm = 48  ! Max number of symmetry matrix for the FLAPW data   

      complex(kind=db), parameter:: img = ( 0._db, 1._db ) 

      real(kind=db), parameter:: bohr = 0.529177249_db 
      real(kind=db), parameter:: pi = 3.1415926535897932384626433832_db 
      real(kind=db), parameter:: quatre_pi = 4 * pi 
      real(kind=db), parameter:: rydb = 13.605698_db 

      real(kind=db), parameter:: eps4 = 1.e-4_db 
      real(kind=db), parameter:: eps6 = 1.e-6_db 
      real(kind=db), parameter:: eps10 = 1.e-10_db 
      real(kind=db), parameter:: epspos = 1.e-4_db     ! precision on the atom and point positions   
        
      end module declarations

!***********************************************************************

! Main of the FDMNES program

      program fdmnes
 
      use declarations
      implicit real(kind=db) (a-h,o-z)  
      include 'mpif.h'

      character(len=1):: mot1
      character(len=8):: dat
      character(len=10):: tim
      character(len=11):: fdmfile
      character(len=50):: com_date, com_time, Revision 
      character(len=132):: fdmnes_error, identmot, mot, Space_file
      character(len=132), dimension(:), allocatable:: fdmnes_inp
       
      common/com_out/ com_date, com_time, fdmnes_error, Revision

      call MPI_Init( mpierr )
      call MPI_Comm_Size(MPI_COMM_WORLD, mpinodes, mpierr)
      call MPI_Comm_Rank(MPI_COMM_WORLD, mpirank, mpierr)

      Revision = '   FDMNES program, Revision 3 July 2009'
      fdmfile = 'fdmfile.txt'
      fdmnes_error = 'fdmnes_error.txt'
      Space_file = 'spacegroup.txt'

      call date_and_time( date = dat, time = tim )
      com_date = '   Date = ' // dat(7:8)
     &        // ' ' // dat(5:6) // ' ' // dat(1:4)
      com_time = '   Time = ' // tim(1:2)  // ' h ' // tim(3:4)
     &        // ' mn ' // tim(5:6) // ' s'

      if( mpirank == 0 ) then

        write(6,'(A/A/A)') Revision, com_date, com_time

        open(1, file = fdmfile, status='old', iostat=istat) 
        if( istat /= 0 ) call write_open_error(fdmfile,istat,1)

        do i = 1,1000
          n = nnombre(1,132)
          if( n > 0 ) exit
          read(1,*)
        end do

        read(1,*) ncalcul
        allocate( fdmnes_inp(ncalcul) )

        j = 0
        do i = 1,ncalcul
          do k = 1,100
            n = nnombre(1,132)
            read(1,'(A)',iostat=istat) mot
            if( istat /= 0 ) then
              call write_error
              do ipr = 6,9,3
                write(ipr,110) i, ncalcul
              end do
              stop
            endif
            mot1 = identmot(mot,1)
            if( mot1 == ' ' ) mot = adjustl( mot )
            mot1 = identmot(mot,1)
            if( mot1 == '!' ) cycle
            exit
          end do
          Open(2,file=mot,status='old',iostat=istat)
          if( istat /= 0 ) then
            call write_open_error(mot,istat,0)
            close(9)
          else
            j = j + 1
            fdmnes_inp(j) = mot
          endif
          Close(2)
        end do
        Close(1)
        ncalcul = j
      endif

      Open( 98, status='SCRATCH' ) ! Erreur MPI

      if( mpinodes > 1 ) then
        call MPI_Bcast(ncalcul,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
        call MPI_BARRIER(MPI_COMM_WORLD,mpierr)
      endif 

      if( mpirank /= 0 ) allocate( fdmnes_inp(ncalcul) )

      do i = 1,ncalcul
        call fit(fdmnes_inp(i),mpirank,mpinodes,Space_file)
      end do

      deallocate( fdmnes_inp )

      Close(98) ! Erreur MPI

      if( mpinodes > 1 ) call MPI_BARRIER(MPI_COMM_WORLD,mpierr)
      call MPI_FINALIZE(mpierr) 

  110 format(//' Problem when reading the file names in fdmfile.txt !'/
     &         '   The problem is for the file number',i4,','/
     &        '   the total number of files is supposed to be',i4,'.'//)
      end

!***********************************************************************

      subroutine write_error

      character(len=50):: com_date, com_time, Revision 
      character(len=132):: fdmnes_error

      common/com_out/ com_date, com_time, fdmnes_error, Revision

      open(9, file = fdmnes_error)
      write(9,'(A/A/A)') Revision, com_date, com_time

      return
      end

!***********************************************************************

      subroutine write_open_error(File_name,istat,istop) 

      character(len=*):: File_name

      call write_error
      do ipr = 6,9,3
        write(ipr,110) File_name, istat
      end do
      if( istop == 1 ) stop

      return
  110 format(//' Error opening the file:',//3x,A,//
     &         10x,'Status =',i5,//
     &         10x,'It does not exist or ',/
     &         10x,'it is not in the good directory or',/
     &         10x,'there is some bad spelling or',/
     &         10x,'the line contains tabulations or any extra hidden',
     &             ' character !'//)
      end

!***********************************************************************

      subroutine fit(fdmnes_inp,mpirank,mpinodes,Space_file)
      
      use declarations
      implicit real(kind=db) (a-h,o-z)
      include 'mpif.h'

      parameter(nkw_all=4,nkw_fdm=126,nkw_conv=30,nkw_fit=1,
     &          nkw_metric=11,nkw_selec=5,nmetricm=4,nparam_tot=27,
     &          nparam_fdm=16)

      character(len=9) grdat, mot9, traduction
      character(len=13) mot13
      character(len=132) comt, convolution_out, fdmfit_out, fdmnes_inp,  
     &                   identmot, mot, nomfich, Space_file 
      character(len=1320) mot1320  
      character(len=2), dimension(nmetricm) :: Nom_Met
      character(len=9), dimension(nkw_all):: kw_all
      character(len=9), dimension(81):: kw_fdm1
      character(len=9), dimension(nkw_fdm-81):: kw_fdm2
      character(len=9), dimension(nkw_fdm):: kw_fdm
      character(len=9), dimension(nkw_conv) :: kw_conv
      character(len=9), dimension(nkw_fit) :: kw_fit
      character(len=9), dimension(nkw_metric) :: kw_metric
      character(len=9), dimension(nkw_selec) :: kw_selec
      character(len=9), dimension(nparam_tot):: param_conv
      character(len=9), dimension(:,:), allocatable :: typepar, typeparc
      character(len=9), dimension(:), allocatable :: typeparg

      integer, dimension(nmetricm) :: ical_Met_min 
      integer, dimension(:), allocatable :: ifile_notskip, indparp, 
     &                            Length_block, npar, npbl, nparam
      integer, dimension(:,:), allocatable :: indice_par

      logical case_fdm, check_file, conv_done, convolution_cal, 
     &        fdmnes_cal, fit_cal, green_plus, Metric_cal, minimfdmok,
     &        minimok, selec_cal

      logical, dimension(:), allocatable :: block_sum 

      real(kind=db), dimension(nmetricm) :: Dist_Min, Gen_Shift_min
      real(kind=db), dimension(:), allocatable :: par_op, parsum,
     &                                           RapIntegrT_min_g 
      real(kind=db), dimension(:,:), allocatable :: Dist_min_g, par, 
     &                                             param, parmax, parmin 

      data kw_all /  'comment  ','filout   ','length_li','length_wo'/

      data kw_conv / 'calculati','check_con','conv_out ','convoluti',
     &   'dec      ','directory','ecent    ','efermi   ','eintmax  ',
     &   'elarg    ','epsii    ','estart   ','forbidden','fprime   ',
     &   'fprime_at','gamma_fix','gamma_hol','gamma_max','gamma_var',
     &   'gaussian ','no_extrap','no_normal','nocut    ','nxan_lib ',
     &   'photo_emi','s0_2     ','scan     ','seah     ','table    ',
     &   'thomson  '/

      data kw_fdm1/  'absorbeur','adimp    ','allsite  ','atom     ',
     &   'ang_spin ','axe_spin ','base_comp','base_reel','base_spin',
     &   'biology  ','bond     ','cartesian','check    ','check_all',
     &   'check_coa','check_pot','check_mat','check_sph','chlib    ',
     &   'crystal  ','crystal_t','clementi ','dafs     ','delta_en_',
     &   'density  ','dilatorb ','dipmag   ','dpos     ','edge     ',
     &   'eimag    ','eneg     ','energphot','etatlie  ','extract  ',
     &   'flapw    ','flapw_psi','flapw_r  ','flapw_s  ','flapw_s_p',
     &   'full_atom','green    ','green_int','hedin    ','hubbard  ',
     &   'iord     ','lmax     ','lmaxfree ','lmaxso   ','lmaxstden',
     &   'ldipimp  ','lmoins1  ','lplus1   ','lquaimp  ','magnetism',
     &   'molecule ','molecule_','muffintin','multrmax ','n_self   ',
     &   'nazimut  ','nchemin  ',
     &   'new_refer','no_dipole','no_fermi ','no_interf','no_octupo',
     &   'no_quadru','no_res_ma','no_res_mo','normaltau','norman   ',
     &   'noncentre','no_check ','non_relat','nonexc   ','not_eneg ',
     &   'octupole ','old_refer','over_rad ','overlap  ','p_self   '/
                
      data kw_fdm2/  'perdew   ','pointgrou','polarized','quadrupol',
     &   'radius   ','range    ',
     &   'rangel   ','raydem   ','rchimp   ','relativis','rhoimp   ',
     &   'r_hubbard','rmt      ','rmtg     ','rmtv0    ','rot_sup  ',
     &   'rpotmax  ','r_self   ','rydberg  ','self_abs ','scf      ',
     &   'scf_exc  ','scf_non_e','screening','single_pr','solsing  ',
     &   'spgroup  ','sphere_al','spherical','spin_reso','spinorbit',
     &   'state_all','supermuf ','symmol   ','symsite  ','temperatu',
     &   'trace    ','vmax     ','v0imp    ','xalpha   ','xan_atom ',
     &   'z_nospino','zero_azim','extractsy','ylm_comp '/

      data kw_fit / 'parameter'/

      data kw_metric/ 'd2       ','detail   ','emin     ','emax     ',
     &    'experimen','fit_out  ','gen_shift','kev      ','metric_ou',
     &    'rx       ','rxg      '/

      data kw_selec/ 'selec_inp','selec_out','energy   ','azimuth  ',
     &    'reflectio'/

! D'abord les parametres de la convolution, puis ceux de fdmnes 
      data param_conv / 'ecent    ','efermi   ','elarg    ',
     &   'gamma_hol','gamma_max','gaussian ','shift    ','aseah    ',
     &   'bseah    ','vibr     ','weight   ','a        ','abc      ',
     &   'anga     ','angb     ','angc     ','b        ','c        ',
     &   'dposx    ','dposy    ','dposz    ','phi      ','poporb   ',
     &   'posx     ','posy     ','posz     ','theta    '/

      check_file = .false.
      nomfich = 'fdmnes_out'
      Length_word = 15
      Length_line = 10 + 201 * Length_word
      green_plus = .true.

      do i = 1,81
        kw_fdm(i) = kw_fdm1(i)
      end do
      do i = 82,nkw_fdm
        kw_fdm(i) = kw_fdm2(i-81)
      end do

      itape = 6
      itape1 = 11
      itape2 = 12
      itape3 = 13
      itape4 = 14
      itape5 = 17
      iscratch = 15
      iscratchconv = 35

      comt = ' '
      convolution_cal = .false.
      fdmnes_cal = .false.
      fit_cal = .false.
      Metric_cal = .false.
      selec_cal = .false.
      ngroup_par = 0
      nparm = 1
      n_shift = 1
      ng = 0

      if( mpirank /= 0 ) goto 1050

      open(1, file=fdmnes_inp, status = 'old')

      boucle_ligne: do ligne = 1,100000

        read(1,'(A)',end=1010) mot
        grdat = identmot(mot,9)

        grdat = traduction(grdat)

        if( grdat(1:1) == '!' .or. grdat(1:1) == ' ' 
     &                        .or. grdat == 'endjump' ) cycle

        if( grdat == 'end' ) exit

        if( grdat == 'jump' ) then
          do i = 1,100000
            read(1,'(A)',end=1010) mot
            grdat = identmot(mot,9)
            grdat = traduction(grdat)
            if( grdat == 'endjump' ) exit
          end do
          read(1,'(A)',end=1010) mot
          grdat = identmot(mot,9)
          grdat = traduction(grdat)
        endif

        boucle_k: do k = 1,6
          select case(k)

            case(1)
              do i = 1,nkw_fdm
                if( grdat /= kw_fdm(i) ) cycle
                itape = itape4
                if( .not. fdmnes_cal ) then
                  if( check_file ) then
                    Open( itape )
                  else
                    Open( itape, status='SCRATCH' )
                  endif
                  fdmnes_cal = .true.
                endif
                exit boucle_k
              end do

            case(2)
              do i = 1,nkw_conv
                if( grdat /= kw_conv(i) ) cycle
                itape = itape1
                if( .not. convolution_cal ) then
                  convolution_cal = .true.
                  if( check_file ) then
                    Open( itape )
                  else
                    Open( itape, status='SCRATCH' )
                  endif
                endif
                exit boucle_k
              end do

            case(3)
              do i = 1,nkw_metric
                if( grdat /= kw_metric(i) ) cycle
                itape = itape2
                if( .not. metric_cal ) then
                  metric_cal = .true.
                  if( check_file ) then
                    Open( itape )
                  else
                    Open( itape, status='SCRATCH' )
                  endif
                endif
                exit boucle_k
              end do

            case(4)
              do i = 1,nkw_fit
                if( grdat /= kw_fit(i) ) cycle
                itape = itape3
                if( .not. fit_cal ) then
                  fit_cal = .true.
                  if( check_file ) then
                    Open( itape )
                  else
                    Open( itape, status='SCRATCH' )
                  endif
                endif
                exit boucle_k
              end do

            case(5)
              do i = 1,nkw_selec
                if( grdat /= kw_selec(i) ) cycle
                itape = itape5
                if( .not. selec_cal ) then
                  selec_cal = .true.
                  if( check_file ) then
                    Open( itape )
                  else
                    Open( itape, status='SCRATCH' )
                  endif
                endif
                exit boucle_k
              end do

            case(6)
              do i = 1,nkw_all
                if( grdat /= kw_all(i) ) cycle
                exit boucle_k
              end do

          end select
        end do boucle_k

        if( k /= 6 ) then
          write(itape,'(A)') grdat
        else

          select case(grdat)

            case('length_wo')
              n = nnombre(1,132)
              read(1,*) Length_word

            case('length_li')
              n = nnombre(1,132)
              read(1,*) Length_line
              n = 10 + 201 * Length_word 
              Length_line = max( Length_line, n )

            case('comment')
              n = nnombre(1,132)
              read(1,'(A)') comt

            case('filout')
              n = nnombre(1,132)
              read(1,'(A)') mot
              nomfich = adjustl( mot )

          end select
          cycle
        endif

        boucle_j: do j = 1,1000000

          read(1,'(A)',end=1010) mot1320
          backspace(1)
          read(1,'(A)',end=1010) mot
          mot9 = identmot(mot,9)

          if( mot9(1:1) == '!' .or. mot9(1:1) == ' ') cycle
          mot9 = traduction(mot9)
          if( mot9 == 'end' ) goto 1010
          if( mot9 == 'jump' ) then
            do i = 1,100000
              read(1,'(A)',end=1010) mot
              mot9 = identmot(mot,9)
              mot9 = traduction(mot9)
              if( mot9 == 'endjump' ) exit
            end do
            read(1,'(A)',end=1010) mot1320
            backspace(1)
            read(1,'(A)',end=1010) mot
            mot9 = identmot(mot,9)
            mot9 = traduction(mot9)
            if( mot9(1:1) == '!' .or. mot9(1:1) == ' ' ) exit
            if( mot9 == 'end' ) exit
          endif

          do k = 1,nkw_fdm
            if( mot9 == kw_fdm(k) ) exit boucle_j
          end do
          do k = 1,nkw_conv
            if( mot9 == kw_conv(k) ) exit boucle_j
          end do
          do k = 1,nkw_metric
            if( mot9 == kw_metric(k) ) exit boucle_j
          end do
          do k = 1,nkw_fit
            if( mot9 /= kw_fit(k) ) cycle
            if( mot9 == 'experimen' ) then
              mot13 = identmot(mot,13)
              if(mot13 /= 'experiment' .and. mot13 /= 'experimen') cycle
            endif
            exit boucle_j
          end do
          do k = 1,nkw_selec
            if( mot9 == kw_selec(k) ) exit boucle_j
          end do
          do k = 1,nkw_all
            if( mot9 == kw_all(k) ) exit boucle_j
          end do

          if( mot9(1:4) == 'par_' ) then
            mot13 = identmot(mot,13)
            mot9 = mot13(5:13)
            mot9 = traduction(mot9)
            mot1320 = ' '
            mot1320(2:10) = mot9 
          endif

          write(itape,'(A)') mot1320

        end do boucle_j         

        backspace(1)

      end do boucle_ligne

 1010 continue

      Close(1)

      if( metric_cal) then
        Rewind(itape2)
        ngroup_par = 1
        do ligne = 1,100000
          read(itape2,'(A)',end=1020) mot
          mot9 = identmot(mot,9)
          if( mot9 == 'gen_shift' ) then
            n = nnombre(itape2,132)
            read(itape2,*) e1, e2, n_shift
          elseif( mot9 == 'experimen' ) then
            boucle_i: do i = 1,100000
              n = nnombre(itape2,132)
              read(itape2,'(A)',end=1020) mot
              mot9 = identmot(mot,9)
              do j = 1,nkw_fdm
                if( mot9 == kw_fdm(j) ) exit boucle_i
              end do
              do j = 1,nkw_conv
                if( mot9 == kw_conv(j) ) exit boucle_i
              end do
              do j = 1,nkw_metric
                if( mot9 == kw_metric(j) ) exit boucle_i
              end do
              do j = 1,nkw_fit
                if( mot9 /= kw_fit(j) ) cycle
                if( mot9 == 'experimen' ) then
                  mot13 = identmot(mot,13)
                  if( mot13 /= 'experiment' .and. 
     &                                mot13 /= 'experimen' ) cycle
                endif
                exit boucle_i
              end do
              if( n == 0 ) ng = ng + 1  ! nombre de spectres
            end do boucle_i
            backspace(itape2)
          endif
        end do
      endif

 1020 continue

      nblock = 0
      if( fit_cal) then
        Rewind(itape3)
        do ligne = 1,100000
          read(itape3,'(A)',end=1030) mot
          mot9 = identmot(mot,9)
          if( mot9 == 'parameter' ) then
            ngroup_par = ngroup_par + 1
            nblock = nblock + 1
            nn = 0

            do ligne2 = 1,100000
              n = nnombre(itape3,132)
              read(itape3,'(A)',end=1030) mot
              if( mot == ' ' ) cycle
              mot9 = identmot(mot,9)
              if( mot9 == 'parameter' ) then
                backspace(itape3)
                exit
              else
                if( n == 0 ) then
                  cycle
                elseif( n > 2 ) then
                  nn = nn + 1 
                else
                  ngroup_par = ngroup_par + nn - 1
                  exit
                endif
              endif
            end do
          endif

        end do
 1030   continue
        Rewind(itape3)
      endif

      allocate( npar(ngroup_par) )
      allocate( npbl(nblock) )
      allocate( block_sum(nblock) )
      if( ngroup_par > 0 ) then
        block_sum(:) = .false.
        npar(:) = 0
      endif

      if( fit_cal) then

        Rewind(itape3)
        do ibl = 1,nblock
          n = nnombre(itape3,132)
          read(itape3,'(A)') mot
          mot9 = identmot(mot,9)
          if( mot9 == 'parameter' ) then
            do ligne = 1,100000
              n = nnombre(itape3,132)
              read(itape3,'(A)',end=1035) mot
              if( mot == ' ' ) cycle
              mot9 = identmot(mot,9)
              if( mot9 == 'parameter' ) then
                backspace(itape3)
                exit
              else
                if( n == 1 .or. n == 2 ) then
                  block_sum(ibl) = .true.
                  exit
                else
                  cycle
                endif
              endif
            end do
          endif

        end do
 1035   continue
      endif

      if( Metric_cal ) then
        nparm = n_shift
        npar(1) = 1
      endif 

      if( fit_cal ) then

        Rewind(itape3)
        igr = 1
        npbl(:) = 0
        do ibl = 1,nblock
          i1 = 1
          n = nnombre(itape3,132)
          read(itape3,'(A)')
          if( .not. Block_sum(ibl) ) igr = igr + 1
          do ligne = 1,100000
            n = nnombre(itape3,132)
            read(itape3,'(A)',end=1040) mot
            if( n > 0 ) cycle
            mot9 = identmot(mot,9)
            if( mot9 == 'parameter' ) then
              backspace(itape3)
              exit
            else
              do i = 1,nparam_tot
                if( mot9 /= param_conv(i) ) cycle
                if( block_sum(ibl) ) then
                  n = nnombre(itape3,132)
                  if( n > 2 ) then
                    igr = igr + 1
                    if( i1 == 1 ) then
                      npar(igr) = 2
                      i1 = 2     ! pour eviter d'ajouter 2 fois dans lectur
                    else
                      npar(igr) = 1
                    endif
                  endif
                else
                  npar(igr) = npar(igr) + 1
                endif
                npbl(ibl) = npbl(ibl) + 1
                exit        
              end do
              if( i > nparam_tot .and. mpirank == 0 ) then
                call write_error
                do ipr = 6,9,3
                  write(ipr,110) mot9
                end do
                close(9)
                stop
              endif
            endif
          end do
 1040     nparm = max( nparm, npar(igr) )
        end do

      endif

 1050 continue   ! Arrivee en cas de calcul parallele

      if( mpinodes > 1 ) then
        call MPI_Bcast(nblock,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(ngroup_par,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(Length_line,1,MPI_INTEGER,0,
     &                 MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(green_plus,1,MPI_LOGICAL,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(Length_word,1,MPI_INTEGER,0,
     &                 MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(e1,1,MPI_REAL8,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(e2,1,MPI_REAL8,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(n_shift,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(nparm,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(ng,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(fit_cal,1,MPI_LOGICAL,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(fdmnes_cal,1,MPI_LOGICAL,0,MPI_COMM_WORLD,mpierr)
        call MPI_BARRIER(MPI_COMM_WORLD,mpierr) 
      endif

      if( mpirank > 0 ) then
        allocate( npbl(nblock) )
        allocate( block_sum(nblock) )
        allocate( npar(ngroup_par) )
      endif

      allocate( Length_block(ngroup_par) )
      allocate( param(ngroup_par,nparm) )
      allocate( parmax(ngroup_par,nparm) )
      allocate( parmin(ngroup_par,nparm) )
      allocate( parsum(ngroup_par) )
      allocate( nparam(ngroup_par) )
      allocate( indice_par(ngroup_par,nparm) )
      allocate( typepar(ngroup_par,nparm) )
      allocate( typeparc(ngroup_par,nparm) )
      allocate( typeparg(ngroup_par) )
      if( ngroup_par > 0 ) then
        indice_par(:,:) = 0
        Length_block(:) = 0
        parmin(1,1) = e1
        parmax(1,1) = e2
        nparam(1) = n_shift
        typepar(1,1) = 'Gen_Shift'
        typeparg(1) = 'Gen_Shift'
      endif      

      if( mpirank == 0 ) then
        allocate( Dist_min_g(ng,nmetricm) ) 
        allocate( RapIntegrT_min_g(ng) ) 
      endif

      if( fit_cal .and. mpirank == 0 ) then

        rewind(itape3)
        igr = 1
        do ibl = 1,nblock
          n = nnombre(itape3,132)
          read(itape3,'(A)') mot
          mot9 = identmot(mot,9)
          if( .not. Block_sum(ibl) ) igr = igr + 1
          do ipbl = 1,npbl(ibl)
            if( .not. Block_sum(ibl) ) then
              ipar = ipbl
            else
              if( ipbl == npbl(ibl) ) then
                ipar = 2
              else
                ipar = 1
                igr = igr + 1
              endif
            endif
            if( Block_sum(ibl) ) then
              if( ipbl < npbl(ibl) - 1 ) then
                Length_block(igr) = - 1
              elseif( ipbl == npbl(ibl) - 1 ) then
                Length_block(igr) = npbl(ibl) - 1
              endif 
            endif 
            n = nnombre(itape3,132)
            read(itape3,'(A)') mot
            mot9 = identmot(mot,9) 
            if( Block_sum(ibl) .and. ipar == 2 ) then 
              typepar(igr-npbl(ibl)+2:igr,ipar) = mot9 
            else
              typepar(igr,ipar) = mot9
            endif
            n = nnombre(itape3,132)
            select case( n )
              case(1)
                read(itape3,*) x
                parsum(igr-npbl(ibl)+2:igr) = x
              case(2)
                read(itape3,*) x, i
                parsum(igr-npbl(ibl)+2:igr) = x
                indice_par(igr-npbl(ibl)+2:igr,ipar) = i
              case(3)
                read(itape3,*) parmin(igr,ipar), parmax(igr,ipar),
     &                    nparam(igr)
              case(4)
                read(itape3,*) parmin(igr,ipar), parmax(igr,ipar),
     &                    nparam(igr), indice_par(igr,ipar)
            end select
          end do
        end do
        Close(itape3)

      endif

      if( mpinodes > 1 ) then
        if( ngroup_par > 0 ) then
          call MPI_Bcast(Length_block,ngroup_par,MPI_INTEGER,0,
     &                 MPI_COMM_WORLD,mpierr)
          call MPI_Bcast(npar,ngroup_par,MPI_INTEGER,0,MPI_COMM_WORLD,
     &                 mpierr)
          call MPI_Bcast(npbl,nblock,MPI_INTEGER,0,MPI_COMM_WORLD,
     &                 mpierr)
        endif
        call MPI_BARRIER(MPI_COMM_WORLD,mpierr)
        if( fit_cal ) then
          call MPI_Bcast(nparam,ngroup_par,MPI_INTEGER,0,
     &                   MPI_COMM_WORLD,mpierr)
          call MPI_Bcast(indice_par,nparm*ngroup_par,MPI_INTEGER,0,
     &                   MPI_COMM_WORLD,mpierr)
          call MPI_Bcast(parmin,nparm*ngroup_par,MPI_REAL8,0,
     &                   MPI_COMM_WORLD,mpierr)
          call MPI_Bcast(parmax,nparm*ngroup_par,MPI_REAL8,0,
     &                   MPI_COMM_WORLD,mpierr)
          call MPI_Bcast(parsum,ngroup_par,MPI_REAL8,0,
     &                   MPI_COMM_WORLD,mpierr)
          do igr = 1,ngroup_par
            do ipar = 1,npar(igr)
              if( mpirank /= 0 ) mot = ' ' 
              if( mpirank == 0 ) mot = typepar(igr,ipar) 
              do i = 1,9
                if( mpirank == 0 ) j = iachar( mot(i:i) )
                call MPI_Bcast(j,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
                call MPI_BARRIER(MPI_COMM_WORLD,mpierr) 
                if( mpirank /= 0 ) mot(i:i) = achar( j ) 
              end do  
              if( mpirank /= 0 ) typepar(igr,ipar) = mot 
            end do
          end do
        endif
      endif 

      ncal = 1
      if( fit_cal ) then

        do igr = 2,ngroup_par
          if( abs(parmin(igr,1) - parmax(igr,1)) < 0.00000001_db )
     &      nparam(igr) = 1
        end do

        do igr = 2,ngroup_par
          nparam(igr) = max( nparam(igr), 1 )
          ncal = ncal * nparam(igr)
        end do

      endif

      nparam_conv = nparam_tot - nparam_fdm

! Mise en ordre des parametres : d'abord shift, puis conv, puis fdm.
      boucle_igr: do igr = 2,ngroup_par
        do ipar = 1,npar(igr)
          do i = 1,nparam_conv
            if( typepar(igr,ipar) == param_conv(i) ) exit
          end do
          if( i > nparam_conv ) exit 
        end do
        case_fdm = .true.
        if( i <= nparam_conv ) then
! On teste si pur shift ou gaus
          do ipar = 1,npar(igr)
            if( typepar(igr,ipar) /= 'shift' .and. 
     &          typepar(igr,ipar) /= 'weight' .and. 
     &          typepar(igr,ipar) /= 'gaussian' ) exit
          end do
          if( ipar > npar(igr) ) cycle
          case_fdm = .false.
        endif 

        do jgr = igr+1,ngroup_par
          if( case_fdm ) then
            do jpar = 1,npar(jgr)
              do j = 1,nparam_conv
                if( typepar(jgr,jpar) == param_conv(j) ) exit
              end do
              if( j > nparam_conv ) exit 
            end do
            if( j > nparam_conv ) cycle
          else
            do jpar = 1,npar(jgr)
              if( typepar(jgr,jpar) /= 'shift' .and. 
     &            typepar(jgr,jpar) /= 'weight' .and. 
     &            typepar(jgr,jpar) /= 'gaussian' ) exit
            end do
            if( jpar <= npar(jgr) ) cycle
          endif 

          n = Length_block(igr)
          Length_block(igr) = Length_block(jgr)
          Length_block(jgr) = n
          n = nparam(igr)
          nparam(igr) = nparam(jgr)
          nparam(jgr) = n
          n = npar(igr)
          npar(igr) = npar(jgr)
          npar(jgr) = n
          n = min(npar(igr),npar(jgr)) 
          do ipar = 1,n
            x = parmin(igr,ipar)
            parmin(igr,ipar) = parmin(jgr,ipar)
            parmin(jgr,ipar) = x
            x = parmax(igr,ipar)
            parmax(igr,ipar) = parmax(jgr,ipar)
            parmax(jgr,ipar) = x
            mot9 = typepar(igr,ipar)
            typepar(igr,ipar) = typepar(jgr,ipar)
            typepar(jgr,ipar) = mot9
            m = indice_par(igr,ipar)
            indice_par(igr,ipar) = indice_par(jgr,ipar) 
            indice_par(jgr,ipar) = m 
          end do
          do ipar = n+1,npar(igr)
            parmin(igr,ipar) = parmin(jgr,ipar)
            parmin(jgr,ipar) = 0._db
            parmax(igr,ipar) = parmax(jgr,ipar)
            parmax(jgr,ipar) = 0._db
            typepar(igr,ipar) = typepar(jgr,ipar)
            typepar(jgr,ipar) = '         '
            indice_par(igr,ipar) = indice_par(jgr,ipar) 
            indice_par(jgr,ipar) = 0 
          end do
          do ipar = n+1,npar(jgr)
            parmin(jgr,ipar) = parmin(igr,ipar)
            parmin(igr,ipar) = 0._db
            parmax(jgr,ipar) = parmax(igr,ipar)
            parmax(igr,ipar) = 0._db
            typepar(jgr,ipar) = typepar(igr,ipar)
            typepar(igr,ipar) = '         '
            indice_par(jgr,ipar) = indice_par(igr,ipar) 
            indice_par(igr,ipar) = 0 
          end do
          x = parsum(igr)
          parsum(igr) = parsum(jgr)
          parsum(jgr) = x

        end do
          
      end do boucle_igr

      ncal_nonfdm = 1
      boucle_ext: do igr = 2,ngroup_par
        do ipar = 1,npar(igr)
          do i = 1,nparam_conv
            if( typepar(igr,ipar) == param_conv(i) ) exit
          end do
          if( i > nparam_conv ) exit boucle_ext
        end do
        ncal_nonfdm = ncal_nonfdm * nparam(igr)
      end do boucle_ext
      ngroup_par_conv = igr - 1
  
      do igr = 1,ngroup_par
        do ipar = 1,npar(igr)
          typeparc(igr,ipar) = typepar(igr,ipar)
          if( indice_par(igr,ipar) /= 0 ) then
            mot9 = typepar(igr,ipar)
            l = len_trim(mot9) + 1
            mot9(l:l) = '_'
            call ad_number(indice_par(igr,ipar),mot9,9)
            typeparc(igr,ipar) = mot9
          endif
        end do
      end do

      if( ngroup_par > 1 ) then
        allocate( indparp(ngroup_par) ) 
        indparp(:) = 0
      endif

      nnotskipm = 0
      do igr = 2,ngroup_par
        nnotskipm = nnotskipm + npar(igr)
      end do
      allocate( ifile_notskip( nnotskipm ) )

      n_atom_proto_p = 0

      do ical = 1,ncal

        if( ical == 1 ) then
          conv_done = .false.
        else
          conv_done = .true.
        endif
           
        ndem = 1
        inotskip = 0
        do igr = 2,ngroup_par

          j = ( ical - 1 ) / ndem
          indpar = mod( j, nparam(igr) ) + 1 
          ndem = ndem * nparam(igr)

          if( nparam(igr) < 2 ) then
            param(igr,1:npar(igr)) = parmin(igr,1:npar(igr))
          else
            do ipar = 1,npar(igr)
              if( Length_block(igr) /= 0 .and. ipar > 1 ) exit
              param(igr,ipar) = parmin(igr,ipar) + ( indpar - 1 )
     &                        * ( parmax(igr,ipar) - parmin(igr,ipar) )
     &                        / ( nparam(igr) - 1 )
            end do
          endif

          if( Length_block(igr) > 0 ) then ! on est a la fin du block
            param_dep = parsum(igr)
     &                   - sum( param(igr-Length_block(igr)+1:igr,1) )
            param(igr-Length_block(igr)+1:igr,2) = param_dep
          endif
         
          if( indpar /= indparp(igr) ) then

            do ipar = 1,npar(igr)
              if( typepar(igr,ipar) /= 'shift' .and.
     &            typepar(igr,ipar) /= 'weight' .and. 
     &            typepar(igr,ipar) /= 'gaussian' ) conv_done = .false.
            end do

            if( ical > 1 ) then
              do ipar = 1,npar(igr)
                if( typepar(igr,ipar) /= 'dposx' .and.
     &              typepar(igr,ipar) /= 'dposy' .and. 
     &              typepar(igr,ipar) /= 'dposz' .and. 
     &              typepar(igr,ipar) /= 'posx' .and. 
     &              typepar(igr,ipar) /= 'posy' .and. 
     &              typepar(igr,ipar) /= 'posz' .and. 
     &              typepar(igr,ipar) /= 'poporb' ) cycle
                inotskip = inotskip + 1
                ifile_notskip(inotskip) = indice_par(igr,ipar)
              end do
            endif

          endif
          indparp(igr) = indpar
        end do
        nnotskip = inotskip

        if( ncal_nonfdm == 1 ) then
          ifdm = 1
        else
          ifdm = mod(ical,ncal_nonfdm)
        endif 

        if( mpinodes > 1 ) call MPI_BARRIER(MPI_COMM_WORLD,mpierr) 

        if( fdmnes_cal .and. ifdm == 1 ) call fdm(comt,convolution_cal,
     &      fit_cal,green_plus,ifile_notskip,indice_par,iscratch,
     &      itape1,itape4,Length_word,mpinodes,mpirank,
     &      n_atom_proto_p,ngroup_par,nnotskip,nnotskipm,
     &      nomfich,npar,nparm,param,Space_file,typepar)

        if( mpirank /= 0 ) cycle

        if( convolution_cal ) call convolution(conv_done,
     &           convolution_out,fit_cal,green_plus,ical,
     &           indice_par,iscratchconv,itape1,kw_conv,length_line,
     &           Length_word,ngroup_par,nkw_conv,nomfich,npar,
     &           nparm,param,typepar,ncal)

        if( Metric_cal ) then

          if( ical == 1 ) then
            if( nomfich == 'fdmnes_out' ) then
              fdmfit_out = 'fdmfit_out.txt'
            else
              mot = nomfich
              l = len_trim(mot)
              mot(l+1:l+8) = '_fit.txt'
              fdmfit_out = mot
            endif
          endif

          call metric(comt,convolution_out,Dist_min,
     &           Dist_min_g,fdmfit_out,fit_cal,Gen_Shift_min,ical,
     &           ical_Met_min,index_Met_Fit,iscratchconv,itape2,
     &           length_line,
     &           Length_block,ncal,ndm,ng,ngroup_par,nmetric,nmetricm,
     &           Nom_Met,npar,nparam,nparm,param,parmax,parmin,
     &           RapIntegrT_min_g,typeparc)

        endif

      end do

      if( ngroup_par > 1 ) deallocate( indparp ) 

      allocate( par_op(ngroup_par) )

      if( Metric_cal .and. mpirank == 0 ) then

        npm = nparam(1) 
        do igr = 2,ngroup_par
          npm = max( npm, nparam(igr) )
        end do

        allocate( par(ngroup_par,npm) )
        do igr = 1,ngroup_par
          typeparg(igr) = typeparc(igr,1)
          do ip = 1,nparam(igr)
            if( nparam(igr) < 2 ) then
              par(igr,ip) = parmin(igr,1)
            else
              fac = ( parmax(igr,1) - parmin(igr,1) )
     &            / ( nparam(igr) - 1 )
              par(igr,ip) = parmin(igr,1) +  fac * ( ip - 1 )
            endif
          end do
        end do

        call minim(fdmfit_out,index_Met_Fit,minimfdmok,minimok,ncal,ndm,
     &             ngroup_par,ngroup_par_conv,nmetric,nmetricm,Nom_Met,
     &             nparam,npm,par,par_op,typeparg)

        deallocate( par )

      endif

      if( mpinodes > 1 ) then
        call MPI_Bcast(minimfdmok,1,MPI_LOGICAL,0,MPI_COMM_WORLD,mpierr)
        call MPI_Bcast(minimok,1,MPI_LOGICAL,0,MPI_COMM_WORLD,mpierr)
        call MPI_BARRIER(MPI_COMM_WORLD,mpierr)
      endif 

      if( fit_cal .and. minimok .and. ncal > 1 ) then

        if( mpinodes > 1 ) call MPI_Bcast(par_op,ngroup_par,
     &                               MPI_REAL8,0,MPI_COMM_WORLD,mpierr)

        ndem = 1
        inotskip = 0
        do igr = 2,ngroup_par

          j = ( ical - 1 ) / ndem
          indpar = mod( j, nparam(igr) ) + 1 
          ndem = ndem * nparam(igr)

          if( nparam(igr) > 1 ) then
            prop = ( par_op(igr) - parmin(igr,1) )
     &           / ( parmax(igr,1) - parmin(igr,1) )
            do ipar = 1,npar(igr)
              if( Length_block(igr) /= 0 .and. ipar > 1 ) exit
              param(igr,ipar) = ( 1 - prop ) * parmin(igr,ipar)
     &                        + prop * parmax(igr,ipar)
            end do
          endif

          if( Length_block(igr) > 0 ) then ! on est a la fin du block
            param_dep = parsum(igr)
     &                   - sum( param(igr-Length_block(igr)+1:igr,1) )
            param(igr-Length_block(igr)+1:igr,2) = param_dep
          endif
         
          do ipar = 1,npar(igr)
            if( typepar(igr,ipar) /= 'shift' .and.
     &          typepar(igr,ipar) /= 'weight' .and. 
     &          typepar(igr,ipar) /= 'gaussian' ) conv_done = .false.
          end do

          do ipar = 1,npar(igr)
            if( typepar(igr,ipar) /= 'dposx' .and.
     &          typepar(igr,ipar) /= 'dposy' .and. 
     &          typepar(igr,ipar) /= 'dposz' .and. 
     &          typepar(igr,ipar) /= 'posx' .and. 
     &          typepar(igr,ipar) /= 'posy' .and. 
     &          typepar(igr,ipar) /= 'posz' .and. 
     &          typepar(igr,ipar) /= 'poporb' ) cycle
            inotskip = inotskip + 1
            ifile_notskip(inotskip) = indice_par(igr,ipar)
          end do

        end do
        nnotskip = inotskip

        if( fdmnes_cal .and. minimfdmok .and. ncal /= ncal_nonfdm )
     &    call fdm(comt,convolution_cal,
     &      fit_cal,green_plus,ifile_notskip,indice_par,iscratch,
     &      itape1,itape4,Length_word,mpinodes,mpirank,
     &      n_atom_proto_p,ngroup_par,nnotskip,nnotskipm,
     &      nomfich,npar,nparm,param,Space_file,typepar)

        if( mpirank == 0 ) then
          if( convolution_cal ) call convolution(.false.,
     &           convolution_out,fit_cal,green_plus,ical,
     &           indice_par,iscratchconv,itape1,kw_conv,length_line,
     &           Length_word,ngroup_par,nkw_conv,nomfich,npar,
     &           nparm,param,typepar,ncal)

          call metric(comt,convolution_out,Dist_min,
     &           Dist_min_g,fdmfit_out,fit_cal,Gen_Shift_min,ical,
     &           ical_Met_min,index_Met_Fit,iscratchconv,itape2,
     &           length_line,
     &           Length_block,ncal,ndm,ng,ngroup_par,nmetric,nmetricm,
     &           Nom_Met,npar,nparam,nparm,param,parmax,parmin,
     &           RapIntegrT_min_g,typeparc)
        endif

      endif

      deallocate( par_op )
      deallocate( ifile_notskip )

      deallocate( block_sum )
      deallocate( npbl ) 
      deallocate( Length_block ) 
      deallocate( npar ) 
      deallocate( param )
      deallocate( parmax )
      deallocate( parmin )
      deallocate( parsum )
      deallocate( nparam )
      deallocate( indice_par )
      deallocate( typepar )
      deallocate( typeparc )
      deallocate( typeparg )

      if( mpirank == 0 ) then
        deallocate( Dist_min_g )
        deallocate( RapIntegrT_min_g )
      endif

      if( mpirank == 0 .and. selec_cal ) call selec(itape5)

      if( mpirank == 0 ) then
        if( convolution_cal ) Close(itape1)
        if( Metric_cal ) Close(itape2)
        if( fdmnes_cal ) Close(itape4)
        if( selec_cal ) Close(itape5)
      endif

      return
 110  format(///' The parameter called ',a9,' does not exist !'/,
     &       ' Check your indata file under the keyword parameter !'//)
      end

!*********************************************************************

      function identmot(mot,longueur)

      use declarations
      implicit real(kind=db) (a-h,o-z)

      character(len=1) let(52)
      character(len=132) identmot, mot

      data let/'a','b','c','d','e','f','g','h','i','j','k','l','m',
     & 'n','o','p','q','r','s','t','u','v','w','x','y','z','A','B','C',
     & 'D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S',
     & 'T','U','V','W','X','Y','Z'/

      do i = 1,131-longueur
        if(mot(i:i) /= ' ') exit
      end do
      idebut = i

      do i = idebut+1,idebut+longueur
        if(mot(i:i) == ' ') exit
      end do
      ifin = i-1
      identmot = mot(idebut:ifin)

! Convertion de majuscule en minuscule
      do i = 1,longueur
        do l = 27,52
          if( identmot(i:i) == let(l) ) then
            identmot(i:i) = let(l-26)
            exit
          endif
        end do
        if( identmot(i:i) == '-' ) identmot(i:i) = '_'
      end do

      return
      end

!*********************************************************************

      function traduction(grdat)

      use declarations
      implicit real(kind=db) (a-h,o-z)

      character(len=9) grdat, traduction

      traduction = grdat

      select case(grdat)
        case('end_jump')
          traduction = 'endjump'
        case('fin','fine')
          traduction = 'end'
        case('biologie')
          traduction = 'biology'
        case('gamme')
          traduction = 'range'
        case('gammel')
          traduction = 'rangel'
        case('iabsorbeu','absorbor','assorbito','absorber')
          traduction = 'absorbeur'
        case('potimag  ')
          traduction = 'eimag'
        case('quadripol')
          traduction = 'quadrupol'
        case('spin_orbi')
          traduction = 'spinorbit'
        case('magnetiqu','magnetic')
          traduction = 'magnetism'
        case('polarise','polarised','polarisat','polarizat','polar')
          traduction = 'polarized'
        case('dafsscan','dafscan','rxs')
          traduction = 'dafs'
        case('atome','atomes','atoms','atomo','atomi')
          traduction = 'atom'
        case('atome_ch','atomch','atomech','atomoch','atomo_ch',
     &       'atomich','atomi_ch')
          traduction = 'atom_ch'
        case('crist','cryst','cristallo','cristal')
          traduction = 'crystal'
        case('molec','molecola')
          traduction = 'molecule'
        case('lapw','wien')
          traduction = 'flapw'
        case('lapw_s','wien_s','flapw_sau','lapw_sauv','wien_sauv',
     &       'flapw_sav','lapw_save','wien_save')
          traduction = 'flapw_s'
        case('lapw_r','wien_r','flapw_rec','lapw_recu','wien_recu')
          traduction = 'flapw_r'
        case('icheck')
          traduction = 'check'
        case('checkall','allcheck','all_check')
          traduction = 'check_all'
        case('checkcoab')
          traduction = 'check_coa'
        case('checkmat')
          traduction = 'check_mat'
        case('checkpot')
          traduction = 'check_pot'
        case('checksph','checksphe')
          traduction = 'check_sph'
        case('cartesien')
          traduction = 'cartesian'
        case('extractio')
          traduction = 'extrac'
        case('extract_s')
          traduction = 'extracsy'
        case('enrgpsii','epsiia')
          traduction = 'epsii'
        case('enrgphot','energpho')
          traduction = 'energphot'
        case('state_den','statedens','densite')
          traduction = 'density'
        case('stateall')
          traduction = 'state_all'
        case('oldrefere')
          traduction = 'old_refer'
        case('newrefere')
          traduction = 'new_refer'
        case('eclie')
          traduction = 'etatlie'
        case('noteneg')
          traduction = 'not_eneg'
        case('chemin')
          traduction = 'nchemin'
        case('lminus1')
          traduction = 'lmoins1'
        case('basereel','real_basi','real_base')
          traduction = 'base_reel'
        case('basecomp')
          traduction = 'base_comp'
        case('spinresol')
          traduction = 'spin_reso'
        case('angspin','spin_ang')
          traduction = 'ang_spin'
        case('axespin','spin_axe','spin_axes','spinaxe')
          traduction = 'axe_spin'
        case('ang_rotsu','angrotsup','rotsup')
          traduction = 'rot_sup'
        case('relat')
          traduction = 'relativis'
        case('nonrelat','nonrelati')
          traduction = 'non_relat'
        case('rayon','rsort','rsorte','raggio')
          traduction = 'radius'
        case('r_selfcon','rself','rselfcons')
          traduction = 'r_self' 
        case('p_self0','pself','pself0')
          traduction = 'p_self' 
        case('selfcons','nself','self_cons')
          traduction = 'scf'   
        case('selfexc','self_exc','scfexc')
          traduction = 'scf_exc'   
        case('selfnonex','self_non_','self_none','scfnonexc',
     &       'scf_nonex')
          traduction = 'scf_non_e'   
        case('converge','deltae','delta_e','delta_en')
          traduction = 'delta_en_'   
        case('nofermi')
          traduction = 'no_fermi'   
        case('overad','overrad')
          traduction = 'over_rad'
        case('ecrantage')
          traduction = 'screening'
        case('non_exc','nonexcite','non_excit')
          traduction = 'nonexc'
        case('seuil','threshold','soglia')
          traduction = 'edge'
        case('dipole_oc','dip_oct')
          traduction = 'octupole'
        case('magdip','dip_mag','mag_dip','m1_m1','m1m1')
          traduction = 'dipmag'
        case('notdipole','nondipole','nodipole')
          traduction = 'no_dipole'
        case('notquadru','nonquadru','noquadrup')
          traduction = 'no_quadru'
        case('notoctupo','nonoctupo','nooctupol')
          traduction = 'no_octupo'
        case('notinterf','noninterf','nointerf','no_dipqua','nodipquad',
     &       'nondipqua')
          traduction = 'no_interf'
        case('lquadimp')
          traduction = 'lquaimp'
        case('normal_ta')
          traduction = 'normaltau'
        case('nphim','nphi','nazimuth')
          traduction = 'nazimut'
        case('singsol')
          traduction = 'solsing'
        case('raychimp')
          traduction = 'rchimp'
        case('rmtimp','rmt_imp','rmtgimp','rmtg_imp')
          traduction = 'rmtg'
        case('rmtvo')
          traduction = 'rmtv0'
        case('muffin_ti')
          traduction = 'muffintin'
        case('interpoin','inter_poi')
          traduction = 'adimp'
        case('lmaxat','lmaxat0','lmax_at','lmax_atom')
          traduction = 'lmax'
        case('lamstdens')
          traduction = 'lmaxstden'
        case('lmaxso','lmaxso0','lmax_so','lmax_sor','lmaxsort')
          traduction = 'lmaxso'
        case('chlibre','freech')
          traduction = 'chlib'
        case('hedin_lun','hedinlund')
          traduction = 'hedin'
        case('selfabsor','self_abso','selfabs')
          traduction = 'self_abs'
        case('xalfa','alfpot')
          traduction = 'xalpha'
        case('hubard')
          traduction = 'hubbard'
        case('dilat','dilatati','dilat_or')
          traduction = 'dilatorb'
        case('voimp','v0bdcfim','vmoyf','korigimp')
          traduction = 'v0imp'
        case('rho_imp')
          traduction = 'rhoimp'
        case('v_intmax','v_max')
          traduction = 'vmax'
        case('sym','point_gro')
          traduction = 'pointgrou'
        case('sym_mol')
          traduction = 'symmol'
        case('fileout','file_out')
          traduction = 'filout'
        case('sphere_si')
          traduction = 'sphere_al'
        case('ylm_compl','ylmcomp','ylmcomple','harmo_com','harmocomp')
          traduction = 'ylm_comp'
! Convolution
        case('arc')
          traduction = 'convoluti'
        case('resolutio','gaus','gauss')
          traduction = 'gaussian'
        case('fprim')
          traduction = 'fprime'
        case('fprim_ato','fprimatom','fprimeato')
          traduction = 'fprime_at'
        case('seah_denc')
          traduction = 'seah'
        case('noextrap')
          traduction = 'no_extrap'
        case('photoemis','photo')
          traduction = 'photo_emi'
        case('s02','so2','so_2')
          traduction = 'photo_emi'
! Fit
        case('pop_orb')
          traduction = 'poporb'
        case('fit_rx')
          traduction = 'rx'
      end select

      return
      end

!***********************************************************************

! Fonction donnant le nombre de chiffres dans la prochaine ligne non vide
! et se place devant cette ligne.
! Si character nnombre = 0

      function nnombre(irec,length_line)

      use declarations
      implicit real(kind=db) (a-h,o-z)

      parameter(nlet = 53)

      character(len=length_line):: mot, test
      character(len=1), dimension(nlet):: let

      data let /'d','e','D','E',
     &          'a','A','b','B','c','C','f','F',
     &          'g','G','h','H','i','I','j','J',
     &          'k','K','l','L','m','M','n','N',
     &          'o','O','p','P','q','Q','r','R',
     &          's','T','t','T','u','U','v','V',
     &          'w','W','x','X','y','Y','z','Z','/'/

      do i = 1,length_line
        mot(i:i) = 'P'
      end do

      nmots = 0

      boucle_ligne: do ligne = 1,1000000

        read(irec,'(A)',end=1030) mot
        do i = 1,length_line 
          if( mot(i:i) /= ' ' ) exit boucle_ligne
        end do

      end do boucle_ligne

      backspace(irec)

      Open( 16, status='SCRATCH' )

      n = 0
      i = 0
      do icol = 1,length_line 
        i = i + 1
        if( i > length_line ) exit         
        if( mot(i:i) == ' ' ) cycle
        do j = i+1,length_line
          if( mot(j:j) == ' ' ) exit
        end do
        j = j - 1
        n = n + 1
        write(16,'(A)') mot(i:j)
        i = j
      end do

      Rewind(16)

      do i = 1,n
        read(16,*,err=1000,end=1000) x
        backspace(16)
        read(16,'(A)',err=1000,end=1000) test
        l = len_trim(test)
        do j = 1,l
          do k = 5,nlet
            if( test(j:j) == let(k) ) goto 1000          
          end do
        end do
        do k = 1,4
          if( test(1:1) == let(k) .or. test(l:l) == let(k) ) goto 1000   
        end do
        nmots = nmots + 1
      end do
 1000 Close(16)

! ESRF changes start
      nnombre = nmots
      return

! An "END=" condition leaves the file position after the endfile record,
! thus backspace to be just before the endfile record. This way a
! subsequent READ with another "END=" clause can execute correctly.

 1030 nnombre = 0
      backspace(irec)
! ESRF changes end

      return
      end

