/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                         fit.c                                 $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: fit.c,v 1.6 2007/03/16 14:48:42 wilcke Exp $

****************************************************************************/

/***************************************************************************

  Changes:
Update 16/03/2007 R. Wilcke (wilcke@esrf.fr)
                  remove include of <lsqfit.h> (is now in "rod.h").
                  create new routine init_fitpar() to copy all fit parameters
                  to array FITPAR;
                  create new routine update_par() to copy values from array
                  FITPAR to corresponding model parameters;
                  fit(): use new routines init_fitpar() and update_par();
                  fit(), ffit(), init_fitpar() and update_par(): use new
                  variable "nsf" instead of macro "NSF" for parameter
                  calculations (better compatibility with experimental version).
Update 12/10/01 E.Vlieg (vlieg@sci.kun.nl)
	modify ASA calling conventions
Update 01/03/01 E.Vlieg (vlieg@sci.kun.nl)
		Prepare for adaptive simulated annealing (ASA).
Update 05/01/01 E.Vlieg (vlieg@sci.kun.nl)
                Replace keat_calc() by energy().

  21/10/91:  Introduced penalty factors for fit parameters
  24/10/91:  Calculate "goodness" of fit
  11/11/91:  penalty factors are now always 0 or 10000 

****************************************************************************/

#define LARGE_PENALTY 1e4	/* useful penalty factor */
#define NO_PENALTY    0

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define FIT
#include "rod.h"

/***************************************************************************/
void fit(void)
/***************************************************************************/

    /*
    Fit the experimental rod data.
    */

{

    int freepar[MAXFIT],nfree;
    int i,ntot,numpar,nomodel;
    float chisqr,q;
    float penalties[MAXFIT];
    int rc;
    int nsf = NSF;
    float index[MAXDATA];

    /* Check whether data has been read in */

    if (NDAT < 1)
	{
	errtype("ERROR, no data read in");
	clear_command();
	return;
	}

    /* Check whether model has been read in */

    nomodel = NSURF+NBULK < 1;
    if(nomodel)
	{
	errtype("ERROR, no model read in");
	clear_command();
	return;
	}

    /* Check whether there are not too many fit parameters */

    numpar = nsf+NDISTOT+NDWTOT+NDWTOT2+NOCCTOT;
    if (numpar > MAXFIT)
	{
	errtype("ERROR, number of fitting parameters exceeds MAXFIT");
	clear_command();
	return;
	}

    /* total number of parameters */
    ntot = init_fitpar();

    /* Add fake data point when Keating energy is added in the
       least-squares minimization */

    if (KEAT_PLUS_CHI)
	{
	if (NDAT == MAXDATA)
	    {
	    errtype("ERROR, too many data points to add Keating");
	    KEAT_PLUS_CHI = FALSE;
	    }
	else
	    {
	    NDAT++;
	    HDAT[NDAT-1] = 100.;
	    KDAT[NDAT-1] = 0.;
	    LDAT[NDAT-1] = 0.1;
	    FDAT[NDAT-1] = 0.;
	    ERRDAT[NDAT-1] = 1.;
	    }
	}

    /* Compute weight of data points */

    for (i = 0; i < NDAT; i++)
	{
	if (ERRDAT[i] > 1e-5)
	    FWEIGHT[i] = 1/sqr(ERRDAT[i]);
	else
	    FWEIGHT[i] = 0.;
	}

    /* Generate array with index numbers of (hkl)-points to be used as
       x-values of fit */

    for (i = 0; i < NDAT; i++)
	index[i] = (float) i;

#ifdef SPEEDUP

    /* Calculate and store constant values, in order to speed up fitting */
    f_calc_fit_init();

#endif /* SPEEDUP */

    /*
       Allow user to manipulate / plot parameters. 
       rc is the flag (return code) for the minimization method to be used
     */

    rc = set_fitpar("ROD.FIT>",
	ntot,
	FITPAR, FITMIN, FITMAX,
	FITPEN, FIXPAR,
	FITTXT[0], TITLE_LENGTH,
	index,FDAT,FWEIGHT,NDAT, ffit
	);

    /* Get number of free parameters and make array with free parameter
       numbers */

    nfree = 0;
    for (i = 0; i < ntot; i++)
	{
	if (!FIXPAR[i])
	    {
	    freepar[nfree] = i+1;
	    nfree++;
	    }
	}

    /* Prepare penalty factors for lsqfit: */

    for (i=0; i<ntot; i++)
	penalties[i] = (FITPEN[i] == TRUE ? LARGE_PENALTY : NO_PENALTY);

    /* Do the fit */

    if (rc != LSQ_NOFIT)
	rc = lsqfit(index,FDAT,FWEIGHT,NDAT,FITPAR,ntot,
	    freepar,nfree,FITERR,FITMIN,FITMAX,penalties,&chisqr,ffit,
#ifdef ASAROD
	    ffit_asa,
#endif /* ASAROD */
	    FITTXT[0], TITLE_LENGTH, 0, rc, 1);

    /* Copy values in array 'FITPAR' to corresponding model parameters and
       update structure model */

    for(i=0; i<ntot; i++)
	FITPEN[i] = (penalties[i] != 0 ? TRUE : FALSE);

    update_par();
    update_model();

    /* Remove fake data point used for Keating energy */

    if (KEAT_PLUS_CHI) NDAT--;

    /* Calculate theoretical structure factors for all data points */

    for (i = 0; i < NDAT; i++)
	{
	HTH[i] = HDAT[i];
	KTH[i] = KDAT[i];
	LTH[i] = LDAT[i];
	f_calc(HDAT[i],KDAT[i],LDAT[i],ATTEN,LBR[i],
	       &FTH[0][i],&FTH[1][i],&FTH[2][i],&PHASE[i]);
	}
    NTH = NDAT;

    }

/***************************************************************************/
void ffit(float x, float a[], float *y, float dyda[], int na, int derivatives)
/***************************************************************************/

    /*
    Function called by fitting routines to compute y-values and, optionally,
    derivatives with respect to fitting parameters.
    */

{

    int i,j;
    int nsf = NSF;
    float fbulk,fsurf,phase;

    SCALE = a[0];
    SCALE2= a[1];
    BETA = a[2];
    SURFFRAC = a[3];
    for (j = 0; j < NDISTOT; j++)
	{
	DISPL[j] = a[j+nsf];
	}
    for (j = 0; j < NDWTOT; j++)
	{
	DEBWAL[j] = a[j+nsf+NDISTOT];
	}
    for (j = 0; j < NDWTOT2; j++)
	{
	DEBWAL2[j] = a[j+nsf+NDISTOT+NDWTOT];
	}
    for (j = 0; j < NOCCTOT; j++)
	{
	OCCUP[j] = a[j+nsf+NDISTOT+NDWTOT+NDWTOT2];
	}
    
    /* Calculate structure model with new parameters */

    update_model();

    /* Calculate theoretical structure factor */

    i = (int) x; /* this is the index of the current data point */
    LBRAGG = LBR[i];

    /* Try to add lattice energy */
    if (HDAT[i] > 99.)
	{
	*y = sqrt(energy_calc());
	}
    else
	{
#ifdef SPEEDUP

	*y = f_calc_fit(HDAT[i],KDAT[i],LDAT[i],LBR[i],Q_PAR[i],Q_PERP[i],
	    FAT[i],RE_BULK[i],IM_BULK[i],FBULK_SQR[i]);
#else
	f_calc(HDAT[i],KDAT[i],LDAT[i],ATTEN,LBR[i],&fbulk,&fsurf,y,&phase);
#endif /* SPEEDUP */
	}

    dyda=dyda; na=na; derivatives=derivatives; /* prevent compiler warning */

}

/***************************************************************************/
int init_fitpar(void)
/***************************************************************************/

/*
   Copy all fitting parameters to one array FITPAR, get starting values, and
   return the total number of fit parameters
 */

{

    int i,ndx,ntot;
    int nsf = NSF;

    FITPAR[0] = SCALE;
    FITMIN[0] = SCALELIM[0];
    FITMAX[0] = SCALELIM[1];
    FITPEN[0] = SCALEPEN;

    FITPAR[1] = SCALE2;
    FITMIN[1] = SCALE2LIM[0];
    FITMAX[1] = SCALE2LIM[1];
    FITPEN[1] = SCALE2PEN;

    FITPAR[2] = BETA;
    FITMIN[2] = BETALIM[0];
    FITMAX[2] = BETALIM[1];
    FITPEN[2] = BETAPEN;

    FITPAR[3] = SURFFRAC;
    FITMIN[3] = SURFFRACLIM[0];
    FITMAX[3] = SURFFRACLIM[1];
    FITPEN[3] = SURFFRACPEN;

    sprintf(FITTXT[0],"Scale");
    sprintf(FITTXT[1],"Scale2");
    sprintf(FITTXT[2],"Beta");
    sprintf(FITTXT[3],"Surface fraction");

    for (i = 0; i < NDISTOT; i++)
	{
	ndx = i+nsf;
	FITPAR[ndx] = DISPL[i];
	FITMIN[ndx] = DISPLLIM[i][0];
	FITMAX[ndx] = DISPLLIM[i][1];
	FITPEN[ndx] = DISPLPEN[i];
	sprintf(FITTXT[ndx],"Displacement %2d",i+1);
	}
    for (i = 0; i < NDWTOT; i++)
	{
	ndx = i+nsf+NDISTOT;
	FITPAR[ndx] = DEBWAL[i];
	FITMIN[ndx] = DEBWALLIM[i][0];
	FITMAX[ndx] = DEBWALLIM[i][1];
	FITPEN[ndx] = DEBWALPEN[i];
	sprintf(FITTXT[ndx],"In-plane DW %2d",i+1);
	}
    for (i = 0; i < NDWTOT2; i++)
	{
	ndx = i+nsf+NDISTOT+NDWTOT;
	FITPAR[ndx] = DEBWAL2[i];
	FITMIN[ndx] = DEBWAL2LIM[i][0];
	FITMAX[ndx] = DEBWAL2LIM[i][1];
	FITPEN[ndx] = DEBWAL2PEN[i];
	sprintf(FITTXT[ndx],"Out-of-plane DW %2d",i+1);
	}
    for (i = 0; i < NOCCTOT; i++)
	{
	ndx = i+nsf+NDISTOT+NDWTOT+NDWTOT2;
	FITPAR[ndx] = OCCUP[i];
	FITMIN[ndx] = OCCUPLIM[i][0];
	FITMAX[ndx] = OCCUPLIM[i][1];
	FITPEN[ndx] = OCCUPPEN[i];
	sprintf(FITTXT[ndx],"Occupancy %2d",i+1);
	}

    /* total number of parameters */
    ntot = nsf+NDISTOT+NDWTOT+NDWTOT2+NOCCTOT;

    return (ntot);

}

/***************************************************************************/
void update_par(void)
/***************************************************************************/

/*
   Copy values in array 'FITPAR' to corresponding model parameters
 */

{
    int i,ndx;
    int nsf = NSF;

    SCALE = FITPAR[0];
    SCALELIM[0] = FITMIN[0];
    SCALELIM[1] = FITMAX[0];
    SCALEPEN = FITPEN[0];

    SCALE2 = FITPAR[1];
    SCALE2LIM[0] = FITMIN[1];
    SCALE2LIM[1] = FITMAX[1];
    SCALE2PEN = FITPEN[1];

    BETA = FITPAR[2];
    BETALIM[0] = FITMIN[2];
    BETALIM[1] = FITMAX[2];
    BETAPEN = FITPEN[2];

    SURFFRAC = FITPAR[3];
    SURFFRACLIM[0] = FITMIN[3];
    SURFFRACLIM[1] = FITMAX[3];
    SURFFRACPEN = FITPEN[3];

    for (i = 0; i < NDISTOT; i++)
	{
	ndx = i+nsf;
	DISPL[i] = FITPAR[ndx];
	DISPLLIM[i][0] = FITMIN[ndx];
	DISPLLIM[i][1] = FITMAX[ndx];
	DISPLPEN[i] = FITPEN[ndx];
	}
    for (i = 0; i < NDWTOT; i++)
	{
	ndx = i+nsf+NDISTOT;
	DEBWAL[i] = FITPAR[ndx];
	DEBWALLIM[i][0] = FITMIN[ndx];
	DEBWALLIM[i][1] = FITMAX[ndx];
	DEBWALPEN[i] = FITPEN[ndx];
	}
    for (i = 0; i < NDWTOT2; i++)
	{
	ndx = i+nsf+NDISTOT+NDWTOT;
	DEBWAL2[i] = FITPAR[ndx];
	DEBWAL2LIM[i][0] = FITMIN[ndx];
	DEBWAL2LIM[i][1] = FITMAX[ndx];
	DEBWAL2PEN[i] = FITPEN[ndx];
	}
    for (i = 0; i < NOCCTOT; i++)
	{
	ndx = i+nsf+NDISTOT+NDWTOT+NDWTOT2;
	OCCUP[i] = FITPAR[ndx];
	OCCUPLIM[i][0] = FITMIN[ndx];
	OCCUPLIM[i][1] = FITMAX[ndx];
	OCCUPPEN[i] = FITPEN[ndx];
	}
}

#ifdef ASAROD
/***************************************************************************/
float ffit_asa(float a[], int *cost_flag)
/***************************************************************************/

    /*
    Function called by adaptive simulated annealing fitting routine.

    a[]:        array with parameter values
    *cost_flag: some sort of flag
    return:     chi^2 value
    */

    {

    int i,j,imax;
    float chisqr,y;
    float fbulk,fsurf,phase;


    SCALE = a[0];
    SCALE2= a[1];
    BETA = a[2];
    SURFFRAC = a[3];
    for (j = 0; j < NDISTOT; j++)
	{
	DISPL[j] = a[j+NSF];
	}
    for (j = 0; j < NDWTOT; j++)
	{
	DEBWAL[j] = a[j+NSF+NDISTOT];
	}
    for (j = 0; j < NDWTOT2; j++)
	{
	DEBWAL2[j] = a[j+NSF+NDISTOT+NDWTOT];
	}
    for (j = 0; j < NOCCTOT; j++)
	{
	OCCUP[j] = a[j+NSF+NDISTOT+NDWTOT+NDWTOT2];
	}

    /* Calculate structure model with new parameters */

    update_model();

    /*
       Calculate chi^2 value

       In order to combine the levenberg algorithm with a lattice energy
       calculation, a fake data point is added. This is not necessary here, and
       thus the number of data points needs to be reduced by one if this option
       is on.
     */

    chisqr = 0.;
    if (KEAT_PLUS_CHI)
	imax = NDAT-1;
    else
	imax = NDAT;

    for (i = 0; i < imax; i++)
	{
#ifdef SPEEDUP
	y = f_calc_fit(HDAT[i],KDAT[i],LDAT[i],LBR[i],Q_PAR[i],Q_PERP[i],
	    FAT[i],RE_BULK[i],IM_BULK[i],FBULK_SQR[i]);
#else
	f_calc(HDAT[i],KDAT[i],LDAT[i],ATTEN,LBR[i],&fbulk,&fsurf,&y,&phase);
#endif /* SPEEDUP */
	chisqr += sqr(y-FDAT[i])*FWEIGHT[i];
	}

    /* Add lattice energy to chi^2 if requested (easy in asa!) */

    if (KEAT_PLUS_CHI)
	chisqr += sqrt(energy_calc());

    *cost_flag = TRUE;
    return(chisqr);
    }
#endif /* ASAROD */
