#define VFSR_ID \
"/* $Id: vfsr.c,v 1.4 2007/03/16 14:52:37 wilcke Exp $ */"

/* Quite drastically modified version for ROD-ANA of the Lester Ingberg
adaptive simulated annealing (asa) code. An early version of the code is
used (version 6.38 for 29 Dec 1992), when the name "very fast simulated
reannealing (vfsr)" was still used. */

/*
Update 19/01/2004 R. Wilcke (wilcke@esrf.fr) 
        removed declarations of fopen() (is declared in "stdio.h").
Update 13/08/2003 E. Vlieg (vlieg@sci.kun.nl)
	Improve test for repeating chi^2 values as criterium for exiting ASA.
Update 28/10/2001 E. Vlieg (vlieg@sci.kun.nl)
	Use USER_INITIAL_PARAMETERS as flag set by user.
Update 21/10/2001 E. Vlieg (vlieg@sci.kun.nl)
	Add test in reanneal() when calculating new index_cost_acceptances:
	the rounding sometimes yields a zero, which gives a very high
	annealing temperature.
	Asign correct memory for all parameters (thus remove the extra bytes
        and correct curvatures in particular.
Update 14/10/2001 E. Vlieg (vlieg@sci.kun.nl)
	Remove HAVE_ANSI flags, because we always use ANSI.
	Add (temporary) asatest file.
	Changed accept_new_state to int in order to use it as flag.
*/

/* vfsr.h for Very Fast Simulated Reannealing */

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <lsqfit.h>
#include <time.h>
//#include <dos.h>
#ifdef ASA
/* INCLUDE THIS FILE IF YOUR C COMPILER ONLY
   ACCEPTS IDENTIFIERS OF LESS THAN 32 CHARACTERS. */
/* #include longname.h */

#define TRUE	1
#define FALSE	0

#define INTEGER_TYPE	1
#define REAL_TYPE	0

//#define NPRINT 2 	/* frequency of printing annealing status */

#define ONE ((float) 1.0)
#define TWO ((float) 2.0)
#define TEN ((float) 10.0)
#define HALF ((float) 0.5)

#define NORMAL_EXIT		0
#define P_TEMP_TOO_SMALL	1
#define C_TEMP_TOO_SMALL	2
#define COST_REPEATING		3

 /* DEFAULT PARAMETERS SETTINGS */
#ifndef HAVE_ANSI
#define HAVE_ANSI TRUE
#endif

//#define VFSR_PRINT TRUE
#ifndef VFSR_PRINT
#define VFSR_PRINT FALSE
#endif

#ifndef VFSR_OUT
#define VFSR_OUT "vfsr_out.txt"
#endif

//#ifndef USER_INITIAL_PARAMETERS
//#define USER_INITIAL_PARAMETERS TRUE
//#endif


//#ifndef ACCEPTED_TO_GENERATED_RATIO	/* determines when to test */
//#define ACCEPTED_TO_GENERATED_RATIO ACCEPTED_TO_GENERATED_RATIO_ASA
#define ACCEPTED_TO_GENERATED_RATIO 1.0E-6
//#endif

//#ifndef LIMIT_ACCEPTANCES	/* max number of acceptances */
//#define LIMIT_ACCEPTANCES 1000
//#endif

//#ifndef TEMPERATURE_RATIO_SCALE
//#define TEMPERATURE_RATIO_SCALE 1.0E-4
/*#define TEMPERATURE_RATIO_SCALE 1.0E-5*/
//#endif

//#ifndef TEMPERATURE_ANNEAL_SCALE
//#define TEMPERATURE_ANNEAL_SCALE 100 /* LIMIT_ACCEPTANCES/TEN ACCEPTED_LIMIT*/
//#endif

//#ifndef COST_PARAMETER_SCALE
//#define COST_PARAMETER_SCALE 1.0
//#endif

//#ifndef TESTING_FREQUENCY_MODULUS
//#define TESTING_FREQUENCY_MODULUS 100
//#endif

#ifndef MAXIMUM_REANNEAL_INDEX
#define MAXIMUM_REANNEAL_INDEX 1.0E8
#endif

#ifndef REANNEAL_RESCALE
#define REANNEAL_RESCALE 3.0
#endif

#ifndef INITIAL_PARAMETER_TEMPERATURE
#define INITIAL_PARAMETER_TEMPERATURE 1.0
#endif

#ifndef NUMBER_COST_SAMPLES
#define NUMBER_COST_SAMPLES 10
#endif

#ifndef MAXIMUM_COST_REPEAT
#define MAXIMUM_COST_REPEAT 20
#endif

#ifndef DELTA_X			/* delta of derivatives in cost_derivatives */
#define DELTA_X 0.001
#endif

#ifndef INCLUDE_INTEGER_PARAMETERS	/* in derivative calculations */
#define INCLUDE_INTEGER_PARAMETERS FALSE
#endif

#ifndef SMALL_FLOAT
#define SMALL_FLOAT 1.0E-24
#endif

#ifndef COST_PRECISION		/* precision at MAXIMUM_COST_REPEAT */
//#define COST_PRECISION SMALL_FLOAT
#define COST_PRECISION 0.05  /* EV: enough for chi^2 in ROD */
#endif

#ifndef IO_PROTOTYPES
#define IO_PROTOTYPES FALSE
#endif

#ifndef ACTIVATE_REANNEAL
#define ACTIVATE_REANNEAL TRUE
#endif

 /* essential DEFINES and MACROS */
#define MAXIMUM_EXPONENT ((float) 2.302585/((float) SMALL_FLOAT))

 /* macros */

 /* VFOR
    is a simple macro to iterate on each parameter index. */
#define VFOR(index_v) \
 for (index_v = 0; index_v < number_par; ++index_v)

 /* EXPONENT_CHECK
    checks that an exponent x is within a valid range and,
    if not, reduces its magnitude to fit in the range. */
#define EXPONENT_CHECK(x) \
 (fabs(x) < MAXIMUM_EXPONENT ? x : ((x * MAXIMUM_EXPONENT)/fabs(x)))

 /* PARAMETER_RANGE_TOO_SMALL(x)
    checks if the range of parameter x is too small to work with
    If user_cost_function were modified to pass parameter ranges,
    this test could be (and has been) used to adaptively bypass
    some parameters, e.g., depending on constraints. */
#define PARAMETER_RANGE_TOO_SMALL(x) \
 (fabs(parameter_minimum[x] - parameter_maximum[x]) < (float) SMALL_FLOAT)

 /* INTEGER_PARAMETER(x)
    determines if the parameter is an integer type. */
#define INTEGER_PARAMETER(x) (parameter_type[x] == INTEGER_TYPE)

 /* ROW_COL_INDEX (i, j)
    converts from row i, column j to an index. */
#define ROW_COL_INDEX(i, j) (i + number_par * j)

 /* The State of the system, its parameters and their resulting function
    value */
typedef struct
    {
    float cost;
    float *parameter;
    }
    STATE;

 /* The 3 states that are kept track of during the annealing process */
STATE current_generated_state, last_saved_state, best_generated_state;

 /* The array of parameter bounds */
float *parameter_minimum, *parameter_maximum;

 /* The array of tangents (absolute value of the numerical derivatives),
    and the maximum |tangent| of the array */
float *tangents, maximum_tangent;

 /* The parameter curvature/covariance about the best state */
float *curvature;

 /* ratio of acceptances to generated points - determines when to
    test/reanneal */
float accepted_to_generated_ratio;

 /* temperature parameters */
double temperature_scale, temperature_scale_parameters;
 /* relative scalings of cost and parameters to temperature_scale */
double temperature_scale_cost;
double *current_parameter_temperature, *initial_parameter_temperature;
double current_cost_temperature, initial_cost_temperature;
double temperature_rescale_power;	/* used when applying REANNEAL_RESCALE */
double log_new_temperature_ratio;	/* current temp = initial temp *
					   exp(log_new_temperature_ratio) */
float one_over_number_parameters;	/* 1/number_parameters */
int index_exit_v;		/* information for vfsr_exit */

 /* flag to determine if curvatures should be calculated */
int curvature_flag,  number_parameters, number_par;

 /* counts of generated states and acceptances */
long int *index_parameter_generations;
long int number_generated, best_number_generated_saved;
long int recent_number_generated, number_accepted;
long int recent_number_acceptances, index_cost_acceptances;
long int number_acceptances_saved, best_number_accepted_saved;

/* Flag indicates that the parameters generated were
   invalid according to the cost function validity criteria. */
int valid_state_generated_flag;
long int number_invalid_generated_states;
//long int steps_print;
/* parameter type is real or integer */
int *parameter_type;

 /* file ptr to output file */
#if VFSR_PRINT
FILE *ptr_vfsr_out;
#endif

/* for debugging purposes */
//FILE *asatest;
//static int STARTASA = TRUE;

 /* system function prototypes */
//#if HAVE_ANSI
#if VFSR_PRINT
//#if 0				/* This block gives trouble under some Ultrix */
//int fprintf(FILE * fp, char *string,...);
//int fflush(FILE * fp);
//int fclose(FILE * fp);
//void exit(int code);
//#endif
//#if IO_PROTOTYPES
//int fprintf();
//int fflush();
//int fclose();
//void exit();
//#endif
#endif

int accept_new_state(RandomASA user_random_generator);
void generate_new_state(RandomASA user_random_generator);
void reanneal(void);
void cost_derivatives(FitFunctionASA user_cost_function);
float generate_vfsr_state(float temp,
			   RandomASA user_random_generator);

float vfsr(
	FitFunctionASA user_cost_function,
	RandomASA user_random_generator,
	int number_parameters,int number_param, int *parameter_type,
	float *parameter_initial_final,
	float *parameter_minimum, float *parameter_maximum,
	int *exit_status);

void vfsr_exit(FitFunctionASA user_cost_function,
	       float *parameter_initial_final,
	       float *final_cost,
	       int *exit_status);
/*changed by Daniel Kaminski*/

void print_state(void);

/***********************************************************************
* Author: Lester Ingber, Bruce Rosen (copyright) (c)
* Date	 5 Nov 92
* Procedure: vfsr
* Parameters:
* Inputs:
*     float (*user_cost_function) ()	- user's cost function
*     float (*user_random_generator) ()	-user's rng
*     int g_number_parameters		- number of parameters
*     int *g_parameter_type		- type of parameter
*     float *parameter_initial_final	- initial and final parameters
*     float *g_parameter_minimum	- minimum parameter values
*     float *g_parameter_maximum	- maximum parameter values
*     float *g_tangents		- holds 1st derivatives
*     float *g_curvature		- holds 2nd derivatives
*     int *exit_status                  - reason for exiting
* Outputs:
*     float *parameter_initial_final	- holds final parameters
* Global Variables:
* Returns:
*     float final_cost		        - best cost found
* Calls:
* Description:
*	This procedure implements the full VFSR function optimization.
* Local Variables:
*	int index_cost_constraint	- index initial cost functions
*					- to set initial temperature
*	int temp_var_int		- integer temporary value
*	int index_v			- iteration index
*	float index_cost_repeat	- test when MAXIMUM_COST_REPEAT
 *      float sampled_cost_sum	        - temporary initial costs
*	float tmp_var_db1, tmp_var_db2	- temporary floats
* Portability:
* Other:
***********************************************************************/

float vfsr(			/* return final_cost */
	FitFunctionASA user_cost_function,
	RandomASA user_random_generator,
	int g_number_parameters,	/* number of parameters */
	int g_number_param,      /*parameters to calculations */
	int *g_parameter_type,	/* INTEGER_TYPE, REAL_TYPE */
	float *parameter_initial_final,	/* initial and final parameters */
	float *g_parameter_minimum,	/* minimum parameter values */
	float *g_parameter_maximum,	/* maximum parameter values */
	//  float *g_tangents,	/* numerical 1st derivatives of cost */
	//  float *g_curvature,	/* numerical 2nd derivatives of cost */
	int *exit_status /* exit code */
	)
{
    int index_cost_constraint,	/* index initial cost functions averaged
				   to set initial temperature */
    temp_var_int,		/* integer temporary value */
    index_v;			/* iteration index */
    float index_cost_repeat,	/* test COST_PRECISION when this reaches
				   MAXIMUM_COST_REPEAT */
    sampled_cost_sum,		/* temporary initial costs */
    final_cost,			/* best cost to return to user */
    tmp_var_db1, tmp_var_db2;	/* temporary floats */
//    struct	time t;

//    if (STARTASA)
//	{
//	STARTASA = FALSE;
//	if ((asatest = fopen("asatest.txt","w+")) == NULL)
//		errtype("Failed to open asatest file");
//	}

    /* initializations */

/* g_... are defined to make local information passed to vfsr() by
   the user into global variables useful for internal vfsr() routines.
   This was considered a reasonable tradeoff to passing several
   parameters between several routines. */

    number_parameters=g_number_param;
    number_par=g_number_parameters; /*number all parameters */
    parameter_type = g_parameter_type;
    parameter_minimum=g_parameter_minimum;
    parameter_maximum = g_parameter_maximum;

/*Changed by Daniel Kaminski*/
    // tangents = g_tangents;
    // curvature = g_curvature;
    //steps_print=0;

#if VFSR_PRINT
    /* open the output file */
    ptr_vfsr_out = fopen(VFSR_OUT, "w");
#endif

    /* set indices and counts to 0 */
    best_number_generated_saved =
	number_generated =
	recent_number_generated =
	recent_number_acceptances = 0;
    index_cost_acceptances =
	best_number_accepted_saved =
	number_accepted =
	number_acceptances_saved = 1;
    index_cost_repeat = 0;

    /* do not calculate curvatures initially */
    curvature_flag = FALSE;

    /* allocate storage for all parameters */
    current_generated_state.parameter =
	(float *) calloc(number_par, sizeof(float));
    last_saved_state.parameter =
	(float *) calloc(number_par, sizeof(float));
    best_generated_state.parameter =
	(float *) calloc(number_par, sizeof(float));
    current_parameter_temperature =
	(double *) calloc(number_par, sizeof(double));
    initial_parameter_temperature =
	(double *) calloc(number_par, sizeof(double));
    index_parameter_generations =
	(long int *) calloc(number_par, sizeof(long int));
    /* Add these parameters: omitted in original version?
    Daniel Kaminski 3/4/01 */
    tangents =
	(float *) calloc(number_par, sizeof(float));
/* this seems a rather strange memory allocation */
//    curvature =
//	(float *) calloc(ROW_COL_INDEX(index_v+1, index_v+1)+1, sizeof(float));
    curvature =
	(float *) calloc(number_par*number_par, sizeof(float));

    /* set the user's initial parameters to the initial system state */
    VFOR(index_v) {
	current_generated_state.parameter[index_v] =
	    parameter_initial_final[index_v];
    }

/* Change by Daniel Kaminski*/

//#if VFSR_PRINT
    /* print header information as defined by user */
//    printf(
//	    "\t\tVERY FAST SIMULATED REANNEALING\n\n");
//
//   printf( "%s\n\n", VFSR_ID);

//   printf( "HAVE_ANSI = %d\n",
//	    (int) HAVE_ANSI);
//   printf( "VFSR_PRINT = %d\n",
//	    (int) VFSR_PRINT);
//   printf( "VFSR_OUT = %s\n",
//	VFSR_OUT);
//   printf( "USER_INITIAL_PARAMETERS = %d\n",
//	    (int) USER_INITIAL_PARAMETERS);
//   printf( "ACCEPTED_TO_GENERATED_RATIO = %g\n",
//	    (float) ACCEPTED_TO_GENERATED_RATIO);
//   printf( "LIMIT_ACCEPTANCES = %d\n",
//	    (int) LIMIT_ACCEPTANCES);
//   printf( "TEMPERATURE_RATIO_SCALE = %g\n",
//	    (float) TEMPERATURE_RATIO_SCALE);
//   printf( "TEMPERATURE_ANNEAL_SCALE = %g\n",
 //	    (float) TEMPERATURE_ANNEAL_SCALE);
//   printf( "COST_PARAMETER_SCALE = %g\n",
////	    (float) COST_PARAMETER_SCALE);
//   printf( "TESTING_FREQUENCY_MODULUS = %d\n",
//	    (int) TESTING_FREQUENCY_MODULUS);
//   printf( "MAXIMUM_REANNEAL_INDEX = %d\n",
//	    (int) MAXIMUM_REANNEAL_INDEX);
//   printf( "REANNEAL_RESCALE = %g\n",
//	    (float) REANNEAL_RESCALE);
//   printf( "INITIAL_PARAMETER_TEMPERATURE = %g\n",
//	    (float) INITIAL_PARAMETER_TEMPERATURE);
//   printf( "NUMBER_COST_SAMPLES = %d\n",
//	    (int) NUMBER_COST_SAMPLES);
//   printf( "MAXIMUM_COST_REPEAT = %d\n",
//	    (int) MAXIMUM_COST_REPEAT);
//   printf( "DELTA_X = %g\n",
//	    (float) DELTA_X);
//   printf( "INCLUDE_INTEGER_PARAMETERS = %d\n",
//	    (int) INCLUDE_INTEGER_PARAMETERS);
//   printf( "SMALL_FLOAT = %g\n",
//	    (float) SMALL_FLOAT);
//   printf( "COST_PRECISION = %g\n",
//	    (float) COST_PRECISION);
//   printf( "IO_PROTOTYPES = %d\n",
//	    (int) IO_PROTOTYPES);
//   printf( "ACTIVATE_REANNEAL = %d\n",
//	    (int) ACTIVATE_REANNEAL);
//   printf( "\n\n");

   //printf( "number_parameters = %d\n",number_parameters);
   printf ("n = %4d; c_cost = %6.3f; c_par = %6.3f\n",number_parameters,
	-log(TEMPERATURE_RATIO_SCALE)*
	exp(-log(TEMPERATURE_ANNEAL_SCALE)/number_parameters)*
	COST_PARAMETER_SCALE,
	-log(TEMPERATURE_RATIO_SCALE)*
	exp(-log(TEMPERATURE_ANNEAL_SCALE)/number_parameters));
   printf ("%10s %10s %10s %10s %10s %10s %10s\n","T_anneal","T_par_min",
	"T_par_max","N_gener.","N_accep.","curr. cost","best cost");

   /* write current time to asa test file */
   //gettime(&t);
   //fprintf(asatest,"%2d:%02d ",t.ti_hour, t.ti_min);
   if (LOGFILE)
	{
	fprintf (FIT_LOGFILE,"n= %2d c_cost= %6.3f c_par = %6.3f ",
	    number_parameters,
	    -log(TEMPERATURE_RATIO_SCALE)*
	    exp(-log(TEMPERATURE_ANNEAL_SCALE)/number_parameters)*
	    COST_PARAMETER_SCALE,
	    -log(TEMPERATURE_RATIO_SCALE)*
	    exp(-log(TEMPERATURE_ANNEAL_SCALE)/number_parameters));
	}

//#endif

    VFOR(index_v) {
	/* set all starting and current temperatures */
	current_parameter_temperature[index_v] =
	    initial_parameter_temperature[index_v] =
	    ((float) INITIAL_PARAMETER_TEMPERATURE);

	/* set all states to the initial parameter values */
	best_generated_state.parameter[index_v] =
	    last_saved_state.parameter[index_v] =
	    current_generated_state.parameter[index_v];

#if VFSR_PRINT
//	fprintf(ptr_vfsr_out, " %-8d %-18.5g %-17.5g %-15.5g %-7d\n",
//		index_v,
//		parameter_minimum[index_v],
//		parameter_maximum[index_v],
//		current_generated_state.parameter[index_v],
//		parameter_type[index_v]);
/*	printf(" %-8d %-18.5g %-17.5g %-15.5g %-7d\n",
		index_v,
		parameter_minimum[index_v],
		parameter_maximum[index_v],
		current_generated_state.parameter[index_v],
		parameter_type[index_v]);
*/
#endif
    }
#if VFSR_PRINT
//    fprintf(ptr_vfsr_out, "\n\n");
#endif

    /* set the initial index of parameter generations to 1 */
    VFOR(index_v) {
	/* ignore parameters that have too small a range */
	if (PARAMETER_RANGE_TOO_SMALL(index_v))
	    continue;
	index_parameter_generations[index_v] = 1;
    }

    /* calculate the average cost over samplings of the cost function */
    sampled_cost_sum = 0.0;
    for (index_cost_constraint = 0;
	 index_cost_constraint < ((int) NUMBER_COST_SAMPLES);
	 ++index_cost_constraint) {
	number_invalid_generated_states = 0;
	do {
	    ++number_invalid_generated_states;
	    valid_state_generated_flag = TRUE;
	    generate_new_state(user_random_generator);
	    current_generated_state.cost =
		(*user_cost_function)
		(current_generated_state.parameter,
		 &valid_state_generated_flag);
	} while (valid_state_generated_flag == FALSE);
	--number_invalid_generated_states;
	sampled_cost_sum += fabs(current_generated_state.cost);
    }

    /* set all cost temperatures to the average cost */
    initial_cost_temperature = current_cost_temperature =
	sampled_cost_sum / ((float) NUMBER_COST_SAMPLES);

    /* establish an initial current state */

    /* user's initial guess of parameters */
    VFOR(index_v) {
	current_generated_state.parameter[index_v] =
	    parameter_initial_final[index_v];
    }

//#if USER_INITIAL_PARAMETERS	/* if using user's initial parameters */
    if (USER_INITIAL_PARAMETERS == TRUE)
	{
	valid_state_generated_flag = TRUE;
	current_generated_state.cost =
	    (*user_cost_function)
	    (current_generated_state.parameter,
	    &valid_state_generated_flag);
	}

#if VFSR_PRINT
//    if (valid_state_generated_flag == FALSE)
//	fprintf(ptr_vfsr_out,
//		"user's initial parameters generated \
//FALSE valid_state_generated_flag\n");
#endif
//#else				/* let vfsr generate valid initial parameters */
    else
	{
	do {
	    ++number_invalid_generated_states;
	    valid_state_generated_flag = TRUE;
	    generate_new_state(user_random_generator);
	    current_generated_state.cost =
		(*user_cost_function)
		(current_generated_state.parameter,
		&valid_state_generated_flag);
	} while (valid_state_generated_flag == FALSE);
	--number_invalid_generated_states;
	}
//#endif				/* USER_INITIAL_PARAMETERS */

    /* set all states to the last one generated */
    VFOR(index_v) {
	/* ignore parameters that have too small a range */
	if (PARAMETER_RANGE_TOO_SMALL(index_v))
	    continue;
	best_generated_state.parameter[index_v] =
	    last_saved_state.parameter[index_v] =
	    current_generated_state.parameter[index_v];
    }

    /* set all costs to the last one generated */
    best_generated_state.cost = last_saved_state.cost =
	current_generated_state.cost;

    accepted_to_generated_ratio = ONE;

    /* compute 1/number_parameters for efficiency */
    one_over_number_parameters = ONE / number_parameters;
    temperature_rescale_power = ONE / pow(
					     (float) REANNEAL_RESCALE,
					     one_over_number_parameters);

    tmp_var_db1 = -log(((float) TEMPERATURE_RATIO_SCALE));

    /* compute temperature scales */
    tmp_var_db2 = log((float) TEMPERATURE_ANNEAL_SCALE);
    temperature_scale = tmp_var_db1 * exp(-tmp_var_db2
					  / (float) number_parameters);

    temperature_scale_parameters = temperature_scale;
    temperature_scale_cost = temperature_scale
	* (float) COST_PARAMETER_SCALE;

    /* do not calculate curvatures initially */
    curvature_flag = FALSE;

#if VFSR_PRINT
//    fprintf(ptr_vfsr_out,
//	    "temperature_scale = %10.5g\n", temperature_scale);
    fprintf(ptr_vfsr_out, "T_scale_parameters = %10.5g,\
	T_scale_cost = %10.5g\n",
	    temperature_scale_parameters, temperature_scale_cost);
    fprintf(ptr_vfsr_out, "\n");
    //print_state();
    fprintf (ptr_vfsr_out,"%10s %10s %10s %10s %10s\n","T_anneal",
	"N_gener.","N_accep.","curr. cost","best cost");
#endif

    /* reset the current cost and the number of generations performed */
    number_invalid_generated_states = 0;
    best_number_generated_saved =
	number_generated = recent_number_generated = 0;
    VFOR(index_v) {
	/* ignore parameters that have too small a range */
	if (PARAMETER_RANGE_TOO_SMALL(index_v))
	    continue;
	index_parameter_generations[index_v] = 1;
    }

    /* MAIN ANNEALING LOOP */

    while (number_accepted <= ((long int) LIMIT_ACCEPTANCES)) {

//////////////* CALCULATE NEW TEMPERATURES *///////////////////////////////

	/* calculate new parameter temperatures */
	VFOR(index_v) {
	    /* skip parameters with too small a range */
	    if (PARAMETER_RANGE_TOO_SMALL(index_v))
		continue;

	    log_new_temperature_ratio = -temperature_scale_parameters *
		pow((float) index_parameter_generations[index_v],
		    one_over_number_parameters);
	    /* check (and correct) for too large an exponent */
	    log_new_temperature_ratio =
		EXPONENT_CHECK(log_new_temperature_ratio);
	    current_parameter_temperature[index_v] =
		initial_parameter_temperature[index_v]
		* exp(log_new_temperature_ratio);

	    /* check for too small a parameter temperature */
	    if (exp(log_new_temperature_ratio) < (float) SMALL_FLOAT) {
		*exit_status = (int) P_TEMP_TOO_SMALL;
		index_exit_v = index_v;

/* Note that this abridged exit can hold and print old values
   of some variables, such as current_cost_temperature. */
		vfsr_exit(user_cost_function,
			  parameter_initial_final,
			  &final_cost,
			  exit_status);
		return (final_cost);
	    }
	}

	/* calculate new cost temperature */
	log_new_temperature_ratio =
	    -temperature_scale_cost * pow((float)
		     index_cost_acceptances, one_over_number_parameters);
	log_new_temperature_ratio =
	    EXPONENT_CHECK(log_new_temperature_ratio);
	current_cost_temperature = initial_cost_temperature
	    * exp(log_new_temperature_ratio);

	/* check for too small a cost temperature */
	if (exp(log_new_temperature_ratio) < (float) SMALL_FLOAT) {
	    *exit_status = (int) C_TEMP_TOO_SMALL;
/* Note that this abridged exit can hold and print old values
   of some variables. */
	    vfsr_exit(user_cost_function,
		      parameter_initial_final,
		      &final_cost,
		      exit_status);
	    return (final_cost);
	}

////////////////////////////////////////////////////////////////////////////
//////////////////* GENERATE NEW PARAMETERS *///////////////////////////////
////////////////////////////////////////////////////////////////////////////

   // printf( "x = %ld   parameter_range_v = %g \n ",  current_parameter_temperature[0], initial_parameter_temperature);


	/* generate a new valid set of parameters */
	do {
	    ++number_invalid_generated_states;
	    valid_state_generated_flag = TRUE;
	    generate_new_state(user_random_generator);
	    current_generated_state.cost =
		(*user_cost_function)
		(current_generated_state.parameter,
		 &valid_state_generated_flag);
	} while (valid_state_generated_flag == FALSE);
	--number_invalid_generated_states;

	/* ACCEPT/REJECT NEW PARAMETERS */

	/* decide to accept/reject the new state */
    if (accept_new_state(user_random_generator))    /* EV 19/9/01 */
	    {
	    if (number_accepted%ASA_NPRINT == 0) print_state();
	    /* try this here */
	    if (fabs(last_saved_state.cost
		     - best_generated_state.cost) <
		(float) COST_PRECISION) {
		++index_cost_repeat;
		if (index_cost_repeat == ((int) MAXIMUM_COST_REPEAT)) {
		    *exit_status = (int) COST_REPEATING;
/* Note that this abridged exit can hold and print old values
   of some variables. */
		    vfsr_exit(user_cost_function,
			      parameter_initial_final,
			      &final_cost,
			      exit_status);
		    return (final_cost);
		}
	    } else
		index_cost_repeat = 0;
#if VFSR_PRINT
	    fprintf(ptr_vfsr_out,"%10.2e %10ld %10ld %10.3f %10.3f",
		current_cost_temperature,
		number_generated,
		number_accepted,
		current_generated_state.cost,
		best_generated_state.cost);
//	    VFOR(index_v)  /* print all parameters for testing!! */
//		{
//		fprintf(ptr_vfsr_out," %9.2e",
//			current_generated_state.parameter[index_v]);
//		}
		fprintf(ptr_vfsr_out,"\n");
#endif
	    }

	/* calculate the ratio of acceptances to generated states */
	accepted_to_generated_ratio =
	    (float) (recent_number_acceptances + 1) /
	    (float) (recent_number_generated + 1);

	/* CHECK FOR NEW MINIMUM */

	if (current_generated_state.cost < best_generated_state.cost) {
	    /* NEW MINIMUM FOUND */

	    /* reset the recent acceptances and generated counts */
	    recent_number_acceptances = recent_number_generated = 0;
	    best_number_generated_saved = number_generated;
	    best_number_accepted_saved = number_accepted;

	    /* copy the current state into the best_generated state */
	    best_generated_state.cost = current_generated_state.cost;
	    VFOR(index_v) {
		/* ignore parameters that have too small a range */
		if (PARAMETER_RANGE_TOO_SMALL(index_v))
		    continue;
		best_generated_state.parameter[index_v] =
		    current_generated_state.parameter[index_v];
	    }

	    /* printout the new minimum state and value */
#if VFSR_PRINT
//	    fprintf(ptr_vfsr_out,      /* EV */
//		    "best_generated_state=%-12.5g\
//			number_accepted=%ld number_generated=%ld\n",
//		    best_generated_state.cost,
//		    number_accepted, number_generated);

/* hz: print refined parameters */

//     if ((float)(number_accepted)/prntstate - number_accepted/prntstate == 0) print_state
//();


#endif
	}
	/* PERIODIC TESTING/REANNEALING/PRINTING SECTION */
//if (number_accepted == LIMIT_ACCEPTANCES)
//{
//vfsr_exit(user_cost_function,parameter_initial_final,
//&final_cost,exit_status);
//return (final_cost);
//}

	temp_var_int = (int) (number_accepted %
			      ((long int) TESTING_FREQUENCY_MODULUS));
	if ((temp_var_int == 0 && number_acceptances_saved
	     == number_accepted)
	    || accepted_to_generated_ratio
	    < ((float) ACCEPTED_TO_GENERATED_RATIO)) {
	    if (accepted_to_generated_ratio
		< ((float) ACCEPTED_TO_GENERATED_RATIO))
		recent_number_acceptances = recent_number_generated = 0;

	    /* if best.cost repeats MAXIMUM_COST_REPEAT then exit */
	    /* is only tested with TESTING-FREQUENCY-MODULUS, thus it takes
	    a long time to exit */
//	    if (fabs(last_saved_state.cost
//		     - best_generated_state.cost) <
//		(float) COST_PRECISION) {
//		++index_cost_repeat;
//		if (index_cost_repeat == ((int) MAXIMUM_COST_REPEAT)) {
//		    *exit_status = (int) COST_REPEATING;
/* Note that this abridged exit can hold and print old values
   of some variables. */
//		    vfsr_exit(user_cost_function,
//			      parameter_initial_final,
//			      &final_cost,
//			      exit_status);
//		    return (final_cost);
//		}
//	    } else
//		index_cost_repeat = 0;

#if ACTIVATE_REANNEAL		/* set to FALSE to skip reannealing */
	    /* calculate tangents, not curvatures, to reanneal */
	    curvature_flag = FALSE;
	    cost_derivatives(user_cost_function);
	    reanneal();
#endif

/* Changed by Daniel Kaminski*/


//#if VFSR_PRINT


//steps_print = steps_print + 1;

//if ((steps_print % (long int) 1) == 0)
 //{
//  print_state();  /* no print at reanneal at the moment */
 //}

//	    fprintf(ptr_vfsr_out, "\n");
//	    fflush(ptr_vfsr_out);
//#endif
	}
    }
    /* FINISHED ANNEALING and MINIMIZATION */

    *exit_status = (int) NORMAL_EXIT;
    vfsr_exit(user_cost_function,
	      parameter_initial_final,
	      &final_cost,
	      exit_status);
    return (final_cost);
}

/***********************************************************************
* Author: Lester Ingber, Bruce Rosen (copyright) (c)
* Date	 5 Nov 92
* Procedure: vfsr_exit
*	This procedures cleans up the system and frees storage, and
*	copies the best parameters and cost into final_cost and
*	parameter_initial_final
* Parameters:
* Inputs:
*	float user_cost_function - user's cost function
	int *exit_status           - exit code
* Outputs:
*	float *parameter_initial_final   - final parameters state
*	float *final_cost - final cost at state parameter_initial_final
* Global Variables:
*	curvature_flag - reset to compute curvature
*	best_generated_state.parameter    - array of best parameters
*	current_generated_state.parameter - free this storage
*	last_saved_state.parameter        - free this storage
*	best_generated_state.parameter    - free this storage
*	current_parameter_temperature     - free this storage
*	initial_parameter_temperature     - free this storage
*	index_parameter_generations       - free this storage
*	ptr_vfsr_out - output file
* Returns: None
* Calls:
*	cost_derivatives()
*	print_state()
*	free()
*	fclose()
* Description:
* Local Variables:
*	int index_v - iteration index
* Portability:
* Other:
***********************************************************************/
//#if HAVE_ANSI
void vfsr_exit(
		  FitFunctionASA user_cost_function,	/* user's cost function */
		  float *parameter_initial_final,	/* initial and final
							   parameters */
		  float *final_cost,	/* best cost found to return */
		  int *exit_status /* exit code */ )
//#else
//void vfsr_exit(user_cost_function, parameter_initial_final,
//	       final_cost, exit_status)
//float (*user_cost_function) ();
//float *parameter_initial_final;
//float *final_cost;
//int *exit_status;
//#endif
{
    int index_v;		/* iteration index */

    /* calculate curvatures and tangents at best point */
    /* why would you want to do this? */
//    curvature_flag = TRUE;
//    cost_derivatives(user_cost_function);

    /* return final function minimum and associated parameters */
    *final_cost = best_generated_state.cost;
    VFOR(index_v) {
	parameter_initial_final[index_v] =
	    best_generated_state.parameter[index_v];
    }
    if (LOGFILE)
	{
	fprintf(FIT_LOGFILE,"N_gen= %8ld N_acc= %8ld ",
	    number_generated,number_accepted);
//	fprintf(FIT_LOGFILE,"final_cost= %10.5f ",*final_cost);
//	VFOR(index_v)
//	    {
//	    fprintf(FIT_LOGFILE,"%9.2e ",best_generated_state.parameter[index_v]);
//	    }
//	fprintf(FIT_LOGFILE,"\n");
	}

//#if VFSR_PRINT
 //   print_state();

    switch (*exit_status) {
    case 0:
	printf(
		"\n\n NORMAL_EXIT exit_status \n "
		);
	break;
    case 1:
	printf(
		"\n\n P_TEMP_TOO_SMALL exit_status \n "
		);
	printf(
			"\n current_parameter_temperature[%d] too small = %10.5g\n",
	      index_exit_v, current_parameter_temperature[index_exit_v]);
	break;
    case 2:
	printf(
		"\n\n C_TEMP_TOO_SMALL exit_status = %d\n",
		*exit_status);
	printf(
		"\n current_cost_temperature too small = %10.5g\n",
		current_cost_temperature);
	break;
    case 3:
	printf(
		"\n\n COST_REPEATING exit_status = %d\n",
		*exit_status);
	break;
    default:
	printf( "\n\n ERR: no exit code available = %d\n",
		*exit_status);
    }

//   printf(
//	    "final_cost = best_generated_state.cost = %-15.5g\n",
//	    *final_cost);
//    printf(
//	    "number_accepted at best_generated_state.cost = %ld\n",
//	    best_number_accepted_saved);
//    printf(
//	    "number_generated at best_generated_state.cost = %ld\n",
//	    best_number_generated_saved);
//#endif

    /* return unused space */
    free(current_generated_state.parameter);
    free(last_saved_state.parameter);
    free(best_generated_state.parameter);
    free(current_parameter_temperature);
    free(initial_parameter_temperature);
    free(index_parameter_generations);
    /* Add these parameters: omitted in original version?
    Daniel Kaminski 3/4/01 */
    free(tangents);
    free(curvature);

#if VFSR_PRINT
    fclose(ptr_vfsr_out);
#endif
}

/***********************************************************************
* Author: Lester Ingber, Bruce Rosen (copyright) (c)
* Date	 5 Nov 92
* Procedure: generate_new_state
* Parameters:
* Inputs:
*	float (*user_random_generator) () - the random number generator
* Outputs: None
* Global Variables:
* Returns: None
* Calls:
*	fabs                     - absolute value function
*	generate_vfsr_state      - to get a vfsr distributed state
* Description:
*	Generates a valid new state from the old state
* Local Variables:
*	int index_v		 - iteration index v
*	float x		 - the new parameter value
*	float parameter_v	 - the vth parameter value
*	float min_parameter_v 	 - the vth parameter lower bound
*	float max_parameter_v 	 - the vth parameter upper bound
*	float temperature_v	 - the vth parameter temperature
*	float parameter_range_v - the vth parameter range
* Portability:
* Other:
***********************************************************************/
 /* generate a new state from the old state that lies between
    the min and max parameter values */
//#if HAVE_ANSI
//void generate_new_state(float (*user_random_generator) ())
void generate_new_state(RandomASA user_random_generator)
//#else
//void generate_new_state(user_random_generator)
//float (*user_random_generator) ();
//#endif
{
    int index_v;
    float x;
    float parameter_v, min_parameter_v, max_parameter_v, temperature_v,
     parameter_range_v;

    /* generate a new value for each parameter */
    VFOR(index_v) {
	min_parameter_v = parameter_minimum[index_v];
	max_parameter_v = parameter_maximum[index_v];
	parameter_range_v = max_parameter_v - min_parameter_v;

	/* ignore parameters that have too small a range */
	if (fabs(parameter_range_v) < (float) SMALL_FLOAT)
	    continue;

	temperature_v = current_parameter_temperature[index_v];
	parameter_v = last_saved_state.parameter[index_v];

	/* handle discrete integer parameters */
	if (INTEGER_PARAMETER(index_v)) {
	    min_parameter_v -= HALF;
	    max_parameter_v += HALF;
	    parameter_range_v = max_parameter_v - min_parameter_v;
	}
	/* generate a new state x until within the parameter bounds */
	for (;;) {
	    x = parameter_v
		+ generate_vfsr_state(temperature_v, user_random_generator)
		* parameter_range_v;
	    /* exit the loop if within its valid parameter range */
	    if (x <= max_parameter_v - (float) SMALL_FLOAT
		&& x >= min_parameter_v + (float) SMALL_FLOAT)
		break;
	}

	/* handle discrete integer parameters */
/* You might have to check rounding on your machine. */
	if (INTEGER_PARAMETER(index_v)) {
	    if (x < min_parameter_v + HALF)
		x = min_parameter_v + HALF + (float) SMALL_FLOAT;
	    if (x > max_parameter_v - HALF)
		x = max_parameter_v - HALF + (float) SMALL_FLOAT;
	    x = (int) x;
	}
	/* save the newly generated value */

       //	 printf( "x = %ld   parameter_range_v = %g \n ", x,parameter_range_v);
	current_generated_state.parameter[index_v] = x;

    }

}

/***********************************************************************
* Author: Lester Ingber, Bruce Rosen (copyright) (c)
* Date	 5 Nov 92
* Procedure: generate_vfsr_state
* Parameters:
* Inputs: float Temperature
*	float user_random_generator() returns random number between 0,1
* Outputs: None
* Global Variables: None
* Returns: float: A random variable VFSR distributed
* Calls:
*	float pow() power function
*	float fabs() absolute value function
* Description:
*	This function generates a single value according to the
*	VFSR generating function and the passed temperature
* Local Variables:
*	float x - temporary variable
*	float y - temporary variable
*	float z - temporary variable
* Portability:
* Fully portable
* Other:
***********************************************************************/
//#if HAVE_ANSI
float generate_vfsr_state(float temp, RandomASA user_random_generator)
//#else
//float generate_vfsr_state(temp, user_random_generator)
//float temp;
//float (*user_random_generator) ();
//#endif
{
    float x, y, z;

    x = (*user_random_generator) ();
    y = x < HALF ? -ONE : ONE;
    z = y * temp * (pow((ONE + ONE / temp), fabs(TWO * x - ONE)) - ONE);

    return (z);

}

/***********************************************************************
* Author: Lester Ingber, Bruce Rosen (copyright) (c)
* Date	 5 Nov 92
* Procedure: accept_new_state
*	This procedure accepts or rejects a newly generated state,
*	depending on whether the difference between new and old
*	cost functions passes a statistical test. If accepted,
*	the current state is updated.
* Parameters:
* Inputs: None
* float user_random_generator() returning a random number between 0,1
* Outputs: None
* Global Variables:
* Returns: None
* Calls:
* Description:
* Local Variables:
* Portability:
* Other:
***********************************************************************/
 /* determine whether to accept or reject the new state */
//#if HAVE_ANSI
//void accept_new_state(float (*user_random_generator) ())
//int accept_new_state(float (*user_random_generator) ())   /* EV */
int accept_new_state(RandomASA user_random_generator)   /* EV */
//#else
//void accept_new_state(user_random_generator)
//int accept_new_state(user_random_generator)      /* EV */
//float (*user_random_generator) ();
//#endif
{
    float delta_cost;
    int index_v;

    /* update accepted and generated count */
    ++number_acceptances_saved;
    ++recent_number_generated;
    ++number_generated;

    /* increment the parameter index generation for each parameter */
    VFOR(index_v) {
	/* ignore parameters with too small a range */
	if (PARAMETER_RANGE_TOO_SMALL(index_v))
	    continue;
	++(index_parameter_generations[index_v]);
    }

    /* effective cost function for testing acceptance criteria,
       calculate the cost difference and divide by the temperature */
    delta_cost = (current_generated_state.cost - last_saved_state.cost)
	/ current_cost_temperature;

    /* decide to accept/reject the new state */
    if (exp(-EXPONENT_CHECK(delta_cost)) > (*user_random_generator) ()) {
	/* copy the current generated parameters to the last saved state */
	last_saved_state.cost = current_generated_state.cost;
	VFOR(index_v) {
	    /* ignore parameters with too small a range */
	    if (PARAMETER_RANGE_TOO_SMALL(index_v))
		continue;
	    last_saved_state.parameter[index_v] =
		current_generated_state.parameter[index_v];
	}
	/* update acceptance counts */
	++recent_number_acceptances;
	++number_accepted;
	++index_cost_acceptances;
	number_acceptances_saved = number_accepted;
	return(1); /* EV */
    }
    return(0); /* EV */
}

/***********************************************************************
* Author: Lester Ingber, Bruce Rosen (copyright) (c)
* Date	 5 Nov 92
* Procedure: reanneal
* Parameters:
* Inputs: None
* Outputs: None
* Global Variables:
* Returns: None
* Calls:
* Description:
*	Readjust temperatures of generating and acceptance functions
* Local Variables:
*	int index_v				- vth iteration index
*	float tmp_var_db2			-
*	float new_temperature			- the new temperature
*	float log_initial_current_temperature_ratio - for efficiency
* Portability:
* Other:
***********************************************************************/
 /* REANNEALING - readjust the temperatures */
//#if HAVE_ANSI
void reanneal(void)
//#else
//void reanneal()
//#endif
{
    int index_v;
    float tmp_var_db2;
    float new_temperature;	/* the temperature */
    float log_initial_current_temperature_ratio;

    VFOR(index_v) {
	/* ignore parameters with too small a range */
	if (PARAMETER_RANGE_TOO_SMALL(index_v))
	    continue;
	/* ignore parameters with too small tangents */
	if (fabs(tangents[index_v]) < (float) SMALL_FLOAT)
	    continue;

	/* reset the index of parameter generations appropriately */
	new_temperature =
	    current_parameter_temperature[index_v] *
	    (maximum_tangent / fabs(tangents[index_v]));
	if (new_temperature < initial_parameter_temperature[index_v]) {
	    log_initial_current_temperature_ratio =
		log(((float) SMALL_FLOAT + initial_parameter_temperature[index_v])
		    / ((float) SMALL_FLOAT + new_temperature));
	    index_parameter_generations[index_v] =
		(long int) ((float) SMALL_FLOAT
			    + pow(log_initial_current_temperature_ratio
				  / temperature_scale_parameters,
				  (float) number_parameters));
	} else
	    index_parameter_generations[index_v] = 1;

	/* Reset index_parameter_generations if index reset too large,
           and also reset the initial_parameter_temperature, to achieve
           the same new temperature. */
	while (index_parameter_generations[index_v]
	       > ((long int) MAXIMUM_REANNEAL_INDEX)) {
	    log_new_temperature_ratio = -temperature_scale_parameters *
		pow((float) index_parameter_generations[index_v],
		    one_over_number_parameters);
	    log_new_temperature_ratio =
		EXPONENT_CHECK(log_new_temperature_ratio);
	    new_temperature =
		initial_parameter_temperature[index_v]
		* exp(log_new_temperature_ratio);
	    index_parameter_generations[index_v] /=
		(long int) REANNEAL_RESCALE;
	    initial_parameter_temperature[index_v] =
		new_temperature * pow(
		initial_parameter_temperature[index_v] / new_temperature,
					 temperature_rescale_power);
	}
    }

    /* reanneal : Reset the index of cost acceptances to take
       into account finer detail in cost terrain. */

    /* set the starting cost_temperature to last cost found so far */
    if (initial_cost_temperature > fabs(last_saved_state.cost))
	initial_cost_temperature = fabs(last_saved_state.cost);

    if (current_cost_temperature > fabs(best_generated_state.cost)) {
	tmp_var_db2 =
	    log(((float) SMALL_FLOAT + initial_cost_temperature) /
		((float) SMALL_FLOAT + fabs(best_generated_state.cost)));
	index_cost_acceptances = (long int) ((float) SMALL_FLOAT
					     + pow(tmp_var_db2
						/ temperature_scale_cost,
					    (float) number_parameters));
    } else {
	log_initial_current_temperature_ratio =
	    log(((float) SMALL_FLOAT + initial_cost_temperature) /
		((float) SMALL_FLOAT + current_cost_temperature));
	index_cost_acceptances = (long int) ((float) SMALL_FLOAT
			      + pow(log_initial_current_temperature_ratio
				    / temperature_scale_cost,
				    (float) number_parameters));
	/* Due to rounding errors, the index may become 0: recalculate the
	initial temperature in that case */
	if (index_cost_acceptances < 1)
	    {
	    index_cost_acceptances = 1;
	    initial_cost_temperature = current_cost_temperature/
		exp(-temperature_scale_cost);
	    }
    }

    /* reset index_cost_temperature if index reset too large */
    while (index_cost_acceptances > ((long int) MAXIMUM_REANNEAL_INDEX)) {
	log_new_temperature_ratio = -temperature_scale_cost *
	    pow((float) index_cost_acceptances,
		one_over_number_parameters);
	log_new_temperature_ratio =
	    EXPONENT_CHECK(log_new_temperature_ratio);
	new_temperature = initial_cost_temperature
	    * exp(log_new_temperature_ratio);
	index_cost_acceptances /= (int)REANNEAL_RESCALE;
/*	index_cost_acceptances /= (float) REANNEAL_RESCALE;  original */
	initial_cost_temperature =
	    new_temperature * pow(
			      initial_cost_temperature / new_temperature,
				     temperature_rescale_power);
    }
}

/***********************************************************************
* Author: Lester Ingber, Bruce Rosen (copyright) (c)
* Date	 5 Nov 92
* Procedure: cost_derivatives
*	This procedure calculates the derivatives of the cost function
*	with respect to its parameters.  The first derivatives are
*	used as a sensitivity measure for reannealing.  The second
*	derivatives are calculated only if curvature_flag=TRUE;
*	these are a measure of the covariance of the fit when a
*	minimum is found.
* Parameters:
* Inputs: (float (*user_cost_function) () - the user's cost function
* Outputs: None
* Global Variables:
* Returns: None
* Calls:
* Description:
* Local Variables:
*	int index_v, index_vv, index_v_vv - iteration indices
*	long int saved_number_invalid_generated_states - temporary variable
*	float parameter_v, parameter_vv
*		- values of the v_th and vv_th parameters
*	float recent_best_cost - the most recently found best cost
*	float new_cost_state_1, new_cost_state_2, new_cost_state_3;
*		- cost values taken at sample points 1,2 and 3
* Portability:
* Other:
***********************************************************************/
 /* Calculate the numerical derivatives of the best
    generated state found so far */

/* In this implementation of VFSR, no checks are made for
   valid_state_generated_flag=FALSE for differential neighbors
   to the current best state. */

/* Assuming no information is given about the metric of the parameter
   space, use simple Cartesian space to calculate curvatures. */

//#if HAVE_ANSI
void cost_derivatives(FitFunctionASA user_cost_function)
//#else
//void cost_derivatives(user_cost_function)
//float (*user_cost_function) ();
//#endif
{
    int index_v, index_vv, index_v_vv, index_vv_v;
    long int saved_number_invalid_generated_states;
    float parameter_v, parameter_vv, recent_best_cost;
    float new_cost_state_1, new_cost_state_2, new_cost_state_3;

    /* save the best cost */
    recent_best_cost = best_generated_state.cost;

    /* copy the best state into the current state */
    VFOR(index_v) {
	/* ignore parameters with too small ranges */
	if (PARAMETER_RANGE_TOO_SMALL(index_v))
	    continue;
	current_generated_state.parameter[index_v] =
	    best_generated_state.parameter[index_v];
    }

    /* set parameters (& possibly constraints) to best state */
    saved_number_invalid_generated_states =
	number_invalid_generated_states;
    valid_state_generated_flag = TRUE;
    current_generated_state.cost =
	(*user_cost_function)
	(current_generated_state.parameter,
	 &valid_state_generated_flag);
#if VFSR_PRINT
    if (valid_state_generated_flag == FALSE)	/* extra check */
	fprintf(ptr_vfsr_out,
		"Generated an invalid best state when calculating \
 the derivatives\n");
#endif
    number_invalid_generated_states =
	saved_number_invalid_generated_states;
    valid_state_generated_flag = TRUE;

    VFOR(index_v) {
	/* skip parameters with too small range or integer parameters */
#if INCLUDE_INTEGER_PARAMETERS
	if (PARAMETER_RANGE_TOO_SMALL(index_v))
	    continue;
#else
	if (PARAMETER_RANGE_TOO_SMALL(index_v) ||
	    INTEGER_PARAMETER(index_v))
	    continue;
#endif

	/* save parameter_v */
	parameter_v = best_generated_state.parameter[index_v];

	/* generate the first sample point and
           calculate cost at current point + DELTA_X */
	current_generated_state.parameter[index_v] =
	    (ONE + (float) DELTA_X) * parameter_v;
	/* generate the first sample point */
	current_generated_state.cost =
	    (*user_cost_function)
	    (current_generated_state.parameter,
	     &valid_state_generated_flag);
	new_cost_state_1 = current_generated_state.cost;
	valid_state_generated_flag = TRUE;

	/* restore the parameter state */
	current_generated_state.parameter[index_v] = parameter_v;

	/* calculate the numerical derivative */
	tangents[index_v] = (new_cost_state_1 - recent_best_cost)
	    / (((float) DELTA_X) * parameter_v + (float) SMALL_FLOAT);

	/* calculate the diagonal curvatures */
	if (curvature_flag == TRUE) {

	    /* generate the second sample point and
               calculate cost at current point - DELTA_X */
	    current_generated_state.parameter[index_v] =
		(ONE - (float) DELTA_X) * parameter_v;
	    current_generated_state.cost =
		(*user_cost_function)
		(current_generated_state.parameter,
		 &valid_state_generated_flag);
	    new_cost_state_2 = current_generated_state.cost;
	    valid_state_generated_flag = TRUE;

	    /* restore the parameter state */
	    current_generated_state.parameter[index_v] =
		parameter_v;

	    /* index_v_vv: row index_v, column index_v */
	    index_v_vv = ROW_COL_INDEX(index_v, index_v);

	    /* calculate and store the curvature */
	    curvature[index_v_vv] =
		(new_cost_state_2 - TWO * recent_best_cost
		 + new_cost_state_1) / (((float) DELTA_X)
		     * parameter_v * parameter_v + (float) SMALL_FLOAT);
	}
    }

    /* calculate off-diagonal curvatures */
    if (curvature_flag == TRUE) {
	VFOR(index_v) {
	    /* skip parms with too small a range or integer parameters
	       index_v */
#if INCLUDE_INTEGER_PARAMETERS
	    if (PARAMETER_RANGE_TOO_SMALL(index_v))
		continue;
#else
	    if (PARAMETER_RANGE_TOO_SMALL(index_v) ||
		INTEGER_PARAMETER(index_v))
		continue;
#endif

	    VFOR(index_vv) {
		/* skip parms with too small range or integer parameters
		   index_vv */
#if INCLUDE_INTEGER_PARAMETERS
		if (PARAMETER_RANGE_TOO_SMALL(index_v))
		    continue;
#else
		if (PARAMETER_RANGE_TOO_SMALL(index_v) ||
		    INTEGER_PARAMETER(index_v))
		    continue;
#endif
		/* calculate only the upper diagonal */
		if (index_v <= index_vv)
		    continue;

		/* index_v_vv: row index_v, column index_vv */
		index_v_vv = ROW_COL_INDEX(index_v, index_vv);
		index_vv_v = ROW_COL_INDEX(index_vv, index_v);

		/* save the v_th parameter */
		parameter_v =
		    current_generated_state.parameter[index_v];

		/* save the vv_th parameter */
		parameter_vv =
		    current_generated_state.parameter[index_vv];

		/* generate first sample point */
		current_generated_state.parameter[index_v] =
		    (ONE + (float) DELTA_X) * parameter_v;

		current_generated_state.parameter[index_vv] =
		    (ONE + (float) DELTA_X) * parameter_vv;
		current_generated_state.cost =
		    (*user_cost_function)
		    (current_generated_state.parameter,
		     &valid_state_generated_flag);
		new_cost_state_1 = current_generated_state.cost;
		valid_state_generated_flag = TRUE;

		/* restore the v_th parameter */
		current_generated_state.parameter[index_v] =
		    parameter_v;

		/* generate second sample point */
		current_generated_state.cost =
		    (*user_cost_function)
		    (current_generated_state.parameter,
		     &valid_state_generated_flag);
		new_cost_state_2 = current_generated_state.cost;
		valid_state_generated_flag = TRUE;

		/* restore the vv_th parameter */
		current_generated_state.parameter[index_vv] =
		    parameter_vv;

		/* generate third sample point */
		current_generated_state.parameter[index_v] =
		    (ONE + (float) DELTA_X)
		    * best_generated_state.parameter[index_v];
		current_generated_state.cost =
		    (*user_cost_function)
		    (current_generated_state.parameter,
		     &valid_state_generated_flag);
		new_cost_state_3 = current_generated_state.cost;
		valid_state_generated_flag = TRUE;

		/* restore the v_th parameter */
		current_generated_state.parameter[index_v] =
		    parameter_v;

		/* calculate and store the curvature */
		curvature[index_vv_v] =
		    curvature[index_v_vv] =
		    (new_cost_state_1 - new_cost_state_2
		     - new_cost_state_3 + recent_best_cost)
		    / (((float) DELTA_X) * ((float) DELTA_X)
		       * parameter_v * parameter_vv
		       + (float) SMALL_FLOAT);
	    }
	}
    }
    /* restore the best cost function value */
    current_generated_state.cost = recent_best_cost;

    /* find the maximum |tangent| from all tangents */
    maximum_tangent = 0;
    VFOR(index_v) {
	/* ignore too small ranges and integer parameters types */
#if INCLUDE_INTEGER_PARAMETERS
	if (PARAMETER_RANGE_TOO_SMALL(index_v))
	    continue;
#else
	if (PARAMETER_RANGE_TOO_SMALL(index_v) ||
	    INTEGER_PARAMETER(index_v))
	    continue;
#endif

	/* find the maximum |tangent| (from all tangents) */
	if (fabs(tangents[index_v]) > maximum_tangent) {
	    maximum_tangent = fabs(tangents[index_v]);
	}
    }

}

//#if VFSR_PRINT
/***********************************************************************
* Author: Lester Ingber, Bruce Rosen (copyright) (c)
* Date	 5 Nov 92
* Procedure: print_state
* Parameters:
* Inputs: None
* Outputs: None
* Global Variables:
* Returns: None
* Calls:
* Description:
*	Prints a description of the current state of the system
* Local Variables:
*	int index variables index_v, index_vv, index_v_vv
* Portability:
* Other:
***********************************************************************/
 /* print the state of the system to the user */
//#if HAVE_ANSI
void print_state(void)
//#else
//void print_state()
//#endif
{
    int index_v, index_vv, index_v_vv;
    float parameter_temp_min,parameter_temp_max;

//    printf( "\n");

//    printf(
//     "index_cost_acceptances = %ld, current_cost_temperature = %10.5g\n",
//	    index_cost_acceptances, current_cost_temperature);

/*Canged by Daniel Kaminski*/

//    fprintf(ptr_vfsr_out,
//	    "accepted_to_generated_ratio = %10.5g,\
// number_invalid_generated_states = %ld\n",
//	    accepted_to_generated_ratio,
//	    number_invalid_generated_states);

//    printf(
//	    "number_generated = %ld, number_accepted = %ld\n",
//	number_generated, number_accepted);

//    printf(
//	    "best_generated_state.cost = %10.5g,\
// last_saved_state.cost = %10.5g\n",
//	    best_generated_state.cost, last_saved_state.cost);

/* Note that tangents will not be calculated until reanneal is called,
   and therefore their listing in the printout only is relevant then */

  /*  VFOR(index_v) {
	/* ignore too small ranges */
   /*	if (PARAMETER_RANGE_TOO_SMALL(index_v))
	    continue;
	fprintf(ptr_vfsr_out,
		"best_generated_state.parameter[%d] = %10.5g\n\
 current_parameter_temperature[%d] = %10.5g\n\
 tangents[%d]: %10.5g\n",
		index_v, best_generated_state.parameter[index_v],
		index_v, current_parameter_temperature[index_v],
		index_v, tangents[index_v]);
    }
    */

    /*Added by Daniel Kaminski*/
    /* Find minimum and maximum parameter temperatures */

    parameter_temp_min = INITIAL_PARAMETER_TEMPERATURE;
    parameter_temp_max = 0.;

VFOR(index_v)
    {
     if (current_parameter_temperature[index_v]<=parameter_temp_min)
     {
      parameter_temp_min = current_parameter_temperature[index_v] ;
     }
     if ((current_parameter_temperature[index_v] > parameter_temp_max)
     && (current_parameter_temperature[index_v] < 0.999))
     {
      parameter_temp_max = current_parameter_temperature[index_v] ;
     }
    }

     //printf("Ta %8.2e Tpmn %8.2e Tpmx %8.2e #gen %6ld #acc %6ld chi^2 %7.3f\n",
     printf("%10.2e %10.2e %10.2e %10ld %10ld %10.3f %10.3f\n",
	current_cost_temperature,
	parameter_temp_min,
	parameter_temp_max,
	number_generated,
	number_accepted,
	current_generated_state.cost,
	best_generated_state.cost);

    if (curvature_flag == TRUE) {
	/* print curvatures */
	VFOR(index_v) {
	    /* ignore too small ranges */
	    if (PARAMETER_RANGE_TOO_SMALL(index_v))
		continue;
	    VFOR(index_vv) {
		/* ignore too small ranges (index_vv) */
		if (PARAMETER_RANGE_TOO_SMALL(index_vv))
		    continue;
		/* only print upper diagonal of matrix */
		if (index_v < index_vv)
		    continue;

		/* index_v_vv: row index_v, column index_vv */
		index_v_vv = ROW_COL_INDEX(index_v, index_vv);

		printf( "curvature[%d][%d] = %10.5g\n",
			index_v, index_vv, curvature[index_v_vv]);
	    }
	}
    }
//    printf( "\n");
    //fflush(ptr_vfsr_out);

}

//#endif  /* VFSR_PRINT */
#endif /* ASA */
