/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                          set.c                                $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/*
Update 28/06/2007 R. Wilcke (wilcke@esrf.fr)
                  remove all code associated with the COOL preprocessor macro,
                  as this methode has now been replaced by ASA.
Update 27/02/2007 R. Wilcke (wilcke@esrf.fr)
                  created this file by moving the routines for the SET
                  FITTING PARAMETERS menu from the file "lsqfit.c" into it.
*/

/***************************************************************************/
/*      module flags                                                       */
/***************************************************************************/

//#define INCL_PLOTCHI2

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>

#if defined(__unix) || defined(__APPLE__)
#include <memory.h>
#define max(a,b) (a>b?a:b)
#define min(a,b) (a>b?b:a)
#elif defined(MSDOS)
#include <mem.h>
#include <malloc.h>
#endif /* defined(__unix) || defined(__APPLE__) */

#include <float.h>
#include <signal.h>
#include <platform.h>

#include <menu.h>
#include <lsqfit.h>

#ifdef INCL_PLOTCHI2
#include <myplot.h>
#endif /* INCL_PLOTCHI2 */

/***************************************************************************/
/*      global parameters                                                  */
/***************************************************************************/

#ifdef INCL_PLOTCHI2
#define NPLOT1 250             /* number of points one-dimensional plots */
#define NPLOT2 51              /* number of points per axis of contour plots */
#endif /* INCL_PLOTCHI2 */

/***************************************************************************/
/*      internal function prototypes                                       */
/***************************************************************************/

int get_parameter(char *text, int npar);

/***************************************************************************/
int EXPORT set_fitpar(char *prompt, int npar,
    float *fpar, float *fmin, float *fmax,
    int *check, int *fix, char *names, int namelength,
    float *xdata, float *ydata, float *wgtdata,
    int ndata, FitFunction funcs )
/***************************************************************************/

/*
   Set and list fit parameters. Also plot chi^2 as a function of fit pars.
   Returns
     LSQ_LEVENBERG (1) - Levenberg Marquardt Fit requested
     LSQ_COOLDOWN  (2) - Cooldown Fit requested
     LSQ_ASA       (4) - Adaptive Simulated Annealing requested
     LSQ_NOFIT     (0) - Return (no fit) requested
*/

{

    enum idx {
	i_zero,
	i_value,
	i_lower,
	i_upper,
	i_check,
	i_fix,
	i_loose,
	i_center,
	i_fall,
	i_lall,
	i_call,
#ifdef INCL_PLOTCHI2
	i_plot1,
	i_plot2,
#endif /* INCL_PLOTCHI2 */
	i_control,
	i_list,
	i_help,
	i_run,
#ifdef ASA
	i_asa,
#endif /* ASA */
	i_return
	};

    /* define setfitpar_menu */

    static struct MENU setfpar_menu[i_return] =
	{
	"value",         1,  i_value,  "Set parameter value",
	"lower",         3,  i_lower,  "Set lower parameter limit",
	"upper",         1,  i_upper,  "Set upper parameter limit",
	"check",         2,  i_check,  "Toggle range checking",
	"fix",           1,  i_fix,    "Fix a parameter value",
	"loose",         2,  i_loose,  "Make parameter free",
	"center",	 1,  i_center, "Center parameter within range",
	"fall",		 4,  i_fall,   "Fix all parameters",
	"lall",		 4,  i_lall,   "Make all parameters free",
	"call",		 4,  i_call,   "Center all parameters",
#ifdef INCL_PLOTCHI2
        "plot",          1,  i_plot1,  "Plot chi^2 for one parameter",
        "map",           1,  i_plot2,  "Contour-map of chi^2 for two pars.",
#endif /* INCL_PLOTCHI2 */
	"control",	 2,  i_control,"Set control parameters",
	"list",          1,  i_list,   "List parameter values",
	"help",          1,  i_help,   "Display menu",
	"run",           2,  i_run,    "Start fit (Levenberg Marquardt)",
#ifdef ASA
	"asa",           2,  i_asa,    "Start fit (adaptive sim. ann.)",
#endif /* ASA */
	"return",        1,  i_return, "Return without fitting"
	};

    int stop = FALSE;
    char token[100], STRING[200];
    int parameter,i, j, rc=0, fpar_save, par1, par2, fp_sav1, fp_sav2,
        nlevel, *linestyles, nfree, mfit;
    long done = 0, olddone = 0, total;
    float *x, *y, *e, *z, zmin=1e30, zmax=-1e30, z0, z1, zstep, pmin, pmax;

/*    put_command("list"); */
    while (!stop)
	{
	if (!get_token(token,prompt)) return 0;
	switch (cmnd_match(token,setfpar_menu,i_return))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case i_value:
		if (FALSE != (parameter =
		    get_parameter("Parameter value to be changed",npar)))
		    {
		    parameter--;
		    sprintf(STRING,
			"%s value [%8.4f]: ",names+(parameter*namelength),
			fpar[parameter]);
		    fpar[parameter] = get_real(fpar[parameter],STRING);
		    }
		break;
	    case i_fix:
		if (FALSE != (parameter =
		    get_parameter("Parameter to be fixed",npar)))
		    {
		    fix[parameter-1] = TRUE;
		    }
		break;
	    case i_loose:
		if (FALSE != (parameter =
		    get_parameter("Parameter to be freed",npar)))
		    {
		    fix[parameter-1] = FALSE;
		    }
		break;
	    case i_lower:
		if (FALSE != (parameter =
		    get_parameter("Minimum of parameter",npar)))
		    {
		    parameter -= 1;
		    sprintf(STRING,
			"%s minimum [%8.4f]: ",names+(parameter*namelength),
			fmin[parameter]);
		    fmin[parameter] = get_real(fmin[parameter],STRING);
		    }
		break;
	    case i_upper:
		if (FALSE != (parameter =
		    get_parameter("Maximum of parameter",npar)))
		    {
		    parameter -= 1;
		    sprintf(STRING,
			"%s maximum [%8.4f]: ",names+(parameter*namelength),
			fmax[parameter]);
		    fmax[parameter] = get_real(fmax[parameter],STRING);
		    }
		break;
	    case i_check:
		if (FALSE != (parameter =
		    get_parameter("Range checking of parameter",npar)))
		    {
		    parameter -= 1;
		    sprintf(STRING,
			"%s range checking [%s]: ",names+(parameter*namelength),
			yesnostr(check[parameter]));
		    check[parameter] = yesno(check[parameter],STRING);
		    }
		break;

	    case i_fall:
		for (i=0; i < npar; i++) fix[i] = TRUE; break;

	    case i_lall:
		for (i=0; i < npar; i++) fix[i] = FALSE; break;

	    case i_center:
		if (FALSE != (parameter =
		    get_parameter("Center parameter",npar)))
		    {
		    parameter -= 1;
		    fpar[parameter]=(fmin[parameter]+fmax[parameter])/2.0;
		    }
		break;

	    case i_call:
		for (i=0; i < npar; i++)
		    if (!fix[i]) fpar[i]=(fmin[i]+fmax[i])/2.;
		break;

#ifdef INCL_PLOTCHI2
	    case i_plot1:
                if (!ndata) break;
                mfit = 0;
                for (i=0; i<npar; i++)
		    if (!fix[i]) mfit++;
                nfree = max(1, ndata-mfit);
		if (FALSE != (parameter =
		    get_parameter("Parameter for x-axis",npar)))
		    {
		    parameter --;
                    if (fmin[parameter] == fmax[parameter])
                        {
                        errtype("ERROR, parameter minimum=maximum");
                        break;
                        }
                        fpar_save = fpar[parameter];
                        x = calloc(sizeof(*x), NPLOT1);
                        y = calloc(sizeof(*y), NPLOT1);
                        e = calloc(sizeof(*e), NPLOT1);
                        for (i=0; i<NPLOT1; i++)
                            {
                            x[i] = fmin[parameter] +
				(fmax[parameter] - fmin[parameter])
				* (float)i / (float)(NPLOT1-1);
                            fpar[parameter] = x[i];
                            y[i] = calc_chisq(xdata-1,ydata-1,wgtdata-1,ndata,
				fpar-1,npar,funcs)/(float)nfree;
                            e[i] = 0.;
                            }
                        fpar[parameter] = fpar_save;
                        plotxye(x, y, e, NPLOT1);
                        free (x); free(y); free(e);

                    }

                break;

            case i_plot2:

                if (!ndata) break;
		if (!(par1=get_parameter("Parameter for x-axis",npar))) break;
  	        par1--;
                if (fmin[par1] == fmax[par1])
                    {
                    errtype("ERROR, parameter minimum=maximum");
                    break;
                    }

		if (!(par2=get_parameter("Parameter for y-axis",npar))) break;
  	        par2--;
                if (fmin[par2] == fmax[par2])
                    {
                    errtype("ERROR, parameter minimum=maximum");
                    break;
                    }

                if (!(nlevel=get_int(20, "Number of contours [20]: "))) break;

                pmin=get_real(0., "Minimum to plot [0.]: ");
                pmax=get_real(1., "Maximum to plot [1.]: ");

                mfit = 0;
                for (i=0; i<npar; i++)
                   if (!fix[i]) mfit++;
                nfree = max(1, ndata-mfit);

                x = calloc(sizeof(*x), NPLOT2);
                y = calloc(sizeof(*y), NPLOT2);
                z = calloc(sizeof(*z), NPLOT2*NPLOT2);
                linestyles=calloc(sizeof(*linestyles),nlevel*2);

                fp_sav1 = fpar[par1];
                fp_sav2 = fpar[par2];

                for (i=0; i<NPLOT2; i++)
                    {
                    x[i] = fmin[par1] + (fmax[par1] - fmin[par1])
			* (float)i / (float)(NPLOT2-1);
                    }

                for (i=0; i<NPLOT2; i++)
                    {
		    y[i] = fmin[par2] + (fmax[par2] - fmin[par2])
			* (float)i / (float)(NPLOT2-1);
                    }

                total = (long)(NPLOT2)*(long)(NPLOT2);
                olddone = 0;

                for (i=0; i<NPLOT2; i++)
                    {
                    fpar[par2] = y[i];
                    for (j=0; j<NPLOT2; j++)
                        {
                        fpar[par1] = x[j];
                        z[i*NPLOT2+j] = log10( ( calc_chisq(xdata-1, ydata-1,
			    wgtdata-1, ndata, fpar-1, npar, funcs) + 1e-7)
			    /(float) nfree );
                        zmin = min(zmin, z[i*NPLOT2+j]);
                        zmax = max(zmax, z[i*NPLOT2+j]);
#ifdef DEBUG
                        printf(
			    "i = %2d    j = %2d    x = %.3f    y = %.3f    z = %.3f\n",
			    i, j, x[j], y[i], z[i*NPLOT2+j]);
#endif /* DEBUG */
                        done = 100*((i+1)*NPLOT2+(j+1));
                        done /= total;
                        if (done != olddone)
                            {
                            done = min(100, done);
                            sprintf(STRING, "%3d%% done\r", done);
                            type_line(STRING);
                            olddone = done;
                            }
                        }
                    }

                type_line("\n");
                fpar[par1] = fp_sav1;
                fpar[par2] = fp_sav2;

                z0 = zmin+pmin*(zmax-zmin);
                z1 = zmin+pmax*(zmax-zmin);
                zstep = (z1-z0)/(float)(nlevel-1);

                for (i = 0; i < nlevel*2; i++)
        	    {
                    if (z0+(float)i*zstep < (z1-z0)/2.)
                        linestyles[i] = 111;
                    else
          	        linestyles[i] = 141;
        	    }

                plotcon(z,x,NPLOT2,y,NPLOT2,z0,zstep, z1,linestyles,0);

                free(x); free(y); free(z); free(linestyles);

                break;
#endif /* INCL_PLOTCHI2 */
	    case i_control:
                strcpy(STRING, prompt);
                strcpy(STRING+strlen(STRING)-1, ".CONTROL>");
                set_controlpars(STRING);
                break;

	    case i_list:
		clear_screen();
		type_list(STRING,0);
		for (i = 0; i < npar; i++)
		    {
		    sprintf(STRING,"%2d %*s = %8.4f  [%8.4f , %8.4f]",
			i+1,namelength,
			names+(i*namelength),fpar[i],fmin[i],fmax[i]);
		    type_line(STRING);
		    if (check[i] == TRUE)
			type_line("  checked");
		    else
			type_line("         ");
		    if (fix[i]) type_line("  fixed");
		    type_list("\n",ROWS);
		    }
		break;
	    case i_help:
		list_menu("SET FITTING PARAMETERS", setfpar_menu, i_return);
		break;
	    case i_run:
                if (!ndata) break;
		stop = TRUE;
                rc = LSQ_LEVENBERG;
		break;
#ifdef ASA
	    case i_asa:
		if (!ndata) break;
		stop = TRUE;
		rc = LSQ_ASA;
		break;
#endif /* ASA */
	    case i_return:
                stop = TRUE;
                rc = LSQ_NOFIT;
                break;
	    }
	}
    return rc;
}

/***************************************************************************/
int get_parameter(char *text, int npar)
/***************************************************************************/

{
    int parameter;
    char STRING[200];

    sprintf(STRING,"%s [1]: ",text);
    parameter = get_int(1,STRING);
    if ((parameter < 1) || (parameter > npar))
    {
	errtype("ERROR, parameter out of range");
        clear_command();
	return(0);
    }
    return(parameter);
}
