/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                        execute.c                              $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: execute.c,v 1.7 2006/03/28 15:33:59 wilcke Exp $

****************************************************************************/

/*

Update 22/02/2006 R. Wilcke (wilcke@esrf.fr) 
                  added macros max(a,b) and min(a,b) for __APPLE__
Update 19/01/2004 R. Wilcke (wilcke@esrf.fr) 
                  removed declarations of fopen() (is declared in "stdio.h").
Update 05/07/2001 O. Svensson (svensson@esrf.fr)
                  Removed definitions of min and max for CYGWIN
                  since they are now defined in Cygwin 1.X.
Update 21/12/2000 E. Vlieg (vlieg@sci.kun.nl)
                  Added code for weight and friedel.
Update 19/07/2000 R. Wilcke (wilcke@esrf.fr)
                  list_inplane(): removed declaration for filename (not used);
                  list_rods(): replaced length of filename by the ISO-C defined
                  macro FILENAME_MAX.
Update 26/06/2000 R. Wilcke (wilcke@esrf.fr)
                  replace CYGWIN by __CYGWIN__ (is defined by CPP on the 
                  CYGNUS for WINDOWS system).
*/

/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define EXT extern
#include "ave.h"
#undef EXT
#include <stdlib.h>
#if defined(__APPLE__)
#define max(a,b) (a>b?a:b)
#define min(a,b) (a>b?b:a)
#endif /* defined(__APPLE__) */

/***************************************************************************/
void    average(void)
/***************************************************************************/

    /*
    Average a data file
    */

    {

    /* Check if space group has been read in */

    if (NTRANS == 0)
	{
	errtype("ERROR, no space group set");
	clear_command();
	return;
	}

    find_equivalents();

    average_equivalents();

    sort_average();

    compute_agreement();

    list_average();

    }

/***************************************************************************/
void    average_equivalents(void)
/***************************************************************************/

    /*
    Compute for each set of equivalent reflections:
	- weighted average
	- variance (sigma_1)
	- statistical error (sigma_2)
    */

    {

    int         i,j,nref;
    float       f,fsqr,wgt,sigma, nom;

    for (i = 0; i < NNONEQU; i++)
	{
	f = fsqr = wgt = sigma = 0;
	for (j = 0; j < EQULIST[i].NEQU; j++)
	    {
	    nref = EQULIST[i].IDS[j];
	    f += REFIN[nref].F/sqr(REFIN[nref].SIG);
	    fsqr += sqr(REFIN[nref].F/REFIN[nref].SIG);
	    wgt += sqr(1/REFIN[nref].SIG);
	    sigma += REFIN[nref].SIG;
	    }
	REFAVE[i].F = f/wgt;
	if (REFAVE[i].F < FMIN) REFAVE[i].F = FMIN; /* No zero's in data file */
	if (EQULIST[i].NEQU > 1)
	    {
	    nom = max(1e-8, fsqr/wgt-sqr(REFAVE[i].F));
	    REFAVE[i].SIG1 = sqrt(EQULIST[i].NEQU/(EQULIST[i].NEQU-1)*nom);
	    }
	else
	    REFAVE[i].SIG1 = -1;

	if (SIG2MODE == 0)
	    REFAVE[i].SIG2 = sqrt(1/wgt);
	else if (SIG2MODE == 1)
	    REFAVE[i].SIG2 = sigma/EQULIST[i].NEQU;
	else if (SIG2MODE == 2)
	    REFAVE[i].SIG2 = sqrt(EQULIST[i].NEQU/wgt);
	}

    }

/***************************************************************************/
void    compute_agreement(void)
/***************************************************************************/

    /*
    Calculate the average agreement factor of all equivalent reflections
    that are measured accurately enough.
    */

    {

    int i;

    EPSILON = 0;
    NEPSILON = 0;
    for (i = 0; i < NAVE; i++)
	{
	if ((REFAVE[i].F > EPSCUTOFF*REFAVE[i].SIG2) &&
	    (EQULIST[i].NEQU > 1))
	    {
	    EPSILON += REFAVE[i].SIG1/REFAVE[i].F;
	    NEPSILON++;
	    }
	}

    if (NEPSILON == 0)
	{
	errtype(
	    "Not enough good equivalent reflections to get an agreement factor");
	}
    else
	{
	EPSILON /= NEPSILON;
	sprintf(STRING,"Average agreement factor of %d reflections: %6.3f\n",
		NEPSILON,EPSILON);
	type_line(STRING);
	}

    }

/***************************************************************************/
void    find_equivalents(void)
/***************************************************************************/

    /*
    Make list of all symmetry-equivalent reflections
    */

    {

    int         i,j;

    NNONEQU = 0;
    EQULIST[0].NEQU = 0;
    NAVE = 0;
    for (i = 0; i < NIN; i++)
	{
	if (!REFIN[i].DONE)
	    {
	    generate_equivalents(REFIN[i].H,REFIN[i].K,REFIN[i].L);
	    get_sorted_hkl(&REFAVE[NAVE].H,&REFAVE[NAVE].K,&REFAVE[NAVE].L);
	    for (j = i; j < NIN; j++)
		{
		if (!REFIN[j].DONE)
		    {
		    if (in_equ_set(REFIN[j].H,REFIN[j].K,REFIN[j].L))
			{
			EQULIST[NNONEQU].IDS[EQULIST[NNONEQU].NEQU] = j;
			EQULIST[NNONEQU].NEQU++;
			if (EQULIST[NNONEQU].NEQU == MAXEQU)
			    {
			    errtype(
				"ERROR, maximum number of equivalent reflections reached");
			    EQULIST[NNONEQU].NEQU--;
			    }
			REFIN[j].DONE = TRUE;
			}
		    }
		}
	    REFIN[i].DONE = TRUE;
	    NNONEQU++;
	    if (NNONEQU == MAXNEQU)
		{
		errtype(
		    "ERROR, maximum number of non-equivalent reflections reached");
		NNONEQU--;
		}
	    EQULIST[NNONEQU].NEQU = 0;
	    NAVE++;
	    }
	}
    }

/***************************************************************************/
void    generate_equivalents(int h, int k, float l)
/***************************************************************************/

    /*
    Generate all symmetry-equivalent reflections for given (hkl) as
    determined by the plane-group symmetry.
    In addition to the NTRANS transformation matrices, also generate
    Friedel pairs, unless flag is set not do to so.
    */

    {

    int i,j,hnew,knew;
    float   lnew;
    int in_set;

    HKLEQU.NEQU = 0;
    for (i = 0; i < NTRANS; i++)
	{
	hnew = TRANS[i][0][0]*h+TRANS[i][0][1]*k;
	knew = TRANS[i][1][0]*h+TRANS[i][1][1]*k;
	lnew = l;

	/* Check whether this reflection is not allready in the set */

	in_set = FALSE;
	for (j = 0; j < HKLEQU.NEQU; j++)
	    if ((hnew == HKLEQU.H[j]) &&
		(knew == HKLEQU.K[j]) &&
		(fabs(lnew-HKLEQU.L[j]) < L_SEPARATION)) in_set = TRUE;
	if (!in_set)
	    {
	    HKLEQU.H[HKLEQU.NEQU] = hnew;
	    HKLEQU.K[HKLEQU.NEQU] = knew;
	    HKLEQU.L[HKLEQU.NEQU] = lnew;
	    HKLEQU.NEQU++;
	    }
	}

    /* Same thing for Friedel pairs, if desired */

    if (FRIEDEL)
	{
	for (i = 0; i < NTRANS; i++)
	    {
	    hnew = -(TRANS[i][0][0]*h+TRANS[i][0][1]*k);
	    knew = -(TRANS[i][1][0]*h+TRANS[i][1][1]*k);
	    lnew = -l;

	    /* Check whether this reflection is not allready in the set */
	    
	    in_set = FALSE;
	    for (j = 0; j < HKLEQU.NEQU; j++)
		if ((hnew == HKLEQU.H[j]) &&
		    (knew == HKLEQU.K[j]) &&
		    (fabs(lnew-HKLEQU.L[j]) < L_SEPARATION)) in_set = TRUE;
	    if (!in_set)
		{
		HKLEQU.H[HKLEQU.NEQU] = hnew;
		HKLEQU.K[HKLEQU.NEQU] = knew;
		HKLEQU.L[HKLEQU.NEQU] = lnew;
		HKLEQU.NEQU++;
		}
	    }
	}
    }

/***************************************************************************/
void    get_sorted_hkl(int *h, int *k, float *l)
/***************************************************************************/

    /*
    Get a particular (hkl) out of the equivalent set, according to
    the specified (i.e. fixed for the time being) sorting criteria.
    */

    {

    int j,i;
    float       value,new_value;

    /* Find a reflection with h,k >= 0 */

    j = 0;
    while (((HKLEQU.H[j] < 0) || (HKLEQU.K[j] < 0)) && (j < HKLEQU.NEQU))
	j++;
    if (j == HKLEQU.NEQU)	/* No reflection found, release k >= 0 */
	{			/* Probably only for plane group p1 */
	j = 0;
	while ((HKLEQU.H[j] < 0) && (j < HKLEQU.NEQU)) j++;
	}
    if (j == HKLEQU.NEQU) /* still nothing found ! */
    	j = 0;

    *h = HKLEQU.H[j];
    *k = HKLEQU.K[j];
    *l = HKLEQU.L[j];
    value = *h*30+*k*900-*l;

    for (i = j; i < HKLEQU.NEQU; i++)
	{
	new_value = HKLEQU.H[i]*30+HKLEQU.K[i]*900-HKLEQU.L[i];
	if ((new_value < value) && (HKLEQU.H[i] >= 0) && (HKLEQU.K[i] >= 0))
	    {
	    *h = HKLEQU.H[i];
	    *k = HKLEQU.K[i];
	    *l = HKLEQU.L[i];
	    value = new_value;
	    }
	}

    }



/***************************************************************************/
int     in_equ_set(int h, int k, float l)
/***************************************************************************/

    /*
    Check whether (hkl) is part of the current equivalent set
    */

    {

    int i;

    for (i = 0; i < HKLEQU.NEQU; i++)
	{
	if ((h == HKLEQU.H[i]) && (k == HKLEQU.K[i]) &&
	    (fabs(l - HKLEQU.L[i]) < L_SEPARATION))
	    return(TRUE);
	}
    return(FALSE);

    }

/***************************************************************************/
void    list_average(void)
/***************************************************************************/

    /*
    Make file with the results of the averaging
    */

    {

    FILE    *outfile;
    int i,j,k,nref;

    /* Open output file */

    replace_extension(FILENAME,"ave");
    if ((outfile = fopen(FILENAME,"w")) == NULL)
	{
	sprintf(STRING,"Failed to open output file '%s'",FILENAME);
	errtype(STRING);
	clear_command();
	return;
	}

    for (k = 0; k < NAVE; k++)
	{
	i = SORTAVE[k];
	fprintf(outfile,"%4d %4d %4d %7.3f %8.2f %8.2f %8.2f %4d ",
		k+1,REFAVE[i].H,REFAVE[i].K,REFAVE[i].L,
		REFAVE[i].F,REFAVE[i].SIG1,REFAVE[i].SIG2,EQULIST[i].NEQU);
	if ((EQULIST[i].NEQU > 1) && (REFAVE[i].F > 0.01))
	    {
	    if (REFAVE[i].F > EPSCUTOFF*REFAVE[i].SIG2)
		fprintf(outfile,"%6.3f*\n",REFAVE[i].SIG1/(REFAVE[i].F+1e-6));
	    else
		fprintf(outfile,"%6.3f\n",REFAVE[i].SIG1/(REFAVE[i].F+1e-6));
	    }
	else
	    fprintf(outfile,"\n");
	for (j = 0; j < EQULIST[i].NEQU; j++)
	    {
	    nref = EQULIST[i].IDS[j];
	    fprintf(outfile,"%30s %4d %4d %4d %7.3f %8.2f %8.2f"," ",
		    REFIN[nref].ID,REFIN[nref].H,REFIN[nref].K,REFIN[nref].L,
		    REFIN[nref].F,REFIN[nref].SIG);
	    if (fabs(REFIN[nref].F-REFAVE[i].F) > 5*REFIN[nref].SIG)
		{
		fprintf(outfile,"W\n");
		}
	    else
		{
		fprintf(outfile,"\n");
		}
	    }
	}

    /* Some general comments */

    fprintf(outfile,"\n%s %s \n","Plane group: ",PLANEGROUP);
    fprintf(outfile,"%-45s%4d\n","Total number of reflections",NIN);
    fprintf(outfile,"%-45s%4d\n","Total number of non-equivalent reflections",
	    NAVE);
    fprintf(outfile,"Sigma_2 mode = %1d; Cutoff factor = %3.1f\n",
	    SIG2MODE,EPSCUTOFF);
    fprintf(outfile,"Average agreement factor of %d reflections: %6.3f",
	    NEPSILON,EPSILON);

    fclose(outfile);

    }

/***************************************************************************/
void    list_inplane(void)
/***************************************************************************/

    /*
    Make file with only 'inplane' reflections. Structure factors close to
    l = 0 are averaged to give estimated value for l = 0.
    */

    {

    FILE    *outfile;
    int     i,j,n,h,k,nsame,imin1,imin2;
    double  nn;
    float   offset,lmin1,lmin2,finp,sigma_ave,sigma_var,finp_sigma;

    /* open output file */

    replace_extension(FILENAME,"inp");
    if ((outfile = fopen(FILENAME,"w")) == NULL)
	{
	sprintf(STRING,"Failed to open output file '%s'",FILENAME);
	errtype(STRING);
	clear_command();
	return;
	}
    fprintf(outfile,"in-plane data %s\n",COMMENTS);

    /* go through entire averaged data set */

    for (j = 0; j < NAVE; j+=nsame)
	{

	/* See how many data points have the same hk */

	nsame = 0;
	i = SORTAVE[j];
	h = REFAVE[i].H;
	k = REFAVE[i].K;
	while ((REFAVE[SORTAVE[j+nsame]].H == h) && (REFAVE[SORTAVE[j+nsame]].K == k)
	       && (j+nsame < NAVE))
	    nsame++;

	/* find the two l-values closest to zero for this (hk)-rod */

	if (nsame == 1)
	    {
	    lmin1 = lmin2 = REFAVE[i].L;
	    imin1 = imin2 = i;
	    }
	else
	    {
	    if (fabs(REFAVE[SORTAVE[j]].L) < fabs(REFAVE[SORTAVE[j+1]].L))
		{
		imin1 = SORTAVE[j];
		lmin1 = REFAVE[imin1].L;
		imin2 = SORTAVE[j+1];
		lmin2 = REFAVE[imin2].L;
		}
	    else
		{
		imin2 = SORTAVE[j];
		lmin2 = REFAVE[imin2].L;
		imin1 = SORTAVE[j+1];
		lmin1 = REFAVE[imin1].L;
		}
	    }
	for (n = 2; n < nsame; n++)
	    {
	    i = SORTAVE[j+n];
	    if (fabs(REFAVE[i].L) < fabs(lmin2)+0.001)
		{
		if (fabs(REFAVE[i].L) < fabs(lmin1)+0.001)
		    {
		    lmin2 = lmin1;
		    imin2 = imin1;
		    lmin1 = REFAVE[i].L;
		    imin1 = i;
		    }
		else
		    {
		    lmin2 = REFAVE[i].L;
		    imin2 = i;
		    }
		}
	    }

	/* only continue if l-values small enough */

	if ((fabs(lmin1) < LINPLANE) && (fabs(lmin2) < 2*LINPLANE))
	    {

	    /* find f for l=0 by linear interpolation, this works fine for
	       1. only a single point
	       2. two points with positive and negative l-value
	       3. two positive l-values of rod */

	    finp = REFAVE[imin1].F-lmin1*(REFAVE[imin2].F-REFAVE[imin1].F)/
		(lmin2-lmin1+1e-6);

	    /* estimate the sigma by the square addition of the average
	       sigma and half the difference between the f's */

	    sigma_ave = 0.5*(
		sqrt(sqr(REFAVE[imin1].SIG2)+sqr(REFAVE[imin1].F*EPSILON))+
		sqrt(sqr(REFAVE[imin2].SIG2)+sqr(REFAVE[imin2].F*EPSILON)));
	    sigma_var = 0.5*fabs(REFAVE[imin1].F-REFAVE[imin2].F);
	    finp_sigma = sqrt(sqr(sigma_ave)+sqr(sigma_var));
	    fprintf(outfile,"%4d %4d %7.3f %8.2f %8.2f",
		    REFAVE[imin1].H,REFAVE[imin1].K,0.01,finp,finp_sigma);

	    /* If requested, list l-value of nearest Bragg peak */

	    if (LIST_LBRAGG)
		fprintf(outfile,"%6d\n",REFAVE[imin1].LBRAGG);
	    else
		fprintf(outfile,"\n");

	    }
	}
    fclose(outfile);
    }



/***************************************************************************/
void    list_rods(void)
/***************************************************************************/

    /*
    Make file for each measured rod in the data set with at least RODMIN
    data points.
    */

    {

    FILE    *outfile;
    int     i,j,n,h,k,nsame;
    double  nn;
    float   offset;
    char    filename[FILENAME_MAX];


    for (j = 0; j < NAVE; j+=nsame)
	{

	/* See how many data points have the same hk */

	nsame = 0;
	i = SORTAVE[j];
	h = REFAVE[i].H;
	k = REFAVE[i].K;
	while ((REFAVE[SORTAVE[j+nsame]].H == h) && (REFAVE[SORTAVE[j+nsame]].K == k)
	       && (j+nsame < NAVE))
	    nsame++;
	if (nsame >= RODMIN)  /* Indeed write the rod to a file */
	    {

	    /* Open output file */

	    del_extension(FILENAME);
	    sprintf(filename,"%s%1d%1d.dat",FILENAME,h,k);
	    if ((outfile = fopen(filename,"w")) == NULL)
		{
		sprintf(STRING,"Failed to open output file '%s'",FILENAME);
		errtype(STRING);
		clear_command();
		return;
		}

	    /* write comments line to data file */

	    fprintf(outfile,"(%1d,%1d)-rod %s\n",h,k,COMMENTS);
	    for (n = j; n < j+nsame; n++)
		{
		i = SORTAVE[n];
		if ((NOINTL) && (fabs(modf(REFAVE[i].L,&nn)) < L_SEPARATION))
		    offset = ((REFAVE[i].L >= 0)-0.5)*2*L_SEPARATION;
		else
		    offset = 0;
		fprintf(outfile,"%4d %4d %7.3f ",
			REFAVE[i].H,REFAVE[i].K,REFAVE[i].L*LMULT+offset);

		/* If number of symmetry equivalent reflections of a particular
		   reflections is less than EQUMIN, use EPSILON to estimate the error */

		if (EQULIST[i].NEQU >= EQUMIN)
		    fprintf(outfile,"%8.2f %8.2f ",
			    REFAVE[i].F,
			    sqrt(sqr(REFAVE[i].SIG1)+sqr(REFAVE[i].SIG2)));
		else
		    fprintf(outfile,"%8.2f %8.2f ",
			    REFAVE[i].F,
			    sqrt(sqr(REFAVE[i].SIG2)+sqr(REFAVE[i].F*EPSILON)));

		/* If requested, list l-value of nearest Bragg peak */

		if (LIST_LBRAGG)
		    fprintf(outfile,"%6d\n",REFAVE[i].LBRAGG);
		else
		    fprintf(outfile,"\n");
		}
	    fclose(outfile);
	    }
	}
    }

/***************************************************************************/
void    list_weight(int weightmode)
/***************************************************************************/

    /*
    Make file with all non-equivalent reflections and with weighted
    errors.
    if weightmode == normal
	- extension .dat
	- columns h, k, l, f, sigma, (lBragg)
    if weightmode == special
	- extension .wgt
	- columns scannr, h, k, l, f, sigma, (lBragg)
    */

    {

    FILE    *outfile;
    int     i,j;
    double  nn;
    float   offset;

    /* Open output file */

    if (weightmode == WEIGHTNORMAL)
	replace_extension(FILENAME,"dat");
    if (weightmode == WEIGHTSPECIAL)
	replace_extension(FILENAME,"wgt");
    if ((outfile = fopen(FILENAME,"w")) == NULL)
	{
	sprintf(STRING,"Failed to open output file '%s'",FILENAME);
	errtype(STRING);
	clear_command();
	return;
	}

    /* Put a comments line in the weight file */

    get_string(COMMENTS,"Comments: ");
    fprintf(outfile,"%s\n",COMMENTS);

    for (j = 0; j < NAVE; j++)
	{
	i = SORTAVE[j];

	/* list scannumber of first equivalent reflection if desired */

	if (weightmode == WEIGHTSPECIAL)
	    fprintf(outfile,"%8d ",REFIN[EQULIST[i].IDS[0]].ID);

	if ((NOINTL) && (fabs(modf(REFAVE[i].L,&nn)) < L_SEPARATION))
	    offset = ((REFAVE[i].L >= 0)-0.5)*2*L_SEPARATION;
	else
	    offset = 0;
	fprintf(outfile,"%4d %4d %7.3f ",
		REFAVE[i].H,REFAVE[i].K,REFAVE[i].L*LMULT+offset);

	/* If number of symmetry equivalent reflections of a particular
	   reflections is less than EQUMIN, use EPSILON to estimate the error */

	if (EQULIST[i].NEQU >= EQUMIN)
	    fprintf(outfile,"%8.2f %8.2f ",
		    REFAVE[i].F,
		    sqrt(sqr(REFAVE[i].SIG1)+sqr(REFAVE[i].SIG2)));
	else
	    fprintf(outfile,"%8.2f %8.2f ",
		    REFAVE[i].F,
		    sqrt(sqr(REFAVE[i].SIG2)+sqr(REFAVE[i].F*EPSILON)));

	/* If requested, list l-value of nearest Bragg peak */

	if (LIST_LBRAGG)
	    fprintf(outfile,"%6d\n",REFAVE[i].LBRAGG);
	else
	    fprintf(outfile,"\n");

	}
    fclose(outfile);

    }

/***************************************************************************/
void    sort_average(void)
/***************************************************************************/

    /*
    Generate list with sorted serial numbers of averaged reflections.
    */

    {

    int i,j,n;
    float       value;

    for (i = 0; i < NAVE; i++)
	SORTAVE[i] = i;

    /* Sort with l fastest, and h slowest varying */

    for (j = 1; j < NAVE; j++)
	{
	n = SORTAVE[j];
	value = REFAVE[n].H*900+REFAVE[n].K*30+REFAVE[n].L;
	i = j-1;
	while ((i >= 0) &&
	       (REFAVE[SORTAVE[i]].H*900+REFAVE[SORTAVE[i]].K*30+
		REFAVE[SORTAVE[i]].L > value))
	    {
	    SORTAVE[i+1] = SORTAVE[i];
	    i--;
	    }
	SORTAVE[i+1] = n;
	}
    }

/***************************************************************************/
double  sqr(double x)
/***************************************************************************/

    /*
    return square of x
    */

    {
    return(x*x);
    }

