/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                       theory.c                                $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: theory.c,v 1.6 2006/03/28 15:31:06 wilcke Exp $

****************************************************************************/
/*
Update 05/07/2001 R. Wilcke (wilcke@esrf.fr)
                  pseudo_voigt(): corrected error in the calculation of the
                  derivatives.
Update 05/07/2001 E. Vlieg (vlieg@sci.kun.nl)
                  Changed menus to follow Rainer's scheme.
update 09/10/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Change in all profiles hwhm to fwhm. (Done for complete
		  equations, not just multiplier).
*/
/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define EXT	extern
#include "ana.h"
#undef EXT

/***************************************************************************/
void    gauss_double(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Double Gaussian.
    */

    {

    float       xnew1,ex1,arg1,fac1,xnew2,ex2,arg2,fac2;

    if (mode == INIT)
	{
	F_PTR = gauss_double;
	sprintf(TH.TITLE,"Double Gaussian");
	TH.NPAR = 8;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Center position of Gaussian1");
	sprintf(TH.NAME[3],"Height of Gaussian1");
	sprintf(TH.NAME[4],"FWHM of Gaussian1");
	sprintf(TH.NAME[5],"Center position of Gaussian2");
	sprintf(TH.NAME[6],"Height of Gaussian2");
	sprintf(TH.NAME[7],"FWHM of Gaussian2");
	sprintf(TH.SHORT[0],"bkgrd");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"pos1");
	sprintf(TH.SHORT[3],"hght1");
	sprintf(TH.SHORT[4],"fwhm1");
	sprintf(TH.SHORT[5],"pos2");
	sprintf(TH.SHORT[6],"hght2");
	sprintf(TH.SHORT[7],"fwhm2");
	}
    else if (mode == CALC)
	{
	*y = a[0]+(x-a[2])*a[1]
	    +a[3]*exp(-GAUSIG2*sqr(x-a[2])
		      /(sqr(a[4])+1e-10))
	    +a[6]*exp(-GAUSIG2*sqr(x-a[5])
		      /(sqr(a[7])+1e-10));
	}
    else if (mode == DERV)
	{
	xnew1 = x-a[2];
	arg1 = xnew1/(a[4]+1e-10);
	ex1 = exp(-GAUSIG2*arg1*arg1);
	fac1 = a[3]*ex1*2*GAUSIG2*arg1;
	xnew2 = x-a[5];
	arg2 = xnew2/(a[7]+1e-10);
	ex2 = exp(-GAUSIG2*arg2*arg2);
	fac2 = a[6]*ex2*2*GAUSIG2*arg2;
	*y = a[0]+a[1]*xnew1+a[3]*ex1+a[6]*ex2;
	dyda[0] = 1.;
	dyda[1] = xnew1;
	dyda[2] = -a[1]+fac1/(a[4]+1e-10);
	dyda[3] = ex1;
	dyda[4] = fac1*arg1/(a[4]+1e-10);
	dyda[5] = fac2/(a[7]+1e-10);
	dyda[6] = ex2;
	dyda[7] = fac2*arg2/(a[7]+1e-10);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 2;
	TH.PEAKINT[0] = TH.PAR[3]*TH.PAR[4]*GCONST2;
	if ((fabs(TH.PAR[3]) < 1e-6) || (fabs(TH.PAR[4]) < 1e-6))
	    {
	    TH.PEAKERR[0] = 0;
	    }
	else
	    {
	    TH.PEAKERR[0] = TH.PEAKINT[0]*
		sqrt(sqr(TH.ERR[3]/TH.PAR[3])+sqr(TH.ERR[4]/TH.PAR[4]));
	    }
	TH.PEAKINT[1] = TH.PAR[6]*TH.PAR[7]*GCONST2;
	if ((fabs(TH.PAR[6]) < 1e-6) || (fabs(TH.PAR[7]) < 1e-6))
	    {
	    TH.PEAKERR[1] = 0;
	    }
	else
	    {
	    TH.PEAKERR[1] = TH.PEAKINT[1]*
		sqrt(sqr(TH.ERR[6]/TH.PAR[6])+sqr(TH.ERR[7]/TH.PAR[7]));
	    }
	}
    }

/***************************************************************************/
void    gauss_lorentz(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Gaussian plus Lorentzian.
    */

    {

    float       xnew1,ex1,arg1,fac1,xnew2,num2,den2;

    if (mode == INIT)
	{
	F_PTR = gauss_lorentz;
	sprintf(TH.TITLE,"Gaussian plus Lorentzian");
	TH.NPAR = 8;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Center position of Gaussian");
	sprintf(TH.NAME[3],"Height of Gaussian");
	sprintf(TH.NAME[4],"FWHM of Gaussian");
	sprintf(TH.NAME[5],"Center position of Lorentzian");
	sprintf(TH.NAME[6],"Height of Lorentzian");
	sprintf(TH.NAME[7],"FWHM of Lorentzian");
	sprintf(TH.SHORT[0],"bkgrd");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"gpos");
	sprintf(TH.SHORT[3],"ghght");
	sprintf(TH.SHORT[4],"gfwhm");
	sprintf(TH.SHORT[5],"lpos");
	sprintf(TH.SHORT[6],"lhght");
	sprintf(TH.SHORT[7],"lfwhm");
	}
    else if (mode == CALC)
	{
	*y = a[0]+(x-a[2])*a[1]
	    +a[3]*exp(-GAUSIG2*sqr(x-a[2])
		      /(sqr(a[4])+1e-10))
	    +a[6]*sqr(a[7])
	    /(4*sqr(x-a[5])+sqr(a[7])+1e-10);
	}
    else if (mode == DERV)
	{
	xnew1 = x-a[2];
	arg1 = xnew1/(a[4]+1e-10);
	ex1 = exp(-GAUSIG2*arg1*arg1);
	fac1 = a[3]*ex1*2*GAUSIG2*arg1;
	xnew2 = x-a[5];
	num2 = a[6]*a[7]*a[7];
	den2 = 4*xnew2*xnew2+a[7]*a[7]+1e-10;
	*y = a[0]+xnew1*a[1]+a[3]*ex1+num2/den2;
	dyda[0] = 1.;
	dyda[1] = xnew1;
	dyda[2] = -a[1]+fac1/(a[4]+1e-10);
	dyda[3] = ex1;
	dyda[4] = fac1*arg1/(a[4]+1e-10);
	dyda[5] = 8*xnew2*num2/(den2*den2);
	dyda[6] = a[7]*a[7]/den2;
	dyda[7] = 2*a[6]*a[7]/den2 - 2*num2*a[7]/(den2*den2);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 2;
	TH.PEAKINT[0] = TH.PAR[3]*TH.PAR[4]*GCONST2;
	if ((fabs(TH.PAR[3]) < 1e-6) || (fabs(TH.PAR[4]) < 1e-6))
	    {
	    TH.PEAKERR[0] = 0;
	    }
	else
	    {
	    TH.PEAKERR[0] = TH.PEAKINT[0]*
		sqrt(sqr(TH.ERR[3]/TH.PAR[3])+sqr(TH.ERR[4]/TH.PAR[4]));
	    }
	TH.PEAKINT[1] = TH.PAR[6]*TH.PAR[7]*PI/2;
	if ((fabs(TH.PAR[6]) < 1e-6) || (fabs(TH.PAR[7]) < 1e-6))
	    {
	    TH.PEAKERR[1] = 0;
	    }
	else
	    {
	    TH.PEAKERR[1] = TH.PEAKINT[1]*
		sqrt(sqr(TH.ERR[6]/TH.PAR[6])+sqr(TH.ERR[7]/TH.PAR[7]));
	    }
	}
    }

/***************************************************************************/
void    gauss_single(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Single Gaussian.
    */

    {

    float       xnew,ex,arg,fac;

    if (mode == INIT)
	{
	F_PTR = gauss_single;
	sprintf(TH.TITLE,"Single Gaussian");
	TH.NPAR = 5;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Center position of Gaussian");
	sprintf(TH.NAME[3],"Height of Gaussian");
	sprintf(TH.NAME[4],"FWHM of Gaussian");
	sprintf(TH.SHORT[0],"bkgrd");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"pos");
	sprintf(TH.SHORT[3],"hght");
	sprintf(TH.SHORT[4],"fwhm");
	}
    else if (mode == CALC)
	{
	*y = a[0]+(x-a[2])*a[1]
	    +a[3]*exp(-GAUSIG2*sqr(x-a[2])
		      /(sqr(a[4])+1e-10));
	}
    else if (mode == DERV)
	{
	xnew = x-a[2];
	arg = xnew/(a[4]+1e-10);
	ex = exp(-GAUSIG2*arg*arg);
	fac = a[3]*ex*2*GAUSIG2*arg;
	*y = a[0]+a[1]*xnew+a[3]*ex;
	dyda[0] = 1.;
	dyda[1] = xnew;
	dyda[2] = -a[1]+fac/(a[4]+1e-10);
	dyda[3] = ex;
	dyda[4] = fac*arg/(a[4]+1e-10);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 1;
	TH.PEAKINT[0] = TH.PAR[3]*TH.PAR[4]*GCONST2;
	if ((fabs(TH.PAR[3]) < 1e-6) || (fabs(TH.PAR[4]) < 1e-6))
	    {
	    TH.PEAKERR[0] = 0;
	    }
	else
	    {
	    TH.PEAKERR[0] = TH.PEAKINT[0]*
		sqrt(sqr(TH.ERR[3]/TH.PAR[3])+sqr(TH.ERR[4]/TH.PAR[4]));
	    }
	}
    }

/***************************************************************************/
void    lorentz_gauss(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Lorentzian plus Gaussian.
    */

    {

    float       xnew1,num1,den1,xnew2,ex2,arg2,fac2;

    if (mode == INIT)
	{
	F_PTR = lorentz_gauss;
	sprintf(TH.TITLE,"Lorentzian plus Gaussian");
	TH.NPAR = 8;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Center position of Lorentzian");
	sprintf(TH.NAME[3],"Height of Lorentzian");
	sprintf(TH.NAME[4],"FWHM of Lorentzian");
	sprintf(TH.NAME[5],"Center position of Gaussian");
	sprintf(TH.NAME[6],"Height of Gaussian");
	sprintf(TH.NAME[7],"FWHM of Gaussian");
	sprintf(TH.SHORT[0],"bkgrd");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"lpos");
	sprintf(TH.SHORT[3],"lhght");
	sprintf(TH.SHORT[4],"lfwhm");
	sprintf(TH.SHORT[5],"gpos");
	sprintf(TH.SHORT[6],"ghght");
	sprintf(TH.SHORT[7],"gfwhm");
	}
    else if (mode == CALC)
	{
	*y = a[0]+(x-a[2])*a[1]
	    +a[3]*sqr(a[4])
	    /(4*sqr(x-a[2])+sqr(a[4])+1e-10)
	    +a[6]*exp(-GAUSIG2*sqr(x-a[5])
		      /(sqr(a[7])+1e-10));
	}
    else if (mode == DERV)
	{
	xnew1 = x-a[2];
	num1 = a[3]*a[4]*a[4];
	den1 = 4*xnew1*xnew1+a[4]*a[4]+1e-10;
	xnew2 = x-a[5];
	arg2 = xnew2/(a[7]+1e-10);
	ex2 = exp(-GAUSIG2*arg2*arg2);
	fac2 = a[6]*ex2*2*GAUSIG2*arg2;
	*y = a[0]+xnew1*a[1]+num1/den1+a[6]*ex2;
	dyda[0] = 1.;
	dyda[1] = xnew1;
	dyda[2] = -a[1]+8*xnew1*num1/(den1*den1);
	dyda[3] = a[4]*a[4]/den1;
	dyda[4] = 2*a[3]*a[4]/den1 - 2*num1*a[4]/(den1*den1);
	dyda[5] = fac2/(a[7]+1e-10);
	dyda[6] = ex2;
	dyda[7] = fac2*arg2/(a[7]+1e-10);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 2;
	TH.PEAKINT[0] = TH.PAR[3]*TH.PAR[4]*PI/2;
	if ((fabs(TH.PAR[3]) < 1e-6) || (fabs(TH.PAR[4]) < 1e-6))
	    {
	    TH.PEAKERR[0] = 0;
	    }
	else
	    {
	    TH.PEAKERR[0] = TH.PEAKINT[0]*
		sqrt(sqr(TH.ERR[3]/TH.PAR[3])+sqr(TH.ERR[4]/TH.PAR[4]));
	    }
	TH.PEAKINT[1] = TH.PAR[6]*TH.PAR[7]*GCONST2;
	if ((fabs(TH.PAR[6]) < 1e-6) || (fabs(TH.PAR[7]) < 1e-6))
	    {
	    TH.PEAKERR[1] = 0;
	    }
	else
	    {
	    TH.PEAKERR[1] = TH.PEAKINT[1]*
		sqrt(sqr(TH.ERR[6]/TH.PAR[6])+sqr(TH.ERR[7]/TH.PAR[7]));
	    }
	}
    }

/***************************************************************************/
void    lorentz_double(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Double Lorentzian.
    */

    {

    float       xnew1,num1,den1,xnew2,num2,den2;

    if (mode == INIT)
	{
	F_PTR = lorentz_double;
	sprintf(TH.TITLE,"Double Lorentzian");
	TH.NPAR = 8;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Center position Lorentzian1");
	sprintf(TH.NAME[3],"Height of Lorentzian1");
	sprintf(TH.NAME[4],"FWHM of Lorentzian1");
	sprintf(TH.NAME[5],"Center position Lorentzian2");
	sprintf(TH.NAME[6],"Height of Lorentzian2");
	sprintf(TH.NAME[7],"FWHM of Lorentzian2");
	sprintf(TH.SHORT[0],"bkgrd");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"pos1");
	sprintf(TH.SHORT[3],"hght1");
	sprintf(TH.SHORT[4],"fwhm1");
	sprintf(TH.SHORT[5],"pos2");
	sprintf(TH.SHORT[6],"hght2");
	sprintf(TH.SHORT[7],"fwhm2");
	}
    else if (mode == CALC)
	{
	*y = a[0]+(x-a[2])*a[1]
	    +a[3]*sqr(a[4])
	    /(4*sqr(x-a[2])+sqr(a[4])+1e-10)
	    +a[6]*sqr(a[7])
	    /(4*sqr(x-a[5])+sqr(a[7])+1e-10);
	}
    else if (mode == DERV)
	{
	xnew1 = x-a[2];
	num1 = a[3]*a[4]*a[4];
	den1 = 4*xnew1*xnew1+a[4]*a[4]+1e-10;
	xnew2 = x-a[5];
	num2 = a[6]*a[7]*a[7];
	den2 = 4*xnew2*xnew2+a[7]*a[7]+1e-10;
	*y = a[0]+xnew1*a[1]+num1/den1+num2/den2;
	dyda[0] = 1.;
	dyda[1] = xnew1;
	dyda[2] = -a[1]+8*xnew1*num1/(den1*den1);
	dyda[3] = a[4]*a[4]/den1;
	dyda[4] = 2*a[3]*a[4]/den1 - 2*num1*a[4]/(den1*den1);
	dyda[5] = 8*xnew2*num2/(den2*den2);
	dyda[6] = a[7]*a[7]/den2;
	dyda[7] = 2*a[6]*a[7]/den2 - 2*num2*a[7]/(den2*den2);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 2;
	TH.PEAKINT[0] = TH.PAR[3]*TH.PAR[4]*PI/2;
	if ((fabs(TH.PAR[3]) < 1e-6) || (fabs(TH.PAR[4]) < 1e-6))
	    {
	    TH.PEAKERR[0] = 0;
	    }
	else
	    {
	    TH.PEAKERR[0] = TH.PEAKINT[0]*
		sqrt(sqr(TH.ERR[3]/TH.PAR[3])+sqr(TH.ERR[4]/TH.PAR[4]));
	    }
	TH.PEAKINT[1] = TH.PAR[6]*TH.PAR[7]*PI/2;
	if ((fabs(TH.PAR[6]) < 1e-6) || (fabs(TH.PAR[7]) < 1e-6))
	    {
	    TH.PEAKERR[1] = 0;
	    }
	else
	    {
	    TH.PEAKERR[1] = TH.PEAKINT[1]*
		sqrt(sqr(TH.ERR[6]/TH.PAR[6])+sqr(TH.ERR[7]/TH.PAR[7]));
	    }
	}
    }

/***************************************************************************/
void    lorentz_single(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Single Lorentzian.
    */

    {

    float       xnew,num,den;

    if (mode == INIT)
	{
	F_PTR = lorentz_single;
	sprintf(TH.TITLE,"Single Lorentzian");
	TH.NPAR = 5;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Center position of Lorentzian");
	sprintf(TH.NAME[3],"Height of Lorentzian");
	sprintf(TH.NAME[4],"FWHM of Lorentzian");
	sprintf(TH.SHORT[0],"bkgrd");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"pos");
	sprintf(TH.SHORT[3],"hght");
	sprintf(TH.SHORT[4],"fwhm");
	}
    else if (mode == CALC)
	{
	*y = a[0]+(x-a[2])*a[1]
	    +a[3]*sqr(a[4])
	    /(4*sqr(x-a[2])+sqr(a[4])+1e-10);
	}
    else if (mode == DERV)
	{
	xnew = x-a[2];
	num = a[3]*a[4]*a[4];
	den = 4*xnew*xnew+a[4]*a[4];
	if (den < 1e-10) den = 1e-10;
	*y = a[0]+xnew*a[1]+num/den;
	dyda[0] = 1.;
	dyda[1] = xnew;
	dyda[2] = -a[1]+8*xnew*num/(den*den);
	dyda[3] = a[4]*a[4]/den;
	dyda[4] = 2*a[3]*a[4]/den - 2*num*a[4]/(den*den);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 1;
	TH.PEAKINT[0] = TH.PAR[3]*TH.PAR[4]*PI/2;
	if ((fabs(TH.PAR[3]) < 1e-6) || (fabs(TH.PAR[4]) < 1e-6))
	    {
	    TH.PEAKERR[0] = 0;
	    }
	else
	    {
	    TH.PEAKERR[0] = TH.PEAKINT[0]*
		sqrt(sqr(TH.ERR[3]/TH.PAR[3])+sqr(TH.ERR[4]/TH.PAR[4]));
	    }
	}
    }

/***************************************************************************/
void    power(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Exponential function, but with small constant added to denumerator,
    to prevent crashing. Should work fine away from peak.
    */

    {

    float       xnew,arg,denom;

    if (mode == INIT)
	{
	F_PTR = power;
	sprintf(TH.TITLE,"Power law");
	TH.NPAR = 6;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Peak position");
	sprintf(TH.NAME[3],"Amplitude");
	sprintf(TH.NAME[4],"Width");
	sprintf(TH.NAME[5],"Exponent");
	sprintf(TH.SHORT[0],"bkgrd");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"pos");
	sprintf(TH.SHORT[3],"hght");
	sprintf(TH.SHORT[4],"wdth");
	sprintf(TH.SHORT[5],"exp");
	}
    else if (mode == CALC)
	{
	*y = a[0]+(x-a[2])*a[1]
	    +a[3]/pow(4*(x-a[2])*(x-a[2])/a[4]/a[4]+1,a[5]/2);
	}
    else if (mode == DERV)
	{
	xnew = x-a[2];
	arg = 4*xnew*xnew/a[4]/a[4]+1;
	denom = pow(arg,a[5]/2);
	*y = a[0]+a[1]*xnew+a[3]/denom;
	dyda[0] = 1.;
	dyda[1] = xnew;
	dyda[2] = -a[1]+4*a[3]*a[5]*xnew/(denom*arg*a[4]*a[4]);
	dyda[3] = 1/denom;
	dyda[4] = 4*a[3]*a[5]*xnew*xnew/(denom*arg*a[4]*a[4]*a[4]);
	dyda[5] = -0.5*a[3]*log(arg)/denom;
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 0;
	}
    }

/***************************************************************************/
void    pseudo_voigt(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Pseudo voigt function with independent Lorentzian and Gaussian widths.
    */

    {

    float       xnew,num1,den1,lor,arg2,fac2,ex;

    if (mode == INIT)
	{
	F_PTR = pseudo_voigt;
	sprintf(TH.TITLE,"Pseudo Voigt");
	TH.NPAR = 7;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Peak position");
	sprintf(TH.NAME[3],"Height of peak");
	sprintf(TH.NAME[4],"FWHM of Lorentzian");
	sprintf(TH.NAME[5],"FWHM of Gaussian");
	sprintf(TH.NAME[6],"Lorentz fraction eta");
	sprintf(TH.SHORT[0],"bkgrd");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"pos");
	sprintf(TH.SHORT[3],"hght");
	sprintf(TH.SHORT[4],"lfwhm");
	sprintf(TH.SHORT[5],"gfwhm");
	sprintf(TH.SHORT[6],"eta");
	}
    else if (mode == CALC)
	{
	*y = a[0]+(x-a[2])*a[1]
	    +a[3]*(
		a[6]*sqr(a[4])/(4*sqr(x-a[2])+sqr(a[4])+1e-10)
		+(1-a[6])*exp(-GAUSIG2*sqr(x-a[2])/(sqr(a[5])+1e-10)));
	}
    else if (mode == DERV)
	{
	xnew = x-a[2];
	num1 = a[4]*a[4];
	den1 = 4*xnew*xnew+num1+1e-10;
	lor = num1/den1;
	arg2 = xnew/(a[5]+1e-10);
	ex = exp(-GAUSIG2*arg2*arg2);
	fac2 = ex*2*GAUSIG2*arg2/(a[5]+1e-10);
	*y = a[0]+xnew*a[1]+a[3]*(a[6]*lor+(1-a[6])*ex);
	dyda[0] = 1;
	dyda[1] = xnew;
	dyda[2] = -a[1]+a[3]*(a[6]*8*xnew*lor/den1+(1-a[6])*fac2);
	dyda[3] = a[6]*lor+(1-a[6])*ex;
	dyda[4] = 2*a[3]*a[6]*(a[4]*den1-num1*a[4])/(den1*den1);
	dyda[5] = a[3]*(1-a[6])*fac2*arg2;
	dyda[6] = a[3]*(lor-ex);
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 0;
	}
    }

/***************************************************************************/
void    slit(float x, float a[], float *y, float dyda[], int mode)
/***************************************************************************/

    /*
    Slit function.
    */

    {

    float       smear,xstep,xstart,xval,ysum,ssum;
    int		i;
#define	smearsteps	21

    if (mode == INIT)
	{
	F_PTR = slit;
	sprintf(TH.TITLE,"Slit");
	TH.NPAR = 9;
	sprintf(TH.NAME[0],"Constant background");
	sprintf(TH.NAME[1],"Slope of background");
	sprintf(TH.NAME[2],"Center of slit function");
	sprintf(TH.NAME[3],"Height of slit function");
	sprintf(TH.NAME[4],"Slit opening");
	sprintf(TH.NAME[5],"Blade offset");
	sprintf(TH.NAME[6],"Screen distance");
	sprintf(TH.NAME[7],"Wavelength");
	sprintf(TH.NAME[8],"Smearing");
	sprintf(TH.SHORT[0],"bkgrd");
	sprintf(TH.SHORT[1],"slope");
	sprintf(TH.SHORT[2],"pos");
	sprintf(TH.SHORT[3],"hght");
	sprintf(TH.SHORT[4],"s");
	sprintf(TH.SHORT[5],"h");
	sprintf(TH.SHORT[6],"l");
	sprintf(TH.SHORT[7],"lam");
	sprintf(TH.SHORT[8],"smear");
	}
    else if (mode == CALC)
	{
	xstep = 4*a[8]/(smearsteps-1.);
	xstart = -xstep*(smearsteps-1.)/2.;
	ysum = 0;
	ssum = 0;
	for (i = 0; i < smearsteps; i++)
	    {
	    xval = x-a[2]+xstart+i*xstep;
	    smear = exp(-sqr((xstart+i*xstep)/a[8]));
	    ssum += smear;
	    ysum += smear*(a[0]+xval*a[1]
			   +a[3]*sqr(1000*
				     sin(
					 PI*xval*(a[4]/1000.+xval*a[5]/(a[6]*1000))/(a[6]*a[7]/10000)
					 )
				     /(PI*xval/(a[6]*a[7]/10000)+1e-10)
			       ));
	    }
	*y = ysum/ssum;
	}
    else if (mode == DERV)
	{
	}
    else if (mode == PEAK)
	{
	TH.NPEAK = 0;
	}
    }

/***************************************************************************/
void    theory(void)
/***************************************************************************/

    /*
    Compute theoretical spectrum.
    */

    {

    /* define theory_menu */

#define theory_length 10       /* number of commands in theory menu */

    static struct   MENU theory_menu[theory_length] =
	{
	"theory",    1,  1,  "Choose type of theory",
	"lowerx",    2,  2,  "Lower x-value",
	"upperx",    1,  3,  "Upper x-value",
	"stepx",     1,  4,  "Step size in x",
	"specx",     2,  5,  "Use x-range of spectrum (if 0, use range)",
	"value",     1,  6,  "Set parameter value",
	"compute",   1, 10,  "Compute theoretical spectrum",
	"list",      1, 11,  "List parameter values",
	"help",      1, 20,  "Display menu",
	"return",    1, 21,  "Return to main menu",
	};

    int     stop = FALSE;
    char    token[100];
    static float   lowerx,upperx,stepx;
    int     parameter,i;
    static int  spectrum = 0;
    int     outspec;
    float   *a,*dyda;

    while (!stop)
	{
	type_header(0);
	while (!get_token(token,"ANA.THEORY>")) return;
	switch (cmnd_match(token,theory_menu,theory_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		theory_type();
		break;
	    case 2:
		sprintf(STRING,"Lower value of x-range [%6.3f]: ",lowerx);
		lowerx = get_real(lowerx,STRING);
		break;
	    case 3:
		sprintf(STRING,"Upper value of x-range [%6.3f]: ",upperx);
		upperx = get_real(upperx,STRING);
		break;
	    case 4:
		sprintf(STRING,"Step size of x-range [%6.3f]: ",stepx);
		stepx = get_real(stepx,STRING);
		break;
	    case 5:
		spectrum = get_int(0,
				   "Spectrum to take x-values from (0 for manual range): ");
		if ((spectrum < 0) || (spectrum > MAX_SPECTRUM))
		    {
		    errtype("Error, invalid spectrum number");
		    clear_command();
		    spectrum = 0;
		    }
		if ((spectrum != 0) && (DATA[spectrum-1].NPNT < 1))
		    {
		    errtype("Error, no data in spectrum");
		    clear_command();
		    spectrum = 0;
		    }
		break;
	    case 6:
		parameter = get_int(1,"Parameter number to change [1]: ");
		if ((parameter < 1) || (parameter > TH.NPAR))
		    {
		    errtype("ERROR, parameter out of range");
		    clear_command();
		    }
		else
		    {
		    parameter -= 1;
		    sprintf(STRING,"%s [%7.3f]: ",TH.NAME[parameter],
			    TH.PAR[parameter]);
		    TH.PAR[parameter] = get_real(TH.PAR[parameter],STRING);
		    }
		break;
	    case 10:

		/* get output spectrum number */

		get_inspec(&outspec);

		/* Compute x-values */

		if (spectrum == 0)
		    {
		    if (fabs(stepx) < 1e-6) stepx = upperx-lowerx;
		    if ((upperx-lowerx)*stepx < 0) stepx *= -1;
		    DATA[outspec].NPNT = (int) ((upperx-lowerx)/stepx+1.5);
		    if (DATA[outspec].NPNT > MAX_DATA)
			DATA[outspec].NPNT = MAX_DATA;
		    for (i = 0; i < DATA[outspec].NPNT; i++)
			{
			DATA[outspec].X[i] = lowerx+i*stepx;
			}
		    }
		else
		    {
		    DATA[outspec].NPNT = DATA[spectrum-1].NPNT;
		    for (i = 0; i < DATA[outspec].NPNT; i++)
			{
			DATA[outspec].X[i] = DATA[spectrum-1].X[i];
			}
		    }

		/* Calculate y-values, set error to unity */

		(*F_PTR)(DATA[outspec].X[0],TH.PAR,&DATA[outspec].Y[0],
			 dyda,CALC);
		DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
		DATA[outspec].E[0] = 1.;
		for (i = 1; i < DATA[outspec].NPNT; i++)
		    {
		    (*F_PTR)(DATA[outspec].X[i],TH.PAR,&DATA[outspec].Y[i],
			     dyda,CALC);
		    DATA[outspec].E[i] = 1.;
		    if (DATA[outspec].Y[i] < DATA[outspec].MINY)
			DATA[outspec].MINY = DATA[outspec].Y[i];
		    if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
			DATA[outspec].MAXY = DATA[outspec].Y[i];
		    }

		/* Generate title */

		sprintf(DATA[outspec].TITLE,"%s",TH.TITLE);
		type_header(0);
		break;

	    case 11:
		sprintf(STRING,"Theory: %s\n",TH.TITLE);
		type_line(STRING);
		for (i = 0; i < TH.NPAR; i++)
		    {
		    sprintf(STRING,"%2d %-30s = %8.3f\n",
			    i+1,TH.NAME[i],TH.PAR[i]);
		    type_line(STRING);
		    }
		if (spectrum == 0)
		    {
		    sprintf(STRING,
			    "Range of x-axis, lower %6.3f, upper %6.3f, step %6.3f\n",
			    lowerx,upperx,stepx);
		    }
		else
		    {
		    sprintf(STRING,"x-values equal to spectrum %1d\n",
			    spectrum);
		    }
		type_line(STRING);
		break;
	    case 20:
		list_menu("THEORY MENU",theory_menu,theory_length);
		break;
	    case 21:
		stop = TRUE;
		break;
	    }
	}
    }

/***************************************************************************/
void    theory_type(void)
/***************************************************************************/

    /*
    Get type of theory.
    */

    {

    /* define th_type_menu */

#define th_type_length 10       /* number of commands in th_type menu */

    static struct   MENU th_type_menu[th_type_length] =
	{
	"lorentz",   2,  1,  "Single Lorentzian",
	"llorentz",  2,  2,  "Double Lorentzian",
	"gauss",     1,  3,  "Single Gaussian",
	"ggauss",    2,  4,  "Double Gaussian",
	"lgauss",    2,  5,  "Lorentzian plus Gaussian",
	"glorentz",  2,  6,  "Gaussian plus Lorentzian",
	"power",     1,  7,  "Power-law lineshape",
	"voigt",     1,  8,  "Pseudo-Voigt lineshape",
	"slit",      1,  9,  "Slit function",
	"help",      1,  20, "Display menu",
	};

    char    token[100];
    float   *a,*dyda,y;

    while (TRUE)
	{
	type_header(0);
	while (!get_token(token,"Give type of theory: "));
	switch (cmnd_match(token,th_type_menu,th_type_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		lorentz_single(0.,a,&y,dyda,INIT);
		return;
	    case 2:
		lorentz_double(0.,a,&y,dyda,INIT);
		return;
	    case 3:
		gauss_single(0.,a,&y,dyda,INIT);
		return;
	    case 4:
		gauss_double(0.,a,&y,dyda,INIT);
		return;
	    case 5:
		lorentz_gauss(0.,a,&y,dyda,INIT);
		return;
	    case 6:
		gauss_lorentz(0.,a,&y,dyda,INIT);
		return;
	    case 7:
		power(0.,a,&y,dyda,INIT);
		return;
	    case 8:
		pseudo_voigt(0.,a,&y,dyda,INIT);
		return;
	    case 9:
		slit(0.,a,&y,dyda,INIT);
		return;
	    case 20:
		list_menu("THEORY TYPES",th_type_menu,th_type_length);
		break;
	    }
	}
    }
