/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
/*$$$$$                       operate.c                               $$$$$*/
/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/

/****************************************************************************

  CVS information:

  $Id: operate.c,v 1.7 2002/03/28 11:48:31 wilcke Exp $

****************************************************************************/
/*
Update 26/03/2002 R. Wilcke (wilcke@esrf.fr)
                  operate(): changed abbreviations for menu commands to the
                  agreed set.
Update 26/11/2001 O. Svensson
                  Fixed bug in line 1528 (fprintf).
Update 11/06/2001 E. Vlieg (vlieg@sci.kun.nl)
                  Normalize integrated intensity to STEPSIZE (and not only the
		  structure factor). Different stepsizes will now yield
                  similar integrated intensity, as was already the case for F.
Update 08/06/2001 E. Vlieg (vlieg@sci.kun.nl)
                  Use FILETYPE in automerge() to make it useful for all file
                  types.
Update 20/12/2000 E. Vlieg (vlieg@sci.kun.nl)
		  Small changes in menu help texts
		  Remove REFLECT function (not used anymore)
Update 06/10/2000 E. Vlieg (vlieg@sci.kun.nl)
                  Add correction factor for transmission.
		  Add level() function.
*/
/***************************************************************************/
/*      include files                                                      */
/***************************************************************************/

#define	EXT extern
#include "ana.h"
#undef EXT

/***************************************************************************/
void    add(void)
/***************************************************************************/

    /*
    Add a constant to the y-values of a spectrum.
    */

    {

    float   constant;
    int     inspec,outspec,i;

    /* Get constant value and spectrum numbers */

    constant = get_real(0.,"Give constant to be added [0.]: ");
    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Add constant to y-values */

    for (i = 0; i < DATA[inspec].NPNT; i++)
	{
	DATA[outspec].X[i] = DATA[inspec].X[i];
	DATA[outspec].Y[i] = DATA[inspec].Y[i]+constant;
	DATA[outspec].E[i] = DATA[inspec].E[i];
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"%10.3g added to spectrum %1d",
	    constant,inspec+1);
    DATA[outspec].NPNT = DATA[inspec].NPNT;
    DATA[outspec].MINY = DATA[inspec].MINY+constant;
    DATA[outspec].MAXY = DATA[inspec].MAXY+constant;
    type_header(1);

    }

/***************************************************************************/
void    append(void)
/***************************************************************************/

    /*
    Append two spectra.
    */

    {

    int     in_spec1,in_spec2,out_spec;
    int     i;

    /* Get input and output spectrum numbers */

    get_2inoutspec(&in_spec1,&in_spec2,&out_spec);

    if ((DATA[in_spec1].NPNT < 1) || (DATA[in_spec2].NPNT < 1))
	{
	errtype("No data points in one of the spectra");
	clear_command();
	return;
	}

    /* Copy first spectrum to temporary spectrum */

    for (i = 0; i < DATA[in_spec1].NPNT; i++)
	{
	XD[i] = DATA[in_spec1].X[i];
	YD[i] = DATA[in_spec1].Y[i];
	ED[i] = DATA[in_spec1].E[i];
	}

    /* Copy second spectrum to temporary spectrum */

    for (i = 0; i < DATA[in_spec2].NPNT; i++)
	{
	XD[DATA[in_spec1].NPNT+i] = DATA[in_spec2].X[i];
	YD[DATA[in_spec1].NPNT+i] = DATA[in_spec2].Y[i];
	ED[DATA[in_spec1].NPNT+i] = DATA[in_spec2].E[i];
	}

    /* Update spectrum information */

    DATA[out_spec].NPNT = DATA[in_spec1].NPNT+DATA[in_spec2].NPNT;
    if (DATA[in_spec1].MINY < DATA[in_spec2].MINY)
	{
	DATA[out_spec].MINY = DATA[in_spec1].MINY;
	}
    else
	{
	DATA[out_spec].MINY = DATA[in_spec2].MINY;
	}
    if (DATA[in_spec1].MAXY > DATA[in_spec2].MAXY)
	{
	DATA[out_spec].MAXY = DATA[in_spec1].MAXY;
	}
    else
	{
	DATA[out_spec].MAXY = DATA[in_spec2].MAXY;
	}
    sprintf(DATA[out_spec].TITLE,"Appending of spectra %1d and %1d",
	    in_spec1+1,in_spec2+1);

    /* Store result in output spectrum */

    for (i = 0; i < DATA[out_spec].NPNT; i++)
	{
	DATA[out_spec].X[i] = XD[i];
	DATA[out_spec].Y[i] = YD[i];
	DATA[out_spec].E[i] = ED[i];
	}

    }

/***************************************************************************/
void    auto_merge(void)
/***************************************************************************/

    /*
    Looks for next-higher scan file number and sees whether this has the
    same (hkl). If this is the case, it's read into the next-higher scan
    number and merged with the first one in the initial spectrum number.
    This is unfortunately an option that is useful as long as no integrated
    peak and background scans can be performed in SURFD.
    */

    {

    float	h_save,k_save,l_save;
    int		read_save,scan_save,npnt_save;


    if (READSPEC+2 == MAX_SPECTRUM)
    	{
        errtype("Can't do automerge with first file as last spectrum");
        return;
        }

    h_save = HMIL;
    k_save = KMIL;
    l_save = LMIL;
    read_save = READSPEC;
    scan_save = SCANNR;
    npnt_save = DATA[READSPEC+1].NPNT;
    DATA[READSPEC+1].NPNT = 0; /* used for checking the read operation */

    /* Try reading in the next run number */

    sprintf(STRING,"%s %1d %1d",FILETYPE,SCANNR+1,READSPEC+2);
    put_command(STRING);
    read_type(); /* Will change value of READSPEC! */
    READSPEC = read_save;
    SCANNR = scan_save;

    if (DATA[READSPEC].NPNT == 0)
    	{
        errtype("Automerge failed");
        HMIL = h_save;
        KMIL = k_save;
        LMIL = l_save;
        DATA[READSPEC+1].NPNT = npnt_save;
        return;
        }

    /* Check whether new file has same (hkl) */

    if ((fabs(HMIL-h_save) < 1e-2) && (fabs(KMIL-k_save) < 1e-2) &&
    	(fabs(LMIL-l_save) < 1e-2))
        {
        sprintf(STRING,"%1d %1d %1d",read_save+1,READSPEC+2,read_save+1);
        put_command(STRING);
        merge();
        }
    else
    	{
        errtype("Automerge failed, (hkl) of adjacent scans not equal");
        HMIL = h_save;
        KMIL = k_save;
        LMIL = l_save;
        return;
        }
    }


/***************************************************************************/
void    center(void)
/***************************************************************************/

    /*
    Shift the x-values of a spectrum such that the maximum y value
    corresponds to x = 0.
    */

    {

    float   offset;
    int     inspec,outspec,i;

    /* Get spectrum numbers */

    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Find x value corresponding to maximum y value */

    for (i = 0; i < DATA[inspec].NPNT; i++)
	{
	if (fabs(DATA[inspec].Y[i] - DATA[inspec].MAXY) < 1e-6)
	    {
	    offset = DATA[inspec].X[i];
	    break;
	    }
	}

    /* Shift spectrum by offset */

    for (i = 0; i < DATA[inspec].NPNT; i++)
	{
	DATA[outspec].X[i] = DATA[inspec].X[i]-offset;
	DATA[outspec].Y[i] = DATA[inspec].Y[i];
	DATA[outspec].E[i] = DATA[inspec].E[i];
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d centered",inspec+1);
    DATA[outspec].NPNT = DATA[inspec].NPNT;
    DATA[outspec].MINY = DATA[inspec].MINY;
    DATA[outspec].MAXY = DATA[inspec].MAXY;
    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[outspec].X[0],DATA[outspec].X[DATA[outspec].NPNT-1]);
    type_line(STRING);
    type_header(1);

    }


/***************************************************************************/
void    cutoff(void)
/***************************************************************************/

    /*
    Cutoff the y-values of a spectrum:
	- above value if cutoff positive
	- below value if cutoff negative.
    */

    {

    float   cutval;
    int     inspec,outspec,i;
    int     npoints,sign;

    /* Get cutoff value and spectrum numbers */

    cutval = get_real(0.,
		      "Give cutoff (+/- y-values above/below value are deleted) [0.]: ");
    if (fabs(cutval) < 1e-6)
	{
	clear_command();
	return;
	}
    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Make cutoff spectrum */

    npoints = 0;
    if (cutval < 0.)
	{
	sign = -1.;
	}
    else
	{
	sign = 1.;
	}
    for (i = 0; i < DATA[inspec].NPNT; i++)
	{
	if (sign*DATA[inspec].Y[i] < cutval)
	    {
	    DATA[outspec].X[npoints] = DATA[inspec].X[i];
	    DATA[outspec].Y[npoints] = DATA[inspec].Y[i];
	    DATA[outspec].E[npoints] = DATA[inspec].E[i];
	    npoints++;
	    }
	}

    /* Generate spectrum information */

    if (cutval < 0.)
	{
	sprintf(DATA[outspec].TITLE,"spectrum %1d cutoff below %10.3g",
		inspec+1,-cutval);
	}
    else
	{
	sprintf(DATA[outspec].TITLE,"spectrum %1d cutoff above %10.3g",
		inspec+1,cutval);
	}
    DATA[outspec].NPNT = npoints;
    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    for (i = 1; i < DATA[outspec].NPNT; i++)
	{
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}
    type_header(1);

    }
/***************************************************************************/
void    delete_points(void)
/***************************************************************************/

    /*
    Delete range of points in spectrum.
    */

    {

    int first,last;
    int inspec,outspec,i;

    /* Get first and last point to be deleted and spectrum numbers */

    first = get_int(1,"Give first point to be deleted [1]: ");
    last = get_int(1,"Give last point to be deleted [1]: ");
    if ((last < first) || (first < 1) || (last < 1))
	{
	errtype("ERROR, invalid range");
	clear_command();
	return;
	}
    get_inoutspec(&inspec,&outspec);
    if ((DATA[inspec].NPNT < first) ||  (DATA[inspec].NPNT < last))
	{
	errtype("ERROR, not that many points in spectrum");
	clear_command();
	return;
	}

    /* Delete the specified range */

    for (i = 0; i < first-1; i++)
	{
	DATA[outspec].X[i] = DATA[inspec].X[i];
	DATA[outspec].Y[i] = DATA[inspec].Y[i];
	DATA[outspec].E[i] = DATA[inspec].E[i];
	}
    for (i = last; i < DATA[inspec].NPNT; i++)
	{
	DATA[outspec].X[i-last+first-1] = DATA[inspec].X[i];
	DATA[outspec].Y[i-last+first-1] = DATA[inspec].Y[i];
	DATA[outspec].E[i-last+first-1] = DATA[inspec].E[i];
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d points %1d to %1d deleted",
	    inspec+1,first,last);
    DATA[outspec].NPNT = DATA[inspec].NPNT+first-last-1;
    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    for (i = 1; i < DATA[outspec].NPNT; i++)
	{
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}
    type_header(1);

    }

/***************************************************************************/
void    despike(void)
/***************************************************************************/

    /*
    Try to remove spikes from a spectrum. Spikes are single points that are
    several standard deviations higher/lower than their neighbouring
    background points.
    */

    {

    int     inspec,outspec,i,npoints;
    float   xleft,xright,yleft,yright,xmiddle,ymiddle,slope,offset;
    float   y1,y2,y3;

    /* Get spectrum numbers */

    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 2*NLEVELING)
	{
	errtype("ERROR, too few data points for despiking");
	clear_command();
	return;
	}

    /* Determine the (sloping) background in spectrum in order to
       calculate expected y-values */

    /* Determine average x and y values on the left and right sides of
       the spectrum */

    xleft=yleft=xright=yright=0.;
    for (i = 0; i < NLEVELING; i++)
	{
	yleft += DATA[inspec].Y[i];
	xleft += DATA[inspec].X[i];
	}
    yleft /= NLEVELING;
    xleft /= NLEVELING;
    for (i = DATA[inspec].NPNT-NLEVELING; i < DATA[inspec].NPNT; i++)
	{
	yright += DATA[inspec].Y[i];
	xright += DATA[inspec].X[i];
	}
    yright /= NLEVELING;
    xright /= NLEVELING;

    /* Determine slope and middle of the spectrum */

    slope = (yright-yleft)/(xright-xleft+1e-10);
    xmiddle = (xleft+xright)/2;
    ymiddle = (yleft+yright)/2;

    /* Remove spikes */

    XD[0] = DATA[inspec].X[0];
    YD[0] = DATA[inspec].Y[0];
    ED[0] = DATA[inspec].E[0];
    npoints = 1;
    for (i = 1; i < DATA[inspec].NPNT-1; i++)
	{
	/* calculate expected y-values around the data point */
	y1 = ymiddle-(DATA[inspec].X[i-1]-xmiddle)*slope;
	y2 = ymiddle-(DATA[inspec].X[i]-xmiddle)*slope;
	y3 = ymiddle-(DATA[inspec].X[i+1]-xmiddle)*slope;
	if (
	    (fabs((DATA[inspec].Y[i-1]-y1)/DATA[inspec].E[i-1]) < 2) &&
	    (fabs((DATA[inspec].Y[i]-y2)/DATA[inspec].E[i]) > 6) &&
	    (fabs((DATA[inspec].Y[i+1]-y3)/DATA[inspec].E[i+1]) < 2)
	    )
	    continue; /* it was a spike! */
	else
	    {
	    XD[npoints] = DATA[inspec].X[i];
	    YD[npoints] = DATA[inspec].Y[i];
	    ED[npoints] = DATA[inspec].E[i];
	    npoints++;
	    }
	}
    i = DATA[inspec].NPNT-1;
    XD[npoints] = DATA[inspec].X[i];
    YD[npoints] = DATA[inspec].Y[i];
    ED[npoints] = DATA[inspec].E[i];
    npoints++;

    /* Write spectrum to output file and generate spectrum information */

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d, de-spiked",inspec+1);
    DATA[outspec].NPNT = npoints;
    DATA[outspec].MINY = DATA[outspec].MAXY = YD[0];
    for (i = 0; i < DATA[outspec].NPNT; i++)
	{
	DATA[outspec].X[i] = XD[i];
	DATA[outspec].Y[i] = YD[i];
	DATA[outspec].E[i] = ED[i];
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}
    type_header(1);

    }

/***************************************************************************/
void    divide(void)
/***************************************************************************/

    /* 
    Divide two buffers by each other
    */

    {
    int     in_spec1,in_spec2,out_spec;
    int     i, n;
    double  del1, del2;
    char    string[200];

    /* Get input and output spectrum numbers */

    get_2inoutspec(&in_spec1,&in_spec2,&out_spec);

    if (DATA[in_spec1].NPNT != DATA[in_spec2].NPNT)
	{
	errtype("ERROR, Buffers must have same number of points");
	clear_command();
	return;
	}


    for (n=0, i=0 ;i < DATA[in_spec1].NPNT ; i++ ) {
    if (fabs(DATA[in_spec2].Y[i]) > 1e-20) {
    if (DATA[in_spec1].X[i] != DATA[in_spec2].X[i]) {
    sprintf(string, "WARNING, different x values: %.5f %.5f",
	    DATA[in_spec1].X[i], DATA[in_spec2].X[i]);
    errtype(string);
    } 
    DATA[out_spec].X[n] = DATA[in_spec1].X[i];
    DATA[out_spec].Y[n] = DATA[in_spec1].Y[i] / DATA[in_spec2].Y[i];
    DATA[out_spec].E[n] = sqrt(  sqr(DATA[in_spec1].E[i] / DATA[in_spec2].Y[i])
				 + sqr(DATA[in_spec1].Y[i] * DATA[in_spec2].E[i]
				       /sqr(DATA[in_spec2].Y[i])
				     )
	);
    n++;
    } /* endif */
    } /* endfor */

    /* Generate spectrum information */

    sprintf(DATA[out_spec].TITLE,"spectrum %1d divided by spectrum %1d",
	    in_spec1+1, in_spec2+1);
    DATA[out_spec].NPNT = n;
    DATA[out_spec].MINY = DATA[out_spec].MAXY = DATA[out_spec].Y[0];
    for (i = 1; i < DATA[out_spec].NPNT; i++)
	{
	if (DATA[out_spec].Y[i] < DATA[out_spec].MINY)
	    DATA[out_spec].MINY = DATA[out_spec].Y[i];
	if (DATA[out_spec].Y[i] > DATA[out_spec].MAXY)
	    DATA[out_spec].MAXY = DATA[out_spec].Y[i];
	}
    type_header(1);
    }
 
/***************************************************************************/
void    differentiate(void)
/***************************************************************************/

    /*
    Numerically differentiate a spectrum.
    */

    {

    int     inspec,outspec,i,j;

    /* Get spectrum numbers */

    get_inoutspec(&inspec,&outspec);
    if ((DATA[inspec].NPNT < 3))
	{
	errtype("ERROR, too few datapoints in spectrum");
	clear_command();
	return;
	}

    /* Do differentiating */

    for (i = 1; i < DATA[inspec].NPNT-1; i++)
	{
	DATA[outspec].X[i-1] = DATA[inspec].X[i];
	DATA[outspec].Y[i-1] = (DATA[inspec].Y[i+1]-DATA[inspec].Y[i-1])/
	    (DATA[inspec].X[i+1]-DATA[inspec].X[i-1]);
	DATA[outspec].E[i] = sqrt(sqr(DATA[inspec].E[i+1])+
				  sqr(DATA[inspec].E[i-1]));
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d differentiated",
	    inspec+1);
    DATA[outspec].NPNT = DATA[inspec].NPNT-2;
    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    for (i = 1; i < DATA[outspec].NPNT; i++)
	{
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}
    type_header(1);

    }

/***************************************************************************/
void    integrate(void)
/***************************************************************************/

    /*
    Numerically integrate a spectrum. If DATARUN the structure factor is
    also calculated.
    */

    {

    int     inspec,i,istart;
    float   background,backerr,step,stepsum,sum,sumerr;
    float   f,ferr,correction,lorentz,area,polarization,intercept;
    float   transmission,tthb, cosepssqr;

    /* Get value and spectrum numbers */

    get_inspec(&inspec);
    if ((DATA[inspec].NPNT < 3) || (DATA[inspec].NPNT <= 2*NBACK))
	{
	errtype("ERROR, too few datapoints in spectrum");
	clear_command();
	return;
	}

    /* Input background */

    if (AUTO_BACK)
	{
	background = backerr = 0;
	for (i = 0; i < NBACK; i++)
	    {
	    background += DATA[inspec].Y[i];
	    backerr += sqr(DATA[inspec].E[i]);
	    }
	for (i = DATA[inspec].NPNT-NBACK; i < DATA[inspec].NPNT; i++)
	    {
	    background += DATA[inspec].Y[i];
	    backerr += sqr(DATA[inspec].E[i]);
	    }
	if (NBACK > 0)
	    {
	    background = background/(2*NBACK);
	    backerr = sqrt(backerr)/(2*NBACK);
	    }
	}
    else
	{
	sprintf(STRING,
		"Background to be subtracted (< 0 gives default) [%10.3e]: ",
		TH.PAR[0]);
	background = get_real(TH.PAR[0],STRING);
	if (background < 0) background = TH.PAR[0];
	sprintf(STRING,"Error in background (< 0 gives default) [%10.3e]: ",
		TH.ERR[0]);
	backerr = get_real(TH.ERR[0],STRING);
	if (backerr < 0) backerr = TH.ERR[0];
	}

    /* Compute integrated intensity. If no points are skipped, take for
       the first point  a step-size equal to the one for the second point */

    if (NBACK == 0)
	{
	if ((INT_RANGE == 0) ||
	    (fabs(DATA[inspec].X[0]-TH.PAR[2]) < fabs(INT_RANGE*TH.PAR[4])))
	    {
	    step = fabs(DATA[inspec].X[1]-DATA[inspec].X[0]);
	    stepsum = step;
	    sum = DATA[inspec].Y[0]*step;
	    sumerr = sqr(step*DATA[inspec].E[0]);
	    }
	else
	    {
	    stepsum = sum = sumerr = 0;
	    }
	istart = 1;
	}
    else
	{
	stepsum = sum = sumerr = 0;
	istart = NBACK;
	}

    for (i = istart; i < DATA[inspec].NPNT-NBACK; i++)
	{
	if ((INT_RANGE == 0) ||
	    (fabs(DATA[inspec].X[i]-TH.PAR[2]) < fabs(INT_RANGE*TH.PAR[4])))
	    {
	    step = fabs(DATA[inspec].X[i]-DATA[inspec].X[i-1]);
	    stepsum += step;
	    sum += DATA[inspec].Y[i]*step;
	    sumerr += sqr(step*DATA[inspec].E[i]);
	    }
	}
    sum = (sum-stepsum*background)/STEPSIZE;
    if (sum < 0) sum = 0;
    sumerr = sqrt(sumerr+sqr(stepsum*backerr))/STEPSIZE;

    sprintf(STRING,"Integrated intensity: %10.4f +/- %10.4f\n",
	    sum,sumerr);
    type_line(STRING);

    /* If DATARUN, also calculate the structure factor */

    if (DATARUN)
	{

	/* Calculate correction factors */

	comp_cor(&lorentz, &polarization, &intercept, &area, &transmission);

	correction = lorentz*area*polarization*intercept*transmission;

	f = sqrt(sum*correction);
	ferr = sqrt(sqr(f)+sumerr*correction)-f;

	sprintf(STRING,"Structure factor    : %10.4f +/- %10.4f\n",
		f,ferr);
	type_line(STRING);
	}

    /* If requested, write results to output file */

    if (OUTINT)
	{
	if (DATARUN)
	    {
	    if (!SCANSWRITTEN)
		{
		fprintf(INTFILE,
			"%4s %8s %8s %8s %10s %7s %9s %9s %9s %9s %9s",
			"run#","H","K","L","F","dF",
			"tot.corr","area","polar","interc","transm");
		if (RSCAN)
		    {
		    fprintf(INTFILE, " %9s\n","lorentz");
		    }
		else
		    {
		    fprintf(INTFILE, "\n");
		    }
		}
	    SCANSWRITTEN++;
	    fprintf(INTFILE,
		    "%4d %8.3f %8.3f %8.3f %10.2f %7.2f %9.4f %9.4f %9.4f %9.4f %9.4f",
		    SCANNR,
		    HMIL*T11+KMIL*T12+LMIL*T13,
		    HMIL*T21+KMIL*T22+LMIL*T23,
		    HMIL*T31+KMIL*T32+LMIL*T33,
		    f,ferr,correction,area,polarization,intercept,transmission);
	    if (RSCAN)
		{
		fprintf(INTFILE, "%9.4f\n", lorentz);
		}
	    else
		{
		fprintf(INTFILE, "\n");
		}
	    }
	else
	    {
	    if (!SCANSWRITTEN)
		{
		fprintf(INTFILE, "%4s %8s %8s %8s %10s %10s\n",
			"run#", "    H   ", "    K   ", "    L   ",
			"     I    ", "     dI    ");
		}
	    SCANSWRITTEN++;
	    fprintf(INTFILE,"%4d %8.3f %8.3f %8.3f %10.2f %10.2f\n",
		    SCANNR,
		    HMIL*T11+KMIL*T12+LMIL*T13,
		    HMIL*T21+KMIL*T22+LMIL*T23,
		    HMIL*T31+KMIL*T32+LMIL*T33,
		    sum,sumerr);
	    }
	}

    type_header(0);

    }

/***************************************************************************/
void	fit_subtract(void)
/***************************************************************************/

    /*
    Subtract fit function from spectrum
    */

    {

    int inspec, outspec, i;
    float fit;


    get_inoutspec(&inspec,&outspec);

    if ((DATA[inspec].NPNT < 3))
	{
	errtype("ERROR, too few datapoints in spectrum");
	clear_command();
	return;
	}

    for (i=0; i<DATA[inspec].NPNT; i++)
        {
        DATA[outspec].X[i] = DATA[inspec].X[i];
        DATA[outspec].E[i] = DATA[inspec].E[i];

        (*F_PTR)(DATA[inspec].X[i], TH.PAR, &fit, NULL, CALC);

        DATA[outspec].Y[i] = DATA[inspec].Y[i] - fit;
        }

    DATA[outspec].NPNT = DATA[inspec].NPNT;
    sprintf(DATA[outspec].TITLE, "spectrum %d minus last fit", inspec+1);

    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    for (i = 1; i < DATA[outspec].NPNT; i++)
	{
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}
    type_header(1);
    }

/***************************************************************************/
void    level(void)
/***************************************************************************/

    /*
    Try to level the sloping background of a spectrum. NLEVELING points on
    either side of the spectrum are used to determine the slope in the
    background.
    */

    {

    int     inspec,outspec,i;
    float   xleft,xright,yleft,yright,xmiddle,slope,offset;

    /* Get spectrum numbers */

    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 2*NLEVELING)
	{
	errtype("ERROR, too few data points for leveling");
	clear_command();
	return;
	}

    /* Determine average x and y values on the left and right sides of
       the spectrum */

    xleft=yleft=xright=yright=0.;
    for (i = 0; i < NLEVELING; i++)
	{
	yleft += DATA[inspec].Y[i];
	xleft += DATA[inspec].X[i];
	}
    yleft /= NLEVELING;
    xleft /= NLEVELING;
    for (i = DATA[inspec].NPNT-NLEVELING; i < DATA[inspec].NPNT; i++)
	{
	yright += DATA[inspec].Y[i];
	xright += DATA[inspec].X[i];
	}
    yright /= NLEVELING;
    xright /= NLEVELING;

    /* Deterimine slope and middle of the spectrum */

    slope = (yright-yleft)/(xright-xleft+1e-10);
    xmiddle = (xleft+xright)/2;

    /* Write leveled x-values to output spectrum */

    for (i = 0; i < DATA[inspec].NPNT; i++)
	{
	offset = (DATA[inspec].X[i]-xmiddle)*slope;
	DATA[outspec].X[i] = DATA[inspec].X[i];
	DATA[outspec].Y[i] = DATA[inspec].Y[i]-offset;
	DATA[outspec].E[i] = sqrt(sqr(DATA[inspec].E[i])+offset);
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d, leveled",inspec+1);
    DATA[outspec].NPNT = DATA[inspec].NPNT;
    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    for (i = 1; i < DATA[outspec].NPNT; i++)
	{
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}
    type_header(1);

    }

/***************************************************************************/
void    lump(void)
/***************************************************************************/

    /*
    Lump given number of data points together. Possible residual points at
    the end of a spectrum are deleted.
    */

    {

    int     nlump;
    int     inspec,outspec,i,j;

    /* Get nlump and spectrum numbers */

    nlump = get_int(1,"Number of points to be lumped together [1]: ");
    if (nlump < 1)
	{
	errtype("Invalid number of lumping points");
	return;
	}
    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < nlump)
	{
	errtype("ERROR, too few data points for lumping");
	clear_command();
	return;
	}

    /* Lump x-values in temporary spectrum */

    for (i = 0; i < DATA[inspec].NPNT/nlump; i++)
	{
	XD[i] = 0.;
	YD[i] = 0.;
	ED[i] = 0.;
	for (j = 0; j < nlump; j++)
	    {
	    XD[i] += DATA[inspec].X[i*nlump+j];
	    YD[i] += DATA[inspec].Y[i*nlump+j];
	    ED[i] += sqr(DATA[inspec].E[i*nlump+j]);
	    }
	}

    /* Write values to output spectrum, and keep original normalization */

    for (i = 0; i < DATA[inspec].NPNT/nlump; i++)
	{
	DATA[outspec].X[i] = XD[i]/nlump;
	DATA[outspec].Y[i] = YD[i]/nlump;
	DATA[outspec].E[i] = sqrt(ED[i])/nlump;
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d, %1d points lumped together",
	    inspec+1,nlump);
    DATA[outspec].NPNT = DATA[inspec].NPNT/nlump;
    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    for (i = 1; i < DATA[outspec].NPNT; i++)
	{
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}
    type_header(1);

    }

/***************************************************************************/
void    merge(void)
/***************************************************************************/

    /*
    Merge two spectra, using a weighting of the y-values based on the errors.
    Input spectra with decreasing x-values are allowed, but the output
    spectrum will always have the x-values in increasing order.
    */

    {

    int     in_spec1,in_spec2,out_spec;
    int     i,j,npoints,iplus,jplus,iend,jend;
    int     done1,done2;

    /* Get input and output spectrum numbers */

    get_2inoutspec(&in_spec1,&in_spec2,&out_spec);

    if ((DATA[in_spec1].NPNT < 2) || (DATA[in_spec2].NPNT < 2))
	{
	errtype("Too few datapoints for merging");
	clear_command();
	return;
	}

    /* Check whether the x-values are not constant */

    if (fabs(DATA[in_spec1].X[0] - DATA[in_spec1].X[1]) < 1e-6)
	{
	sprintf(STRING,"Error, x-values constant in spectrum %1d",
		in_spec1+1);
	errtype(STRING);
	return;
	}
    if (fabs(DATA[in_spec2].X[0] - DATA[in_spec2].X[1]) < 1e-6)
	{
	sprintf(STRING,"Error, x-values constant in spectrum %1d",
		in_spec2+1);
	errtype(STRING);
	return;
	}

    /* Determine whether the x-values are increasing or decreasing */

    if (DATA[in_spec1].X[0] < DATA[in_spec1].X[1])
	{
	i = 0;
	iplus = 1;
	iend = DATA[in_spec1].NPNT-1;
	}
    else
	{
	i = DATA[in_spec1].NPNT-1;
	iplus = -1;
	iend = 0;
	}
    if (DATA[in_spec2].X[0] < DATA[in_spec2].X[1])
	{
	j = 0;
	jplus = 1;
	jend = DATA[in_spec2].NPNT-1;
	}
    else
	{
	j = DATA[in_spec2].NPNT-1;
	jplus = -1;
	jend = 0;
	}

    /* Merge the two spectra while both have still points left */

    npoints = 0;
    done1 = done2 = FALSE;
    while (!(done1 || done2))
	{
	if ((fabs(DATA[in_spec1].X[i]-DATA[in_spec2].X[j]) <
	     1e-5*fabs(DATA[in_spec1].X[i])) ||
	    (fabs(DATA[in_spec1].X[i]-DATA[in_spec2].X[j]) < 1e-6))
	    {
	    XD[npoints] = DATA[in_spec1].X[i];
	    YD[npoints] = (DATA[in_spec1].Y[i]*sqr(DATA[in_spec2].E[j])
			   +DATA[in_spec2].Y[j]*sqr(DATA[in_spec1].E[i]))
		/(sqr(DATA[in_spec1].E[i])+sqr(DATA[in_spec2].E[j]));
	    ED[npoints] = 1/sqrt(1/sqr(DATA[in_spec1].E[i])
				 +1/sqr(DATA[in_spec2].E[j]));
	    i+=iplus;
	    j+=jplus;
	    }
	else if (DATA[in_spec1].X[i] < DATA[in_spec2].X[j])
	    {
	    XD[npoints] = DATA[in_spec1].X[i];
	    YD[npoints] = DATA[in_spec1].Y[i];
	    ED[npoints] = DATA[in_spec1].E[i];
	    i+=iplus;
	    }
	else
	    {
	    XD[npoints] = DATA[in_spec2].X[j];
	    YD[npoints] = DATA[in_spec2].Y[j];
	    ED[npoints] = DATA[in_spec2].E[j];
	    j+=jplus;
	    }
	npoints++;
	if (i == iend+iplus) done1 = TRUE;
	if (j == jend+jplus) done2 = TRUE;
	if (npoints == MAX_DATA)
	    {
	    errtype("ERROR, too many datapoints");
	    done1 = done2 = TRUE;
	    }
	}

    /* Complete merged spectrum with remaining points in first spectrum */

    while (!done1)
	{
	XD[npoints] = DATA[in_spec1].X[i];
	YD[npoints] = DATA[in_spec1].Y[i];
	ED[npoints] = DATA[in_spec1].E[i];
	i+=iplus;
	npoints++;
	if (i == iend+iplus) done1 = TRUE;
	if (npoints == MAX_DATA)
	    {
	    errtype("ERROR, too many datapoints");
	    done1 = done2 = TRUE;
	    }
	}

    /* Complete merged spectrum with remaining points in second spectrum */

    while (!done2)
	{
	XD[npoints] = DATA[in_spec2].X[j];
	YD[npoints] = DATA[in_spec2].Y[j];
	ED[npoints] = DATA[in_spec2].E[j];
	j+=jplus;
	npoints++;
	if (j == jend+jplus) done2 = TRUE;
	if (npoints == MAX_DATA)
	    {
	    errtype("ERROR, too many datapoints");
	    done1 = done2 = TRUE;
	    }
	}

    /* Store result in output spectrum */

    for (i = 0; i < npoints; i++)
	{
	DATA[out_spec].X[i] = XD[i];
	DATA[out_spec].Y[i] = YD[i];
	DATA[out_spec].E[i] = ED[i];
	}

    /* Get spectrum information */

    DATA[out_spec].NPNT = npoints;
    DATA[out_spec].MINY = DATA[out_spec].MAXY = DATA[out_spec].Y[0];
    for (i = 1; i < DATA[out_spec].NPNT; i++)
	{
	if (DATA[out_spec].Y[i] < DATA[out_spec].MINY)
	    DATA[out_spec].MINY = DATA[out_spec].Y[i];
	if (DATA[out_spec].Y[i] > DATA[out_spec].MAXY)
	    DATA[out_spec].MAXY = DATA[out_spec].Y[i];
	}
    sprintf(DATA[out_spec].TITLE,"Merging of spectra %1d and %1d",
	    in_spec1+1,in_spec2+1);
    type_header(1);

    }

/***************************************************************************/
void    multiply(void)
/***************************************************************************/

    /*
    Multiply the y-values of a spectrum by a given factor.
    */

    {

    float   factor, dummy;
    int     inspec,outspec,i;

    /* Get value and spectrum numbers */

    factor = get_real(1.,"Give multiplication factor [1.]: ");
    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Multiply y-values */

    for (i = 0; i < DATA[inspec].NPNT; i++)
	{
	DATA[outspec].X[i] = DATA[inspec].X[i];
	DATA[outspec].Y[i] = DATA[inspec].Y[i]*factor;
	DATA[outspec].E[i] = DATA[inspec].E[i]*fabs(factor);
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d multiplied by %10.3g",
	    inspec+1,factor);
    DATA[outspec].NPNT = DATA[inspec].NPNT;
    if (factor < 0) {
    dummy = DATA[inspec].MAXY*factor;
    DATA[outspec].MAXY = DATA[inspec].MINY*factor;
    DATA[outspec].MINY = dummy;
    }
    else {
    DATA[outspec].MINY = DATA[inspec].MINY*factor;
    DATA[outspec].MAXY = DATA[inspec].MAXY*factor;
    }

    type_header(1);

    }

/***************************************************************************/
void    operate(void)
/***************************************************************************/

    /*
    Ordinate and spectrum operation menu.
    */

    {

    /* define operate_menu */

#define operate_length 24      /* number of commands in operate menu */

    static struct   MENU operate_menu[operate_length] =
	{
	"merge",     1,  1,  "Merge two spectra",
	"append",    2,  2,  "Append two spectra",
	"sum",       1,  3,  "Sum y-values of two spectra",
	"shift",     2,  4,  "Shift x-values of spectrum",
	"center",    1,  5,  "Shift x-values to put maximum at x=0",
	"scale",     2,  6,  "Multiply x-values by constant",
	"delete",    1,  7,  "Delete range of points from spectrum",
	"wrap",      1,  8,  "Wrap x-range to avoid 360 degr jumps",
	"multiply",  2,  9,  "Multiply y-values by constant",
	"add",       1,  10, "Add constant to y-values",
	"cutoff",    2,  11, "Cut off points; +/- = above/below value",
	"lump",      2,  12, "Lump data points",
	"integrate", 1,  13, "Numerically integrate a spectrum",
/*	"reflect",   3,  14, "Get str. factor for reflectivity scan",  */
        "automerge", 2,  15, "Try merging with next-higher run number",
        "different", 2,  16, "Differentiate spectrum",
        "fitsub",    2,  17, "Subtract last fit from spectrum",
        "split",     2,  18, "Split spectrum into peak and background",
	"div",       3,  19, "Divide two buffers by each other",
	"log10",     2,  20, "Take log10 of y values",
	"reftof",    4,  21, "Convert refl. ridge scan to str. factors",
	"level",     2,  22, "Level spectrum with sloping background",
	"despike",   3,  23, "Remove spikes from spectrum",
	"help",      1,  30, "Display menu",
	"return",    1,  31, "Return to main menu"
	};

    int     stop = FALSE;
    char    token[100];

    while (!stop)
	{
	type_header(0);
	if (!get_token(token,"ANA.OPERATE>")) return;
	switch (cmnd_match(token,operate_menu,operate_length))
	    {
	    case -1:
		break;
	    case 0:
		break;
	    case 1:
		merge();
		return;
	    case 2:
		append();
		return;
	    case 3:
		sum();
		return;
	    case 4:
		shift();
		return;
	    case 5:
		center();
		return;
	    case 6:
		scale();
		return;
	    case 7:
		delete_points();
		return;
	    case 8:
		wrap();
		return;
	    case 9:
		multiply();
		return;
	    case 10:
		add();
		return;
	    case 11:
		cutoff();
		return;
	    case 12:
		lump();
		return;
	    case 13:
		integrate();
		return;
/*	    case 14:
	    ref_integrate();
	    return;           */
            case 15:
            	auto_merge();
		return;
	    case 16:
		differentiate();
                return;
            case 17:
		fit_subtract();
                return;
            case 18:
            	split();
                return;
	    case 19:
                divide();
		return;
	    case 20:
		ylog10();
		return;
	    case 21:
		ref_to_f();
		return;
	    case 22:
		level();
		return;
	    case 23:
		despike();
		return;
	    case 30:
		list_menus("Ordinate/spectrum operation menu",
			   operate_menu,operate_length,ROWS-MAX_SPECTRUM-3);
		break;
	    case 31:
		stop = TRUE;
	    }
	}
    }

#if defined (neverheardoff)
/***************************************************************************/
void    ref_integrate(void)
/***************************************************************************/

    /*
    Calculate peak intensity and structure factor for reflectivity scan.
    */

    {

    int     inspec,i,ipos;
    float   background,backerr,position,dif;
    float   peak,peakerr,f,ferr;

    /* Get value and spectrum numbers */

    get_inspec(&inspec);
    if (DATA[inspec].NPNT < 3)
	{
	errtype("ERROR, too few datapoints in spectrum");
	clear_command();
	return;
	}

    /* Input background */

    sprintf(STRING,
	    "Background to be subtracted (< 0 gives default) [%10.3e]: ",
	    TH.PAR[0]);
    background = get_real(TH.PAR[0],STRING);
    if (background < 0) background = TH.PAR[0];
    sprintf(STRING,"Error in background (< 0 gives default) [%10.3e]: ",
	    TH.ERR[0]);
    backerr = get_real(TH.ERR[0],STRING);
    if (backerr < 0) backerr = TH.ERR[0];

    /* Input peak position */

    sprintf(STRING,
	    "Peak position (< 0 gives default) [%10.4e]: ",
	    TH.PAR[2]);
    position = get_real(TH.PAR[2],STRING);
    if (position < 0) position = TH.PAR[2];

    /* Find point on x axis that is closest to peak position */

    dif = fabs(DATA[inspec].X[0]-position);
    for (i = 1; i < DATA[inspec].NPNT; i++)
	{
	if (fabs(DATA[inspec].X[i]-position) < dif)
	    {
	    dif = fabs(DATA[inspec].X[i]-position);
	    ipos = i;
	    }
	}

    /* Calculate peak intensity and structure factor, plus errors */

    peak = DATA[inspec].Y[ipos]-background;
    peakerr = sqrt(sqr(DATA[inspec].E[ipos])+sqr(backerr));
    f = LMIL*sqrt(peak);
    ferr = LMIL*sqrt(peak+peakerr)-f;

    /* Output results */

    sprintf(STRING,"Peak intensity  : %10.4f +/- %10.4f\n",
	    peak,peakerr);
    type_line(STRING);
    sprintf(STRING,"Structure factor: %10.4f +/- %10.4f\n",
	    f,ferr);
    type_line(STRING);

    /* If requested, write results to output file */

    if (OUTINT)
	{
	fprintf(INTFILE,"%4d %8.3f %8.3f %8.3f %10.2f %10.2f\n",
		SCANNR,
		HMIL*T11+KMIL*T12+LMIL*T13,
		HMIL*T21+KMIL*T22+LMIL*T23,
		HMIL*T31+KMIL*T32+LMIL*T33,
		f,ferr);
	}

    type_header(0);

    }
#endif

/***************************************************************************/
void    ref_to_f(void)
/***************************************************************************/

    /*
    Convert reflectivity scan into structure factors. It is assumed that
    X contains the angle of incidence, while Y is the background-subtracted
    peak intensity.
    Output spectrum contains diffraction index l versus structure factor.
    */

    {

    int     inspec,outspec,i,istart;
    float   lconvert;
    float   f,ferr,correction,lorentz,area,polarization,transmission;

    /* Get spectrum numbers and l-conversion factor */

    get_inoutspec(&inspec,&outspec);
    if ((DATA[inspec].NPNT < 3) || (DATA[inspec].NPNT <= 2*NBACK))
	{
	errtype("ERROR, too few datapoints in spectrum");
	clear_command();
	return;
	}
    lconvert = get_real(6.,"Value of 2*K/b3 (to convert angles to l: ");

    /* Do the conversion */

    for (i = 0; i < DATA[inspec].NPNT; i++)
	{

	/* Calculate correction factors */

	comp_cor_ridge(DATA[inspec].X[i]/RAD, &lorentz, &polarization, &area,
		       &transmission);
	correction = lorentz*area*polarization*transmission;

	/* Calculate l and structure factor */

	LMIL = lconvert*sin(DATA[inspec].X[i]/RAD);
	f = sqrt(DATA[inspec].Y[i]*correction);
	ferr = sqrt((DATA[inspec].Y[i]+DATA[inspec].E[i])*correction)-f;

	/* Write data to output spectrum */

	DATA[outspec].X[i] = LMIL;
	DATA[outspec].Y[i] = f;
	DATA[outspec].E[i] = ferr;

	/* If requested, write results to output file */

	if (OUTINT)
	    {
	    if (!SCANSWRITTEN)
		{
		fprintf(INTFILE,"%4s %5s %8s %8s %9s %10s %13s %5s %10s %11s %9s",
				"run#","H","K","L","F","dF",
				"tot.corr.","area","polar","interc.","transm.");
		fprintf(INTFILE, "%9s\n","lorentz");
		SCANSWRITTEN = TRUE;
		}
	    fprintf(INTFILE,
		    "%4d %8.3f %8.3f %8.3f %10.2f %7.2f %9.4f %9.4f %9.4f %9.4f %9.4f",
		    SCANNR,
		    LMIL*T13,
		    LMIL*T23,
		    LMIL*T33,
		    f,ferr,correction,area,polarization,1.,transmission);
	    fprintf(INTFILE, "%9.4f\n", lorentz);
	    }
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d converted to f's",
	    inspec+1);
    DATA[outspec].NPNT = DATA[inspec].NPNT;
    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    for (i = 1; i < DATA[outspec].NPNT; i++)
	{
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}
    type_header(1);

    }

/***************************************************************************/
void    scale(void)
/***************************************************************************/

    /*
    Multiply the x-values of a spectrum by a constant.
    */

    {

    float   scale;
    int     inspec,outspec,i;

    /* Get scale factor and spectrum numbers */

    scale = get_real(1.,"Give scale factor [1.]: ");
    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Scale x-axis */

    for (i = 0; i < DATA[inspec].NPNT; i++)
	{
	DATA[outspec].X[i] = DATA[inspec].X[i]*scale;
	DATA[outspec].Y[i] = DATA[inspec].Y[i];
	DATA[outspec].E[i] = DATA[inspec].E[i];
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d x-scale * %10.3g",
	    inspec+1,scale);
    DATA[outspec].NPNT = DATA[inspec].NPNT;
    DATA[outspec].MINY = DATA[inspec].MINY;
    DATA[outspec].MAXY = DATA[inspec].MAXY;
    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[outspec].X[0],DATA[outspec].X[DATA[outspec].NPNT-1]);
    type_line(STRING);
    type_header(1);

    }

/***************************************************************************/
void    shift(void)
/***************************************************************************/

    /*
    Shift the x-values of a spectrum by a given offset.
    */

    {

    float   offset;
    int     inspec,outspec,i;

    /* Get offset and spectrum numbers */

    offset = get_real(0.,"Give shift value [0.]: ");
    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Shift spectrum */

    for (i = 0; i < DATA[inspec].NPNT; i++)
	{
	DATA[outspec].X[i] = DATA[inspec].X[i]+offset;
	DATA[outspec].Y[i] = DATA[inspec].Y[i];
	DATA[outspec].E[i] = DATA[inspec].E[i];
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d shifted by %10.3g",
	    inspec+1,offset);
    DATA[outspec].NPNT = DATA[inspec].NPNT;
    DATA[outspec].MINY = DATA[inspec].MINY;
    DATA[outspec].MAXY = DATA[inspec].MAXY;
    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[outspec].X[0],DATA[outspec].X[DATA[outspec].NPNT-1]);
    type_line(STRING);
    type_header(1);

    }

/***************************************************************************/
void	split(void)
/***************************************************************************/

    /*
    Split spectrum into peak and background parts
    */

    {

    int     in_spec, peak_spec, bg_spec;
    int     i, bg_points, pp, bb;

    bg_points = get_int(5, "Number of background points on each side (-1 = use fit) [5]: ");

    /* Get input and output spectrum numbers */

    get_in2outspec(&in_spec,&peak_spec,&bg_spec);

    if (DATA[in_spec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    if ( bg_points != -1 && (bg_points < 1 ||  2*bg_points > DATA[in_spec].NPNT))
	{
	errtype("ERROR, wrong number of background points");
	clear_command();
	return;
	}

    /* Do the splitting */

    pp = bb = 0;

    if (bg_points == -1)

	{

	for (i=0; i<DATA[in_spec].NPNT; i++)
	    {
	    if (fabs(DATA[in_spec].X[i]-TH.PAR[2]) < fabs(INT_RANGE*TH.PAR[4]))
		{
		DATA[peak_spec].X[pp] = DATA[in_spec].X[i];
		DATA[peak_spec].Y[pp] = DATA[in_spec].Y[i];
		DATA[peak_spec].E[pp] = DATA[in_spec].E[i];
		pp++;
		}
	    else
		{
		DATA[bg_spec].X[bb] = DATA[in_spec].X[i];
		DATA[bg_spec].Y[bb] = DATA[in_spec].Y[i];
		DATA[bg_spec].E[bb] = DATA[in_spec].E[i];
		bb++;
		}
	    }
	}
    else
	{

	for (i=0; i<bg_points; i++)
	    {
	    DATA[bg_spec].X[bb] = DATA[in_spec].X[i];
	    DATA[bg_spec].Y[bb] = DATA[in_spec].Y[i];
	    DATA[bg_spec].E[bb] = DATA[in_spec].E[i];
	    bb++;
	    }
	for (; i<DATA[in_spec].NPNT - bg_points; i++)
	    {
	    DATA[peak_spec].X[pp] = DATA[in_spec].X[i];
	    DATA[peak_spec].Y[pp] = DATA[in_spec].Y[i];
	    DATA[peak_spec].E[pp] = DATA[in_spec].E[i];
	    pp++;
	    }
	for (; i<DATA[in_spec].NPNT; i++)
	    {
	    DATA[bg_spec].X[bb] = DATA[in_spec].X[i];
	    DATA[bg_spec].Y[bb] = DATA[in_spec].Y[i];
	    DATA[bg_spec].E[bb] = DATA[in_spec].E[i];
	    bb++;
	    }
	}

    /* Generate spectrum information */

    sprintf(DATA[bg_spec].TITLE,"background of spectrum %1d ", in_spec+1);
    sprintf(DATA[peak_spec].TITLE,"peak of spectrum %1d ", in_spec+1);
    DATA[bg_spec].NPNT = bb;
    DATA[peak_spec].NPNT = pp;

    DATA[bg_spec].MINY = DATA[bg_spec].MAXY = DATA[bg_spec].Y[0];
    for (i = 1; i < DATA[bg_spec].NPNT; i++)
	{
	if (DATA[bg_spec].Y[i] < DATA[bg_spec].MINY)
	    DATA[bg_spec].MINY = DATA[bg_spec].Y[i];
	if (DATA[bg_spec].Y[i] > DATA[bg_spec].MAXY)
	    DATA[bg_spec].MAXY = DATA[bg_spec].Y[i];
	}
    DATA[peak_spec].MINY = DATA[peak_spec].MAXY = DATA[peak_spec].Y[0];
    for (i = 1; i < DATA[peak_spec].NPNT; i++)
	{
	if (DATA[peak_spec].Y[i] < DATA[peak_spec].MINY)
	    DATA[peak_spec].MINY = DATA[peak_spec].Y[i];
	if (DATA[peak_spec].Y[i] > DATA[peak_spec].MAXY)
	    DATA[peak_spec].MAXY = DATA[peak_spec].Y[i];
	}
    type_header(1);
    }

/***************************************************************************/
void    sum(void)
/***************************************************************************/

    /*
    Sum two spectra, assuming that they have the same x-values. Errors are
    set to unity.
    */

    {

    int     in_spec1,in_spec2,out_spec;
    int     i;

    /* Get input and output spectrum numbers */

    get_2inoutspec(&in_spec1,&in_spec2,&out_spec);

    if (DATA[in_spec1].NPNT != DATA[in_spec2].NPNT)
	{
	errtype("Spectra don't have same number of points");
	clear_command();
	return;
	}

    /* Sum the two spectra */

    for (i = 0; i < DATA[in_spec1].NPNT; i++)
	{
	DATA[out_spec].X[i] = DATA[in_spec1].X[i];
	DATA[out_spec].Y[i] = DATA[in_spec1].Y[i]+DATA[in_spec2].Y[i];
	DATA[out_spec].E[i] = sqrt(sqr(DATA[in_spec1].E[i])+sqr(DATA[in_spec2].E[i]));
	}

    /* Get spectrum information */

    DATA[out_spec].NPNT = DATA[in_spec1].NPNT;
    DATA[out_spec].MINY = DATA[out_spec].MAXY = DATA[out_spec].Y[0];
    for (i = 1; i < DATA[out_spec].NPNT; i++)
	{
	if (DATA[out_spec].Y[i] < DATA[out_spec].MINY)
	    DATA[out_spec].MINY = DATA[out_spec].Y[i];
	if (DATA[out_spec].Y[i] > DATA[out_spec].MAXY)
	    DATA[out_spec].MAXY = DATA[out_spec].Y[i];
	}
    sprintf(DATA[out_spec].TITLE,"Sum of spectra %1d and %1d",
	    in_spec1+1,in_spec2+1);

    }

/***************************************************************************/
void    wrap(void)
/***************************************************************************/

    /*
    Wrap the x-values of a spectrum such that no jumps of 360 degrees
    exist.
    */

    {

#define maxjump 10

    int     inspec,outspec,i,ijump;

    /* Get spectrum numbers */

    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 2)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Look for a jump in x-values */

    for (ijump = 1; ijump < DATA[inspec].NPNT; ijump++)
	{
	if (fabs(DATA[inspec].X[ijump]-DATA[inspec].X[ijump-1]) > maxjump)
	    break;
	}

    if (ijump != DATA[inspec].NPNT)
	{
	if (DATA[inspec].X[ijump-1] > 360-maxjump)
	    {
	    for (i = 0; i < ijump; i++)
		{
		DATA[outspec].X[i] = DATA[inspec].X[i]-360;
		DATA[outspec].Y[i] = DATA[inspec].Y[i];
		DATA[outspec].E[i] = DATA[inspec].E[i];
		}
	    for (i = ijump; i < DATA[inspec].NPNT; i++)
		{
		DATA[outspec].X[i] = DATA[inspec].X[i];
		DATA[outspec].Y[i] = DATA[inspec].Y[i];
		DATA[outspec].E[i] = DATA[inspec].E[i];
		}
	    }
	else if (DATA[inspec].X[ijump] > 360-maxjump)
	    {
	    for (i = 0; i < ijump; i++)
		{
		DATA[outspec].X[i] = DATA[inspec].X[i];
		DATA[outspec].Y[i] = DATA[inspec].Y[i];
		DATA[outspec].E[i] = DATA[inspec].E[i];
		}
	    for (i = ijump; i < DATA[inspec].NPNT; i++)
		{
		DATA[outspec].X[i] = DATA[inspec].X[i]-360;
		DATA[outspec].Y[i] = DATA[inspec].Y[i];
		DATA[outspec].E[i] = DATA[inspec].E[i];
		}
	    }
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"spectrum %1d wrapped",inspec+1);
    DATA[outspec].NPNT = DATA[inspec].NPNT;
    DATA[outspec].MINY = DATA[inspec].MINY;
    DATA[outspec].MAXY = DATA[inspec].MAXY;
    sprintf(STRING,"Range of x-axis: %10.3g to %10.3g\n",
	    DATA[outspec].X[0],DATA[outspec].X[DATA[outspec].NPNT-1]);
    type_line(STRING);
    type_header(1);

    }
/***************************************************************************/
void    ylog10(void)
/***************************************************************************/

    /*
    Replace y-values by log10(y).
    */

    {

    int     inspec,outspec,i,j;

    /* Get spectrum numbers */

    get_inoutspec(&inspec,&outspec);
    if (DATA[inspec].NPNT < 1)
	{
	errtype("ERROR, no data in spectrum");
	clear_command();
	return;
	}

    /* Take log of y-values */

    for (i = 0; i < DATA[inspec].NPNT; i++)
	{
	DATA[outspec].X[i] = DATA[inspec].X[i];
	if (DATA[inspec].Y[i] < 1e-40)
	    DATA[outspec].Y[i] = -40;
	else
	    DATA[outspec].Y[i] = log10(DATA[inspec].Y[i]);
	if (DATA[inspec].E[i] < 1e-40)
	    DATA[outspec].E[i] = -40;
	else
	    DATA[outspec].E[i] = log10(DATA[inspec].E[i]);
	}

    /* Generate spectrum information */

    sprintf(DATA[outspec].TITLE,"log10 of spectrum %1d",
	    inspec+1);
    DATA[outspec].NPNT = DATA[inspec].NPNT;
    DATA[outspec].MINY = DATA[outspec].MAXY = DATA[outspec].Y[0];
    for (i = 1; i < DATA[outspec].NPNT; i++)
	{
	if (DATA[outspec].Y[i] < DATA[outspec].MINY)
	    DATA[outspec].MINY = DATA[outspec].Y[i];
	if (DATA[outspec].Y[i] > DATA[outspec].MAXY)
	    DATA[outspec].MAXY = DATA[outspec].Y[i];
	}
    type_header(1);

    }



