
C
C     PROGRAM  crystal_tt    
C
C     Calculates perfect crystal diffraction profiles solving
C     numerically the Takagi-Taupin equations.
C
C     It uses DABAX library to get structure factors.
C     Takagi-Taupin subroutines [tt_bragg and tt_laue] are adapted
C     from the originals of Dr. Jerzy Gronkowski.
C
C     Bragg [reflection] and Laue [transmission] cases
C
C     Authors: M. Sanchez del Rio, V. Mocella and C. Ferrero
C       --------------------
C
C	1997-5-15 First version: only flat crystals  
C	1997-9-30 Working version: Laue bent seems OK, Bragg bent
C		seems to do not deliver correct results.
C	1997-10-10 Version for XOP 1.9
C
C	(c) 1997 European Synchrotron Radiation Facility
C	    Programming Group
C	    Contact: M. Sanchez del Rio, srio@esrf.fr
C
	program cryst_tt
	implicit none

	integer		nsteps,j,jold
	real*8		tmp,cgross,d_ext,l_ext,l_pen,l_abs,d_abs
        	complex*16	chibchih
c input
        character*80    file_refl,tmps
        character*160    top_title
c                                                   
	real*8		phot,y0,ymax,dy0,alpha,t,kstar
	integer*4	i_geometry,irun,ipol

	real*8		todeg,twopi,r0,torad
	integer*4	idebugger
	integer*4	i,iunits(3),iunit
	real*8		thetab,rn,d_spacing,lambda
        complex*16      f_0,f_h,f_hbar
        complex*16      psi_0,psi_h,psi_hbar
        complex*16      refrac
        complex*16      struct
	real*8		d,v,mu,pi


	real*8 chio(2),chih(2),chib(2)
	real*8 teta,fi,window,elem,window_file,elem_file
	integer*4 net_file,all_file
	real*8 thick
	real*8 eta,wavel,ref
	real*8 yzac,yzac1,yzac0
	real*8 radius
        real*8 fact1,fact2
	integer*4	iblank2

	common /inp_cry/teta,fi,chio,chih,chib,thick
	common /inp_net/window,elem
	common /uu2/wavel,radius,kstar,i_geometry

      DATA    TODEG   / 57.2957 79513 08232 08767 98155 D0 /
      DATA    TORAD   /  0.0174 53292 51994 32957 69237 D0 /
      DATA    TWOPI   /  6.2831 85307 17958 64679 25287 D0 /
      DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
C      DATA    R0      /  2.818 D-15/
      DATA    R0      /  2.817940289458 D-15/

	

*case 1: Laue Si bent crystal
	file_refl = 'xcrystal.bra'		!crystal input file
	PHOT = 8000.0		!photon energy [eV]
	I_GEOMETRY = 2		! 1: Bragg, 2: Laue
	Y0 = -25.	! Ymin
	YMAX = 25.		! Ymax
	DY0 = 0.2		! Ystep
	IPOL = 1			! polarization [1=s, 2=p, 3=unpol]
	ALPHA = 90.00		! asymmetry angle
	radius = 12.98		! R [m]
	T = 0.050		! crystal thickness [mm]
	kstar = 0.224D0      ! Poisson ratio                              


	idebugger = 0
C
C open output files
C
	open(10,file='cryst_tt.dat',status='unknown')
	open (11,file='cryst_tt.inf',status='unknown')
	iunits(1) = 6
	iunits(2) = 11
	iunits(3) = 10
C
C---------------------- starts input section ---------------------
C
      DO I=1,2
      IUNIT = IUNITS(I)
      WRITE (iunit,*) '  '
      WRITE (iunit,*) '***********************************************'
      WRITE (iunit,*) '************** crystal_tt 1.0 *****************'
      WRITE (iunit,*) '***********************************************'
      write (iunit,*)  ' '
      WRITE (iunit,*) 'THIS PROGRAM CALCULATES REFLECTIVITY CURVES FOR'
      WRITE (iunit,*) '  BENT CRYSTALS BY SOLVING THE TAKAGI-TAUPIN   '
      WRITE (iunit,*) '  EQUATION WITH THE FINITE-DIFFERENCE-METHOD.  '
      WRITE (iunit,*) '  '
      ENDDO

      write(*,*)  ' '
C      write(*,*)  ' Run crystal_ml with the input parameters from: '
C      write(*,*)  '     [0] default values'
C      write(*,*)  '     [1] input from keyboard'
C      write(*,*)  '     [2] input from file cryst_ml.inp'
C	read(*,*) IRUN
	IRUN = 2

C	if (irun.eq.1) call crystal_inp(file_refl,PHOT,
C     $I_GEOMETRY,Y0,YMAX,DY0,IPOL,ALPHA,radius,T,KSTAR)

	if (irun.eq.2) then
	  write(*,*) 'Reading inputs from file cryst_tt.inp'
	  iunit = 25
	  open (iunit,file='cryst_tt.inp',status='old')
	  read(iunit,'(a)') file_refl
	  read(iunit,'(a)') top_title
	  read(iunit,*) PHOT
	  read(iunit,*) I_GEOMETRY
	  read(iunit,*) Y0
	  read(iunit,*) YMAX
	  read(iunit,*) DY0
	  read(iunit,*) IPOL
	  read(iunit,*) radius
	  read(iunit,*) ALPHA
	  read(iunit,*) T
	  read(iunit,*) KSTAR
	  close(iunit)
	  write(*,*) 'Done.'
	endif
C----------------------- end of input section -------------------------


	write(10,'(a)') '#F crystal_tt.dat'
	write(10,'(a)') '#S 1 crystal_tt run'
	write(10,'(a)') '#C results of crystal_tt [Takagi-Taupin]  run'
	write(10,'(a)') '#UT0 '//top_title(1:iblank2(top_title))
	write(10,'(a)') '#N 4'
	write(10,'(a)') '#L theta-inc[mrad]  theta-inc[deg]  y  
     1Reflectivity'

c
c write inputs in the info file
c
      WRITE (11,*) '**Input section:'
      write(11,*) 'crystal input file: ',file_refl
      write(11,*) 'photon energy [eV]: ',PHOT
      write(11,*) 'geometry [1: Bragg, 2: Laue]',I_GEOMETRY
      write(11,*) 'Y min: ',Y0
      write(11,*) 'Y max: ',YMAX
      write(11,*) 'Y step: ',DY0
      write(11,*) 'Polarization [1=s,2=p,3=unpolarized]: ',IPOL
      if (radius.eq.0) then
	write(11,*) 'crystal curvature radius [m]: INFINITY'
      else
	write(11,*) 'crystal curvature radius [m]: ',radius
      endif
      write(11,*) 'asymmetry angle: ',ALPHA
      if (t.ne.0) then 
	write(11,*) 'crystal thickness [mm]: ',T
      else
	write(11,*) 'crystal thickness [mm]: THICK'
      endif
      write(11,*) 'Poisson ratio   : ',kstar


	if (idebugger.EQ.1) then 
          write(*,*) 'End of Input Section, Parameters are: '
	  write(*,*) '>> Selected inp file : ',file_refl
	  write (*,*) '>> energy [eV] = ',phot
	  write (*,*) '>> i_geometry = ',i_geometry
          write (*,*) '>> Y0 = ',Y0
          write (*,*) '>> YMAX = ',YMAX
	  write (*,*) '>> DY0 = ',DY0
          write (*,*) '>> IPOL [polarization]  = ',IPOL
	  write(*,*) '>> radius : ',radius
	  write(*,*) '>> ALPHA [asymmetry angle]: ',ALPHA
          write(*,*) '>> End of Input Section Parameters. '
          write(*,*) ' '
	endif
C---------------------- start calculations ----------------------------


C--------------------- starts struct fact  calculations ---------------------

C Call crystal_fh to read the BRAGG file
	    call crystal_fh(-1,file_refl,phot,thetab,
     $f_h,f_hbar,f_0,psi_h,psi_hbar,psi_0,refrac,
     $rn,d_spacing)


C calculates Bragg angle
	lambda = 12398.54/phot
	thetab = dasin(lambda*1.0E-8/2.0D0/d_spacing)

           if (ipol.eq.1) then                                             
             cgross = 1.00D0                                             
           else if (ipol.eq.2) then                                      
             cgross = dabs(dcos(2.00d0*thetab))                         
           else if (ipol.eq.3) then                                    
             cgross = (1.00d0+dabs(dcos(2.00d0*thetab)))/2.00D0       
           else                                                     
		do i=6,11,5
		write (i,*) 'cryst_tt aborted: Wrong polarization'
     $               //' number.'
		enddo
		stop
           end if

C
C interlattice distances and cell volume
C
	d = d_spacing*1d-2     ! d in meters
	v = 2.817939 D-13/rn   ! V in cm^3
	v = v*1d-6             ! V in m^3

c	     if (idebugger.eq.1) then 
	     write(11,*)' '
	     write(11,*)'Lambda [A]: ',Lambda
	     write(11,*)'Phot [eV]: ',Phot
	     write(11,*)'Theta Bragg [deg]: ',thetab*todeg
	     write(11,*)'polarization (1=s, 2=p, 3=unpol): ',IPOL
	     write(11,*) 'Volume of the crystal unit cell [m^3]= ',v
	     write(11,*)' '
c	     endif

C Structure factor

	     write(11,*)' '
	     write(11,*)' '
	     write(11,*)'##########################################'
	     write(11,*)'Starts data from Struct Fact calculation.'
	     call crystal_fh(11,file_refl,phot,thetab,
     $f_h,f_hbar,f_0,psi_h,psi_hbar,psi_0,refrac,
     $rn,d_spacing)
	     write(11,*)'Ends data from Structure Factor calculation.'
	     write(11,*)'##########################################'

#if lahey
	     mu =  2.0D0 * twopi *(-aimag(refrac)) / (lambda*1.0d-8)
#else
	     mu =  2.0D0 * twopi *(-Dimag(refrac)) / (lambda*1.0d-8)
#endif
	     mu = mu * 1.0d2              !m^-1
	     struct = sqrt(f_h * f_hbar)
c     lambda = lambda * 1.0d-10    !m

	     if (idebugger.EQ.1) then
	       write(*,*)'-------- str factors : -------'
	       write (*,*) '** RN         : ',RN
	       write (*,*) '** D_SPACING  : ',D_SPACING
	       write (*,*) '** F_H     : ',F_H
	       write (*,*) '** F_HBAR  : ',F_HBAR
	       write (*,*) '** F_0     : ',F_0
	       write (*,*) ' '
	       write (*,*) '** PSI_H   : ',PSI_H
	       write (*,*) '** PSI_HBAR: ',PSI_HBAR
	       write (*,*) '** PSI_0   : ',PSI_0
	       write (*,*) ' '
	       write (*,*) '** REFRAC  : ',REFRAC
	       write(*,*)'-------- derived from str factors : -------'
	       write (*,*) '** STRUCT : ',STRUCT
	       write (*,*) '** MU [m^-1]  : ',MU
	       write(*,*)'-------- end from str factors : -------'
	     endif

C--------------------- end struct fact  calculations ---------------------

c
c input parameters
c
	TETA  = THETAB*180.0/PI
	if (i_geometry.eq.2) then
	  fi    = ALPHA - 90.0
	else
c changed by srio 97/10/10 to be consistent with the other programs
c	  fi    = ALPHA
	  fi    = -1.0D0*ALPHA
	endif
	chio(1) = - (psi_0)
	chih(1) = - Dreal(psi_h)
	chib(1) = - Dreal(psi_hbar)

#if lahey
	chio(2) = - aimag(psi_0)
	chih(2) = - aimag(psi_h)
	chib(2) = - aimag(psi_hbar)
#else
	chio(2) = - Dimag(psi_0)
	chih(2) = - Dimag(psi_h)
	chib(2) = - Dimag(psi_hbar)
#endif

	thick = t*1000.0     ! thickness in micron

	wavel = lambda*1.0d-4    ! wavelength in micron
c
c calculate Darwin width
c
	yzac0 = yzac(i_geometry,0,cgross)
	yzac1 = yzac(i_geometry,1,cgross)
	  write(11,*) ' '
	  write(11,*) '> 1*y [rad] : ',yzac0
	  write(11,*) '> dy shift [rad] : ',yzac1
	  write(11,*) ' '

c pendelloesung(for transmission) and extinction(for reflection) 
c distance (A. Authier, Tables of X-ray cryst., $ 5.1): 2 names
c for the same thing
	                  
#if lahey
      chibchih=cmplx( chih(1)*chib(1)-chih(2)*chib(2) ,
     $chih(1)*chib(2)+chib(1)*chih(2), 8 )
#else
csrio      chibchih=cmplx( chih(1)*chib(1)-chih(2)*chib(2) ,
      chibchih=dcmplx( chih(1)*chib(1)-chih(2)*chib(2) ,
     $chih(1)*chib(2)+chib(1)*chih(2) )
#endif
      
      if (i_geometry.eq.2) then
#if lahey
      l_pen=wavel*sqrt(dcos((teta+fi)*torad)*dcos((teta-fi)*torad))
     &/(cgross*real(sqrt(chibchih)))
#else
      l_pen=wavel*dsqrt(dcos((teta+fi)*torad)*dcos((teta-fi)*torad))
     &/(cgross*dreal(cdsqrt(chibchih)))
#endif
      else
#if lahey
      l_pen=wavel*sqrt(dsin((teta+fi)*torad)*dsin((teta-fi)*torad))
     &/(cgross*real(sqrt(chibchih)))
#else
      l_pen=wavel*dsqrt(dsin((teta+fi)*torad)*dsin((teta-fi)*torad))
     &/(cgross*dreal(cdsqrt(chibchih)))
#endif
      endif
      
c       l_pen = real(wavel*cos(teta*torad)/cgross/csqrt(chibchih) )
cd_ext = real(wavel*cos(teta*torad)/cgross/csqrt(chibchih)/2.0/pi)
cl_ext = real(wavel*cos(teta*torad)/cgross/csqrt(chibchih)/2.0/pi)/
c    $cos( torad*(teta+fi) )
      d_abs = 1.0/(mu/1e6)/(1.d0/cos(torad*(teta+fi)) +
     $1.d0/abs(cos(torad*(teta+fi))))
      l_abs = 1.0/(mu/1e6)

cwrite(*,*) 'EXTINCTION DEPTH: ',d_ext
c	write(*,*) 'ABSORPTION COEFF [m^-1]: ',mu
c	    write(*,*) 'ABSORPTION COEFF [microns^-1]: ',mu/1e6

	write(11,*) ' '
	write(11,*) 
     $'PENDELLOSUNG PERIOD or EXTINCTION LENGTH [microns] ',l_pen
	write(11,*) 'ABSORPTION LENGTH [microns]: ',l_abs
	write(11,*) 'ABSORPTION DEPTH [microns]: ',d_abs
	write(11,*) ' '


        IF (I_GEOMETRY.EQ.1) THEN      ! BRAGG
          if (radius.eq.0.0) then
c           -- good for flat crystals
            window = l_pen*400.0
            elem = window/100.0
          else
            fact1 =120.0
            fact2 = 50.0
            window = l_pen*fact1
            elem = window/fact2
          endif
        ELSE
	  window = 3.0*(thick+tan(thetab+fi*pi/180))
	  elem=window/300.0
        ENDIF
	 


	DO I=1,2
	write(iunits(i),*) 'theta Bragg [deg]: ',teta
	write(iunits(i),*) ' '
	write(iunits(i),*) '>> INTEGRATION NETWORK: '
	write(iunits(i),*) '>> window [microns]: ',window
	write(iunits(i),*) '>> element [microns]: ',elem
	write(iunits(i),*) ' '
	ENDDO
c by default do not write network and all_profiles files.
	net_file = 0
	all_file = 0
	

C
C if file cryst_tt.win exists, then take the window values from there
C
	open (28,file='cryst_tt.win',status='old',ERR=1150,
     $ FORM='FORMATTED')
	read(28,'(a)') tmps
C	write(*,*) '<><> tmps: ',tmps
	read(28,*) window_file
C	write(*,*) '<><> window_file: ',window_file
	read(28,'(a)') tmps
C	write(*,*) '<><> tmps: ',tmps
	read(28,*) elem_file
C	write(*,*) '<><> elem_file: ',elem_file
	read(28,'(a)') tmps
C	write(*,*) '<><> tmps: ',tmps
	read(28,*) net_file
C	write(*,*) '<><> net_file: ',net_file
	read(28,'(a)') tmps
C	write(*,*) '<><> tmps: ',tmps
	read(28,*) all_file
C	write(*,*) '<><> all_file: ',all_file
	close(28)
	if (window_file.lt.-1.0D-8) then
	  window = window*(-1.0D0*window_file)
	else if (window_file.gt.1.0D-8) then
	  window = window_file 
	else
	  window = window
	endif
	if (elem_file.lt.-1.0D-8) then
	  elem = elem*(-1.0D0*elem_file )
	else if (elem_file.gt.1.0D-8) then
	  elem = elem_file
	else
	  elem = elem
	endif
	DO I=1,2
	write(iunits(i),*) 
     $    '>> USING NEW INTEGRATION NETWORK READ FROM cryst_tt.win: '
	write(iunits(i),*) '>> window [microns]: ',window
	write(iunits(i),*) '>> element [microns]: ',elem
	write(iunits(i),*) 
     $'>> write NETWORK file (0-No, 1=Yes): ',net_file
	write(iunits(i),*) 
     $'>> write ALL PROFILES file (0-No, 1=Yes): ',all_file
	write(iunits(i),*) ' '
	ENDDO
1150	continue

C
C writes some outputs
C
        if (i_geometry.eq.2) then
          call tt_laue(1,wavel,0.0D0,ref)
        else
          call tt_bragg(1,wavel,0.0D0,ref,i_geometry,cgross)
        endif

	write(11,*) ' <<>> teta  =  ',teta
	write(11,*) ' <<>> fi    =  ',fi
	write(11,*) ' chio(1) = ',chio(1)
	write(11,*) ' chio(2) = ',chio(2)
	write(11,*) ' chih(1) = ',chih(1)
	write(11,*) ' chih(2) = ',chih(2)
c	write(11,*) ' thick   = ',thick
c	write(11,*) ' '
c	write(11,*) ' elem    = ',elem
c	write(11,*) ' window  = ',window
c	write(11,*) ' 1*y [rad]  = ',yzac0
c	write(11,*) ' y0 shift [rad]  = ',yzac1
c	write(11,*) ' extinction length [microns]  = ',l_ext
c	write(11,*) ' extinction depth [microns]  = ',d_ext
c	write(11,*) ' pendellosung period [microns]  = ',l_pen
c	write(11,*) ' absorption length [microns]  = ',l_abs
c	write(11,*) ' absorption depth [microns]  = ',d_abs
c	write(11,*) ' '
	write(11,*) ' '

	nsteps = ((ymax-y0)/dy0)+1
	i=0
	jold = 0
	j = 0

c
c write the file with network (cryst_tt.net) , if selected
c
	if (net_file.eq.1) then
	  eta   = 0.0
	  write(*,*) 'Writing integration net in file cryst.net...'
	  if (i_geometry.eq.2) then
	    call tt_laue(2,wavel,eta,ref)
	  else
	    call tt_bragg(2,wavel,eta,ref,i_geometry,cgross)
	  endif
	  write(*,*) 'Done.'
	endif

c all_file = 0 means do not print, all_file = 3 means print all profiles.
	if (all_file.eq.1) all_file=3

c
c mainloop
c

	write(*,*) 'done (%) :   0'
	do eta=Y0,YMAX,DY0
	  i=i+1
	  if (i_geometry.eq.2) then
	    call tt_laue(all_file,wavel,eta,ref)
	  else
	    call tt_bragg(all_file,wavel,eta,ref,i_geometry,cgross)
	  endif
#if lahey
	  tmp=real(i,8)/real(nsteps,8)*100.0
#else
	  tmp=Dfloat(i)/Dfloat(nsteps)*100.0
#endif
	  j = int(tmp/10.0)
	  if (j.ne.jold) then
	    write(*,*) 'done (%) : ',int(tmp)
	    jold = j
	  endif
c	  write(10,*) eta*yzac0+yzac1,eta,ref
c in mrads
	  write(10,*) 1.0D3*(eta*yzac0+yzac1),
     1  180.0/PI*(eta*yzac0+yzac1),eta,ref
c  write(*,*) eta*yzac0+yzac1,eta,ref
	enddo
	close(11)
	close(10)
	end


c===================================================================
	real*8 function yzac(i_geometry,iflag,cgross)
c
c	iflag = 0 : returns 1*y width (rads)
c	iflag = 1 : returns shift theta_b - theta_b_corrected
c
        implicit real*8 (a-h,o-z)
c
        complex*16 chibchih  
        real*8  first,second           
	dimension chio(2),chih(2),chib(2)
	common /inp_cry/teta,fi,chio,chih,chib,thick
      DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /


	tet1 = teta*pi/180.0
	tpf = (teta + fi)*pi/180.0
	tmf = (teta - fi)*pi/180.0
      if (i_geometry.eq.1) then ! bragg
     	  go = sin(tmf)
     	  gh = -sin(tpf)
     	else
	  go = cos(tpf)
	  gh = cos(tmf)
     	endif


csrio-----------------
	chibchih1=chih(1)*chib(1)-chih(2)*chib(2)
	chibchih2=chih(1)*chib(2)+chib(1)*chih(2)
#if lahey
	chibchih=cmplx(chibchih1,chibchih2,8)
           first=sqrt(go/abs(gh))/(cgross*sqrt(abs(chibchih)))
#else
csrio	chibchih=cmplx(chibchih1,chibchih2)
	chibchih=dcmplx(chibchih1,chibchih2)
           first=sqrt(go/abs(gh))/(cgross*dsqrt(cdabs(chibchih)))
#endif

	if (iflag.eq.0) then 
	second1=sin(2.0*tet1)
	second2=0.0
#if lahey
	second=cmplx(second1,second2,8)
#else
csrio	second=cmplx(second1,second2)
	second=dcmplx(second1,second2)
#endif
	yzac = 1.0/Dreal(first*second)
	endif

      if (iflag.eq.1) then 
          if (i_geometry.eq.1) then
	        yzac=(1.0/sin(2.0*tet1))*(chio(1)/2.0*(gh/go-1.0))
	  else
	        yzac=(1.0/sin(2.0*tet1))*(chio(1)/2.0*(gh/go-1.0))
	  endif
      endif
	return
	end

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C                                                                          C
C  Program BRAGG.F to solve the Takagi-Taupin equations in the Bragg case  C
C  for an arbitrary incident wave falling on a deformed single crystal.    C
C  More detailed information can be found in a provisional manual and in   C
C  the list of the program. For further explanations contact:              C
C                                                                          C
C  Dr. Jerzy Gronkowski, Institute of Experimental Physics, University of  C
C                        Warsaw, Hoza 69, 00-681 Warsaw, Poland            C
C                        tel.(48) (2) 628 30 31, fax: (48) (2) 622 61 54   C
C                        email: gronko@fuw.edu.pl                          C
C                                                                          C
C Modified to be called as subroutine.                                     C
C Changed input /output . May 14 1997                                      C
C Renamed step->elem to be consistent with tt_laue.                        C
C                                                                          C
C                                                                          C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC



	subroutine tt_bragg(iflag,wavel,eta,ref,i_geometry,cgross)

c iflag = 0 : default (=makes calculation)
c iflag = 1 : prints info comment
c iflag = 2 : writes integration network: file cryst_tt.net
c iflag = 3 : default (=makes calculation) + 
c		writes all reflectivity curves: file cryst_tt.all

       implicit real*8 (a-h,o-z)
	
	dimension chio(2),chih(2),chib(2),reflex(10000)
c added by srio
	dimension do(10000,2),dh(10000,2)
	common /inp_cry/teta,fi,chio,chih,chib,thick
	common /inp_net/window,elem

	common /uu/ao,go,ah,gh,sf,cf,st

      DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /


c        if (iflag.eq.2) open (55,file='cryst_tt.net',status='unknown')
#if lahey
c formatter (ascii) output for PC (by now).
        if (iflag.eq.2) open (56,file='cryst_tt.net',status='unknown')
#else
        if (iflag.eq.2) open (56,file='cryst_tt.net',
     $FORM='UNFORMATTED',status='unknown')
#endif
        if (iflag.eq.3) open (66,file='cryst_tt.all',status='unknown')
	
C#########################################################################
C
C  Read  data from the data file
C
C#########################################################################
C
C  Line # 1
C
C  wavel: the wavelength of the incident radiation [micrometers]
C  teta:  the Bragg angle [degrees]
C  fi:    the asymmetry angles [degrees]                                  
C
c	read (3,*)wavel,teta,fi
c       if (iflag.eq.1) then
c         write(*,*) '>>>>>>>>>>>>>>>>'
c         write(*,*) 'wavel,eta: ',wavel,eta
c         write(*,*) 'teta,fi: ',wavel,teta,fi
c       endif

C
C  Line # 2
C
C  chio:  the zeroth Fourier component of the electric susceptibility
C         ( two numbers: the real and imaginary part )
C  chih:  the h_th Fourier component of the electric susceptibility
C         ( two numbers: the real and imaginary part )
C  chib:  the -h_th Fourier component of the electric susceptibility
C         ( two numbers: the real and imaginary part )
C
c	read (3,*)chio,chih,chib
c       if (iflag.eq.1) then
c         write(*,*) 'chio: ',chio
c         write(*,*) 'chih: ',chih
c         write(*,*) 'chib: ',chib
c       endif

C
C  Line # 3
C
C  thick: the thickness of the crystal [micrometers]
C         ( thick = 0 is equivalent to assuming that the crystal is
C           infinitely thick; thick > 0 - the calculations are made
C           for the thin crystal model )
C  elem:  the elementary step of integration along the entrance surface
C         [micrometers]
C  window:the width of the illuminated window on the entrance surface
C         [micrometers] 
C  eta:   the value of the real part of the incidence parameter
C         [dimensionless]
C
c	read (3,*)thick,step,window,eta
c        if (iflag.eq.1) write(*,*) 'thick: ',thick
c        if (iflag.eq.1) write(*,*) 'elem,window: ',elem,window

C
C  Line # 4
C
C  ys:    the value of the y coordinate for the first plane of diffraction
C         [micrometers]
C  ytot:  the total height of the calculated map [micrometers]
C         ( ytot = 0 if only one line is to be calculated )
C
c	read (3,*)ys,ytot
	ys = 0.0
        ytot = 0.0
C
C  End of data reading
C
C#########################################################################

C#########################################################################
C
C  Calculations of general parameters
C
c	write (*,*) '0*** cgross = ',cgross
	vect = 2.0*pi/wavel
	tet1 = teta*pi/180.0
	tpf = (teta + fi)*pi/180.0
	tmf = (teta - fi)*pi/180.0
C
C  go, gh: direction cosines
C
	go = sin(tmf)
	gh = -sin(tpf)
        st = sin(tet1)
	ct = cos(tet1)
	ao = cos(tmf)
	ah = cos(tpf)
	to = ao/go
	th = ah/gh
	sf = sin(fi*pi/180.0)
	cf = cos(fi*pi/180.0)
C
C prints parameters
C
	if (iflag.eq.1) then 
	  write(11,*) ' <<>>'
	  write(11,*) '     Incident angle [deg]: ',tmf*180.0/PI
	  write(11,*) '     Reflected angle [deg]: ',tpf*180.0/PI
	  write(11,*) '     Gamma_0: ',go
	  write(11,*) '     Gamma_H: ',gh
	  write(11,*) '     Asymmetry factor b: ',go/gh
	  write(11,*) ' <<>>'
	  return
	endif
C	
C  deltat: Delta(theta) - angular deviation of the incidence angle from
C                         the exact Bragg angle [radians]
C


      yzac0 = yzac(i_geometry,0,cgross)
      yzac1 = yzac(i_geometry,1,cgross)
 	    deltat = eta*yzac0+yzac1

C
C  b,a: elementary integration steps along the s_o and s_h directions
C
cvito	b = go*elem/sin(2.0*tet1)
cvito	a = -gh*elem/sin(2.0*tet1)
	b = go*elem
	a = -gh*elem

	w = a*go

C
C  l:  the number of integration steps along the entrance surface 
C      ( l should be smaller than 9,999 unless the dimension of the arrays 
C        is changed in the declarations )
C
	l = int(0.5 + window/elem)
	if (thick .gt. 0.0)then
C
C  lt: the number of integration steps along the s_o direction 
C      ( l should be smaller than 9,999 unless the dimension of the arrays 
C        is changed in the declarations )
C
	lt = int(thick/w + 1.5)

	if (lt .gt. 9999)then
	write(*,*)'crystal too thick'
	stop
	endif
	endif	
	if(l .gt. 9999)then
	write(*,*)'too many points on the surface'
	stop
	endif

	tmptmp = 0
	do i=1,l
	  tmptmp = tmptmp + l - i + 1
	enddo
C
C  Calculations of constant coefficients in the half-step derivative
C  algorithm to solve the T-T equations
C  ( "r" generally means: "real part of", "i" - "imaginary part of" )
C
	c1r = 1.0 - a*vect*chio(2)/4.0
	c1i = a*vect*chio(1)/4.0
	c2r = vect*chih(2)*b/4.0
	c2i = -vect*chih(1)*b/4.0
	c3r = vect*chib(2)*a/4.0
	c3i = -vect*chib(1)*a/4.0
	c1m = c1r*c1r + c1i*c1i
	c4r = (c3r*c1r + c3i*c1i)/c1m
	c4i = (c3i*c1r - c3r*c1i)/c1m
	c5r = (c2r*c1r + c2i*c1i)/c1m
	c5i = (c2i*c1r - c2r*c1i)/c1m
	c6r = 1.0 - c4r*c2r + c4i*c2i
	c6i = -c4r*c2i - c4i*c2r
c
cvito	a3 = -b*vect*(sin(2.0*tet1)*deltat + chio(1)/2.0)*0.5
cvito2   a3 = -b*vect*(sin(2.0*tet1)*deltat - chio(1)/2.0)*0.5
ca3=-b*vect*(sin(2.0*tet1)*deltat - chio(1)/2.0*(-gh/go-1.0))*0.5
c
	a3=-b*vect*(sin(2.0*tet1)*deltat + chio(1)/2.0)*0.5
c
	pp = -vect*b*0.5
	b1r = 1.0 + b*vect*chio(2)/4.0
	b2r = c6r - b1r + 1.0
	c7r = -c1r + 2.0
	c7i = -c1i
	cc = 0.5*b*sqrt(1.0 - gh*gh)
	c1 = a*sqrt(1.0 - go*go)

cvito asymmetry factor^-1 for rocking curve
	bb=abs(gh)/go
c     write (*,*) '0*** asymmetry factor = ',1.0/bb
	
	k = l + 1
C
C  dy:    the distance between two successive planes of incidence
C         lines
C  lines: the number of planes of incidence calculated
C
	dy = elem*to
	lines = int(ytot/dy + 1.5)
C<<>>
C
C  600: loop over the planes of incidence
C	
	do 600 nplan = 1,lines
	y1 = ys + (nplan - 1)*dy
C
C  500: loop to set the initial conditions along the s_o characteristics
C
	do 500 j = 1,k  	
	do(j,1) = 0.0
	do(j,2) = 0.0
	dh(j,1) = 0.0
	dh(j,2) = 0.0
  500   continue

	inet = 0
C
C calculate the number of knots
C
	if (iflag.eq.2) then 
	  tmp2=0
  	  DO m = 1,l
            k1 = l - m + 1
            if (thick .gt. 0.0) k1 = lt
            tmp2=tmp2+k1-1
	  ENDDO
	write(*,*) 
     $'>> Number of knots in the integration network: ',tmp2
#if lahey
	if (iflag.eq.2) write(56,*) real(tmp2,4)
#else
	if (iflag.eq.2) write(56) SNGL(tmp2)
#endif
	endif
	
	
C
C  400: the main loop over the points of the integration network
C       along the entrance surface
C	
  	do 400 m = 1,l
	x = elem*m - cc
	z = 0.5*w
C
C  Initial condition for D_o in one point on the entrance surface
C
	do(1,1) = d(x)
	do(1,2) = 0.0
	
c	b1i = a3 + pp*u(x,y,z)
	b1i = a3 + pi*b*u(x,y,z)
   	p = b1r*dh(2,1) - b1i*dh(2,2) + c2r*(do(2,1) + do(1,1)) - c2i
     *    *(do(2,2) + do(1,2))
     	q = b1r*dh(2,2) + b1i*dh(2,1) + c2r*(do(2,2) + do(1,2)) + c2i
     *    *(do(2,1) + do(1,1))
     	re = (2.0 - b1r)**2 + b1i*b1i
	dh(1,1) = (p*(2.0 - b1r) - q*b1i)/re
	dh(1,2) = (q*(2.0 - b1r) + p*b1i)/re
C	
C  reflex(m): the intensity of the diffracted wave at the entrance
C             surface	
C	
  	reflex(m) = dh(1,1)**2 + dh(1,2)**2
C	
C  k1:  the number of points along the s_o characteristics ( diminishing
C       with m )
C	
    	k1 = l - m + 1
C	
C  Switch to the thick crystal model	
C	
	if (thick .gt. 0.0) k1 = lt

C
C  The inner loop over the points along the s_o characteristics
C
	do 444 j = 2,k1
	z = (j - 1)*w + 0.5*w
	x = elem*m + (j - 1)*c1 - cc
cb1i = a3 + pp*u(x,y,z)
	b1i = a3 + pi*b*u(x,y,z)
   	b2i = c6i - b1i
	f1 = do(j-1,1)*c7r-do(j-1,2)*c7i+dh(j-1,1)*c3r-dh(j-1,2)*c3i
	f2 = do(j-1,1)*c7i+do(j-1,2)*c7r+dh(j-1,1)*c3i+dh(j-1,2)*c3r
	f3 = b1r*dh(j+1,1)-b1i*dh(j+1,2)+c2r*do(j+1,1)-c2i*do(j+1,2)
	f4 = b1r*dh(j+1,2)+b1i*dh(j+1,1)+c2r*do(j+1,2)+c2i*do(j+1,1)
	p = f3 + c5r*f1 - c5i*f2
	q = f4 + c5r*f2 + c5i*f1
	re = b2r*b2r + b2i*b2i
	dh(j,1) = (b2r*p + b2i*q)/re
	dh(j,2) = (b2r*q - b2i*p)/re
	do(j,1) = (f1*c1r + f2*c1i)/c1m + c4r*dh(j,1) - c4i*dh(j,2)
	do(j,2) = (f2*c1r - f1*c1i)/c1m + c4r*dh(j,2) + c4i*dh(j,1)

	inet = inet+1
c        if (iflag.eq.2) write(55,10101) inet,x,y,z,
c     $do(j,1)**2+do(j,2)**2,dh(j,1)**2+dh(j,2)**2
c10101   format (1x,I,1x,5(1pe13.6,1x))
#if lahey
        if (iflag.eq.2) write(56,*) real(x,4),real(y,4),real(z,4),
     $real(do(j,1)**2+do(j,2)**2,4),real(dh(j,1)**2+dh(j,2)**2,4)
#else
        if (iflag.eq.2) write(56) SNGL(x),SNGL(y),SNGL(z),
     $SNGL(do(j,1)**2+do(j,2)**2),SNGL(dh(j,1)**2+dh(j,2)**2)
#endif

  444	continue
cvito Add condition in exit surface
c         dh(k1,1)=0.0
c         dh(k1,2)=0.0

  400   continue  

c added by Vito/Manolo
	nexit = int(l/2)
	if (nexit.eq.0) nexit=1
c     write (*,*) '0*** nexit, ref = ',nexit,ref
	ref = bb*reflex(nexit)
	if (iflag.eq.3) then
          do itmp=1,l
            write(66,*) itmp,eta,reflex(itmp)
          enddo
	endif



c	write (10,668) jj,nplan
c	write (10,667) (reflex(m), m = 1,l)
  600   continue	
c  	write (*,*)' results: file ',nfil
c  999	format (a) 
c  667   format (1x,e12.5)	  
c  668	format (1x,i4,1x,i4)
	end
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C                                                                          C
C
C  Program LAUE.F to solve the Takagi-Taupin equations in the Laue case    C
C  for an arbitrary incident wave falling on a deformed single crystal.    C
C  More detailed information can be found in a provisional manual and in   C
C  the list of the program. For further explanations contact:              C
C                                                                          C
C  Dr. Jerzy Gronkowski, Institute of Experimental Physics, University of  C
C                        Warsaw, Hoza 69, 00-681 Warsaw, Poland            C
C                        tel.(48) (2) 628 30 31, fax: (48) (2) 622 61 54   C
C                        email: gronko@fuw.edu.pl                          C
C                                                                          C
C Modified to be called as                                                 C 
C subroutine and changed inputs/outputs. May 13 1997
C                                                                          C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
	subroutine tt_laue(iflag,wavel,eta,ref)

c iflag = 0 : default
c iflag = 1 : prints debug comments
c iflag = 2 : writes integration network:file cryst_tt.net
c iflag = 3 : writes all reflectivity curves:file cryst_tt.all

       implicit real*8 (a-h,o-z)
	complex*16 ab,f,e,dd,b,a,phi           
csrio	character *12 nfil,nfild
	dimension chio(2),chih(2),chib(2),dielep(2),dieleq(2)
	dimension betah(2),c1(2),c2(2),reflex(5000)
	DIMENSION phi(2,5000)
csrio	common /imp/limit,nplan,actz,elem,transv,cos2t,kpol
	
c added by srio
	common /inp_cry/teta,fi,chio,chih,chib,thick
	common /inp_net/window,elem

	common /uu/ao,go,ah,gh,sf,cf,st


      DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
  
c	if (iflag.eq.2) open (55,file='cryst_tt.neta',status='unknown')
#if lahey
        if (iflag.eq.2) open (56,file='cryst_tt.net',status='unknown')
#else
        if (iflag.eq.2) open (56,file='cryst_tt.net',
     $FORM='UNFORMATTED',status='unknown')
#endif

	if (iflag.eq.3) open (66,file='cryst_tt.all',status='unknown')
	
C#########################################################################
C
C  Read of the data from the data file
C
C#########################################################################
C
C  Line # 1
C
C  wavel: the wavelength of the incident radiation [micrometers]
C  teta:  the Bragg angle [degrees]
C  fi:    the asymmetry angles [degrees] (see manual for the convention)
C	
csrio	read (3,*)wavel,teta,fi
c	if (iflag.eq.1) then 
c	  write(*,*) '>>>>>>>>>>>>>>>>'
c	  write(*,*) 'wavel,eta: ',wavel,eta
c	  write(*,*) 'teta,fi: ',wavel,teta,fi
c	endif
C
C  Line # 2
C
C  chio:  the zeroth Fourier component of the electric susceptibility
C         ( two numbers: the real and imaginary part )
C  chih:  the h_th Fourier component of the electric susceptibility
C         ( two numbers: the real and imaginary part )
C  chib:  the -h_th Fourier component of the electric susceptibility
C         ( two numbers: the real and imaginary part )
C

csrio	read (3,*)chio,chih,chib
cif (iflag.eq.1) then
c  write(*,*) 'chio: ',chio
c  write(*,*) 'chih: ',chih
c  write(*,*) 'chib: ',chib
cendif
C
C  Line # 3
C
C  thick: the thickness of the crystal [micrometers]
C  elem:  the elementary step of integration in depth [micrometers]
C  window:the width of the illuminated window on the entrance surface
C         [micrometers] 
C  eta:   the value of the real part of the incidence parameter
C         [dimensionless]
C
csrio	read (3,*)thick,elem,window,eta
cif (iflag.eq.1) write(*,*) 'thick: ',thick
cif (iflag.eq.1) write(*,*) 'elem,window: ',elem,window
	
C
C  Line # 4
C
C  ys:    the value of the y coordinate for the first plane of diffraction
C         [micrometers]
C  ytot:  the total height of the calculated map [micrometers]
C         ( ytot = 0 if only one line is to be calculated )
C
csrio	read (3,*)ys,ytot
	ys = 0.0
	ytot = 0.0
C
C#########################################################################
c
C#########################################################################
C
C  Calculations of general parameters
C
	vector = pi/wavel
	tet1 = teta*pi/180.0
	tpf = (teta + fi)*pi/180.0
	tmf = (teta - fi)*pi/180.0
C
C  go, gh: direction cosines
C
	go = cos(tpf)
	gh = cos(tmf)
        st = sin(tet1)
	ct = cos(tet1)
	ao = sin(tpf)
	ah = sin(tmf)
	to = ao/go
	th = ah/gh
	sf = sin(fi*pi/180.0)
	cf = cos(fi*pi/180.)
	transv = elem*(to + th)
#if lahey
	lmaxt = int(thick/elem + 0.5)
#else
csrio	lmaxt = ifix(thick/elem + 0.5)
	lmaxt = idint(thick/elem + 0.5)
#endif

	if (iflag.eq.1) then 
	  write(11,*) ' <<>>'
c	  write(11,*) '     Incident angle [deg]: ',tpf*180.0/PI
c	  write(11,*) '     Reflected angle [deg]: ',tmf*180.0/PI
	  write(11,*) '     Gamma_0: ',go
	  write(11,*) '     Gamma_H: ',gh
	  write(11,*) '     Asymmetry factor b: ',go/gh
	  write(11,*) ' <<>>'
	  return
	endif
C
C  p,q: elementary integration steps along the s_o and s_h directions
C
	p = elem/go
	q = elem/gh
	sq2 = 0.5*elem*to
	dv1 = 0.5*vector*p
	dv2 = 0.5*vector*q
#if lahey
	window = max(window,transv)
	lmin = int(window/transv + 1.5)
#else
csrio	window = amax1(window,transv)
csrio	lmin = ifix(window/transv + 1.5)
	window = dmax1(window,transv)
	lmin = idint(window/transv + 1.5)
#endif
	betah(1) = -chih(1)*sqrt(gh/go)*eta
	betah(2) = -chio(2)/2.0*(gh/go - 1.0)
	absorp = -chio(2)*2.0*pi/wavel
	limb = 5001 - lmin
	lmax = min0(lmaxt + lmin,4999) - 1
	c1(1) = 1.0 - vector*q*betah(2)
	c2(1) = 1.0 + vector*q*betah(2)
	w1 = vector*q*betah(1)
	dielep(1) = dv1*chib(2)
	dielep(2) = -dv1*chib(1)
	dieleq(1) = dv2*chih(2)
	dieleq(2) = -dv2*chih(1)
#if lahey
	a = cmplx(dielep(1),dielep(2),8)
	b = cmplx(dieleq(1),dieleq(2),8)
#else
csrio	a = cmplx(dielep(1),dielep(2))
csrio	b = cmplx(dieleq(1),dieleq(2))
	a = Dcmplx(dielep(1),dielep(2))
	b = Dcmplx(dieleq(1),dieleq(2))
#endif
	ab = a*b
	phi(1,5000) = (0.0,0.0)
	phi(2,5000) = (0.0,0.0)

cvito factor including anomalous absorption

c	chibchih1=chih(1)*chib(1)-chih(2)*chib(2)
c        chibchih2=chih(1)*chib(2)+chib(1)*chih(2)
c        chibchih=cmplx(chibchih1,chibchih2)
c        first=sqrt(go/abs(gh))/csqrt(chibchih)
c	xoj2=0.5*go/(first*wavel)*(eta+sqrt(eta**2+1.0))
c	absorp2=-4.0*pi*(xoj2+chio(2)/(2.0*wavel))/go


Cvito asymmetry factor^-1 for rocking curve
 
	bb=abs(gh)/go
c     write (*,*) '0*** asymmetry factor =',1.0/bb


C  dy:    the distance between two successive planes of incidence
C         (two lines in the final map)
C  lines: the number of planes of incidence calculated
C
	dy = transv*gh
	lines = int(ytot/dy + 1.5)

C
C calculate number of knots in the integration network
C
	if (iflag.eq.2) then
	tmp = 0.0
	do i=lmin,lmax
	  tmp = tmp+(4999 - (5000 - i) + 1)
	enddo
	write(*,*)
     $'>> Number of knots in the integration network: ',tmp
#if lahey
	write(56,*) real(tmp,4)
#else
	write(56) SNGL(tmp)
#endif
	endif

c inet node index
	inet = 0
C
C  5000: loop over the planes of incidence
C	
	do 5000 nplan = 1,lines
	y = ys + (nplan - 1)*dy
	r1 = 1.0
C
C  8: loop to set the initial conditions along the entrance surface
C
	do 8 k = limb,4999
	x = transv*(k - limb)
#if lahey
	phi(1,k) = cmplx(d(x),0.0D0,8)
	phi(2,k) = cmplx(0.0D0,0.0D0,8)
#else
csrio	phi(1,k) = cmplx(d(x),0.0)
csrio	phi(2,k) = cmplx(0.0,0.0)
	phi(1,k) = Dcmplx(d(x),0.0D0)
	phi(2,k) = Dcmplx(0.0D0,0.0D0)
#endif
   8    continue
   
   	actz = 0.0
C
C  22: the main loop over the lines of the integration network
C       parallel to the entrance surface
C	
	do 22 l = lmin,lmax
	actz = actz + elem
	factor = exp(-absorp*actz/gh)
c	factor = exp(-2.0*absorp*actz/gh)
c	factor = exp(-absorp*actz)
cvito factor2 with anomalous absorption
	factor2=exp(absorp2*actz/gh)
	zi = actz*th + sq2
	limit = 5000 - l
	phi(1,limit) = (0.0,0.0)
	phi(2,limit) = (0.0,0.0)
	z = actz - 0.5*elem
C
C  21:  the main loop over the points of the integration network
C       within each line parallel to the entrance surface
C	
	jj = 0
	do 21 ks = limit,4999
	jj = jj + 1
	x = zi - (ks - limit)*transv
c	if (iflag.eq.2) write(55,*) inet,x,y,z
	uxyz = u(x,y,z)
    	c1(2) = w1 - pi*q*uxyz
	c2(2) = -c1(2)
#if lahey
	dd = cmplx(c2(1),c2(2),8) - ab
     	e = phi(1,ks + 1) + a*phi(2,ks + 1)
	f = b*phi(1,ks) + cmplx(c1(1),c1(2),8)*phi(2,ks)
	phi(1,ks) = (cmplx(c2(1),c2(2),8)*e + a*f)/dd
#else
csrio	dd = cmplx(c2(1),c2(2)) - ab
csrio     	e = phi(1,ks + 1) + a*phi(2,ks + 1)
csrio	f = b*phi(1,ks) + cmplx(c1(1),c1(2))*phi(2,ks)
csrio	phi(1,ks) = (cmplx(c2(1),c2(2))*e + a*f)/dd
	dd = Dcmplx(c2(1),c2(2)) - ab
     	e = phi(1,ks + 1) + a*phi(2,ks + 1)
	f = b*phi(1,ks) + Dcmplx(c1(1),c1(2))*phi(2,ks)
	phi(1,ks) = (Dcmplx(c2(1),c2(2))*e + a*f)/dd
#endif

	phi(2,ks) = (b*e + f)/dd
	inet = inet+1
c	if (iflag.eq.2) write(55,10101) inet,x,y,z,
c     $factor*cdabs(phi(1,ks))**2,factor*cdabs(phi(2,ks))**2
c10101	format (1x,I,1x,5(1pe13.6,1x))
#if lahey
	if (iflag.eq.2) write(56,*) real(x,4),real(y,4),real(z,4),
     $real(factor*abs(phi(1,ks))**2,4),
     $real(factor*abs(phi(2,ks))**2,4)
#else
	if (iflag.eq.2) write(56) SNGL(x),SNGL(y),SNGL(z),
     $SNGL(factor*cdabs(phi(1,ks))**2),
     $SNGL(factor*cdabs(phi(2,ks))**2)
#endif

C	
C  reflex(jj): the intensity of the diffracted wave at the exit surface	
C
#if lahey
	reflex(jj) =bb*factor*abs(phi(2,ks))**2
#else
	reflex(jj) =bb*factor*cdabs(phi(2,ks))**2
#endif

c	factor3=exp(-abs(absorp2*thick))
c	write(*,*)absorp2,thick,factor3
c	write(*,*)factor
c	factor4 = exp(-2.0*absorp*thick)
c	reflex(jj) = bb*factor4*cabs(phi(2,ks))**2
   21	continue
   22	continue
	ref = reflex(int(jj/2))
cendif
	if (iflag.eq.3) then
	  do itmp=1,jj
	    write(66,*) itmp,eta,reflex(itmp)
	  enddo
	endif
csrio	write (10,668) jj,nplan
csrio  668	format (1x,i4,1x,i4)
csrio	write (10,667) (reflex(m), m = 1,jj)
csrio  667   format (1x,e12.5)	
 5000   continue
	end

c======================================================================
      double precision function u(x,y,z)
      implicit real*8 (a-h,o-z)
	

        common /uu/ao,go,ah,gh,sf,cf,st
	common /inp_net/window,elem
	common /uu2/wavel,radius,rnu,i_geometry
	

c conversion of r0 in micron
	
        rm=radius*1.e6

        ccc1=2.0*st/wavel
	if (radius.eq.0.0) then
	    u=0.0
	else
      if (i_geometry.eq.2) then
	   dux=z/rm
	   duz=(window/2.0+x)/rm
	   dwx=   2.0*((window/2.0+x)/rm)
	   dwz=-1.0*(rnu/(1.0-rnu))*z/rm

c deformation for Laue case u=d(u*h)/d(sh)
	  
	   u=ccc1*((dux*ah-duz*gh)*cf-(dwx*ah-dwz*gh)*sf)
      else
       dux=z/rm
       duz=(-window/2.0+x)/rm
       dwx=-2.0*((-window/2.0+x)/rm)
       dwz=-1.0*(rnu/(1.0-rnu))*z/rm
    	   u=-ccc1*((dux*ah+duz*gh)*sf+(dwx*ah+dwz*gh)*cf)
      endif
	endif

        return
        end

	     double precision function d(x)
      implicit real*8 (a-h,o-z)
	common /inp_net/window,elem
      DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /

c	x0=window/2.0
cc	x0=0.0
c	x1=x0+x
c        cz=0.2*window
c        ww=x0*cz
c	if (x1.gt.0.0) then
c       d=sin(pi*(x1-cz)/ww)/(pi*(x1-cz)/ww)
c	endif
c	if (x1.lt.0.0) then
c	   d=sin(pi*(x1+cz)/ww)/(pi*(x1+cz)/ww)
c	endif
c	if (abs(x1).le.cz) d=1.0	


	d = 1.0
	
	return
	end
c======================================================================




