;+
; NAME:
;	nrbGetKnots
;
;
; PURPOSE:
;	This function computes the knots to be used for calculating the 
;	interpolated coefficients for an input surface (or curve)
;
; CATEGORY:
;	NURBS library
;
; CALLING SEQUENCE:
;	nrbGetKnots(grid)
;
; INPUTS:
;    grid: grid of coefficients in the form (nx,4,ny) matrix 
;               (as prepared by rrbConvSurf). 
;	   In case of curve, the format is XXX
;
; KEYWORD PARAMETERS:
;   M = NURBS degree along u parametric variable (default=3)
;   N = NURBS degree along v parametric variable (default=3)
;
; OUTPUTS:
;   An structure with the two arrays in under the tag names:
;	uKnots , vKnots
;   (in case that a curve is input, only uKonts is produced)
;
; OUTPUT KEYWORDS:
;   UU= parametric points by centripetal method along the U direction
;   VV= parametric points by centripetal method along the V direction
;
; PROCEDURE:
;	Using the "centripetal method" 
;       Reference: 
;                "Rational B-splines Curves and Surfaces for CAD and Graphics"
;                by Les A. Piegl  
;                in State of Art in Computer Graphics, Springer- Verlad, 1991, pp. 225-269.
;
; EXAMPLE:
;	a=nrbconvsurf('testsurf.dat')  ; see doc in nrbconvsurf
;
;	b=nrbgetknots(a,uu=uu,vv=vv)
;	help,/str,b
;	
;	
; MODIFICATION HISTORY:
;	written by Davide Bianchi. ESRF. 2006
;
;-	

FUNCTION nrbgetknots, grid, p=p, q=q, uu=uu, vv=vv
;--------------------------------------------------;

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'NURBS: nrbGetKnots error caught: '+!err_string)
   catch, /cancel
   on_error,2
   RETURN,0
endif

IF N_Elements(p) EQ 0 THEN p=3
IF N_Elements(q) EQ 0 THEN q=3


nu=uint(n_elements(reform(grid(*,1,0))))
n=nu-1
nv=uint(n_elements(reform(grid(0,1,*))))
m=nv-1

p1 = (nu-2)
IF p GT p1 THEN BEGIN
  message,'The Nurbs degree along U direction ('+StrCompress(p)+$
          ') is larger than the maximum allowed ('+StrCompress(p1)+$
          '). Set to : '+StrCompress(p1), /continue
  p=p1
ENDIF
p=double(p)

;q1 = (nv-2)
;IF q GT q1 THEN BEGIN
;  message,'The Nurbs degree along V direction ('+StrCompress(q)+$
;          ') is larger than the maximum allowed ('+StrCompress(q1)+$
;          '). Set to : '+StrCompress(q1), /continue
;  q=q1
;ENDIF
;q=double(q)


;/*==========paramentric U array by centripetal method (miss ref)========*/;
uumat=dblarr(nv,nu);

FOR ss=0,m DO BEGIN
    uumat(ss,n)=1.0
    chl=0.0;
    ; Sum of Sqrt ( || Qi - Qi-1 || ) (denominator of Eq 51) 
    FOR tt=1,n DO chl += sqrt(norm((grid(tt,0:2,ss) - grid(tt-1,0:2,ss)),/double))

    ; Eq. 51
    FOR rr=1,nu-2 DO BEGIN
        uumat(ss,rr)=uumat(ss,rr-1) + sqrt(norm((grid(rr,0:2,ss) - grid(rr-1,0:2,ss)),/double))/chl;
    ENDFOR
ENDFOR

; Convert matrix in u's (because we have a surface)  in array 
uu = dblarr(nu)
uu(n) = 1.0

FOR rr=1,n-1 DO BEGIN
    sum=0.0
    FOR ss=0,nv-1 DO sum += uumat(ss,rr)
    uu(rr)=sum/(nv);
ENDFOR


;====NON PERIODIC KNOTS ARRAY along U====;
knots1=dblarr(n-p)

FOR kk=0,n-p-1 DO BEGIN
     sum=0.0
     FOR hh=kk+1,kk+p DO sum += uu(hh)
     knots1(kk)=1/p*sum
ENDFOR

au=dblarr(p+1)
bu=dblarr(p+1)+1D
u_knots=[au, knots1, bu]

;/*==========paramentric V array by centripetal method (miss ref)========*/;
IF nv GT 1 THEN BEGIN

  q1 = (nv-2)
  IF q GT q1 THEN BEGIN
    message,'The Nurbs degree along V direction ('+StrCompress(q)+$
          ') is larger than the maximum allowed ('+StrCompress(q1)+$
          '). Set to : '+StrCompress(q1), /continue
    q=q1
  ENDIF
  q=double(q)

  vv = dblarr(nv)
  vv(m) = 1.0
  vvmat=dblarr(nu,nv)

  FOR ss=0,n DO BEGIN
    vvmat(ss,m)=1.0
    chl=0.0
    FOR tt=1,m DO chl += sqrt(norm((grid(ss,0:2,tt) - grid(ss,0:2,tt-1)),lnorm=2,/double))
    FOR rr=1,m-1 DO BEGIN
        vvmat(ss,rr)=vvmat(ss,rr-1) + sqrt(norm((grid(ss,0:2,rr) - grid(ss,0:2,rr-1)),lnorm=2,/double))/chl
    ENDFOR
  ENDFOR

  FOR rr=1,m-1 DO BEGIN
    sum=0.0
    FOR ss=0,n DO sum += vvmat(ss,rr)
    vv(rr) = sum/(nu)
  ENDFOR


  knots2=dblarr(m-q)


  ;/*====NON PERIODIC KNOTS ARRAY along V====*/;
  FOR kk=0,m-q-1 DO BEGIN
     sum=0.0;
     FOR hh=kk+1,kk+q DO sum += vv(hh)
     knots2(kk)=(1/q)*sum
  ENDFOR

  av = dblarr(q+1)
  bv = dblarr(q+1)
  bv(*)=1D
  v_knots=[av, knots2, bv]

  knots={uknots:u_knots, vknots:v_knots}
  ;return, knots
ENDIF ELSE BEGIN
  knots={uknots:u_knots}
ENDELSE

return, knots

END
