
;+
; NAME:
;	RCIL
; PURPOSE:
;	Calculates the tangential and sagittal focusing radii, given the
;	focal distances. 
; CATEGORY:
;	Optics
; CALLING SEQUENCE:
;	rcil
; KEYWORD PARAMETERS:
;    GROUP
;	The widget id of the parent. Also used for centered the input/output 
;	windows.
; OUTPUTS:
;	The outputs are shown in a text window.
; COMMON BLOCKS:
;	COMMON rcil str   
;	(to store the inputs for following runs)
; SIDE EFFECTS:
;	None.
; RESTRICTIONS:
;	If the IDL session support widgets, the i/o is done via widgets, 
;	otherwise it is done by terminal.
;	None.
; PROCEDURE:
;	User the well known formulas:
;	rt=2*p*q/(p+q)/sin_theta;
;	rs=2*p*q*sin_theta/(p+q);
;
; MODIFICATION HISTORY:
;	Created 2001-03-08 by M. Sanchez del Rio from a c code
;	01/03/14 srio@esrf.fr force to use cm and eV
;	06/02/16 srio@esrf.fr makes it more readible the mirror inputs
;-
;/****************************************************************************
; *  
; *  File:                 rcil.c
; *
; *  Project:              X-ray optics utilities
; *
; *  Description:          Calculates the sagittal and tangential radius 
; *                        of a mirror
; *
; *  Author:               Manolo Sanchez del Rio 
; *
; *  Date:                 November 1991
; *
; *****************************************************************************/
PRO rcil,Group=group

on_error,2

COMMON rcil,str_rcil

TOANGS = physical_constants('hc')
TORAD  = !dpi/180

;
; Inputs
;
   IF N_Elements(str_rcil) EQ 0 THEN BEGIN
   str_rcil = {p:3000.0,q:1000.0, $
	  fmenu:['0','Crystal Energy [eV]','Crystal Wavelength [A]',$
		'GRAZING or BRAGG angle in deg', $
		'GRAZING or BRAGG angle in mrad'] , $
          sin_theta:10000.0, $
          cmenu:['0','Si 111','Si 220','Ge 111','Ge 220','Si 331','Si 311']}
   ENDIF
   flags = ['1','1','1','1','w(2) EQ 0 OR w(2) EQ 1']
   titles = ['distance P [cm]','distance Q [cm]','Angles or Energy', $
	'input the amount','Type of cristal']
          

   action=''
   XScrMenu,str_rcil,/Inter,/NoType,Flags=flags,Titles=titles,Action=action,$
	Wtitle='Focusing radii', Dialog_Parent=group
   IF action EQ 'DONT' THEN RETURN

   p = str_rcil.p
   q = str_rcil.q
   sin_theta = str_rcil.sin_theta
   f = Fix(str_rcil.fmenu[0])
   c = Fix(str_rcil.cmenu[0])
    

; 
; End input section
;


;    /*
;     * Choose the constant depending on crystal type.
;     */
    CASE Fix(c) OF
       0: d=3.135E0;
       1: d=1.92;
       2: d=3.262;
       3: d=1.997;
       4: d=1.246;
       5: d=1.637;
       else: Print," Urecognized input"
    ENDCASE
;    /*
;     * Calculate sin_theta.
;     */
    CASE f OF
       0: sin_theta = (TOANGS/(sin_theta*2.0*d));
       1: sin_theta=0.5*sin_theta/d;
       2: sin_theta=sin(TORAD*sin_theta);
       3: sin_theta=sin(sin_theta/1000);
    ENDCASE

;    /*
;     * Final radii calculation.
;     */
    rt=2*p*q/(p+q)/sin_theta;
    rs=2*p*q*sin_theta/(p+q);

;    /*
;     * Present output.
;     */
    out = [" Focusing conditions (rcil) "," ",$
    " p is "+StrCompress(p,/Rem), $
    " q is "+StrCompress(q,/Rem)]
    IF Fix(str_rcil.fmenu[0]) LE 1 THEN BEGIN
      out=[out, $
      " d is "+StrCompress(d,/Rem), $
      " sin(theta) is "+StrCompress(sin_theta,/Rem)]
    ENDIF
    out=[out, $
    " grazing or Bragg angle is "+ $
        StrCompress(Asin(sin_theta)*180.0/!pi,/Rem)+' deg = '+ $
        StrCompress(Asin(sin_theta)*1000,/Rem)+' mrad', $
    " ",$
    " Tangential radius is "+StrCompress(rt,/Rem), $
    " Sagittal radius is "+StrCompress(rs,/Rem) ]

    XDisplayFile1,Text=out, Dialog_Parent=group, Title='rcil' 

    FOR i=0,N_Elements(out)-1 DO print,out[i]
END
