PRO mirsur,input,zz,xx,yy,QUINTIC=quintic,SHOW=show,GS=gs,LIMITS=limits, $
 NXPOINTS=nxpoints,NYPOINTS=nypoints,EXTRAPOLATE=extrapolate, $
 INVERT=invert,CUTPOINTS=cutpoints
;+
; NAME:
;	MIRSUR
;
; PURPOSE:
;	Plots the surface of a mirror starting from the 
;	SHADOW file (i.e. mirr.xx)
;
; CATEGORY:
;	SHADOW utilities.
;
; CALLING SEQUENCE:
;	MIRSUR,'mirr.01'
;
; INPUTS:
;	input: 	a SHADOW file or idl structure containing the SHADOW
;		data. It can also be a fltarr(3,npts) with the x,y,z
;		coordinates of the surface points.
;
; OPTIONAL INPUTS:
;	z,x,y	The idl variables to strore the (output) surface for
;		additional processing.
;	
; KEYWORD PARAMETERS:
;	QUINTIC: it is passed to the TRIGRID function:
;	        If QUINTIC is set, smooth interpolation is per-
;	        formed using Akima's quintic polynomials from "A 
;	        Method of Bivariate Interpolation and Smooth Surface 
;	        Fitting for Irregularly Distributed Data Points" in 
;	        ACM Transactions on Mathematical Software, 4, 148-
;	        159. The default method is linear interpolation.
;	SHOW:	When set, shows the triangulation before the mirror
;		surface.
;	GS: 	passed to TRIGRID:
;		If present, GS should be a two-element vector 
;		[XS, YS], where XS is the horizontal spacing between 
;	        grid points and YS is the vertical spacing. The 
;	        default is based on the extents of X and Y. If the 
;	        grid starts at X value x0 and ends at x1,then the 
;	        horizontal spacing is
;	LIMITS=[min_colx,min_coly,max_coly,max_coly] optinal limits
;		for the mirror (default=min and max from the SHADOW
;		data)
;	NXPOINTS = number of points in x (default=21)
;	NYPOINTS = number of points in y (default=51)
;	EXTRAPOLATE = passed to TRIGRID:
;        	This keyword sets the quintic interpolation 
;	        mode. TRIGRID extrapolates to grid points outside 
;	        the triangulation if this keyword is assigned to the 
;	        array of boundary node indices returned by the 
;	        optional parameter B of the TRIANGULATE procedure. 
;	        The extrapolation is not smooth, but should give 
;	        acceptable results in most cases.
;	INVERT: When set, switch the X and Y columns
;	CUTPOINTS=n the number of rays used for the plot (default=# of
;		rays in the SHADOW input file)
; OUTPUTS:
;	A grafic.
;
; OPTIONAL OUTPUTS:
;	The surface data in the (optional) input variables x,y and z.
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;	this entry.
;
; RESTRICTIONS:
;	It may be quite time  consuming. The graphic output may not be
;	very well looking.
;
; PROCEDURE:
;	As the mirror file of SHADOW contains the (x,y,z) coordinates
;	for all the rays, and these points are irregularly spread over
;	the mirror surface, a triangulation is necessary. For such a 
;	purpose we use the TRIGRID and TRIANGULATE idl's routines.
;
; EXAMPLE:
;	SURMIR,'mirr.01',z,x,y
;	surface,z,x,y	; to repeat the plot
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio. ESRF. May 1994
;		94/07/08 MSR extends the input to a fltarr(3,*)
;		96/01/30 MSR display 3D triangulation when /show is set.
;-
;
on_error,2
if not(keyword_set(nxpoints)) then nxpoints = 21
if not(keyword_set(nypoints)) then nypoints = 51
if not(keyword_set(extrapolate)) then extrapolate = 0
print,'SURMIR: npoints = ',nxpoints,nypoints

input_size = (size(input))((size(input))(0)+1)
if input_size eq 4 then begin		; input=fltarr(3,npts)
  a1 = input(0,*)
  a2 = input(1,*)
  a3 = input(2,*)
  xtitle=''
  ytitle=''
endif else begin			; shadow input
  a = readsh(input)
  if keyword_set(cutpoints) then cut=cutpoints else cut=a.npoint
  xtitle='Column 1'
  ytitle='Column 2'
  a1 = a.ray(0,0:cut-1)
  a2 = a.ray(1,0:cut-1)
  a3 = a.ray(2,0:cut-1)
endelse
if keyword_set(invert) then begin
  tmp = a1
  a1 = a2
  a2 = tmp
  xtitle='Column 2'
  ytitle='Column 1'
endif
nrays = n_elements(a1) 

if not(keyword_set(quintic)) then quintic = 0 

if not(keyword_set(limits)) then $
  limits = [min(a1,max=tmp),min(a2,max=tmp2),tmp,tmp2 ] else $
  limits = float(limits)
print,'SURMIR: limits = ',limits

if not(keyword_set(gs)) then gs = $ 
  [(limits(2)-limits(0))/float(nxpoints-1) ,(limits(3)-limits(1))/ $
  float(nypoints-1)] else gs=gs
print,'MIRSUR: gs = ',gs

if keyword_set(show) then  begin
  plot,a1,a2,psym=1,title=' col 1, col 2 points'
  pause
endif
;
; triangulate
;
if keyword_set(extrapolate) then triangulate,a1,a2,tr,B else $
  triangulate,a1,a2,tr
if keyword_set(show) then  begin
  plot,a1,a2,psym=1,title='triangulation'
  for i=0,n_elements(tr)/3-1 do begin
    t = [tr(*,i),tr(0,i)]
    plots,a1(t),a2(t)
  endfor
  pause
endif
if keyword_set(show) then  begin
  surface,[[0,0],[0,0]],/save, /nodata, $
    xrange=[min(a1),max(a1)],yrange=[min(a2),max(a2)],zrange=[min(a3),max(a3)]
  ;plot,a1,a2,psym=1,title='triangulation'
  for i=0,n_elements(tr)/3-1 do begin
    t = [tr(*,i),tr(0,i)]
    plots,a1(t),a2(t),a3(t),/t3d,color=125
  endfor
  pause
endif
;
; show surface
;
;
if keyword_set(extrapolate) then zz = $
  trigrid(a1,a2,a3,tr,gs,limits,quintic=quintic,extrapolate=B) else $
  zz = trigrid(a1,a2,a3,tr,gs,limits,quintic=quintic)
xx = findgen(nxpoints)*gs(0) + limits(0)
yy = findgen(nypoints)*gs(1) + limits(1)
help,zz
help,xx
help,yy
zz=zz(0:nxpoints-1,0:nypoints-1)
;surface,zz,xtitle=xtitle,ytitle=ytitle  ;,xx,yy
if keyword_set(show) then tmp = 1 else tmp=0
surface,zz,xx,yy,xtitle=xtitle,ytitle=ytitle ,noerase=tmp, $
  xrange=[min(a1),max(a1)],yrange=[min(a2),max(a2)],zrange=[min(a3),max(a3)]
if keyword_set(quintic) then $
   xyouts,.5,.9,'Quintic Interpolation', align=.5,/NORMAL else $
   xyouts,.5,.9,'Linear Interpolation', align=.5,/NORMAL

end

